#!/usr/bin/python
#   Copyright 2021 Red Hat, Inc.
#
#   Licensed under the Apache License, Version 2.0 (the "License"); you may
#   not use this file except in compliance with the License. You may obtain
#   a copy of the License at
#
#        http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#   WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#   License for the specific language governing permissions and limitations
#   under the License.

from ansible.module_utils.basic import AnsibleModule

import tripleo_yum_config.yum_config as cfg

DOCUMENTATION = r'''
---
module: tripleo_yum_config

short_description: Update yum configuration files for tripleo deployments.

version_added: "1.0"

description:
    - Update specific options for different yum configuration files like
      yum repos, yum modules and yum global configuration. 

options:
    type:
        description: 
          - The type of yum configuration to be changed.
        required: true
        type: str
        choices: [repo, module, global]
    name:
        description: 
          - Name of the repo or module to be changed. This options is 
            mandatory only for repo and module types.
        required: false
        type: str
    enabled:
        description: 
          - Change the yum repo or module to enabled or disabled.
          - This options is ignored for yum global configuration.
        required: false
        type: bool
    set_options:
        description: 
          - Dictionary with options to be updated. All dictionary values must
            be string or list of strings.
        required: false
        type: dict
    file_path:
        description:
          - Absolute path of the configuration file to be changed.
        required: false
        type: str
    dir_path:
        description: 
          - Absolute path of the directory that contains the configuration
            file to be changed.
        required: false
        type: str

author:
    - Douglas Viroel (@viroel)
'''

EXAMPLES = r'''
# Set yum 'appstream' repo to enabled and exclude a list of packages
- name: Enable appstream repo and exclude nodejs and mariadb packages
  become: true
  become_user: root
  tripleo_yum_config:
    type: repo
    name: appstream
    enabled: true
    set_options:
      exclude:
        - nodejs*
        - mariadb*

# Set yum global configuration options
- name: Set yum global options
  become: true
  become_user: root
  tripleo_yum_config:
    type: global
    file_path: /etc/dnf/dnf.conf
    set_options:
      skip_if_unavailable: "False"
      keepcache: "0"
'''


def run_module():
    # define available arguments/parameters a user can pass to the module
    supported_config_types = ['repo', 'module', 'global']
    module_args = dict(
        type=dict(type='str', required=True, choices=supported_config_types),
        name=dict(type='str', required=False),
        enabled=dict(type='bool', required=False),
        set_options=dict(type='dict', required=False),
        file_path=dict(type='str', required=False),
        dir_path=dict(type='str', required=False),
    )

    # seed the result dict in the object
    # we primarily care about changed and state
    # changed is if this module effectively modified the target
    # state will include any data that you want your module to pass back
    # for consumption, for example, in a subsequent task
    result = dict(
        changed=False,
        msg=''
    )

    # the AnsibleModule object will be our abstraction working with Ansible
    # this includes instantiation, a couple of common attr would be the
    # args/params passed to the execution, as well as if the module
    # supports check mode
    module = AnsibleModule(
        argument_spec=module_args,
        supports_check_mode=True
    )

    m_type = module.params.get('type')
    m_name = module.params.get('name')
    m_set_opts = module.params.get('set_options', {})
    m_enabled = module.params.get('enabled')
    m_file_path = module.params.get('file_path')
    m_dir_path = module.params.get('dir_path')

    # Sanity checks
    if m_type in ['repo', 'module'] and m_name is None:
        result['msg'] = (
            "The parameter 'name' is mandatory when 'type' is set to 'repo' "
            "or 'module'.")
        module.fail_json(**result)

    # 'set_options' expects a dict that can hold, as value, strings and list
    # of strings. List of strings will be converted to a comma-separated list.
    invalid_set_opts_msg = (
        "The provided value for 'set_options' parameter has an invalid "
        "format. All dict values must be string or a list of strings.")
    for k, v in m_set_opts.items():
        if isinstance(v, list):
            if not all(isinstance(elem, str) for elem in v):
                result['msg'] = invalid_set_opts_msg
                module.fail_json(**result)
            m_set_opts[k] = ','.join(v)
        elif not isinstance(v, str):
            result['msg'] = invalid_set_opts_msg
            module.fail_json(**result)

    if module.check_mode:
        # Checks were already made above
        module.exit_json(**result)

    # Module execution
    try:
        if m_type == 'repo':
            config_obj = cfg.TripleOYumRepoConfig(
                file_path=m_file_path,
                dir_path=m_dir_path)
            config_obj.update_section(m_name, m_set_opts, enable=m_enabled)

        elif m_type == 'module':
            config_obj = cfg.TripleOYumModuleConfig(
                file_path=m_file_path,
                dir_path=m_dir_path)
            config_obj.update_section(m_name, m_set_opts, enable=m_enabled)

        elif m_type == 'global':
            config_obj = cfg.TripleOYumGlobalConfig(file_path=m_file_path)
            config_obj.update_section('main', m_set_opts)

    except Exception as exc:
        result['msg'] = str(exc)
        module.fail_json(**result)

    # Successful module execution
    result['changed'] = True
    result['msg'] = (
        "Yum {} configuration was successfully updated.".format(m_type)
    )
    module.exit_json(**result)


def main():
    run_module()


if __name__ == '__main__':
    main()
