# kraken/logic/activity_logger.py

import sys
import traceback
from datetime import datetime, timezone, timedelta
from typing import Optional
from uuid import UUID

from agptools.logs import logger
from ..storage import SurrealistStorage  # Import the correct storage type
from ..syncmodels import WaveStorage

from ..model.activity import ActivityStatus

# Remove incorrect import if it exists
# from kraken.storage.surreal import SurrealStorageManagerAsync

# Configuration constants removed, will be read from config
# ACTIVITY_TABLE = "activity_log"
# DEFAULT_NAMESPACE = "kraken"
# DEFAULT_DATABASE = "kraken"

log = logger(__name__)


class ActivityLogger:
    """Handles logging activity status updates to SurrealDB."""

    ACTIVITIES_URL = "system://monitor/activity"

    def __init__(
        self,
        storage: WaveStorage,
        config: Optional[dict] = None,  # Expect SurrealistStorage directly
    ):
        config = config or {}
        self.enabled = config.get("activity_logging_enabled", True)
        self.activity_log_table = config.get("activity_log_table", "activity_log")
        self.storage = storage  # Assign the passed storage object directly
        # Assuming namespace/db comes from storage manager's connection details if needed
        # self.table_path = f"{DEFAULT_NAMESPACE}://{DEFAULT_DATABASE}/{self.activity_log_table}" # Old path calculation

    # Removed _ensure_connection method as SurrealistStorage handles connection internally

    async def log_activity_start(
        self,
        activity_type: str,
        name: str,
        wave_timestamp: Optional[datetime] = None,
        details: Optional[dict] = None,
    ) -> Optional[UUID]:
        """
        Logs the start of a new activity and returns its generated UUID.
        Sets status to 'pending'.
        """
        if not self.enabled:
            return None
        try:
            # await self._ensure_connection() # Removed call
            activity = ActivityStatus(
                wave_timestamp=wave_timestamp or datetime.now(tz=timezone.utc),
                activity_type=activity_type,
                name=name,
                status="pending",  # Initial status
                details=details or {},
            )
            # Use SurrealDB's record ID generation by creating without specifying ID
            # The ID will be like 'activity_log:xxxxxxxx'
            created_record = await self.storage.put(
                self.activity_log_table,
                data=activity.model_dump(exclude={"activity_id"}),
            )

            if (
                created_record
                and isinstance(created_record, list)
                and created_record[0].get("id")
            ):
                # Extract the UUID part if SurrealDB ID format is known, or use the generated one
                # Assuming SurrealDB returns the full record including its generated ID
                # If SurrealDB uses 'table:id' format, parse it. Otherwise, use the UUID from the model.
                # For simplicity, let's assume we use the UUID generated by Pydantic model
                # and update the record *after* creation to set the correct ID field if needed,
                # or structure the table to use the UUID as the primary key part.

                # Let's assume we use the Pydantic generated UUID and want it as the Surreal ID part
                surreal_id = f"{self.activity_log_table}:{activity.activity_id}"
                # We might need to update the record we just created to ensure the activity_id field matches
                # Or ideally, structure the table so `activity_id` is the primary key.
                # For now, let's just return the generated UUID.
                log.debug(
                    f"Activity started and logged: {activity.activity_id} ({name})"
                )
                return activity.activity_id
            else:
                log.error(
                    f"Failed to create activity log entry for {name}. Record: {created_record}"
                )
                return None

        except Exception as e:
            log.error(f"Error logging activity start for {name}: {e}")
            log.error("".join(traceback.format_exception(*sys.exc_info())))
            return None

    async def update_activity(
        self,
        activity: ActivityStatus,
    ):
        """
        Updates an existing activity log entry by its UUID.
        Uses SurrealDB's MERGE functionality.
        """
        try:
            # await self._ensure_connection() # Removed call
            now = datetime.now(tz=timezone.utc)
            activity.update_time = now
            activity.duration_seconds = int((now - activity.start_time).total_seconds())
            # activity.details = str(activity.arguments)

            speed = activity.duration_seconds / (activity.progress_percentage or 1)
            activity.arguments["speed"] = speed
            remain = timedelta(seconds=speed * (100 - activity.progress_percentage))
            activity.eta_time = now + remain

            uid = activity.id
            uri = f"{self.ACTIVITIES_URL}:{uid}"

            # Use MERGE to update only specified fields
            # SurrealistStorage might handle serialization.
            # log.debug(f"Updating activity {activity_id} with: {update_data}")
            data = activity.model_dump()
            result = await self.storage.update(uri, data)

            if not result:
                log.warning(
                    f"Failed to update activity log for {activity}. Update result: {result}"
                )

        except Exception as e:
            log.error(f"Error updating activity {activity}: {e}")
            log.error("".join(traceback.format_exception(*sys.exc_info())))


class HIDE_ActivityLogger:
    """Handles logging activity status updates to SurrealDB."""

    def __init__(
        self,
        storage: WaveStorage,
        config: Optional[dict] = None,  # Expect SurrealistStorage directly
    ):
        config = config or {}
        self.enabled = config.get("activity_logging_enabled", True)
        self.activity_log_table = config.get("activity_log_table", "activity_log")
        self.storage = storage  # Assign the passed storage object directly
        # Assuming namespace/db comes from storage manager's connection details if needed
        # self.table_path = f"{DEFAULT_NAMESPACE}://{DEFAULT_DATABASE}/{self.activity_log_table}" # Old path calculation

    # Removed _ensure_connection method as SurrealistStorage handles connection internally

    async def log_activity_start(
        self,
        activity_type: str,
        name: str,
        wave_timestamp: Optional[datetime] = None,
        details: Optional[dict] = None,
    ) -> Optional[UUID]:
        """
        Logs the start of a new activity and returns its generated UUID.
        Sets status to 'pending'.
        """
        if not self.enabled:
            return None
        try:
            # await self._ensure_connection() # Removed call
            activity = ActivityStatus(
                wave_timestamp=wave_timestamp or datetime.now(tz=timezone.utc),
                activity_type=activity_type,
                name=name,
                status="pending",  # Initial status
                details=details or {},
            )
            # Use SurrealDB's record ID generation by creating without specifying ID
            # The ID will be like 'activity_log:xxxxxxxx'
            created_record = await self.storage.put(
                self.activity_log_table,
                data=activity.model_dump(exclude={"activity_id"}),
            )

            if (
                created_record
                and isinstance(created_record, list)
                and created_record[0].get("id")
            ):
                # Extract the UUID part if SurrealDB ID format is known, or use the generated one
                # Assuming SurrealDB returns the full record including its generated ID
                # If SurrealDB uses 'table:id' format, parse it. Otherwise, use the UUID from the model.
                # For simplicity, let's assume we use the UUID generated by Pydantic model
                # and update the record *after* creation to set the correct ID field if needed,
                # or structure the table to use the UUID as the primary key part.

                # Let's assume we use the Pydantic generated UUID and want it as the Surreal ID part
                surreal_id = f"{self.activity_log_table}:{activity.activity_id}"
                # We might need to update the record we just created to ensure the activity_id field matches
                # Or ideally, structure the table so `activity_id` is the primary key.
                # For now, let's just return the generated UUID.
                log.debug(
                    f"Activity started and logged: {activity.activity_id} ({name})"
                )
                return activity.activity_id
            else:
                log.error(
                    f"Failed to create activity log entry for {name}. Record: {created_record}"
                )
                return None

        except Exception as e:
            log.error(f"Error logging activity start for {name}: {e}")
            log.error("".join(traceback.format_exception(*sys.exc_info())))
            return None

    async def update_activity(
        self,
        activity_id: UUID,
        status: Optional[str] = None,
        start_time: Optional[datetime] = None,
        end_time: Optional[datetime] = None,
        progress_percentage: Optional[float] = None,
        details_update: Optional[dict] = None,
        error_info: Optional[Exception] = None,
    ):
        """
        Updates an existing activity log entry by its UUID.
        Uses SurrealDB's MERGE functionality.
        """
        if not self.enabled or not activity_id:
            if not self.enabled:
                log.debug("Activity logging disabled, skipping update.")
            else:  # activity_id is None
                log.warning("Attempted to update activity with no ID.")
            return

        try:
            # await self._ensure_connection() # Removed call
            surreal_record_id = f"{self.activity_log_table}:{activity_id}"
            update_data = {}

            if status:
                update_data["status"] = status
            if start_time:
                # Ensure timezone-aware
                update_data["start_time"] = (
                    start_time.astimezone(timezone.utc)
                    if start_time.tzinfo
                    else start_time.replace(tzinfo=timezone.utc)
                )
            if end_time:
                # Ensure timezone-aware
                update_data["end_time"] = (
                    end_time.astimezone(timezone.utc)
                    if end_time.tzinfo
                    else end_time.replace(tzinfo=timezone.utc)
                )
                # Let Pydantic model handle duration calculation if possible,
                # otherwise, calculate here if start_time is known.
                # For simplicity, we rely on reading the record back or assume model handles it.
            if progress_percentage is not None:  # Check explicitly for None
                # Validate progress here or rely on Pydantic model validation if reading back
                if 0 <= progress_percentage <= 100:
                    update_data["progress_percentage"] = progress_percentage
                else:
                    log.warning(
                        f"Invalid progress percentage ({progress_percentage}) for activity {activity_id}. Ignored."
                    )

            if details_update:
                # For MERGE, we need to handle nested dict updates carefully.
                # A simple merge might overwrite the whole details dict.
                # Fetch-modify-update might be safer for nested dicts if needed.
                # For now, let's assume we overwrite or add keys.
                # A safer approach might be needed depending on SurrealDB MERGE behavior with dicts.
                # Example: update_data["details"] = {"$merge": details_update} # Check SurrealDB syntax
                update_data["details"] = details_update  # Simple overwrite for now

            if error_info:
                update_data["status"] = "failed"  # Ensure status is failed on error
                error_details = {
                    "error_type": type(error_info).__name__,
                    "error_message": str(error_info),
                    "traceback": "".join(
                        traceback.format_exception_only(type(error_info), error_info)
                    ),
                }
                # Merge error details with existing or new details
                update_data["details"] = {
                    **(update_data.get("details", {})),
                    **error_details,
                }

            if not update_data:
                log.debug(f"No updates provided for activity {activity_id}.")
                return

            # Use MERGE to update only specified fields
            # Note: Ensure datetime objects are properly serialized for SurrealDB (e.g., ISO format string)
            # Pydantic models usually handle this, but direct dict construction needs care.
            # SurrealistStorage might handle serialization.
            log.debug(f"Updating activity {activity_id} with: {update_data}")
            result = await self.storage.update(
                surreal_record_id, data=update_data
            )  # Assuming update performs a MERGE

            if not result:
                log.warning(
                    f"Failed to update activity log for {activity_id}. Update result: {result}"
                )

        except Exception as e:
            log.error(f"Error updating activity {activity_id}: {e}")
            log.error("".join(traceback.format_exception(*sys.exc_info())))
