"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
module.exports = {
    'default stream'(test) {
        const stack = new core_1.Stack();
        new lib_1.Stream(stack, 'MyStream');
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyStream5C050E93": {
                    "Type": "AWS::Kinesis::Stream",
                    "Properties": {
                        "RetentionPeriodHours": 24,
                        "ShardCount": 1
                    }
                }
            }
        });
        test.done();
    },
    'stream from attributes'(test) {
        const stack = new core_1.Stack();
        const s = lib_1.Stream.fromStreamAttributes(stack, 'MyStream', {
            streamArn: 'arn:aws:kinesis:region:account-id:stream/stream-name'
        });
        test.equals(s.streamArn, 'arn:aws:kinesis:region:account-id:stream/stream-name');
        test.done();
    },
    "uses explicit shard count"(test) {
        const stack = new core_1.Stack();
        new lib_1.Stream(stack, 'MyStream', {
            shardCount: 2
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyStream5C050E93": {
                    "Type": "AWS::Kinesis::Stream",
                    "Properties": {
                        "RetentionPeriodHours": 24,
                        "ShardCount": 2
                    }
                }
            }
        });
        test.done();
    },
    "uses explicit retention period"(test) {
        const stack = new core_1.Stack();
        new lib_1.Stream(stack, 'MyStream', {
            retentionPeriodHours: 168
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyStream5C050E93": {
                    "Type": "AWS::Kinesis::Stream",
                    "Properties": {
                        "RetentionPeriodHours": 168,
                        "ShardCount": 1
                    }
                }
            }
        });
        test.done();
    },
    "retention period must be between 24 and 168 hours"(test) {
        test.throws({
            block: () => {
                new lib_1.Stream(new core_1.Stack(), 'MyStream', {
                    retentionPeriodHours: 169
                });
            },
            message: "retentionPeriodHours must be between 24 and 168 hours"
        });
        test.throws({
            block: () => {
                new lib_1.Stream(new core_1.Stack(), 'MyStream', {
                    retentionPeriodHours: 23
                });
            },
            message: "retentionPeriodHours must be between 24 and 168 hours"
        });
        test.done();
    },
    "auto-creates KMS key if encryption type is KMS but no key is provided"(test) {
        const stack = new core_1.Stack();
        new lib_1.Stream(stack, 'MyStream', {
            encryption: lib_1.StreamEncryption.KMS
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "MyStreamKey76F3300E": {
                    "Type": "AWS::KMS::Key",
                    "Properties": {
                        "Description": "Created by MyStream",
                        "KeyPolicy": {
                            "Statement": [
                                {
                                    "Action": [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion",
                                        "kms:GenerateDataKey"
                                    ],
                                    "Effect": "Allow",
                                    "Principal": {
                                        "AWS": {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        "Ref": "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        "Ref": "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    "Resource": "*"
                                }
                            ],
                            "Version": "2012-10-17"
                        }
                    },
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain"
                },
                "MyStream5C050E93": {
                    "Type": "AWS::Kinesis::Stream",
                    "Properties": {
                        "RetentionPeriodHours": 24,
                        "ShardCount": 1,
                        "StreamEncryption": {
                            "EncryptionType": "KMS",
                            "KeyId": {
                                "Fn::GetAtt": [
                                    "MyStreamKey76F3300E",
                                    "Arn"
                                ]
                            }
                        }
                    }
                }
            }
        });
        test.done();
    },
    "uses explicit KMS key if encryption type is KMS and a key is provided"(test) {
        const stack = new core_1.Stack();
        const explicitKey = new kms.Key(stack, 'ExplicitKey', {
            description: `Explicit Key`
        });
        new lib_1.Stream(stack, 'MyStream', {
            encryption: lib_1.StreamEncryption.KMS,
            encryptionKey: explicitKey
        });
        assert_1.expect(stack).toMatch({
            "Resources": {
                "ExplicitKey7DF42F37": {
                    "Type": "AWS::KMS::Key",
                    "Properties": {
                        "Description": "Explicit Key",
                        "KeyPolicy": {
                            "Statement": [
                                {
                                    "Action": [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion",
                                        "kms:GenerateDataKey"
                                    ],
                                    "Effect": "Allow",
                                    "Principal": {
                                        "AWS": {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        "Ref": "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        "Ref": "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    "Resource": "*"
                                }
                            ],
                            "Version": "2012-10-17"
                        }
                    },
                    "DeletionPolicy": "Retain",
                    "UpdateReplacePolicy": "Retain"
                },
                "MyStream5C050E93": {
                    "Type": "AWS::Kinesis::Stream",
                    "Properties": {
                        "RetentionPeriodHours": 24,
                        "ShardCount": 1,
                        "StreamEncryption": {
                            "EncryptionType": "KMS",
                            "KeyId": {
                                "Fn::GetAtt": [
                                    "ExplicitKey7DF42F37",
                                    "Arn"
                                ]
                            }
                        }
                    }
                }
            }
        });
        test.done();
    },
    "permissions": {
        "with encryption": {
            "grantRead creates and attaches a policy with read only access to Stream and EncryptionKey"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream', {
                    encryption: lib_1.StreamEncryption.KMS
                });
                const user = new iam.User(stack, "MyUser");
                stream.grantRead(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStreamKey76F3300E": {
                            "Type": "AWS::KMS::Key",
                            "Properties": {
                                "Description": "Created by MyStream",
                                "KeyPolicy": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kms:Create*",
                                                "kms:Describe*",
                                                "kms:Enable*",
                                                "kms:List*",
                                                "kms:Put*",
                                                "kms:Update*",
                                                "kms:Revoke*",
                                                "kms:Disable*",
                                                "kms:Get*",
                                                "kms:Delete*",
                                                "kms:ScheduleKeyDeletion",
                                                "kms:CancelKeyDeletion",
                                                "kms:GenerateDataKey"
                                            ],
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::Join": [
                                                        "",
                                                        [
                                                            "arn:",
                                                            {
                                                                "Ref": "AWS::Partition"
                                                            },
                                                            ":iam::",
                                                            {
                                                                "Ref": "AWS::AccountId"
                                                            },
                                                            ":root"
                                                        ]
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        },
                                        {
                                            "Action": "kms:Decrypt",
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::GetAtt": [
                                                        "MyUserDC45028B",
                                                        "Arn"
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                }
                            },
                            "DeletionPolicy": "Retain",
                            "UpdateReplacePolicy": "Retain"
                        },
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1,
                                "StreamEncryption": {
                                    "EncryptionType": "KMS",
                                    "KeyId": {
                                        "Fn::GetAtt": [
                                            "MyStreamKey76F3300E",
                                            "Arn"
                                        ]
                                    }
                                }
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:GetRecords",
                                                "kinesis:GetShardIterator"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        },
                                        {
                                            "Action": "kms:Decrypt",
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStreamKey76F3300E",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            },
            "grantWrite creates and attaches a policy with write only access to Stream and EncryptionKey"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream', {
                    encryption: lib_1.StreamEncryption.KMS
                });
                const user = new iam.User(stack, "MyUser");
                stream.grantWrite(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStreamKey76F3300E": {
                            "Type": "AWS::KMS::Key",
                            "Properties": {
                                "Description": "Created by MyStream",
                                "KeyPolicy": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kms:Create*",
                                                "kms:Describe*",
                                                "kms:Enable*",
                                                "kms:List*",
                                                "kms:Put*",
                                                "kms:Update*",
                                                "kms:Revoke*",
                                                "kms:Disable*",
                                                "kms:Get*",
                                                "kms:Delete*",
                                                "kms:ScheduleKeyDeletion",
                                                "kms:CancelKeyDeletion",
                                                "kms:GenerateDataKey"
                                            ],
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::Join": [
                                                        "",
                                                        [
                                                            "arn:",
                                                            {
                                                                "Ref": "AWS::Partition"
                                                            },
                                                            ":iam::",
                                                            {
                                                                "Ref": "AWS::AccountId"
                                                            },
                                                            ":root"
                                                        ]
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        },
                                        {
                                            "Action": [
                                                "kms:Encrypt",
                                                "kms:ReEncrypt*",
                                                "kms:GenerateDataKey*"
                                            ],
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::GetAtt": [
                                                        "MyUserDC45028B",
                                                        "Arn"
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                }
                            },
                            "DeletionPolicy": "Retain",
                            "UpdateReplacePolicy": "Retain"
                        },
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1,
                                "StreamEncryption": {
                                    "EncryptionType": "KMS",
                                    "KeyId": {
                                        "Fn::GetAtt": [
                                            "MyStreamKey76F3300E",
                                            "Arn"
                                        ]
                                    }
                                }
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:PutRecord",
                                                "kinesis:PutRecords"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        },
                                        {
                                            "Action": [
                                                "kms:Encrypt",
                                                "kms:ReEncrypt*",
                                                "kms:GenerateDataKey*",
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStreamKey76F3300E",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            },
            "grantReadWrite creates and attaches a policy with access to Stream and EncryptionKey"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream', {
                    encryption: lib_1.StreamEncryption.KMS
                });
                const user = new iam.User(stack, "MyUser");
                stream.grantReadWrite(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStreamKey76F3300E": {
                            "Type": "AWS::KMS::Key",
                            "Properties": {
                                "Description": "Created by MyStream",
                                "KeyPolicy": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kms:Create*",
                                                "kms:Describe*",
                                                "kms:Enable*",
                                                "kms:List*",
                                                "kms:Put*",
                                                "kms:Update*",
                                                "kms:Revoke*",
                                                "kms:Disable*",
                                                "kms:Get*",
                                                "kms:Delete*",
                                                "kms:ScheduleKeyDeletion",
                                                "kms:CancelKeyDeletion",
                                                "kms:GenerateDataKey"
                                            ],
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::Join": [
                                                        "",
                                                        [
                                                            "arn:",
                                                            {
                                                                "Ref": "AWS::Partition"
                                                            },
                                                            ":iam::",
                                                            {
                                                                "Ref": "AWS::AccountId"
                                                            },
                                                            ":root"
                                                        ]
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        },
                                        {
                                            "Action": [
                                                "kms:Decrypt",
                                                "kms:Encrypt",
                                                "kms:ReEncrypt*",
                                                "kms:GenerateDataKey*"
                                            ],
                                            "Effect": "Allow",
                                            "Principal": {
                                                "AWS": {
                                                    "Fn::GetAtt": [
                                                        "MyUserDC45028B",
                                                        "Arn"
                                                    ]
                                                }
                                            },
                                            "Resource": "*"
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                }
                            },
                            "DeletionPolicy": "Retain",
                            "UpdateReplacePolicy": "Retain"
                        },
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1,
                                "StreamEncryption": {
                                    "EncryptionType": "KMS",
                                    "KeyId": {
                                        "Fn::GetAtt": [
                                            "MyStreamKey76F3300E",
                                            "Arn"
                                        ]
                                    }
                                }
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:GetRecords",
                                                "kinesis:GetShardIterator",
                                                "kinesis:PutRecord",
                                                "kinesis:PutRecords"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        },
                                        {
                                            "Action": [
                                                "kms:Decrypt",
                                                "kms:Encrypt",
                                                "kms:ReEncrypt*",
                                                "kms:GenerateDataKey*"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStreamKey76F3300E",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            }
        },
        "with no encryption": {
            "grantRead creates and associates a policy with read only access to Stream"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream');
                const user = new iam.User(stack, "MyUser");
                stream.grantRead(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:GetRecords",
                                                "kinesis:GetShardIterator"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            },
            "grantWrite creates and attaches a policy with write only access to Stream"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream');
                const user = new iam.User(stack, "MyUser");
                stream.grantWrite(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:PutRecord",
                                                "kinesis:PutRecords"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            },
            "greatReadWrite creates and attaches a policy with write only access to Stream"(test) {
                const stack = new core_1.Stack();
                const stream = new lib_1.Stream(stack, 'MyStream');
                const user = new iam.User(stack, "MyUser");
                stream.grantReadWrite(user);
                assert_1.expect(stack).toMatch({
                    "Resources": {
                        "MyStream5C050E93": {
                            "Type": "AWS::Kinesis::Stream",
                            "Properties": {
                                "RetentionPeriodHours": 24,
                                "ShardCount": 1
                            }
                        },
                        "MyUserDC45028B": {
                            "Type": "AWS::IAM::User"
                        },
                        "MyUserDefaultPolicy7B897426": {
                            "Type": "AWS::IAM::Policy",
                            "Properties": {
                                "PolicyDocument": {
                                    "Statement": [
                                        {
                                            "Action": [
                                                "kinesis:DescribeStream",
                                                "kinesis:GetRecords",
                                                "kinesis:GetShardIterator",
                                                "kinesis:PutRecord",
                                                "kinesis:PutRecords"
                                            ],
                                            "Effect": "Allow",
                                            "Resource": {
                                                "Fn::GetAtt": [
                                                    "MyStream5C050E93",
                                                    "Arn"
                                                ]
                                            }
                                        }
                                    ],
                                    "Version": "2012-10-17"
                                },
                                "PolicyName": "MyUserDefaultPolicy7B897426",
                                "Users": [
                                    {
                                        "Ref": "MyUserDC45028B"
                                    }
                                ]
                            }
                        }
                    }
                });
                test.done();
            }
        }
    },
    "cross-stack permissions": {
        "no encryption"(test) {
            const app = new core_1.App();
            const stackA = new core_1.Stack(app, 'stackA');
            const streamFromStackA = new lib_1.Stream(stackA, 'MyStream');
            const stackB = new core_1.Stack(app, 'stackB');
            const user = new iam.User(stackB, 'UserWhoNeedsAccess');
            streamFromStackA.grantRead(user);
            assert_1.expect(stackA).toMatch({
                "Resources": {
                    "MyStream5C050E93": {
                        "Type": "AWS::Kinesis::Stream",
                        "Properties": {
                            "RetentionPeriodHours": 24,
                            "ShardCount": 1
                        }
                    }
                },
                "Outputs": {
                    "ExportsOutputFnGetAttMyStream5C050E93Arn4ABF30CD": {
                        "Value": {
                            "Fn::GetAtt": [
                                "MyStream5C050E93",
                                "Arn"
                            ]
                        },
                        "Export": {
                            "Name": "stackA:ExportsOutputFnGetAttMyStream5C050E93Arn4ABF30CD"
                        }
                    }
                }
            });
            assert_1.expect(stackB).toMatch({
                "Resources": {
                    "UserWhoNeedsAccessF8959C3D": {
                        "Type": "AWS::IAM::User"
                    },
                    "UserWhoNeedsAccessDefaultPolicy6A9EB530": {
                        "Type": "AWS::IAM::Policy",
                        "Properties": {
                            "PolicyDocument": {
                                "Statement": [
                                    {
                                        "Action": [
                                            "kinesis:DescribeStream",
                                            "kinesis:GetRecords",
                                            "kinesis:GetShardIterator"
                                        ],
                                        "Effect": "Allow",
                                        "Resource": {
                                            "Fn::ImportValue": "stackA:ExportsOutputFnGetAttMyStream5C050E93Arn4ABF30CD"
                                        }
                                    }
                                ],
                                "Version": "2012-10-17"
                            },
                            "PolicyName": "UserWhoNeedsAccessDefaultPolicy6A9EB530",
                            "Users": [
                                {
                                    "Ref": "UserWhoNeedsAccessF8959C3D"
                                }
                            ]
                        }
                    }
                }
            });
            test.done();
        },
        "fails with encryption due to cyclic dependency"(test) {
            const app = new core_1.App();
            const stackA = new core_1.Stack(app, 'stackA');
            const streamFromStackA = new lib_1.Stream(stackA, 'MyStream', {
                encryption: lib_1.StreamEncryption.KMS
            });
            const stackB = new core_1.Stack(app, 'stackB');
            const user = new iam.User(stackB, 'UserWhoNeedsAccess');
            streamFromStackA.grantRead(user);
            test.throws(() => app.synth(), /'stackB' depends on 'stackA'/);
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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