"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const iam = require("../lib");
test('use of cross-stack role reference does not lead to URLSuffix being exported', () => {
    // GIVEN
    const app = new core_1.App();
    const first = new core_1.Stack(app, 'First');
    const second = new core_1.Stack(app, 'Second');
    // WHEN
    const role = new iam.Role(first, 'Role', {
        assumedBy: new iam.ServicePrincipal('s3.amazonaws.com'),
    });
    new core_1.CfnOutput(second, 'Output', {
        value: role.roleArn,
    });
    // THEN
    app.synth();
    expect(first).toMatchTemplate({
        Resources: {
            Role1ABCC5F0: {
                Type: 'AWS::IAM::Role',
                Properties: {
                    AssumeRolePolicyDocument: {
                        Statement: [
                            {
                                Action: 'sts:AssumeRole',
                                Effect: 'Allow',
                                Principal: { Service: 's3.amazonaws.com' },
                            },
                        ],
                        Version: '2012-10-17',
                    },
                },
            },
        },
        Outputs: {
            ExportsOutputFnGetAttRole1ABCC5F0ArnB4C0B73E: {
                Value: { 'Fn::GetAtt': ['Role1ABCC5F0', 'Arn'] },
                Export: {
                    Name: 'First:ExportsOutputFnGetAttRole1ABCC5F0ArnB4C0B73E',
                },
            },
        },
    });
});
test('cannot have multiple principals with different conditions in the same statement', () => {
    const stack = new core_1.Stack(undefined, 'First');
    const user = new iam.User(stack, 'User');
    expect(() => {
        user.addToPolicy(new iam.PolicyStatement({
            principals: [
                new iam.ServicePrincipal('myService.amazon.com', {
                    conditions: {
                        StringEquals: {
                            hairColor: 'blond',
                        },
                    },
                }),
                new iam.ServicePrincipal('yourservice.amazon.com', {
                    conditions: {
                        StringEquals: {
                            hairColor: 'black',
                        },
                    },
                }),
            ],
        }));
    }).toThrow(/All principals in a PolicyStatement must have the same Conditions/);
});
test('can have multiple principals the same conditions in the same statement', () => {
    const stack = new core_1.Stack(undefined, 'First');
    const user = new iam.User(stack, 'User');
    user.addToPolicy(new iam.PolicyStatement({
        principals: [
            new iam.ServicePrincipal('myService.amazon.com'),
            new iam.ServicePrincipal('yourservice.amazon.com'),
        ],
    }));
    user.addToPolicy(new iam.PolicyStatement({
        principals: [
            new iam.ServicePrincipal('myService.amazon.com', {
                conditions: {
                    StringEquals: { hairColor: 'blond' },
                },
            }),
            new iam.ServicePrincipal('yourservice.amazon.com', {
                conditions: {
                    StringEquals: { hairColor: 'blond' },
                },
            }),
        ],
    }));
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicHJpbmNpcGFscy50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsicHJpbmNpcGFscy50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsZ0NBQThCO0FBQzlCLHdDQUFzRDtBQUN0RCw4QkFBOEI7QUFFOUIsSUFBSSxDQUFDLDZFQUE2RSxFQUFFLEdBQUcsRUFBRTtJQUN2RixRQUFRO0lBQ1IsTUFBTSxHQUFHLEdBQUcsSUFBSSxVQUFHLEVBQUUsQ0FBQztJQUN0QixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssQ0FBQyxHQUFHLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFDdEMsTUFBTSxNQUFNLEdBQUcsSUFBSSxZQUFLLENBQUMsR0FBRyxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBRXhDLE9BQU87SUFDUCxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtRQUN2QyxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsa0JBQWtCLENBQUM7S0FDeEQsQ0FBQyxDQUFDO0lBRUgsSUFBSSxnQkFBUyxDQUFDLE1BQU0sRUFBRSxRQUFRLEVBQUU7UUFDOUIsS0FBSyxFQUFFLElBQUksQ0FBQyxPQUFPO0tBQ3BCLENBQUMsQ0FBQztJQUVILE9BQU87SUFDUCxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFFWixNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsZUFBZSxDQUFDO1FBQzVCLFNBQVMsRUFBRTtZQUNULFlBQVksRUFBRTtnQkFDWixJQUFJLEVBQUUsZ0JBQWdCO2dCQUN0QixVQUFVLEVBQUU7b0JBQ1Ysd0JBQXdCLEVBQUU7d0JBQ3hCLFNBQVMsRUFBRTs0QkFDVDtnQ0FDRSxNQUFNLEVBQUUsZ0JBQWdCO2dDQUN4QixNQUFNLEVBQUUsT0FBTztnQ0FDZixTQUFTLEVBQUUsRUFBRSxPQUFPLEVBQUUsa0JBQWtCLEVBQUU7NkJBQzNDO3lCQUNGO3dCQUNELE9BQU8sRUFBRSxZQUFZO3FCQUN0QjtpQkFDRjthQUNGO1NBQ0Y7UUFDRCxPQUFPLEVBQUU7WUFDUCw0Q0FBNEMsRUFBRTtnQkFDNUMsS0FBSyxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsY0FBYyxFQUFFLEtBQUssQ0FBRSxFQUFFO2dCQUNsRCxNQUFNLEVBQUU7b0JBQ04sSUFBSSxFQUFFLG9EQUFvRDtpQkFDM0Q7YUFDRjtTQUNGO0tBQ0YsQ0FDQSxDQUFDO0FBQ0osQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsaUZBQWlGLEVBQUUsR0FBRyxFQUFFO0lBQzNGLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUM1QyxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBRXpDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7UUFDVixJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN2QyxVQUFVLEVBQUU7Z0JBQ1YsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsc0JBQXNCLEVBQUU7b0JBQy9DLFVBQVUsRUFBRTt3QkFDVixZQUFZLEVBQUU7NEJBQ1osU0FBUyxFQUFFLE9BQU87eUJBQ25CO3FCQUNGO2lCQUNGLENBQUM7Z0JBQ0YsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsd0JBQXdCLEVBQUU7b0JBQ2pELFVBQVUsRUFBRTt3QkFDVixZQUFZLEVBQUU7NEJBQ1osU0FBUyxFQUFFLE9BQU87eUJBQ25CO3FCQUNGO2lCQUNGLENBQUM7YUFDSDtTQUNGLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLG1FQUFtRSxDQUFDLENBQUM7QUFDbEYsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsd0VBQXdFLEVBQUUsR0FBRyxFQUFFO0lBQ2xGLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxDQUFDLFNBQVMsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUM1QyxNQUFNLElBQUksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO0lBRXpDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1FBQ3ZDLFVBQVUsRUFBRTtZQUNWLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHNCQUFzQixDQUFDO1lBQ2hELElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHdCQUF3QixDQUFDO1NBQ25EO0tBQ0YsQ0FBQyxDQUFDLENBQUM7SUFFSixJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztRQUN2QyxVQUFVLEVBQUU7WUFDVixJQUFJLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQyxzQkFBc0IsRUFBRTtnQkFDL0MsVUFBVSxFQUFFO29CQUNWLFlBQVksRUFBRSxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUU7aUJBQ3JDO2FBQ0YsQ0FBQztZQUNGLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLHdCQUF3QixFQUFFO2dCQUNqRCxVQUFVLEVBQUU7b0JBQ1YsWUFBWSxFQUFFLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRTtpQkFDckM7YUFDRixDQUFDO1NBQ0g7S0FDRixDQUFDLENBQUMsQ0FBQztBQUNOLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICdAYXdzLWNkay9hc3NlcnQvamVzdCc7XG5pbXBvcnQgeyBBcHAsIENmbk91dHB1dCwgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICcuLi9saWInO1xuXG50ZXN0KCd1c2Ugb2YgY3Jvc3Mtc3RhY2sgcm9sZSByZWZlcmVuY2UgZG9lcyBub3QgbGVhZCB0byBVUkxTdWZmaXggYmVpbmcgZXhwb3J0ZWQnLCAoKSA9PiB7XG4gIC8vIEdJVkVOXG4gIGNvbnN0IGFwcCA9IG5ldyBBcHAoKTtcbiAgY29uc3QgZmlyc3QgPSBuZXcgU3RhY2soYXBwLCAnRmlyc3QnKTtcbiAgY29uc3Qgc2Vjb25kID0gbmV3IFN0YWNrKGFwcCwgJ1NlY29uZCcpO1xuXG4gIC8vIFdIRU5cbiAgY29uc3Qgcm9sZSA9IG5ldyBpYW0uUm9sZShmaXJzdCwgJ1JvbGUnLCB7XG4gICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ3MzLmFtYXpvbmF3cy5jb20nKSxcbiAgfSk7XG5cbiAgbmV3IENmbk91dHB1dChzZWNvbmQsICdPdXRwdXQnLCB7XG4gICAgdmFsdWU6IHJvbGUucm9sZUFybixcbiAgfSk7XG5cbiAgLy8gVEhFTlxuICBhcHAuc3ludGgoKTtcblxuICBleHBlY3QoZmlyc3QpLnRvTWF0Y2hUZW1wbGF0ZSh7XG4gICAgUmVzb3VyY2VzOiB7XG4gICAgICBSb2xlMUFCQ0M1RjA6IHtcbiAgICAgICAgVHlwZTogJ0FXUzo6SUFNOjpSb2xlJyxcbiAgICAgICAgUHJvcGVydGllczoge1xuICAgICAgICAgIEFzc3VtZVJvbGVQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICBBY3Rpb246ICdzdHM6QXNzdW1lUm9sZScsXG4gICAgICAgICAgICAgICAgRWZmZWN0OiAnQWxsb3cnLFxuICAgICAgICAgICAgICAgIFByaW5jaXBhbDogeyBTZXJ2aWNlOiAnczMuYW1hem9uYXdzLmNvbScgfSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSxcbiAgICBPdXRwdXRzOiB7XG4gICAgICBFeHBvcnRzT3V0cHV0Rm5HZXRBdHRSb2xlMUFCQ0M1RjBBcm5CNEMwQjczRToge1xuICAgICAgICBWYWx1ZTogeyAnRm46OkdldEF0dCc6IFsgJ1JvbGUxQUJDQzVGMCcsICdBcm4nIF0gfSxcbiAgICAgICAgRXhwb3J0OiB7XG4gICAgICAgICAgTmFtZTogJ0ZpcnN0OkV4cG9ydHNPdXRwdXRGbkdldEF0dFJvbGUxQUJDQzVGMEFybkI0QzBCNzNFJyxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgfSxcbiAgfSxcbiAgKTtcbn0pO1xuXG50ZXN0KCdjYW5ub3QgaGF2ZSBtdWx0aXBsZSBwcmluY2lwYWxzIHdpdGggZGlmZmVyZW50IGNvbmRpdGlvbnMgaW4gdGhlIHNhbWUgc3RhdGVtZW50JywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjayh1bmRlZmluZWQsICdGaXJzdCcpO1xuICBjb25zdCB1c2VyID0gbmV3IGlhbS5Vc2VyKHN0YWNrLCAnVXNlcicpO1xuXG4gIGV4cGVjdCgoKSA9PiB7XG4gICAgdXNlci5hZGRUb1BvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICBwcmluY2lwYWxzOiBbXG4gICAgICAgIG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbXlTZXJ2aWNlLmFtYXpvbi5jb20nLCB7XG4gICAgICAgICAgY29uZGl0aW9uczoge1xuICAgICAgICAgICAgU3RyaW5nRXF1YWxzOiB7XG4gICAgICAgICAgICAgIGhhaXJDb2xvcjogJ2Jsb25kJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSksXG4gICAgICAgIG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgneW91cnNlcnZpY2UuYW1hem9uLmNvbScsIHtcbiAgICAgICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgICAgICBTdHJpbmdFcXVhbHM6IHtcbiAgICAgICAgICAgICAgaGFpckNvbG9yOiAnYmxhY2snLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KSxcbiAgICAgIF0sXG4gICAgfSkpO1xuICB9KS50b1Rocm93KC9BbGwgcHJpbmNpcGFscyBpbiBhIFBvbGljeVN0YXRlbWVudCBtdXN0IGhhdmUgdGhlIHNhbWUgQ29uZGl0aW9ucy8pO1xufSk7XG5cbnRlc3QoJ2NhbiBoYXZlIG11bHRpcGxlIHByaW5jaXBhbHMgdGhlIHNhbWUgY29uZGl0aW9ucyBpbiB0aGUgc2FtZSBzdGF0ZW1lbnQnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKHVuZGVmaW5lZCwgJ0ZpcnN0Jyk7XG4gIGNvbnN0IHVzZXIgPSBuZXcgaWFtLlVzZXIoc3RhY2ssICdVc2VyJyk7XG5cbiAgdXNlci5hZGRUb1BvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgcHJpbmNpcGFsczogW1xuICAgICAgbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdteVNlcnZpY2UuYW1hem9uLmNvbScpLFxuICAgICAgbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCd5b3Vyc2VydmljZS5hbWF6b24uY29tJyksXG4gICAgXSxcbiAgfSkpO1xuXG4gIHVzZXIuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgIHByaW5jaXBhbHM6IFtcbiAgICAgIG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgnbXlTZXJ2aWNlLmFtYXpvbi5jb20nLCB7XG4gICAgICAgIGNvbmRpdGlvbnM6IHtcbiAgICAgICAgICBTdHJpbmdFcXVhbHM6IHsgaGFpckNvbG9yOiAnYmxvbmQnIH0sXG4gICAgICAgIH0sXG4gICAgICB9KSxcbiAgICAgIG5ldyBpYW0uU2VydmljZVByaW5jaXBhbCgneW91cnNlcnZpY2UuYW1hem9uLmNvbScsIHtcbiAgICAgICAgY29uZGl0aW9uczoge1xuICAgICAgICAgIFN0cmluZ0VxdWFsczogeyBoYWlyQ29sb3I6ICdibG9uZCcgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pLFxuICAgIF0sXG4gIH0pKTtcbn0pOyJdfQ==