#!/usr/bin/python
# -*- coding: utf-8 -*-

# Hive DigitalOcean API
# Copyright (c) 2008-2020 Hive Solutions Lda.
#
# This file is part of Hive DigitalOcean API.
#
# Hive DigitalOcean API is free software: you can redistribute it and/or modify
# it under the terms of the Apache License as published by the Apache
# Foundation, either version 2.0 of the License, or (at your option) any
# later version.
#
# Hive DigitalOcean API is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# Apache License for more details.
#
# You should have received a copy of the Apache License along with
# Hive DigitalOcean API. If not, see <http://www.apache.org/licenses/>.

__author__ = "João Magalhães <joamag@hive.pt>"
""" The author(s) of the module """

__copyright__ = "Copyright (c) 2008-2020 Hive Solutions Lda."
""" The copyright for the module """

__license__ = "Apache License, Version 2.0"
""" The license for the module """

import appier

from . import action
from . import droplet

BASE_URL = "https://API.digitalocean.com/v2/"
""" The default base URL to be used when no other
base URL value is provided to the constructor """

AUTH_URL = "https://cloud.digitalocean.com/v1/"
""" The default auth URL to be used as the basis
for the authentication part of the API """

CLIENT_ID = None
""" The default value to be used for the client id
in case no client id is provided to the API client """

CLIENT_SECRET = None
""" The secret value to be used for situations where
no client secret has been provided to the client """

REDIRECT_URL = "http://localhost:8080/oauth"
""" The redirect URL used as default (fallback) value
in case none is provided to the API (client) """

SCOPE = ("read", "write")
""" The list of permissions to be used to create the
scope string for the OAuth value """


class API(appier.OAuth2API, action.ActionAPI, droplet.DropletAPI):

    def __init__(self, *args, **kwargs):
        appier.OAuth2API.__init__(self, *args, **kwargs)
        self.client_id = appier.conf("DO_ID", CLIENT_ID)
        self.client_secret = appier.conf("DO_SECRET", CLIENT_SECRET)
        self.redirect_url = appier.conf("DO_REDIRECT_URL", REDIRECT_URL)
        self.base_url = kwargs.get("base_url", BASE_URL)
        self.auth_url = kwargs.get("auth_url", AUTH_URL)
        self.client_id = kwargs.get("client_id", self.client_id)
        self.client_secret = kwargs.get("client_secret", self.client_secret)
        self.redirect_url = kwargs.get("redirect_url", self.redirect_url)
        self.scope = kwargs.get("scope", SCOPE)
        self.access_token = kwargs.get("access_token", None)
        self.refresh_token = kwargs.get("refresh_token", None)
        self.email = kwargs.get("email", None)

    def oauth_authorize(self, state=None):
        url = self.auth_url + "oauth/authorize"
        values = dict(
            client_id=self.client_id,
            redirect_uri=self.redirect_url,
            response_type="code",
            scope=" ".join(self.scope),
        )
        if state:
            values["state"] = state
        data = appier.legacy.urlencode(values)
        url = url + "?" + data
        return url

    def oauth_access(self, code):
        url = self.auth_url + "oauth/token"
        contents = self.post(
            url,
            token=False,
            client_id=self.client_id,
            client_secret=self.client_secret,
            grant_type="authorization_code",
            redirect_uri=self.redirect_url,
            code=code,
        )
        self.access_token = contents["access_token"]
        self.refresh_token = contents["refresh_token"]
        self.email = contents["info"]["email"]
        self.trigger("access_token", self.access_token)
        return self.access_token

    def oauth_refresh(self, refresh_token):
        url = self.auth_url + "oauth/token"
        contents = self.post(
            url,
            token=False,
            client_id=self.client_id,
            client_secret=self.client_secret,
            grant_type="refresh_token",
            redirect_uri=self.redirect_url,
            refresh_token=refresh_token,
        )
        self.access_token = contents["access_token"]
        self.refresh_token = contents["refresh_token"]
        self.trigger("access_token", self.access_token)
        self.trigger("refresh_token", self.refresh_token)
        return self.access_token
