from .action import Action
import os
import json
from . import utils
from .logger import logger

class CacheAction(Action):
    """
    ####################################################################################################################
    Cache Action
    ####################################################################################################################
    Currently, the following actions are supported for Cache:
    1. find/search
    2. show
    3. list
    4. remove(rm)

    """

    def __init__(self, parent=None):
        #super().__init__(parent)
        self.parent = parent
        self.__dict__.update(vars(parent))
    
    def search(self, i):
        """
    ####################################################################################################################
    Target: Cache
    Action: Find (Alias: Search)
    ####################################################################################################################

    The `find` (or `search`) action retrieves the path of cache generated by a particular script. 

    Syntax:

    mlc find cache --tags=<list_of_tag_used_to_run_the_particular_script> 

    Example Command:

    mlc find cache --tags=get,dataset,igbh

        """
        i['target_name'] = "cache"
        #logger.debug(f"Searching for cache with input: {i}")
        return self.parent.search(i)

    find = search

    def rm(self, i):
        """
    ####################################################################################################################
    Target: Cache
    Action: Remove(rm)
    ####################################################################################################################

    The `rm` action removes one or more caches generated while running scripts through MLCFlow.

    Syntax:

    mlc rm cache --tags=<list_of_tag_used_to_run_the_particular_script> 

    Options:
        1. `-f`: Force removes caches without confirmation. Without `-f`, the user will be prompted for confirmation before deletion.

    To remove all generated caches, use:
        
    mlc rm cache

    Example Command:

    mlc rm cache --tags=get,dataset,igbh

        """
        i['target_name'] = "cache"
        #logger.debug(f"Removing cache with input: {i}")
        return self.parent.rm(i)

    def show(self, run_args):
        """
    ####################################################################################################################
    Target: Cache
    Action: Show
    ####################################################################################################################

    Retrieves the path and metadata of caches generated while running scripts through MLCFlow.
        
    Syntax:
        
    mlc show cache --tags=<list_of_tags_used_while_running_script>

    Example Command:

    mlc show cache --tags=get,dataset,igbh

    Note:
    - The `find` action is a subset of `show`. It retrieves only the path of the searched cache.

        """
        self.action_type = "cache"
        res = self.search(run_args)
        logger.info(f"Showing cache with tags: {run_args.get('tags')}")
        cached_meta_keys_to_show = ["uid", "tags", "dependent_cached_path", "associated_script_item"]
        cached_state_keys_to_show = ["new_env", "new_state", "version"]
        for item in res['list']:
            print(f"""Location: {item.path}:
Cache Meta:""")
            for key in cached_meta_keys_to_show:
                if key in item.meta:
                    print(f"""    {key}: {item.meta[key]}""")
            print("""Cached State:""")
            cached_state_meta_file = os.path.join(item.path, "mlc-cached-state.json")
            if not os.path.exists(cached_state_meta_file):
                continue
            try:
                # Load and parse the JSON file containing the cached state
                with open(cached_state_meta_file, 'r') as file:
                    meta = json.load(file)
                    for key in cached_state_keys_to_show:
                        if key in meta:
                            print(f"""    {key}:""", end="")
                            if meta[key] and isinstance(meta[key], dict):
                                print("")
                                utils.printd(meta[key], yaml=False, sort_keys=True, begin_spaces=8)
                            else:
                                print(f""" {meta[key]}""")
            except json.JSONDecodeError as e:
                logger.error(f"Error decoding JSON: {e}")
            print("......................................................")
            print("")
            
        return {'return': 0}

    def list(self, args):
        """
    ####################################################################################################################
    Target: Cache
    Action: List
    ####################################################################################################################

    Lists all cached items along with their paths.

    Example Command:

    mlc list cache

        """
        self.action_type = "cache"
        run_args = {"fetch_all": True}  # to fetch the details of all the caches generated
        
        res = self.search(run_args)
        if res['return'] > 0:
            return res
        
        logger.info(f"Listing all the caches and their paths")
        print("......................................................")
        for item in res['list']:
            print(f"tags: {item.meta['tags'] if item.meta.get('tags') else 'None'}")
            print(f"Location: {item.path}")
            print("......................................................")

        return {'return': 0}
