import { IResource, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { IGroup } from './group';
import { PolicyDocument } from './policy-document';
import { PolicyStatement } from './policy-statement';
import { IRole } from './role';
import { IUser } from './user';
/**
 * Represents an IAM Policy.
 *
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/access_policies_manage.html
 * @stability stable
 */
export interface IPolicy extends IResource {
    /**
     * The name of this policy.
     *
     * @stability stable
     * @attribute true
     */
    readonly policyName: string;
}
/**
 * Properties for defining an IAM inline policy document.
 *
 * @stability stable
 */
export interface PolicyProps {
    /**
     * The name of the policy.
     *
     * If you specify multiple policies for an entity,
     * specify unique names. For example, if you specify a list of policies for
     * an IAM role, each policy must have a unique name.
     *
     * @default - Uses the logical ID of the policy resource, which is ensured
     * to be unique within the stack.
     * @stability stable
     */
    readonly policyName?: string;
    /**
     * Users to attach this policy to.
     *
     * You can also use `attachToUser(user)` to attach this policy to a user.
     *
     * @default - No users.
     * @stability stable
     */
    readonly users?: IUser[];
    /**
     * Roles to attach this policy to.
     *
     * You can also use `attachToRole(role)` to attach this policy to a role.
     *
     * @default - No roles.
     * @stability stable
     */
    readonly roles?: IRole[];
    /**
     * Groups to attach this policy to.
     *
     * You can also use `attachToGroup(group)` to attach this policy to a group.
     *
     * @default - No groups.
     * @stability stable
     */
    readonly groups?: IGroup[];
    /**
     * Initial set of permissions to add to this policy document.
     *
     * You can also use `addStatements(...statement)` to add permissions later.
     *
     * @default - No statements.
     * @stability stable
     */
    readonly statements?: PolicyStatement[];
    /**
     * Force creation of an `AWS::IAM::Policy`.
     *
     * Unless set to `true`, this `Policy` construct will not materialize to an
     * `AWS::IAM::Policy` CloudFormation resource in case it would have no effect
     * (for example, if it remains unattached to an IAM identity or if it has no
     * statements). This is generally desired behavior, since it prevents
     * creating invalid--and hence undeployable--CloudFormation templates.
     *
     * In cases where you know the policy must be created and it is actually
     * an error if no statements have been added to it, you can se this to `true`.
     *
     * @default false
     * @stability stable
     */
    readonly force?: boolean;
    /**
     * Initial PolicyDocument to use for this Policy.
     *
     * If omited, any
     * `PolicyStatement` provided in the `statements` property will be applied
     * against the empty default `PolicyDocument`.
     *
     * @default - An empty policy.
     * @stability stable
     */
    readonly document?: PolicyDocument;
}
/**
 * The AWS::IAM::Policy resource associates an IAM policy with IAM users, roles, or groups.
 *
 * For more information about IAM policies, see [Overview of IAM
 * Policies](http://docs.aws.amazon.com/IAM/latest/UserGuide/policies_overview.html)
 * in the IAM User Guide guide.
 *
 * @stability stable
 */
export declare class Policy extends Resource implements IPolicy {
    /**
     * Import a policy in this app based on its name.
     *
     * @stability stable
     */
    static fromPolicyName(scope: Construct, id: string, policyName: string): IPolicy;
    /**
     * The policy document.
     *
     * @stability stable
     */
    readonly document: PolicyDocument;
    private readonly _policyName;
    private readonly roles;
    private readonly users;
    private readonly groups;
    private readonly force;
    private referenceTaken;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: PolicyProps);
    /**
     * Adds a statement to the policy document.
     *
     * @stability stable
     */
    addStatements(...statement: PolicyStatement[]): void;
    /**
     * Attaches this policy to a user.
     *
     * @stability stable
     */
    attachToUser(user: IUser): void;
    /**
     * Attaches this policy to a role.
     *
     * @stability stable
     */
    attachToRole(role: IRole): void;
    /**
     * Attaches this policy to a group.
     *
     * @stability stable
     */
    attachToGroup(group: IGroup): void;
    /**
     * The name of this policy.
     *
     * @stability stable
     * @attribute true
     */
    get policyName(): string;
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    protected validate(): string[];
    /**
     * Whether the policy resource has been attached to any identity
     */
    private get isAttached();
}
