"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const core_1 = require("@aws-cdk/core");
/**
 * A PolicyDocument is a collection of statements
 */
class PolicyDocument {
    constructor(props = {}) {
        this.statements = new Array();
        this.creationStack = core_1.captureStackTrace();
        this.autoAssignSids = !!props.assignSids;
        this.addStatements(...props.statements || []);
    }
    resolve(context) {
        context.registerPostProcessor(new RemoveDuplicateStatements(this.autoAssignSids));
        return this.render();
    }
    get isEmpty() {
        return this.statements.length === 0;
    }
    /**
     * The number of statements already added to this policy.
     * Can be used, for example, to generate uniuqe "sid"s within the policy.
     */
    get statementCount() {
        return this.statements.length;
    }
    /**
     * Adds a statement to the policy document.
     *
     * @param statement the statement to add.
     */
    addStatements(...statement) {
        this.statements.push(...statement);
    }
    /**
     * Encode the policy document as a string
     */
    toString() {
        return cdk.Token.asString(this, {
            displayHint: 'PolicyDocument'
        });
    }
    /**
     * JSON-ify the document
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return this.render();
    }
    render() {
        if (this.isEmpty) {
            return undefined;
        }
        const doc = {
            Statement: this.statements.map(s => s.toStatementJson()),
            Version: '2012-10-17'
        };
        return doc;
    }
}
exports.PolicyDocument = PolicyDocument;
/**
 * Removes duplicate statements and assign Sids if necessary
 */
class RemoveDuplicateStatements {
    constructor(autoAssignSids) {
        this.autoAssignSids = autoAssignSids;
    }
    postProcess(input, _context) {
        if (!input || !input.Statement) {
            return input;
        }
        const jsonStatements = new Set();
        const uniqueStatements = [];
        for (const statement of input.Statement) {
            const jsonStatement = JSON.stringify(statement);
            if (!jsonStatements.has(jsonStatement)) {
                uniqueStatements.push(statement);
                jsonStatements.add(jsonStatement);
            }
        }
        // assign unique SIDs (the statement index) if `autoAssignSids` is enabled
        const statements = uniqueStatements.map((s, i) => {
            if (this.autoAssignSids && !s.Sid) {
                s.Sid = i.toString();
            }
            return s;
        });
        return {
            ...input,
            Statement: statements
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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