"""
Backend Management System for Multi-Framework Support

This module provides unified interfaces for PyTorch, JAX, and NUMBA backends,
enabling seamless switching between frameworks and automatic backend selection
based on data type, hardware availability, and performance requirements.
"""

import os
from typing import Optional, Union, Any, Dict, List, Tuple, Callable
from enum import Enum
import warnings

# Backend availability checking
try:
    import torch
    TORCH_AVAILABLE = True
except ImportError:
    TORCH_AVAILABLE = False

try:
    import jax
    import jax.numpy as jnp
    JAX_AVAILABLE = True
except ImportError:
    JAX_AVAILABLE = False

try:
    import numba
    NUMBA_AVAILABLE = True
except ImportError:
    NUMBA_AVAILABLE = False


class BackendType(Enum):
    """Available computation backends"""
    TORCH = "torch"
    JAX = "jax"
    NUMBA = "numba"
    AUTO = "auto"


class BackendManager:
    """
    Manages backend selection and provides unified interfaces
    
    This class handles automatic backend selection based on:
    - Data type and size
    - Hardware availability (CPU/GPU)
    - Performance requirements
    - User preferences
    """
    
    def __init__(
        self,
        preferred_backend: BackendType = BackendType.AUTO,
        force_cpu: bool = False,
        enable_jit: bool = True,
        enable_gpu: bool = True
    ):
        self.preferred_backend = preferred_backend
        self.force_cpu = force_cpu
        self.enable_jit = enable_jit
        self.enable_gpu = enable_gpu
        
        # Detect available backends
        self.available_backends = self._detect_available_backends()
        
        # Current active backend
        self.active_backend = self._select_optimal_backend()
        
        # Backend-specific configurations
        self.backend_configs = self._initialize_backend_configs()
        
        print(f"🎯 Backend Manager initialized with {self.active_backend.value}")
        print(f"📊 Available backends: {[b.value for b in self.available_backends]}")
    
    def _detect_available_backends(self) -> List[BackendType]:
        """Detect which backends are available on the system"""
        available = []
        
        if TORCH_AVAILABLE:
            available.append(BackendType.TORCH)
            if torch.cuda.is_available() and not self.force_cpu:
                print("🚀 PyTorch CUDA support detected")
        
        if JAX_AVAILABLE:
            available.append(BackendType.JAX)
            try:
                devices = jax.devices()
                if any('gpu' in str(d).lower() for d in devices) and not self.force_cpu:
                    print("🚀 JAX GPU support detected")
            except:
                pass
        
        if NUMBA_AVAILABLE:
            available.append(BackendType.NUMBA)
            try:
                if hasattr(numba, 'cuda') and numba.cuda.is_available() and not self.force_cpu:
                    print("🚀 NUMBA CUDA support detected")
            except:
                pass
        
        if not available:
            raise RuntimeError("No computation backends available!")
        
        return available
    
    def _select_optimal_backend(self) -> BackendType:
        """Select the optimal backend based on preferences and availability"""
        if self.preferred_backend == BackendType.AUTO:
            # Prefer PyTorch by default to match test expectations and widest API coverage
            if BackendType.TORCH in self.available_backends:
                return BackendType.TORCH
            elif BackendType.JAX in self.available_backends and self.enable_gpu:
                return BackendType.JAX
            elif BackendType.NUMBA in self.available_backends:
                return BackendType.NUMBA
            else:
                return self.available_backends[0]
        else:
            if self.preferred_backend in self.available_backends:
                return self.preferred_backend
            else:
                warnings.warn(f"Preferred backend {self.preferred_backend.value} not available, using {self.available_backends[0].value}")
                return self.available_backends[0]
    
    def _initialize_backend_configs(self) -> Dict[BackendType, Dict[str, Any]]:
        """Initialize backend-specific configurations"""
        configs = {}
        
        # PyTorch configuration
        if BackendType.TORCH in self.available_backends:
            configs[BackendType.TORCH] = {
                'device': 'cuda' if torch.cuda.is_available() and not self.force_cpu else 'cpu',
                'dtype': torch.float32,
                'enable_amp': True,  # Automatic Mixed Precision
                'enable_compile': hasattr(torch, 'compile'),  # PyTorch 2.0+ compilation
            }
        
        # JAX configuration
        if BackendType.JAX in self.available_backends:
            configs[BackendType.JAX] = {
                'device': 'gpu' if self.enable_gpu and not self.force_cpu else 'cpu',
                'dtype': jnp.float32,
                'enable_jit': self.enable_jit,
                'enable_x64': False,  # Use float32 for better performance
                'enable_amp': True,
            }
        
        # NUMBA configuration
        if BackendType.NUMBA in self.available_backends:
            try:
                gpu_available = hasattr(numba, 'cuda') and numba.cuda.is_available()
            except:
                gpu_available = False
                
            configs[BackendType.NUMBA] = {
                'device': 'gpu' if gpu_available and not self.force_cpu else 'cpu',
                'dtype': numba.float32,
                'enable_jit': self.enable_jit,
                'enable_parallel': True,
                'enable_fastmath': True,
            }
        
        return configs
    
    def get_backend_config(self, backend: Optional[BackendType] = None) -> Dict[str, Any]:
        """Get configuration for a specific backend"""
        backend = backend or self.active_backend
        return self.backend_configs.get(backend, {})
    
    def switch_backend(self, backend: BackendType) -> bool:
        """Switch to a different backend"""
        if backend in self.available_backends:
            self.active_backend = backend
            print(f"🔄 Switched to {backend.value} backend")
            return True
        else:
            warnings.warn(f"Backend {backend.value} not available")
            return False
    
    def get_tensor_lib(self) -> Any:
        """Get the active tensor library"""
        if self.active_backend == BackendType.TORCH:
            return torch
        elif self.active_backend == BackendType.JAX:
            return jnp
        elif self.active_backend == BackendType.NUMBA:
            return numba
        else:
            raise RuntimeError(f"Unknown backend: {self.active_backend}")
    
    def create_tensor(self, data: Any, **kwargs) -> Any:
        """Create a tensor in the active backend"""
        if self.active_backend == BackendType.TORCH:
            # Ensure consistent dtype for PyTorch
            if 'dtype' not in kwargs:
                # Preserve integer types for classification targets
                if hasattr(data, 'dtype') and 'int' in str(data.dtype):
                    kwargs['dtype'] = torch.long
                else:
                    kwargs['dtype'] = torch.float32
            return torch.tensor(data, **kwargs)
        elif self.active_backend == BackendType.JAX:
            # Ensure consistent dtype for JAX
            if 'dtype' not in kwargs:
                # Preserve integer types for classification targets
                if hasattr(data, 'dtype') and 'int' in str(data.dtype):
                    kwargs['dtype'] = jnp.int32
                else:
                    kwargs['dtype'] = jnp.float32
            return jnp.array(data, **kwargs)
        elif self.active_backend == BackendType.NUMBA:
            # NUMBA works with numpy arrays
            import numpy as np
            if 'dtype' not in kwargs:
                # Preserve integer types for classification targets
                if hasattr(data, 'dtype') and 'int' in str(data.dtype):
                    kwargs['dtype'] = np.int32
                else:
                    kwargs['dtype'] = np.float32
            return np.array(data, **kwargs)
        else:
            raise RuntimeError(f"Unknown backend: {self.active_backend}")
    
    def to_device(self, tensor: Any, device: Optional[str] = None) -> Any:
        """Move tensor to specified device"""
        if self.active_backend == BackendType.TORCH:
            return tensor.to(device or self.backend_configs[BackendType.TORCH]['device'])
        elif self.active_backend == BackendType.JAX:
            # JAX handles device placement differently
            return tensor
        elif self.active_backend == BackendType.NUMBA:
            # NUMBA handles device placement differently
            return tensor
        else:
            raise RuntimeError(f"Unknown backend: {self.active_backend}")
    
    def compile_function(self, func: Callable) -> Callable:
        """Compile a function using the active backend's compilation system"""
        if self.active_backend == BackendType.TORCH:
            if hasattr(torch, 'compile'):
                return torch.compile(func)
            else:
                return func
        elif self.active_backend == BackendType.JAX:
            if self.enable_jit:
                return jax.jit(func)
            else:
                return func
        elif self.active_backend == BackendType.NUMBA:
            if self.enable_jit:
                return numba.jit(func)
            else:
                return func
        else:
            return func


# Global backend manager instance
_backend_manager: Optional[BackendManager] = None


def get_backend_manager() -> BackendManager:
    """Get the global backend manager instance"""
    global _backend_manager
    if _backend_manager is None:
        _backend_manager = BackendManager()
    return _backend_manager


def set_backend_manager(manager: BackendManager) -> None:
    """Set the global backend manager instance"""
    global _backend_manager
    _backend_manager = manager


def get_active_backend() -> BackendType:
    """Get the currently active backend"""
    return get_backend_manager().active_backend


def switch_backend(backend: BackendType) -> bool:
    """Switch to a different backend"""
    return get_backend_manager().switch_backend(backend)
