from typing import Any, List, Optional, Tuple, Type

import numpy as np
import pandas as pd
from scipy.stats import qmc

from optimization_problem_inspector import models


class OPISampler:
    """Base class providing interfaces to implement to support
    integartion of a sampling method into the Optimization problem inspector
    framework.
    """

    # TODO: make it an abstract base class that enforces interfaces
    def __init__(
        self,
        real_variables: List[Tuple[str, float, float, float]],
        integer_variables: List[Tuple[str, int, int, int]],
        categorical_variables: List[Tuple[str, List[Any]]],
        random_seed: Optional[int] = None,
        *args,
        **kwargs,
    ) -> None:
        self.real_variables = real_variables
        self.integer_variables = integer_variables
        self.categorical_variables = categorical_variables
        self.random_seed = random_seed
        raise NotImplementedError

    def sample(self, N: int, **kwargs) -> pd.DataFrame:
        """Generate a sample of variables of size N. Pass on the kwargs
        to a particular OPISampler implementation.

        Args:
            N (int): Size of the sample.

        Returns:
            pd.DataFrame: Dataframe with N samples generated by the
                chosen OPISampler implementation.
        """
        raise NotImplementedError

    def reset_random_seed(self):
        """Reset the random seed to the one specified in the __init__."""
        if self.random_seed is not None:
            np.random.seed(self.random_seed)


class RandomSampler(OPISampler):
    """Random sampling of variables that works with any type of variable."""

    def __init__(
        self,
        real_variables: List[Tuple[str, float, float, float]],
        integer_variables: List[Tuple[str, int, int, int]],
        categorical_variables: List[Tuple[str, List[Any]]],
        random_seed: Optional[int] = None,
        *args,
        **kwargs,
    ) -> None:
        """Random sampling of variables. Works with any type of variable.
        In case of step-less real variable, np.random.random is used, in all
        other cases (real variable with step, integer, or categorical) all
        options are enumerated and np.random.randint is used to sample
        the data.

        Args:
            real_variables (List[Tuple[str, float, float, float]]):
                List of real variables (name, start, end, step).
            integer_variables (List[Tuple[str, int, int, int]]):
                List of integer variables (name, start, end, step).
            categorical_variables (List[Tuple[str, List[Any]]]):
                List of categorical variables (name, values).
            random_seed (Optional[int], optional): The random seed to set
                before sampling. Defaults to None.
        """
        self.real_variables = (
            real_variables if real_variables is not None else []
        )
        self.integer_variables = (
            integer_variables if integer_variables is not None else []
        )
        self.categorical_variables = (
            categorical_variables if categorical_variables is not None else []
        )
        self.random_seed = random_seed

    def sample(self, N: int, **kwargs) -> pd.DataFrame:
        self.reset_random_seed()
        vals = {}

        for n, lb, ub, step in self.real_variables:
            if step is None:
                vals[n] = np.random.random(N) * (ub - lb) + lb
            else:
                ub_modulo_step = ub % step
                step_divides_ub = np.isclose(
                    ub_modulo_step, step
                ) or np.isclose(ub_modulo_step, 0)
                values = np.arange(lb, ub + int(step_divides_ub) * step, step)
                positions = np.random.randint(
                    low=0, high=values.shape[0], size=N
                )
                vals[n] = np.clip(values[positions], lb, ub)

        for n, lb, ub, step in self.integer_variables:
            if step is None:
                vals[n] = np.random.randint(low=lb, high=ub, size=N)
            else:
                ub_modulo_step = ub % step
                step_divides_ub = np.isclose(
                    ub_modulo_step, step
                ) or np.isclose(ub_modulo_step, 0)
                values = np.arange(lb, ub + int(step_divides_ub) * step, step)
                positions = np.random.randint(
                    low=0, high=values.shape[0], size=N
                )
                vals[n] = np.clip(values[positions], lb, ub)

        for n, cat_vals in self.categorical_variables:
            positions = np.random.randint(low=0, high=len(cat_vals), size=N)
            vals[n] = np.array(cat_vals)[positions]

        return pd.DataFrame(vals)


class SobolSampler(OPISampler):
    """Sampling method for real-type variables."""

    def __init__(
        self,
        real_variables: List[Tuple[str, float, float, float]],
        integer_variables: List[Tuple[str, int, int, int]],
        random_seed: Optional[int] = None,
        *args,
        **kwargs,
    ) -> None:
        """Use sampling from scipy.stats.qmc.Sobol to generate real-type
        samples.

        Args:
            real_variables (List[Tuple[str, float, float, float]]):
                List of real variables (name, start, end, step).
            integer_variables (List[Tuple[str, int, int, int]]):
                List of integer variables (name, start, end, step).
            random_seed (Optional[int], optional): The random seed to set
                before sampling. Defaults to None.
        """
        self.real_variables = (
            real_variables if real_variables is not None else []
        )
        self.integer_variables = (
            integer_variables if integer_variables is not None else []
        )

    def sample(self, N: int, **kwargs) -> pd.DataFrame:
        sampler = qmc.Sobol(
            d=len(self.real_variables) + len(self.integer_variables)
        )
        sample = sampler.random(n=N)

        res = pd.DataFrame(
            sample,
            columns=[
                x[0] for x in self.real_variables + self.integer_variables
            ],
        )

        for var in self.real_variables:
            if var[3] is None:
                res[var[0]] = res[var[0]] * (var[2] - var[1]) + var[1]
            else:
                res[var[0]] = (
                    round(
                        (
                            res[var[0]] * (var[2] - var[1] + var[3])
                            + var[1]
                            - var[3] / 2
                        )
                        / var[3]
                    )
                    * var[3]
                )

        for var in self.integer_variables:
            if var[3] is None:
                res[var[0]] = round(res[var[0]] * (var[2] - var[1]) + var[1])
            else:
                res[var[0]] = (
                    round(
                        (
                            res[var[0]] * (var[2] - var[1] + var[3])
                            + var[1]
                            - var[3] / 2
                        )
                        / var[3]
                    )
                    * var[3]
                )

        return res


class LHSSampler(OPISampler):
    def __init__(
        self,
        real_variables: List[Tuple[str, float, float]],
        integer_variables: List[Tuple[str, int, int, int]],
        random_seed: Optional[int] = None,
        *args,
        **kwargs,
    ) -> None:
        """Use sampling from scipy.stats.qmc.LatinHypercube to
        generate real-type samples.

        Args:
            real_variables (List[Tuple[str, float, float, float]]):
                List of real variables (name, start, end, step).
            integer_variables (List[Tuple[str, int, int, int]]):
                List of integer variables (name, start, end, step).
            random_seed (Optional[int], optional): The random seed to set
                before sampling. Defaults to None.
        """
        self.real_variables = (
            real_variables if real_variables is not None else []
        )
        self.integer_variables = (
            integer_variables if integer_variables is not None else []
        )
        self.random_seed = random_seed

    def sample(self, N: int, **kwargs) -> pd.DataFrame:
        self.reset_random_seed()
        sampler = qmc.LatinHypercube(
            d=len(self.real_variables) + len(self.integer_variables)
        )
        sample = sampler.random(n=N)

        res = pd.DataFrame(
            sample,
            columns=[
                x[0] for x in self.real_variables + self.integer_variables
            ],
        )

        for var in self.real_variables:
            if var[3] is None:
                res[var[0]] = res[var[0]] * (var[2] - var[1]) + var[1]
            else:
                res[var[0]] = (
                    round(
                        (
                            res[var[0]] * (var[2] - var[1] + var[3])
                            + var[1]
                            - var[3] / 2
                        )
                        / var[3]
                    )
                    * var[3]
                )

        for var in self.integer_variables:
            if var[3] is None:
                res[var[0]] = round(res[var[0]] * (var[2] - var[1]) + var[1])
            else:
                res[var[0]] = (
                    round(
                        (
                            res[var[0]] * (var[2] - var[1] + var[3])
                            + var[1]
                            - var[3] / 2
                        )
                        / var[3]
                    )
                    * var[3]
                )

        return res


def OPISamplerFactory(
    opi_sampler: Type[OPISampler],
    problem_spec: models.Spec,
    random_seed: Optional[int] = None,
    *args,
    **kwargs,
) -> OPISampler:
    """Helper factory to generate OPISampler instance, based on the
    input Optimization problem inspector specification `problem_spec`
    and `opi_sampler` class.

    Args:
        opi_sampler (Type[OPISampler]): The class of sampler to generate.
        problem_spec (models.Spec): Optimization problem inspector
            specification
        random_seed (Optional[int], optional): Random seed to use when
            sampling. Defaults to None.

    Raises:
        ValueError: Could not produce sampler instance.

    Returns:
        OPISampler: OPISampler instance adjusted to the problem and ready
            to generate samples of data.
    """
    real_variables = []
    int_variables = []
    cat_variables = []

    for parameter in problem_spec.dataSpec.parameters:
        if parameter.parameterGroup != models.ParameterGroup.VARIABLES:
            continue

        if isinstance(parameter, models.ParameterReal):
            real_variables.append(
                (
                    parameter.name,
                    parameter.lower_bound,
                    parameter.upper_bound,
                    parameter.step,
                )
            )
        elif isinstance(parameter, models.ParameterInt):
            int_variables.append(
                (
                    parameter.name,
                    parameter.lower_bound,
                    parameter.upper_bound,
                    parameter.step,
                )
            )
        elif isinstance(parameter, models.ParameterCategorical):
            cat_variables.append(
                (
                    parameter.name,
                    parameter.values,
                )
            )
        else:
            raise ValueError(
                f"Unknown parameter type {type(parameter).__name__}"
            )

    return opi_sampler(
        real_variables=real_variables,
        integer_variables=int_variables,
        categorical_variables=cat_variables,
        random_seed=random_seed,
        *args,
        **kwargs,
    )


OPI_SAMPLERS = [
    RandomSampler,
    SobolSampler,
    LHSSampler,
]
"""List of registered OPISampler classes.
"""
