"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BucketNotifications = void 0;
const cdk = require("@aws-cdk/core");
const destination_1 = require("../destination");
const notifications_resource_handler_1 = require("./notifications-resource-handler");
/**
 * A custom CloudFormation resource that updates bucket notifications for a
 * bucket. The reason we need it is because the AWS::S3::Bucket notification
 * configuration is defined on the bucket itself, which makes it impossible to
 * provision notifications at the same time as the target (since
 * PutBucketNotifications validates the targets).
 *
 * Since only a single BucketNotifications resource is allowed for each Bucket,
 * this construct is not exported in the public API of this module. Instead, it
 * is created just-in-time by `s3.Bucket.onEvent`, so a 1:1 relationship is
 * ensured.
 *
 * @see
 * https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-s3-bucket-notificationconfig.html
 */
class BucketNotifications extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaNotifications = new Array();
        this.queueNotifications = new Array();
        this.topicNotifications = new Array();
        this.bucket = props.bucket;
    }
    /**
     * Adds a notification subscription for this bucket.
     * If this is the first notification, a BucketNotification resource is added to the stack.
     *
     * @param event The type of event
     * @param target The target construct
     * @param filters A set of S3 key filters
     */
    addNotification(event, target, ...filters) {
        const resource = this.createResourceOnce();
        // resolve target. this also provides an opportunity for the target to e.g. update
        // policies to allow this notification to happen.
        const targetProps = target.bind(this, this.bucket);
        const commonConfig = {
            Events: [event],
            Filter: renderFilters(filters),
        };
        // if the target specifies any dependencies, add them to the custom resource.
        // for example, the SNS topic policy must be created /before/ the notification resource.
        // otherwise, S3 won't be able to confirm the subscription.
        if (targetProps.dependencies) {
            resource.node.addDependency(...targetProps.dependencies);
        }
        // based on the target type, add the the correct configurations array
        switch (targetProps.type) {
            case destination_1.BucketNotificationDestinationType.LAMBDA:
                this.lambdaNotifications.push({ ...commonConfig, LambdaFunctionArn: targetProps.arn });
                break;
            case destination_1.BucketNotificationDestinationType.QUEUE:
                this.queueNotifications.push({ ...commonConfig, QueueArn: targetProps.arn });
                break;
            case destination_1.BucketNotificationDestinationType.TOPIC:
                this.topicNotifications.push({ ...commonConfig, TopicArn: targetProps.arn });
                break;
            default:
                throw new Error('Unsupported notification target type:' + destination_1.BucketNotificationDestinationType[targetProps.type]);
        }
    }
    renderNotificationConfiguration() {
        return {
            LambdaFunctionConfigurations: this.lambdaNotifications.length > 0 ? this.lambdaNotifications : undefined,
            QueueConfigurations: this.queueNotifications.length > 0 ? this.queueNotifications : undefined,
            TopicConfigurations: this.topicNotifications.length > 0 ? this.topicNotifications : undefined,
        };
    }
    /**
     * Defines the bucket notifications resources in the stack only once.
     * This is called lazily as we add notifications, so that if notifications are not added,
     * there is no notifications resource.
     */
    createResourceOnce() {
        if (!this.resource) {
            const handlerArn = notifications_resource_handler_1.NotificationsResourceHandler.singleton(this);
            this.resource = new cdk.CfnResource(this, 'Resource', {
                type: 'Custom::S3BucketNotifications',
                properties: {
                    ServiceToken: handlerArn,
                    BucketName: this.bucket.bucketName,
                    NotificationConfiguration: cdk.Lazy.anyValue({ produce: () => this.renderNotificationConfiguration() }),
                },
            });
        }
        return this.resource;
    }
}
exports.BucketNotifications = BucketNotifications;
function renderFilters(filters) {
    if (!filters || filters.length === 0) {
        return undefined;
    }
    const renderedRules = new Array();
    let hasPrefix = false;
    let hasSuffix = false;
    for (const rule of filters) {
        if (!rule.suffix && !rule.prefix) {
            throw new Error('NotificationKeyFilter must specify `prefix` and/or `suffix`');
        }
        if (rule.suffix) {
            if (hasSuffix) {
                throw new Error('Cannot specify more than one suffix rule in a filter.');
            }
            renderedRules.push({ Name: 'suffix', Value: rule.suffix });
            hasSuffix = true;
        }
        if (rule.prefix) {
            if (hasPrefix) {
                throw new Error('Cannot specify more than one prefix rule in a filter.');
            }
            renderedRules.push({ Name: 'prefix', Value: rule.prefix });
            hasPrefix = true;
        }
    }
    return {
        Key: {
            FilterRules: renderedRules,
        },
    };
}
//# sourceMappingURL=data:application/json;base64,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