import argparse
import json
import logging
import sys
from typing import Optional

from annofabapi.models import CommentType

import annofabcli
import annofabcli.common.cli
from annofabcli.comment.put_comment import PutCommentMain, convert_cli_comments
from annofabcli.common.cli import (
    COMMAND_LINE_ERROR_STATUS_CODE,
    AbstractCommandLineInterface,
    ArgumentParser,
    build_annofabapi_resource_and_login,
)
from annofabcli.common.facade import AnnofabApiFacade

logger = logging.getLogger(__name__)


class PutInspectionComment(AbstractCommandLineInterface):
    COMMON_MESSAGE = "annofabcli comment put_onhold: error:"

    def validate(self, args: argparse.Namespace) -> bool:

        if args.parallelism is not None and not args.yes:
            print(
                f"{self.COMMON_MESSAGE} argument --parallelism: '--parallelism'を指定するときは、必ず '--yes' を指定してください。",
                file=sys.stderr,
            )
            return False

        return True

    def main(self):
        args = self.args
        if not self.validate(args):
            sys.exit(COMMAND_LINE_ERROR_STATUS_CODE)

        super().validate_project(args.project_id)

        dict_comments = annofabcli.common.cli.get_json_from_args(args.json)
        comments_for_task_list = convert_cli_comments(
            dict_comments,
            comment_type=CommentType.ONHOLD,
        )
        main_obj = PutCommentMain(
            self.service, project_id=args.project_id, comment_type=CommentType.ONHOLD, all_yes=self.all_yes
        )
        main_obj.add_comments_for_task_list(
            comments_for_task_list=comments_for_task_list,
            parallelism=args.parallelism,
        )


def main(args: argparse.Namespace):
    service = build_annofabapi_resource_and_login(args)
    facade = AnnofabApiFacade(service)
    PutInspectionComment(service, facade, args).main()


def parse_args(parser: argparse.ArgumentParser):
    argument_parser = ArgumentParser(parser)

    argument_parser.add_project_id()

    SAMPLE_JSON = {"task1": {"input_data1": [{"comment": "type属性が間違っています。"}]}}
    parser.add_argument(
        "--json",
        type=str,
        help=(
            "付与する保留コメントの情報をJSON形式で指定してください。"
            "``file://`` を先頭に付けると、JSON形式のファイルを指定できます。\n\n"
            f"(ex)  ``{json.dumps(SAMPLE_JSON, ensure_ascii=False)}``"
        ),
    )

    parser.add_argument(
        "--parallelism", type=int, help="使用するプロセス数（並列度）を指定してください。指定する場合は必ず ``--yes`` を指定してください。指定しない場合は、逐次的に処理します。"
    )

    parser.set_defaults(subcommand_func=main)


def add_parser(subparsers: Optional[argparse._SubParsersAction] = None):
    subcommand_name = "put_onhold"
    subcommand_help = "保留コメントを付与します"

    parser = annofabcli.common.cli.add_parser(subparsers, subcommand_name, subcommand_help)
    parse_args(parser)
    return parser
