"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnknownPrincipal = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * A principal for use in resources that need to have a role but it's unknown
 *
 * Some resources have roles associated with them which they assume, such as
 * Lambda Functions, CodeBuild projects, StepFunctions machines, etc.
 *
 * When those resources are imported, their actual roles are not always
 * imported with them. When that happens, we use an instance of this class
 * instead, which will add user warnings when statements are attempted to be
 * added to it.
 */
class UnknownPrincipal {
    constructor(props) {
        this.assumeRoleAction = 'sts:AssumeRole';
        this.resource = props.resource;
        this.grantPrincipal = this;
    }
    get policyFragment() {
        throw new Error(`Cannot get policy fragment of ${this.resource.node.path}, resource imported without a role`);
    }
    addToPrincipalPolicy(statement) {
        const stack = core_1.Stack.of(this.resource);
        const repr = JSON.stringify(stack.resolve(statement));
        this.resource.node.addWarning(`Add statement to this resource's role: ${repr}`);
        // Pretend we did the work. The human will do it for us, eventually.
        return { statementAdded: true, policyDependable: new core_1.ConcreteDependable() };
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
}
exports.UnknownPrincipal = UnknownPrincipal;
//# sourceMappingURL=data:application/json;base64,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