"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const iam = require("../lib");
// Test cross-account grant scenario's for principals
//
// When doing a grant on a resource with a resource policy:
//
// - Permissions are added to the identity if possible.
// - Trust is added to the resource if necessary (identity is in
//   a different account than the resource).
let app;
const stack1Account = '1234';
let stack1;
const stack2Account = '5678';
let stack2;
const thirdAccount = '123456789012';
beforeEach(() => {
    app = new cdk.App();
    stack1 = new cdk.Stack(app, 'Stack1', { env: { account: stack1Account, region: 'us-bla-5' } });
    stack2 = new cdk.Stack(app, 'Stack2', { env: { account: stack2Account, region: 'us-bla-5' } });
});
test('cross-account Role grant creates permissions AND trust', () => {
    // GIVEN
    const role = new iam.Role(stack1, 'Role', {
        roleName: cdk.PhysicalName.GENERATE_IF_NEEDED,
        assumedBy: new iam.ServicePrincipal('some.service'),
    });
    const resource = new FakeResource(stack2, 'Resource');
    // WHEN
    doGrant(resource, role);
    // THEN
    assertPolicyCreated(stack1);
    assertTrustCreated(stack2, {
        AWS: {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    `:iam::${stack1Account}:role/stack1stack1rolef3c14260253562f428b7`,
                ]],
        },
    });
});
test('Service Principal grant creates trust', () => {
    const resource = new FakeResource(stack2, 'Resource');
    // WHEN
    doGrant(resource, new iam.ServicePrincipal('service.amazonaws.com'));
    // THEN
    assertTrustCreated(stack2, { Service: 'service.amazonaws.com' });
});
test('Imported Role with definitely different account grant creates trust', () => {
    const resource = new FakeResource(stack2, 'Resource');
    const role = iam.Role.fromRoleArn(stack2, 'Role', `arn:aws:iam::${thirdAccount}:role/S3Access`, { mutable: true });
    // WHEN
    doGrant(resource, role);
    // THEN
    noPolicyCreated(stack2);
    assertTrustCreated(stack2, {
        AWS: `arn:aws:iam::${thirdAccount}:role/S3Access`,
    });
});
test('Imported Role with partition token in ARN (definitely different account) grant creates trust', () => {
    const resource = new FakeResource(stack2, 'Resource');
    const role = iam.Role.fromRoleArn(stack2, 'Role', `arn:${stack2.partition}:iam::${thirdAccount}:role/S3Access`, { mutable: true });
    // WHEN
    doGrant(resource, role);
    // THEN
    noPolicyCreated(stack2);
    assertTrustCreated(stack2, {
        AWS: {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    `:iam::${thirdAccount}:role/S3Access`,
                ]],
        },
    });
});
test('Imported Role with definitely same account grant does not create trust', () => {
    const resource = new FakeResource(stack2, 'Resource');
    const role = iam.Role.fromRoleArn(stack2, 'Role', `arn:aws:iam::${stack2Account}:role/S3Access`, { mutable: true });
    // WHEN
    doGrant(resource, role);
    // THEN
    assertPolicyCreated(stack2);
    noTrustCreated(stack2);
});
test('Imported Role with partition token and definitely same account grant does not create trust', () => {
    const resource = new FakeResource(stack2, 'Resource');
    const role = iam.Role.fromRoleArn(stack2, 'Role', `arn:${stack2.partition}:iam::5678:role/S3Access`, { mutable: true });
    // WHEN
    doGrant(resource, role);
    // THEN
    assertPolicyCreated(stack2);
    noTrustCreated(stack2);
});
test('Agnostic stack with concrete imported role adds trust', () => {
    // GIVEN
    const stack = new cdk.Stack(app, 'AgStack');
    const resource = new FakeResource(stack, 'Resource');
    const role = iam.Role.fromRoleArn(stack2, 'Role', 'arn:aws:iam::5678:role/S3Access', { mutable: true });
    // WHEN
    doGrant(resource, role);
    // THEN
    assertTrustCreated(stack, { AWS: 'arn:aws:iam::5678:role/S3Access' });
    noPolicyCreated(stack);
});
test('Agnostic stack with agnostic imported role does not add trust', () => {
    // GIVEN
    const stack = new cdk.Stack(app, 'AgStack');
    const resource = new FakeResource(stack, 'Resource');
    const role = iam.Role.fromRoleArn(stack2, 'Role', `arn:aws:iam::${cdk.Aws.ACCOUNT_ID}:role/S3Access`, { mutable: true });
    // WHEN
    doGrant(resource, role);
    // THEN
    assertPolicyCreated(stack2);
    noTrustCreated(stack);
});
test('Immutable role in same account adds no policy and no trust', () => {
    // GIVEN
    const resource = new FakeResource(stack2, 'Resource');
    const role = iam.Role.fromRoleArn(stack2, 'Role', `arn:aws:iam::${stack2Account}:role/S3Access`, { mutable: false });
    // require('util').inspect.defaultOptions.customInspect = false; // ?
    // WHEN
    doGrant(resource, role);
    // THEN
    noTrustCreated(stack2);
    noPolicyCreated(stack2);
});
class FakeResource extends cdk.Resource {
    constructor(scope, id) {
        super(scope, id);
        this.arn = 'arn:aws:resource';
        this.policy = new iam.PolicyDocument();
        new cdk.CfnResource(this, 'Default', {
            type: 'Test::Fake::Resource',
            properties: {
                ResourcePolicy: cdk.Lazy.anyValue({ produce: () => this.policy }),
            },
        });
    }
    addToResourcePolicy(statement) {
        this.policy.addStatements(statement);
        return { statementAdded: true, policyDependable: this.policy };
    }
}
function doGrant(resource, principal) {
    iam.Grant.addToPrincipalOrResource({
        actions: ['some:action'],
        grantee: principal,
        resourceArns: [resource.arn],
        resource,
    });
}
function assertTrustCreated(stack, principal) {
    expect(stack).toHaveResource('Test::Fake::Resource', {
        ResourcePolicy: {
            Statement: [
                {
                    Action: 'some:action',
                    Effect: 'Allow',
                    Resource: 'arn:aws:resource',
                    Principal: principal,
                },
            ],
            Version: '2012-10-17',
        },
    });
}
function noTrustCreated(stack) {
    expect(stack).not.toHaveResourceLike('Test::Fake::Resource', {
        ResourcePolicy: {
            Statement: [
                {},
            ],
        },
    });
}
function assertPolicyCreated(stack) {
    expect(stack).toHaveResource('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'some:action',
                    Effect: 'Allow',
                    Resource: 'arn:aws:resource',
                },
            ],
            Version: '2012-10-17',
        },
    });
}
function noPolicyCreated(stack) {
    expect(stack).not.toHaveResource('AWS::IAM::Policy');
}
//# sourceMappingURL=data:application/json;base64,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