"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessPoint = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const s3 = require("aws-cdk-lib/aws-s3");
const core = require("aws-cdk-lib/core");
const aws_s3objectlambda_1 = require("aws-cdk-lib/aws-s3objectlambda");
class AccessPointBase extends core.Resource {
    /** Implement the `IAccessPoint.domainName` field. */
    get domainName() {
        const urlSuffix = this.stack.urlSuffix;
        return `${this.accessPointName}-${this.env.account}.s3-object-lambda.${urlSuffix}`;
    }
    /** Implement the `IAccessPoint.regionalDomainName` field. */
    get regionalDomainName() {
        const urlSuffix = this.stack.urlSuffix;
        const region = this.env.region;
        return `${this.accessPointName}-${this.env.account}.s3-object-lambda.${region}.${urlSuffix}`;
    }
    /** Implement the `IAccessPoint.virtualHostedUrlForObject` method. */
    virtualHostedUrlForObject(key, options) {
        const domainName = options?.regional ?? true ? this.regionalDomainName : this.domainName;
        const prefix = `https://${domainName}`;
        if (!key) {
            return prefix;
        }
        if (key.startsWith('/')) {
            key = key.slice(1);
        }
        if (key.endsWith('/')) {
            key = key.slice(0, -1);
        }
        return `${prefix}/${key}`;
    }
}
/**
 * Checks the access point name against the rules in https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-names
 * @param name The name of the access point
 */
function validateAccessPointName(name) {
    if (name.length < 3 || name.length > 50) {
        throw new Error('Access point name must be between 3 and 50 characters long');
    }
    if (name.endsWith('-s3alias')) {
        throw new Error('Access point name cannot end with the suffix -s3alias');
    }
    if (name[0] === '-' || name[name.length - 1] === '-') {
        throw new Error('Access point name cannot begin or end with a dash');
    }
    if (!/^[0-9a-z](.(?![\.A-Z_]))+[0-9a-z]$/.test(name)) {
        throw new Error('Access point name must begin with a number or lowercase letter and not contain underscores, uppercase letters, or periods');
    }
}
/**
  * An S3 object lambda access point for intercepting and
  * transforming `GetObject` requests.
  */
class AccessPoint extends AccessPointBase {
    /**
     * Reference an existing AccessPoint defined outside of the CDK code.
     */
    static fromAccessPointAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_s3objectlambda_alpha_AccessPointAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromAccessPointAttributes);
            }
            throw error;
        }
        const arn = core.Arn.split(attrs.accessPointArn, core.ArnFormat.SLASH_RESOURCE_NAME);
        if (!arn.resourceName) {
            throw new Error('Unable to parse acess point name');
        }
        const name = arn.resourceName;
        class Import extends AccessPointBase {
            constructor() {
                super(...arguments);
                this.accessPointArn = attrs.accessPointArn;
                this.accessPointCreationDate = attrs.accessPointCreationDate;
                this.accessPointName = name;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.accessPointName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_s3objectlambda_alpha_AccessPointProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AccessPoint);
            }
            throw error;
        }
        if (props.accessPointName) {
            validateAccessPointName(props.accessPointName);
        }
        const supporting = new s3.CfnAccessPoint(this, 'SupportingAccessPoint', {
            bucket: props.bucket.bucketName,
        });
        const allowedFeatures = [];
        if (props.supportsGetObjectPartNumber) {
            allowedFeatures.push('GetObject-PartNumber');
        }
        if (props.supportsGetObjectRange) {
            allowedFeatures.push('GetObject-Range');
        }
        const accessPoint = new aws_s3objectlambda_1.CfnAccessPoint(this, id, {
            name: this.physicalName,
            objectLambdaConfiguration: {
                allowedFeatures,
                cloudWatchMetricsEnabled: props.cloudWatchMetricsEnabled,
                supportingAccessPoint: supporting.attrArn,
                transformationConfigurations: [
                    {
                        actions: ['GetObject'],
                        contentTransformation: {
                            AwsLambda: {
                                FunctionArn: props.handler.functionArn,
                                FunctionPayload: props.payload ? JSON.stringify(props.payload) : undefined,
                            },
                        },
                    },
                ],
            },
        });
        this.accessPointName = accessPoint.ref;
        this.accessPointArn = accessPoint.attrArn;
        this.accessPointCreationDate = accessPoint.attrCreationDate;
        props.handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['s3-object-lambda:WriteGetObjectResponse'],
            resources: ['*'],
        }));
    }
}
exports.AccessPoint = AccessPoint;
_a = JSII_RTTI_SYMBOL_1;
AccessPoint[_a] = { fqn: "@aws-cdk/aws-s3objectlambda-alpha.AccessPoint", version: "2.166.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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