"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const defaults = require("@aws-solutions-constructs/core");
const cdk = require("aws-cdk-lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const lib_1 = require("../lib");
const ec2 = require("aws-cdk-lib/aws-ec2");
const assertions_1 = require("aws-cdk-lib/assertions");
const testPort = 12321;
const testFunctionName = "something-unique";
const testClusterName = "something-else";
test("When provided a VPC, does not create a second VPC", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    new lib_1.LambdaToElasticachememcached(stack, "testStack", {
        existingVpc,
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: ".handler",
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::EC2::VPC", 1);
});
test("When provided an existingCache, does not create a second cache", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingCache = defaults.CreateTestCache(stack, "test-cache", existingVpc, testPort);
    new lib_1.LambdaToElasticachememcached(stack, "testStack", {
        existingVpc,
        existingCache,
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: ".handler",
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::ElastiCache::CacheCluster", 1);
    template.hasResourceProperties("AWS::ElastiCache::CacheCluster", {
        Port: testPort,
    });
});
test("When provided an existingFunction, does not create a second function", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingFunction = new lambda.Function(stack, "test-function", {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: ".handler",
        vpc: existingVpc,
        functionName: testFunctionName,
    });
    new lib_1.LambdaToElasticachememcached(stack, "testStack", {
        existingVpc,
        existingLambdaObj: existingFunction,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Lambda::Function", 1);
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: testFunctionName,
    });
});
test("Test custom environment variable name", () => {
    const stack = new cdk.Stack();
    const testEnvironmentVariableName = "CUSTOM_CLUSTER_NAME";
    new lib_1.LambdaToElasticachememcached(stack, "test-construct", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: ".handler",
        },
        cacheEndpointEnvironmentVariableName: testEnvironmentVariableName,
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                CUSTOM_CLUSTER_NAME: {
                    "Fn::Join": [
                        "",
                        [
                            {
                                "Fn::GetAtt": [
                                    "testconstructtestconstructclusterCF9DF48A",
                                    "ConfigurationEndpoint.Address",
                                ],
                            },
                            ":",
                            {
                                "Fn::GetAtt": [
                                    "testconstructtestconstructclusterCF9DF48A",
                                    "ConfigurationEndpoint.Port",
                                ],
                            },
                        ],
                    ],
                },
            },
        },
    });
});
test("Test setting custom function properties", () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: ".handler",
            functionName: testFunctionName,
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: testFunctionName,
    });
});
test("Test setting custom cache properties", () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: ".handler",
        },
        cacheProps: {
            clusterName: testClusterName,
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ElastiCache::CacheCluster", {
        ClusterName: testClusterName,
    });
});
test("Test setting custom VPC properties", () => {
    const stack = new cdk.Stack();
    const testCidrBlock = "192.168.0.0/16";
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: ".handler",
        },
        vpcProps: {
            ipAddresses: ec2.IpAddresses.cidr(testCidrBlock),
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: testCidrBlock,
    });
});
test("Test all default values", () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: ".handler",
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Lambda::Function", 1);
    template.resourceCountIs("AWS::ElastiCache::CacheCluster", 1);
    template.resourceCountIs("AWS::EC2::VPC", 1);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: "1",
                CACHE_ENDPOINT: {
                    "Fn::Join": [
                        "",
                        [
                            {
                                "Fn::GetAtt": [
                                    "testcachetestcachecluster27D08FAD",
                                    "ConfigurationEndpoint.Address",
                                ],
                            },
                            ":",
                            {
                                "Fn::GetAtt": [
                                    "testcachetestcachecluster27D08FAD",
                                    "ConfigurationEndpoint.Port",
                                ],
                            },
                        ],
                    ],
                },
            },
        },
        Handler: ".handler",
        Runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_STRING,
    });
    // All values taken from elasticache-defaults.ts
    template.hasResourceProperties("AWS::ElastiCache::CacheCluster", {
        CacheNodeType: "cache.t3.medium",
        Engine: "memcached",
        NumCacheNodes: 2,
        Port: 11222,
        AZMode: "cross-az",
    });
    template.hasResourceProperties("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
test('Test for the proper self referencing security group', () => {
    const stack = new cdk.Stack();
    new lib_1.LambdaToElasticachememcached(stack, "test-cache", {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: ".handler",
        },
        cacheProps: {
            port: 22223
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::SecurityGroupIngress", {
        IpProtocol: "TCP",
        FromPort: 22223,
        ToPort: 22223,
        GroupId: {
            "Fn::GetAtt": [
                "testcachetestcachecachesg74A03DA4",
                "GroupId"
            ]
        },
        SourceSecurityGroupId: {
            "Fn::GetAtt": [
                "testcachetestcachecachesg74A03DA4",
                "GroupId"
            ]
        },
    });
});
test("Test error from existingCache and no VPC", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingCache = defaults.CreateTestCache(stack, "test-cache", existingVpc);
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            existingCache,
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                handler: ".handler",
            },
        });
    };
    expect(app).toThrowError("If providing an existing Cache or Lambda Function, you must also supply the associated existingVpc");
});
test("Test error from existing function and no VPC", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingFunction = new lambda.Function(stack, "test-function", {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: ".handler",
        vpc: existingVpc,
    });
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            existingLambdaObj: existingFunction,
        });
    };
    expect(app).toThrowError("If providing an existing Cache or Lambda Function, you must also supply the associated existingVpc");
});
test("Test error from existingCache and cacheProps", () => {
    const stack = new cdk.Stack();
    const existingVpc = defaults.getTestVpc(stack);
    const existingCache = defaults.CreateTestCache(stack, "test-cache", existingVpc);
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            existingCache,
            existingVpc,
            cacheProps: {
                numCacheNodes: 4,
            },
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                handler: ".handler",
            },
        });
    };
    expect(app).toThrowError("Cannot specify existingCache and cacheProps");
});
test("Test error from trying to launch Redis", () => {
    const stack = new cdk.Stack();
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            cacheProps: {
                numCacheNodes: 4,
                engine: "redis",
            },
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                handler: ".handler",
            },
        });
    };
    expect(app).toThrowError("This construct can only launch memcached clusters");
});
test("Test error from existingCache and no VPC", () => {
    const stack = new cdk.Stack();
    const vpc = defaults.getTestVpc(stack);
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, "testStack", {
            lambdaFunctionProps: {
                code: lambda.Code.fromAsset(`${__dirname}/lambda`),
                runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
                handler: ".handler",
            },
            vpcProps: {},
            existingVpc: vpc
        });
    };
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Confirm call to CheckLambdaProps', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const testVpc = defaults.getTestVpc(stack);
    const lambdaFunction = new lambda.Function(stack, 'a-function', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        vpc: testVpc,
    });
    const props = {
        existingVpc: testVpc,
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingLambdaObj: lambdaFunction,
    };
    const app = () => {
        new lib_1.LambdaToElasticachememcached(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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