from __future__ import annotations

from typing import Annotated

from pydantic import BaseModel, Field, constr

from . import (
    AddressId,
    BillingDetails,
    BusinessId,
    CollectionMode,
    CreatedAt,
    CurrencyCode,
    CustomData,
    CustomerId,
    DiscountId,
    Event,
    NotificationId,
    OriginTransaction,
    StatusTransaction,
    SubscriptionId,
    TimePeriod,
    Timestamp,
    TransactionDetails,
    TransactionId,
    TransactionItem,
    TransactionPaymentAttempt,
    UpdatedAt,
)


class Checkout(BaseModel):
    url: Annotated[str, Field(min_length=1, max_length=2048)] | None = Field(
        None,
        description="Paddle Checkout URL for this transaction, composed of the URL passed in the request or your default payment URL + `?_ptxn=` and the Paddle ID for this transaction.",
    )


class Data(BaseModel):
    id: TransactionId | None = None
    status: StatusTransaction | None = None
    customer_id: CustomerId | None = Field(
        None,
        description="Paddle ID of the customer that this transaction is for, prefixed with `ctm_`.",
    )
    address_id: AddressId | None = Field(
        None,
        description="Paddle ID of the address that this transaction is for, prefixed with `add_`.",
    )
    business_id: BusinessId | None = Field(
        None,
        description="Paddle ID of the business that this transaction is for, prefixed with `biz_`.",
    )
    custom_data: CustomData | None = Field(
        None, description="Your own structured key-value data."
    )
    currency_code: CurrencyCode | None = Field(
        None,
        description="Supported three-letter ISO 4217 currency code. Must be `USD`, `EUR`, or `GBP` if `collection_mode` is `manual`.",
    )
    origin: OriginTransaction | None = None
    subscription_id: SubscriptionId | None = Field(
        None,
        description="Paddle ID of the subscription that this transaction is for, prefixed with `sub_`.",
    )
    invoice_id: Annotated[str, Field(pattern="^inv_[a-z\\d]{26}$")] | None = Field(
        None,
        description="Paddle ID of the invoice that this transaction is related to, prefixed with `inv_`. Used for compatibility with the Paddle Invoice API, which is now deprecated. This field is scheduled to be removed in the next version of the Paddle API.",
        examples=["inv_01ghbk4xjn4qdsmstcwzgcgg35"],
    )
    invoice_number: str | None = Field(
        None,
        description="Invoice number for this transaction. Automatically generated by Paddle when you mark a transaction as `billed` where `collection_mode` is `manual`.",
        examples=["123-45678"],
    )
    collection_mode: CollectionMode | None = Field(
        "automatic",
        description="How payment is collected for this transaction. `automatic` for checkout, `manual` for invoices.",
    )
    discount_id: DiscountId | None = Field(
        None,
        description="Paddle ID of the discount applied to this transaction, prefixed with `dsc_`.",
    )
    billing_details: BillingDetails | None = Field(
        None,
        description="Details for invoicing. Required if `collection_mode` is `manual`.",
    )
    billing_period: TimePeriod | None = Field(
        None,
        description="Time period that this transaction is for. Set automatically by Paddle for subscription renewals to describe the period that charges are for.",
    )
    items: list[TransactionItem] | None = Field(
        None,
        description="List of items on this transaction. For calculated totals, use `details.line_items`.",
        max_length=100,
        min_length=1,
    )
    details: TransactionDetails | None = None
    payments: list[TransactionPaymentAttempt] | None = Field(
        None,
        description="List of payment attempts for this transaction, including successful payments. Sorted by `created_at` in descending order, so most recent attempts are returned first.",
    )
    checkout: Checkout | None = Field(
        None,
        description="Paddle Checkout details for this transaction. Returned for automatically-collected transactions and where `billing_details.enable_checkout` is `true` for manually-collected transactions; `null` otherwise.",
    )
    created_at: CreatedAt | None = None
    updated_at: UpdatedAt | None = None
    billed_at: Timestamp | None = Field(
        None,
        description="RFC 3339 datetime string of when this transaction was marked as `billed`. `null` for transactions that aren't `billed` or `completed`. Set automatically by Paddle.",
    )
    revised_at: Timestamp | None = Field(
        None,
        description="RFC 3339 datetime string of when a transaction was revised. Revisions describe an update to customer information for a billed or completed transaction. `null` if not revised. Set automatically by Paddle.",
    )


class Billed(Event):
    notification_id: NotificationId | None = None
    data: Data | None = Field(
        None, description="New or changed entity.", title="Transaction Notification"
    )


class Canceled(Event):
    notification_id: NotificationId | None = None
    data: Data | None = Field(
        None, description="New or changed entity.", title="Transaction Notification"
    )


class Completed(Event):
    notification_id: NotificationId | None = None
    data: Data | None = Field(
        None, description="New or changed entity.", title="Transaction Notification"
    )


class Created(Event):
    notification_id: NotificationId | None = None
    data: Data | None = Field(
        None, description="New or changed entity.", title="Transaction Notification"
    )


class Paid(Event):
    notification_id: NotificationId | None = None
    data: Data | None = Field(
        None, description="New or changed entity.", title="Transaction Notification"
    )


class PastDue(Event):
    notification_id: NotificationId | None = None
    data: Data | None = Field(
        None, description="New or changed entity.", title="Transaction Notification"
    )


class PaymentFailed(Event):
    notification_id: NotificationId | None = None
    data: Data | None = Field(
        None, description="New or changed entity.", title="Transaction Notification"
    )


class Ready(Event):
    notification_id: NotificationId | None = None
    data: Data | None = Field(
        None, description="New or changed entity.", title="Transaction Notification"
    )


class Revised(Event):
    notification_id: NotificationId | None = None
    data: Data | None = Field(
        None, description="New or changed entity.", title="Transaction Notification"
    )


class Updated(Event):
    notification_id: NotificationId | None = None
    data: Data | None = Field(
        None, description="New or changed entity.", title="Transaction Notification"
    )
