# Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
# SPDX-License-Identifier: Apache-2.0

import json
import logging
import os
import uuid
from typing import List

import boto3
import textractcaller as tc
from textractmanifest import IDPManifest, IDPManifestSchema, Query

logger = logging.getLogger(__name__)
version = "0.0.7"
s3 = boto3.client('s3')


def convert_manifest_queries_config_to_caller(
        queries_config: List[Query]) -> tc.QueriesConfig:
    if queries_config:
        return tc.QueriesConfig(queries=[
            tc.Query(text=x.text, alias=x.alias, pages=x.pages)
            for x in queries_config
        ])
    else:
        return tc.QueriesConfig(queries=[])


def convert_manifest_features_to_caller(
        features: List[str]) -> List[tc.Textract_Features]:
    if features:
        return [tc.Textract_Features[x] for x in features]
    else:
        return []


def lambda_handler(event, _):
    log_level = os.environ.get('LOG_LEVEL', 'INFO')
    logger.setLevel(log_level)
    logger.info(f"version: {version}")
    logger.info(json.dumps(event))
    textract_endpoint_url = os.environ.get('TEXTRACT_ENDPOINT_URL', None)

    s3_output_bucket = os.environ.get('S3_OUTPUT_BUCKET')
    s3_output_prefix = os.environ.get('S3_OUTPUT_PREFIX')

    if not s3_output_bucket or not s3_output_prefix:
        raise ValueError(
            f"no s3_output_bucket: {s3_output_bucket} or s3_output_prefix: {s3_output_prefix} defined."
        )
    logger.info(f"LOG_LEVEL: {log_level} \n \
                S3_OUTPUT_BUCKET: {s3_output_bucket} \n \
                S3_OUTPUT_PREFIX: {s3_output_prefix} \n \
                TEXTRACT_ENDPOINT_URL: {textract_endpoint_url}")

    if textract_endpoint_url:
        textract = boto3.client("textract", endpoint_url=textract_endpoint_url)
    else:
        textract = boto3.client("textract")

    manifest: IDPManifest = IDPManifestSchema().load(
        event['manifest'])  #type: ignore
    logger.debug(manifest)

    s3_path = manifest.s3_path
    s3_filename, _ = os.path.splitext(os.path.basename(s3_path))

    features = convert_manifest_features_to_caller(manifest.textract_features)
    queries_config = convert_manifest_queries_config_to_caller(
        manifest.queries_config)
    logger.debug(f"before call_textract\n \
        input_document: {s3_path} \n \
        features: {features}\n \
        queries_config: {queries_config}")
    textract_response = tc.call_textract(input_document=s3_path,
                                         boto3_textract_client=textract,
                                         features=features,
                                         queries_config=queries_config)

    output_bucket_key = s3_output_prefix + "/" + str(
        uuid.uuid4()) + "/" + s3_filename + ".json"
    s3.put_object(Body=bytes(
        json.dumps(textract_response, indent=4).encode('UTF-8')),
                  Bucket=s3_output_bucket,
                  Key=output_bucket_key)
    return {
        "TextractOutputJsonPath":
        f"s3://{s3_output_bucket}/{output_bucket_key}"
    }
