# Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
# SPDX-License-Identifier: Apache-2.0

import datetime
import json
import logging
import os
import time
import uuid
import boto3

from typing import List

import textractcaller as tc
import textractmanifest as tm

logger = logging.getLogger(__name__)

region = os.environ['AWS_REGION']
dynamo_db_client = boto3.client("dynamodb")
__version__ = "0.0.1"


def convert_manifest_queries_config_to_caller(
        queries_config: List[tm.Query]) -> tc.QueriesConfig:
    if queries_config:
        return tc.QueriesConfig(queries=[
            tc.Query(text=x.text, alias=x.alias, pages=x.pages)
            for x in queries_config
        ])
    else:
        return tc.QueriesConfig(queries=[])


def convert_manifest_features_to_caller(
        features: List[str]) -> List[tc.Textract_Features]:
    if features:
        return [tc.Textract_Features[x] for x in features]
    else:
        return []


def lambda_handler(event, _):
    log_level = os.environ.get('LOG_LEVEL', 'INFO')
    logger.setLevel(log_level)
    logger.info(json.dumps(event))
    logger.info(
        f"version: {__version__}\ntextractmanifest version: {tm.__version__}\nboto3 version: {boto3.__version__}\ntextractcaller version: {tc.__version__}."
    )
    notification_sns = os.environ.get('NOTIFICATION_SNS', None)
    if not notification_sns:
        raise Exception("no NOTIFICATION_SNS set")

    s3_output_bucket = os.environ.get('S3_OUTPUT_BUCKET', None)
    if not s3_output_bucket:
        raise Exception("no S3_OUTPUT_BUCKET set")

    s3_temp_output_prefix = os.environ.get('S3_TEMP_OUTPUT_PREFIX', None)
    if not s3_temp_output_prefix:
        raise Exception("no S3_TEMP_OUTPUT_PREFIX set")

    notification_role_arn = os.environ.get('NOTIFICATION_ROLE_ARN', None)
    if not notification_role_arn:
        raise Exception("no NOTIFICATION_ROLE_ARN set")

    token_store_ddb = os.environ.get('TOKEN_STORE_DDB', None)
    if not token_store_ddb:
        raise Exception("no TOKEN_STORE_DDB set")

    textract_endpoint_url = os.environ.get('TEXTRACT_ENDPOINT_URL', None)

    logger.info(f"LOG_LEVEL: {log_level} \n \
                NOTIFICATION_SNS: {notification_sns} \n \
                NOTIFICATION_ROLE_ARN: {notification_role_arn} \n \
                S3_TEMP_OUTPUT_PREFIX: {s3_temp_output_prefix} \n \
                S3_OUTPUT_BUCKET: {s3_output_bucket} \n \
                TOKEN_STORE_DDB: {token_store_ddb} \n \
                TEXTRACT_ENDPOINT_URL: {textract_endpoint_url}")

    if textract_endpoint_url:
        textract = boto3.client("textract", endpoint_url=textract_endpoint_url)
    else:
        textract = boto3.client("textract")

    manifest: tm.IDPManifest = tm.IDPManifestSchema().load(
        event["Payload"]['manifest'])  #type: ignore

    s3_path = manifest.s3_path

    token = event['Token']
    execution_id = event['ExecutionId']

    logger.info(f"s3_path: {s3_path} \n \
                token: {token} \n \
                execution_id: {execution_id}")

    try:
        uuid_key = str(uuid.uuid4())
        logger.debug(f"uuid_key: {uuid_key}")
        ddb_response = dynamo_db_client.put_item(
            TableName=token_store_ddb,
            Item={
                "ID": {
                    'S': uuid_key
                },
                "Type": {
                    'S': "textract_async"
                },
                "Token": {
                    'S': token
                },
                "WorkflowId": {
                    'S': execution_id
                },
                "ttltimestamp": {
                    'N':
                    str(
                        int(time.time()) +
                        int(datetime.timedelta(days=7).total_seconds()))
                }
            })
        logger.debug(f"ddb_response: {ddb_response}")
        nc: tc.NotificationChannel = tc.NotificationChannel(
            role_arn=notification_role_arn, sns_topic_arn=notification_sns)
        output_config: tc.OutputConfig = tc.OutputConfig(
            s3_bucket=s3_output_bucket, s3_prefix=s3_temp_output_prefix)
        response = tc.call_textract(
            input_document=s3_path,
            boto3_textract_client=textract,
            output_config=output_config,
            notification_channel=nc,
            job_tag=uuid_key,
            return_job_id=True,
            force_async_api=True,
            features=convert_manifest_features_to_caller(
                manifest.textract_features),
            queries_config=convert_manifest_queries_config_to_caller(
                manifest.queries_config))
        logger.debug(f"Textract-response: {response}")
        return {"TextractResponse": response}
    except Exception as e:
        logger.error(e)
        raise ValueError(e)
