"""
SQS_Task_Token_Listener Lambda in Step Functions
Listens to SQS queue for TaskTokens and calls StepFunctions
"""
import json
import logging
import os
import re
import boto3
import textractcaller as tc
import uuid

logger = logging.getLogger(__name__)

step_functions_client = boto3.client(service_name='stepfunctions')
dynamo_db_client = boto3.client('dynamodb')
textract = boto3.client('textract')
s3 = boto3.client('s3')


def lambda_handler(event, context):

    log_level = os.environ.get('LOG_LEVEL', 'INFO')
    logger.setLevel(log_level)
    logger.info(json.dumps(event))

    token_store_ddb = os.environ.get('TOKEN_STORE_DDB', None)
    if not token_store_ddb:
        raise Exception("no TOKEN_STORE_DDB set")

    s3_output_bucket = os.environ.get('S3_OUTPUT_BUCKET')
    s3_output_prefix = os.environ.get('S3_OUTPUT_PREFIX')

    s3_temp_output_prefix = os.environ.get('S3_TEMP_OUTPUT_PREFIX', None)
    if not s3_temp_output_prefix:
        raise Exception("no S3_TEMP_OUTPUT_PREFIX set")

    if not s3_output_bucket or not s3_output_prefix:
        raise ValueError(
            f"no s3_output_bucket: {s3_output_bucket} or s3_output_prefix: {s3_output_prefix} defined."
        )

    logger.info(f"LOG_LEVEL: {log_level} \n \
                TOKEN_STORE_DDB: {token_store_ddb} \n \
                OUTPUT_BUCKET: {s3_output_bucket} \n \
                S3_TEMP_OUTPUT_PREFIX: {s3_temp_output_prefix} \n \
                OUTPUT_BUCKET_PREFIX: {s3_output_prefix}")

    for record in event['Records']:
        if "eventSource" in record and record["eventSource"] == "aws:sqs":
            body = json.loads(record["body"])
            message = json.loads(body['Message'])
        elif "EventSource" in record and record["EventSource"] == "aws:sns":
            message = json.loads(record['Sns']['Message'])
            logger.debug(f"message: {message}")
        else:
            raise Exception("not supported event source.")

        input_bucket_key = message['DocumentLocation']['S3ObjectName']
        s3_filename, _ = os.path.splitext(input_bucket_key)
        output_bucket_prefix = re.sub('/', '', s3_output_prefix)
        job_status = message['Status']
        job_tag = message['JobTag']
        job_id = message['JobId']
        logger.info(
            f"job_tag: '{job_tag}, job_id: {job_id}, job_status: {job_status}'"
        )

        ddb_response = dynamo_db_client.get_item(TableName=token_store_ddb,
                                                 Key={"ID": {
                                                     'S': job_tag
                                                 }})
        logger.debug(f"ddb_response: {ddb_response}")
        task_token = ddb_response['Item']['Token']['S']
        output_bucket_key = output_bucket_prefix + "/" + s3_filename + ".json"

        if job_status == 'SUCCEEDED':
            output_config: tc.OutputConfig = tc.OutputConfig(
                s3_bucket=s3_output_bucket, s3_prefix=s3_temp_output_prefix)
            textract_response = tc.get_full_json_from_output_config(
                output_config=output_config, job_id=job_id)

            output_bucket_key = s3_output_prefix + "/" + str(
                uuid.uuid4()) + "/" + s3_filename + ".json"
            s3.put_object(Body=bytes(
                json.dumps(textract_response, indent=4).encode('UTF-8')),
                          Bucket=s3_output_bucket,
                          Key=output_bucket_key)
            result_location = f"s3://{s3_output_bucket}/{output_bucket_key}"

            step_functions_client.send_task_success(
                taskToken=task_token,
                output=json.dumps({"TextractOutputJsonPath": result_location}))
        else:
            step_functions_client.send_task_failure(taskToken=task_token,
                                                    error=job_status,
                                                    cause="see output")
