"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TextractStepFunctionsStartExecution = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const sns = require("aws-cdk-lib/aws-sns");
const aws_sns_subscriptions_1 = require("aws-cdk-lib/aws-sns-subscriptions");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
// import * as tb from 'schadem-cdk-construct-sfn-token-bucket';
/**
 * Verifies that a validation pattern is supported for a service integration
 *
 */
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
// interface AcquireLockProps{
//   concurrentJobLimit: number;
//   concurrencyTable: dynamodb.Table;
// }
// class AcquireLockClass extends sfn.StateMachineFragment {
//   public readonly startState: sfn.State;
//   public readonly endStates: sfn.INextable[];
//   constructor(parent: Construct, id: string, props: AcquireLockProps) {
//     super(parent, id);
//     //  CONCURRENCY CONTROL
//     // TODO: check for task table in props
//     // const concurrencyTableName = concurrencyTable.tableName;
//     // TODO: Token Bucket implementation for TPS
//     // TODO: generate all state names, all have to have unique names
//     const acquireLock = new tasks.DynamoUpdateItem(this, 'AcquireLock', {
//       table: props.concurrencyTable,
//       key: {
//         LockName: tasks.DynamoAttributeValue.fromString('MySempahore'),
//       },
//       expressionAttributeNames: {
//         '#currentlockcount': 'currentlockcount',
//         '#lockownerid.$': '$$.Execution.Id',
//       },
//       expressionAttributeValues: {
//         ':increase': tasks.DynamoAttributeValue.fromNumber(1),
//         ':limit': tasks.DynamoAttributeValue.fromNumber(5),
//         ':lockacquiredtime': tasks.DynamoAttributeValue.fromString(sfn.JsonPath.stringAt('$$.State.EnteredTime')),
//       },
//       updateExpression: 'SET #currentlockcount = #currentlockcount + :increase, #lockownerid = :lockacquiredtime',
//       conditionExpression: 'currentlockcount <> :limit and attribute_not_exists(#lockownerid)',
//       returnValues: tasks.DynamoReturnValues.UPDATED_NEW,
//     });
//     const initializeLockItemTask = new tasks.DynamoPutItem(this, 'InitializeLockItem', {
//       table: props.concurrencyTable,
//       item: {
//         LockName: tasks.DynamoAttributeValue.fromString('MySempahore'),
//         currentlockcount: tasks.DynamoAttributeValue.fromNumber(0),
//       },
//       conditionExpression: 'LockName <> :lockname',
//       expressionAttributeValues: {
//         ':lockname': tasks.DynamoAttributeValue.fromString('MySempahore'),
//       },
//     });
//     const getCurrentLockRecord = new tasks.DynamoGetItem(this, 'GetCurrentLockRecord', {
//       table: props.concurrencyTable,
//       key: {
//         LockName: tasks.DynamoAttributeValue.fromString('MySempahore'),
//       },
//       expressionAttributeNames: {
//         '#lockownerid.$': '$$.Execution.Id',
//       },
//       projectionExpression: [new tasks.DynamoProjectionExpression().withAttribute('#lockownerid')],
//       resultSelector: [{ 'Item.$': '$.Item' }, { 'ItemString.$': 'States.JsonToString($.Item)' }],
//     });
//     acquireLock.addRetry({
//       interval: Duration.seconds(1),
//       maxAttempts: 0,
//       backoffRate: 1,
//       errors: ['DynamoDB.AmazonDynamoDBException'],
//     });
//     acquireLock.addRetry({
//       interval: Duration.seconds(1),
//       maxAttempts: 6,
//       backoffRate: 2,
//       errors: [sfn.Errors.ALL],
//     });
//     acquireLock.addCatch(initializeLockItemTask, {
//       resultPath: '$.lockinfo.acquisitionerror',
//       errors: ['DynamoDB.AmazonDynamoDBException'],
//     });
//     acquireLock.addCatch(getCurrentLockRecord, {
//       resultPath: '$.lockinfo.acquisitionerror',
//       errors: ['DynamoDB.ConditionalCheckFailedException'],
//     });
//     initializeLockItemTask.addCatch(acquireLock, {
//       errors: [sfn.Errors.ALL],
//     });
//     const continueBecauseLockWasAlreadyAcquired = new sfn.Pass(this, 'ContinueBecauseLockWasAlreadyAcquired');
//     const waitToGetLock = new sfn.Wait(this, 'WaitToGetLock', { time: sfn.WaitTime.duration(Duration.seconds(3)) });
//     waitToGetLock.next(acquireLock);
//     const checkIfLockAlreadyAcquired = new sfn.Choice(this, 'CheckIfLockAlreadyAcquired')
//       .when(
//         sfn.Condition.and(sfn.Condition.isPresent('$.lockinfo.currentlockitem.ItemString'),
//           sfn.Condition.stringMatches('$.lockinfo.currentlockitem.ItemString', '*Z*'),
//         ), continueBecauseLockWasAlreadyAcquired)
//       .otherwise(waitToGetLock);
//     getCurrentLockRecord.next(checkIfLockAlreadyAcquired);
//     this.startState = acquireLock;
//     this.endStates = [acquireLock, continueBecauseLockWasAlreadyAcquired];
//   }
// }
// interface ReleaseLockProps{
//   concurrencyTable: dynamodb.Table;
// }
// class ReleaseLockClass extends sfn.StateMachineFragment {
//   public readonly startState: sfn.State;
//   public readonly endStates: sfn.INextable[];
//   constructor(parent: Construct, id: string, props: ReleaseLockProps) {
//     super(parent, id);
//     const releaseLock = new tasks.DynamoUpdateItem(this, 'ReleaseLock', {
//       table: props.concurrencyTable,
//       key: {
//         LockName: tasks.DynamoAttributeValue.fromString('MySempahore'),
//       },
//       expressionAttributeNames: {
//         '#currentlockcount': 'currentlockcount',
//         '#lockownerid.$': '$$.Execution.Id',
//       },
//       expressionAttributeValues: {
//         ':decrease': tasks.DynamoAttributeValue.fromNumber(1),
//       },
//       updateExpression: 'SET #currentlockcount = #currentlockcount + :decrease REMOVE #lockownerid',
//       conditionExpression: 'attribute_exists(#lockownerid)',
//       returnValues: tasks.DynamoReturnValues.UPDATED_NEW,
//     });
//     releaseLock.addRetry({
//       maxAttempts: 0,
//       errors: ['DynamoDB.ConditionalCheckFailedException'],
//     });
//     releaseLock.addRetry({
//       maxAttempts: 5,
//       backoffRate: 1.5,
//       errors: [sfn.Errors.ALL],
//     });
//     const successState = new sfn.Pass(this, 'ReleaseLockSuccessState');
//     releaseLock.addCatch(successState, {
//       errors: ['DynamoDB.ConditionalCheckFailedException'],
//     });
//     releaseLock.next(successState);
//     this.endStates = [successState];
//     this.startState = releaseLock;
//   }
// }
/**
 * A Step Functions Task to call Textract. Requires
 *
 * It supports three service integration patterns: REQUEST_RESPONSE, RUN_JOB, and WAIT_FOR_TASK_TOKEN.
 */
//TODO: Should I extend StateMachineFragment instead?
class TextractStepFunctionsStartExecution extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, TextractStepFunctionsStartExecution.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        if (!props.s3OutputBucket || !props.s3OutputPrefix) {
            throw new Error('no s3OutputBucket and s3OutputPrefix passed in');
        }
        var defaultClassification = '';
        if (typeof (props.defaultClassification != 'undefined') && props.defaultClassification) {
            defaultClassification = props.defaultClassification;
        }
        // const concurrencyTable = new dynamodb.Table(this, 'TextractConcurrencyTable', {
        //   partitionKey: {
        //     name: 'LockName',
        //     type: dynamodb.AttributeType.STRING,
        //   },
        //   billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
        // });
        // ACQUIRE-LOCK
        // const acquireLockClass = new AcquireLockClass(this, 'AcquireLockClass', {
        //   concurrencyTable: concurrencyTable,
        //   concurrentJobLimit: 600,
        // });
        // const parallel = new sfn.Parallel(this, 'AcquireLockParallel', { resultPath: sfn.JsonPath.DISCARD })
        //   .branch(acquireLockClass);;
        // END ACQUIRE-LOCK
        /**
         * TEXTRACT CALLS
         */
        // TODO: check for task table in props
        const taskTokenTable = new dynamodb.Table(this, 'TextractTaskTokenTable', {
            partitionKey: {
                name: 'ID',
                type: dynamodb.AttributeType.STRING,
            },
            billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
            timeToLiveAttribute: 'ttltimestamp',
        });
        const taskTokenTableName = taskTokenTable.tableName;
        // DECIDER
        const deciderFunction = new lambda.DockerImageFunction(this, 'TextractDecider', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/decider/')),
            memorySize: 128,
            environment: {
                DEFAULT_CLASSIFICATION: defaultClassification,
            },
        });
        deciderFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['s3:GetObject'], resources: ['*'] }));
        const deciderLambdaInvoke = new tasks.LambdaInvoke(this, 'deciderLambdaInvoke', {
            lambdaFunction: deciderFunction,
            timeout: aws_cdk_lib_1.Duration.seconds(100),
            outputPath: '$.Payload',
        });
        deciderLambdaInvoke.addRetry({
            interval: aws_cdk_lib_1.Duration.seconds(1),
            maxAttempts: 6,
            backoffRate: 2,
            errors: [sfn.Errors.ALL],
        });
        // TEXTRACT SYNC
        const textractSyncCallFunction = new lambda.DockerImageFunction(this, 'TextractSyncCall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/sync/')),
            memorySize: 128,
            environment: {
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_OUTPUT_PREFIX: props.s3OutputPrefix,
            },
        });
        textractSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['textract:*'], resources: ['*'] }));
        // TODO split up role into put for output_prefix and get for input_prefixes
        // TODO: S3 access granted to * buckets for testing, should be param as well
        textractSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                's3:GetObject', 's3:ListBucket', 's3:PutObject',
            ],
            resources: [`arn:aws:s3:::${props.s3OutputBucket}`, `arn:aws:s3:::${props.s3OutputBucket}/*`, '*'],
        }));
        // single page sync call
        const detectText = new tasks.LambdaInvoke(this, 'TextractSyncTask', {
            lambdaFunction: textractSyncCallFunction,
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            outputPath: '$.Payload',
        });
        detectText.addRetry({
            interval: aws_cdk_lib_1.Duration.seconds(1),
            maxAttempts: 6,
            backoffRate: 2,
            errors: [sfn.Errors.ALL],
        });
        // TEXTRACT EXPENSE SYNC
        const textractExpenseSyncCallFunction = new lambda.DockerImageFunction(this, 'TextractExpenseSyncCall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/expense_sync/')),
            memorySize: 128,
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            environment: {
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_OUTPUT_PREFIX: props.s3OutputPrefix,
            },
        });
        textractExpenseSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['textract:*'], resources: ['*'] }));
        // TODO split up role into put for output_prefix and get for input_prefixes
        // TODO: S3 access granted to * buckets for testing, should be param as well
        textractExpenseSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                's3:GetObject', 's3:ListBucket', 's3:PutObject',
            ],
            resources: [`arn:aws:s3:::${props.s3OutputBucket}`, `arn:aws:s3:::${props.s3OutputBucket}/*`, '*'],
        }));
        // single page sync call
        const expenseSyncTask = new tasks.LambdaInvoke(this, 'TextractExpenseSync', {
            lambdaFunction: textractExpenseSyncCallFunction,
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            outputPath: '$.Payload',
        });
        expenseSyncTask.addRetry({
            interval: aws_cdk_lib_1.Duration.seconds(1),
            maxAttempts: 6,
            backoffRate: 2,
            errors: [sfn.Errors.ALL],
        });
        // IDENTITY SYNC
        const textractIdentitySyncCallFunction = new lambda.DockerImageFunction(this, 'TextractIdentitySyncCall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/analyzeid_sync/')),
            memorySize: 128,
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            environment: {
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_OUTPUT_PREFIX: props.s3OutputPrefix,
            },
        });
        textractIdentitySyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['textract:*'], resources: ['*'] }));
        // TODO split up role into put for output_prefix and get for input_prefixes
        // TODO: S3 access granted to * buckets for testing, should be param as well
        textractIdentitySyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                's3:GetObject', 's3:ListBucket', 's3:PutObject',
            ],
            resources: [`arn:aws:s3:::${props.s3OutputBucket}`, `arn:aws:s3:::${props.s3OutputBucket}/*`, '*'],
        }));
        const identitySyncTask = new tasks.LambdaInvoke(this, 'TextractIdentitySync', {
            lambdaFunction: textractIdentitySyncCallFunction,
            outputPath: '$.Payload',
        });
        identitySyncTask.addRetry({
            interval: aws_cdk_lib_1.Duration.seconds(1),
            maxAttempts: 6,
            backoffRate: 2,
            errors: [sfn.Errors.ALL],
        });
        // TEXTRACT ASYNC
        const textractAsyncSNSRole = new iam.Role(this, 'TextractAsyncSNSRole', {
            assumedBy: new iam.ServicePrincipal('textract.amazonaws.com'),
            managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSQSFullAccess'),
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSNSFullAccess'),
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonS3ReadOnlyAccess'),
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonTextractFullAccess')],
        });
        // TODO: change log level, make them a prop
        const textractAsyncSNS = new sns.Topic(this, 'TextractAsyncSNS');
        const textractAsyncCallFunction = new lambda.DockerImageFunction(this, 'TextractAsyncCall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/textract_async/')),
            memorySize: 128,
            environment: {
                NOTIFICATION_SNS: textractAsyncSNS.topicArn,
                NOTIFICATION_ROLE_ARN: textractAsyncSNSRole.roleArn,
                TOKEN_STORE_DDB: taskTokenTableName,
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_TEMP_OUTPUT_PREFIX: props.s3TempOutputPrefix,
                LOG_LEVEL: 'DEBUG',
            },
        });
        textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['textract:*'], resources: ['*'] }));
        // TODO split up role into put for output_prefix and get for input_prefixes
        // TODO: S3 access granted to * buckets for testing, should be param as well
        textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                's3:GetObject', 's3:ListBucket', 's3:PutObject',
            ],
            resources: [`arn:aws:s3:::${props.s3OutputBucket}`, `arn:aws:s3:::${props.s3OutputBucket}/*`, '*'],
        }));
        textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['sns:*'], resources: ['*'] }));
        textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['dynamodb:*'], resources: ['*'] }));
        const textractAsyncLambdaInvoke = new tasks.LambdaInvoke(this, 'TextractASync', {
            lambdaFunction: textractAsyncCallFunction,
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            payload: sfn.TaskInput.fromObject({
                Token: sfn.JsonPath.taskToken,
                ExecutionId: sfn.JsonPath.stringAt('$$.Execution.Id'),
                Payload: sfn.JsonPath.entirePayload,
            }),
        });
        textractAsyncLambdaInvoke.addRetry({
            interval: aws_cdk_lib_1.Duration.seconds(1),
            maxAttempts: 6,
            backoffRate: 2,
            errors: [sfn.Errors.ALL],
        });
        // ASYNC SNS LISTENER
        const textractAsyncSNSFunction = new lambda.DockerImageFunction(this, 'TextractAsyncSNSFunction', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/textract_async_sns_listener/')),
            memorySize: 2048,
            timeout: aws_cdk_lib_1.Duration.seconds(900),
            environment: {
                TOKEN_STORE_DDB: taskTokenTableName,
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_OUTPUT_PREFIX: props.s3OutputPrefix,
                S3_TEMP_OUTPUT_PREFIX: props.s3TempOutputPrefix,
                LOG_LEVEL: 'DEBUG',
            },
        });
        // EXPENSE ASYNC
        const textractExpenseAsyncCallFunction = new lambda.DockerImageFunction(this, 'TextractExpenseAsyncCall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/expense_async/')),
            memorySize: 128,
            environment: {
                NOTIFICATION_SNS: textractAsyncSNS.topicArn,
                NOTIFICATION_ROLE_ARN: textractAsyncSNSRole.roleArn,
                TOKEN_STORE_DDB: taskTokenTableName,
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_TEMP_OUTPUT_PREFIX: props.s3TempOutputPrefix,
                LOG_LEVEL: 'DEBUG',
            },
        });
        textractExpenseAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['textract:*'], resources: ['*'] }));
        // TODO split up role into put for output_prefix and get for input_prefixes
        // TODO: S3 access granted to * buckets for testing, should be param as well
        textractExpenseAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                's3:GetObject', 's3:ListBucket', 's3:PutObject',
            ],
            resources: [`arn:aws:s3:::${props.s3OutputBucket}`, `arn:aws:s3:::${props.s3OutputBucket}/*`, '*'],
        }));
        textractExpenseAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['sns:*'], resources: ['*'] }));
        textractExpenseAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['dynamodb:*'], resources: ['*'] }));
        const textractExpenseAsyncLambdaInvoke = new tasks.LambdaInvoke(this, 'TextractExpenseASync', {
            lambdaFunction: textractExpenseAsyncCallFunction,
            integrationPattern: sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            payload: sfn.TaskInput.fromObject({
                Token: sfn.JsonPath.taskToken,
                ExecutionId: sfn.JsonPath.stringAt('$$.Execution.Id'),
                Payload: sfn.JsonPath.entirePayload,
            }),
        });
        textractExpenseAsyncLambdaInvoke.addRetry({
            interval: aws_cdk_lib_1.Duration.seconds(1),
            maxAttempts: 6,
            backoffRate: 2,
            errors: [sfn.Errors.ALL],
        });
        textractAsyncSNS.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(textractAsyncSNSFunction));
        textractAsyncSNSFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['states:SendTaskSuccess', 'states:SendTaskFailure'], resources: ['*'] }));
        textractAsyncSNSFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['textract:*'], resources: ['*'] }));
        textractAsyncSNSFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['dynamodb:*'], resources: ['*'] }));
        textractAsyncSNSFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['s3:*'], resources: ['*'] }));
        // STEP FUNCTION FLOW
        const unsupportedMineType = new sfn.Fail(this, 'TextractUnsupportedMimeType');
        const expenseMimeTypeChoice = new sfn.Choice(this, 'ExpenseMimeTypeChoice')
            .when(sfn.Condition.or(sfn.Condition.stringEquals('$.mime', 'image/png'), sfn.Condition.stringEquals('$.mime', 'image/jpeg')), expenseSyncTask)
            .when(sfn.Condition.or(sfn.Condition.stringEquals('$.mime', 'application/pdf'), sfn.Condition.stringEquals('$.mime', 'image/tiff')), textractExpenseAsyncLambdaInvoke)
            .otherwise(unsupportedMineType);
        const singleMultiChoice = new sfn.Choice(this, 'singleMultiPageChoice')
            .when(sfn.Condition.numberGreaterThan('$.numberOfPages', 1), textractAsyncLambdaInvoke)
            .otherwise(detectText);
        const textractMimeTypeChoice = new sfn.Choice(this, 'TextractMimeTypeChoice')
            .when(sfn.Condition.or(sfn.Condition.stringEquals('$.mime', 'image/png'), sfn.Condition.stringEquals('$.mime', 'image/jpeg')), detectText)
            .when(sfn.Condition.or(sfn.Condition.stringEquals('$.mime', 'application/pdf'), sfn.Condition.stringEquals('$.mime', 'image/tiff')), singleMultiChoice)
            .otherwise(unsupportedMineType);
        // EXPENSE OR NOT
        const expenseIdentityOrNotChoice = new sfn.Choice(this, 'ExpenseIdentityOrNotChoice')
            .when(sfn.Condition.and(sfn.Condition.isPresent('$.classification'), sfn.Condition.stringEquals('$.classification', 'EXPENSE')), expenseMimeTypeChoice)
            .when(sfn.Condition.and(sfn.Condition.isPresent('$.classification'), sfn.Condition.stringEquals('$.classification', 'IDENTITY')), identitySyncTask)
            .otherwise(textractMimeTypeChoice);
        // const tokenBucketStep = new tb.StepFunctionsTPSTokenBucket(this, 'StepFunctionsTPCTokenBucket', { tokenLimit: 10 });
        // const releaseLockClass = new ReleaseLockClass(this, 'ReleaseLock', { concurrencyTable: concurrencyTable });
        // const workflow_chain = sfn.Chain.start(parallel)
        const workflow_chain = sfn.Chain.start(deciderLambdaInvoke)
            // .next(tokenBucketStep)
            // .next(deciderLambdaInvoke)
            .next(expenseIdentityOrNotChoice);
        // detectText.next(releaseLockClass);
        // textractAsyncLambdaInvoke.next(releaseLockClass);
        this.stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: workflow_chain,
            timeout: aws_cdk_lib_1.Duration.hours(4),
        });
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
         * @internal
         */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
         * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
         *
         * the scoped access policy should be generated accordingly.
         *
         * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
         */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.TextractStepFunctionsStartExecution = TextractStepFunctionsStartExecution;
_a = JSII_RTTI_SYMBOL_1;
TextractStepFunctionsStartExecution[_a] = { fqn: "schadem-cdk-construct-sfn-test.TextractStepFunctionsStartExecution", version: "0.0.12" };
TextractStepFunctionsStartExecution.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,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