import mock
import os
import pandas as pd
from datetime import datetime
from flexmock import flexmock
from sportsipy import utils
from sportsipy.constants import AWAY
from sportsipy.ncaaf.constants import BOXSCORE_URL, BOXSCORES_URL
from sportsipy.ncaaf.boxscore import Boxscore, Boxscores


MONTH = 10
YEAR = 2020

BOXSCORE = '2020-09-12-wake-forest'


def read_file(filename):
    filepath = os.path.join(os.path.dirname(__file__), 'ncaaf', filename)
    return open('%s' % filepath, 'r', encoding='utf8').read()


def mock_pyquery(url):
    class MockPQ:
        def __init__(self, html_contents):
            self.status_code = 200
            self.html_contents = html_contents
            self.text = html_contents

    if url == BOXSCORES_URL % (9, 12, 2020):
        return MockPQ(read_file('boxscores-9-12-2020.html'))
    if url == BOXSCORES_URL % (9, 13, 2020):
        return MockPQ(read_file('boxscores-9-13-2020.html'))
    boxscore = read_file('%s.html' % BOXSCORE)
    return MockPQ(boxscore)


class MockDateTime:
    def __init__(self, year, month):
        self.year = year
        self.month = month


class TestNCAAFBoxscore:
    @mock.patch('requests.get', side_effect=mock_pyquery)
    def setup_method(self, *args, **kwargs):
        self.results = {
            'date': 'Saturday Sep 12, 2020',
            'time': '7:30 PM ET',
            'stadium': '',
            'winner': AWAY,
            'winning_name': 'Clemson',
            'winning_abbr': 'CLEMSON',
            'losing_name': 'Wake Forest',
            'losing_abbr': 'WAKE-FOREST',
            'away_points': 37,
            'away_first_downs': 27,
            'away_rush_attempts': 46,
            'away_rush_yards': 185,
            'away_rush_touchdowns': 3,
            'away_pass_completions': 26,
            'away_pass_attempts': 36,
            'away_pass_yards': 376,
            'away_pass_touchdowns': 1,
            'away_interceptions': 1,
            'away_total_yards': 561,
            'away_fumbles': 0,
            'away_fumbles_lost': 0,
            'away_turnovers': 1,
            'away_penalties': 5,
            'away_yards_from_penalties': 55,
            'home_points': 13,
            'home_first_downs': 15,
            'home_rush_attempts': 34,
            'home_rush_yards': 37,
            'home_rush_touchdowns': 0,
            'home_pass_completions': 19,
            'home_pass_attempts': 38,
            'home_pass_yards': 293,
            'home_pass_touchdowns': 1,
            'home_interceptions': 0,
            'home_total_yards': 330,
            'home_fumbles': 0,
            'home_fumbles_lost': 0,
            'home_turnovers': 0,
            'home_penalties': 1,
            'home_yards_from_penalties': 6
        }
        flexmock(utils) \
            .should_receive('_todays_date') \
            .and_return(MockDateTime(YEAR, MONTH))

        self.boxscore = Boxscore(BOXSCORE)

    def test_ncaaf_boxscore_returns_requested_boxscore(self):
        for attribute, value in self.results.items():
            assert getattr(self.boxscore, attribute) == value
        assert getattr(self.boxscore, 'summary') == {
            'away': [14, 13, 10, 0],
            'home': [0, 0, 3, 10]
        }

    def test_invalid_url_yields_empty_class(self):
        flexmock(Boxscore) \
            .should_receive('_retrieve_html_page') \
            .and_return(None)

        boxscore = Boxscore(BOXSCORE)

        for key, value in boxscore.__dict__.items():
            if key == '_uri':
                continue
            assert value is None

    def test_ncaaf_boxscore_dataframe_returns_dataframe_of_all_values(self):
        df = pd.DataFrame([self.results], index=[BOXSCORE])

        # Pandas doesn't natively allow comparisons of DataFrames.
        # Concatenating the two DataFrames (the one generated during the test
        # and the expected one above) and dropping duplicate rows leaves only
        # the rows that are unique between the two frames. This allows a quick
        # check of the DataFrame to see if it is empty - if so, all rows are
        # duplicates, and they are equal.
        frames = [df, self.boxscore.dataframe]
        df1 = pd.concat(frames).drop_duplicates(keep=False)

        assert df1.empty

    def test_ncaaf_boxscore_players(self):
        boxscore = Boxscore(BOXSCORE)

        assert len(boxscore.home_players) == 37
        assert len(boxscore.away_players) == 45

        for player in boxscore.home_players:
            assert not player.dataframe.empty
        for player in boxscore.away_players:
            assert not player.dataframe.empty

    def test_ncaaf_boxscore_string_representation(self):
        expected = ('Boxscore for Clemson at Wake Forest '
                    '(Saturday Sep 12, 2020)')

        boxscore = Boxscore(BOXSCORE)

        assert boxscore.__repr__() == expected


class TestNCAAFBoxscores:
    def setup_method(self):
        self.expected = {
            '9-12-2020': [
                {'boxscore': '2020-09-12-appalachian-state',
                 'away_name': 'Charlotte',
                 'away_abbr': 'charlotte',
                 'away_score': 20,
                 'away_rank': None,
                 'home_name': 'Appalachian State',
                 'home_abbr': 'appalachian-state',
                 'home_score': 35,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Appalachian State',
                 'winning_abbr': 'appalachian-state',
                 'losing_name': 'Charlotte',
                 'losing_abbr': 'charlotte'},
                {'boxscore': '2020-09-12-army',
                 'away_name': 'Louisiana-Monroe',
                 'away_abbr': 'louisiana-monroe',
                 'away_score': 7,
                 'away_rank': None,
                 'home_name': 'Army',
                 'home_abbr': 'army',
                 'home_score': 37,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Army',
                 'winning_abbr': 'army',
                 'losing_name': 'Louisiana-Monroe',
                 'losing_abbr': 'louisiana-monroe'},
                {'boxscore': '2020-09-12-florida-state',
                 'away_name': 'Georgia Tech',
                 'away_abbr': 'georgia-tech',
                 'away_score': 16,
                 'away_rank': None,
                 'home_name': 'Florida State',
                 'home_abbr': 'florida-state',
                 'home_score': 13,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Georgia Tech',
                 'winning_abbr': 'georgia-tech',
                 'losing_name': 'Florida State',
                 'losing_abbr': 'florida-state'},
                {'boxscore': '2020-09-12-georgia-southern',
                 'away_name': 'Campbell',
                 'away_abbr': 'Campbell',
                 'away_score': 26,
                 'away_rank': None,
                 'home_name': 'Georgia Southern',
                 'home_abbr': 'georgia-southern',
                 'home_score': 27,
                 'home_rank': None,
                 'non_di': True,
                 'top_25': False,
                 'winning_name': 'Georgia Southern',
                 'winning_abbr': 'georgia-southern',
                 'losing_name': 'Campbell',
                 'losing_abbr': 'Campbell'},
                {'boxscore': '2020-09-12-iowa-state',
                 'away_name': 'Louisiana',
                 'away_abbr': 'louisiana-lafayette',
                 'away_score': 31,
                 'away_rank': None,
                 'home_name': 'Iowa State',
                 'home_abbr': 'iowa-state',
                 'home_score': 14,
                 'home_rank': 23,
                 'non_di': False,
                 'top_25': True,
                 'winning_name': 'Louisiana',
                 'winning_abbr': 'louisiana-lafayette',
                 'losing_name': 'Iowa State',
                 'losing_abbr': 'iowa-state'},
                {'boxscore': '2020-09-12-kansas',
                 'away_name': 'Coastal Carolina',
                 'away_abbr': 'coastal-carolina',
                 'away_score': 38,
                 'away_rank': None,
                 'home_name': 'Kansas',
                 'home_abbr': 'kansas',
                 'home_score': 23,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Coastal Carolina',
                 'winning_abbr': 'coastal-carolina',
                 'losing_name': 'Kansas',
                 'losing_abbr': 'kansas'},
                {'boxscore': '2020-09-12-kansas-state',
                 'away_name': 'Arkansas State',
                 'away_abbr': 'arkansas-state',
                 'away_score': 35,
                 'away_rank': None,
                 'home_name': 'Kansas State',
                 'home_abbr': 'kansas-state',
                 'home_score': 31,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Arkansas State',
                 'winning_abbr': 'arkansas-state',
                 'losing_name': 'Kansas State',
                 'losing_abbr': 'kansas-state'},
                {'boxscore': '2020-09-12-louisville',
                 'away_name': 'Western Kentucky',
                 'away_abbr': 'western-kentucky',
                 'away_score': 21,
                 'away_rank': None,
                 'home_name': 'Louisville',
                 'home_abbr': 'louisville',
                 'home_score': 35,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Louisville',
                 'winning_abbr': 'louisville',
                 'losing_name': 'Western Kentucky',
                 'losing_abbr': 'western-kentucky'},
                {'boxscore': '2020-09-12-north-carolina',
                 'away_name': 'Syracuse',
                 'away_abbr': 'syracuse',
                 'away_score': 6,
                 'away_rank': None,
                 'home_name': 'North Carolina',
                 'home_abbr': 'north-carolina',
                 'home_score': 31,
                 'home_rank': 18,
                 'non_di': False,
                 'top_25': True,
                 'winning_name': 'North Carolina',
                 'winning_abbr': 'north-carolina',
                 'losing_name': 'Syracuse',
                 'losing_abbr': 'syracuse'},
                {'boxscore': '2020-09-12-notre-dame',
                 'away_name': 'Duke',
                 'away_abbr': 'duke',
                 'away_score': 13,
                 'away_rank': None,
                 'home_name': 'Notre Dame',
                 'home_abbr': 'notre-dame',
                 'home_score': 27,
                 'home_rank': 10,
                 'non_di': False,
                 'top_25': True,
                 'winning_name': 'Notre Dame',
                 'winning_abbr': 'notre-dame',
                 'losing_name': 'Duke',
                 'losing_abbr': 'duke'},
                {'boxscore': '2020-09-12-oklahoma',
                 'away_name': 'Missouri State',
                 'away_abbr': 'Missouri State',
                 'away_score': 0,
                 'away_rank': None,
                 'home_name': 'Oklahoma',
                 'home_abbr': 'oklahoma',
                 'home_score': 48,
                 'home_rank': 5,
                 'non_di': True,
                 'top_25': True,
                 'winning_name': None,
                 'winning_abbr': None,
                 'losing_name': None,
                 'losing_abbr': None},
                {'boxscore': '2020-09-12-pittsburgh',
                 'away_name': 'Austin Peay',
                 'away_abbr': 'Austin Peay',
                 'away_score': 0,
                 'away_rank': None,
                 'home_name': 'Pitt',
                 'home_abbr': 'pittsburgh',
                 'home_score': 55,
                 'home_rank': None,
                 'non_di': True,
                 'top_25': False,
                 'winning_name': None,
                 'winning_abbr': None,
                 'losing_name': None,
                 'losing_abbr': None},
                {'boxscore': '2020-09-12-south-alabama',
                 'away_name': 'Tulane',
                 'away_abbr': 'tulane',
                 'away_score': 27,
                 'away_rank': None,
                 'home_name': 'South Alabama',
                 'home_abbr': 'south-alabama',
                 'home_score': 24,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Tulane',
                 'winning_abbr': 'tulane',
                 'losing_name': 'South Alabama',
                 'losing_abbr': 'south-alabama'},
                {'boxscore': '2020-09-12-south-florida',
                 'away_name': 'Citadel',
                 'away_abbr': 'Citadel',
                 'away_score': 6,
                 'away_rank': None,
                 'home_name': 'South Florida',
                 'home_abbr': 'south-florida',
                 'home_score': 27,
                 'home_rank': None,
                 'non_di': True,
                 'top_25': False,
                 'winning_name': 'South Florida',
                 'winning_abbr': 'south-florida',
                 'losing_name': 'Citadel',
                 'losing_abbr': 'Citadel'},
                {'boxscore': '2020-09-12-texas',
                 'away_name': 'UTEP',
                 'away_abbr': 'texas-el-paso',
                 'away_score': 3,
                 'away_rank': None,
                 'home_name': 'Texas',
                 'home_abbr': 'texas',
                 'home_score': 59,
                 'home_rank': 14,
                 'non_di': False,
                 'top_25': True,
                 'winning_name': 'Texas',
                 'winning_abbr': 'texas',
                 'losing_name': 'UTEP',
                 'losing_abbr': 'texas-el-paso'},
                {'boxscore': '2020-09-12-texas-state',
                 'away_name': 'UTSA',
                 'away_abbr': 'texas-san-antonio',
                 'away_score': 51,
                 'away_rank': None,
                 'home_name': 'Texas State',
                 'home_abbr': 'texas-state',
                 'home_score': 48,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'UTSA',
                 'winning_abbr': 'texas-san-antonio',
                 'losing_name': 'Texas State',
                 'losing_abbr': 'texas-state'},
                {'boxscore': '2020-09-12-texas-tech',
                 'away_name': 'Houston Baptist',
                 'away_abbr': 'Houston Baptist',
                 'away_score': 33,
                 'away_rank': None,
                 'home_name': 'Texas Tech',
                 'home_abbr': 'texas-tech',
                 'home_score': 35,
                 'home_rank': None,
                 'non_di': True,
                 'top_25': False,
                 'winning_name': 'Texas Tech',
                 'winning_abbr': 'texas-tech',
                 'losing_name': 'Houston Baptist',
                 'losing_abbr': 'Houston Baptist'},
                {'boxscore': '2020-09-12-wake-forest',
                 'away_name': 'Clemson',
                 'away_abbr': 'clemson',
                 'away_score': 37,
                 'away_rank': 1,
                 'home_name': 'Wake Forest',
                 'home_abbr': 'wake-forest',
                 'home_score': 13,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': True,
                 'winning_name': 'Clemson',
                 'winning_abbr': 'clemson',
                 'losing_name': 'Wake Forest',
                 'losing_abbr': 'wake-forest'},
                {'boxscore': '2020-09-12-west-virginia',
                 'away_name': 'Eastern Kentucky',
                 'away_abbr': 'Eastern Kentucky',
                 'away_score': 10,
                 'away_rank': None,
                 'home_name': 'West Virginia',
                 'home_abbr': 'west-virginia',
                 'home_score': 56,
                 'home_rank': None,
                 'non_di': True,
                 'top_25': False,
                 'winning_name': 'West Virginia',
                 'winning_abbr': 'west-virginia',
                 'losing_name': 'Eastern Kentucky',
                 'losing_abbr': 'Eastern Kentucky'},
                {'boxscore': '2020-09-10-miami-fl',
                 'away_name': 'UAB',
                 'away_abbr': 'alabama-birmingham',
                 'away_score': 14,
                 'away_rank': None,
                 'home_name': 'Miami (FL)',
                 'home_abbr': 'miami-fl',
                 'home_score': 31,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Miami (FL)',
                 'winning_abbr': 'miami-fl',
                 'losing_name': 'UAB',
                 'losing_abbr': 'alabama-birmingham'}
            ]
        }

    @mock.patch('requests.get', side_effect=mock_pyquery)
    def test_boxscores_search(self, *args, **kwargs):
        result = Boxscores(datetime(2020, 9, 12)).games

        assert result == self.expected

    @mock.patch('requests.get', side_effect=mock_pyquery)
    def test_boxscores_search_invalid_end(self, *args, **kwargs):
        result = Boxscores(datetime(2020, 9, 12), datetime(2020, 9, 11)).games

        assert result == self.expected

    @mock.patch('requests.get', side_effect=mock_pyquery)
    def test_boxscores_search_multiple_days(self, *args, **kwargs):
        expected = {
            '9-12-2020': [
                {'boxscore': '2020-09-12-appalachian-state',
                 'away_name': 'Charlotte',
                 'away_abbr': 'charlotte',
                 'away_score': 20,
                 'away_rank': None,
                 'home_name': 'Appalachian State',
                 'home_abbr': 'appalachian-state',
                 'home_score': 35,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Appalachian State',
                 'winning_abbr': 'appalachian-state',
                 'losing_name': 'Charlotte',
                 'losing_abbr': 'charlotte'},
                {'boxscore': '2020-09-12-army',
                 'away_name': 'Louisiana-Monroe',
                 'away_abbr': 'louisiana-monroe',
                 'away_score': 7,
                 'away_rank': None,
                 'home_name': 'Army',
                 'home_abbr': 'army',
                 'home_score': 37,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Army',
                 'winning_abbr': 'army',
                 'losing_name': 'Louisiana-Monroe',
                 'losing_abbr': 'louisiana-monroe'},
                {'boxscore': '2020-09-12-florida-state',
                 'away_name': 'Georgia Tech',
                 'away_abbr': 'georgia-tech',
                 'away_score': 16,
                 'away_rank': None,
                 'home_name': 'Florida State',
                 'home_abbr': 'florida-state',
                 'home_score': 13,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Georgia Tech',
                 'winning_abbr': 'georgia-tech',
                 'losing_name': 'Florida State',
                 'losing_abbr': 'florida-state'},
                {'boxscore': '2020-09-12-georgia-southern',
                 'away_name': 'Campbell',
                 'away_abbr': 'Campbell',
                 'away_score': 26,
                 'away_rank': None,
                 'home_name': 'Georgia Southern',
                 'home_abbr': 'georgia-southern',
                 'home_score': 27,
                 'home_rank': None,
                 'non_di': True,
                 'top_25': False,
                 'winning_name': 'Georgia Southern',
                 'winning_abbr': 'georgia-southern',
                 'losing_name': 'Campbell',
                 'losing_abbr': 'Campbell'},
                {'boxscore': '2020-09-12-iowa-state',
                 'away_name': 'Louisiana',
                 'away_abbr': 'louisiana-lafayette',
                 'away_score': 31,
                 'away_rank': None,
                 'home_name': 'Iowa State',
                 'home_abbr': 'iowa-state',
                 'home_score': 14,
                 'home_rank': 23,
                 'non_di': False,
                 'top_25': True,
                 'winning_name': 'Louisiana',
                 'winning_abbr': 'louisiana-lafayette',
                 'losing_name': 'Iowa State',
                 'losing_abbr': 'iowa-state'},
                {'boxscore': '2020-09-12-kansas',
                 'away_name': 'Coastal Carolina',
                 'away_abbr': 'coastal-carolina',
                 'away_score': 38,
                 'away_rank': None,
                 'home_name': 'Kansas',
                 'home_abbr': 'kansas',
                 'home_score': 23,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Coastal Carolina',
                 'winning_abbr': 'coastal-carolina',
                 'losing_name': 'Kansas',
                 'losing_abbr': 'kansas'},
                {'boxscore': '2020-09-12-kansas-state',
                 'away_name': 'Arkansas State',
                 'away_abbr': 'arkansas-state',
                 'away_score': 35,
                 'away_rank': None,
                 'home_name': 'Kansas State',
                 'home_abbr': 'kansas-state',
                 'home_score': 31,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Arkansas State',
                 'winning_abbr': 'arkansas-state',
                 'losing_name': 'Kansas State',
                 'losing_abbr': 'kansas-state'},
                {'boxscore': '2020-09-12-louisville',
                 'away_name': 'Western Kentucky',
                 'away_abbr': 'western-kentucky',
                 'away_score': 21,
                 'away_rank': None,
                 'home_name': 'Louisville',
                 'home_abbr': 'louisville',
                 'home_score': 35,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Louisville',
                 'winning_abbr': 'louisville',
                 'losing_name': 'Western Kentucky',
                 'losing_abbr': 'western-kentucky'},
                {'boxscore': '2020-09-12-north-carolina',
                 'away_name': 'Syracuse',
                 'away_abbr': 'syracuse',
                 'away_score': 6,
                 'away_rank': None,
                 'home_name': 'North Carolina',
                 'home_abbr': 'north-carolina',
                 'home_score': 31,
                 'home_rank': 18,
                 'non_di': False,
                 'top_25': True,
                 'winning_name': 'North Carolina',
                 'winning_abbr': 'north-carolina',
                 'losing_name': 'Syracuse',
                 'losing_abbr': 'syracuse'},
                {'boxscore': '2020-09-12-notre-dame',
                 'away_name': 'Duke',
                 'away_abbr': 'duke',
                 'away_score': 13,
                 'away_rank': None,
                 'home_name': 'Notre Dame',
                 'home_abbr': 'notre-dame',
                 'home_score': 27,
                 'home_rank': 10,
                 'non_di': False,
                 'top_25': True,
                 'winning_name': 'Notre Dame',
                 'winning_abbr': 'notre-dame',
                 'losing_name': 'Duke',
                 'losing_abbr': 'duke'},
                {'boxscore': '2020-09-12-oklahoma',
                 'away_name': 'Missouri State',
                 'away_abbr': 'Missouri State',
                 'away_score': 0,
                 'away_rank': None,
                 'home_name': 'Oklahoma',
                 'home_abbr': 'oklahoma',
                 'home_score': 48,
                 'home_rank': 5,
                 'non_di': True,
                 'top_25': True,
                 'winning_name': None,
                 'winning_abbr': None,
                 'losing_name': None,
                 'losing_abbr': None},
                {'boxscore': '2020-09-12-pittsburgh',
                 'away_name': 'Austin Peay',
                 'away_abbr': 'Austin Peay',
                 'away_score': 0,
                 'away_rank': None,
                 'home_name': 'Pitt',
                 'home_abbr': 'pittsburgh',
                 'home_score': 55,
                 'home_rank': None,
                 'non_di': True,
                 'top_25': False,
                 'winning_name': None,
                 'winning_abbr': None,
                 'losing_name': None,
                 'losing_abbr': None},
                {'boxscore': '2020-09-12-south-alabama',
                 'away_name': 'Tulane',
                 'away_abbr': 'tulane',
                 'away_score': 27,
                 'away_rank': None,
                 'home_name': 'South Alabama',
                 'home_abbr': 'south-alabama',
                 'home_score': 24,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Tulane',
                 'winning_abbr': 'tulane',
                 'losing_name': 'South Alabama',
                 'losing_abbr': 'south-alabama'},
                {'boxscore': '2020-09-12-south-florida',
                 'away_name': 'Citadel',
                 'away_abbr': 'Citadel',
                 'away_score': 6,
                 'away_rank': None,
                 'home_name': 'South Florida',
                 'home_abbr': 'south-florida',
                 'home_score': 27,
                 'home_rank': None,
                 'non_di': True,
                 'top_25': False,
                 'winning_name': 'South Florida',
                 'winning_abbr': 'south-florida',
                 'losing_name': 'Citadel',
                 'losing_abbr': 'Citadel'},
                {'boxscore': '2020-09-12-texas',
                 'away_name': 'UTEP',
                 'away_abbr': 'texas-el-paso',
                 'away_score': 3,
                 'away_rank': None,
                 'home_name': 'Texas',
                 'home_abbr': 'texas',
                 'home_score': 59,
                 'home_rank': 14,
                 'non_di': False,
                 'top_25': True,
                 'winning_name': 'Texas',
                 'winning_abbr': 'texas',
                 'losing_name': 'UTEP',
                 'losing_abbr': 'texas-el-paso'},
                {'boxscore': '2020-09-12-texas-state',
                 'away_name': 'UTSA',
                 'away_abbr': 'texas-san-antonio',
                 'away_score': 51,
                 'away_rank': None,
                 'home_name': 'Texas State',
                 'home_abbr': 'texas-state',
                 'home_score': 48,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'UTSA',
                 'winning_abbr': 'texas-san-antonio',
                 'losing_name': 'Texas State',
                 'losing_abbr': 'texas-state'},
                {'boxscore': '2020-09-12-texas-tech',
                 'away_name': 'Houston Baptist',
                 'away_abbr': 'Houston Baptist',
                 'away_score': 33,
                 'away_rank': None,
                 'home_name': 'Texas Tech',
                 'home_abbr': 'texas-tech',
                 'home_score': 35,
                 'home_rank': None,
                 'non_di': True,
                 'top_25': False,
                 'winning_name': 'Texas Tech',
                 'winning_abbr': 'texas-tech',
                 'losing_name': 'Houston Baptist',
                 'losing_abbr': 'Houston Baptist'},
                {'boxscore': '2020-09-12-wake-forest',
                 'away_name': 'Clemson',
                 'away_abbr': 'clemson',
                 'away_score': 37,
                 'away_rank': 1,
                 'home_name': 'Wake Forest',
                 'home_abbr': 'wake-forest',
                 'home_score': 13,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': True,
                 'winning_name': 'Clemson',
                 'winning_abbr': 'clemson',
                 'losing_name': 'Wake Forest',
                 'losing_abbr': 'wake-forest'},
                {'boxscore': '2020-09-12-west-virginia',
                 'away_name': 'Eastern Kentucky',
                 'away_abbr': 'Eastern Kentucky',
                 'away_score': 10,
                 'away_rank': None,
                 'home_name': 'West Virginia',
                 'home_abbr': 'west-virginia',
                 'home_score': 56,
                 'home_rank': None,
                 'non_di': True,
                 'top_25': False,
                 'winning_name': 'West Virginia',
                 'winning_abbr': 'west-virginia',
                 'losing_name': 'Eastern Kentucky',
                 'losing_abbr': 'Eastern Kentucky'},
                {'boxscore': '2020-09-10-miami-fl',
                 'away_name': 'UAB',
                 'away_abbr': 'alabama-birmingham',
                 'away_score': 14,
                 'away_rank': None,
                 'home_name': 'Miami (FL)',
                 'home_abbr': 'miami-fl',
                 'home_score': 31,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Miami (FL)',
                 'winning_abbr': 'miami-fl',
                 'losing_name': 'UAB',
                 'losing_abbr': 'alabama-birmingham'}
            ],
            '9-13-2020': [
                {'boxscore': '2020-09-12-appalachian-state',
                 'away_name': 'Charlotte',
                 'away_abbr': 'charlotte',
                 'away_score': 20,
                 'away_rank': None,
                 'home_name': 'Appalachian State',
                 'home_abbr': 'appalachian-state',
                 'home_score': 35,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Appalachian State',
                 'winning_abbr': 'appalachian-state',
                 'losing_name': 'Charlotte',
                 'losing_abbr': 'charlotte'},
                {'boxscore': '2020-09-12-army',
                 'away_name': 'Louisiana-Monroe',
                 'away_abbr': 'louisiana-monroe',
                 'away_score': 7,
                 'away_rank': None,
                 'home_name': 'Army',
                 'home_abbr': 'army',
                 'home_score': 37,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Army',
                 'winning_abbr': 'army',
                 'losing_name': 'Louisiana-Monroe',
                 'losing_abbr': 'louisiana-monroe'},
                {'boxscore': '2020-09-12-florida-state',
                 'away_name': 'Georgia Tech',
                 'away_abbr': 'georgia-tech',
                 'away_score': 16,
                 'away_rank': None,
                 'home_name': 'Florida State',
                 'home_abbr': 'florida-state',
                 'home_score': 13,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Georgia Tech',
                 'winning_abbr': 'georgia-tech',
                 'losing_name': 'Florida State',
                 'losing_abbr': 'florida-state'},
                {'boxscore': '2020-09-12-georgia-southern',
                 'away_name': 'Campbell',
                 'away_abbr': 'Campbell',
                 'away_score': 26,
                 'away_rank': None,
                 'home_name': 'Georgia Southern',
                 'home_abbr': 'georgia-southern',
                 'home_score': 27,
                 'home_rank': None,
                 'non_di': True,
                 'top_25': False,
                 'winning_name': 'Georgia Southern',
                 'winning_abbr': 'georgia-southern',
                 'losing_name': 'Campbell',
                 'losing_abbr': 'Campbell'},
                {'boxscore': '2020-09-12-iowa-state',
                 'away_name': 'Louisiana',
                 'away_abbr': 'louisiana-lafayette',
                 'away_score': 31,
                 'away_rank': None,
                 'home_name': 'Iowa State',
                 'home_abbr': 'iowa-state',
                 'home_score': 14,
                 'home_rank': 23,
                 'non_di': False,
                 'top_25': True,
                 'winning_name': 'Louisiana',
                 'winning_abbr': 'louisiana-lafayette',
                 'losing_name': 'Iowa State',
                 'losing_abbr': 'iowa-state'},
                {'boxscore': '2020-09-12-kansas',
                 'away_name': 'Coastal Carolina',
                 'away_abbr': 'coastal-carolina',
                 'away_score': 38,
                 'away_rank': None,
                 'home_name': 'Kansas',
                 'home_abbr': 'kansas',
                 'home_score': 23,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Coastal Carolina',
                 'winning_abbr': 'coastal-carolina',
                 'losing_name': 'Kansas',
                 'losing_abbr': 'kansas'},
                {'boxscore': '2020-09-12-kansas-state',
                 'away_name': 'Arkansas State',
                 'away_abbr': 'arkansas-state',
                 'away_score': 35,
                 'away_rank': None,
                 'home_name': 'Kansas State',
                 'home_abbr': 'kansas-state',
                 'home_score': 31,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Arkansas State',
                 'winning_abbr': 'arkansas-state',
                 'losing_name': 'Kansas State',
                 'losing_abbr': 'kansas-state'},
                {'boxscore': '2020-09-12-louisville',
                 'away_name': 'Western Kentucky',
                 'away_abbr': 'western-kentucky',
                 'away_score': 21,
                 'away_rank': None,
                 'home_name': 'Louisville',
                 'home_abbr': 'louisville',
                 'home_score': 35,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Louisville',
                 'winning_abbr': 'louisville',
                 'losing_name': 'Western Kentucky',
                 'losing_abbr': 'western-kentucky'},
                {'boxscore': '2020-09-12-north-carolina',
                 'away_name': 'Syracuse',
                 'away_abbr': 'syracuse',
                 'away_score': 6,
                 'away_rank': None,
                 'home_name': 'North Carolina',
                 'home_abbr': 'north-carolina',
                 'home_score': 31,
                 'home_rank': 18,
                 'non_di': False,
                 'top_25': True,
                 'winning_name': 'North Carolina',
                 'winning_abbr': 'north-carolina',
                 'losing_name': 'Syracuse',
                 'losing_abbr': 'syracuse'},
                {'boxscore': '2020-09-12-notre-dame',
                 'away_name': 'Duke',
                 'away_abbr': 'duke',
                 'away_score': 13,
                 'away_rank': None,
                 'home_name': 'Notre Dame',
                 'home_abbr': 'notre-dame',
                 'home_score': 27,
                 'home_rank': 10,
                 'non_di': False,
                 'top_25': True,
                 'winning_name': 'Notre Dame',
                 'winning_abbr': 'notre-dame',
                 'losing_name': 'Duke',
                 'losing_abbr': 'duke'},
                {'boxscore': '2020-09-12-oklahoma',
                 'away_name': 'Missouri State',
                 'away_abbr': 'Missouri State',
                 'away_score': 0,
                 'away_rank': None,
                 'home_name': 'Oklahoma',
                 'home_abbr': 'oklahoma',
                 'home_score': 48,
                 'home_rank': 5,
                 'non_di': True,
                 'top_25': True,
                 'winning_name': None,
                 'winning_abbr': None,
                 'losing_name': None,
                 'losing_abbr': None},
                {'boxscore': '2020-09-12-pittsburgh',
                 'away_name': 'Austin Peay',
                 'away_abbr': 'Austin Peay',
                 'away_score': 0,
                 'away_rank': None,
                 'home_name': 'Pitt',
                 'home_abbr': 'pittsburgh',
                 'home_score': 55,
                 'home_rank': None,
                 'non_di': True,
                 'top_25': False,
                 'winning_name': None,
                 'winning_abbr': None,
                 'losing_name': None,
                 'losing_abbr': None},
                {'boxscore': '2020-09-12-south-alabama',
                 'away_name': 'Tulane',
                 'away_abbr': 'tulane',
                 'away_score': 27,
                 'away_rank': None,
                 'home_name': 'South Alabama',
                 'home_abbr': 'south-alabama',
                 'home_score': 24,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Tulane',
                 'winning_abbr': 'tulane',
                 'losing_name': 'South Alabama',
                 'losing_abbr': 'south-alabama'},
                {'boxscore': '2020-09-12-south-florida',
                 'away_name': 'Citadel',
                 'away_abbr': 'Citadel',
                 'away_score': 6,
                 'away_rank': None,
                 'home_name': 'South Florida',
                 'home_abbr': 'south-florida',
                 'home_score': 27,
                 'home_rank': None,
                 'non_di': True,
                 'top_25': False,
                 'winning_name': 'South Florida',
                 'winning_abbr': 'south-florida',
                 'losing_name': 'Citadel',
                 'losing_abbr': 'Citadel'},
                {'boxscore': '2020-09-12-texas',
                 'away_name': 'UTEP',
                 'away_abbr': 'texas-el-paso',
                 'away_score': 3,
                 'away_rank': None,
                 'home_name': 'Texas',
                 'home_abbr': 'texas',
                 'home_score': 59,
                 'home_rank': 14,
                 'non_di': False,
                 'top_25': True,
                 'winning_name': 'Texas',
                 'winning_abbr': 'texas',
                 'losing_name': 'UTEP',
                 'losing_abbr': 'texas-el-paso'},
                {'boxscore': '2020-09-12-texas-state',
                 'away_name': 'UTSA',
                 'away_abbr': 'texas-san-antonio',
                 'away_score': 51,
                 'away_rank': None,
                 'home_name': 'Texas State',
                 'home_abbr': 'texas-state',
                 'home_score': 48,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'UTSA',
                 'winning_abbr': 'texas-san-antonio',
                 'losing_name': 'Texas State',
                 'losing_abbr': 'texas-state'},
                {'boxscore': '2020-09-12-texas-tech',
                 'away_name': 'Houston Baptist',
                 'away_abbr': 'Houston Baptist',
                 'away_score': 33,
                 'away_rank': None,
                 'home_name': 'Texas Tech',
                 'home_abbr': 'texas-tech',
                 'home_score': 35,
                 'home_rank': None,
                 'non_di': True,
                 'top_25': False,
                 'winning_name': 'Texas Tech',
                 'winning_abbr': 'texas-tech',
                 'losing_name': 'Houston Baptist',
                 'losing_abbr': 'Houston Baptist'},
                {'boxscore': '2020-09-12-wake-forest',
                 'away_name': 'Clemson',
                 'away_abbr': 'clemson',
                 'away_score': 37,
                 'away_rank': 1,
                 'home_name': 'Wake Forest',
                 'home_abbr': 'wake-forest',
                 'home_score': 13,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': True,
                 'winning_name': 'Clemson',
                 'winning_abbr': 'clemson',
                 'losing_name': 'Wake Forest',
                 'losing_abbr': 'wake-forest'},
                {'boxscore': '2020-09-12-west-virginia',
                 'away_name': 'Eastern Kentucky',
                 'away_abbr': 'Eastern Kentucky',
                 'away_score': 10,
                 'away_rank': None,
                 'home_name': 'West Virginia',
                 'home_abbr': 'west-virginia',
                 'home_score': 56,
                 'home_rank': None,
                 'non_di': True,
                 'top_25': False,
                 'winning_name': 'West Virginia',
                 'winning_abbr': 'west-virginia',
                 'losing_name': 'Eastern Kentucky',
                 'losing_abbr': 'Eastern Kentucky'},
                {'boxscore': '2020-09-10-miami-fl',
                 'away_name': 'UAB',
                 'away_abbr': 'alabama-birmingham',
                 'away_score': 14,
                 'away_rank': None,
                 'home_name': 'Miami (FL)',
                 'home_abbr': 'miami-fl',
                 'home_score': 31,
                 'home_rank': None,
                 'non_di': False,
                 'top_25': False,
                 'winning_name': 'Miami (FL)',
                 'winning_abbr': 'miami-fl',
                 'losing_name': 'UAB',
                 'losing_abbr': 'alabama-birmingham'}
            ]
        }
        result = Boxscores(datetime(2020, 9, 12), datetime(2020, 9, 13)).games

        assert result == expected

    @mock.patch('requests.get', side_effect=mock_pyquery)
    def test_boxscores_search_string_representation(self, *args, **kwargs):
        result = Boxscores(datetime(2020, 9, 12))

        assert result.__repr__() == 'NCAAF games for 9-12-2020'
