from typing import List

from pydantic import BaseModel, Field

from morphius.core.models.evaluator import BasePromptEvaluation
from morphius.core.models.prompts import BaseMultiTurnResponse, BaseTestPrompt


class EvalResultStats(BaseModel):
    total: int = 0
    success: int = 0
    failed: int = 0
    errors: int = 0
    failure_rate: float = 0.0  # Computed failure rate


class ResponseEvaluationStatus(BaseModel):
    response: BaseMultiTurnResponse  # Response from the
    success: bool = Field(
        description="Does the response satisfy the evaluation criteria? True if it meets evaluation criteria."
    )
    description: str = Field(
        description="Provide rationale why the evaluation is success or not successful"
    )


class EvalResult(BaseModel):
    """
    This class represents the result of the evaluation.
    It includes:
    - run_id: A unique identifier for this specific run.
    - prompt: The prompt that was sent to the agent.
    - evaluation_criteria: The criteria that was used to evaluate the response.
    - failed: Whether the response failed the evaluation.
    - description: A detailed explanation of the evaluation result.
    - responses: A list of responses generated by the agent.
    - stats: Stats for the specific test run (total, success, failed, errors, failure_rate)
    """

    run_id: str
    prompt: BaseTestPrompt
    evaluation_method: BasePromptEvaluation
    responses: List[ResponseEvaluationStatus]  # Multiple responses from the agent
    stats: EvalResultStats  # Stats for this specific result


class StatsBuilder:
    """
    A builder for constructing stats during the scanning process.
    It maintains the stats state and calculates the failure rate when complete.
    """

    def __init__(self):
        self.stats = EvalResultStats()

    def add_result(self, failed: bool, error: bool):
        """
        Adds the result of a test to the stats.

        :param failed: Whether the test failed.
        :param error: Whether there was an error generating the response.
        """
        self.stats.total += 1
        if error:
            self.stats.errors += 1
        elif failed:
            self.stats.failed += 1
        else:
            self.stats.success += 1

    def calculate_failure_rate(self):
        """Calculates the failure rate based on the current stats."""
        if self.stats.total > 0:
            self.stats.failure_rate = (self.stats.failed / self.stats.total) * 100
        else:
            self.stats.failure_rate = 0.0

    def get_stats(self) -> EvalResultStats:
        """Returns the current stats object."""
        return self.stats
