import torch
from torch import nn
import fmot
import pytest


class OutputNameTestCase(nn.Module):
    expected_names: list[str]
    n_inputs: int
    xfail = False
    xfail_reason = ""


class UserNames(OutputNameTestCase):
    expected_names = ["c", "d"]
    n_inputs = 2

    def forward(self, a, b):
        c = a + b
        d = a - b
        c = fmot.tag(c, "c")
        d = fmot.tag(d, "d")
        return c, d


class PartialUserNames(OutputNameTestCase):
    expected_names = ["c", "output.1"]
    n_inputs = 2

    def forward(self, a, b):
        c = a + b
        d = a - b
        c = fmot.tag(c, "c")
        return c, d


class RepeatedUntagged(OutputNameTestCase):
    # expect that FQIR passes will create a copy of the output
    # and duplicate it
    expected_names = ["output.0", "output.1"]
    n_inputs = 1

    def forward(self, a):
        c = a + 3
        return c, c


class OutputList(OutputNameTestCase):
    expected_names = ["a", "b", "output.2", "d", "output.4"]
    n_inputs = 1

    def forward(self, x):
        a = x + 1
        b = x + 2
        c = x + 3
        d = x + 4
        e = x + 5

        fmot.tag(a, "a")
        fmot.tag(b, "b")
        fmot.tag(d, "d")

        return a, [b, c], [d, e]


class KernelizedNonlinearity(OutputNameTestCase):
    # had to make a change to kernelize-lut to make sure that autogen names
    # prepend the % for lut variables
    expected_names = ["output.0"]
    n_inputs = 1

    def forward(self, x):
        return torch.sigmoid(x)


class UserNameMatchesAutogenPattern(OutputNameTestCase):
    expected_names = ["%my_output"]
    n_inputs = 1
    xfail = True
    xfail_reason = r"current tools recognize names starting with % as autogenerated"

    def forward(self, x):
        y = x + 1
        # autogen names start with %... this will be incorrectly changed
        # to output.0
        fmot.tag(y, "%my_output")
        return y


@pytest.mark.parametrize(
    "testcase",
    [
        UserNames,
        PartialUserNames,
        RepeatedUntagged,
        OutputList,
        UserNameMatchesAutogenPattern,
        KernelizedNonlinearity,
    ],
)
def test_output_naming(testcase: type[OutputNameTestCase]):
    model = testcase()

    if model.xfail:
        pytest.xfail(model.xfail_reason)

    cmodel = fmot.ConvertedModel(model)

    calib = []
    for i in range(4):
        sample = []
        for _ in range(model.n_inputs):
            sample.append(torch.randn(8, 16))
        calib.append(tuple(sample))

    cmodel.quantize(calib)
    graph = cmodel.trace()

    # test that we get the expected output names
    arith = graph.subgraphs["ARITH"]
    for i, x in enumerate(arith.outputs):
        expected = model.expected_names[i]
        if x.name != expected:
            raise ValueError(f"Expected output name {i} to be {expected}, got {x.name}")


if __name__ == "__main__":
    test_output_naming(UserNames)
