# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/04_fragments.ipynb.

# %% ../nbs/04_fragments.ipynb 3
from __future__ import annotations
from .imports import *
from .utils import *
from .chem import to_mol, to_smile, canon_smile, Molecule

from rdkit import Chem
from rdkit.Chem import rdMMPA

# %% auto 0
__all__ = ['fuse_mol_on_atom_mapping', 'fuse_smile_on_atom_mapping', 'remove_fragment_mapping', 'add_fragment_mapping',
           'get_dummy_mol', 'combine_dummies', 'is_mapped', 'match_mapping', 'generate_mapping_permutations',
           'match_and_map', 'fragment_smile', 'clean_fragments', 'shred_smiles']

# %% ../nbs/04_fragments.ipynb 4
def fuse_mol_on_atom_mapping(mol: Chem.Mol # input rdkit Mol
                            ) -> Union[Chem.Mol, None]: # output fused Mol, returns None if failed
    '''
    Attempts to fuse mapped molecules into a single molecule (ie `[*:1]C.[*:1]N -> CN`). 
    Returns None if fusion failed
    '''
    try:
        return Chem.molzip(mol)
    except:
        return None
    
def fuse_smile_on_atom_mapping(smile: str # input SMILES string
                              ) -> str: # output fused SMILES string
    '''
    Attempts to fuse mapped SMILES into a single molecule (ie `[*:1]C.[*:1]N -> CN`). 
    Returns None if fusion failed
    '''
    mol = to_mol(smile)
    mol = fuse_mol_on_atom_mapping(mol)
    if mol is not None:
        return to_smile(mol)
    else:
        return ''

# %% ../nbs/04_fragments.ipynb 6
def remove_fragment_mapping(smile: str # mapped SMILES string
                           ) -> str: # unmapped SMILES string
    patt = re.compile('\[\*(.*?)]')
    smile = patt.sub('[*]', smile)
    return canon_smile(smile)

def add_fragment_mapping(smile:    str, # SMILES string 
                         map_nums: list[int] # fragment mapping ints
                        ) -> str: # mapped SMILES
    
    '''
    Given an unmapped fragment SMILES string and a list of mapping ints, 
    adds mapping to SMILES.
    
    ie `add_fragment_mapping('*C*', [3,4]) -> [*:3]C[*:4]`
    
    Number of * dummy atoms should match length of `map_nums`
    '''
    
    assert smile.count('*') == len(map_nums)
    
    smile = remove_fragment_mapping(smile)
    
    new_smile = ''
    map_idx = 0
    for char in smile:
        if char=='*':
            new_smile += f'[*:{map_nums[map_idx]}]'
            map_idx += 1
        else:
            new_smile += char
            
    return canon_smile(new_smile)

# %% ../nbs/04_fragments.ipynb 8
def get_dummy_mol(name:     str, # dummy name
                  map_nums: list[int], # dummy mapping nums
                  id:       Optional[int]=None # optional dummy ID
                 ) -> Chem.Mol: # returns dummy mol
    templates = {
        0 : '[Zr]',
        1 : '[*][Zr]',
        2 : '[*][Zr][*]',
        3 : '[*][Zr]([*])[*]',
        4 : '[*][Zr]([*])([*])[*]'
    }
    
    num_attachments = len(map_nums)
    mapping_idx = 0
    template = templates[num_attachments]
    template = add_fragment_mapping(template, map_nums)
    
    if id is not None:
        template = template.replace('Zr', f'Zr:{id}')
            
    mol = to_mol(template)
    for atom in mol.GetAtoms():
        if atom.GetAtomicNum() != 0:
            atom.SetProp('atomLabel', name)
            
    return mol

def combine_dummies(dummies: list[Chem.Mol], # list of dummy mols 
                    fuse:    bool=True # if mols should be fused
                   ) -> Chem.Mol: # returns output mol
    combo = Chem.MolFromSmiles('')
    for mol in dummies:
        combo = Chem.CombineMols(combo, mol)
        
    if fuse:
        combo = Chem.molzip(combo)
        
    return combo

# %% ../nbs/04_fragments.ipynb 10
def is_mapped(smile: str # SMILES string
             ) -> bool: # True if mapped, else False
    
    'determines mapping status by matching number of * dummy atoms with number of [*:x] mapping IDs'
    
    patt = re.compile('\[\*(.*?)]')
    return len(patt.findall(smile)) == smile.count('*')

def match_mapping(molecule: Molecule, # input Molecule
                  mapping_idxs: list[int] # mapping ints
                 ) -> bool: # True if mapping matches, else False
    patt = re.compile('\[\*(.*?)]')
    frag_idxs = [int(i[1:]) for i in patt.findall(molecule.smile)]
    return sorted(frag_idxs) == sorted(mapping_idxs)

# %% ../nbs/04_fragments.ipynb 12
def generate_mapping_permutations(smile:    str, # SMILES string
                                  map_nums: list[int], # possible mapping ints
                                  exact:    bool=False # if True, number of `map_nums` must match number of `*` atoms
                                 ) -> list[str]: # list of mapped SMILES
    
    n_attachments = smile.count('*')
    
    if map_nums is None:
        map_nums = list(range(1, n_attachments+1))
    
    if exact:
        assert n_attachments == len(map_nums)
    else:
        assert n_attachments <= len(map_nums)
    
    perms = permutations(map_nums, n_attachments)
    outputs = []
    for p in perms:
        outputs.append(add_fragment_mapping(smile, p))
        
    return outputs

# %% ../nbs/04_fragments.ipynb 14
def match_and_map(fragment: str, # fragment SMILES
                  mapping_idxs: list[int] # mapping ints
                 ) -> list[str]: # list of mapped SMILES
    outputs = []
    if is_mapped(fragment):
        if match_mapping(Molecule(fragment), mapping_idxs):
            outputs = [fragment]
    elif fragment.count('*')==len(mapping_idxs):
        outputs = generate_mapping_permutations(fragment, mapping_idxs, exact=True)
    
    return outputs

# %% ../nbs/04_fragments.ipynb 16
def fragment_smile(smile: str, # input SMILES string
                   cuts: list[int] # number of cuts, ie [1,2,3]
                  ) -> list[str]: # list of fragments
    mol = to_mol(smile)
    fragments = []
    for cut in cuts:
        frags = rdMMPA.FragmentMol(mol, maxCuts=cut, resultsAsMols=False)
        frags = deduplicate_list(flatten_list(frags))
        fragments += frags
        
    fragments = deduplicate_list(fragments)
    return fragments

def clean_fragments(fragments: list[str], # list of input fragments
                    remove_mapping: bool=True # if mapping should be removed (ie [*:1]C -> *C)
                   ) -> list[str]: # list of cleaned fragments
    '''
    cleans fragments, deduplicates them, and splits multi-compound fragments
    '''
    clean_fragments = []
    
    while fragments:
        current = fragments.pop()
        
        if not current:
            continue
            
        if '.' in current:
            fragments += current.split('.')
        else:
            if remove_mapping:
                current = remove_fragment_mapping(current)
                
            current = canon_smile(current)
            if current:
                clean_fragments.append(current)
                
    clean_fragments = deduplicate_list(clean_fragments)
    return clean_fragments

def shred_smiles(smiles: list[str], 
                 cuts: list[int], 
                 max_fragment_length: int, 
                 generations: int, 
                 keep_long_fragments: bool, 
                 worker_pool: Optional[Pool]=None
                ) -> list[str]:
    '''
    given a list of SMILES `smiles`, each SMILES string is fragmented with `cuts` 
    (see `fragment_smile`). After fragmentation, all fragments longer than 
    `max_fragment_length` are re-fragmented. Repeats for `generations` iterations. 
    If `keep_long_fragments=True`, all fragments are returned. Else, only fragments
    shorter than `max_fragment_length` are returned.
    
    `keep_long_fragments=False` is recommended as molecules tend to generate very 
    large fragments (ie just cleaving off a methyl group)
    '''
    processed = set()
    output = []
    
    current_inputs = smiles
    
    for g in range(generations):
        print(g, len(current_inputs))
        processed.update(set(current_inputs))
        
        if worker_pool:
            func = partial(fragment_smile, cuts=cuts)
            frags = worker_pool.map(func, current_inputs)
            frags = flatten_list(worker_pool.map(clean_fragments, frags))
            frags = deduplicate_list(frags)
            
        else:
            frags = flatten_list([fragment_smile(i, cuts) for i in current_inputs])
            frags = clean_fragments(frags)
            frags = deduplicate_list(frags)
        
        
        current_inputs = []
        for f in frags:
            if len(f)>max_fragment_length:
                if not (f in processed):
                    current_inputs.append(f)
                
                if keep_long_fragments:
                    output.append(f)
                    
            else:
                output.append(f)
                
        if not current_inputs:
            break
            
    return deduplicate_list(output)
