"use strict";
const fs = require("fs");
const os = require("os");
const path = require("path");
const fs_1 = require("../../lib/fs");
module.exports = {
    files: {
        'does not change with the file name'(test) {
            // GIVEN
            const workdir = fs.mkdtempSync(path.join(os.tmpdir(), 'hash-tests'));
            const content = 'Hello, world!';
            const input1 = path.join(workdir, 'input1.txt');
            const input2 = path.join(workdir, 'input2.txt');
            const input3 = path.join(workdir, 'input3.txt');
            fs.writeFileSync(input1, content);
            fs.writeFileSync(input2, content);
            fs.writeFileSync(input3, content + '.'); // add one character, hash should be different
            // WHEN
            const hash1 = fs_1.FileSystem.fingerprint(input1);
            const hash2 = fs_1.FileSystem.fingerprint(input2);
            const hash3 = fs_1.FileSystem.fingerprint(input3);
            // THEN
            test.deepEqual(hash1, hash2);
            test.notDeepEqual(hash3, hash1);
            test.done();
        },
        'works on empty files'(test) {
            // GIVEN
            const workdir = fs.mkdtempSync(path.join(os.tmpdir(), 'hash-tests'));
            const input1 = path.join(workdir, 'empty');
            const input2 = path.join(workdir, 'empty');
            fs.writeFileSync(input1, '');
            fs.writeFileSync(input2, '');
            // WHEN
            const hash1 = fs_1.FileSystem.fingerprint(input1);
            const hash2 = fs_1.FileSystem.fingerprint(input2);
            // THEN
            test.deepEqual(hash1, hash2);
            test.done();
        },
    },
    directories: {
        'works on directories'(test) {
            // GIVEN
            const srcdir = path.join(__dirname, 'fixtures', 'symlinks');
            const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
            fs_1.FileSystem.copyDirectory(srcdir, outdir);
            // WHEN
            const hashSrc = fs_1.FileSystem.fingerprint(srcdir);
            const hashCopy = fs_1.FileSystem.fingerprint(outdir);
            // THEN
            test.deepEqual(hashSrc, hashCopy);
            test.done();
        },
        'ignores requested files'(test) {
            // GIVEN
            const srcdir = path.join(__dirname, 'fixtures', 'symlinks');
            const outdir = fs.mkdtempSync(path.join(os.tmpdir(), 'copy-tests'));
            fs_1.FileSystem.copyDirectory(srcdir, outdir);
            // WHEN
            const hashSrc = fs_1.FileSystem.fingerprint(srcdir, { exclude: ['*.ignoreme'] });
            fs.writeFileSync(path.join(outdir, `${hashSrc}.ignoreme`), 'Ignore me!');
            const hashCopy = fs_1.FileSystem.fingerprint(outdir, { exclude: ['*.ignoreme'] });
            // THEN
            test.deepEqual(hashSrc, hashCopy);
            test.done();
        },
        'changes with file names'(test) {
            // GIVEN
            const srcdir = path.join(__dirname, 'fixtures', 'symlinks');
            const cpydir = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            fs_1.FileSystem.copyDirectory(srcdir, cpydir);
            // be careful not to break a symlink
            fs.renameSync(path.join(cpydir, 'normal-dir', 'file-in-subdir.txt'), path.join(cpydir, 'move-me.txt'));
            // WHEN
            const hashSrc = fs_1.FileSystem.fingerprint(srcdir);
            const hashCopy = fs_1.FileSystem.fingerprint(cpydir);
            // THEN
            test.notDeepEqual(hashSrc, hashCopy);
            test.done();
        },
    },
    symlinks: {
        'changes with the contents of followed symlink referent'(test) {
            // GIVEN
            const dir1 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const dir2 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const target = path.join(dir1, 'boom.txt');
            const content = 'boom';
            fs.writeFileSync(target, content);
            fs.symlinkSync(target, path.join(dir2, 'link-to-boom.txt'));
            // now dir2 contains a symlink to a file in dir1
            // WHEN
            const original = fs_1.FileSystem.fingerprint(dir2);
            // now change the contents of the target
            fs.writeFileSync(target, 'changning you!');
            const afterChange = fs_1.FileSystem.fingerprint(dir2);
            // revert the content to original and expect hash to be reverted
            fs.writeFileSync(target, content);
            const afterRevert = fs_1.FileSystem.fingerprint(dir2);
            // THEN
            test.notDeepEqual(original, afterChange);
            test.deepEqual(afterRevert, original);
            test.done();
        },
        'does not change with the contents of un-followed symlink referent'(test) {
            // GIVEN
            const dir1 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const dir2 = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const target = path.join(dir1, 'boom.txt');
            const content = 'boom';
            fs.writeFileSync(target, content);
            fs.symlinkSync(target, path.join(dir2, 'link-to-boom.txt'));
            // now dir2 contains a symlink to a file in dir1
            // WHEN
            const original = fs_1.FileSystem.fingerprint(dir2, { follow: fs_1.SymlinkFollowMode.NEVER });
            // now change the contents of the target
            fs.writeFileSync(target, 'changning you!');
            const afterChange = fs_1.FileSystem.fingerprint(dir2, { follow: fs_1.SymlinkFollowMode.NEVER });
            // revert the content to original and expect hash to be reverted
            fs.writeFileSync(target, content);
            const afterRevert = fs_1.FileSystem.fingerprint(dir2, { follow: fs_1.SymlinkFollowMode.NEVER });
            // THEN
            test.deepEqual(original, afterChange);
            test.deepEqual(afterRevert, original);
            test.done();
        },
    },
    exclude: {
        'encodes exclude patterns'(test) {
            // GIVEN
            const dir = fs.mkdtempSync(path.join(os.tmpdir(), 'fingerprint-tests'));
            const options1 = { path: dir, exclude: ['**', '!file.py'], sourcePath: dir };
            const options2 = { path: dir, exclude: ['**', '!otherfile.py'], sourcePath: dir };
            // WHEN
            const f1 = fs_1.FileSystem.fingerprint(dir, options1);
            const f2 = fs_1.FileSystem.fingerprint(dir, options2);
            // THEN
            test.notDeepEqual(f1, f2);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,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