"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const s3 = require("../lib");
module.exports = {
    'when notification is added a custom s3 bucket notification resource is provisioned'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket');
        bucket.addEventNotification(s3.EventType.OBJECT_CREATED, {
            bind: () => ({
                arn: 'ARN',
                type: s3.BucketNotificationDestinationType.TOPIC,
            }),
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::S3::Bucket'));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications', {
            NotificationConfiguration: {
                TopicConfigurations: [
                    {
                        Events: [
                            's3:ObjectCreated:*',
                        ],
                        TopicArn: 'ARN',
                    },
                ],
            },
        }));
        test.done();
    },
    'can specify prefix and suffix filter rules'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket');
        bucket.addEventNotification(s3.EventType.OBJECT_CREATED, {
            bind: () => ({
                arn: 'ARN',
                type: s3.BucketNotificationDestinationType.TOPIC,
            }),
        }, { prefix: 'images/', suffix: '.png' });
        assert_1.expect(stack).to(assert_1.haveResource('Custom::S3BucketNotifications', {
            NotificationConfiguration: {
                TopicConfigurations: [
                    {
                        Events: [
                            's3:ObjectCreated:*',
                        ],
                        Filter: {
                            Key: {
                                FilterRules: [
                                    {
                                        Name: 'suffix',
                                        Value: '.png',
                                    },
                                    {
                                        Name: 'prefix',
                                        Value: 'images/',
                                    },
                                ],
                            },
                        },
                        TopicArn: 'ARN',
                    },
                ],
            },
        }));
        test.done();
    },
    'the notification lambda handler must depend on the role to prevent executing too early'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket');
        bucket.addEventNotification(s3.EventType.OBJECT_CREATED, {
            bind: () => ({
                arn: 'ARN',
                type: s3.BucketNotificationDestinationType.TOPIC,
            }),
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Type: 'AWS::Lambda::Function',
            Properties: {
                Role: {
                    'Fn::GetAtt': [
                        'BucketNotificationsHandler050a0587b7544547bf325f094a3db834RoleB6FB88EC',
                        'Arn',
                    ],
                },
            },
            DependsOn: ['BucketNotificationsHandler050a0587b7544547bf325f094a3db834RoleDefaultPolicy2CF63D36',
                'BucketNotificationsHandler050a0587b7544547bf325f094a3db834RoleB6FB88EC'],
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'throws with multiple prefix rules in a filter'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket');
        test.throws(() => bucket.addEventNotification(s3.EventType.OBJECT_CREATED, {
            bind: () => ({
                arn: 'ARN',
                type: s3.BucketNotificationDestinationType.TOPIC,
            }),
        }, { prefix: 'images/' }, { prefix: 'archive/' }), /prefix rule/);
        test.done();
    },
    'throws with multiple suffix rules in a filter'(test) {
        const stack = new cdk.Stack();
        const bucket = new s3.Bucket(stack, 'MyBucket');
        test.throws(() => bucket.addEventNotification(s3.EventType.OBJECT_CREATED, {
            bind: () => ({
                arn: 'ARN',
                type: s3.BucketNotificationDestinationType.TOPIC,
            }),
        }, { suffix: '.png' }, { suffix: '.zip' }), /suffix rule/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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