import os
import requests


def publish_docs(docs_url_project: str) -> None:
    """
    Publishes dbt documentation to `dbt.intern.nav.no`.

    This function uploads documentation files generated by the `dbt docs generate` command to 
    `dbt.intern.nav.no`. The files include `manifest.json`, `catalog.json`, and `index.html` 
    from the `target` directory. The URL used for publishing is constructed using the provided 
    `docs_url_project`, which should be the last segment of the URL path.

    Note:
        - Ensure that `dbt docs generate` has been executed beforehand to create the necessary 
          documentation files.
        - The `docs_url_project` should use hyphens (`-`) instead of underscores (`_`).

    Args:
        docs_url_project (str): The last segment of the URL path for the documentation site, e.g., 'dvh-aap'.

    Raises:
        requests.RequestException: For issues during the HTTP PUT request.

    Examples:
        >>> publish_docs(docs_url_project='dvh-aap')
        Gathering target/manifest.json (1024 kB)
        Gathering target/catalog.json (2048 kB)
        Gathering target/index.html (512 kB)
        HTTP PUT status: 200 OK
    """     
    docs_url_complete = "https://dbt.intern.nav.no/docs/spenn/" + docs_url_project
    files = ["target/manifest.json", "target/catalog.json", "target/index.html"]
    multipart_form_data = {}
    for file_path in files:
        file_name = os.path.basename(file_path)
        with open(file_path, "rb") as file:
            file_contents = file.read()
            print(f"Gathering {file_path} ({len(file_contents)/1024:.0f} kB)")
            multipart_form_data[file_name] = (file_name, file_contents)

    res = requests.put(docs_url_complete, files=multipart_form_data)
    res.raise_for_status()
    print("HTTP PUT status: ", res.status_code, res.text)
    print()


if __name__ == "__main__":
    publish_docs()
