#MAEC Malware Subject Class

#Copyright (c) 2013, The MITRE Corporation
#All rights reserved.

#Compatible with MAEC v4.0
#Last updated 5/15/2013

import maec
import maec.bindings.maec_package as package_binding
from cybox.core.object import Object
from cybox.common import VocabString
from maec.package.analysis import Analysis
from maec.package.malware_subject_reference import MalwareSubjectReference
from maec.bundle.bundle import Bundle
from maec.package.action_equivalence import ActionEquivalenceList
from maec.package.object_equivalence import ObjectEquivalenceList

class MalwareSubject(maec.Entity):
    def __init__(self, id, malware_instance_object_attributes = None):
        super(MalwareSubject, self).__init__()
        self.id = id
        #Set the Malware Instance Object Attributes (a CybOX object) if they are not none
        self.malware_instance_object_attributes = malware_instance_object_attributes
        self.minor_variants = MinorVariants()
        self.field_data = None
        #Instantiate the lists
        self.analyses = Analyses()
        self.findings_bundles = FindingsBundleList()
        self.relationships = MalwareSubjectRelationshipList()

    #Public methods
    #Set the Malware_Instance_Object_Attributes with a CybOX object
    def set_malware_instance_object_attributes(self, malware_instance_object_attributes):
        self.malware_instance_object_attributes = malware_instance_object_attributes

    #Add an Analysis to the Analyses
    def add_analysis(self, analysis):
        self.analyses.append(analysis)

    def get_analyses(self):
        return self.analyses

    #Add a MAEC Bundle to the Findings Bundles
    def add_findings_bundle(self, bundle):
        self.findings_bundles.add_bundle(bundle)

    
    def to_obj(self):
        malware_subject_obj = package_binding.MalwareSubjectType(id = self.id)
        if self.malware_instance_object_attributes is not None: malware_subject_obj.set_Malware_Instance_Object_Attributes(self.malware_instance_object_attributes.to_obj())
        if self.minor_variants is not None : malware_subject_obj.set_Minor_Variants(self.minor_variants.to_obj())
        if self.field_data is not None: malware_subject_obj.set_Field_Data(self.field_data.to_obj())
        if self.analyses is not None : malware_subject_obj.set_Analyses(self.analyses.to_obj())
        if self.findings_bundles is not None : malware_subject_obj.set_Findings_Bundles(self.findings_bundles.to_obj())
        if self.relationships is not None : malware_subject_obj.set_Relationships(self.relationships.to_obj())
        return malware_subject_obj

    def to_dict(self):
        malware_subject_dict = {}
        if self.id is not None : malware_subject_dict['id'] = self.id
        if self.malware_instance_object_attributes is not None: malware_subject_dict['malware_instance_object_attributes'] = self.malware_instance_object_attributes.to_dict()
        if self.minor_variants is not None : malware_subject_dict['minor_variants'] = self.minor_variants.to_list()
        if self.field_data is not None: malware_subject_dict['field_data'] = self.field_data.to_dict()
        if self.analyses is not None : malware_subject_dict['analyses'] = self.analyses.to_list()
        if self.findings_bundles is not None : malware_subject_dict['findings_bundles'] = self.findings_bundles.to_dict()
        if self.relationships is not None : malware_subject_dict['relationships'] = self.relationships.to_list()
        return malware_subject_dict

    #Build the Malware Subject from the input dictionary
    @staticmethod
    def from_dict(malware_subject_dict):
        if not malware_subject_dict:
            return None
        malware_subject_ = MalwareSubject(None)
        malware_subject_.id = malware_subject_dict.get('id')
        malware_subject_.malware_instance_object_attributes = Object.from_dict(malware_subject_dict.get('malware_instance_object_attributes'))
        malware_subject_.minor_variants = MinorVariants.from_list(malware_subject_dict.get('minor_variants'))
        malware_subject_.field_data = None #TODO: add support
        malware_subject_.analyses = Analyses.from_list(malware_subject_dict.get('analyses'))
        malware_subject_.findings_bundles = FindingsBundleList.from_dict(malware_subject_dict.get('findings_bundles'))
        malware_subject_.relationships = MalwareSubjectRelationshipList.from_list(malware_subject_dict.get('id'))
        return malware_subject_

    @staticmethod
    def from_obj(malware_subject_obj):
        if not malware_subject_obj:
            return None
        malware_subject_ = MalwareSubject(None)
        malware_subject_.id = malware_subject_obj.get_id()
        malware_subject_.malware_instance_object_attributes = Object.from_obj(malware_subject_obj.get_Malware_Instance_Object_Attributes())
        malware_subject_.minor_variants = MinorVariants.from_obj(malware_subject_obj.get_Minor_Variants())
        malware_subject_.field_data = None #TODO: add support
        malware_subject_.analyses = Analyses.from_obj(malware_subject_obj.get_Analyses())
        malware_subject_.findings_bundles = FindingsBundleList.from_obj(malware_subject_obj.get_Findings_Bundles())
        malware_subject_.relationships = MalwareSubjectRelationshipList.from_obj(malware_subject_obj.get_Relationships())
        return malware_subject_

    
class MinorVariants(maec.EntityList):
    _contained_type = Object
    _binding_class = package_binding.MinorVariantListType

    def __init__(self):
        super(MinorVariants, self).__init__()

    @staticmethod
    def _set_list(binding_obj, list_):
        binding_obj.set_Minor_Variant(list_)

    @staticmethod
    def _get_list(binding_obj):
        return binding_obj.get_Minor_Variant()

class Analyses(maec.EntityList):
    _contained_type = Analysis
    _binding_class = package_binding.AnalysisListType

    def __init__(self):
        super(Analyses, self).__init__()

    @staticmethod
    def _set_list(binding_obj, list_):
        binding_obj.set_Analysis(list_)

    @staticmethod
    def _get_list(binding_obj):
        return binding_obj.get_Analysis()

class MalwareSubjectRelationship(maec.Entity):

    def __init__(self):
        super(MalwareSubjectRelationship, self).__init__()
        self.type = None
        self.malware_subject_references = []

    def to_obj(self):
        malware_subject_relationship_obj = package_binding.MalwareSubjectRelationshipType()
        if self.type is not None : malware_subject_relationship_obj.set_Type(self.type.to_obj())
        if len(self.malware_subject_references) > 0:
            for malware_subject_reference in self.malware_subject_references:
                malware_subject_relationship_obj.add_Malware_Subject_Reference(malware_subject_reference.to_obj())
        return malware_subject_relationship_obj

    def to_dict(self):
        malware_subject_relationship_dict = {}
        if self.type is not None : malware_subject_relationship_dict['type'] = self.type.to_dict()
        if len(self.malware_subject_references) > 0:
            malware_subject_refs = []
            for malware_subject_reference in self.malware_subject_references:
                malware_subject_refs.append(malware_subject_reference.to_dict())
            malware_subject_relationship_dict['malware_subject_references'] = malware_subject_refs
        return malware_subject_relationship_dict

    @staticmethod
    def from_dict(malware_subject_relationship_dict):
        if not malware_subject_relationship_dict:
            return None
        malware_subject_relationship_ = MalwareSubjectRelationship()
        malware_subject_relationship_.type = VocabString.from_dict(malware_subject_relationship_dict.get('type'))
        malware_subject_relationship_.malware_subject_references = [MalwareSubjectReference.from_dict(x) for x in malware_subject_relationship_dict.get('malware_subject_references', [])]
        return malware_subject_relationship_

    @staticmethod
    def from_obj(malware_subject_relationship_obj):
        if not malware_subject_relationship_obj:
            return None
        malware_subject_relationship_ = MalwareSubjectRelationship()
        malware_subject_relationship_.type = VocabString.from_obj(malware_subject_relationship_obj.get_Type())
        malware_subject_relationship_.malware_subject_references = [MalwareSubjectReference.from_obj(x) for x in malware_subject_relationship_obj.get_Malware_Subject_Reference()]
        return malware_subject_relationship_

class MalwareSubjectRelationshipList(maec.EntityList):
    _contained_type = MalwareSubjectRelationship
    _binding_class = package_binding.MalwareSubjectRelationshipListType

    def __init__(self):
        super(MalwareSubjectRelationshipList, self).__init__()

    @staticmethod
    def _set_list(binding_obj, list_):
        binding_obj.set_Relationship(list_)

    @staticmethod
    def _get_list(binding_obj):
        return binding_obj.get_Relationship()

class FindingsBundleList(maec.Entity):

    def __init__(self):
        super(FindingsBundleList, self).__init__()
        self.meta_analysis = None
        self.bundles = []
        self.bundle_external_references = []

    def add_bundle(self, bundle):
        self.bundles.append(bundle)

    def add_bundle_external_reference(self, bundle_external_reference):
        self.bundle_external_references.append(bundle_external_reference)

    def to_obj(self):
        findings_bundle_list_obj = package_binding.FindingsBundleListType()
        if self.meta_analysis is not None : findings_bundle_list_obj.set_Meta_Analysis(self.meta_analysis.to_obj())
        if len(self.bundles) > 0: 
            for bundle in self.bundles: findings_bundle_list_obj.add_Bundle(bundle.to_obj())
        if len(self.bundle_external_references) > 0: 
            for bundle_external_reference in self.bundle_external_references: findings_bundle_list_obj.add_Bundle_External_Reference(bundle_external_reference)
        return findings_bundle_list_obj

    def to_dict(self):
        findings_bundle_list_dict = {}
        if self.meta_analysis is not None : findings_bundle_list_dict['meta_analysis'] = self.meta_analysis.to_dict()
        if len(self.bundles) > 0: 
            bundle_list = []
            for bundle in self.bundles: bundle_list.append(bundle.to_dict())
            findings_bundle_list_dict['bundles'] = bundle_list
        if len(self.bundle_external_references) > 0: 
            bundle_external_refs_list = []
            for bundle_external_reference in self.bundle_external_references: bundle_external_refs_list.append(bundle_external_reference)
            findings_bundle_list_dict['bundle_external_references'] = bundle_external_refs_list
        return findings_bundle_list_dict

    @staticmethod
    def from_dict(findings_bundle_list_dict):
        if not findings_bundle_list_dict:
            return None
        findings_bundle_list_ = FindingsBundleList()
        findings_bundle_list_.meta_analysis = MetaAnalysis.from_dict(findings_bundle_list_dict.get('meta_analysis'))
        findings_bundle_list_.bundles = [Bundle.from_dict(x) for x in findings_bundle_list_dict.get('bundles', [])]
        findings_bundle_list_.bundle_external_references = [x for x in findings_bundle_list_dict.get('bundle_external_references', [])]
        return findings_bundle_list_

    @staticmethod
    def from_obj(findings_bundle_list_obj):
        if not findings_bundle_list_obj:
            return None
        findings_bundle_list_ = FindingsBundleList()
        findings_bundle_list_.meta_analysis = MetaAnalysis.from_obj(findings_bundle_list_obj.get_Meta_Analysis())
        findings_bundle_list_.bundles = [Bundle.from_obj(x) for x in findings_bundle_list_obj.get_Bundle()]
        findings_bundle_list_.bundle_external_references = [x for x in findings_bundle_list_obj.get_Bundle_External_Reference()]
        return findings_bundle_list_

class MetaAnalysis(maec.Entity):

    def __init__(self):
        super(MetaAnalysis, self).__init__()
        self.action_equivalences = None
        self.object_equivalences = None

    def to_obj(self):
        meta_analysis_obj = package_binding.MetaAnalysisType()
        if self.action_equivalences is not None : meta_analysis_obj.set_Action_Equivalences(self.action_equivalences.to_obj())
        if self.object_equivalences is not None : meta_analysis_obj.set_Object_Equivalences(self.object_equivalences.to_obj())
        return meta_analysis_obj

    def to_dict(self):
        meta_analysis_dict = {}
        if self.action_equivalences is not None : meta_analysis_dict['action_equivalences'] = self.action_equivalences.to_list()
        if self.object_equivalences is not None : meta_analysis_dict['object_equivalences'] = self.object_equivalences.to_list()
        return meta_analysis_dict

    @staticmethod
    def from_dict(meta_analysis_dict):
        if not meta_analysis_dict:
            return None
        meta_analysis_ = MetaAnalysis()
        meta_analysis_.action_equivalences = ActionEquivalenceList.from_list(meta_analysis_dict.get('action_equivalences'))
        meta_analysis_.object_equivalences = ObjectEquivalenceList.from_list(meta_analysis_dict.get('object_equivalences'))
        return meta_analysis_

    @staticmethod
    def from_obj(meta_analysis_obj):
        if not meta_analysis_obj:
            return None
        meta_analysis_ = MetaAnalysis()
        meta_analysis_.action_equivalences = ActionEquivalenceList.from_obj(meta_analysis_obj.get_Action_Equivalences())
        meta_analysis_.object_equivalences = ObjectEquivalenceList.from_obj(meta_analysis_obj.get_Object_Equivalences())
        return meta_analysis_

class MalwareSubjectList(maec.EntityList):
    _contained_type = MalwareSubject
    _binding_class = package_binding.MalwareSubjectListType

    def __init__(self):
        super(MalwareSubjectList, self).__init__()

    @staticmethod
    def _set_list(binding_obj, list_):
        binding_obj.set_Malware_Subject(list_)

    @staticmethod
    def _get_list(binding_obj):
        return binding_obj.get_Malware_Subject()