//! Contains [`Error`]

/// List of features whose non-activation may cause a runtime error.
/// Used to indicate which lack of feature caused [`Error::FeatureNotActive`].
#[derive(Debug, Clone, Copy, PartialEq, Eq, Hash)]
#[non_exhaustive]
pub enum Feature {
    /// Snappy compression and decompression
    Snappy,
    /// Brotli compression and decompression
    Brotli,
    /// Gzip compression and decompression
    Gzip,
    /// Lz4 raw compression and decompression
    Lz4,
    /// Zstd compression and decompression
    Zstd,
}

/// Errors generated by this crate
#[derive(Debug, Clone)]
#[non_exhaustive]
pub enum ParquetError {
    /// When the parquet file is known to be out of spec.
    OutOfSpec(String),
    /// Error presented when trying to use a code branch that requires activating a feature.
    FeatureNotActive(Feature, String),
    /// Error presented when trying to use a feature from parquet that is not yet supported
    FeatureNotSupported(String),
    /// When encoding, the user passed an invalid parameter
    InvalidParameter(String),
    /// When decoding or decompressing, the page would allocate more memory than allowed
    WouldOverAllocate,
}

impl ParquetError {
    /// Create an OutOfSpec error from any Into<String>
    pub(crate) fn oos<I: Into<String>>(message: I) -> Self {
        Self::OutOfSpec(message.into())
    }

    /// Create an FeatureNotSupported error from any Into<String>
    pub(crate) fn not_supported<I: Into<String>>(message: I) -> Self {
        Self::FeatureNotSupported(message.into())
    }
}

impl std::error::Error for ParquetError {}

impl std::fmt::Display for ParquetError {
    fn fmt(&self, fmt: &mut std::fmt::Formatter) -> std::fmt::Result {
        match self {
            ParquetError::OutOfSpec(message) => {
                write!(fmt, "File out of specification: {}", message)
            },
            ParquetError::FeatureNotActive(feature, reason) => {
                write!(
                    fmt,
                    "The feature \"{:?}\" needs to be active to {}",
                    feature, reason
                )
            },
            ParquetError::FeatureNotSupported(reason) => {
                write!(fmt, "Not yet supported: {}", reason)
            },
            ParquetError::InvalidParameter(message) => {
                write!(fmt, "Invalid parameter: {}", message)
            },
            ParquetError::WouldOverAllocate => {
                write!(fmt, "Operation would exceed memory use threshold")
            },
        }
    }
}

#[cfg(feature = "snappy")]
impl From<snap::Error> for ParquetError {
    fn from(e: snap::Error) -> ParquetError {
        ParquetError::OutOfSpec(format!("underlying snap error: {}", e))
    }
}

#[cfg(feature = "lz4_flex")]
impl From<lz4_flex::block::DecompressError> for ParquetError {
    fn from(e: lz4_flex::block::DecompressError) -> ParquetError {
        ParquetError::OutOfSpec(format!("underlying lz4_flex error: {}", e))
    }
}

#[cfg(feature = "lz4_flex")]
impl From<lz4_flex::block::CompressError> for ParquetError {
    fn from(e: lz4_flex::block::CompressError) -> ParquetError {
        ParquetError::OutOfSpec(format!("underlying lz4_flex error: {}", e))
    }
}

impl From<polars_parquet_format::thrift::Error> for ParquetError {
    fn from(e: polars_parquet_format::thrift::Error) -> ParquetError {
        ParquetError::OutOfSpec(format!("Invalid thrift: {}", e))
    }
}

impl From<std::io::Error> for ParquetError {
    fn from(e: std::io::Error) -> ParquetError {
        ParquetError::OutOfSpec(format!("underlying IO error: {}", e))
    }
}

impl From<std::collections::TryReserveError> for ParquetError {
    fn from(e: std::collections::TryReserveError) -> ParquetError {
        ParquetError::OutOfSpec(format!("OOM: {}", e))
    }
}

impl From<std::num::TryFromIntError> for ParquetError {
    fn from(e: std::num::TryFromIntError) -> ParquetError {
        ParquetError::OutOfSpec(format!("Number must be zero or positive: {}", e))
    }
}

impl From<std::array::TryFromSliceError> for ParquetError {
    fn from(e: std::array::TryFromSliceError) -> ParquetError {
        ParquetError::OutOfSpec(format!("Can't deserialize to parquet native type: {}", e))
    }
}

/// A specialized `Result` for Parquet errors.
pub type ParquetResult<T> = std::result::Result<T, ParquetError>;

impl From<ParquetError> for polars_error::PolarsError {
    fn from(e: ParquetError) -> polars_error::PolarsError {
        polars_error::PolarsError::ComputeError(format!("parquet: {}", e).into())
    }
}

impl From<polars_error::PolarsError> for ParquetError {
    fn from(e: polars_error::PolarsError) -> ParquetError {
        ParquetError::OutOfSpec(format!("OOM: {}", e))
    }
}
