import cxapi = require('@aws-cdk/cx-api');
import { DockerImageAssetLocation, DockerImageAssetSource, FileAssetLocation, FileAssetSource } from './assets';
import { Construct, IConstruct, ISynthesisSession } from './construct';
import { Environment } from './environment';
export interface StackProps {
    /**
     * A description of the stack.
     *
     * @default - No description.
     */
    readonly description?: string;
    /**
     * The AWS environment (account/region) where this stack will be deployed.
     *
     * @default - The `default-account` and `default-region` context parameters will be
     * used. If they are undefined, it will not be possible to deploy the stack.
     */
    readonly env?: Environment;
    /**
     * Name to deploy the stack with
     *
     * @default - Derived from construct path.
     */
    readonly stackName?: string;
    /**
     * Stack tags that will be applied to all the taggable resources and the stack itself.
     *
     * @default {}
     */
    readonly tags?: {
        [key: string]: string;
    };
}
/**
 * A root construct which represents a single CloudFormation stack.
 */
export declare class Stack extends Construct implements ITaggable {
    /**
     * Return whether the given object is a Stack.
     *
     * We do attribute detection since we can't reliably use 'instanceof'.
     */
    static isStack(x: any): x is Stack;
    /**
     * Looks up the first stack scope in which `construct` is defined. Fails if there is no stack up the tree.
     * @param construct The construct to start the search from.
     */
    static of(construct: IConstruct): Stack;
    /**
     * Tags to be applied to the stack.
     */
    readonly tags: TagManager;
    /**
     * Options for CloudFormation template (like version, transform, description).
     */
    readonly templateOptions: ITemplateOptions;
    /**
     * The AWS region into which this stack will be deployed (e.g. `us-west-2`).
     *
     * This value is resolved according to the following rules:
     *
     * 1. The value provided to `env.region` when the stack is defined. This can
     *    either be a concerete region (e.g. `us-west-2`) or the `Aws.region`
     *    token.
     * 3. `Aws.region`, which is represents the CloudFormation intrinsic reference
     *    `{ "Ref": "AWS::Region" }` encoded as a string token.
     *
     * Preferably, you should use the return value as an opaque string and not
     * attempt to parse it to implement your logic. If you do, you must first
     * check that it is a concerete value an not an unresolved token. If this
     * value is an unresolved token (`Token.isUnresolved(stack.region)` returns
     * `true`), this implies that the user wishes that this stack will synthesize
     * into a **region-agnostic template**. In this case, your code should either
     * fail (throw an error, emit a synth error using `node.addError`) or
     * implement some other region-agnostic behavior.
     */
    readonly region: string;
    /**
     * The AWS account into which this stack will be deployed.
     *
     * This value is resolved according to the following rules:
     *
     * 1. The value provided to `env.account` when the stack is defined. This can
     *    either be a concerete account (e.g. `585695031111`) or the
     *    `Aws.accountId` token.
     * 3. `Aws.accountId`, which represents the CloudFormation intrinsic reference
     *    `{ "Ref": "AWS::AccountId" }` encoded as a string token.
     *
     * Preferably, you should use the return value as an opaque string and not
     * attempt to parse it to implement your logic. If you do, you must first
     * check that it is a concerete value an not an unresolved token. If this
     * value is an unresolved token (`Token.isUnresolved(stack.account)` returns
     * `true`), this implies that the user wishes that this stack will synthesize
     * into a **account-agnostic template**. In this case, your code should either
     * fail (throw an error, emit a synth error using `node.addError`) or
     * implement some other region-agnostic behavior.
     */
    readonly account: string;
    /**
     * The environment coordinates in which this stack is deployed. In the form
     * `aws://account/region`. Use `stack.account` and `stack.region` to obtain
     * the specific values, no need to parse.
     *
     * You can use this value to determine if two stacks are targeting the same
     * environment.
     *
     * If either `stack.account` or `stack.region` are not concrete values (e.g.
     * `Aws.account` or `Aws.region`) the special strings `unknown-account` and/or
     * `unknown-region` will be used respectively to indicate this stack is
     * region/account-agnostic.
     */
    readonly environment: string;
    /**
     * If this is a nested stack, this represents its `AWS::CloudFormation::Stack`
     * resource. `undefined` for top-level (non-nested) stacks.
     *
     * @experimental
     */
    readonly nestedStackResource?: CfnResource;
    /**
     * An attribute (late-bound) that represents the URL of the template file
     * in the deployment bucket.
     *
     * @experimental
     */
    readonly templateUrl: string;
    /**
     * The name of the CloudFormation template file emitted to the output
     * directory during synthesis.
     *
     * @example MyStack.template.json
     */
    readonly templateFile: string;
    /**
     * The ID of the cloud assembly artifact for this stack.
     */
    readonly artifactId: string;
    /**
     * Logical ID generation strategy
     */
    private readonly _logicalIds;
    /**
     * Other stacks this stack depends on
     */
    private readonly _stackDependencies;
    /**
     * Lists all missing contextual information.
     * This is returned when the stack is synthesized under the 'missing' attribute
     * and allows tooling to obtain the context and re-synthesize.
     */
    private readonly _missingContext;
    /**
     * Includes all parameters synthesized for assets (lazy).
     */
    private _assetParameters?;
    private _templateUrl?;
    private readonly _stackName;
    /**
     * Creates a new stack.
     *
     * @param scope Parent of this stack, usually a Program instance.
     * @param id The construct ID of this stack. If `stackName` is not explicitly
     * defined, this id (and any parent IDs) will be used to determine the
     * physical ID of the stack.
     * @param props Stack properties.
     */
    constructor(scope?: Construct, id?: string, props?: StackProps);
    /**
     * Resolve a tokenized value in the context of the current stack.
     */
    resolve(obj: any): any;
    /**
     * Convert an object, potentially containing tokens, to a JSON string
     */
    toJsonString(obj: any, space?: number): string;
    /**
     * Indicate that a context key was expected
     *
     * Contains instructions which will be emitted into the cloud assembly on how
     * the key should be supplied.
     *
     * @param report The set of parameters needed to obtain the context
     */
    reportMissingContext(report: cxapi.MissingContext): void;
    /**
     * Rename a generated logical identities
     *
     * To modify the naming scheme strategy, extend the `Stack` class and
     * override the `createNamingScheme` method.
     */
    renameLogicalId(oldId: string, newId: string): void;
    /**
     * Allocates a stack-unique CloudFormation-compatible logical identity for a
     * specific resource.
     *
     * This method is called when a `CfnElement` is created and used to render the
     * initial logical identity of resources. Logical ID renames are applied at
     * this stage.
     *
     * This method uses the protected method `allocateLogicalId` to render the
     * logical ID for an element. To modify the naming scheme, extend the `Stack`
     * class and override this method.
     *
     * @param element The CloudFormation element for which a logical identity is
     * needed.
     */
    getLogicalId(element: CfnElement): string;
    /**
     * Add a dependency between this stack and another stack.
     *
     * This can be used to define dependencies between any two stacks within an
     * app, and also supports nested stacks.
     */
    addDependency(target: Stack, reason?: string): void;
    /**
     * Return the stacks this stack depends on
     */
    get dependencies(): Stack[];
    /**
     * The concrete CloudFormation physical stack name.
     *
     * This is either the name defined explicitly in the `stackName` prop or
     * allocated based on the stack's location in the construct tree. Stacks that
     * are directly defined under the app use their construct `id` as their stack
     * name. Stacks that are defined deeper within the tree will use a hashed naming
     * scheme based on the construct path to ensure uniqueness.
     *
     * If you wish to obtain the deploy-time AWS::StackName intrinsic,
     * you can use `Aws.stackName` directly.
     */
    get stackName(): string;
    /**
     * The partition in which this stack is defined
     */
    get partition(): string;
    /**
     * The Amazon domain suffix for the region in which this stack is defined
     */
    get urlSuffix(): string;
    /**
     * The ID of the stack
     *
     * @example After resolving, looks like arn:aws:cloudformation:us-west-2:123456789012:stack/teststack/51af3dc0-da77-11e4-872e-1234567db123
     */
    get stackId(): string;
    /**
     * Returns the list of notification Amazon Resource Names (ARNs) for the current stack.
     */
    get notificationArns(): string[];
    /**
     * Indicates if this is a nested stack, in which case `parentStack` will include a reference to it's parent.
     */
    get nested(): boolean;
    /**
     * Creates an ARN from components.
     *
     * If `partition`, `region` or `account` are not specified, the stack's
     * partition, region and account will be used.
     *
     * If any component is the empty string, an empty string will be inserted
     * into the generated ARN at the location that component corresponds to.
     *
     * The ARN will be formatted as follows:
     *
     *   arn:{partition}:{service}:{region}:{account}:{resource}{sep}}{resource-name}
     *
     * The required ARN pieces that are omitted will be taken from the stack that
     * the 'scope' is attached to. If all ARN pieces are supplied, the supplied scope
     * can be 'undefined'.
     */
    formatArn(components: ArnComponents): string;
    /**
     * Given an ARN, parses it and returns components.
     *
     * If the ARN is a concrete string, it will be parsed and validated. The
     * separator (`sep`) will be set to '/' if the 6th component includes a '/',
     * in which case, `resource` will be set to the value before the '/' and
     * `resourceName` will be the rest. In case there is no '/', `resource` will
     * be set to the 6th components and `resourceName` will be set to the rest
     * of the string.
     *
     * If the ARN includes tokens (or is a token), the ARN cannot be validated,
     * since we don't have the actual value yet at the time of this function
     * call. You will have to know the separator and the type of ARN. The
     * resulting `ArnComponents` object will contain tokens for the
     * subexpressions of the ARN, not string literals. In this case this
     * function cannot properly parse the complete final resourceName (path) out
     * of ARNs that use '/' to both separate the 'resource' from the
     * 'resourceName' AND to subdivide the resourceName further. For example, in
     * S3 ARNs:
     *
     *    arn:aws:s3:::my_corporate_bucket/path/to/exampleobject.png
     *
     * After parsing the resourceName will not contain
     * 'path/to/exampleobject.png' but simply 'path'. This is a limitation
     * because there is no slicing functionality in CloudFormation templates.
     *
     * @param arn The ARN string to parse
     * @param sepIfToken The separator used to separate resource from resourceName
     * @param hasName Whether there is a name component in the ARN at all. For
     * example, SNS Topics ARNs have the 'resource' component contain the topic
     * name, and no 'resourceName' component.
     *
     * @returns an ArnComponents object which allows access to the various
     * components of the ARN.
     *
     * @returns an ArnComponents object which allows access to the various
     *      components of the ARN.
     */
    parseArn(arn: string, sepIfToken?: string, hasName?: boolean): ArnComponents;
    /**
     * Returnst the list of AZs that are availability in the AWS environment
     * (account/region) associated with this stack.
     *
     * If the stack is environment-agnostic (either account and/or region are
     * tokens), this property will return an array with 2 tokens that will resolve
     * at deploy-time to the first two availability zones returned from CloudFormation's
     * `Fn::GetAZs` intrinsic function.
     *
     * If they are not available in the context, returns a set of dummy values and
     * reports them as missing, and let the CLI resolve them by calling EC2
     * `DescribeAvailabilityZones` on the target environment.
     */
    get availabilityZones(): string[];
    addFileAsset(asset: FileAssetSource): FileAssetLocation;
    addDockerImageAsset(asset: DockerImageAssetSource): DockerImageAssetLocation;
    /**
     * If this is a nested stack, returns it's parent stack.
     */
    get nestedStackParent(): Stack | undefined;
    /**
     * Returns the parent of a nested stack.
     *
     * @deprecated use `nestedStackParent`
     */
    get parentStack(): Stack | undefined;
    /**
     * Add a Transform to this stack. A Transform is a macro that AWS
     * CloudFormation uses to process your template.
     *
     * Duplicate values are removed when stack is synthesized.
     *
     * @example addTransform('AWS::Serverless-2016-10-31')
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/transform-section-structure.html
     *
     * @param transform The transform to add
     */
    addTransform(transform: string): void;
    /**
     * Returns the naming scheme used to allocate logical IDs. By default, uses
     * the `HashedAddressingScheme` but this method can be overridden to customize
     * this behavior.
     *
     * In order to make sure logical IDs are unique and stable, we hash the resource
     * construct tree path (i.e. toplevel/secondlevel/.../myresource) and add it as
     * a suffix to the path components joined without a separator (CloudFormation
     * IDs only allow alphanumeric characters).
     *
     * The result will be:
     *
     *   <path.join('')><md5(path.join('/')>
     *     "human"      "hash"
     *
     * If the "human" part of the ID exceeds 240 characters, we simply trim it so
     * the total ID doesn't exceed CloudFormation's 255 character limit.
     *
     * We only take 8 characters from the md5 hash (0.000005 chance of collision).
     *
     * Special cases:
     *
     * - If the path only contains a single component (i.e. it's a top-level
     *   resource), we won't add the hash to it. The hash is not needed for
     *   disamiguation and also, it allows for a more straightforward migration an
     *   existing CloudFormation template to a CDK stack without logical ID changes
     *   (or renames).
     * - For aesthetic reasons, if the last components of the path are the same
     *   (i.e. `L1/L2/Pipeline/Pipeline`), they will be de-duplicated to make the
     *   resulting human portion of the ID more pleasing: `L1L2Pipeline<HASH>`
     *   instead of `L1L2PipelinePipeline<HASH>`
     * - If a component is named "Default" it will be omitted from the path. This
     *   allows refactoring higher level abstractions around constructs without affecting
     *   the IDs of already deployed resources.
     * - If a component is named "Resource" it will be omitted from the user-visible
     *   path, but included in the hash. This reduces visual noise in the human readable
     *   part of the identifier.
     *
     * @param cfnElement The element for which the logical ID is allocated.
     */
    protected allocateLogicalId(cfnElement: CfnElement): string;
    /**
     * Prepare stack
     *
     * Find all CloudFormation references and tell them we're consuming them.
     *
     * Find all dependencies as well and add the appropriate DependsOn fields.
     */
    protected prepare(): void;
    protected synthesize(session: ISynthesisSession): void;
    /**
     * Exports a resolvable value for use in another stack.
     *
     * @returns a token that can be used to reference the value from the producing stack.
     */
    protected prepareCrossReference(sourceStack: Stack, reference: Reference): IResolvable;
    private getCreateExportsScope;
    /**
     * Determine the various stack environment attributes.
     *
     */
    private parseEnvironment;
    /**
     * Check whether this stack has a (transitive) dependency on another stack
     *
     * Returns the list of reasons on the dependency path, or undefined
     * if there is no dependency.
     */
    private stackDependencyReasons;
    private collectMetadata;
    /**
     * Calculcate the stack name based on the construct path
     */
    private generateUniqueId;
    private generateExportName;
    private get assetParameters();
    private determineCrossReferenceFactory;
    /**
     * Returns all the tokens used within the scope of the current stack.
     */
    private findTokens;
}
/**
 * CloudFormation template options for a stack.
 */
export interface ITemplateOptions {
    /**
     * Gets or sets the description of this stack.
     * If provided, it will be included in the CloudFormation template's "Description" attribute.
     */
    description?: string;
    /**
     * Gets or sets the AWSTemplateFormatVersion field of the CloudFormation template.
     */
    templateFormatVersion?: string;
    /**
     * Gets or sets the top-level template transform for this stack (e.g. "AWS::Serverless-2016-10-31").
     *
     * @deprecated use `transforms` instead.
     */
    transform?: string;
    /**
     * Gets or sets the top-level template transform(s) for this stack (e.g. `["AWS::Serverless-2016-10-31"]`).
     */
    transforms?: string[];
    /**
     * Metadata associated with the CloudFormation template.
     */
    metadata?: {
        [key: string]: any;
    };
}
import { ArnComponents } from './arn';
import { CfnElement } from './cfn-element';
import { CfnResource } from './cfn-resource';
import { Reference } from './reference';
import { IResolvable } from './resolvable';
import { ITaggable, TagManager } from './tag-manager';
