from typing import List
import copy
from dataclasses import dataclass, field

from mindsdb_sql.exceptions import PlanningException
from mindsdb_sql.parser import ast
from mindsdb_sql.parser.ast import (Select, Identifier, BetweenOperation, Join, Star, BinaryOperation, Constant,
                                    NativeQuery, Parameter)
from mindsdb_sql.planner.steps import (FetchDataframeStep, JoinStep, ApplyPredictorStep, SubSelectStep, QueryStep)
from mindsdb_sql.planner.utils import (query_traversal, filters_to_bin_op)
from mindsdb_sql.planner.plan_join_ts import PlanJoinTSPredictorQuery


@dataclass
class TableInfo:
    integration: str
    table: Identifier
    aliases: List[str] = field(default_factory=List)
    conditions: List = None
    sub_select: ast.ASTNode = None
    predictor_info: dict = None


class PlanJoin:

    def __init__(self, planner):
        self.planner = planner

    def is_timeseries(self, query):

        join = query.from_table
        l_predictor = self.planner.get_predictor(join.left) if isinstance(join.left, Identifier) else None
        r_predictor = self.planner.get_predictor(join.right) if isinstance(join.right, Identifier) else None
        if l_predictor and l_predictor.get('timeseries'):
            return True
        if r_predictor and r_predictor.get('timeseries'):
            return True

    def check_single_integration(self, query):
        query_info = self.planner.get_query_info(query)

        # can we send all query to integration?

        # one integration and not mindsdb objects in query
        if (
                len(query_info['mdb_entities']) == 0
                and len(query_info['integrations']) == 1
                and 'files' not in query_info['integrations']
                and 'views' not in query_info['integrations']
        ):

            int_name = list(query_info['integrations'])[0]
            # if is sql database
            if self.planner.integrations.get(int_name, {}).get('class_type') != 'api':

                # send to this integration
                return int_name
        return None

    def plan(self, query, integration=None):
        # FIXME: Tableau workaround, INFORMATION_SCHEMA with Where
        # if isinstance(join.right, Identifier) \
        #         and self.resolve_database_table(join.right)[0] == 'INFORMATION_SCHEMA':
        #     pass

        # send join to integration as is?
        integration_to_send = self.check_single_integration(query)
        if integration_to_send:
            self.planner.prepare_integration_select(integration_to_send, query)

            last_step = self.planner.plan.add_step(FetchDataframeStep(integration=integration_to_send, query=query))
            return last_step
        elif self.is_timeseries(query):
            return PlanJoinTSPredictorQuery(self.planner).plan(query, integration)
        else:
            return PlanJoinTablesQuery(self.planner).plan(query)


class PlanJoinTablesQuery:

    def __init__(self, planner):
        self.planner = planner

        # index to lookup tables
        self.tables_idx = None

        self.step_stack = None
        self.query_context = {}

    def plan(self, query):
        self.tables_idx = {}
        join_step = self.plan_join_tables(query)

        if (
                query.group_by is not None
                or query.order_by is not None
                or query.having is not None
                or query.distinct is True
                or query.where is not None
                or query.limit is not None
                or query.offset is not None
                or len(query.targets) != 1
                or not isinstance(query.targets[0], Star)
        ):
            query2 = copy.deepcopy(query)
            query2.from_table = None
            query2.using = None
            sup_select = QueryStep(query2, from_table=join_step.result)
            self.planner.plan.add_step(sup_select)
            return sup_select
        return join_step

    def resolve_table(self, table):
        # gets integration for table and name to access to it
        table = copy.deepcopy(table)
        # get possible table aliases
        aliases = []
        if table.alias is not None:
            # to lowercase
            parts = tuple(map(str.lower, table.alias.parts))
            aliases.append(parts)
        else:
            for i in range(0, len(table.parts)):
                parts = table.parts[i:]
                parts = tuple(map(str.lower, parts))
                aliases.append(parts)

        # try to use default namespace
        integration = self.planner.default_namespace
        if len(table.parts) > 0:
            if table.parts[0] in self.planner.databases:
                integration = table.parts.pop(0)
            else:
                integration = self.planner.default_namespace

        if integration is None and not hasattr(table, 'sub_select'):
            raise PlanningException(f'Integration not found for: {table}')

        sub_select = getattr(table, 'sub_select', None)

        return TableInfo(integration, table, aliases, conditions=[], sub_select=sub_select)

    def get_table_for_column(self, column: Identifier):

        # to lowercase
        parts = tuple(map(str.lower, column.parts[:-1]))
        if parts in self.tables_idx:
            return self.tables_idx[parts]

    def get_join_sequence(self, node):
        sequence = []
        if isinstance(node, Identifier):
            # resolve identifier

            table_info = self.resolve_table(node)
            for alias in table_info.aliases:
                self.tables_idx[alias] = table_info

            table_info.predictor_info = self.planner.get_predictor(node)

            sequence.append(table_info)

        elif isinstance(node, Join):
            # create sequence: 1)table1, 2)table2, 3)join 1 2, 4)table 3, 5)join 3 4

            # put all tables before
            sequence2 = self.get_join_sequence(node.left)
            for item in sequence2:
                sequence.append(item)

            sequence2 = self.get_join_sequence(node.right)
            if len(sequence2) != 1:
                raise PlanningException('Unexpected join nesting behavior')

            # put next table
            sequence.append(sequence2[0])

            # put join
            sequence.append(node)

        else:
            raise NotImplementedError()
        return sequence

    def check_node_condition(self, node):

        col_idx = 0
        if len(node.args) == 2:
            if not isinstance(node.args[col_idx], Identifier):
                # try to use second arg, could be: 'x'=col
                col_idx = 1

        # check the case col <condition> constant, col between constant and constant
        for i, arg in enumerate(node.args):
            if i == col_idx:
                if not isinstance(arg, Identifier):
                    return
            else:
                if not isinstance(arg, (Constant, Parameter)):
                    return

        # checked, find table and store condition

        node2 = copy.deepcopy(node)

        arg1 = node2.args[col_idx]

        if len(arg1.parts) < 2:
            return

        table_info = self.get_table_for_column(arg1)
        if table_info is None:
            raise PlanningException(f'Table not found for identifier: {arg1.to_string()}')

        # keep only column name
        arg1.parts = [arg1.parts[-1]]

        node2._orig_node = node
        table_info.conditions.append(node2)

    def check_query_conditions(self, query):
        # get conditions for tables
        binary_ops = []

        def _check_node_condition(node, **kwargs):
            if isinstance(node, BetweenOperation):
                self.check_node_condition(node)

            if isinstance(node, BinaryOperation):
                binary_ops.append(node.op)

                self.check_node_condition(node)

        query_traversal(query.where, _check_node_condition)

        self.query_context['binary_ops'] = binary_ops

    def check_use_limit(self, query_in, join_sequence):
        # use limit for first table?
        # if only models
        use_limit = False
        if query_in.having is None or query_in.group_by is None and query_in.limit is not None:

            join = None
            use_limit = True
            for item in join_sequence:
                if isinstance(item, TableInfo):
                    if item.predictor_info is None and item.sub_select is None:
                        if join is not None:
                            if join.join_type.upper() != 'LEFT JOIN':
                                use_limit = False
                elif isinstance(item, Join):
                    join = item
        self.query_context['use_limit'] = use_limit

    def plan_join_tables(self, query_in):

        # plan all nested selects in 'where'
        find_selects = self.planner.get_nested_selects_plan_fnc(self.planner.default_namespace, force=True)
        query_in.targets = query_traversal(query_in.targets, find_selects)
        query_traversal(query_in.where, find_selects)

        query = copy.deepcopy(query_in)

        # replace sub selects, with identifiers with links to original selects
        def replace_subselects(node, **args):
            if isinstance(node, Select) or isinstance(node, NativeQuery) or isinstance(node, ast.Data):
                name = f't_{id(node)}'
                node2 = Identifier(name, alias=node.alias)

                # save in attribute
                if isinstance(node, NativeQuery) or isinstance(node, ast.Data):
                    # wrap to select
                    node = Select(targets=[Star()], from_table=node)
                node2.sub_select = node
                return node2

        query_traversal(query.from_table, replace_subselects)

        # get all join tables, form join sequence
        join_sequence = self.get_join_sequence(query.from_table)

        # find tables for identifiers used in query
        def _check_identifiers(node, is_table, **kwargs):
            if not is_table and isinstance(node, Identifier):
                if len(node.parts) > 1:
                    table_info = self.get_table_for_column(node)
                    if table_info is None:
                        raise PlanningException(f'Table not found for identifier: {node.to_string()}')

                    # # replace identifies name
                    col_parts = list(table_info.aliases[-1])
                    col_parts.append(node.parts[-1])
                    node.parts = col_parts

        query_traversal(query, _check_identifiers)

        self.check_query_conditions(query)

        # workaround for 'model join table': swap tables:
        if len(join_sequence) == 3 and join_sequence[0].predictor_info is not None:
            join_sequence = [join_sequence[1], join_sequence[0], join_sequence[2]]

        self.check_use_limit(query_in, join_sequence)

        # create plan
        # TODO add optimization: one integration without predictor

        self.step_stack = []
        for item in join_sequence:
            if isinstance(item, TableInfo):

                if item.sub_select is not None:
                    self.process_subselect(item)
                elif item.predictor_info is not None:
                    self.process_predictor(item, query_in)
                else:
                    # is table
                    self.process_table(item, query_in)

            elif isinstance(item, Join):
                step_right = self.step_stack.pop()
                step_left = self.step_stack.pop()

                new_join = copy.deepcopy(item)

                # TODO
                new_join.left = Identifier('tab1')
                new_join.right = Identifier('tab2')
                new_join.implicit = False

                step = self.planner.plan.add_step(JoinStep(left=step_left.result, right=step_right.result, query=new_join))

                self.step_stack.append(step)

        query_in.where = query.where
        return self.step_stack.pop()

    def process_subselect(self, item):
        # is sub select
        item.sub_select.alias = None
        item.sub_select.parentheses = False
        step = self.planner.plan_select(item.sub_select)

        where = filters_to_bin_op(item.conditions)

        # apply table alias
        query2 = Select(targets=[Star()], where=where)
        if item.table.alias is None:
            raise PlanningException(f'Subselect in join have to be aliased: {item.sub_select.to_string()}')
        table_name = item.table.alias.parts[-1]

        add_absent_cols = False
        if hasattr(item.sub_select, 'from_table') and \
                isinstance(item.sub_select.from_table, ast.Data):
            add_absent_cols = True

        step2 = SubSelectStep(query2, step.result, table_name=table_name, add_absent_cols=add_absent_cols)
        step2 = self.planner.plan.add_step(step2)
        self.step_stack.append(step2)

    def process_table(self, item, query_in):
        query2 = Select(from_table=item.table, targets=[Star()])
        # parts = tuple(map(str.lower, table_name.parts))
        conditions = item.conditions
        if 'or' in self.query_context['binary_ops']:
            # not use conditions
            conditions = []

        if self.query_context['use_limit']:
            order_by = None
            if query_in.order_by is not None:
                order_by = []
                # all order column be from this table
                for col in query_in.order_by:
                    if self.get_table_for_column(col.field).table != item.table:
                        order_by = False
                        break
                    col = copy.deepcopy(col)
                    col.field.parts = [col.field.parts[-1]]
                    order_by.append(col)

            if order_by is not False:
                # copy limit from upper query
                query2.limit = query_in.limit
                # move offset from upper query
                query2.offset = query_in.offset
                query_in.offset = None
                # copy order
                query2.order_by = order_by

            self.query_context['use_limit'] = False
        for cond in conditions:
            if query2.where is not None:
                query2.where = BinaryOperation('and', args=[query2.where, cond])
            else:
                query2.where = cond

        # step = self.planner.get_integration_select_step(query2)
        step = FetchDataframeStep(integration=item.integration, query=query2)
        self.planner.plan.add_step(step)
        self.step_stack.append(step)

    def process_predictor(self, item, query_in):
        if len(self.step_stack) == 0:
            raise NotImplementedError("Predictor can't be first element of join syntax")
        if item.predictor_info.get('timeseries'):
            raise NotImplementedError("TS predictor is not supported here yet")
        data_step = self.step_stack[-1]
        row_dict = None

        predict_target = item.predictor_info.get('to_predict')
        if isinstance(predict_target, list) and len(predict_target) > 0:
            predict_target = predict_target[0]
        if predict_target is not None:
            predict_target = predict_target.lower()

        if item.conditions:
            row_dict = {}
            for i, el in enumerate(item.conditions):
                if isinstance(el.args[0], Identifier) and el.op == '=':
                    col_name = el.args[0].parts[-1]
                    if col_name.lower() == predict_target:
                        # don't add predict target to parameters
                        continue

                    if isinstance(el.args[1], (Constant, Parameter)):
                        row_dict[el.args[0].parts[-1]] = el.args[1].value

                    # exclude condition
                    el._orig_node.args = [Constant(0), Constant(0)]

        # params for model
        model_params = None

        if query_in.using is not None:
            model_params = {}
            for param, value in query_in.using.items():
                if '.' in param:
                    alias = param.split('.')[0]
                    if (alias,) in item.aliases:
                        new_param = '.'.join(param.split('.')[1:])
                        model_params[new_param] = value
                else:
                    model_params[param] = value

        predictor_step = self.planner.plan.add_step(ApplyPredictorStep(
            namespace=item.integration,
            dataframe=data_step.result,
            predictor=item.table,
            params=model_params,
            row_dict=row_dict,
        ))
        self.step_stack.append(predictor_step)
