"""Vulnerability scanning using multiple security tools and LLM analysis."""

import json
import logging
import re
import shutil
import subprocess
import tempfile
from concurrent.futures import ThreadPoolExecutor
from pathlib import Path
from typing import Any, Optional

logger = logging.getLogger(__name__)


class VulnerabilityScanner:
    """Scan code for security vulnerabilities using tools and patterns."""

    def __init__(self, config: Any):
        """Initialize vulnerability scanner with configuration."""
        self.config = config
        self.vulnerability_patterns = {
            name: re.compile(pattern) for name, pattern in config.vulnerability_patterns.items()
        }

        # Check which tools are available
        self.available_tools = self._detect_available_tools()

    def scan_files(self, files_changed: list[str], repo_path: Path) -> list[dict]:
        """Scan changed files for vulnerabilities.

        Args:
            files_changed: List of changed file paths
            repo_path: Path to repository

        Returns:
            List of vulnerability findings
        """
        findings = []

        # Quick pattern-based scanning
        pattern_findings = self._scan_with_patterns(files_changed, repo_path)
        findings.extend(pattern_findings)

        # Tool-based scanning (run in parallel for performance)
        if self.available_tools:
            tool_findings = self._scan_with_tools(files_changed, repo_path)
            findings.extend(tool_findings)

        return findings

    def _detect_available_tools(self) -> dict[str, bool]:
        """Detect which security tools are installed."""
        tools = {}

        # Check for Semgrep
        if self.config.enable_semgrep:
            tools["semgrep"] = self._is_tool_available("semgrep")
            if not tools["semgrep"]:
                logger.info("Semgrep not found. Install with: pip install semgrep")

        # Check for Bandit (Python)
        if self.config.enable_bandit:
            tools["bandit"] = self._is_tool_available("bandit")
            if not tools["bandit"]:
                logger.info("Bandit not found. Install with: pip install bandit")

        # Check for gosec (Go)
        if self.config.enable_gosec:
            tools["gosec"] = self._is_tool_available("gosec")
            if not tools["gosec"]:
                logger.info("Gosec not found. Install from: https://github.com/securego/gosec")

        return tools

    def _is_tool_available(self, tool_name: str) -> bool:
        """Check if a tool is available in PATH."""
        return shutil.which(tool_name) is not None

    def _scan_with_patterns(self, files_changed: list[str], repo_path: Path) -> list[dict]:
        """Quick pattern-based vulnerability detection."""
        findings = []

        for file_path in files_changed:
            full_path = repo_path / file_path
            if not full_path.exists() or not full_path.is_file():
                continue

            try:
                content = full_path.read_text(encoding="utf-8", errors="ignore")

                for vuln_type, pattern in self.vulnerability_patterns.items():
                    for match in pattern.finditer(content):
                        line_num = content[: match.start()].count("\n") + 1
                        finding = {
                            "type": "vulnerability",
                            "vulnerability_type": vuln_type,
                            "severity": self._get_vuln_severity(vuln_type),
                            "file": file_path,
                            "line": line_num,
                            "message": f"Potential {vuln_type.replace('_', ' ')} detected",
                            "tool": "pattern_matcher",
                            "confidence": "medium",
                        }
                        findings.append(finding)
            except Exception as e:
                logger.debug(f"Error scanning {file_path}: {e}")

        return findings

    def _scan_with_tools(self, files_changed: list[str], repo_path: Path) -> list[dict]:
        """Run security tools on changed files."""
        all_findings = []

        # Group files by language for efficient tool execution
        files_by_language = self._group_files_by_language(files_changed)

        with ThreadPoolExecutor(max_workers=4) as executor:
            futures = []

            # Run Semgrep if available (works on all languages)
            if self.available_tools.get("semgrep"):
                future = executor.submit(self._run_semgrep, files_changed, repo_path)
                futures.append(("semgrep", future))

            # Run Bandit on Python files
            if self.available_tools.get("bandit") and files_by_language.get("python"):
                future = executor.submit(self._run_bandit, files_by_language["python"], repo_path)
                futures.append(("bandit", future))

            # Run gosec on Go files
            if self.available_tools.get("gosec") and files_by_language.get("go"):
                future = executor.submit(self._run_gosec, files_by_language["go"], repo_path)
                futures.append(("gosec", future))

            # Collect results
            for tool_name, future in futures:
                try:
                    findings = future.result(timeout=30)
                    all_findings.extend(findings)
                except Exception as e:
                    logger.warning(f"Error running {tool_name}: {e}")

        return all_findings

    def _group_files_by_language(self, files: list[str]) -> dict[str, list[str]]:
        """Group files by programming language."""
        groups = {}

        language_extensions = {
            "python": [".py"],
            "go": [".go"],
            "javascript": [".js", ".jsx", ".ts", ".tsx"],
            "java": [".java"],
            "ruby": [".rb"],
        }

        for file_path in files:
            path = Path(file_path)
            for language, extensions in language_extensions.items():
                if path.suffix in extensions:
                    if language not in groups:
                        groups[language] = []
                    groups[language].append(file_path)
                    break

        return groups

    def _run_semgrep(self, files: list[str], repo_path: Path) -> list[dict]:
        """Run Semgrep security scanning."""
        findings = []

        if not files:
            return findings

        try:
            # Create temporary file list for semgrep
            with tempfile.NamedTemporaryFile(mode="w", suffix=".txt", delete=False) as f:
                for file_path in files:
                    f.write(f"{file_path}\n")
                file_list_path = f.name

            cmd = [
                "semgrep",
                "--config=auto",  # Use automatic rules
                "--json",
                "--no-error",
                f"--include-list={file_list_path}",
                str(repo_path),
            ]

            result = subprocess.run(cmd, capture_output=True, text=True, cwd=repo_path)

            if result.returncode == 0 and result.stdout:
                data = json.loads(result.stdout)
                for finding in data.get("results", []):
                    findings.append(
                        {
                            "type": "vulnerability",
                            "vulnerability_type": finding.get("check_id", "unknown"),
                            "severity": self._map_semgrep_severity(
                                finding.get("extra", {}).get("severity")
                            ),
                            "file": Path(finding["path"]).relative_to(repo_path).as_posix(),
                            "line": finding.get("start", {}).get("line", 0),
                            "message": finding.get("extra", {}).get(
                                "message", "Security issue detected"
                            ),
                            "tool": "semgrep",
                            "confidence": "high",
                        }
                    )

            # Clean up temp file
            Path(file_list_path).unlink()

        except Exception as e:
            logger.warning(f"Error running Semgrep: {e}")

        return findings

    def _run_bandit(self, files: list[str], repo_path: Path) -> list[dict]:
        """Run Bandit for Python security scanning."""
        findings = []

        if not files:
            return findings

        try:
            # Bandit expects full paths
            full_paths = [str(repo_path / f) for f in files if (repo_path / f).exists()]

            if not full_paths:
                return findings

            cmd = ["bandit", "-f", "json", "-ll", *full_paths]  # Low severity and higher

            result = subprocess.run(cmd, capture_output=True, text=True)

            if result.stdout:
                data = json.loads(result.stdout)
                for finding in data.get("results", []):
                    findings.append(
                        {
                            "type": "vulnerability",
                            "vulnerability_type": finding.get("test_id", "unknown"),
                            "severity": finding.get("issue_severity", "medium").lower(),
                            "file": Path(finding["filename"]).relative_to(repo_path).as_posix(),
                            "line": finding.get("line_number", 0),
                            "message": finding.get("issue_text", "Security issue detected"),
                            "tool": "bandit",
                            "confidence": finding.get("issue_confidence", "medium").lower(),
                        }
                    )

        except Exception as e:
            logger.warning(f"Error running Bandit: {e}")

        return findings

    def _run_gosec(self, files: list[str], repo_path: Path) -> list[dict]:
        """Run gosec for Go security scanning."""
        findings = []

        if not files:
            return findings

        try:
            # gosec works on directories, so we scan the whole repo but filter results
            cmd = ["gosec", "-fmt", "json", "./..."]

            result = subprocess.run(cmd, capture_output=True, text=True, cwd=repo_path)

            if result.stdout:
                data = json.loads(result.stdout)
                for finding in data.get("Issues", []):
                    file_path = Path(finding["file"]).relative_to(repo_path).as_posix()

                    # Only include findings for changed files
                    if file_path in files:
                        findings.append(
                            {
                                "type": "vulnerability",
                                "vulnerability_type": finding.get("rule_id", "unknown"),
                                "severity": self._map_gosec_severity(finding.get("severity")),
                                "file": file_path,
                                "line": int(finding.get("line", "0")),
                                "message": finding.get("details", "Security issue detected"),
                                "tool": "gosec",
                                "confidence": finding.get("confidence", "medium").lower(),
                            }
                        )

        except Exception as e:
            logger.warning(f"Error running gosec: {e}")

        return findings

    def _get_vuln_severity(self, vuln_type: str) -> str:
        """Map vulnerability type to severity."""
        critical_types = ["sql_injection", "command_injection", "path_traversal"]
        high_types = ["xss", "weak_crypto"]

        if vuln_type in critical_types:
            return "critical"
        elif vuln_type in high_types:
            return "high"
        else:
            return "medium"

    def _map_semgrep_severity(self, severity: Optional[str]) -> str:
        """Map Semgrep severity to our severity scale."""
        if not severity:
            return "medium"
        severity = severity.upper()
        if severity == "ERROR":
            return "critical"
        elif severity == "WARNING":
            return "high"
        else:
            return "medium"

    def _map_gosec_severity(self, severity: Optional[str]) -> str:
        """Map gosec severity to our severity scale."""
        if not severity:
            return "medium"
        severity = severity.upper()
        if severity == "HIGH":
            return "critical"
        elif severity == "MEDIUM":
            return "high"
        else:
            return "medium"
