import json
import logging
import os
import sys

from tabulate import tabulate
import tempfile
import subprocess
import uuid
from pingsafe_cli.cli.registry import MissingRequiredFlags, CodeTypeSubParser, LogColors, MissingDependencies, PINGSAFE_JSON, DEFECT_DOJO_GENERIC_FORMAT, OutputFormat, SEVERITY_MAP
from pingsafe_cli.cli.utils import read_json_file, print_output_on_file, get_config_path, get_severity_color, \
    get_version, get_exit_code_on_crash, get_os_and_architecture, get_wrapping_length, wrap_text, get_priority, \
    get_output_file_and_format, get_sarif_payload

LOGGER = logging.getLogger("cli")


def read_result(output_file):
    if os.path.exists(output_file) and os.path.getsize(output_file) > 0:
        issues = read_json_file(output_file)
        issue_cnt = len(issues) if issues is not None else 0
        return issues, issue_cnt
    else:
        return [], 0


def print_sbom(data):
    required_length = get_wrapping_length(4)
    if len(data) > 0:
        table_data = []
        for row in data:
            table_data.append({
                "Name": wrap_text(row["name"], required_length),
                "Version": wrap_text(row["version"], required_length),
                "Type": wrap_text(row["type"], required_length),
                "Purl": wrap_text(row["purl"], required_length)
            })
        print(tabulate(table_data, headers="keys", tablefmt="psql"))
    else:
        LOGGER.info("No packages detected")
    return 0


def vulnerability_parser(args, cache_directory):
    vulnerability_pre_evaluation(args, cache_directory)

    global_config_path = get_config_path(cache_directory)
    global_config_data = read_json_file(global_config_path)

    vulnerability_config_path = get_config_path(cache_directory, CodeTypeSubParser.VULN)
    vulnerability_config_data = read_json_file(vulnerability_config_path)

    output_file = os.path.join(tempfile.gettempdir(), f"{uuid.uuid4()}.json")
    if args.generate_sbom:
        output_file, _ = get_output_file_and_format(args, global_config_data)

    call_vulnerability_scanner(args, cache_directory, output_file, global_config_data)

    # means error occurred during vulnerability scanning / sbom generation
    if not os.path.exists(output_file):
        return get_exit_code_on_crash(cache_directory)

    if args.generate_sbom:
        if args.sbom_format == PINGSAFE_JSON:
            with open(output_file, 'r') as file:
                data = json.load(file)
                print_sbom(data["artifacts"])
        LOGGER.info(f"Sbom generated at path: {output_file}")
        return 0
    else:
        issues, issue_cnt = read_result(output_file)
        os.remove(output_file)
        if issue_cnt > 0:
            return vulnerability_post_evaluation(args, issues, global_config_data, vulnerability_config_data)
        else:
            print(LogColors.OKGREEN + "RESULT\tScan completed. No issue found!" + LogColors.ENDC)
        return 0


def vulnerability_pre_evaluation(args, cache_directory):
    operating_sys, arch = get_os_and_architecture()
    runtime = f"{operating_sys}/{arch}"
    not_supported_runtimes = ["windows/386", "windows/arm"]
    if runtime in not_supported_runtimes:
        LOGGER.info(f"{runtime} runtime not supported for vulnerability scanning")
        sys.exit(0)

    global_config_data = read_json_file(get_config_path(cache_directory))
    output_file, output_format = get_output_file_and_format(args, global_config_data)

    if args.generate_sbom and len(output_file) == 0:
        LOGGER.error("--output-file is required to generate sbom")
        sys.exit(1)

    if args.vuln_format == DEFECT_DOJO_GENERIC_FORMAT and len(output_file) == 0:
        LOGGER.error("--output-file is required")
        sys.exit(1)

    if args.generate_sbom and args.sbom_format is None:
        args.sbom_format = PINGSAFE_JSON

    if args.sbom_format is not None and not args.generate_sbom:
        LOGGER.warning("--sbom-format should be used with --generate-sbom")

    if args.directory == "" and args.docker_image == "":
        raise MissingRequiredFlags("Either --d/--directory or --docker-image is required with vuln")

    if output_format == OutputFormat.SARIF or output_format == OutputFormat.CSV:
        if args.generate_sbom:
            LOGGER.error(f"output-format: {output_format} not supported for sbom")
            sys.exit(1)

        if args.vuln_format == DEFECT_DOJO_GENERIC_FORMAT:
            LOGGER.error(f"output-format: {output_format} not supported for DEFECT_DOJO_GENERIC_FORMAT")
            sys.exit(1)


def vulnerability_post_evaluation(args, issues, global_config_data, vulnerability_config_data):
    exit_code = 0
    for issue in issues:
        if exit_code == 0 and evaluate_exit_strategy(issue, vulnerability_config_data) == 1:
            exit_code = 1

    issue_cnt = len(issues)

    if issue_cnt > 0:
        issues = sorted(issues, key=get_priority)
        if args.vuln_format == PINGSAFE_JSON:
            print_issue_on_console(issues, args.quiet, args.verbose)

        if args.vuln_format == DEFECT_DOJO_GENERIC_FORMAT:
            issues = {
                "findings": issues
            }
        save_issues_on_file(args, issues, global_config_data)
        print("RESULT\tScan completed. Found " + str(issue_cnt) + " issues.")
    else:
        print(LogColors.OKGREEN + "RESULT\tScan completed. No issue found!" + LogColors.ENDC)

    return exit_code


def call_vulnerability_scanner(args, cache_directory, output_file, global_config_data):
    if os.path.exists(output_file):
        os.remove(output_file)

    command = generate_command(args, cache_directory, output_file, global_config_data)
    subprocess.run(command)


def generate_command(args, cache_directory, output_path, global_config_data):
    version = get_version()
    binary_path = os.path.join(cache_directory, "bin", version, "bin_vulnerability_scanner")
    if not os.path.exists(binary_path):
        raise MissingDependencies(f"Missing Vulnerability Scanner Binary: {version}")

    scan_path = args.directory
    if args.docker_image:
        scan_path = args.docker_image

    command = [binary_path, "-p", scan_path, "--output-path", output_path]
    skipped_path = args.skip_paths + global_config_data["pathToIgnore"]

    if args.debug:
        command.append("--debug")

    if args.only_fixed:
        command.append("--only-fixed")

    if args.generate_sbom:
        command.append("--generate-sbom")

    if args.sbom_format:
        command.extend(["--sbom-format", args.sbom_format])

    if args.vuln_format:
        command.extend(["--vuln-format", args.vuln_format])

    if args.docker_image:
        command.extend(["--scan-image", "--platform", args.platform,  "--registry", args.registry])

        if args.username != "":
            command.extend(["--username", args.username])
        if args.password != "":
            command.extend(["--password", args.password])
            command.extend(["--auth-token", args.password])

    if len(skipped_path) > 0:
        for path in skipped_path:
            if not args.docker_image:
                if os.path.isabs(path):
                    LOGGER.error("absolute paths are not allowed in skip-path")
                    sys.exit(1)

                """
                in secretScanner we join skip path with base-dir using
                filepath.join() 
                behaviour of filepath.join()
                 1: join(A, *) => A/*
                 2: join(A, ./*) => A/*
                 3: join(A, */) => A/*
                  
                To make it consistent with other scanners we replace
                */ with ./* and **/ with ./**
                inside sbom-scanner => ./ prefix is removed from skip-path

                so final behaviour (same for iac and secret-scanning as well)
                 * = *         ** = **
                ./* = *       ./** = **
                */ = *        **/ = **
                """
                if path == "*" or path == "**":
                    path = "./" + path

                if path == "*/" or path == "**/":
                    path = "./" + path[:-1]

                # done as vul scanner only allows ./, */, **/, if user just want to pass the dir or filename
                if not path.startswith("*/") and not path.startswith("**/") and not path.startswith("./"):
                    path = "./" + path
            command.extend(["--skip-path", path])
    return command


def evaluate_exit_strategy(issue, vulnerability_config_data):
    whitelisted_severity = vulnerability_config_data["exitStrategy"]["severity"]

    if issue["severity"].upper() in whitelisted_severity:
        return 1

    return 0


def print_issue_on_console(issues, quiet, verbose):
    if verbose:
        print(json.dumps(issues, indent=4))
        return

    table_data = []
    for issue in issues:
        if quiet:
            print(LogColors.FAIL + f'[ISSUE]\tFound {issue["id"]} inside package {issue["package"]} for version {issue["version"]}' + LogColors.ENDC)
        else:
            table_data.append(generate_table_row(issue))

    if len(table_data) > 0:
        print(tabulate(table_data, headers="keys", tablefmt="psql"))


def generate_table_row(issue):
    severity_color = get_severity_color(issue["severity"].upper())
    fixed_version = "-"
    if issue["fixedVersions"] != "":
        fixed_version = issue["fixedVersions"]

    required_length = get_wrapping_length(7)
    return {
        "Id": wrap_text(str(issue["id"]), required_length),
        "Severity": wrap_text(severity_color + issue["severity"] + LogColors.ENDC, required_length),
        "Package": wrap_text(str(issue["package"]), required_length),
        "Version": wrap_text(str(issue["version"]), required_length),
        "Fixed In": wrap_text(str(fixed_version), required_length),
        "CVSSScore": wrap_text(str(issue["CVSSScore"]), required_length),
        "Link": wrap_text(issue["link"], required_length)
    }


def save_issues_on_file(args, filtered_issues, global_config_data):
    _, output_format = get_output_file_and_format(args, global_config_data)
    if output_format == OutputFormat.SARIF:
        filtered_issues = convert_issues_to_sarif(filtered_issues)
    print_output_on_file(args, filtered_issues, global_config_data)


def convert_issues_to_sarif(issues):
    rules = []
    results = []
    sarif_result = get_sarif_payload("PingSafe Vulnerability Scanner")

    for issue in issues:
        rule, result = get_sarif_rule_and_result(issue)
        rules.append(rule)
        results.append(result)
    sarif_result["runs"][0]["results"] = results
    sarif_result["runs"][0]["tool"]["driver"]["rules"] = rules
    return sarif_result


def get_sarif_rule_and_result(issue):
    title = issue.get("id", "")
    short_description = issue.get("id", "")
    full_description = issue.get("description", "")
    help_uri = issue.get("link", "")
    severity = issue.get("severity", "LOW").upper()
    file_path = issue.get("paths", "")[0]
    package = issue.get("package", "")
    version = issue.get("version", "")
    fixed_versions = issue.get("fixedVersions", "")

    return {
        "id": title,
        "name": title,
        "shortDescription": {
            "text": short_description
        },
        "fullDescription": {
            "text": full_description
        },
        **({"helpUri": help_uri} if help_uri is not None and len(help_uri) != 0 else {}),
        "properties": {
            "security-severity": SEVERITY_MAP[severity]
        },
        "help": {
            "text": f"Id: {title}\nSeverity: {severity}\nPackage: {package}\nVersion: {version}\nFix Version: {fixed_versions}\nLocation: {file_path}\nLink: {help_uri}\n",
            "markdown": f"| Id | Severity | Package | Version | Fix Version | Location | Link |\n| --- | --- | --- | --- | --- | --- | --- |\n| {title} | {severity} | {package} | {version} | {fixed_versions} | {file_path} | [url]({help_uri})|\n"
        }
    },{
        "ruleId": title,
        "message": {
            "text": title,
        },
        "locations": [{
            "physicalLocation": {
                "artifactLocation": {
                    "uri": file_path
                }
            }
        }]
    }
