# -*- coding: utf-8 -*-

# Mathmaker creates automatically maths exercises sheets
# with their answers
# Copyright 2006-2017 Nicolas Hainaux <nh.techn@gmail.com>

# This file is part of Mathmaker.

# Mathmaker is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# any later version.

# Mathmaker is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with Mathmaker; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

import os
import sys
import logging
import subprocess
from pathlib import Path

MATHMAKER_PATH = None
TEXLIVE_PATH = None
__myname__ = 'mathmaker'


def safe_home():
    """
    Return a usable ‘home’ path even if Path.home() or $HOME is invalid.
    Logs the decision steps.
    """
    logger = logging.getLogger(__name__)

    env_home = os.environ.get("HOME")
    path_home = Path.home()
    logger.debug(f"$HOME={env_home!r} ; Path.home()={str(path_home)!r}")

    # Usual case
    if path_home != Path("/") and path_home.exists():
        logger.debug(f"Using Path.home() : {path_home}")
        return path_home

    # Case of root with invalid home (or / as home)
    if os.geteuid() == 0:
        root_path = Path("/root")
        if root_path.exists():
            logger.warning(
                f"Invalid Path.home() ({path_home}) for root, "
                f"making use of {root_path}"
            )
            return root_path

    # Final fallback
    fallback = Path("/var/lib/mathmaker")
    if not fallback.exists():
        try:
            fallback.mkdir(parents=True, exist_ok=True)
            logger.warning(
                f"Create fallback directory {fallback} "
                f"because Path.home() is invalid ({path_home})"
            )
        except Exception as e:
            logger.error(f"Unable to create the fallback directory "
                         f"{fallback}: {e}")
    else:
        logger.warning(
            f"Making use of fallback directory {fallback} "
            f"because Path.home() is invalid ({path_home})"
        )

    return fallback


def load_search_paths():
    """
    Load search paths, starting with default ones in settings/default,
    possibly overwritten by the contents of /etc/mathmaker/mathmaker_paths.txt,
    then by the contents of ~/.config/mathmaker/mathmaker_paths.txt and
    finally by the dev settings.
    """
    logger = logging.getLogger('server')
    py_version = f"{sys.version_info.major}{sys.version_info.minor}"

    config_files = [
        Path(__file__).parent.parent / "settings/default/mathmaker_paths.txt",
        Path("/etc/mathmaker/mathmaker_paths.txt"),
        Path(safe_home()) / ".config/mathmaker/mathmaker_paths.txt",
        Path(__file__).parent.parent / "settings/dev/mathmaker_paths.txt",
    ]

    paths = []

    for config_file in config_files:
        if config_file.exists():
            logger.debug(f"Loading paths from {config_file}")
            try:
                with open(config_file, 'r') as f:
                    current_paths = []
                    for line in f:
                        line = line.strip()
                        if line and not line.startswith('#'):
                            path = line.replace('%%PY%%', py_version)
                            if path.startswith('~/'):
                                path = str(Path(safe_home()) / path[2:])
                            current_paths.append(path)

                    paths = current_paths
                    logger.debug(f"Loaded {len(paths)} paths "
                                 f"from {config_file}")

            except Exception as e:
                logger.warning(f"Error reading {config_file}: {e}")

    if not paths:
        logger.error("No valid path configuration found!")

    return paths


def find_mathmaker_script():
    """Find mathmaker script using the paths list."""
    logger = logging.getLogger('server')
    search_paths = load_search_paths()

    logger.debug(f"Searching mathmaker in {len(search_paths)} locations")

    for path in search_paths:
        path_obj = Path(path)
        logger.debug(f"Checking: {path}")

        if path_obj.exists() and path_obj.is_file():
            if os.access(path, os.X_OK):
                logger.info(f"Found mathmaker at: {path}")
                return str(path_obj)
            else:
                logger.warning(f"Found existing but not executable "
                               f"file: {path}")

    logger.error("mathmaker not found. Searched paths:")
    for i, path in enumerate(search_paths, 1):
        logger.error(f"  {i}. {path}")

    raise FileNotFoundError(f"mathmaker script not found in "
                            f"{len(search_paths)} locations")


def initialize_mathmaker_path():
    """Initialize and check mathmaker path at start"""
    global MATHMAKER_PATH
    logger = logging.getLogger('server')

    try:
        MATHMAKER_PATH = find_mathmaker_script()
        logger.info(f"Mathmaker initialized: {MATHMAKER_PATH}")
        return True
    except FileNotFoundError as e:
        logger.error(f"Failed to initialize mathmaker: {e}")
        logger.error("Daemon will not start without mathmaker available")
        return False


def get_mathmaker_path():
    """Get mathmaker path (after it has been initialized)"""
    if MATHMAKER_PATH is None:
        raise RuntimeError("Mathmaker not initialized. Call "
                           "initialize_mathmaker_path() first.")
    return MATHMAKER_PATH


def initialize_texlive_path():
    """
    Return TeXLive install directory, based on the result of tlmgr version.
    Remove any /skeleton prefix.

    Example: tlmgr version output:
    tlmgr revision 75204 (2025-05-13 23:48:24 +0200)
    tlmgr using installation: /skeleton/usr/local/texlive/2025
    TeX Live (https://tug.org/texlive) version 2025

    Result: /usr/local/texlive/2025
    """
    global TEXLIVE_PATH
    logger = logging.getLogger('server')
    try:
        output = subprocess.check_output(
            ['tlmgr', 'version'],
            text=True,
            stderr=subprocess.STDOUT
        )
    except subprocess.CalledProcessError as e:
        logger.error(f'Error calling tlmgr: {e.output.strip()}')
        return False
    except FileNotFoundError:
        logger.error('tlmgr impossible to find from PATH')
        return False

    for line in output.splitlines():
        if line.lower().startswith('tlmgr using installation:'):
            path = line.split(':', 1)[1].strip()
            if path.startswith('/skeleton'):
                path = path[len('/skeleton'):]
            TEXLIVE_PATH = path
            logger.info(f'TeXLive found at {TEXLIVE_PATH}')
            return True

    logger.error('TeXLive path cannot be found in tlmgr output')
    return False


def get_texlive_path():
    """Get TeXLive path (after it has been initialized)"""
    if TEXLIVE_PATH is None:
        raise RuntimeError("TeXLive path not initialized. Call "
                           "initialize_texlive_path() first.")
    return TEXLIVE_PATH


USER_LOCAL_SHARE = os.path.join(str(safe_home()), '.local', 'share',
                                __myname__)
