# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, unused-wildcard-import
"""
.. _`German Federal Institute of Hydrology (BfG)`: https://www.bafg.de/EN

Version 2 of H-Land is very similar to |hland_v1|.  The only difference is in runoff
concentration.  |hland_v2| uses a linear storage cascade instead of a triangle Unit
Hydrograph to route both runoff components to the subbasin's outlet.  In some cases,
the results of |hland_v2| should look more natural than the ones of |hland_v1|.  On the
whole, however, the differences should be minor.  The real reason for the modification
lies in the availability of "physical" storage contents instead of purely mathematical
intermediate Unit Hydrograph results as initial conditions, which can simplify data
assimilation.  We implemented |hland_v2| on behalf of the `German Federal Institute of
Hydrology (BfG)`_ for runoff forecasting in large river basins in central Europe.


The following figure shows the general structure of H-Land Version 2:

.. image:: HydPy-H-Land_Version-2.png


Integration tests
=================

.. how_to_understand_integration_tests::

The following configurations agree with the documentation on |hland_v1|.  The only
difference lies in the additional parameter |NmbStorages|.  In |hland_v1|, we can
modify runoff concentration by parameter |MaxBaz| only, which corresponds to the
response function's width.  In |hland_v2|, we can also alter the response function's
shape.  Here, we set |NmbStorages| to five, which results in the highest similarity
with the fixed-shape triangular response function of |hland_v1|:

>>> from hydpy import pub
>>> pub.timegrids = "01.01.2000", "05.01.2000", "1h"

>>> from hydpy.models.hland_v2 import *
>>> parameterstep("1h")
>>> from hydpy import Node, Element
>>> outlet = Node("outlet")
>>> land = Element("land", outlets=outlet)
>>> land.model = model

>>> nmbzones(1)
>>> sclass(1)
>>> area(1.0)
>>> zonearea(1.0)
>>> zonez(1.0)
>>> zrelp(2.0)
>>> zrelt(2.0)
>>> zrele(2.0)
>>> psi(1.0)

>>> from hydpy import IntegrationTest
>>> IntegrationTest.plotting_options.axis1 = inputs.p, fluxes.rt, fluxes.qt
>>> IntegrationTest.plotting_options.axis2 = inputs.t
>>> test = IntegrationTest(land)
>>> test.dateformat = "%d/%m %H:00"

.. _hland_v2_field:

field
_____

>>> zonetype(FIELD)

>>> pcorr(1.2)
>>> pcalt(0.1)
>>> rfcf(1.1)
>>> sfcf(1.3)
>>> tcalt(0.6)
>>> ecorr(0.8)
>>> ecalt(-0.1)
>>> epf(0.1)
>>> etf(0.1)
>>> ered(0.5)
>>> icmax(2.0)
>>> sfdist(1.0)
>>> smax(inf)
>>> sred(0.0)
>>> tt(0.0)
>>> ttint(2.0)
>>> dttm(1.0)
>>> cfmax(0.5)
>>> cfvar(0.1)
>>> gmelt(1.0)
>>> gvar(0.2)
>>> cfr(0.1)
>>> whc(0.4)
>>> fc(200.0)
>>> lp(0.8)
>>> beta(2.0)
>>> percmax(0.5)
>>> cflux(0.1)
>>> alpha(2.0)
>>> k(0.001)
>>> k4(0.005)
>>> gamma(0.0)
>>> maxbaz(3.0)
>>> nmbstorages(5)
>>> resparea(False)
>>> recstep(100)

>>> test.inits = ((states.ic, 0.0),
...               (states.sp, 0.0),
...               (states.wc, 0.0),
...               (states.sm, 100.0),
...               (states.uz, 0.0),
...               (states.lz, 10.0),
...               (states.sc, 0.05))

>>> inputs.p.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.2, 0.0, 0.0, 1.3, 5.6, 2.9, 4.9, 10.6, 0.1, 0.7, 3.0, 2.1, 10.4, 3.5, 3.4,
...     1.2, 0.1, 0.0, 0.0, 0.4, 0.1, 3.6, 5.9, 1.1, 20.7, 37.9, 8.2, 3.6, 7.5, 18.5,
...     15.4, 6.3, 1.9, 4.9, 2.7, 0.5, 0.2, 0.5, 2.4, 0.4, 0.2, 0.0, 0.0, 0.3, 2.6,
...     0.7, 0.3, 0.3, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     1.3, 0.0, 0.0, 0.0, 0.7, 0.4, 0.1, 0.4, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0)
>>> inputs.t.series = (
...     21.2, 19.4, 18.9, 18.3, 18.9, 22.5, 25.1, 28.3, 27.8, 31.4, 32.2, 35.2, 37.1,
...     31.2, 24.3, 25.4, 25.9, 23.7, 21.6, 21.2, 20.4, 19.8, 19.6, 19.2, 19.2, 19.2,
...     18.9, 18.7, 18.5, 18.3, 18.5, 18.8, 18.8, 19.0, 19.2, 19.3, 19.0, 18.8, 18.7,
...     17.8, 17.4, 17.3, 16.8, 16.5, 16.3, 16.2, 15.5, 14.6, 14.7, 14.6, 14.1, 14.3,
...     14.9, 15.7, 16.0, 16.7, 17.1, 16.2, 15.9, 16.3, 16.3, 16.4, 16.5, 18.4, 18.3,
...     18.1, 16.7, 15.2, 13.4, 12.4, 11.6, 11.0, 10.5, 11.7, 11.9, 11.2, 11.1, 11.9,
...     12.2, 11.8, 11.4, 11.6, 13.0, 17.1, 18.2, 22.4, 21.4, 21.8, 22.2, 20.1, 17.8,
...     15.2, 14.5, 12.4, 11.7, 11.9)
>>> inputs.tn.series = inputs.t.series - 1.0
>>> inputs.epn.series = (
...     0.100707, 0.097801, 0.096981, 0.09599, 0.096981, 0.102761, 0.291908, 1.932875,
...     4.369536, 7.317556, 8.264362, 9.369867, 5.126178, 6.62503, 7.397619, 2.39151,
...     1.829834, 1.136569, 0.750986, 0.223895, 0.099425, 0.098454, 0.098128, 0.097474,
...     0.097474, 0.097474, 0.096981, 0.096652, 0.096321, 0.09599, 0.187298, 1.264612,
...     3.045538, 1.930758, 2.461001, 6.215945, 3.374783, 8.821555, 4.046025, 2.110757,
...     2.239257, 2.877848, 1.591452, 0.291604, 0.092622, 0.092451, 0.091248, 0.089683,
...     0.089858, 0.089683, 0.088805, 0.089157, 0.090207, 0.091593, 0.154861, 0.470369,
...     1.173726, 4.202296, 4.359715, 5.305753, 5.376027, 4.658915, 7.789594, 4.851567,
...     5.30692, 3.286036, 1.506216, 0.274762, 0.087565, 0.085771, 0.084317, 0.083215,
...     0.082289, 0.0845, 0.084864, 0.083584, 0.0834, 0.084864, 0.310229, 1.391958,
...     3.195876, 5.191651, 7.155036, 8.391432, 8.391286, 10.715238, 9.383394, 7.861915,
...     6.298329, 2.948416, 1.309232, 0.32955, 0.089508, 0.085771, 0.0845, 0.084864)


>>> test.reset_inits()
>>> conditions = sequences.conditions

Besides slight differences in runoff concentration, the following calculation
reproduces the results of the :ref:`hland_v1_field` experiment of |hland_v1| exactly:

.. integration-test::

    >>> test("hland_v2_field")
    |        date |    p |    t |   tn |       epn | tmean |   tc | fracrain | rfc | sfc |    cfact | swe | gact | contriarea |      pc |        ep |      epc |       ei |        tf | spl | wcl | spg | wcg | glmelt | melt | refr |       in_ |         r |  sr |       ea |       cf |      inuz |     perc |        q0 |  el |       q1 |      inuh |    outuh |       rt |       qt |       ic |  sp |  wc |         sm |        uz |        lz |                                               sc |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |   0.11682 |  0.08411 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.052569 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |     0.05 |      0.05 | 0.163676 | 0.163676 | 0.045466 |      0.0 | 0.0 | 0.0 |  99.947431 |       0.0 |      9.95 | 0.015696  0.019685  0.026398  0.034007  0.040538 | 0.045466 |
    | 01/01 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.113449 | 0.081683 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.051025 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.04975 |   0.04975 | 0.102365 | 0.102365 | 0.028435 |      0.0 | 0.0 | 0.0 |  99.896406 |       0.0 |   9.90025 |  0.01447  0.014747  0.015662  0.017698  0.021131 | 0.028435 |
    | 01/01 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.050572 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049501 |  0.049501 | 0.059979 | 0.059979 | 0.016661 |      0.0 | 0.0 | 0.0 |  99.845834 |       0.0 |  9.850749 | 0.014359  0.014381  0.014465  0.014714  0.015311 | 0.016661 |
    | 01/01 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.111348 | 0.080171 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.05003 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049254 |  0.049254 | 0.050799 | 0.050799 | 0.014111 |      0.0 | 0.0 | 0.0 |  99.795804 |       0.0 |  9.801495 | 0.014286  0.014295  0.014315  0.014353  0.014436 | 0.014111 |
    | 01/01 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.050521 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049007 |  0.049007 |  0.04947 |  0.04947 | 0.013742 |      0.0 | 0.0 | 0.0 |  99.745284 |       0.0 |  9.752488 | 0.014215  0.014223  0.014239   0.01426  0.014286 | 0.013742 |
    | 01/01 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.119203 | 0.085826 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.053505 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048762 |  0.048762 | 0.049126 | 0.049126 | 0.013646 |      0.0 | 0.0 | 0.0 |  99.691779 |       0.0 |  9.703725 | 0.014144  0.014152  0.014168  0.014187  0.014209 | 0.013646 |
    | 01/01 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.338613 | 0.243802 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.151906 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048519 |  0.048519 | 0.048873 | 0.048873 | 0.013576 |      0.0 | 0.0 | 0.0 |  99.539873 |       0.0 |  9.655206 | 0.014073  0.014081  0.014097  0.014116  0.014138 | 0.013576 |
    | 01/01 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  2.242135 | 1.614337 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.004318 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048276 |  0.048276 | 0.048629 | 0.048629 | 0.013508 |      0.0 | 0.0 | 0.0 |  98.535555 |       0.0 |   9.60693 | 0.014003  0.014011  0.014026  0.014046  0.014067 | 0.013508 |
    | 01/01 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  5.068662 | 3.649436 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.247495 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048035 |  0.048035 | 0.048385 | 0.048385 |  0.01344 |      0.0 | 0.0 | 0.0 |  96.288059 |       0.0 |  9.558896 | 0.013932  0.013941  0.013956  0.013976  0.013997 |  0.01344 |
    | 01/01 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  8.488365 | 6.111623 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.677977 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047794 |  0.047794 | 0.048143 | 0.048143 | 0.013373 |      0.0 | 0.0 | 0.0 |  92.610082 |       0.0 |  9.511101 | 0.013863  0.013871  0.013886  0.013906  0.013927 | 0.013373 |
    | 01/01 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |   9.58666 | 6.902395 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.995196 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047556 |  0.047556 | 0.047903 | 0.047903 | 0.013306 |      0.0 | 0.0 | 0.0 |  88.614886 |       0.0 |  9.463546 | 0.013794  0.013802  0.013817  0.013836  0.013857 | 0.013306 |
    | 01/01 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 | 10.869046 | 7.825713 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.334217 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047318 |  0.047318 | 0.047663 | 0.047663 |  0.01324 |      0.0 | 0.0 | 0.0 |   84.28067 |       0.0 |  9.416228 | 0.013725  0.013733  0.013748  0.013767  0.013788 |  0.01324 |
    | 01/01 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  5.946366 | 4.281384 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.255237 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047081 |  0.047081 | 0.047425 | 0.047425 | 0.013174 |      0.0 | 0.0 | 0.0 |  82.025433 |       0.0 |  9.369147 | 0.013656  0.013664  0.013679  0.013698  0.013719 | 0.013174 |
    | 01/01 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  7.685035 | 5.533225 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.836657 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046846 |  0.046846 | 0.047188 | 0.047188 | 0.013108 |      0.0 | 0.0 | 0.0 |  79.188775 |       0.0 |  9.322301 | 0.013588  0.013596  0.013611   0.01363   0.01365 | 0.013108 |
    | 01/01 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  8.581238 | 6.178491 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.05792 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046612 |  0.046612 | 0.046952 | 0.046952 | 0.013042 |      0.0 | 0.0 | 0.0 |  76.130856 |       0.0 |   9.27569 |  0.01352  0.013528  0.013543  0.013562  0.013582 | 0.013042 |
    | 01/01 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  0.2376 |  2.774152 | 1.950491 |   0.2376 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.928078 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046378 |  0.046378 | 0.046717 | 0.046717 | 0.012977 |      0.0 | 0.0 | 0.0 |  75.202777 |       0.0 |  9.229311 | 0.013452   0.01346  0.013475  0.013494  0.013514 | 0.012977 |
    | 01/01 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  2.122607 | 1.528277 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.718317 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046147 |  0.046147 | 0.046484 | 0.046484 | 0.012912 |      0.0 | 0.0 | 0.0 |  74.484461 |       0.0 |  9.183165 | 0.013385  0.013393  0.013408  0.013426  0.013446 | 0.012912 |
    | 01/01 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |   1.31842 | 0.949262 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.441908 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045916 |  0.045916 | 0.046251 | 0.046251 | 0.012848 |      0.0 | 0.0 | 0.0 |  74.042552 |       0.0 |  9.137249 | 0.013318  0.013326   0.01334  0.013359  0.013379 | 0.012848 |
    | 01/01 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  1.5444 |  0.871144 | 0.537465 | 0.537465 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.12436 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045686 |  0.045686 |  0.04602 |  0.04602 | 0.012783 | 1.006935 | 0.0 | 0.0 |  73.918192 |       0.0 |  9.091563 | 0.013251  0.013259  0.013274  0.013292  0.013312 | 0.012783 |
    | 01/01 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  6.6528 |  0.259718 | 0.096141 | 0.096141 |  5.659735 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.659735 |  0.773106 | 0.0 | 0.023694 | 0.060598 |  0.712508 |      0.5 |  0.000002 | 0.0 | 0.047958 |   0.04796 | 0.045968 | 0.045968 | 0.012769 | 1.903859 | 0.0 | 0.0 |  78.841725 |  0.212506 |  9.543605 | 0.013886  0.013813   0.01369  0.013553  0.013439 | 0.012769 |
    | 01/01 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  3.4452 |  0.115333 | 0.058839 | 0.058839 |  3.349059 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.349059 |  0.520445 | 0.0 | 0.015028 | 0.059165 |   0.46128 |      0.5 |  0.000007 | 0.0 | 0.050218 |  0.050225 |  0.04717 |  0.04717 | 0.013103 | 1.941161 | 0.0 | 0.0 |  81.714476 |  0.173779 |  9.993387 | 0.014542  0.014464  0.014323  0.014145  0.013962 | 0.013103 |
    | 01/01 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  5.8212 |  0.114207 | 0.045942 | 0.045942 |  5.762361 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.762361 |  0.961919 | 0.0 | 0.012429 | 0.056743 |  0.905176 |      0.5 |   0.00007 | 0.0 | 0.052467 |  0.052537 | 0.049235 | 0.049235 | 0.013676 | 1.954058 | 0.0 | 0.0 |  86.559232 |  0.578886 |  10.44092 | 0.015212  0.015132  0.014987  0.014802  0.014605 | 0.013676 |
    | 01/01 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 | 12.5928 |  0.113828 | 0.023264 | 0.023264 | 12.546858 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.546858 |  2.350184 | 0.0 | 0.007038 | 0.051622 |  2.298561 |      0.5 |  0.004477 | 0.0 | 0.054705 |  0.059182 | 0.051812 | 0.051812 | 0.014392 | 1.976736 | 0.0 | 0.0 |  96.800491 |   2.37297 | 10.886215 | 0.017097  0.016875  0.016492  0.016037  0.015606 | 0.014392 |
    | 01/01 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  0.1188 |   0.11307 | 0.080449 | 0.080449 |  0.095536 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.095536 |   0.02238 | 0.0 | 0.024367 | 0.051563 | -0.029183 |      0.5 |   0.00942 | 0.0 | 0.056931 |  0.066351 | 0.056635 | 0.056635 | 0.015732 | 1.919551 | 0.0 | 0.0 |  96.900842 |  1.834367 | 11.329284 | 0.019169  0.018921  0.018478  0.017918  0.017337 | 0.015732 |
    | 02/01 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.8316 |   0.11307 | 0.074914 | 0.074914 |  0.751151 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.751151 |  0.176328 | 0.0 | 0.022832 | 0.051262 |  0.125066 |      0.5 |  0.004491 | 0.0 | 0.059146 |  0.063637 | 0.062514 | 0.062514 | 0.017365 | 1.925086 | 0.0 | 0.0 |  97.504096 |  1.454943 | 11.770138 | 0.018479  0.018551  0.018643  0.018681  0.018592 | 0.017365 |
    | 02/01 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |   3.564 |   0.11307 | 0.057003 | 0.057003 |  3.489086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.489086 |  0.829273 | 0.0 | 0.017851 | 0.049918 |  0.779355 |      0.5 |  0.004082 | 0.0 | 0.061351 |  0.065432 | 0.064266 | 0.064266 | 0.017852 | 1.942997 | 0.0 | 0.0 | 100.195975 |  1.730216 | 12.208787 | 0.018959  0.018905  0.018823  0.018741  0.018684 | 0.017852 |
    | 02/01 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  2.4948 |  0.112498 | 0.063115 | 0.063115 |  2.437797 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.437797 |   0.61184 | 0.0 | 0.020132 | 0.048989 |  0.562851 |      0.5 |  0.005447 | 0.0 | 0.063544 |  0.068991 | 0.065067 | 0.065067 | 0.018074 | 1.936885 | 0.0 | 0.0 | 102.050789 |   1.78762 | 12.645243 | 0.019972  0.019852  0.019645  0.019398  0.019167 | 0.018074 |
    | 02/01 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 12.3552 |  0.112116 | 0.023465 | 0.023465 | 12.292085 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.292085 |  3.200356 | 0.0 | 0.008153 | 0.044429 |  3.155927 |      0.5 |  0.035664 | 0.0 | 0.065726 |   0.10139 | 0.069652 | 0.069652 | 0.019348 | 1.976535 | 0.0 | 0.0 | 111.178794 |  4.407884 | 13.079517 | 0.029085  0.028022  0.026218  0.024142  0.022307 | 0.019348 |
    | 02/01 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |   4.158 |  0.111732 |  0.05308 |  0.05308 |  4.134535 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.134535 |  1.277646 | 0.0 | 0.018923 | 0.042982 |  1.234664 |      0.5 |  0.106278 | 0.0 | 0.067898 |  0.174175 | 0.092411 | 0.092411 |  0.02567 |  1.94692 | 0.0 | 0.0 | 114.059742 |   5.03627 | 13.511619 | 0.049789  0.047346  0.043114  0.038048  0.033242 |  0.02567 |
    | 02/01 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.0392 |  0.111348 |  0.05353 |  0.05353 |   3.98612 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.98612 |  1.296448 | 0.0 | 0.019537 | 0.041625 |  1.254823 |      0.5 |  0.153262 | 0.0 | 0.070058 |   0.22332 | 0.143817 | 0.143817 | 0.039949 |  1.94647 | 0.0 | 0.0 | 116.771503 |  5.637831 | 13.941561 | 0.064258  0.062494  0.059245  0.054927  0.050117 | 0.039949 |
    | 02/01 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  1.4256 |  0.217266 | 0.135647 | 0.135647 |   1.37207 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   1.37207 |  0.467724 | 0.0 |   0.0499 | 0.041162 |  0.426562 |      0.5 |  0.167893 | 0.0 | 0.072208 |  0.240101 | 0.197141 | 0.197141 | 0.054761 | 1.864353 | 0.0 | 0.0 |  117.66711 |    5.3965 | 14.369353 | 0.069448  0.068785  0.067462  0.065466  0.062839 | 0.054761 |
    | 02/01 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.1188 |   1.46695 |  1.04373 |  1.04373 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.383924 | 0.041166 | -0.041166 |      0.5 |  0.129717 | 0.0 | 0.074347 |  0.204064 | 0.225543 | 0.225543 | 0.062651 | 0.939422 | 0.0 | 0.0 | 117.324353 |  4.725616 | 14.795007 | 0.059525  0.060656  0.062479  0.064339  0.065523 | 0.062651 |
    | 02/01 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |     0.0 |  3.532824 | 2.543633 | 0.939422 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.735028 | 0.041338 | -0.041338 |      0.5 |  0.086003 | 0.0 | 0.076475 |  0.162478 | 0.214585 | 0.214585 | 0.059607 |      0.0 | 0.0 | 0.0 | 115.630662 |  4.098276 | 15.218532 | 0.047537  0.048969  0.051503  0.054644  0.057762 | 0.059607 |
    | 02/01 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |     0.0 |  2.239679 | 1.612569 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.165815 | 0.042185 | -0.042185 |      0.5 |  0.054998 | 0.0 | 0.078593 |  0.133591 | 0.180987 | 0.180987 | 0.050274 |      0.0 | 0.0 | 0.0 | 114.507032 |  3.501093 | 15.639939 | 0.039038  0.040074  0.041984   0.04453  0.047392 | 0.050274 |
    | 02/01 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.4752 |  2.854761 | 1.960038 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.403262 | 0.042746 | -0.042746 |      0.5 |  0.033312 | 0.0 |   0.0807 |  0.114012 | 0.148331 | 0.148331 | 0.041203 |      0.0 | 0.0 | 0.0 | 113.146516 |  2.925034 | 16.059239 | 0.033265  0.033971  0.035282  0.037058  0.039123 | 0.041203 |
    | 02/01 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.1188 |  7.210496 | 5.130246 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.629327 | 0.043427 | -0.043427 |      0.5 |  0.018617 | 0.0 | 0.082796 |  0.101413 | 0.124317 | 0.124317 | 0.034532 |      0.0 | 0.0 | 0.0 | 109.560616 |  2.362991 | 16.476443 |  0.02954  0.029997  0.030849  0.032016  0.033394 | 0.034532 |
    | 02/01 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.2768 |  3.914748 | 1.837796 | 1.837796 |    2.2768 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    2.2768 |  0.683241 | 0.0 | 0.638626 | 0.044423 |  0.638818 |      0.5 |  0.014291 | 0.0 | 0.084882 |  0.099173 | 0.108584 | 0.108584 | 0.030162 | 0.162204 | 0.0 | 0.0 | 110.559972 |  2.487517 | 16.891561 | 0.028787   0.02889  0.029117  0.029506  0.030086 | 0.030162 |
    | 02/01 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  7.0092 | 10.233004 | 3.655358 |      2.0 |  5.171404 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.171404 |  1.580317 | 0.0 | 2.132116 | 0.042924 |  1.537393 |      0.5 |  0.027763 | 0.0 | 0.086958 |  0.114721 | 0.102503 | 0.102503 | 0.028473 |      0.0 | 0.0 | 0.0 | 112.061866 |  3.497147 | 17.304603 | 0.033118  0.032618  0.031789  0.030887   0.03019 | 0.028473 |
    | 02/01 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  1.3068 |  4.693389 | 2.965278 |   1.3068 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.868067 | 0.043969 | -0.043969 |      0.5 |  0.033174 | 0.0 | 0.089023 |  0.122197 | 0.108928 | 0.108928 | 0.030258 |      0.0 | 0.0 | 0.0 | 110.237768 |  2.920004 |  17.71558 | 0.035359  0.035082  0.034559  0.033844  0.033028 | 0.030258 |
    | 02/01 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 24.5916 |  2.448478 |  0.15074 |  0.15074 |   22.5916 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   22.5916 |  6.863535 | 0.0 | 0.059355 | 0.037017 |  6.826517 |      0.5 |  0.275437 | 0.0 | 0.091078 |  0.366515 | 0.135184 | 0.135184 | 0.037551 |  1.84926 | 0.0 | 0.0 | 125.943496 |  8.971084 | 18.124502 | 0.103899  0.095921  0.082458   0.06712  0.053806 | 0.037551 |
    | 02/01 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 45.0252 |  2.597538 | 0.020724 | 0.020724 |  44.87446 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  44.87446 | 17.794702 | 0.0 | 0.009912 | 0.023488 | 17.771214 |      0.5 |  4.821721 | 0.0 | 0.093123 |  4.914843 | 0.585858 | 0.585858 | 0.162738 | 1.979276 | 0.0 | 0.0 |  153.03683 | 21.420577 |  18.53138 | 1.380769  1.232047  0.980801  0.694047  0.444526 | 0.162738 |
    | 02/01 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  9.7416 |  3.338304 | 0.907373 | 0.907373 |  9.720876 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.720876 |  5.691639 | 0.0 | 0.445428 | 0.021467 |  5.670172 |      0.5 |  7.878038 | 0.0 | 0.095157 |  7.973195 | 3.166508 | 3.166508 | 0.879586 | 1.092627 | 0.0 | 0.0 | 156.642106 |  18.71271 | 18.936223 | 2.280833  2.171176  1.969849  1.704194  1.412823 | 0.879586 |
    | 02/01 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.2768 |  1.846084 | 0.866652 | 0.866652 |  3.369427 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.369427 |  2.066869 | 0.0 | 0.427817 | 0.021028 |  2.045842 |      0.5 |  4.840645 | 0.0 | 0.097181 |  4.937827 | 6.068673 | 6.068673 | 1.685742 | 1.133348 | 0.0 | 0.0 | 157.537874 | 15.417906 | 19.339042 | 1.460579  1.552249  1.693676  1.823303  1.878223 | 1.685742 |
    | 02/01 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |    8.91 |  0.338261 | 0.099914 | 0.099914 |  8.043348 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  8.043348 |  4.990532 | 0.0 | 0.049957 | 0.019705 |  4.970827 |      0.5 |    3.9218 | 0.0 | 0.099195 |  4.020995 | 5.873581 | 5.873581 |  1.63155 | 1.900086 | 0.0 | 0.0 | 160.560437 | 15.966933 | 19.739846 | 1.176014  1.212188  1.282945  1.383657   1.50064 |  1.63155 |
    | 02/01 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  21.978 |  0.107442 |  0.00859 |  0.00859 | 21.878086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 21.878086 | 14.100237 | 0.0 | 0.004295 | 0.015831 | 14.084406 |      0.5 |  7.677383 | 0.0 | 0.101199 |  7.778582 | 4.899384 | 4.899384 |  1.36094 |  1.99141 | 0.0 | 0.0 | 168.349822 | 21.873957 | 20.138647 | 2.219397  2.099309  1.901557  1.688315  1.526065 |  1.36094 |
    | 02/01 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 18.2952 |  0.107243 | 0.012392 | 0.012392 |  18.28661 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  18.28661 | 12.956823 | 0.0 | 0.006196 |  0.01316 | 12.943663 |      0.5 | 11.510418 | 0.0 | 0.103193 | 11.613611 | 6.546898 | 6.546898 | 1.818583 | 1.987608 | 0.0 | 0.0 | 173.686572 | 22.807202 | 20.535454 | 3.329237  3.196156  2.958793  2.660512  2.356657 | 1.818583 |
    | 02/01 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  7.4844 |  0.105848 | 0.036055 | 0.036055 |  7.472008 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.472008 |  5.635206 | 0.0 | 0.018028 | 0.012238 |  5.622968 |      0.5 |  8.843986 | 0.0 | 0.105177 |  8.949163 | 9.482369 | 9.482369 | 2.633991 | 1.963945 | 0.0 | 0.0 | 175.517585 | 19.086184 | 20.930277 | 2.619995  2.697968  2.813823  2.909249  2.927113 | 2.633991 |
    | 02/01 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  2.2572 |  0.104032 | 0.059768 | 0.059768 |  2.221145 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.221145 |  1.710638 | 0.0 | 0.029884 | 0.011986 |  1.698652 |      0.5 |  4.947934 | 0.0 | 0.107151 |  5.055086 | 9.458187 | 9.458187 | 2.627274 | 1.940232 | 0.0 | 0.0 | 176.010193 | 15.336902 | 21.323125 | 1.504869  1.637145  1.868051  2.146369  2.408613 | 2.627274 |
    | 03/01 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  5.8212 |  0.104235 | 0.041931 | 0.041931 |  5.761432 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.761432 |  4.462169 | 0.0 | 0.020966 | 0.011345 |  4.450824 |      0.5 |  3.725205 | 0.0 | 0.109116 |   3.83432 | 6.814636 | 6.814636 | 1.892954 | 1.958069 | 0.0 | 0.0 | 177.299835 | 15.562521 |  21.71401 | 1.125196  1.173796  1.270745  1.415295  1.599702 | 1.892954 |
    | 03/01 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  3.2076 |  0.104032 |  0.05435 |  0.05435 |  3.165669 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.165669 |  2.487838 | 0.0 | 0.027175 | 0.011011 |  2.476827 |      0.5 |  3.288302 | 0.0 |  0.11107 |  3.399372 | 4.670584 | 4.670584 | 1.297385 |  1.94565 | 0.0 | 0.0 | 177.961502 | 14.251047 |  22.10294 | 0.990515  1.007812  1.042897  1.097725  1.174571 | 1.297385 |
    | 03/01 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |   0.594 |  0.103014 | 0.069893 | 0.069893 |   0.53965 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.53965 |  0.427272 | 0.0 | 0.034946 | 0.010963 |  0.416309 |      0.5 |  2.220294 | 0.0 | 0.113015 |  2.333308 |  3.65034 |  3.65034 | 1.013983 | 1.930107 | 0.0 | 0.0 | 178.049897 | 11.947062 | 22.489925 | 0.687237   0.72308  0.785561   0.86189   0.93872 | 1.013983 |
    | 03/01 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.2376 |  0.103422 | 0.072716 | 0.072716 |  0.167707 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.167707 |  0.132916 | 0.0 | 0.036358 | 0.010958 |  0.121958 |      0.5 |  1.351723 | 0.0 |  0.11495 |  1.466673 | 2.778988 | 2.778988 | 0.771941 | 1.927284 | 0.0 | 0.0 | 178.059289 | 10.217297 | 22.874975 | 0.434162  0.464794  0.520455  0.592959  0.671803 | 0.771941 |
    | 03/01 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |   0.594 |   0.10464 | 0.070996 | 0.070996 |  0.521284 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.521284 |  0.413185 | 0.0 | 0.035498 | 0.010916 |  0.402268 |      0.5 |  0.912964 | 0.0 | 0.116875 |  1.029839 | 1.900164 | 1.900164 | 0.527823 | 1.929004 | 0.0 | 0.0 | 178.142807 |  9.206601 |   23.2581 |  0.30322  0.319477   0.35042  0.393954  0.446777 | 0.527823 |
    | 03/01 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  2.8512 |  0.106248 | 0.057521 | 0.057521 |  2.780204 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.780204 |  2.205735 | 0.0 | 0.028761 | 0.010641 |  2.195093 |      0.5 |  0.896634 | 0.0 | 0.118791 |  1.015425 | 1.298338 | 1.298338 |  0.36065 | 1.942479 | 0.0 | 0.0 | 178.699157 |  10.00506 |  23.63931 | 0.294768  0.296299  0.300822  0.310715  0.328331 |  0.36065 |
    | 03/01 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.4752 |  0.179639 | 0.123337 | 0.123337 |  0.417679 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.417679 |  0.333448 | 0.0 | 0.061669 | 0.010608 |  0.322839 |      0.5 |  0.851988 | 0.0 | 0.120697 |  0.972684 | 1.064082 | 1.064082 | 0.295578 | 1.876663 | 0.0 | 0.0 | 178.732328 |  8.975912 | 24.018613 | 0.282506  0.283983  0.286707  0.290558  0.295781 | 0.295578 |
    | 03/01 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.2376 |  0.545628 | 0.383628 | 0.383628 |  0.114263 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.114263 |  0.091254 | 0.0 | 0.191814 | 0.010622 |  0.080631 |      0.5 |  0.604807 | 0.0 | 0.122593 |    0.7274 | 0.979708 | 0.979708 | 0.272141 | 1.616372 | 0.0 | 0.0 | 178.574145 |  7.951736 |  24.39602 | 0.213358  0.221447  0.235238   0.25129  0.265895 | 0.272141 |
    | 03/01 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  1.361522 | 0.980296 | 0.980296 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.490148 | 0.010713 | -0.010713 |      0.5 |  0.419019 | 0.0 |  0.12448 |  0.543499 | 0.823347 | 0.823347 | 0.228707 | 0.636076 | 0.0 | 0.0 |  178.09471 |  7.022004 |  24.77154 | 0.159493  0.166029  0.177944  0.193511  0.210404 | 0.228707 |
    | 03/01 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  4.874663 | 3.509758 | 0.636076 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.19172 | 0.010953 | -0.010953 |      0.5 |  0.290078 | 0.0 | 0.126358 |  0.416436 |   0.6339 |   0.6339 | 0.176083 |      0.0 | 0.0 | 0.0 | 174.913943 |  6.220974 | 25.145182 | 0.122072  0.126643  0.135102  0.146502  0.159598 | 0.176083 |
    | 03/01 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.3564 |  5.057269 | 3.513746 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.335546 | 0.012543 | -0.012543 |      0.5 |  0.201588 | 0.0 | 0.128226 |  0.329814 | 0.482279 | 0.482279 | 0.133966 |      0.0 | 0.0 | 0.0 | 171.590941 |  5.506843 | 25.516957 | 0.096537  0.099659  0.105456  0.113318  0.122481 | 0.133966 |
    | 03/01 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  3.0888 |  6.154673 | 3.253813 |      2.0 |    1.0888 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    1.0888 |  0.801451 | 0.0 | 2.253813 | 0.014061 |   0.78739 |      0.5 |  0.172588 | 0.0 | 0.130085 |  0.302673 | 0.377466 | 0.377466 | 0.104852 |      0.0 | 0.0 | 0.0 | 169.638537 |  5.621644 | 25.886872 |  0.08807   0.08916  0.091363  0.094748  0.099317 | 0.104852 |
    | 03/01 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.8316 |  6.236191 | 4.131769 |   0.8316 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.715969 | 0.015181 | -0.015181 |      0.5 |  0.148076 | 0.0 | 0.131934 |   0.28001 |  0.32125 |  0.32125 | 0.089236 |      0.0 | 0.0 | 0.0 | 165.937749 |  4.958388 | 26.254937 | 0.081434  0.082243  0.083738  0.085778  0.088225 | 0.089236 |
    | 03/01 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.3564 |  5.404341 | 3.754888 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.576688 | 0.017031 | -0.017031 |      0.5 |  0.100609 | 0.0 | 0.133775 |  0.234383 | 0.289661 | 0.289661 | 0.080461 |      0.0 | 0.0 | 0.0 | 162.378092 |  4.340748 | 26.621163 | 0.068425  0.069965  0.072652  0.075927  0.079172 | 0.080461 |
    | 03/01 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.3564 |  9.035929 | 6.278083 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.099883 | 0.018811 | -0.018811 |      0.5 |  0.066455 | 0.0 | 0.135606 |  0.202061 | 0.253824 | 0.253824 | 0.070507 |      0.0 | 0.0 | 0.0 |  156.29702 |  3.755482 | 26.985557 | 0.058929  0.060085  0.062207  0.065017   0.06814 | 0.070507 |
    | 03/01 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  5.627818 | 4.052029 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.958804 | 0.021851 | -0.021851 |      0.5 |  0.042037 | 0.0 | 0.137428 |  0.179465 | 0.218308 | 0.218308 | 0.060641 |      0.0 | 0.0 | 0.0 | 152.360068 |  3.191593 | 27.348129 | 0.052277  0.053089  0.054592   0.05662  0.058956 | 0.060641 |
    | 03/01 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  6.156027 |  4.43234 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.221357 |  0.02382 |  -0.02382 |      0.5 |  0.024952 | 0.0 | 0.139241 |  0.164192 | 0.191186 | 0.191186 | 0.053107 |      0.0 | 0.0 | 0.0 | 148.162531 |  2.642821 | 27.708889 | 0.047773  0.048324  0.049347  0.050737   0.05236 | 0.053107 |
    | 03/01 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  3.811802 | 2.744497 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.541892 | 0.025919 | -0.025919 |      0.5 |  0.013481 | 0.0 | 0.141044 |  0.154525 | 0.172264 | 0.172264 | 0.047851 |      0.0 | 0.0 | 0.0 | 145.646557 |  2.103421 | 28.067844 | 0.044912  0.045263   0.04592   0.04682  0.047887 | 0.047851 |
    | 03/01 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  1.747211 | 1.257992 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.145352 | 0.027177 | -0.027177 |      0.5 |  0.006295 | 0.0 | 0.142839 |  0.149135 | 0.159792 | 0.159792 | 0.044387 |      0.0 | 0.0 | 0.0 | 144.528382 |  1.569949 | 28.425005 | 0.043305  0.043504   0.04388  0.044407  0.045049 | 0.044387 |
    | 03/01 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.318724 | 0.229481 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.207331 | 0.027736 | -0.027736 |      0.5 |  0.002299 | 0.0 | 0.144625 |  0.146924 | 0.152254 | 0.152254 | 0.042293 |      0.0 | 0.0 | 0.0 | 144.348787 |  1.039914 |  28.78038 | 0.042631  0.042716  0.042883   0.04313  0.043453 | 0.042293 |
    | 03/01 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.101575 | 0.073134 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.065993 | 0.027826 | -0.027826 |      0.5 |  0.000516 | 0.0 | 0.146402 |  0.146918 | 0.148425 | 0.148425 | 0.041229 |      0.0 | 0.0 | 0.0 |  144.31062 |  0.511573 | 29.133978 | 0.042607  0.042613  0.042633  0.042681  0.042774 | 0.041229 |
    | 03/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.064624 | 0.027845 | -0.027845 | 0.483697 |  0.000032 | 0.0 | 0.148088 |   0.14812 | 0.147251 | 0.147251 | 0.040903 |      0.0 | 0.0 | 0.0 |  144.27384 |       0.0 | 29.469586 | 0.042943  0.042904   0.04284   0.04277   0.04272 | 0.040903 |
    | 03/01 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.097808 | 0.070422 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |   0.0635 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.147348 |  0.147348 | 0.147578 | 0.147578 | 0.040994 |      0.0 | 0.0 | 0.0 |  144.21034 |       0.0 | 29.322238 | 0.042738  0.042761  0.042794  0.042823  0.042832 | 0.040994 |
    | 03/01 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.096529 | 0.069501 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.062642 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.146611 |  0.146611 | 0.147528 | 0.147528 |  0.04098 |      0.0 | 0.0 | 0.0 | 144.147698 |       0.0 | 29.175627 | 0.042525   0.04255  0.042596  0.042652  0.042707 |  0.04098 |
    | 04/01 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.095455 | 0.068728 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.061918 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.145878 |  0.145878 | 0.146927 | 0.146927 | 0.040813 |      0.0 | 0.0 | 0.0 |  144.08578 |       0.0 | 29.029749 | 0.042312  0.042338  0.042384  0.042443  0.042505 | 0.040813 |
    | 04/01 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.063555 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.145149 |  0.145149 | 0.146207 | 0.146207 | 0.040613 |      0.0 | 0.0 | 0.0 | 144.022225 |       0.0 |   28.8846 | 0.042101  0.042126  0.042172  0.042231  0.042294 | 0.040613 |
    | 04/01 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |   0.0638 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.144423 |  0.144423 | 0.145478 | 0.145478 |  0.04041 |      0.0 | 0.0 | 0.0 | 143.958424 |       0.0 | 28.740177 |  0.04189  0.041915  0.041961   0.04202  0.042083 |  0.04041 |
    | 04/01 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  1.5444 |  0.096957 | 0.059819 | 0.059819 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.026911 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.143701 |  0.143701 |  0.14475 |  0.14475 | 0.040208 | 1.484581 | 0.0 | 0.0 | 143.931513 |       0.0 | 28.596476 | 0.041681  0.041706  0.041751  0.041809  0.041872 | 0.040208 |
    | 04/01 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.096744 | 0.069656 | 0.069656 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.03133 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.142982 |  0.142982 | 0.144026 | 0.144026 | 0.040007 | 1.414925 | 0.0 | 0.0 | 143.900183 |       0.0 | 28.453494 | 0.041472  0.041497  0.041542    0.0416  0.041663 | 0.040007 |
    | 04/01 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 | 0.070878 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.031873 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.142267 |  0.142267 | 0.143306 | 0.143306 | 0.039807 | 1.344047 | 0.0 | 0.0 |  143.86831 |       0.0 | 28.311226 | 0.041265   0.04129  0.041335  0.041392  0.041454 | 0.039807 |
    | 04/01 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.359866 | 0.259103 | 0.259103 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.11649 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.141556 |  0.141556 |  0.14259 |  0.14259 | 0.039608 | 1.084943 | 0.0 | 0.0 |  143.75182 |       0.0 |  28.16967 | 0.041058  0.041083  0.041128  0.041185  0.041247 | 0.039608 |
    | 04/01 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.8316 |  1.614671 | 1.069795 | 1.069795 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.480578 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.140848 |  0.140848 | 0.141877 | 0.141877 |  0.03941 | 0.846748 | 0.0 | 0.0 | 143.271242 |       0.0 | 28.028822 | 0.040853  0.040878  0.040922   0.04098  0.041041 |  0.03941 |
    | 04/01 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.4752 |  3.707216 | 2.545322 | 1.321948 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.751285 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.140144 |  0.140144 | 0.141167 | 0.141167 | 0.039213 |      0.0 | 0.0 | 0.0 | 141.519957 |       0.0 | 27.888678 | 0.040649  0.040673  0.040718  0.040775  0.040836 | 0.039213 |
    | 04/01 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.1188 |  6.022315 | 4.284859 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.789957 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.139443 |  0.139443 | 0.140462 | 0.140462 | 0.039017 |      0.0 | 0.0 | 0.0 |     137.73 |       0.0 | 27.749234 | 0.040446   0.04047  0.040514  0.040571  0.040632 | 0.039017 |
    | 04/01 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.4752 |  8.299842 | 5.698554 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.905386 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.138746 |  0.138746 | 0.139759 | 0.139759 | 0.038822 |      0.0 | 0.0 | 0.0 | 132.824614 |       0.0 | 27.610488 | 0.040243  0.040268  0.040312  0.040368  0.040428 | 0.038822 |
    | 04/01 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  9.734061 | 7.008524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.818153 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.138052 |  0.138052 | 0.139061 | 0.139061 | 0.038628 |      0.0 | 0.0 | 0.0 | 127.006461 |       0.0 | 27.472436 | 0.040042  0.040066   0.04011  0.040166  0.040226 | 0.038628 |
    | 04/01 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  9.733892 | 7.008402 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.563202 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.137362 |  0.137362 | 0.138365 | 0.138365 | 0.038435 |      0.0 | 0.0 | 0.0 | 121.443259 |       0.0 | 27.335074 | 0.039842  0.039866  0.039909  0.039965  0.040025 | 0.038435 |
    | 04/01 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 | 12.429676 | 8.949367 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.792752 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.136675 |  0.136675 | 0.137673 | 0.137673 | 0.038243 |      0.0 | 0.0 | 0.0 | 114.650507 |       0.0 | 27.198398 | 0.039643  0.039667   0.03971  0.039765  0.039825 | 0.038243 |
    | 04/01 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 | 10.884737 | 7.837011 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.615733 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.135992 |  0.135992 | 0.136985 | 0.136985 | 0.038051 |      0.0 | 0.0 | 0.0 | 109.034774 |       0.0 | 27.062406 | 0.039445  0.039468  0.039511  0.039567  0.039626 | 0.038051 |
    | 04/01 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  9.119821 | 6.566271 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |   4.4747 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.135312 |  0.135312 |   0.1363 |   0.1363 | 0.037861 |      0.0 | 0.0 | 0.0 | 104.560075 |       0.0 | 26.927094 | 0.039247  0.039271  0.039314  0.039369  0.039428 | 0.037861 |
    | 04/01 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  7.306062 | 5.260364 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.437651 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.134635 |  0.134635 | 0.135619 | 0.135619 | 0.037672 |      0.0 | 0.0 | 0.0 | 101.122424 |       0.0 | 26.792459 | 0.039051  0.039075  0.039117  0.039172  0.039231 | 0.037672 |
    | 04/01 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  3.420163 | 2.462517 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.556348 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.133962 |  0.133962 | 0.134941 | 0.134941 | 0.037483 |      0.0 | 0.0 | 0.0 |  99.566076 |       0.0 | 26.658496 | 0.038856  0.038879  0.038922  0.038976  0.039034 | 0.037483 |
    | 04/01 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  1.518709 | 1.093471 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.680454 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.133292 |  0.133292 | 0.134266 | 0.134266 | 0.037296 |      0.0 | 0.0 | 0.0 |  98.885622 |       0.0 | 26.525204 | 0.038662  0.038685  0.038727  0.038781  0.038839 | 0.037296 |
    | 04/01 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.382278 |  0.27524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.170108 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.132626 |  0.132626 | 0.133594 | 0.133594 |  0.03711 |      0.0 | 0.0 | 0.0 |  98.715514 |       0.0 | 26.392578 | 0.038468  0.038492  0.038533  0.038587  0.038645 |  0.03711 |
    | 04/01 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.103829 | 0.074757 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046123 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.131963 |  0.131963 | 0.132927 | 0.132927 | 0.036924 |      0.0 | 0.0 | 0.0 |  98.669391 |       0.0 | 26.260615 | 0.038276  0.038299  0.038341  0.038394  0.038452 | 0.036924 |
    | 04/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.044177 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.131303 |  0.131303 | 0.132262 | 0.132262 | 0.036739 |      0.0 | 0.0 | 0.0 |  98.625215 |       0.0 | 26.129312 | 0.038085  0.038108  0.038149  0.038202   0.03826 | 0.036739 |
    | 04/01 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043503 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.130647 |  0.130647 | 0.131601 | 0.131601 | 0.036556 |      0.0 | 0.0 | 0.0 |  98.581712 |       0.0 | 25.998665 | 0.037894  0.037917  0.037958  0.038011  0.038068 | 0.036556 |
    | 04/01 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043671 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.129993 |  0.129993 | 0.130943 | 0.130943 | 0.036373 |      0.0 | 0.0 | 0.0 |  98.538041 |       0.0 | 25.868672 | 0.037705  0.037727  0.037769  0.037821  0.037878 | 0.036373 |

There is no indication of an error in the water balance:

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v2_resparea:

contributing area
_________________

Besides slight differences in runoff concentration, the following calculation
reproduces the results of the :ref:`hland_v1_resparea` experiment of |hland_v1| exactly:

.. integration-test::

    >>> resparea(True)
    >>> test("hland_v2_resparea")
    |        date |    p |    t |   tn |       epn | tmean |   tc | fracrain | rfc | sfc |    cfact | swe | gact | contriarea |      pc |        ep |      epc |       ei |        tf | spl | wcl | spg | wcg | glmelt | melt | refr |       in_ |         r |  sr |       ea |       cf |      inuz |     perc |        q0 |  el |       q1 |      inuh |    outuh |       rt |       qt |       ic |  sp |  wc |         sm |        uz |        lz |                                               sc |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.249737 |     0.0 |   0.11682 |  0.08411 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.052569 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |     0.05 |      0.05 | 0.163676 | 0.163676 | 0.045466 |      0.0 | 0.0 | 0.0 |  99.947431 |       0.0 |      9.95 | 0.015696  0.019685  0.026398  0.034007  0.040538 | 0.045466 |
    | 01/01 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.249482 |     0.0 |  0.113449 | 0.081683 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.051025 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.04975 |   0.04975 | 0.102365 | 0.102365 | 0.028435 |      0.0 | 0.0 | 0.0 |  99.896406 |       0.0 |   9.90025 |  0.01447  0.014747  0.015662  0.017698  0.021131 | 0.028435 |
    | 01/01 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.24923 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.050572 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049501 |  0.049501 | 0.059979 | 0.059979 | 0.016661 |      0.0 | 0.0 | 0.0 |  99.845834 |       0.0 |  9.850749 | 0.014359  0.014381  0.014465  0.014714  0.015311 | 0.016661 |
    | 01/01 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.24898 |     0.0 |  0.111348 | 0.080171 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.05003 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049254 |  0.049254 | 0.050799 | 0.050799 | 0.014111 |      0.0 | 0.0 | 0.0 |  99.795804 |       0.0 |  9.801495 | 0.014286  0.014295  0.014315  0.014353  0.014436 | 0.014111 |
    | 01/01 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.248728 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.050521 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049007 |  0.049007 |  0.04947 |  0.04947 | 0.013742 |      0.0 | 0.0 | 0.0 |  99.745284 |       0.0 |  9.752488 | 0.014215  0.014223  0.014239   0.01426  0.014286 | 0.013742 |
    | 01/01 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.248461 |     0.0 |  0.119203 | 0.085826 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.053505 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048762 |  0.048762 | 0.049126 | 0.049126 | 0.013646 |      0.0 | 0.0 | 0.0 |  99.691779 |       0.0 |  9.703725 | 0.014144  0.014152  0.014168  0.014187  0.014209 | 0.013646 |
    | 01/01 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.247705 |     0.0 |  0.338613 | 0.243802 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.151906 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048519 |  0.048519 | 0.048873 | 0.048873 | 0.013576 |      0.0 | 0.0 | 0.0 |  99.539873 |       0.0 |  9.655206 | 0.014073  0.014081  0.014097  0.014116  0.014138 | 0.013576 |
    | 01/01 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.242731 |     0.0 |  2.242135 | 1.614337 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.004318 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048276 |  0.048276 | 0.048629 | 0.048629 | 0.013508 |      0.0 | 0.0 | 0.0 |  98.535555 |       0.0 |   9.60693 | 0.014003  0.014011  0.014026  0.014046  0.014067 | 0.013508 |
    | 01/01 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.231785 |     0.0 |  5.068662 | 3.649436 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.247495 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048035 |  0.048035 | 0.048385 | 0.048385 |  0.01344 |      0.0 | 0.0 | 0.0 |  96.288059 |       0.0 |  9.558896 | 0.013932  0.013941  0.013956  0.013976  0.013997 |  0.01344 |
    | 01/01 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.214416 |     0.0 |  8.488365 | 6.111623 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.677977 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047794 |  0.047794 | 0.048143 | 0.048143 | 0.013373 |      0.0 | 0.0 | 0.0 |  92.610082 |       0.0 |  9.511101 | 0.013863  0.013871  0.013886  0.013906  0.013927 | 0.013373 |
    | 01/01 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.196315 |     0.0 |   9.58666 | 6.902395 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.995196 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047556 |  0.047556 | 0.047903 | 0.047903 | 0.013306 |      0.0 | 0.0 | 0.0 |  88.614886 |       0.0 |  9.463546 | 0.013794  0.013802  0.013817  0.013836  0.013857 | 0.013306 |
    | 01/01 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.177581 |     0.0 | 10.869046 | 7.825713 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.334217 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047318 |  0.047318 | 0.047663 | 0.047663 |  0.01324 |      0.0 | 0.0 | 0.0 |   84.28067 |       0.0 |  9.416228 | 0.013725  0.013733  0.013748  0.013767  0.013788 |  0.01324 |
    | 01/01 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.168204 |     0.0 |  5.946366 | 4.281384 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.255237 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047081 |  0.047081 | 0.047425 | 0.047425 | 0.013174 |      0.0 | 0.0 | 0.0 |  82.025433 |       0.0 |  9.369147 | 0.013656  0.013664  0.013679  0.013698  0.013719 | 0.013174 |
    | 01/01 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.156772 |     0.0 |  7.685035 | 5.533225 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.836657 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046846 |  0.046846 | 0.047188 | 0.047188 | 0.013108 |      0.0 | 0.0 | 0.0 |  79.188775 |       0.0 |  9.322301 | 0.013588  0.013596  0.013611   0.01363   0.01365 | 0.013108 |
    | 01/01 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.144898 |     0.0 |  8.581238 | 6.178491 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.05792 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046612 |  0.046612 | 0.046952 | 0.046952 | 0.013042 |      0.0 | 0.0 | 0.0 |  76.130856 |       0.0 |   9.27569 |  0.01352  0.013528  0.013543  0.013562  0.013582 | 0.013042 |
    | 01/01 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.141386 |  0.2376 |  2.774152 | 1.950491 |   0.2376 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.928078 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046378 |  0.046378 | 0.046717 | 0.046717 | 0.012977 |      0.0 | 0.0 | 0.0 |  75.202777 |       0.0 |  9.229311 | 0.013452   0.01346  0.013475  0.013494  0.013514 | 0.012977 |
    | 01/01 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.138698 |     0.0 |  2.122607 | 1.528277 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.718317 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046147 |  0.046147 | 0.046484 | 0.046484 | 0.012912 |      0.0 | 0.0 | 0.0 |  74.484461 |       0.0 |  9.183165 | 0.013385  0.013393  0.013408  0.013426  0.013446 | 0.012912 |
    | 01/01 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.137057 |     0.0 |   1.31842 | 0.949262 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.441908 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045916 |  0.045916 | 0.046251 | 0.046251 | 0.012848 |      0.0 | 0.0 | 0.0 |  74.042552 |       0.0 |  9.137249 | 0.013318  0.013326   0.01334  0.013359  0.013379 | 0.012848 |
    | 01/01 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.136597 |  1.5444 |  0.871144 | 0.537465 | 0.537465 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.12436 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045686 |  0.045686 |  0.04602 |  0.04602 | 0.012783 | 1.006935 | 0.0 | 0.0 |  73.918192 |       0.0 |  9.091563 | 0.013251  0.013259  0.013274  0.013292  0.013312 | 0.012783 |
    | 01/01 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.1554 |  6.6528 |  0.259718 | 0.096141 | 0.096141 |  5.659735 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.659735 |  0.773106 | 0.0 | 0.023694 | 0.060598 |  0.712508 |   0.0777 |  0.016629 | 0.0 | 0.045846 |  0.062475 | 0.047004 | 0.047004 | 0.013057 | 1.903859 | 0.0 | 0.0 |  78.841725 |  0.618179 |  9.123417 | 0.017954  0.017407  0.016488  0.015449  0.014561 | 0.013057 |
    | 01/01 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.166931 |  3.4452 |  0.115333 | 0.058839 | 0.058839 |  3.349059 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.349059 |  0.520445 | 0.0 | 0.015028 | 0.059165 |   0.46128 | 0.083466 |  0.100569 | 0.0 | 0.046034 |  0.146603 | 0.061317 | 0.061317 | 0.017033 | 1.941161 | 0.0 | 0.0 |  81.714476 |  0.895425 |  9.160848 | 0.041687  0.038909  0.034172  0.028666   0.02371 | 0.017033 |
    | 01/01 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.187313 |  5.8212 |  0.114207 | 0.045942 | 0.045942 |  5.762361 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.762361 |  0.961919 | 0.0 | 0.012429 | 0.056743 |  0.905176 | 0.093656 |  0.275044 | 0.0 | 0.046273 |  0.321317 | 0.121686 | 0.121686 | 0.033802 | 1.954058 | 0.0 | 0.0 |  86.559232 |  1.431901 |  9.208232 | 0.091446  0.085568  0.075357  0.063078  0.051327 | 0.033802 |
    | 01/01 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.234258 | 12.5928 |  0.113828 | 0.023264 | 0.023264 | 12.546858 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.546858 |  2.350184 | 0.0 | 0.007038 | 0.051622 |  2.298561 | 0.117129 |  0.903885 | 0.0 | 0.046627 |  0.950512 | 0.284334 | 0.284334 | 0.078982 | 1.976736 | 0.0 | 0.0 |  96.800491 |  2.709448 |  9.278734 |  0.26944  0.248548  0.212708  0.170556  0.131702 | 0.078982 |
    | 01/01 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.234744 |  0.1188 |   0.11307 | 0.080449 | 0.080449 |  0.095536 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.095536 |   0.02238 | 0.0 | 0.024367 | 0.051563 | -0.029183 | 0.117372 |  0.805699 | 0.0 | 0.046981 |  0.852679 | 0.658981 | 0.658981 |  0.18305 | 1.919551 | 0.0 | 0.0 |  96.900842 |  1.757194 |  9.349126 | 0.248024   0.24981  0.250446  0.245798  0.232575 |  0.18305 |
    | 02/01 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.237676 |  0.8316 |   0.11307 | 0.074914 | 0.074914 |  0.751151 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.751151 |  0.176328 | 0.0 | 0.022832 | 0.051262 |  0.125066 | 0.118838 |   0.30497 | 0.0 |  0.04734 |   0.35231 |  0.80494 |  0.80494 | 0.223595 | 1.925086 | 0.0 | 0.0 |  97.504096 |  1.458453 |  9.420624 | 0.107086  0.123532  0.151337   0.18285  0.209218 | 0.223595 |
    | 02/01 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.250981 |   3.564 |   0.11307 | 0.057003 | 0.057003 |  3.489086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.489086 |  0.829273 | 0.0 | 0.017851 | 0.049918 |  0.779355 |  0.12549 |  0.292533 | 0.0 | 0.047731 |  0.340263 | 0.567409 | 0.567409 | 0.157614 | 1.942997 | 0.0 | 0.0 | 100.195975 |  1.819784 |  9.498384 |  0.09896  0.100459  0.104834  0.113932  0.128692 | 0.157614 |
    | 02/01 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.260359 |  2.4948 |  0.112498 | 0.063115 | 0.063115 |  2.437797 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.437797 |   0.61184 | 0.0 | 0.020132 | 0.048989 |  0.562851 |  0.13018 |  0.365199 | 0.0 | 0.048143 |  0.413342 | 0.392483 | 0.392483 | 0.109023 | 1.936885 | 0.0 | 0.0 | 102.050789 |  1.887257 |   9.58042 | 0.119164  0.116866  0.113227  0.109828  0.108649 | 0.109023 |
    | 02/01 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.309018 | 12.3552 |  0.112116 | 0.023465 | 0.023465 | 12.292085 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.292085 |  3.200356 | 0.0 | 0.008153 | 0.044429 |  3.155927 | 0.154509 |  1.047457 | 0.0 | 0.048675 |  1.096132 | 0.436706 | 0.436706 | 0.121307 | 1.976535 | 0.0 | 0.0 | 111.178794 |  3.841219 |  9.686255 | 0.311181  0.288779  0.250819   0.20727  0.169111 | 0.121307 |
    | 02/01 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.325241 |   4.158 |  0.111732 |  0.05308 |  0.05308 |  4.134535 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.134535 |  1.277646 | 0.0 | 0.018923 | 0.042982 |  1.234664 |  0.16262 |  1.415468 | 0.0 | 0.049244 |  1.464713 | 0.824756 | 0.824756 | 0.229099 |  1.94692 | 0.0 | 0.0 | 114.059742 |  3.497794 |  9.799631 | 0.420938  0.407423  0.382146  0.347832  0.308778 | 0.229099 |
    | 02/01 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.34089 |  4.0392 |  0.111348 |  0.05353 |  0.05353 |   3.98612 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.98612 |  1.296448 | 0.0 | 0.019537 | 0.041625 |  1.254823 | 0.170445 |  1.088288 | 0.0 |  0.04985 |  1.138139 | 1.226454 | 1.226454 | 0.340682 |  1.94647 | 0.0 | 0.0 | 116.771503 |  3.493884 |  9.920225 | 0.333123  0.342877  0.357693  0.370619  0.374489 | 0.340682 |
    | 02/01 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.346139 |  1.4256 |  0.217266 | 0.135647 | 0.135647 |   1.37207 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   1.37207 |  0.467724 | 0.0 |   0.0499 | 0.041162 |  0.426562 | 0.173069 |  0.792891 | 0.0 | 0.050466 |  0.843358 | 1.219626 | 1.219626 | 0.338785 | 1.864353 | 0.0 | 0.0 |  117.66711 |  2.954486 | 10.042828 | 0.247558  0.257835  0.276176    0.2991  0.321865 | 0.338785 |
    | 02/01 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.344125 |  0.1188 |   1.46695 |  1.04373 |  1.04373 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.383924 | 0.041166 | -0.041166 | 0.172063 |  0.439147 | 0.0 | 0.051074 |  0.490221 |  0.96718 |  0.96718 | 0.268661 | 0.939422 | 0.0 | 0.0 | 117.324353 |   2.30211 | 10.163816 | 0.145716  0.157903  0.179607  0.206973  0.235376 | 0.268661 |
    | 02/01 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.334261 |     0.0 |  3.532824 | 2.543633 | 0.939422 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.735028 | 0.041338 | -0.041338 | 0.167131 |  0.238811 | 0.0 | 0.051655 |  0.290465 | 0.654307 | 0.654307 | 0.181752 |      0.0 | 0.0 | 0.0 | 115.630662 |   1.85483 | 10.279292 | 0.086307  0.093625  0.107353  0.126195  0.148254 | 0.181752 |
    | 02/01 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.327797 |     0.0 |  2.239679 | 1.612569 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.165815 | 0.042185 | -0.042185 | 0.163898 |  0.135278 | 0.0 | 0.052216 |  0.187494 | 0.400961 | 0.400961 | 0.111378 |      0.0 | 0.0 | 0.0 | 114.507032 |   1.51347 | 10.390974 | 0.055449  0.059284  0.066608  0.077013  0.089913 | 0.111378 |
    | 02/01 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.320053 |  0.4752 |  2.854761 | 1.960038 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.403262 | 0.042746 | -0.042746 | 0.160027 |  0.078912 | 0.0 | 0.052755 |  0.131667 | 0.247817 | 0.247817 | 0.068838 |      0.0 | 0.0 | 0.0 | 113.146516 |  1.231784 | 10.498246 | 0.038765  0.040834  0.044777  0.050376  0.057363 | 0.068838 |
    | 02/01 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.300088 |  0.1188 |  7.210496 | 5.130246 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.629327 | 0.043427 | -0.043427 | 0.150044 |  0.050702 | 0.0 | 0.053241 |  0.103943 | 0.164367 | 0.164367 | 0.045658 |      0.0 | 0.0 | 0.0 | 109.560616 |  0.987612 | 10.595049 | 0.030434  0.031472  0.033468  0.036339  0.039979 | 0.045658 |
    | 02/01 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.305588 |  4.2768 |  3.914748 | 1.837796 | 1.837796 |    2.2768 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    2.2768 |  0.683241 | 0.0 | 0.638626 | 0.044423 |  0.638818 | 0.152794 |  0.063517 | 0.0 | 0.053739 |  0.117256 | 0.122575 | 0.122575 | 0.034049 | 0.162204 | 0.0 | 0.0 | 110.559972 |  1.410119 | 10.694103 | 0.033884  0.033518  0.033027  0.032779  0.033167 | 0.034049 |
    | 02/01 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.313947 |  7.0092 | 10.233004 | 3.655358 |      2.0 |  5.171404 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.171404 |  1.580317 | 0.0 | 2.132116 | 0.042924 |  1.537393 | 0.156973 |  0.280085 | 0.0 | 0.054255 |  0.334341 | 0.129995 | 0.129995 |  0.03611 |      0.0 | 0.0 | 0.0 | 112.061866 |  2.510453 | 10.796821 | 0.094833  0.087734  0.075748  0.062097  0.050307 |  0.03611 |
    | 02/01 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.303809 |  1.3068 |  4.693389 | 2.965278 |   1.3068 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.868067 | 0.043969 | -0.043969 | 0.151905 |  0.384322 | 0.0 | 0.054744 |  0.439066 | 0.248142 | 0.248142 | 0.068928 |      0.0 | 0.0 | 0.0 | 110.237768 |  1.930258 | 10.893982 | 0.126234  0.122344  0.114997  0.104883  0.093186 | 0.068928 |
    | 02/01 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.396544 | 24.5916 |  2.448478 |  0.15074 |  0.15074 |   22.5916 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   22.5916 |  6.863535 | 0.0 | 0.059355 | 0.037017 |  6.826517 | 0.198272 |  2.037496 | 0.0 | 0.055461 |  2.092957 | 0.494223 | 0.494223 | 0.137284 |  1.84926 | 0.0 | 0.0 | 125.943496 |  6.521007 | 11.036793 | 0.590756  0.536618  0.445022  0.340087  0.247896 | 0.137284 |
    | 02/01 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.585507 | 45.0252 |  2.597538 | 0.020724 | 0.020724 |  44.87446 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  44.87446 | 17.794702 | 0.0 | 0.009912 | 0.023488 | 17.771214 | 0.292753 |  9.477037 | 0.0 | 0.056648 |  9.533685 | 1.905552 | 1.905552 |  0.52932 | 1.979276 | 0.0 | 0.0 |  153.03683 | 14.522431 | 11.272899 | 2.691497  2.445433  2.025128  1.535109  1.091344 |  0.52932 |
    | 02/01 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.613419 |  9.7416 |  3.338304 | 0.907373 | 0.907373 |  9.720876 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.720876 |  5.691639 | 0.0 | 0.445428 | 0.021467 |  5.670172 | 0.306709 |  8.550769 | 0.0 | 0.057898 |  8.608667 | 6.145428 | 6.145428 | 1.707063 | 1.092627 | 0.0 | 0.0 | 156.642106 | 11.335124 |  11.52171 | 2.503085  2.516697  2.511762  2.443284  2.276922 | 1.707063 |
    | 02/01 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.620455 |  4.2768 |  1.846084 | 0.866652 | 0.866652 |  3.369427 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.369427 |  2.066869 | 0.0 | 0.427817 | 0.021028 |  2.045842 | 0.310227 |  4.140784 | 0.0 |  0.05916 |  4.199943 | 8.057309 | 8.057309 | 2.238142 | 1.133348 | 0.0 | 0.0 | 157.537874 |  8.929955 | 11.772778 | 1.261296  1.406129  1.650522  1.925669  2.150767 | 2.238142 |
    | 02/01 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.644491 |    8.91 |  0.338261 | 0.099914 | 0.099914 |  8.043348 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  8.043348 |  4.990532 | 0.0 | 0.049957 | 0.019705 |  4.970827 | 0.322246 |  3.415715 | 0.0 | 0.060475 |   3.47619 | 6.037831 | 6.037831 | 1.677175 | 1.900086 | 0.0 | 0.0 | 160.560437 | 10.162822 | 12.034548 | 1.016629  1.049956  1.122348  1.240736  1.403074 | 1.677175 |
    | 02/01 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.708542 |  21.978 |  0.107442 |  0.00859 |  0.00859 | 21.878086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 21.878086 | 14.100237 | 0.0 | 0.004295 | 0.015831 | 14.084406 | 0.354271 |  7.923725 | 0.0 | 0.061944 |  7.985669 |   4.4755 |   4.4755 | 1.243194 |  1.99141 | 0.0 | 0.0 | 168.349822 | 15.969232 | 12.326875 | 2.272056  2.127219  1.887704   1.62767  1.428264 | 1.243194 |
    | 02/01 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.754176 | 18.2952 |  0.107243 | 0.012392 | 0.012392 |  18.28661 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  18.28661 | 12.956823 | 0.0 | 0.006196 |  0.01316 | 12.943663 | 0.377088 | 11.304212 | 0.0 |  0.06352 | 11.367732 | 6.400454 | 6.400454 | 1.777904 | 1.987608 | 0.0 | 0.0 | 173.686572 | 17.231595 | 12.640443 |  3.26211  3.142122  2.924014  2.641353  2.340591 | 1.777904 |
    | 02/01 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.770161 |  7.4844 |  0.105848 | 0.036055 | 0.036055 |  7.472008 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.472008 |  5.635206 | 0.0 | 0.018028 | 0.012238 |  5.622968 |  0.38508 |  8.161988 | 0.0 | 0.065128 |  8.227115 | 9.333586 | 9.333586 | 2.592663 | 1.963945 | 0.0 | 0.0 | 175.517585 | 14.307494 | 12.960395 | 2.415396  2.509794  2.654673  2.785757    2.8381 | 2.592663 |
    | 02/01 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.77449 |  2.2572 |  0.104032 | 0.059768 | 0.059768 |  2.221145 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.221145 |  1.710638 | 0.0 | 0.029884 | 0.011986 |  1.698652 | 0.387245 |  4.333944 | 0.0 | 0.066738 |  4.400682 | 8.974887 | 8.974887 | 2.493024 | 1.940232 | 0.0 | 0.0 | 176.010193 | 11.284958 | 13.280902 | 1.314593  1.445757  1.676683  1.959393  2.233088 | 2.493024 |
    | 03/01 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.785881 |  5.8212 |  0.104235 | 0.041931 | 0.041931 |  5.761432 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.761432 |  4.462169 | 0.0 | 0.020966 | 0.011345 |  4.450824 |  0.39294 |  3.332947 | 0.0 | 0.068369 |  3.401316 | 6.190282 | 6.190282 | 1.719523 | 1.958069 | 0.0 | 0.0 | 177.299835 | 12.009895 | 13.605473 | 0.997443  1.038737  1.123325  1.253947  1.427095 | 1.719523 |
    | 03/01 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.791757 |  3.2076 |  0.104032 |  0.05435 |  0.05435 |  3.165669 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.165669 |  2.487838 | 0.0 | 0.027175 | 0.011011 |  2.476827 | 0.395879 |   3.05638 | 0.0 | 0.070007 |  3.126387 | 4.153619 | 4.153619 | 1.153783 |  1.94565 | 0.0 | 0.0 | 177.961502 | 11.034464 | 13.931345 | 0.909712  0.921304  0.945964  0.987137  1.049197 | 1.153783 |
    | 03/01 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.792544 |   0.594 |  0.103014 | 0.069893 | 0.069893 |   0.53965 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.53965 |  0.427272 | 0.0 | 0.034946 | 0.010963 |  0.416309 | 0.396272 |  2.001337 | 0.0 | 0.071638 |  2.072975 | 3.291218 | 3.291218 | 0.914227 | 1.930107 | 0.0 | 0.0 | 178.049897 |  9.053164 | 14.255979 | 0.611562  0.646617  0.707101  0.779582   0.85021 | 0.914227 |
    | 03/01 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.792628 |  0.2376 |  0.103422 | 0.072716 | 0.072716 |  0.167707 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.167707 |  0.132916 | 0.0 | 0.036358 | 0.010958 |  0.121958 | 0.396314 |  1.154152 | 0.0 | 0.073261 |  1.227413 | 2.499315 | 2.499315 | 0.694254 | 1.927284 | 0.0 | 0.0 | 178.059289 |  7.624656 | 14.579032 | 0.364565  0.394463   0.44878  0.519432  0.595931 | 0.694254 |
    | 03/01 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.793371 |   0.594 |   0.10464 | 0.070996 | 0.070996 |  0.521284 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.521284 |  0.413185 | 0.0 | 0.035498 | 0.010916 |  0.402268 | 0.396686 |  0.759428 | 0.0 | 0.074879 |  0.834307 | 1.652215 | 1.652215 | 0.458949 | 1.929004 | 0.0 | 0.0 | 178.142807 |  6.870811 | 14.900839 | 0.246081  0.260864  0.289242   0.32966  0.379413 | 0.458949 |
    | 03/01 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.798335 |  2.8512 |  0.106248 | 0.057521 | 0.057521 |  2.780204 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.780204 |  2.205735 | 0.0 | 0.028761 | 0.010641 |  2.195093 | 0.399167 |   0.80239 | 0.0 |   0.0765 |   0.87889 | 1.086287 | 1.086287 | 0.301746 | 1.942479 | 0.0 | 0.0 | 178.699157 |  7.864347 | 15.223506 | 0.254582  0.254092  0.255045  0.260506   0.27364 | 0.301746 |
    | 03/01 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.798631 |  0.4752 |  0.179639 | 0.123337 | 0.123337 |  0.417679 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.417679 |  0.333448 | 0.0 | 0.061669 | 0.010608 |  0.322839 | 0.399316 |  0.800248 | 0.0 | 0.078114 |  0.878362 | 0.901466 | 0.901466 | 0.250407 | 1.876663 | 0.0 | 0.0 | 178.732328 |  6.987622 | 15.544708 |  0.25472  0.254688  0.254638  0.254805   0.25591 | 0.250407 |
    | 03/01 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.797218 |  0.2376 |  0.545628 | 0.383628 | 0.383628 |  0.114263 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.114263 |  0.091254 | 0.0 | 0.191814 | 0.010622 |  0.080631 | 0.398609 |  0.553214 | 0.0 | 0.079717 |   0.63293 | 0.861981 | 0.861981 | 0.239439 | 1.616372 | 0.0 | 0.0 | 178.574145 |  6.116431 |   15.8636 | 0.185948  0.193943  0.207397  0.222645  0.235777 | 0.239439 |
    | 03/01 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.792943 |     0.0 |  1.361522 | 0.980296 | 0.980296 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.490148 | 0.010713 | -0.010713 | 0.396472 |  0.375567 | 0.0 |   0.0813 |  0.456868 | 0.725391 | 0.725391 | 0.201498 | 0.636076 | 0.0 | 0.0 |  178.09471 |  5.333679 | 16.178771 | 0.134293  0.140568  0.152029  0.167024  0.183272 | 0.201498 |
    | 03/01 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.764872 |     0.0 |  4.874663 | 3.509758 | 0.636076 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.19172 | 0.010953 | -0.010953 | 0.382436 |  0.278561 | 0.0 | 0.082806 |  0.361367 | 0.545717 | 0.545717 | 0.151588 |      0.0 | 0.0 | 0.0 | 174.913943 |   4.66173 | 16.478401 | 0.105791  0.109316  0.115985  0.125287  0.136458 | 0.151588 |
    | 03/01 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.736086 |  0.3564 |  5.057269 | 3.513746 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.335546 | 0.012543 | -0.012543 | 0.368043 |  0.208651 | 0.0 | 0.084232 |  0.292883 | 0.414311 | 0.414311 | 0.115086 |      0.0 | 0.0 | 0.0 | 171.590941 |  4.072493 | 16.762212 | 0.085639  0.088101  0.092665  0.098865  0.106139 | 0.115086 |
    | 03/01 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.719431 |  3.0888 |  6.154673 | 3.253813 |      2.0 |    1.0888 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    1.0888 |  0.801451 | 0.0 | 2.253813 | 0.014061 |   0.78739 | 0.359715 |  0.198136 | 0.0 |  0.08561 |  0.283746 | 0.331366 | 0.331366 | 0.092046 |      0.0 | 0.0 | 0.0 | 169.638537 |  4.302031 | 17.036318 | 0.082399  0.082859  0.083923  0.085829  0.088779 | 0.092046 |
    | 03/01 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.688383 |  0.8316 |  6.236191 | 4.131769 |   0.8316 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.715969 | 0.015181 | -0.015181 | 0.344192 |  0.199364 | 0.0 | 0.086903 |  0.286266 | 0.294601 | 0.294601 | 0.081834 |      0.0 | 0.0 | 0.0 | 165.937749 |  3.743294 | 17.293607 | 0.082996  0.082943  0.082913  0.083051  0.083551 | 0.081834 |
    | 03/01 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.659166 |  0.3564 |  5.404341 | 3.754888 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.576688 | 0.017031 | -0.017031 | 0.329583 |  0.149065 | 0.0 | 0.088116 |  0.237181 | 0.283199 | 0.283199 | 0.078666 |      0.0 | 0.0 | 0.0 | 162.378092 |  3.247615 | 17.535074 | 0.069262  0.070857  0.073538  0.076575  0.079205 | 0.078666 |
    | 03/01 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.610719 |  0.3564 |  9.035929 | 6.278083 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.099883 | 0.018811 | -0.018811 | 0.305359 |  0.121494 | 0.0 | 0.089202 |  0.210696 |  0.25624 |  0.25624 | 0.071178 |      0.0 | 0.0 | 0.0 |  156.29702 |  2.801951 | 17.751232 | 0.061377  0.062347   0.06416  0.066614  0.069394 | 0.071178 |
    | 03/01 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.58034 |     0.0 |  5.627818 | 4.052029 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.958804 | 0.021851 | -0.021851 |  0.29017 |  0.090079 | 0.0 | 0.090207 |  0.180286 | 0.223913 | 0.223913 | 0.062198 |      0.0 | 0.0 | 0.0 | 152.360068 |   2.39985 | 17.951194 | 0.052589  0.053644  0.055534  0.057953  0.060546 | 0.062198 |
    | 03/01 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.548803 |     0.0 |  6.156027 |  4.43234 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.221357 |  0.02382 |  -0.02382 | 0.274402 |  0.066087 | 0.0 | 0.091128 |  0.157215 | 0.194321 | 0.194321 | 0.053978 |      0.0 | 0.0 | 0.0 | 148.162531 |  2.035542 | 18.134468 | 0.045828   0.04665  0.048155  0.050148  0.052378 | 0.053978 |
    | 03/01 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.530323 |     0.0 |  3.811802 | 2.744497 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.541892 | 0.025919 | -0.025919 | 0.265161 |  0.043836 | 0.0 | 0.091998 |  0.135834 | 0.168379 | 0.168379 | 0.046772 |      0.0 | 0.0 | 0.0 | 145.646557 |  1.700626 | 18.307631 | 0.039609  0.040359   0.04172  0.043491  0.045435 | 0.046772 |
    | 03/01 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.522211 |     0.0 |  1.747211 | 1.257992 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.145352 | 0.027177 | -0.027177 | 0.261106 |  0.025957 | 0.0 | 0.092844 |  0.118801 | 0.145999 | 0.145999 | 0.040555 |      0.0 | 0.0 | 0.0 | 144.528382 |  1.386387 | 18.475893 | 0.034626  0.035231  0.036336  0.037796  0.039427 | 0.040555 |
    | 03/01 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.520914 |     0.0 |  0.318724 | 0.229481 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.207331 | 0.027736 | -0.027736 | 0.260457 |  0.013472 | 0.0 | 0.093682 |  0.107154 | 0.127281 | 0.127281 | 0.035356 |      0.0 | 0.0 | 0.0 | 144.348787 |  1.084722 | 18.642669 | 0.031194  0.031614   0.03239  0.033439   0.03465 | 0.035356 |
    | 03/01 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.520639 |     0.0 |  0.101575 | 0.073134 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.065993 | 0.027826 | -0.027826 | 0.260319 |   0.00595 | 0.0 | 0.094515 |  0.100465 |   0.1133 |   0.1133 | 0.031472 |      0.0 | 0.0 | 0.0 |  144.31062 |  0.790627 | 18.808473 | 0.029204   0.02945  0.029913  0.030556   0.03133 | 0.031472 |
    | 03/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.520374 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.064624 | 0.027845 | -0.027845 | 0.260187 |  0.001991 | 0.0 | 0.095343 |  0.097335 | 0.104254 | 0.104254 | 0.028959 |      0.0 | 0.0 | 0.0 |  144.27384 |  0.500604 | 18.973317 |  0.02826  0.028378  0.028606  0.028936  0.029354 | 0.028959 |
    | 03/01 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.520116 |     0.0 |  0.097808 | 0.070422 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.063512 | 0.027863 | -0.027863 | 0.260058 |   0.00037 | 0.0 | 0.096167 |  0.096537 |   0.0993 |   0.0993 | 0.027583 |      0.0 | 0.0 | 0.0 | 144.238191 |  0.212312 | 19.137208 | 0.028005  0.028038  0.028109  0.028225  0.028393 | 0.027583 |
    | 03/01 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.519866 |     0.0 |  0.096529 | 0.069501 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.062667 | 0.027881 | -0.027881 |  0.18442 |  0.000012 | 0.0 | 0.096608 |   0.09662 |  0.09722 |  0.09722 | 0.027005 |      0.0 | 0.0 | 0.0 | 144.203406 |       0.0 |  19.22502 | 0.028019  0.028019  0.028022  0.028037  0.028074 | 0.027005 |
    | 04/01 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.519419 |     0.0 |  0.095455 | 0.068728 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.061942 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.096125 |  0.096125 | 0.096652 | 0.096652 | 0.026848 |      0.0 | 0.0 | 0.0 | 144.141463 |       0.0 | 19.128895 | 0.027881  0.027897  0.027924  0.027956  0.027985 | 0.026848 |
    | 04/01 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.518961 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.063579 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.095644 |  0.095644 | 0.096307 | 0.096307 | 0.026752 |      0.0 | 0.0 | 0.0 | 144.077884 |       0.0 |  19.03325 | 0.027742  0.027758  0.027788  0.027827  0.027866 | 0.026752 |
    | 04/01 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.518501 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.063825 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.095166 |  0.095166 | 0.095857 | 0.095857 | 0.026627 |      0.0 | 0.0 | 0.0 | 144.014059 |       0.0 | 18.938084 | 0.027603   0.02762   0.02765  0.027688   0.02773 | 0.026627 |
    | 04/01 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.518307 |  1.5444 |  0.096957 | 0.059819 | 0.059819 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.026921 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.09469 |   0.09469 | 0.095382 | 0.095382 | 0.026495 | 1.484581 | 0.0 | 0.0 | 143.987137 |       0.0 | 18.843393 | 0.027465  0.027482  0.027512   0.02755  0.027591 | 0.026495 |
    | 04/01 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.518082 |     0.0 |  0.096744 | 0.069656 | 0.069656 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.031342 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.094217 |  0.094217 | 0.094905 | 0.094905 | 0.026362 | 1.414925 | 0.0 | 0.0 | 143.955795 |       0.0 | 18.749176 | 0.027328  0.027344  0.027374  0.027412  0.027453 | 0.026362 |
    | 04/01 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.517852 |     0.0 |  0.098442 | 0.070878 | 0.070878 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.031885 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.093746 |  0.093746 |  0.09443 |  0.09443 | 0.026231 | 1.344047 | 0.0 | 0.0 |  143.92391 |       0.0 | 18.655431 | 0.027191  0.027208  0.027237  0.027275  0.027316 | 0.026231 |
    | 04/01 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.517014 |     0.0 |  0.359866 | 0.259103 | 0.259103 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.116535 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.093277 |  0.093277 | 0.093958 | 0.093958 |   0.0261 | 1.084943 | 0.0 | 0.0 | 143.807375 |       0.0 | 18.562153 | 0.027055  0.027071  0.027101  0.027139  0.027179 |   0.0261 |
    | 04/01 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.513563 |  0.8316 |  1.614671 | 1.069795 | 1.069795 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.480764 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.092811 |  0.092811 | 0.093488 | 0.093488 | 0.025969 | 0.846748 | 0.0 | 0.0 | 143.326611 |       0.0 | 18.469343 |  0.02692  0.026936  0.026965  0.027003  0.027044 | 0.025969 |
    | 04/01 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.501086 |  0.4752 |  3.707216 | 2.545322 | 1.321948 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.751726 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.092347 |  0.092347 | 0.093021 | 0.093021 | 0.025839 |      0.0 | 0.0 | 0.0 | 141.574885 |       0.0 | 18.376996 | 0.026785  0.026801  0.026831  0.026868  0.026908 | 0.025839 |
    | 04/01 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.474607 |  0.1188 |  6.022315 | 4.284859 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.791428 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.091885 |  0.091885 | 0.092556 | 0.092556 |  0.02571 |      0.0 | 0.0 | 0.0 | 137.783457 |       0.0 | 18.285111 | 0.026651  0.026667  0.026696  0.026734  0.026774 |  0.02571 |
    | 04/01 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.441402 |  0.4752 |  8.299842 | 5.698554 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  4.90729 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.091426 |  0.091426 | 0.092093 | 0.092093 | 0.025581 |      0.0 | 0.0 | 0.0 | 132.876167 |       0.0 | 18.193685 | 0.026518  0.026534  0.026563    0.0266   0.02664 | 0.025581 |
    | 04/01 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.403579 |     0.0 |  9.734061 | 7.008524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.820411 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.090968 |  0.090968 | 0.091633 | 0.091633 | 0.025454 |      0.0 | 0.0 | 0.0 | 127.055756 |       0.0 | 18.102717 | 0.026385  0.026401   0.02643  0.026467  0.026507 | 0.025454 |
    | 04/01 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.368998 |     0.0 |  9.733892 | 7.008402 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.565361 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.090514 |  0.090514 | 0.091175 | 0.091175 | 0.025326 |      0.0 | 0.0 | 0.0 | 121.490395 |       0.0 | 18.012203 | 0.026254  0.026269  0.026298  0.026335  0.026374 | 0.025326 |
    | 04/01 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.328874 |     0.0 | 12.429676 | 8.949367 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.795388 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.090061 |  0.090061 | 0.090719 | 0.090719 |   0.0252 |      0.0 | 0.0 | 0.0 | 114.695007 |       0.0 | 17.922142 | 0.026122  0.026138  0.026167  0.026203  0.026242 |   0.0252 |
    | 04/01 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.297445 |     0.0 | 10.884737 | 7.837011 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.617912 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.089611 |  0.089611 | 0.090265 | 0.090265 | 0.025074 |      0.0 | 0.0 | 0.0 | 109.077094 |       0.0 | 17.832532 | 0.025992  0.026007  0.026036  0.026072  0.026111 | 0.025074 |
    | 04/01 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.273532 |     0.0 |  9.119821 | 6.566271 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.476436 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.089163 |  0.089163 | 0.089814 | 0.089814 | 0.024948 |      0.0 | 0.0 | 0.0 | 104.600658 |       0.0 | 17.743369 | 0.025862  0.025877  0.025905  0.025942  0.025981 | 0.024948 |
    | 04/01 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.255842 |     0.0 |  7.306062 | 5.260364 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.438985 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.088717 |  0.088717 | 0.089365 | 0.089365 | 0.024824 |      0.0 | 0.0 | 0.0 | 101.161673 |       0.0 | 17.654652 | 0.025732  0.025748  0.025776  0.025812  0.025851 | 0.024824 |
    | 04/01 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.248028 |     0.0 |  3.420163 | 2.462517 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.556952 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.088273 |  0.088273 | 0.088918 | 0.088918 | 0.024699 |      0.0 | 0.0 | 0.0 |  99.604721 |       0.0 | 17.566379 | 0.025604  0.025619  0.025647  0.025683  0.025721 | 0.024699 |
    | 04/01 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.244649 |     0.0 |  1.518709 | 1.093471 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.680718 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.087832 |  0.087832 | 0.088473 | 0.088473 | 0.024576 |      0.0 | 0.0 | 0.0 |  98.924003 |       0.0 | 17.478547 | 0.025476  0.025491  0.025519  0.025555  0.025593 | 0.024576 |
    | 04/01 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.243808 |     0.0 |  0.382278 |  0.27524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.170174 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.087393 |  0.087393 | 0.088031 | 0.088031 | 0.024453 |      0.0 | 0.0 | 0.0 |  98.753829 |       0.0 | 17.391154 | 0.025348  0.025364  0.025391  0.025427  0.025465 | 0.024453 |
    | 04/01 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.24358 |     0.0 |  0.103829 | 0.074757 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046141 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.086956 |  0.086956 | 0.087591 | 0.087591 | 0.024331 |      0.0 | 0.0 | 0.0 |  98.707688 |       0.0 | 17.304198 | 0.025222  0.025237  0.025264    0.0253  0.025338 | 0.024331 |
    | 04/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.243362 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.044194 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.086521 |  0.086521 | 0.087153 | 0.087153 | 0.024209 |      0.0 | 0.0 | 0.0 |  98.663494 |       0.0 | 17.217677 | 0.025095  0.025111  0.025138  0.025173  0.025211 | 0.024209 |
    | 04/01 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.243147 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043519 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.086088 |  0.086088 | 0.086717 | 0.086717 | 0.024088 |      0.0 | 0.0 | 0.0 |  98.619975 |       0.0 | 17.131589 |  0.02497  0.024985  0.025012  0.025047  0.025085 | 0.024088 |
    | 04/01 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.242932 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043688 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.085658 |  0.085658 | 0.086283 | 0.086283 | 0.023968 |      0.0 | 0.0 | 0.0 |  98.576287 |       0.0 | 17.045931 | 0.024845   0.02486  0.024887  0.024922  0.024959 | 0.023968 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v2_low_accuracy:

low accuracy
____________

The :ref:`hland_v1_low_accuracy` example of |hland_v1| illustrates how decreasing
|RecStep| for gaining computational speed might result in too low numerical accuracies.
For |hland_v2|, such inaccuracies can be even more severe, as |RecStep| controls the
accuracy of the numerical solution both of the upper zone layer and the linear storage
cascade:

.. integration-test::

    >>> recstep(1)
    >>> test("hland_v2_low_accuracy")
    |        date |    p |    t |   tn |       epn | tmean |   tc | fracrain | rfc | sfc |    cfact | swe | gact | contriarea |      pc |        ep |      epc |       ei |        tf | spl | wcl | spg | wcg | glmelt | melt | refr |       in_ |         r |  sr |       ea |       cf |      inuz |     perc |        q0 |  el |       q1 |      inuh |     outuh |        rt |       qt |       ic |  sp |  wc |         sm |       uz |        lz |                      sc |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.249737 |     0.0 |   0.11682 |  0.08411 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.052569 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |     0.05 |      0.05 |       0.3 |       0.3 | 0.083333 |      0.0 | 0.0 | 0.0 |  99.947431 |      0.0 |      9.95 | 0.0  0.0  0.0  0.0  0.0 | 0.083333 |
    | 01/01 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.249482 |     0.0 |  0.113449 | 0.081683 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.051025 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.04975 |   0.04975 |   0.04975 |   0.04975 | 0.013819 |      0.0 | 0.0 | 0.0 |  99.896406 |      0.0 |   9.90025 | 0.0  0.0  0.0  0.0  0.0 | 0.013819 |
    | 01/01 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.24923 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.050572 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049501 |  0.049501 |  0.049501 |  0.049501 |  0.01375 |      0.0 | 0.0 | 0.0 |  99.845834 |      0.0 |  9.850749 | 0.0  0.0  0.0  0.0  0.0 |  0.01375 |
    | 01/01 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.24898 |     0.0 |  0.111348 | 0.080171 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.05003 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049254 |  0.049254 |  0.049254 |  0.049254 | 0.013682 |      0.0 | 0.0 | 0.0 |  99.795804 |      0.0 |  9.801495 | 0.0  0.0  0.0  0.0  0.0 | 0.013682 |
    | 01/01 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.248728 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.050521 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.049007 |  0.049007 |  0.049007 |  0.049007 | 0.013613 |      0.0 | 0.0 | 0.0 |  99.745284 |      0.0 |  9.752488 | 0.0  0.0  0.0  0.0  0.0 | 0.013613 |
    | 01/01 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.248461 |     0.0 |  0.119203 | 0.085826 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.053505 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048762 |  0.048762 |  0.048762 |  0.048762 | 0.013545 |      0.0 | 0.0 | 0.0 |  99.691779 |      0.0 |  9.703725 | 0.0  0.0  0.0  0.0  0.0 | 0.013545 |
    | 01/01 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.247705 |     0.0 |  0.338613 | 0.243802 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.151906 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048519 |  0.048519 |  0.048519 |  0.048519 | 0.013477 |      0.0 | 0.0 | 0.0 |  99.539873 |      0.0 |  9.655206 | 0.0  0.0  0.0  0.0  0.0 | 0.013477 |
    | 01/01 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.242731 |     0.0 |  2.242135 | 1.614337 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.004318 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048276 |  0.048276 |  0.048276 |  0.048276 |  0.01341 |      0.0 | 0.0 | 0.0 |  98.535555 |      0.0 |   9.60693 | 0.0  0.0  0.0  0.0  0.0 |  0.01341 |
    | 01/01 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.231785 |     0.0 |  5.068662 | 3.649436 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.247495 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.048035 |  0.048035 |  0.048035 |  0.048035 | 0.013343 |      0.0 | 0.0 | 0.0 |  96.288059 |      0.0 |  9.558896 | 0.0  0.0  0.0  0.0  0.0 | 0.013343 |
    | 01/01 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.214416 |     0.0 |  8.488365 | 6.111623 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.677977 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047794 |  0.047794 |  0.047794 |  0.047794 | 0.013276 |      0.0 | 0.0 | 0.0 |  92.610082 |      0.0 |  9.511101 | 0.0  0.0  0.0  0.0  0.0 | 0.013276 |
    | 01/01 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.196315 |     0.0 |   9.58666 | 6.902395 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.995196 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047556 |  0.047556 |  0.047556 |  0.047556 |  0.01321 |      0.0 | 0.0 | 0.0 |  88.614886 |      0.0 |  9.463546 | 0.0  0.0  0.0  0.0  0.0 |  0.01321 |
    | 01/01 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.177581 |     0.0 | 10.869046 | 7.825713 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.334217 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047318 |  0.047318 |  0.047318 |  0.047318 | 0.013144 |      0.0 | 0.0 | 0.0 |   84.28067 |      0.0 |  9.416228 | 0.0  0.0  0.0  0.0  0.0 | 0.013144 |
    | 01/01 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.168204 |     0.0 |  5.946366 | 4.281384 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.255237 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.047081 |  0.047081 |  0.047081 |  0.047081 | 0.013078 |      0.0 | 0.0 | 0.0 |  82.025433 |      0.0 |  9.369147 | 0.0  0.0  0.0  0.0  0.0 | 0.013078 |
    | 01/01 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.156772 |     0.0 |  7.685035 | 5.533225 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.836657 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046846 |  0.046846 |  0.046846 |  0.046846 | 0.013013 |      0.0 | 0.0 | 0.0 |  79.188775 |      0.0 |  9.322301 | 0.0  0.0  0.0  0.0  0.0 | 0.013013 |
    | 01/01 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.144898 |     0.0 |  8.581238 | 6.178491 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.05792 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046612 |  0.046612 |  0.046612 |  0.046612 | 0.012948 |      0.0 | 0.0 | 0.0 |  76.130856 |      0.0 |   9.27569 | 0.0  0.0  0.0  0.0  0.0 | 0.012948 |
    | 01/01 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.141386 |  0.2376 |  2.774152 | 1.950491 |   0.2376 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.928078 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046378 |  0.046378 |  0.046378 |  0.046378 | 0.012883 |      0.0 | 0.0 | 0.0 |  75.202777 |      0.0 |  9.229311 | 0.0  0.0  0.0  0.0  0.0 | 0.012883 |
    | 01/01 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.138698 |     0.0 |  2.122607 | 1.528277 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.718317 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.046147 |  0.046147 |  0.046147 |  0.046147 | 0.012818 |      0.0 | 0.0 | 0.0 |  74.484461 |      0.0 |  9.183165 | 0.0  0.0  0.0  0.0  0.0 | 0.012818 |
    | 01/01 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.137057 |     0.0 |   1.31842 | 0.949262 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.441908 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045916 |  0.045916 |  0.045916 |  0.045916 | 0.012754 |      0.0 | 0.0 | 0.0 |  74.042552 |      0.0 |  9.137249 | 0.0  0.0  0.0  0.0  0.0 | 0.012754 |
    | 01/01 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.136597 |  1.5444 |  0.871144 | 0.537465 | 0.537465 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.12436 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.045686 |  0.045686 |  0.045686 |  0.045686 | 0.012691 | 1.006935 | 0.0 | 0.0 |  73.918192 |      0.0 |  9.091563 | 0.0  0.0  0.0  0.0  0.0 | 0.012691 |
    | 01/01 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |     0.1554 |  6.6528 |  0.259718 | 0.096141 | 0.096141 |  5.659735 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.659735 |  0.773106 | 0.0 | 0.023694 | 0.060598 |  0.712508 |   0.0777 |  0.068166 | 0.0 | 0.045846 |  0.114013 |  0.114013 |  0.114013 |  0.03167 | 1.903859 | 0.0 | 0.0 |  78.841725 | 0.566641 |  9.123417 | 0.0  0.0  0.0  0.0  0.0 |  0.03167 |
    | 01/01 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.166931 |  3.4452 |  0.115333 | 0.058839 | 0.058839 |  3.349059 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.349059 |  0.520445 | 0.0 | 0.015028 | 0.059165 |   0.46128 | 0.083466 |  0.181105 | 0.0 | 0.046034 |   0.22714 |   0.22714 |   0.22714 | 0.063094 | 1.941161 | 0.0 | 0.0 |  81.714476 | 0.763351 |  9.160848 | 0.0  0.0  0.0  0.0  0.0 | 0.063094 |
    | 01/01 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.187313 |  5.8212 |  0.114207 | 0.045942 | 0.045942 |  5.762361 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.762361 |  0.961919 | 0.0 | 0.012429 | 0.056743 |  0.905176 | 0.093656 |  0.594339 | 0.0 | 0.046273 |  0.640612 |  0.640612 |  0.640612 | 0.177948 | 1.954058 | 0.0 | 0.0 |  86.559232 | 0.980532 |  9.208232 | 0.0  0.0  0.0  0.0  0.0 | 0.177948 |
    | 01/01 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.234258 | 12.5928 |  0.113828 | 0.023264 | 0.023264 | 12.546858 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.546858 |  2.350184 | 0.0 | 0.007038 | 0.051622 |  2.298561 | 0.117129 |  2.459152 | 0.0 | 0.046627 |  2.505778 |  2.505778 |  2.505778 |  0.69605 | 1.976736 | 0.0 | 0.0 |  96.800491 | 0.702812 |  9.278734 | 0.0  0.0  0.0  0.0  0.0 |  0.69605 |
    | 01/01 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |   0.234744 |  0.1188 |   0.11307 | 0.080449 | 0.080449 |  0.095536 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.095536 |   0.02238 | 0.0 | 0.024367 | 0.051563 | -0.029183 | 0.117372 |  0.013306 | 0.0 | 0.046981 |  0.060286 |  0.060286 |  0.060286 | 0.016746 | 1.919551 | 0.0 | 0.0 |  96.900842 | 0.542951 |  9.349126 | 0.0  0.0  0.0  0.0  0.0 | 0.016746 |
    | 02/01 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.237676 |  0.8316 |   0.11307 | 0.074914 | 0.074914 |  0.751151 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.751151 |  0.176328 | 0.0 | 0.022832 | 0.051262 |  0.125066 | 0.118838 |  0.012336 | 0.0 |  0.04734 |  0.059676 |  0.059676 |  0.059676 | 0.016577 | 1.925086 | 0.0 | 0.0 |  97.504096 | 0.536843 |  9.420624 | 0.0  0.0  0.0  0.0  0.0 | 0.016577 |
    | 02/01 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.250981 |   3.564 |   0.11307 | 0.057003 | 0.057003 |  3.489086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.489086 |  0.829273 | 0.0 | 0.017851 | 0.049918 |  0.779355 |  0.12549 |  0.106781 | 0.0 | 0.047731 |  0.154511 |  0.154511 |  0.154511 |  0.04292 | 1.942997 | 0.0 | 0.0 | 100.195975 | 1.083926 |  9.498384 | 0.0  0.0  0.0  0.0  0.0 |  0.04292 |
    | 02/01 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.260359 |  2.4948 |  0.112498 | 0.063115 | 0.063115 |  2.437797 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.437797 |   0.61184 | 0.0 | 0.020132 | 0.048989 |  0.562851 |  0.13018 |  0.197648 | 0.0 | 0.048143 |  0.245791 |  0.245791 |  0.245791 | 0.068275 | 1.936885 | 0.0 | 0.0 | 102.050789 |  1.31895 |   9.58042 | 0.0  0.0  0.0  0.0  0.0 | 0.068275 |
    | 02/01 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.309018 | 12.3552 |  0.112116 | 0.023465 | 0.023465 | 12.292085 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.292085 |  3.200356 | 0.0 | 0.008153 | 0.044429 |  3.155927 | 0.154509 |  2.732817 | 0.0 | 0.048675 |  2.781491 |  2.781491 |  2.781491 | 0.772636 | 1.976535 | 0.0 | 0.0 | 111.178794 | 1.587552 |  9.686255 | 0.0  0.0  0.0  0.0  0.0 | 0.772636 |
    | 02/01 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.325241 |   4.158 |  0.111732 |  0.05308 |  0.05308 |  4.134535 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.134535 |  1.277646 | 0.0 | 0.018923 | 0.042982 |  1.234664 |  0.16262 |  0.546805 | 0.0 | 0.049244 |  0.596049 |  0.596049 |  0.596049 | 0.165569 |  1.94692 | 0.0 | 0.0 | 114.059742 |  2.11279 |  9.799631 | 0.0  0.0  0.0  0.0  0.0 | 0.165569 |
    | 02/01 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.34089 |  4.0392 |  0.111348 |  0.05353 |  0.05353 |   3.98612 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.98612 |  1.296448 | 0.0 | 0.019537 | 0.041625 |  1.254823 | 0.170445 |  0.825003 | 0.0 |  0.04985 |  0.874853 |  0.874853 |  0.874853 | 0.243015 |  1.94647 | 0.0 | 0.0 | 116.771503 | 2.372166 |  9.920225 | 0.0  0.0  0.0  0.0  0.0 | 0.243015 |
    | 02/01 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.346139 |  1.4256 |  0.217266 | 0.135647 | 0.135647 |   1.37207 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   1.37207 |  0.467724 | 0.0 |   0.0499 | 0.041162 |  0.426562 | 0.173069 |   0.43648 | 0.0 | 0.050466 |  0.486946 |  0.486946 |  0.486946 | 0.135263 | 1.864353 | 0.0 | 0.0 |  117.66711 | 2.189179 | 10.042828 | 0.0  0.0  0.0  0.0  0.0 | 0.135263 |
    | 02/01 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.344125 |  0.1188 |   1.46695 |  1.04373 |  1.04373 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.383924 | 0.041166 | -0.041166 | 0.172063 |  0.189312 | 0.0 | 0.051074 |  0.240387 |  0.240387 |  0.240387 | 0.066774 | 0.939422 | 0.0 | 0.0 | 117.324353 | 1.786638 | 10.163816 | 0.0  0.0  0.0  0.0  0.0 | 0.066774 |
    | 02/01 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.334261 |     0.0 |  3.532824 | 2.543633 | 0.939422 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.735028 | 0.041338 | -0.041338 | 0.167131 |  0.105245 | 0.0 | 0.051655 |    0.1569 |    0.1569 |    0.1569 | 0.043583 |      0.0 | 0.0 | 0.0 | 115.630662 | 1.472924 | 10.279292 | 0.0  0.0  0.0  0.0  0.0 | 0.043583 |
    | 02/01 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.327797 |     0.0 |  2.239679 | 1.612569 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.165815 | 0.042185 | -0.042185 | 0.163898 |  0.057724 | 0.0 | 0.052216 |   0.10994 |   0.10994 |   0.10994 | 0.030539 |      0.0 | 0.0 | 0.0 | 114.507032 | 1.209117 | 10.390974 | 0.0  0.0  0.0  0.0  0.0 | 0.030539 |
    | 02/01 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.320053 |  0.4752 |  2.854761 | 1.960038 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.403262 | 0.042746 | -0.042746 | 0.160027 |  0.031087 | 0.0 | 0.052755 |  0.083842 |  0.083842 |  0.083842 | 0.023289 |      0.0 | 0.0 | 0.0 | 113.146516 | 0.975258 | 10.498246 | 0.0  0.0  0.0  0.0  0.0 | 0.023289 |
    | 02/01 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.300088 |  0.1188 |  7.210496 | 5.130246 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.629327 | 0.043427 | -0.043427 | 0.150044 |  0.017681 | 0.0 | 0.053241 |  0.070923 |  0.070923 |  0.070923 | 0.019701 |      0.0 | 0.0 | 0.0 | 109.560616 | 0.764105 | 10.595049 | 0.0  0.0  0.0  0.0  0.0 | 0.019701 |
    | 02/01 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.305588 |  4.2768 |  3.914748 | 1.837796 | 1.837796 |    2.2768 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    2.2768 |  0.683241 | 0.0 | 0.638626 | 0.044423 |  0.638818 | 0.152794 |  0.068463 | 0.0 | 0.053739 |  0.122202 |  0.122202 |  0.122202 | 0.033945 | 0.162204 | 0.0 | 0.0 | 110.559972 | 1.181666 | 10.694103 | 0.0  0.0  0.0  0.0  0.0 | 0.033945 |
    | 02/01 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.313947 |  7.0092 | 10.233004 | 3.655358 |      2.0 |  5.171404 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.171404 |  1.580317 | 0.0 | 2.132116 | 0.042924 |  1.537393 | 0.156973 |  0.543518 | 0.0 | 0.054255 |  0.597773 |  0.597773 |  0.597773 | 0.166048 |      0.0 | 0.0 | 0.0 | 112.061866 | 2.018568 | 10.796821 | 0.0  0.0  0.0  0.0  0.0 | 0.166048 |
    | 02/01 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.303809 |  1.3068 |  4.693389 | 2.965278 |   1.3068 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.868067 | 0.043969 | -0.043969 | 0.151905 |  0.215943 | 0.0 | 0.054744 |  0.270687 |  0.270687 |  0.270687 | 0.075191 |      0.0 | 0.0 | 0.0 | 110.237768 | 1.606751 | 10.893982 | 0.0  0.0  0.0  0.0  0.0 | 0.075191 |
    | 02/01 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.396544 | 24.5916 |  2.448478 |  0.15074 |  0.15074 |   22.5916 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   22.5916 |  6.863535 | 0.0 | 0.059355 | 0.037017 |  6.826517 | 0.198272 |  8.234997 | 0.0 | 0.055461 |  8.290458 |  8.290458 |  8.290458 | 2.302905 |  1.84926 | 0.0 | 0.0 | 125.943496 |      0.0 | 11.036793 | 0.0  0.0  0.0  0.0  0.0 | 2.302905 |
    | 02/01 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.585507 | 45.0252 |  2.597538 | 0.020724 | 0.020724 |  44.87446 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  44.87446 | 17.794702 | 0.0 | 0.009912 | 0.023488 | 17.771214 | 0.292753 | 17.478461 | 0.0 | 0.056648 | 17.535108 | 17.535108 | 17.535108 | 4.870863 | 1.979276 | 0.0 | 0.0 |  153.03683 |      0.0 | 11.272899 | 0.0  0.0  0.0  0.0  0.0 | 4.870863 |
    | 02/01 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.613419 |  9.7416 |  3.338304 | 0.907373 | 0.907373 |  9.720876 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.720876 |  5.691639 | 0.0 | 0.445428 | 0.021467 |  5.670172 | 0.306709 |  0.668443 | 0.0 | 0.057898 |  0.726341 |  0.726341 |  0.726341 | 0.201762 | 1.092627 | 0.0 | 0.0 | 156.642106 | 4.695019 |  11.52171 | 0.0  0.0  0.0  0.0  0.0 | 0.201762 |
    | 02/01 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.620455 |  4.2768 |  1.846084 | 0.866652 | 0.866652 |  3.369427 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.369427 |  2.066869 | 0.0 | 0.427817 | 0.021028 |  2.045842 | 0.310227 |  1.113349 | 0.0 |  0.05916 |  1.172509 |  1.172509 |  1.172509 | 0.325697 | 1.133348 | 0.0 | 0.0 | 157.537874 | 5.317284 | 11.772778 | 0.0  0.0  0.0  0.0  0.0 | 0.325697 |
    | 02/01 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.644491 |    8.91 |  0.338261 | 0.099914 | 0.099914 |  8.043348 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  8.043348 |  4.990532 | 0.0 | 0.049957 | 0.019705 |  4.970827 | 0.322246 |  3.697378 | 0.0 | 0.060475 |  3.757853 |  3.757853 |  3.757853 | 1.043848 | 1.900086 | 0.0 | 0.0 | 160.560437 | 6.268488 | 12.034548 | 0.0  0.0  0.0  0.0  0.0 | 1.043848 |
    | 02/01 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.708542 |  21.978 |  0.107442 |  0.00859 |  0.00859 | 21.878086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 21.878086 | 14.100237 | 0.0 | 0.004295 | 0.015831 | 14.084406 | 0.354271 | 19.998623 | 0.0 | 0.061944 | 20.060568 | 20.060568 | 20.060568 |  5.57238 |  1.99141 | 0.0 | 0.0 | 168.349822 |      0.0 | 12.326875 | 0.0  0.0  0.0  0.0  0.0 |  5.57238 |
    | 02/01 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.754176 | 18.2952 |  0.107243 | 0.012392 | 0.012392 |  18.28661 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  18.28661 | 12.956823 | 0.0 | 0.006196 |  0.01316 | 12.943663 | 0.377088 |  4.626295 | 0.0 |  0.06352 |  4.689814 |  4.689814 |  4.689814 | 1.302726 | 1.987608 | 0.0 | 0.0 | 173.686572 |  7.94028 | 12.640443 | 0.0  0.0  0.0  0.0  0.0 | 1.302726 |
    | 02/01 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   0.770161 |  7.4844 |  0.105848 | 0.036055 | 0.036055 |  7.472008 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.472008 |  5.635206 | 0.0 | 0.018028 | 0.012238 |  5.622968 |  0.38508 |   5.00981 | 0.0 | 0.065128 |  5.074938 |  5.074938 |  5.074938 | 1.409705 | 1.963945 | 0.0 | 0.0 | 175.517585 | 8.168358 | 12.960395 | 0.0  0.0  0.0  0.0  0.0 | 1.409705 |
    | 02/01 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.77449 |  2.2572 |  0.104032 | 0.059768 | 0.059768 |  2.221145 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.221145 |  1.710638 | 0.0 | 0.029884 | 0.011986 |  1.698652 | 0.387245 |  1.833774 | 0.0 | 0.066738 |  1.900512 |  1.900512 |  1.900512 |  0.52792 | 1.940232 | 0.0 | 0.0 | 176.010193 | 7.645991 | 13.280902 | 0.0  0.0  0.0  0.0  0.0 |  0.52792 |
    | 03/01 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.785881 |  5.8212 |  0.104235 | 0.041931 | 0.041931 |  5.761432 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.761432 |  4.462169 | 0.0 | 0.020966 | 0.011345 |  4.450824 |  0.39294 |  3.303079 | 0.0 | 0.068369 |  3.371448 |  3.371448 |  3.371448 | 0.936513 | 1.958069 | 0.0 | 0.0 | 177.299835 | 8.400796 | 13.605473 | 0.0  0.0  0.0  0.0  0.0 | 0.936513 |
    | 03/01 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.791757 |  3.2076 |  0.104032 |  0.05435 |  0.05435 |  3.165669 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.165669 |  2.487838 | 0.0 | 0.027175 | 0.011011 |  2.476827 | 0.395879 |  2.320194 | 0.0 | 0.070007 |  2.390201 |  2.390201 |  2.390201 | 0.663945 |  1.94565 | 0.0 | 0.0 | 177.961502 |  8.16155 | 13.931345 | 0.0  0.0  0.0  0.0  0.0 | 0.663945 |
    | 03/01 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.792544 |   0.594 |  0.103014 | 0.069893 | 0.069893 |   0.53965 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.53965 |  0.427272 | 0.0 | 0.034946 | 0.010963 |  0.416309 | 0.396272 |  1.100126 | 0.0 | 0.071638 |  1.171764 |  1.171764 |  1.171764 |  0.32549 | 1.930107 | 0.0 | 0.0 | 178.049897 | 7.081462 | 14.255979 | 0.0  0.0  0.0  0.0  0.0 |  0.32549 |
    | 03/01 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.792628 |  0.2376 |  0.103422 | 0.072716 | 0.072716 |  0.167707 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.167707 |  0.132916 | 0.0 | 0.036358 | 0.010958 |  0.121958 | 0.396314 |  0.633402 | 0.0 | 0.073261 |  0.706664 |  0.706664 |  0.706664 | 0.196295 | 1.927284 | 0.0 | 0.0 | 178.059289 | 6.173703 | 14.579032 | 0.0  0.0  0.0  0.0  0.0 | 0.196295 |
    | 03/01 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.793371 |   0.594 |   0.10464 | 0.070996 | 0.070996 |  0.521284 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.521284 |  0.413185 | 0.0 | 0.035498 | 0.010916 |  0.402268 | 0.396686 |  0.472482 | 0.0 | 0.074879 |   0.54736 |   0.54736 |   0.54736 | 0.152045 | 1.929004 | 0.0 | 0.0 | 178.142807 | 5.706804 | 14.900839 | 0.0  0.0  0.0  0.0  0.0 | 0.152045 |
    | 03/01 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.798335 |  2.8512 |  0.106248 | 0.057521 | 0.057521 |  2.780204 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.780204 |  2.205735 | 0.0 | 0.028761 | 0.010641 |  2.195093 | 0.399167 |  0.830047 | 0.0 |   0.0765 |  0.906547 |  0.906547 |  0.906547 | 0.251819 | 1.942479 | 0.0 | 0.0 | 178.699157 | 6.672683 | 15.223506 | 0.0  0.0  0.0  0.0  0.0 | 0.251819 |
    | 03/01 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.798631 |  0.4752 |  0.179639 | 0.123337 | 0.123337 |  0.417679 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.417679 |  0.333448 | 0.0 | 0.061669 | 0.010608 |  0.322839 | 0.399316 |  0.563435 | 0.0 | 0.078114 |  0.641549 |  0.641549 |  0.641549 | 0.178208 | 1.876663 | 0.0 | 0.0 | 178.732328 | 6.032771 | 15.544708 | 0.0  0.0  0.0  0.0  0.0 | 0.178208 |
    | 03/01 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.797218 |  0.2376 |  0.545628 | 0.383628 | 0.383628 |  0.114263 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.114263 |  0.091254 | 0.0 | 0.191814 | 0.010622 |  0.080631 | 0.398609 |  0.368358 | 0.0 | 0.079717 |  0.448075 |  0.448075 |  0.448075 | 0.124465 | 1.616372 | 0.0 | 0.0 | 178.574145 | 5.346435 |   15.8636 | 0.0  0.0  0.0  0.0  0.0 | 0.124465 |
    | 03/01 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.792943 |     0.0 |  1.361522 | 0.980296 | 0.980296 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.490148 | 0.010713 | -0.010713 | 0.396472 |  0.241689 | 0.0 |   0.0813 |  0.322989 |  0.322989 |  0.322989 | 0.089719 | 0.636076 | 0.0 | 0.0 |  178.09471 | 4.697562 | 16.178771 | 0.0  0.0  0.0  0.0  0.0 | 0.089719 |
    | 03/01 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.764872 |     0.0 |  4.874663 | 3.509758 | 0.636076 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.19172 | 0.010953 | -0.010953 | 0.382436 |  0.178198 | 0.0 | 0.082806 |  0.261004 |  0.261004 |  0.261004 | 0.072501 |      0.0 | 0.0 | 0.0 | 174.913943 | 4.125975 | 16.478401 | 0.0  0.0  0.0  0.0  0.0 | 0.072501 |
    | 03/01 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.736086 |  0.3564 |  5.057269 | 3.513746 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.335546 | 0.012543 | -0.012543 | 0.368043 |  0.131736 | 0.0 | 0.084232 |  0.215968 |  0.215968 |  0.215968 | 0.059991 |      0.0 | 0.0 | 0.0 | 171.590941 | 3.613653 | 16.762212 | 0.0  0.0  0.0  0.0  0.0 | 0.059991 |
    | 03/01 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.719431 |  3.0888 |  6.154673 | 3.253813 |      2.0 |    1.0888 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    1.0888 |  0.801451 | 0.0 | 2.253813 | 0.014061 |   0.78739 | 0.359715 |  0.177258 | 0.0 |  0.08561 |  0.262867 |  0.262867 |  0.262867 | 0.073019 |      0.0 | 0.0 | 0.0 | 169.638537 |  3.86407 | 17.036318 | 0.0  0.0  0.0  0.0  0.0 | 0.073019 |
    | 03/01 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.688383 |  0.8316 |  6.236191 | 4.131769 |   0.8316 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.715969 | 0.015181 | -0.015181 | 0.344192 |  0.131965 | 0.0 | 0.086903 |  0.218868 |  0.218868 |  0.218868 | 0.060797 |      0.0 | 0.0 | 0.0 | 165.937749 | 3.372732 | 17.293607 | 0.0  0.0  0.0  0.0  0.0 | 0.060797 |
    | 03/01 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.659166 |  0.3564 |  5.404341 | 3.754888 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.576688 | 0.017031 | -0.017031 | 0.329583 |  0.096755 | 0.0 | 0.088116 |  0.184871 |  0.184871 |  0.184871 | 0.051353 |      0.0 | 0.0 | 0.0 | 162.378092 | 2.929363 | 17.535074 | 0.0  0.0  0.0  0.0  0.0 | 0.051353 |
    | 03/01 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.610719 |  0.3564 |  9.035929 | 6.278083 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.099883 | 0.018811 | -0.018811 | 0.305359 |  0.077624 | 0.0 | 0.089202 |  0.166826 |  0.166826 |  0.166826 | 0.046341 |      0.0 | 0.0 | 0.0 |  156.29702 | 2.527568 | 17.751232 | 0.0  0.0  0.0  0.0  0.0 | 0.046341 |
    | 03/01 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.58034 |     0.0 |  5.627818 | 4.052029 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.958804 | 0.021851 | -0.021851 |  0.29017 |  0.055641 | 0.0 | 0.090207 |  0.145848 |  0.145848 |  0.145848 | 0.040513 |      0.0 | 0.0 | 0.0 | 152.360068 | 2.159906 | 17.951194 | 0.0  0.0  0.0  0.0  0.0 | 0.040513 |
    | 03/01 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.548803 |     0.0 |  6.156027 |  4.43234 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.221357 |  0.02382 |  -0.02382 | 0.274402 |  0.039036 | 0.0 | 0.091128 |  0.130164 |  0.130164 |  0.130164 | 0.036157 |      0.0 | 0.0 | 0.0 | 148.162531 | 1.822648 | 18.134468 | 0.0  0.0  0.0  0.0  0.0 | 0.036157 |
    | 03/01 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.530323 |     0.0 |  3.811802 | 2.744497 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.541892 | 0.025919 | -0.025919 | 0.265161 |  0.024087 | 0.0 | 0.091998 |  0.116085 |  0.116085 |  0.116085 | 0.032246 |      0.0 | 0.0 | 0.0 | 145.646557 | 1.507481 | 18.307631 | 0.0  0.0  0.0  0.0  0.0 | 0.032246 |
    | 03/01 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.522211 |     0.0 |  1.747211 | 1.257992 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.145352 | 0.027177 | -0.027177 | 0.261106 |  0.012726 | 0.0 | 0.092844 |  0.105569 |  0.105569 |  0.105569 | 0.029325 |      0.0 | 0.0 | 0.0 | 144.528382 | 1.206472 | 18.475893 | 0.0  0.0  0.0  0.0  0.0 | 0.029325 |
    | 03/01 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.520914 |     0.0 |  0.318724 | 0.229481 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.207331 | 0.027736 | -0.027736 | 0.260457 |  0.005478 | 0.0 | 0.093682 |   0.09916 |   0.09916 |   0.09916 | 0.027544 |      0.0 | 0.0 | 0.0 | 144.348787 | 0.912801 | 18.642669 | 0.0  0.0  0.0  0.0  0.0 | 0.027544 |
    | 03/01 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.520639 |     0.0 |  0.101575 | 0.073134 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.065993 | 0.027826 | -0.027826 | 0.260319 |  0.001727 | 0.0 | 0.094515 |  0.096242 |  0.096242 |  0.096242 | 0.026734 |      0.0 | 0.0 | 0.0 |  144.31062 | 0.622929 | 18.808473 | 0.0  0.0  0.0  0.0  0.0 | 0.026734 |
    | 03/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.520374 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.064624 | 0.027845 | -0.027845 | 0.260187 |  0.000267 | 0.0 | 0.095343 |   0.09561 |   0.09561 |   0.09561 | 0.026558 |      0.0 | 0.0 | 0.0 |  144.27384 | 0.334631 | 18.973317 | 0.0  0.0  0.0  0.0  0.0 | 0.026558 |
    | 03/01 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.520116 |     0.0 |  0.097808 | 0.070422 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.063512 | 0.027863 | -0.027863 | 0.260058 |  0.000001 | 0.0 | 0.096167 |  0.096168 |  0.096168 |  0.096168 | 0.026713 |      0.0 | 0.0 | 0.0 | 144.238191 | 0.046709 | 19.137208 | 0.0  0.0  0.0  0.0  0.0 | 0.026713 |
    | 03/01 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |   0.519866 |     0.0 |  0.096529 | 0.069501 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.062667 | 0.027881 | -0.027881 | 0.018828 |       0.0 | 0.0 |  0.09578 |   0.09578 |   0.09578 |   0.09578 | 0.026606 |      0.0 | 0.0 | 0.0 | 144.203406 |      0.0 | 19.060256 | 0.0  0.0  0.0  0.0  0.0 | 0.026606 |
    | 04/01 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.519419 |     0.0 |  0.095455 | 0.068728 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.061942 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.095301 |  0.095301 |  0.095301 |  0.095301 | 0.026473 |      0.0 | 0.0 | 0.0 | 144.141463 |      0.0 | 18.964955 | 0.0  0.0  0.0  0.0  0.0 | 0.026473 |
    | 04/01 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.518961 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.063579 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.094825 |  0.094825 |  0.094825 |  0.094825 |  0.02634 |      0.0 | 0.0 | 0.0 | 144.077884 |      0.0 |  18.87013 | 0.0  0.0  0.0  0.0  0.0 |  0.02634 |
    | 04/01 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.518501 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.063825 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.094351 |  0.094351 |  0.094351 |  0.094351 | 0.026209 |      0.0 | 0.0 | 0.0 | 144.014059 |      0.0 |  18.77578 | 0.0  0.0  0.0  0.0  0.0 | 0.026209 |
    | 04/01 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.518307 |  1.5444 |  0.096957 | 0.059819 | 0.059819 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.026921 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.093879 |  0.093879 |  0.093879 |  0.093879 | 0.026077 | 1.484581 | 0.0 | 0.0 | 143.987137 |      0.0 | 18.681901 | 0.0  0.0  0.0  0.0  0.0 | 0.026077 |
    | 04/01 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.518082 |     0.0 |  0.096744 | 0.069656 | 0.069656 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.031342 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 |  0.09341 |   0.09341 |   0.09341 |   0.09341 | 0.025947 | 1.414925 | 0.0 | 0.0 | 143.955795 |      0.0 | 18.588491 | 0.0  0.0  0.0  0.0  0.0 | 0.025947 |
    | 04/01 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.517852 |     0.0 |  0.098442 | 0.070878 | 0.070878 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.031885 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.092942 |  0.092942 |  0.092942 |  0.092942 | 0.025817 | 1.344047 | 0.0 | 0.0 |  143.92391 |      0.0 | 18.495549 | 0.0  0.0  0.0  0.0  0.0 | 0.025817 |
    | 04/01 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.517014 |     0.0 |  0.359866 | 0.259103 | 0.259103 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.116535 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.092478 |  0.092478 |  0.092478 |  0.092478 | 0.025688 | 1.084943 | 0.0 | 0.0 | 143.807375 |      0.0 | 18.403071 | 0.0  0.0  0.0  0.0  0.0 | 0.025688 |
    | 04/01 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.513563 |  0.8316 |  1.614671 | 1.069795 | 1.069795 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.480764 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.092015 |  0.092015 |  0.092015 |  0.092015 |  0.02556 | 0.846748 | 0.0 | 0.0 | 143.326611 |      0.0 | 18.311056 | 0.0  0.0  0.0  0.0  0.0 |  0.02556 |
    | 04/01 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.501086 |  0.4752 |  3.707216 | 2.545322 | 1.321948 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.751726 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.091555 |  0.091555 |  0.091555 |  0.091555 | 0.025432 |      0.0 | 0.0 | 0.0 | 141.574885 |      0.0 |   18.2195 | 0.0  0.0  0.0  0.0  0.0 | 0.025432 |
    | 04/01 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.474607 |  0.1188 |  6.022315 | 4.284859 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.791428 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.091098 |  0.091098 |  0.091098 |  0.091098 | 0.025305 |      0.0 | 0.0 | 0.0 | 137.783457 |      0.0 | 18.128403 | 0.0  0.0  0.0  0.0  0.0 | 0.025305 |
    | 04/01 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.441402 |  0.4752 |  8.299842 | 5.698554 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  4.90729 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.090642 |  0.090642 |  0.090642 |  0.090642 | 0.025178 |      0.0 | 0.0 | 0.0 | 132.876167 |      0.0 | 18.037761 | 0.0  0.0  0.0  0.0  0.0 | 0.025178 |
    | 04/01 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.403579 |     0.0 |  9.734061 | 7.008524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.820411 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.090189 |  0.090189 |  0.090189 |  0.090189 | 0.025052 |      0.0 | 0.0 | 0.0 | 127.055756 |      0.0 | 17.947572 | 0.0  0.0  0.0  0.0  0.0 | 0.025052 |
    | 04/01 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.368998 |     0.0 |  9.733892 | 7.008402 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.565361 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.089738 |  0.089738 |  0.089738 |  0.089738 | 0.024927 |      0.0 | 0.0 | 0.0 | 121.490395 |      0.0 | 17.857834 | 0.0  0.0  0.0  0.0  0.0 | 0.024927 |
    | 04/01 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.328874 |     0.0 | 12.429676 | 8.949367 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.795388 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.089289 |  0.089289 |  0.089289 |  0.089289 | 0.024803 |      0.0 | 0.0 | 0.0 | 114.695007 |      0.0 | 17.768545 | 0.0  0.0  0.0  0.0  0.0 | 0.024803 |
    | 04/01 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.297445 |     0.0 | 10.884737 | 7.837011 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.617912 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.088843 |  0.088843 |  0.088843 |  0.088843 | 0.024679 |      0.0 | 0.0 | 0.0 | 109.077094 |      0.0 | 17.679702 | 0.0  0.0  0.0  0.0  0.0 | 0.024679 |
    | 04/01 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.273532 |     0.0 |  9.119821 | 6.566271 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.476436 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.088399 |  0.088399 |  0.088399 |  0.088399 | 0.024555 |      0.0 | 0.0 | 0.0 | 104.600658 |      0.0 | 17.591304 | 0.0  0.0  0.0  0.0  0.0 | 0.024555 |
    | 04/01 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.255842 |     0.0 |  7.306062 | 5.260364 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.438985 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.087957 |  0.087957 |  0.087957 |  0.087957 | 0.024432 |      0.0 | 0.0 | 0.0 | 101.161673 |      0.0 | 17.503347 | 0.0  0.0  0.0  0.0  0.0 | 0.024432 |
    | 04/01 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.248028 |     0.0 |  3.420163 | 2.462517 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.556952 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.087517 |  0.087517 |  0.087517 |  0.087517 |  0.02431 |      0.0 | 0.0 | 0.0 |  99.604721 |      0.0 | 17.415831 | 0.0  0.0  0.0  0.0  0.0 |  0.02431 |
    | 04/01 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.244649 |     0.0 |  1.518709 | 1.093471 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.680718 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.087079 |  0.087079 |  0.087079 |  0.087079 | 0.024189 |      0.0 | 0.0 | 0.0 |  98.924003 |      0.0 | 17.328751 | 0.0  0.0  0.0  0.0  0.0 | 0.024189 |
    | 04/01 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.243808 |     0.0 |  0.382278 |  0.27524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.170174 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.086644 |  0.086644 |  0.086644 |  0.086644 | 0.024068 |      0.0 | 0.0 | 0.0 |  98.753829 |      0.0 | 17.242108 | 0.0  0.0  0.0  0.0  0.0 | 0.024068 |
    | 04/01 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.24358 |     0.0 |  0.103829 | 0.074757 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046141 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.086211 |  0.086211 |  0.086211 |  0.086211 | 0.023947 |      0.0 | 0.0 | 0.0 |  98.707688 |      0.0 | 17.155897 | 0.0  0.0  0.0  0.0  0.0 | 0.023947 |
    | 04/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.243362 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.044194 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.085779 |  0.085779 |  0.085779 |  0.085779 | 0.023828 |      0.0 | 0.0 | 0.0 |  98.663494 |      0.0 | 17.070118 | 0.0  0.0  0.0  0.0  0.0 | 0.023828 |
    | 04/01 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.243147 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043519 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.085351 |  0.085351 |  0.085351 |  0.085351 | 0.023708 |      0.0 | 0.0 | 0.0 |  98.619975 |      0.0 | 16.984767 | 0.0  0.0  0.0  0.0  0.0 | 0.023708 |
    | 04/01 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |   0.242932 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043688 |      0.0 |       0.0 |      0.0 |       0.0 | 0.0 | 0.084924 |  0.084924 |  0.084924 |  0.084924 |  0.02359 |      0.0 | 0.0 | 0.0 |  98.576287 |      0.0 | 16.899843 | 0.0  0.0  0.0  0.0  0.0 |  0.02359 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v2_sealed:

sealed area
___________

Besides slight differences in runoff concentration, the following calculation
reproduces the results of the :ref:`hland_v1_sealed` experiment of |hland_v1| exactly:

.. integration-test::

    >>> recstep(100)
    >>> zonetype(SEALED)
    >>> test("hland_v2_sealed")
    |        date |    p |    t |   tn |       epn | tmean |   tc | fracrain | rfc | sfc |    cfact | swe | gact | contriarea |      pc |        ep |      epc |       ei |        tf | spl | wcl | spg | wcg | glmelt | melt | refr |       in_ |         r |        sr |  ea |  cf | inuz | perc |  q0 |  el |  q1 |      inuh |     outuh |        rt |       qt |       ic |  sp |  wc |  sm |  uz |  lz |                                                 sc |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |   0.11682 |  0.08411 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.160107 |  0.160107 | 0.044474 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.001685   0.007302  0.016758  0.027475  0.036673 | 0.044474 |
    | 01/01 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.113449 | 0.081683 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.073779 |  0.073779 | 0.020494 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.000057   0.000435  0.001704  0.004551  0.009367 | 0.020494 |
    | 01/01 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.014282 |  0.014282 | 0.003967 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.000002   0.000021  0.000117  0.000439  0.001253 | 0.003967 |
    | 01/01 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.111348 | 0.080171 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.001674 |  0.001674 | 0.000465 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0   0.000001  0.000007  0.000032  0.000119 | 0.000465 |
    | 01/01 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.112498 | 0.080999 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.000147 |  0.000147 | 0.000041 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0  0.000002  0.000009 | 0.000041 |
    | 01/01 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.119203 | 0.085826 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.000011 |  0.000011 | 0.000003 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0  0.000001 | 0.000003 |
    | 01/01 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  0.338613 | 0.243802 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.000001 |  0.000001 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  2.242135 | 1.614337 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  5.068662 | 3.649436 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  8.488365 | 6.111623 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |   9.58666 | 6.902395 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 | 10.869046 | 7.825713 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  5.946366 | 4.281384 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  7.685035 | 5.533225 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  8.581238 | 6.178491 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  0.2376 |  2.774152 | 1.950491 |   0.2376 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |  2.122607 | 1.528277 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |     0.0 |   1.31842 | 0.949262 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  1.5444 |  0.871144 | 0.537465 | 0.537465 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 | 1.006935 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  6.6528 |  0.259718 | 0.096141 | 0.096141 |  5.659735 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.659735 |  5.659735 |  5.659735 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  5.659735 |  0.403991 |  0.403991 |  0.11222 | 1.903859 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  1.586005   1.401611  1.091214  0.739431  0.437484 |  0.11222 |
    | 01/01 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  3.4452 |  0.115333 | 0.058839 | 0.058839 |  3.349059 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.349059 |  3.349059 |  3.349059 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  3.349059 |   3.07289 |   3.07289 | 0.853581 | 1.941161 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.991947     1.0548  1.139885  1.190051   1.15523 | 0.853581 |
    | 01/01 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  5.8212 |  0.114207 | 0.045942 | 0.045942 |  5.762361 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.762361 |  5.762361 |  5.762361 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  5.762361 |   4.04129 |   4.04129 | 1.122581 | 1.954058 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  1.648196   1.574016  1.455511  1.333088  1.242174 | 1.122581 |
    | 01/01 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 | 12.5928 |  0.113828 | 0.023264 | 0.023264 | 12.546858 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.546858 | 12.546858 | 12.546858 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 | 12.546858 |  5.350397 |  5.350397 | 1.486221 | 1.976736 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  3.571506   3.345395  2.956657   2.49859  2.077298 | 1.486221 |
    | 01/01 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |        1.0 |  0.1188 |   0.11307 | 0.080449 | 0.080449 |  0.095536 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.095536 |  0.095536 |  0.095536 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.095536 |   8.57257 |   8.57257 | 2.381269 | 1.919551 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.147144   0.537651  1.169328  1.826995  2.291295 | 2.381269 |
    | 02/01 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.8316 |   0.11307 | 0.074914 | 0.074914 |  0.751151 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.751151 |  0.751151 |  0.751151 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.751151 |  4.886703 |  4.886703 | 1.357417 | 1.925086 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.215451   0.220671  0.272464  0.424294   0.70398 | 1.357417 |
    | 02/01 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |   3.564 |   0.11307 | 0.057003 | 0.057003 |  3.489086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.489086 |  3.489086 |  3.489086 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  3.489086 |  1.629616 |  1.629616 | 0.452671 | 1.942997 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.984994     0.8957  0.747425   0.58866   0.47955 | 0.452671 |
    | 02/01 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  2.4948 |  0.112498 | 0.063115 | 0.063115 |  2.437797 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.437797 |  2.437797 |  2.437797 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  2.437797 |  2.303066 |  2.303066 | 0.639741 | 1.936885 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.716332   0.744558  0.782109  0.802804  0.785259 | 0.639741 |
    | 02/01 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 12.3552 |  0.112116 | 0.023465 | 0.023465 | 12.292085 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.292085 | 12.292085 | 12.292085 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 | 12.292085 |  3.368095 |  3.368095 | 0.935582 | 1.976535 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  3.468705   3.149656  2.615426  2.015191  1.506073 | 0.935582 |
    | 02/01 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |   4.158 |  0.111732 |  0.05308 |  0.05308 |  4.134535 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.134535 |  4.134535 |  4.134535 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  4.134535 |  7.553655 |  7.553655 | 2.098238 |  1.94692 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  1.275511   1.519746  1.895127  2.241003  2.404544 | 2.098238 |
    | 02/01 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.0392 |  0.111348 |  0.05353 |  0.05353 |   3.98612 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.98612 |   3.98612 |   3.98612 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |   3.98612 |   6.80633 |   6.80633 | 1.890647 |  1.94647 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  1.160004   1.181664  1.244361  1.369998  1.559694 | 1.890647 |
    | 02/01 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  1.4256 |  0.217266 | 0.135647 | 0.135647 |   1.37207 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   1.37207 |   1.37207 |   1.37207 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |   1.37207 |  4.443818 |  4.443818 | 1.234394 | 1.864353 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.423586   0.509934  0.658605   0.83732  1.014528 | 1.234394 |
    | 02/01 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.1188 |   1.46695 |  1.04373 |  1.04373 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  2.475866 |  2.475866 | 0.687741 | 0.939422 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.014276   0.064774  0.159592  0.289434  0.440032 | 0.687741 |
    | 02/01 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |     0.0 |  3.532824 | 2.543633 | 0.939422 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.811453 |  0.811453 | 0.225404 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.000481   0.003787  0.015356  0.042994  0.094037 | 0.225404 |
    | 02/01 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |     0.0 |  2.239679 | 1.612569 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.139626 |  0.139626 | 0.038785 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.000016   0.000182  0.001034  0.003993  0.011804 | 0.038785 |
    | 02/01 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.4752 |  2.854761 | 1.960038 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |   0.01559 |   0.01559 |  0.00433 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.000001   0.000008  0.000058  0.000289  0.001084 |  0.00433 |
    | 02/01 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  0.1188 |  7.210496 | 5.130246 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.001336 |  0.001336 | 0.000371 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0  0.000003  0.000018  0.000082 | 0.000371 |
    | 02/01 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.2768 |  3.914748 | 1.837796 | 1.837796 |    2.2768 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    2.2768 |    2.2768 |    2.2768 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |    2.2768 |  0.162614 |  0.162614 | 0.045171 | 0.162204 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.638019    0.56384  0.438974  0.297459  0.175996 | 0.045171 |
    | 02/01 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  7.0092 | 10.233004 | 3.655358 |      2.0 |  5.171404 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.171404 |  5.171404 |  5.171404 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  5.171404 |  1.509136 |  1.509136 | 0.419204 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  1.470665   1.371359   1.19586  0.978349  0.760324 | 0.419204 |
    | 02/01 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  1.3068 |  4.693389 | 2.965278 |   1.3068 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |   3.37502 |   3.37502 | 0.937505 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.049566   0.211441  0.472493  0.741871  0.926165 | 0.937505 |
    | 02/01 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 24.5916 |  2.448478 |  0.15074 |  0.15074 |   22.5916 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   22.5916 |   22.5916 |   22.5916 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |   22.5916 |  3.563656 |  3.563656 | 0.989905 |  1.84926 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  6.332424   5.607414  4.404781  3.080234  2.004627 | 0.989905 |
    | 02/01 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 45.0252 |  2.597538 | 0.020724 | 0.020724 |  44.87446 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  44.87446 |  44.87446 |  44.87446 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  44.87446 | 14.913088 | 14.913088 | 4.142525 | 1.979276 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 12.788415  12.013385   10.6279  8.879071  7.082082 | 4.142525 |
    | 02/01 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  9.7416 |  3.338304 | 0.907373 | 0.907373 |  9.720876 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.720876 |  9.720876 |  9.720876 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  9.720876 | 31.006199 | 31.006199 | 8.612833 | 1.092627 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  3.155056   4.248941  6.000518  7.776078  8.924936 | 8.612833 |
    | 02/01 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  4.2768 |  1.846084 | 0.866652 | 0.866652 |  3.369427 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.369427 |  3.369427 |  3.369427 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  3.369427 | 22.249557 | 22.249557 | 6.180432 | 1.133348 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  1.050537   1.332083  1.925885  2.856168  4.060725 | 6.180432 |
    | 02/01 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |    8.91 |  0.338261 | 0.099914 | 0.099914 |  8.043348 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  8.043348 |  8.043348 |  8.043348 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  8.043348 |  9.097558 |  9.097558 | 2.527099 | 1.900086 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  2.289362   2.154821  1.964014  1.840546  1.922445 | 2.527099 |
    | 02/01 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  21.978 |  0.107442 |  0.00859 |  0.00859 | 21.878086 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 21.878086 | 21.878086 | 21.878086 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 | 21.878086 |  7.877775 |  7.877775 | 2.188271 |  1.99141 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  6.207968   5.747843  4.959388  4.039182  3.217118 | 2.188271 |
    | 02/01 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 | 18.2952 |  0.107243 | 0.012392 | 0.012392 |  18.28661 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  18.28661 |  18.28661 |  18.28661 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  18.28661 | 15.404588 | 15.404588 | 4.279052 | 1.987608 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  5.333613   5.419761  5.512614  5.499941   5.28759 | 4.279052 |
    | 02/01 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  7.4844 |  0.105848 | 0.036055 | 0.036055 |  7.472008 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.472008 |  7.472008 |  7.472008 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  7.472008 | 17.867701 | 17.867701 |  4.96325 | 1.963945 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  2.273612   2.632281  3.243962   3.94898  4.558993 |  4.96325 |
    | 02/01 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |        1.0 |  2.2572 |  0.104032 | 0.059768 | 0.059768 |  2.221145 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.221145 |  2.221145 |  2.221145 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  2.221145 | 11.884713 | 11.884713 | 3.301309 | 1.940232 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.699051   0.894203  1.263271  1.772825  2.364909 | 3.301309 |
    | 03/01 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  5.8212 |  0.104235 | 0.041931 | 0.041931 |  5.761432 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.761432 |  5.761432 |  5.761432 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  5.761432 |  5.639405 |  5.639405 | 1.566501 | 1.958069 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  1.638063   1.535468  1.386057  1.273322  1.283375 | 1.566501 |
    | 03/01 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  3.2076 |  0.104032 |  0.05435 |  0.05435 |  3.165669 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.165669 |  3.165669 |  3.165669 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  3.165669 |  4.603302 |  4.603302 | 1.278695 |  1.94565 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.942311   1.019744  1.139348  1.253681  1.323569 | 1.278695 |
    | 03/01 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |   0.594 |  0.103014 | 0.069893 | 0.069893 |   0.53965 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.53965 |   0.53965 |   0.53965 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |   0.53965 |  3.856116 |  3.856116 | 1.071143 | 1.930107 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.182983   0.273875  0.435213  0.635569  0.834547 | 1.071143 |
    | 03/01 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.2376 |  0.103422 | 0.072716 | 0.072716 |  0.167707 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.167707 |  0.167707 |  0.167707 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.167707 |  1.822639 |  1.822639 | 0.506289 | 1.927284 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.053163    0.07132  0.112376  0.183238   0.28716 | 0.506289 |
    | 03/01 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |   0.594 |   0.10464 | 0.070996 | 0.070996 |  0.521284 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.521284 |  0.521284 |  0.521284 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.521284 |  0.592179 |  0.592179 | 0.164494 | 1.929004 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.147869    0.13747  0.122359  0.111787  0.116876 | 0.164494 |
    | 03/01 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  2.8512 |  0.106248 | 0.057521 | 0.057521 |  2.780204 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.780204 |  2.780204 |  2.780204 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  2.780204 |  0.589542 |  0.589542 | 0.163762 | 1.942479 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.784069   0.709752  0.583564  0.438431  0.311207 | 0.163762 |
    | 03/01 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.4752 |  0.179639 | 0.123337 | 0.123337 |  0.417679 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.417679 |  0.417679 |  0.417679 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.417679 |  1.621645 |  1.621645 | 0.450457 | 1.876663 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.14347   0.215444  0.328213  0.437178  0.498752 | 0.450457 |
    | 03/01 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.2376 |  0.545628 | 0.383628 | 0.383628 |  0.114263 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.114263 |  0.114263 |  0.114263 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |  0.114263 |  1.216937 |  1.216937 | 0.338038 | 1.616372 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.036855   0.051676  0.084428  0.138029  0.209396 | 0.338038 |
    | 03/01 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  1.361522 | 0.980296 | 0.980296 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.409443 |  0.409443 | 0.113734 | 0.636076 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.001242   0.005882  0.015621  0.031809  0.056386 | 0.113734 |
    | 03/01 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  4.874663 | 3.509758 | 0.636076 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.095016 |  0.095016 | 0.026393 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.000042   0.000338  0.001422  0.004206  0.009917 | 0.026393 |
    | 03/01 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.3564 |  5.057269 | 3.513746 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.014283 |  0.014283 | 0.003968 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.000001   0.000016  0.000094  0.000374  0.001156 | 0.003968 |
    | 03/01 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  3.0888 |  6.154673 | 3.253813 |      2.0 |    1.0888 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    1.0888 |    1.0888 |    1.0888 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |    1.0888 |  0.079225 |  0.079225 | 0.022007 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.30511   0.269638  0.209929  0.142276  0.084264 | 0.022007 |
    | 03/01 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.8316 |  6.236191 | 4.131769 |   0.8316 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.545289 |  0.545289 | 0.151469 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.010283   0.043365  0.095068  0.144766  0.172445 | 0.151469 |
    | 03/01 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.3564 |  5.404341 | 3.754888 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.375739 |  0.375739 | 0.104372 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.000347   0.002617  0.010021  0.025965  0.051241 | 0.104372 |
    | 03/01 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |  0.3564 |  9.035929 | 6.278083 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.079619 |  0.079619 | 0.022117 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  0.000012   0.000127  0.000697   0.00257  0.007164 | 0.022117 |
    | 03/01 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  5.627818 | 4.052029 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.009641 |  0.009641 | 0.002678 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0   0.000006   0.00004  0.000191  0.000691 | 0.002678 |
    | 03/01 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  6.156027 |  4.43234 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.000861 |  0.000861 | 0.000239 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0  0.000002  0.000012  0.000054 | 0.000239 |
    | 03/01 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  3.811802 | 2.744497 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.000064 |  0.000064 | 0.000018 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0  0.000001  0.000004 | 0.000018 |
    | 03/01 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  1.747211 | 1.257992 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.000004 |  0.000004 | 0.000001 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 | 0.000001 |
    | 03/01 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.318724 | 0.229481 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 03/01 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.101575 | 0.073134 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 03/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 03/01 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.097808 | 0.070422 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 03/01 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |        1.0 |     0.0 |  0.096529 | 0.069501 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.095455 | 0.068728 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  1.5444 |  0.096957 | 0.059819 | 0.059819 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 | 1.484581 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.096744 | 0.069656 | 0.069656 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 | 1.414925 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 | 0.070878 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 | 1.344047 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.359866 | 0.259103 | 0.259103 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 | 1.084943 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.8316 |  1.614671 | 1.069795 | 1.069795 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 | 0.846748 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.4752 |  3.707216 | 2.545322 | 1.321948 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.1188 |  6.022315 | 4.284859 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |  0.4752 |  8.299842 | 5.698554 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  9.734061 | 7.008524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  9.733892 | 7.008402 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 | 12.429676 | 8.949367 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 | 10.884737 | 7.837011 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  9.119821 | 6.566271 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  7.306062 | 5.260364 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  3.420163 | 2.462517 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  1.518709 | 1.093471 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.382278 |  0.27524 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.103829 | 0.074757 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.099494 | 0.071636 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |   0.09802 | 0.070574 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |
    | 04/01 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 |       0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.0 | 0.0 |       0.0 |       0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0        0.0       0.0       0.0       0.0 |      0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v2_ilake:

internal lake
_____________

Besides slight differences in runoff concentration, the following calculation
reproduces the results of the :ref:`hland_v1_ilake` experiment of |hland_v1| exactly:

.. integration-test::

    >>> recstep(100)
    >>> zonetype(ILAKE)
    >>> ttice(13.0)
    >>> test("hland_v2_ilake")
    |        date |    p |    t |   tn |       epn | tmean |   tc | fracrain | rfc | sfc | cfact | swe | gact | contriarea |      pc |        ep |      epc |  ei |      tf | spl | wcl | spg | wcg | glmelt | melt | refr |     in_ |       r |  sr |  ea |  cf | inuz | perc |  q0 |       el |       q1 |     inuh |    outuh |       rt |       qt |  ic |  sp |  wc |  sm |  uz |         lz |                                               sc |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |   0.11682 |  0.08411 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.08411 | 0.049579 | 0.049579 | 0.163646 | 0.163646 | 0.045457 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    9.86631 | 0.015579  0.019581  0.026317  0.033952  0.040505 | 0.045457 |
    | 01/01 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.113449 | 0.081683 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.081683 | 0.048923 | 0.048923 | 0.102096 | 0.102096 |  0.02836 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.735704 | 0.014235  0.014526  0.015465  0.017534  0.021001 |  0.02836 |
    | 01/01 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.112498 | 0.080999 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.080999 | 0.048274 | 0.048274 | 0.059332 | 0.059332 | 0.016481 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.606431 | 0.014007  0.014043  0.014152  0.014433  0.015066 | 0.016481 |
    | 01/01 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.111348 | 0.080171 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.080171 | 0.047631 | 0.047631 | 0.049752 | 0.049752 |  0.01382 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.478629 |  0.01382  0.013843  0.013887  0.013957  0.014074 |  0.01382 |
    | 01/01 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.112498 | 0.080999 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.080999 | 0.046988 | 0.046988 | 0.048025 | 0.048025 |  0.01334 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.350643 | 0.013633  0.013656  0.013696  0.013749   0.01381 |  0.01334 |
    | 01/01 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.119203 | 0.085826 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.085826 | 0.046324 | 0.046324 | 0.047284 | 0.047284 | 0.013134 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.218493 | 0.013441  0.013464  0.013505  0.013559  0.013616 | 0.013134 |
    | 01/01 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.338613 | 0.243802 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.243802 | 0.044873 | 0.044873 | 0.046565 | 0.046565 | 0.012935 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   8.929818 | 0.013028  0.013077  0.013161  0.013264  0.013363 | 0.012935 |
    | 01/01 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  2.242135 | 1.614337 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.614337 | 0.036577 | 0.036577 | 0.044964 | 0.044964 |  0.01249 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   7.278903 | 0.010689  0.010963  0.011429   0.01197  0.012456 |  0.01249 |
    | 01/01 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  5.068662 | 3.649436 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.649436 | 0.018147 | 0.018147 | 0.038939 | 0.038939 | 0.010816 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   3.611319 | 0.005446  0.006064  0.007137  0.008422  0.009645 | 0.010816 |
    | 01/01 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  8.488365 | 6.111623 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 6.111623 |      0.0 |      0.0 |  0.02544 |  0.02544 | 0.007067 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -2.500304 | 0.000184  0.000816  0.001952    0.0034  0.004923 | 0.007067 |
    | 01/01 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |   9.58666 | 6.902395 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 6.902395 |      0.0 |      0.0 | 0.009374 | 0.009374 | 0.002604 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -9.402699 | 0.000006  0.000048  0.000192  0.000528  0.001126 | 0.002604 |
    | 01/01 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 | 10.869046 | 7.825713 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 7.825713 |      0.0 |      0.0 | 0.001689 | 0.001689 | 0.000469 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -17.228412 |      0.0  0.000002  0.000013   0.00005  0.000145 | 0.000469 |
    | 01/01 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  5.946366 | 4.281384 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 4.281384 |      0.0 |      0.0 | 0.000192 | 0.000192 | 0.000053 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -21.509796 |      0.0       0.0  0.000001  0.000004  0.000013 | 0.000053 |
    | 01/01 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  7.685035 | 5.533225 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 5.533225 |      0.0 |      0.0 | 0.000017 | 0.000017 | 0.000005 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -27.043021 |      0.0       0.0       0.0       0.0  0.000001 | 0.000005 |
    | 01/01 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  8.581238 | 6.178491 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 6.178491 |      0.0 |      0.0 | 0.000001 | 0.000001 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -33.221512 |      0.0       0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.2376 |  2.774152 | 1.950491 | 0.0 |  0.2376 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.2376 |  0.2376 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.950491 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -34.934403 |      0.0       0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  2.122607 | 1.528277 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.528277 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -36.46268 |      0.0       0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |   1.31842 | 0.949262 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.949262 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -37.411942 |      0.0       0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  1.5444 |  0.871144 | 0.537465 | 0.0 |  1.5444 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.5444 |  1.5444 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.537465 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -36.405007 |      0.0       0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  6.6528 |  0.259718 | 0.096141 | 0.0 |  6.6528 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  6.6528 |  6.6528 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.096141 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -29.848348 |      0.0       0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  3.4452 |  0.115333 | 0.058839 | 0.0 |  3.4452 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.4452 |  3.4452 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.058839 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -26.461986 |      0.0       0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  5.8212 |  0.114207 | 0.045942 | 0.0 |  5.8212 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.8212 |  5.8212 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.045942 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -20.686729 |      0.0       0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 | 12.5928 |  0.113828 | 0.023264 | 0.0 | 12.5928 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.5928 | 12.5928 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.023264 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -8.117193 |      0.0       0.0       0.0       0.0       0.0 |      0.0 |
    | 01/01 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.1188 |   0.11307 | 0.080449 | 0.0 |  0.1188 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.1188 |  0.1188 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.080449 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -8.078842 |      0.0       0.0       0.0       0.0       0.0 |      0.0 |
    | 02/01 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.8316 |   0.11307 | 0.074914 | 0.0 |  0.8316 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.8316 |  0.8316 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.074914 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -7.322156 |      0.0       0.0       0.0       0.0       0.0 |      0.0 |
    | 02/01 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |   3.564 |   0.11307 | 0.057003 | 0.0 |   3.564 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.564 |   3.564 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.057003 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -3.815158 |      0.0       0.0       0.0       0.0       0.0 |      0.0 |
    | 02/01 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  2.4948 |  0.112498 | 0.063115 | 0.0 |  2.4948 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.4948 |  2.4948 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.063115 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -1.383473 |      0.0       0.0       0.0       0.0       0.0 |      0.0 |
    | 02/01 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 | 12.3552 |  0.112116 | 0.023465 | 0.0 | 12.3552 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.3552 | 12.3552 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.023465 | 0.054741 | 0.054741 | 0.003907 | 0.003907 | 0.001085 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  10.893521 |  0.01534  0.013556  0.010554  0.007152  0.004231 | 0.001085 |
    | 02/01 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |   4.158 |  0.111732 |  0.05308 | 0.0 |   4.158 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   4.158 |   4.158 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.05308 | 0.074992 | 0.074992 | 0.032762 | 0.032762 | 0.009101 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  14.923448 | 0.021532  0.020752  0.019238  0.017076  0.014466 | 0.009101 |
    | 02/01 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  4.0392 |  0.111348 |  0.05353 | 0.0 |  4.0392 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.0392 |  4.0392 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.05353 | 0.094546 | 0.094546 | 0.063188 | 0.063188 | 0.017552 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  18.814573 |  0.02722  0.026532   0.02528  0.023628  0.021761 | 0.017552 |
    | 02/01 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  1.4256 |  0.217266 | 0.135647 | 0.0 |  1.4256 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.4256 |  1.4256 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.135647 | 0.100523 | 0.100523 | 0.084396 | 0.084396 | 0.023443 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  20.004003 | 0.029086  0.028846  0.028362  0.027621  0.026633 | 0.023443 |
    | 02/01 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.1188 |   1.46695 |  1.04373 | 0.0 |  0.1188 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.1188 |  0.1188 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  1.04373 | 0.095395 | 0.095395 | 0.095736 | 0.095736 | 0.026593 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  18.983677 | 0.027713  0.027864   0.02809   0.02827  0.028272 | 0.026593 |
    | 02/01 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  3.532824 | 2.543633 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 2.543633 |   0.0822 |   0.0822 | 0.095942 | 0.095942 | 0.026651 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  16.357844 | 0.023969  0.024409  0.025166  0.026057  0.026865 | 0.026651 |
    | 02/01 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  2.239679 | 1.612569 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.612569 | 0.073726 | 0.073726 |  0.08759 |  0.08759 | 0.024331 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  14.671548 | 0.021468  0.021773  0.022338  0.023091  0.023932 | 0.024331 |
    | 02/01 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.4752 |  2.854761 | 1.960038 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.4752 |  0.4752 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.960038 | 0.065934 | 0.065934 | 0.077933 | 0.077933 | 0.021648 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  13.120776 |   0.0192  0.019474  0.019971  0.020621  0.021338 | 0.021648 |
    | 02/01 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.1188 |  7.210496 | 5.130246 | 0.0 |  0.1188 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.1188 |  0.1188 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 5.130246 | 0.040547 | 0.040547 |  0.06829 |  0.06829 | 0.018969 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   8.068783 | 0.012009  0.012855  0.014309  0.016034  0.017653 | 0.018969 |
    | 02/01 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  4.2768 |  3.914748 | 1.837796 | 0.0 |  4.2768 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.2768 |  4.2768 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.837796 | 0.052539 | 0.052539 | 0.053006 | 0.053006 | 0.014724 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  10.455248 | 0.015128  0.014793  0.014327  0.014017  0.014128 | 0.014724 |
    | 02/01 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  7.0092 | 10.233004 | 3.655358 | 0.0 |  7.0092 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.0092 |  7.0092 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.655358 | 0.069045 | 0.069045 | 0.050772 | 0.050772 | 0.014103 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  13.740045 | 0.019858  0.019297  0.018318  0.017143  0.016051 | 0.014103 |
    | 02/01 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  1.3068 |  4.693389 | 2.965278 | 0.0 |  1.3068 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.3068 |  1.3068 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 2.965278 | 0.060408 | 0.060408 | 0.060626 | 0.060626 |  0.01684 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  12.021159 | 0.017597  0.017841  0.018188  0.018433  0.018389 |  0.01684 |
    | 02/01 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 | 24.5916 |  2.448478 |  0.15074 | 0.0 | 24.5916 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 24.5916 | 24.5916 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.15074 |  0.18231 |  0.18231 | 0.071344 | 0.071344 | 0.019818 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  36.279708 | 0.051681  0.047727  0.041095  0.033628  0.027283 | 0.019818 |
    | 02/01 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 | 45.0252 |  2.597538 | 0.020724 | 0.0 | 45.0252 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 45.0252 | 45.0252 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.020724 | 0.406421 | 0.406421 | 0.146689 | 0.146689 | 0.040747 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  80.877763 | 0.115632  0.108063   0.09488  0.078951  0.063621 | 0.040747 |
    | 02/01 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  9.7416 |  3.338304 | 0.907373 | 0.0 |  9.7416 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.7416 |  9.7416 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.907373 |  0.44856 |  0.44856 | 0.303499 | 0.303499 | 0.084305 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   89.26343 | 0.129595  0.127717  0.123689  0.117143  0.108063 | 0.084305 |
    | 02/01 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  4.2768 |  1.846084 | 0.866652 | 0.0 |  4.2768 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.2768 |  4.2768 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.866652 | 0.463368 | 0.463368 | 0.411844 | 0.411844 | 0.114401 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   92.21021 | 0.134216  0.133615  0.132364  0.130311  0.127226 | 0.114401 |
    | 02/01 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |    8.91 |  0.338261 | 0.099914 | 0.0 |    8.91 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    8.91 |    8.91 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.099914 | 0.505101 | 0.505101 | 0.454241 | 0.454241 | 0.126178 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 100.515194 | 0.146066  0.144668  0.142239  0.139287  0.136331 | 0.126178 |
    | 02/01 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  21.978 |  0.107442 |  0.00859 | 0.0 |  21.978 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  21.978 |  21.978 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.00859 | 0.612423 | 0.612423 | 0.493069 | 0.493069 | 0.136964 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  121.87218 |  0.17654   0.17295  0.166747  0.159351  0.152359 | 0.136964 |
    | 02/01 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 | 18.2952 |  0.107243 | 0.012392 | 0.0 | 18.2952 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 18.2952 | 18.2952 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.012392 | 0.700775 | 0.700775 | 0.569153 | 0.569153 | 0.158098 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 139.454213 | 0.202325  0.199206  0.193548  0.186203  0.178285 | 0.158098 |
    | 02/01 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  7.4844 |  0.105848 | 0.036055 | 0.0 |  7.4844 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.4844 |  7.4844 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.036055 | 0.734513 | 0.734513 | 0.656065 | 0.656065 |  0.18224 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 146.168045 | 0.212649  0.211343  0.208782  0.205019  0.200223 |  0.18224 |
    | 02/01 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  2.2572 |  0.104032 | 0.059768 | 0.0 |  2.2572 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.2572 |  2.2572 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.059768 | 0.741827 | 0.741827 | 0.712201 | 0.712201 | 0.197834 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 147.623649 | 0.215046  0.214724  0.214021  0.212828  0.211023 | 0.197834 |
    | 03/01 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  5.8212 |  0.104235 | 0.041931 | 0.0 |  5.8212 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.8212 |  5.8212 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.041931 | 0.767015 | 0.767015 | 0.736749 | 0.736749 | 0.204652 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 152.635904 | 0.222185  0.221344  0.219887  0.218124  0.216368 | 0.204652 |
    | 03/01 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  3.2076 |  0.104032 |  0.05435 | 0.0 |  3.2076 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.2076 |  3.2076 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.05435 | 0.778946 | 0.778946 | 0.756072 | 0.756072 |  0.21002 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 155.010208 | 0.225769  0.225324  0.224479  0.223302  0.221907 |  0.21002 |
    | 03/01 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |   0.594 |  0.103014 | 0.069893 | 0.0 |   0.594 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.594 |   0.594 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.069893 | 0.777672 | 0.777672 | 0.771557 | 0.771557 | 0.214321 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 154.756644 | 0.225533  0.225545  0.225513  0.225347  0.224957 | 0.214321 |
    | 03/01 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.2376 |  0.103422 | 0.072716 | 0.0 |  0.2376 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.2376 |  0.2376 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.072716 | 0.774608 | 0.774608 | 0.776757 | 0.776757 | 0.215766 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 154.146921 | 0.224666  0.224768  0.224938  0.225123  0.225253 | 0.215766 |
    | 03/01 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |   0.594 |   0.10464 | 0.070996 | 0.0 |   0.594 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.594 |   0.594 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.070996 |  0.77335 |  0.77335 | 0.775797 | 0.775797 | 0.215499 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 153.896575 | 0.224285  0.224332  0.224424  0.224553  0.224705 | 0.215499 |
    | 03/01 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  2.8512 |  0.106248 | 0.057521 | 0.0 |  2.8512 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.8512 |  2.8512 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.057521 | 0.783451 | 0.783451 | 0.774863 | 0.774863 |  0.21524 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 155.906803 | 0.227103  0.226777  0.226234  0.225632  0.225143 |  0.21524 |
    | 03/01 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.4752 |  0.179639 | 0.123337 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.4752 |  0.4752 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.123337 | 0.781293 | 0.781293 | 0.779109 | 0.779109 | 0.216419 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 155.477372 | 0.226593  0.226641  0.226686  0.226656  0.226497 | 0.216419 |
    | 03/01 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.2376 |  0.545628 | 0.383628 | 0.0 |  0.2376 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.2376 |  0.2376 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.383628 | 0.776657 | 0.776657 | 0.781064 | 0.781064 | 0.216962 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 154.554688 | 0.225276  0.225431  0.225696  0.226001  0.226262 | 0.216962 |
    | 03/01 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  1.361522 | 0.980296 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.980296 | 0.767872 | 0.767872 | 0.778096 | 0.778096 | 0.216138 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  152.80652 |  0.22277  0.223067  0.223584  0.224209  0.224812 | 0.216138 |
    | 03/01 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  4.874663 | 3.509758 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.509758 | 0.746484 | 0.746484 | 0.770501 | 0.770501 | 0.214028 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 148.550278 | 0.216692  0.217409  0.218649  0.220133  0.221542 | 0.214028 |
    | 03/01 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.3564 |  5.057269 | 3.513746 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.3564 |  0.3564 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.513746 | 0.726965 | 0.726965 | 0.755017 | 0.755017 | 0.209727 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 144.665968 | 0.211018  0.211702  0.212934  0.214518  0.216201 | 0.209727 |
    | 03/01 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  3.0888 |  6.154673 | 3.253813 | 0.0 |  3.0888 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.0888 |  3.0888 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.253813 | 0.722505 | 0.722505 |  0.73686 |  0.73686 | 0.204683 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  143.77845 | 0.209577  0.209767  0.210168  0.210808  0.211697 | 0.204683 |
    | 03/01 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.8316 |  6.236191 | 4.131769 | 0.0 |  0.8316 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.8316 |  0.8316 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 4.131769 | 0.702391 | 0.702391 |  0.72481 |  0.72481 | 0.201336 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  139.77589 | 0.203892  0.204559  0.205707   0.20707  0.208371 | 0.201336 |
    | 03/01 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.3564 |  5.404341 | 3.754888 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.3564 |  0.3564 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 3.754888 | 0.681887 | 0.681887 | 0.710112 | 0.710112 | 0.197253 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 135.695514 | 0.197954  0.198667  0.199943  0.201561  0.203248 | 0.197253 |
    | 03/01 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.3564 |  9.035929 | 6.278083 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.3564 |  0.3564 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 6.278083 | 0.648869 | 0.648869 | 0.690032 | 0.690032 | 0.191675 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 129.124962 | 0.188502  0.189625  0.191598  0.194027  0.196459 | 0.191675 |
    | 03/01 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  5.627818 | 4.052029 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 4.052029 | 0.625365 | 0.625365 | 0.663219 | 0.663219 | 0.184228 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 124.447569 | 0.181597  0.182437  0.183981  0.186029  0.188312 | 0.184228 |
    | 03/01 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  6.156027 |  4.43234 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  4.43234 | 0.600076 | 0.600076 | 0.636569 | 0.636569 | 0.176825 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 119.415153 | 0.174277  0.175157  0.176736   0.17876  0.180933 | 0.176825 |
    | 03/01 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  3.811802 | 2.744497 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 2.744497 | 0.583353 | 0.583353 |  0.61199 |  0.61199 | 0.169997 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 116.087303 | 0.169345  0.169948  0.171065  0.172571  0.174298 | 0.169997 |
    | 03/01 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  1.747211 | 1.257992 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.257992 | 0.574147 | 0.574147 | 0.592205 | 0.592205 | 0.164501 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 114.255164 | 0.166598  0.166938  0.167581   0.16848  0.169571 | 0.164501 |
    | 03/01 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.318724 | 0.229481 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.229481 | 0.570128 | 0.570128 | 0.579452 | 0.579452 | 0.160959 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 113.455555 |  0.16538  0.165533  0.165831  0.166268  0.166832 | 0.160959 |
    | 03/01 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.101575 | 0.073134 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.073134 | 0.566912 | 0.566912 | 0.572566 | 0.572566 | 0.159046 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 112.815508 | 0.164437  0.164552  0.164764   0.16505  0.165387 | 0.159046 |
    | 03/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.099494 | 0.071636 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.564078 | 0.564078 | 0.568524 | 0.568524 | 0.157923 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 112.251431 | 0.163611  0.163711  0.163893  0.164131  0.164397 | 0.157923 |
    | 03/01 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.097808 | 0.070422 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.561257 | 0.561257 | 0.565404 | 0.565404 | 0.157057 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 111.690174 | 0.162793  0.162892  0.163069  0.163298  0.163545 | 0.157057 |
    | 03/01 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.096529 | 0.069501 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.558451 | 0.558451 | 0.562534 | 0.562534 | 0.156259 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 111.131723 | 0.161979  0.162077  0.162253   0.16248  0.162724 | 0.156259 |
    | 04/01 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.095455 | 0.068728 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.555659 | 0.555659 | 0.559717 | 0.559717 | 0.155477 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 110.576064 | 0.161169  0.161267  0.161442  0.161668   0.16191 | 0.155477 |
    | 04/01 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |   0.09802 | 0.070574 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 |  0.55288 |  0.55288 | 0.556918 | 0.556918 | 0.154699 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 110.023184 | 0.160363   0.16046  0.160635  0.160859    0.1611 | 0.154699 |
    | 04/01 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.550116 | 0.550116 | 0.554133 | 0.554133 | 0.153926 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 109.473068 | 0.159562  0.159658  0.159832  0.160055  0.160295 | 0.153926 |
    | 04/01 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  1.5444 |  0.096957 | 0.059819 | 0.0 |  1.5444 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.5444 |  1.5444 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.555087 | 0.555087 | 0.551914 | 0.551914 | 0.153309 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 110.462381 | 0.160928  0.160772  0.160521  0.160264   0.16009 | 0.153309 |
    | 04/01 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.096744 | 0.069656 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.552312 | 0.552312 |  0.55302 |  0.55302 | 0.153617 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 109.910069 | 0.160196  0.160276  0.160393  0.160489  0.160513 | 0.153617 |
    | 04/01 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 |  0.54955 |  0.54955 |  0.55292 |  0.55292 | 0.153589 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 109.360518 | 0.159397  0.159493  0.159662  0.159871  0.160074 | 0.153589 |
    | 04/01 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.359866 | 0.259103 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.546803 | 0.546803 |  0.55072 |  0.55072 | 0.152978 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 108.813716 | 0.158601  0.158696  0.158869   0.15909  0.159324 | 0.152978 |
    | 04/01 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.8316 |  1.614671 | 1.069795 | 0.0 |  0.8316 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.8316 |  0.8316 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.548227 | 0.548227 | 0.548332 | 0.548332 | 0.152314 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 109.097089 | 0.158973  0.158933  0.158876  0.158839  0.158854 | 0.152314 |
    | 04/01 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.4752 |  3.707216 | 2.545322 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.4752 |  0.4752 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.547861 | 0.547861 | 0.547848 | 0.547848 |  0.15218 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 109.024428 | 0.158883  0.158892  0.158903  0.158908  0.158902 |  0.15218 |
    | 04/01 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.1188 |  6.022315 | 4.284859 | 0.0 |  0.1188 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.1188 |  0.1188 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.545716 | 0.545716 | 0.547776 | 0.547776 |  0.15216 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 108.597512 | 0.158279  0.158349  0.158469  0.158606  0.158726 |  0.15216 |
    | 04/01 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |  0.4752 |  8.299842 | 5.698554 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.4752 |  0.4752 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 5.698554 | 0.516871 | 0.516871 | 0.544594 | 0.544594 | 0.151276 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 102.857287 | 0.150175   0.15112  0.152718  0.154546  0.156146 | 0.151276 |
    | 04/01 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  9.734061 | 7.008524 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 7.008524 | 0.479244 | 0.479244 | 0.526709 | 0.526709 | 0.146308 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  95.369519 | 0.139358  0.140647  0.142924  0.145744  0.148567 | 0.146308 |
    | 04/01 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  9.733892 | 7.008402 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 7.008402 | 0.441806 | 0.441806 | 0.495083 | 0.495083 | 0.137523 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  87.919312 | 0.128502  0.129808  0.132154  0.135157  0.138342 | 0.137523 |
    | 04/01 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 | 12.429676 | 8.949367 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 8.949367 |  0.39485 |  0.39485 | 0.457873 | 0.457873 | 0.127187 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  78.575095 | 0.114978  0.116594  0.119467  0.123078  0.126821 | 0.127187 |
    | 04/01 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 | 10.884737 | 7.837011 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 7.837011 |  0.35369 |  0.35369 | 0.415496 | 0.415496 | 0.115416 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  70.384394 | 0.102988  0.104437  0.107062  0.110471  0.114175 | 0.115416 |
    | 04/01 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  9.119821 | 6.566271 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 6.566271 | 0.319091 | 0.319091 |  0.37312 |  0.37312 | 0.103645 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  63.499032 | 0.092889  0.094112  0.096339  0.099263  0.102501 | 0.103645 |
    | 04/01 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  7.306062 | 5.260364 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 5.260364 | 0.291193 | 0.291193 | 0.335805 | 0.335805 | 0.093279 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  57.947474 | 0.084731   0.08572  0.087529  0.089919  0.092593 | 0.093279 |
    | 04/01 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  3.420163 | 2.462517 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 2.462517 | 0.277425 | 0.277425 | 0.305456 | 0.305456 | 0.084849 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  55.207533 | 0.080597  0.081111  0.082092   0.08348   0.08518 | 0.084849 |
    | 04/01 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  1.518709 | 1.093471 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 1.093471 |  0.27057 |  0.27057 | 0.285512 | 0.285512 | 0.079309 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  53.843492 | 0.078537  0.078794  0.079288  0.079999    0.0809 | 0.079309 |
    | 04/01 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.382278 |  0.27524 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.27524 | 0.267841 | 0.267841 | 0.274752 | 0.274752 |  0.07632 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   53.30041 | 0.077703  0.077809  0.078017  0.078331  0.078748 |  0.07632 |
    | 04/01 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.103829 | 0.074757 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 | 0.074757 | 0.266128 | 0.266128 | 0.269614 | 0.269614 | 0.074893 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  52.959525 | 0.077195  0.077258  0.077376  0.077543  0.077751 | 0.074893 |
    | 04/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.099494 | 0.071636 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.264798 | 0.264798 | 0.267073 | 0.267073 | 0.074187 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  52.694727 | 0.076805  0.076852   0.07694  0.077057  0.077193 | 0.074187 |
    | 04/01 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |   0.09802 | 0.070574 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.263474 | 0.263474 | 0.265448 | 0.265448 | 0.073735 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  52.431254 | 0.076421  0.076467   0.07655  0.076658  0.076776 | 0.073735 |
    | 04/01 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |        1.0 |     0.0 |  0.098442 | 0.070878 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     0.0 |     0.0 | 0.0 | 0.0 | 0.0 |  0.0 |  0.0 | 0.0 |      0.0 | 0.262156 | 0.262156 | 0.264076 | 0.264076 | 0.073354 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  52.169097 | 0.076039  0.076085  0.076167  0.076274  0.076388 | 0.073354 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v2_snow_classes:

snow classes
____________

Besides slight differences in runoff concentration, the following calculation
reproduces the results of the :ref:`hland_v1_snow_classes` experiment of |hland_v1|
exactly:

.. integration-test::

    >>> sclass(2)
    >>> sfdist(linear=0.2)
    >>> zonetype(FIELD)
    >>> t_series = inputs.t.series.copy()
    >>> tn_series = inputs.tn.series.copy()
    >>> inputs.t.series[:48] = -20.0
    >>> inputs.t.series[48:] = 20.0
    >>> inputs.tn.series = inputs.t.series
    >>> test("hland_v2_snow_classes")
    |        date |    p |     t |    tn |       epn | tmean |    tc | fracrain | rfc | sfc |    cfact |                    swe | gact | contriarea |      pc |        ep |      epc |       ei |        tf | spl | wcl | spg | wcg | glmelt |               melt |      refr |       in_ |        r |  sr |       ea |       cf |      inuz |     perc |       q0 |  el |       q1 |     inuh |    outuh |       rt |       qt |       ic |                     sp |                   wc |         sm |        uz |        lz |                                               sc |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -20.0 | -20.0 |  0.100707 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.24976 |     0.0 |  0.106749 |  0.07686 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.048037 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 |     0.05 |     0.05 | 0.163676 | 0.163676 | 0.045466 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.951963 |       0.0 |      9.95 | 0.015696  0.019685  0.026398  0.034007  0.040538 | 0.045466 |
    | 01/01 01:00 |  0.0 | -20.0 | -20.0 |  0.097801 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.249527 |     0.0 |  0.103669 | 0.074642 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.046629 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 |  0.04975 |  0.04975 | 0.102365 | 0.102365 | 0.028435 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.905334 |       0.0 |   9.90025 |  0.01447  0.014747  0.015662  0.017698  0.021131 | 0.028435 |
    | 01/01 02:00 |  0.0 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.249296 |     0.0 |    0.1028 | 0.074016 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.046216 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.049501 | 0.049501 | 0.059979 | 0.059979 | 0.016661 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.859118 |       0.0 |  9.850749 | 0.014359  0.014381  0.014465  0.014714  0.015311 | 0.016661 |
    | 01/01 03:00 |  0.0 | -20.0 | -20.0 |   0.09599 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.249068 |     0.0 |  0.101749 |  0.07326 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.045723 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.049254 | 0.049254 | 0.050799 | 0.050799 | 0.014111 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.813395 |       0.0 |  9.801495 | 0.014286  0.014295  0.014315  0.014353  0.014436 | 0.014111 |
    | 01/01 04:00 |  0.0 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.248837 |     0.0 |    0.1028 | 0.074016 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.046174 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.049007 | 0.049007 |  0.04947 |  0.04947 | 0.013742 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.767222 |       0.0 |  9.752488 | 0.014215  0.014223  0.014239   0.01426  0.014286 | 0.013742 |
    | 01/01 05:00 |  0.0 | -20.0 | -20.0 |  0.102761 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.248594 |     0.0 |  0.108927 | 0.078427 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.048903 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.048762 | 0.048762 | 0.049126 | 0.049126 | 0.013646 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.718319 |       0.0 |  9.703725 | 0.014144  0.014152  0.014168  0.014187  0.014209 | 0.013646 |
    | 01/01 06:00 |  0.0 | -20.0 | -20.0 |  0.291908 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.247902 |     0.0 |  0.309422 | 0.222784 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.138848 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.048519 | 0.048519 | 0.048873 | 0.048873 | 0.013576 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.579471 |       0.0 |  9.655206 | 0.014073  0.014081  0.014097  0.014116  0.014138 | 0.013576 |
    | 01/01 07:00 |  0.0 | -20.0 | -20.0 |  1.932875 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.243352 |     0.0 |  2.048847 |  1.47517 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.918104 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.048276 | 0.048276 | 0.048629 | 0.048629 | 0.013508 |      0.0 |        0.0         0.0 |       0.0        0.0 |  98.661367 |       0.0 |   9.60693 | 0.014003  0.014011  0.014026  0.014046  0.014067 | 0.013508 |
    | 01/01 08:00 |  0.0 | -20.0 | -20.0 |  4.369536 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.233313 |     0.0 |  4.631708 |  3.33483 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.056368 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.048035 | 0.048035 | 0.048385 | 0.048385 |  0.01344 |      0.0 |        0.0         0.0 |       0.0        0.0 |  96.604999 |       0.0 |  9.558896 | 0.013932  0.013941  0.013956  0.013976  0.013997 |  0.01344 |
    | 01/01 09:00 |  0.0 | -20.0 | -20.0 |  7.317556 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.21731 |     0.0 |  7.756609 | 5.584759 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 3.371973 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.047794 | 0.047794 | 0.048143 | 0.048143 | 0.013373 |      0.0 |        0.0         0.0 |       0.0        0.0 |  93.233026 |       0.0 |  9.511101 | 0.013863  0.013871  0.013886  0.013906  0.013927 | 0.013373 |
    | 01/01 10:00 |  0.0 | -20.0 | -20.0 |  8.264362 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.200514 |     0.0 |  8.760224 | 6.307361 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |  3.67534 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.047556 | 0.047556 | 0.047903 | 0.047903 | 0.013306 |      0.0 |        0.0         0.0 |       0.0        0.0 |  89.557686 |       0.0 |  9.463546 | 0.013794  0.013802  0.013817  0.013836  0.013857 | 0.013306 |
    | 01/01 11:00 |  0.0 | -20.0 | -20.0 |  9.369867 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.182991 |     0.0 |  9.932059 | 7.151082 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 4.002715 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.047318 | 0.047318 | 0.047663 | 0.047663 |  0.01324 |      0.0 |        0.0         0.0 |       0.0        0.0 |  85.554971 |       0.0 |  9.416228 | 0.013725  0.013733  0.013748  0.013767  0.013788 |  0.01324 |
    | 01/01 12:00 |  0.0 | -20.0 | -20.0 |  5.126178 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.174152 |     0.0 |  5.433749 | 3.912299 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.091979 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.047081 | 0.047081 | 0.047425 | 0.047425 | 0.013174 |      0.0 |        0.0         0.0 |       0.0        0.0 |  83.462992 |       0.0 |  9.369147 | 0.013656  0.013664  0.013679  0.013698  0.013719 | 0.013174 |
    | 01/01 13:00 |  0.0 | -20.0 | -20.0 |   6.62503 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.163319 |     0.0 |  7.022532 | 5.056223 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.637547 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.046846 | 0.046846 | 0.047188 | 0.047188 | 0.013108 |      0.0 |        0.0         0.0 |       0.0        0.0 |  80.825446 |       0.0 |  9.322301 | 0.013588  0.013596  0.013611   0.01363   0.01365 | 0.013108 |
    | 01/01 14:00 |  0.0 | -20.0 | -20.0 |  7.397619 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.151996 |     0.0 |  7.841476 | 5.645863 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.852059 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.046612 | 0.046612 | 0.046952 | 0.046952 | 0.013042 |      0.0 |        0.0         0.0 |       0.0        0.0 |  77.973387 |       0.0 |   9.27569 |  0.01352  0.013528  0.013543  0.013562  0.013582 | 0.013042 |
    | 01/01 15:00 |  0.2 | -20.0 | -20.0 |   2.39151 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.148643 |  0.2808 |  2.535001 | 1.774662 |   0.2808 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.864852 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.046378 | 0.046378 | 0.046717 | 0.046717 | 0.012977 |      0.0 |        0.0         0.0 |       0.0        0.0 |  77.108535 |       0.0 |  9.229311 | 0.013452   0.01346  0.013475  0.013494  0.013514 | 0.012977 |
    | 01/01 16:00 |  0.0 | -20.0 | -20.0 |  1.829834 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.14606 |     0.0 |  1.939624 | 1.396529 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.673027 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.046147 | 0.046147 | 0.046484 | 0.046484 | 0.012912 |      0.0 |        0.0         0.0 |       0.0        0.0 |  76.435507 |       0.0 |  9.183165 | 0.013385  0.013393  0.013408  0.013426  0.013446 | 0.012912 |
    | 01/01 17:00 |  0.0 | -20.0 | -20.0 |  1.136569 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.14448 |     0.0 |  1.204763 | 0.867429 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |  0.41439 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.045916 | 0.045916 | 0.046251 | 0.046251 | 0.012848 |      0.0 |        0.0         0.0 |       0.0        0.0 |  76.021117 |       0.0 |  9.137249 | 0.013318  0.013326   0.01334  0.013359  0.013379 | 0.012848 |
    | 01/01 18:00 |  1.3 | -20.0 | -20.0 |  0.750986 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |   0.144049 |  1.8252 |  0.796045 | 0.477532 | 0.477532 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.113445 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.045686 | 0.045686 |  0.04602 |  0.04602 | 0.012783 | 1.347668 |        0.0         0.0 |       0.0        0.0 |  75.907672 |       0.0 |  9.091563 | 0.013251  0.013259  0.013274  0.013292  0.013312 | 0.012783 |
    | 01/01 19:00 |  5.6 | -20.0 | -20.0 |  0.223895 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   5.768054    8.652081 |  0.0 |   0.144049 |  7.8624 |  0.237329 | 0.077844 | 0.077844 |  7.210068 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.045458 | 0.045458 |  0.04579 |  0.04579 | 0.012719 | 1.922156 |   5.768054    8.652081 |       0.0        0.0 |  75.907672 |       0.0 |  9.046105 | 0.013185  0.013193  0.013207  0.013226  0.013246 | 0.012719 |
    | 01/01 20:00 |  2.9 | -20.0 | -20.0 |  0.099425 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   8.963059   13.444589 |  0.0 |   0.144049 |  4.0716 |   0.10539 | 0.050502 | 0.050502 |  3.993756 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.045231 | 0.045231 | 0.045561 | 0.045561 | 0.012656 | 1.949498 |   8.963059   13.444589 |       0.0        0.0 |  75.907672 |       0.0 |  9.000874 | 0.013119  0.013127  0.013141   0.01316  0.013179 | 0.012656 |
    | 01/01 21:00 |  4.9 | -20.0 | -20.0 |  0.098454 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  14.426338   21.639507 |  0.0 |   0.144049 |  6.8796 |  0.104361 | 0.037765 | 0.037765 |  6.829098 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.045004 | 0.045004 | 0.045333 | 0.045333 | 0.012593 | 1.962235 |  14.426338   21.639507 |       0.0        0.0 |  75.907672 |       0.0 |   8.95587 | 0.013054  0.013061  0.013076  0.013094  0.013114 | 0.012593 |
    | 01/01 22:00 | 10.6 | -20.0 | -20.0 |  0.098128 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  26.302045   39.453068 |  0.0 |   0.144049 | 14.8824 |  0.104016 | 0.016908 | 0.016908 | 14.844635 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.044779 | 0.044779 | 0.045106 | 0.045106 |  0.01253 | 1.983092 |  26.302045   39.453068 |       0.0        0.0 |  75.907672 |       0.0 |  8.911091 | 0.012988  0.012996   0.01301  0.013028  0.013048 |  0.01253 |
    | 01/01 23:00 |  0.1 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  26.400839   39.601258 |  0.0 |   0.144049 |  0.1404 |  0.103322 | 0.073355 | 0.073355 |  0.123492 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.044555 | 0.044555 | 0.044881 | 0.044881 | 0.012467 | 1.926645 |  26.400839   39.601258 |       0.0        0.0 |  75.907672 |       0.0 |  8.866535 | 0.012923  0.012931  0.012945  0.012963  0.012983 | 0.012467 |
    | 02/01 00:00 |  0.7 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  27.128395   40.692592 |  0.0 |   0.144049 |  0.9828 |  0.103322 | 0.067429 | 0.067429 |  0.909445 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.044333 | 0.044333 | 0.044656 | 0.044656 | 0.012405 | 1.932571 |  27.128395   40.692592 |       0.0        0.0 |  75.907672 |       0.0 |  8.822202 | 0.012859  0.012867  0.012881  0.012899  0.012918 | 0.012405 |
    | 02/01 01:00 |  3.0 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  30.444052   45.666078 |  0.0 |   0.144049 |   4.212 |  0.103322 | 0.048821 | 0.048821 |  4.144571 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.044111 | 0.044111 | 0.044433 | 0.044433 | 0.012343 | 1.951179 |  30.444052   45.666078 |       0.0        0.0 |  75.907672 |       0.0 |  8.778091 | 0.012794  0.012802  0.012816  0.012834  0.012853 | 0.012343 |
    | 02/01 02:00 |  2.1 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  32.763716   49.145573 |  0.0 |   0.144049 |  2.9484 |    0.1028 | 0.055116 | 0.055116 |  2.899579 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 |  0.04389 |  0.04389 | 0.044211 | 0.044211 | 0.012281 | 1.944884 |  32.763716   49.145573 |       0.0        0.0 |  75.907672 |       0.0 |  8.734201 |  0.01273  0.012738  0.012752   0.01277  0.012789 | 0.012281 |
    | 02/01 03:00 | 10.4 | -20.0 | -20.0 |  0.096652 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  44.400903   66.601354 |  0.0 |   0.144049 | 14.6016 |  0.102451 | 0.017128 | 0.017128 | 14.546484 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.043671 | 0.043671 |  0.04399 |  0.04399 | 0.012219 | 1.982872 |  44.400903   66.601354 |       0.0        0.0 |  75.907672 |       0.0 |   8.69053 | 0.012667  0.012674  0.012688  0.012706  0.012725 | 0.012219 |
    | 02/01 04:00 |  3.5 | -20.0 | -20.0 |  0.096321 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    48.3184     72.4776 |  0.0 |   0.144049 |   4.914 |    0.1021 | 0.044973 | 0.044973 |  4.896872 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.043453 | 0.043453 |  0.04377 |  0.04377 | 0.012158 | 1.955027 |    48.3184     72.4776 |       0.0        0.0 |  75.907672 |       0.0 |  8.647077 | 0.012603  0.012611  0.012625  0.012642  0.012661 | 0.012158 |
    | 02/01 05:00 |  3.4 | -20.0 | -20.0 |   0.09599 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  52.101302   78.151953 |  0.0 |   0.144049 |  4.7736 |  0.101749 | 0.045452 | 0.045452 |  4.728627 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.043235 | 0.043235 | 0.043551 | 0.043551 | 0.012098 | 1.954548 |  52.101302   78.151953 |       0.0        0.0 |  75.907672 |       0.0 |  8.603842 |  0.01254  0.012548  0.012562  0.012579  0.012598 | 0.012098 |
    | 02/01 06:00 |  1.2 | -20.0 | -20.0 |  0.187298 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.412781   80.119171 |  0.0 |   0.144049 |  1.6848 |  0.198536 | 0.120782 | 0.120782 |  1.639348 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.043019 | 0.043019 | 0.043333 | 0.043333 | 0.012037 | 1.879218 |  53.412781   80.119171 |       0.0        0.0 |  75.907672 |       0.0 |  8.560823 | 0.012478  0.012485  0.012499  0.012516  0.012535 | 0.012037 |
    | 02/01 07:00 |  0.1 | -20.0 | -20.0 |  1.264612 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.428476   80.142713 |  0.0 |   0.144049 |  0.1404 |  1.340489 | 0.951696 | 0.951696 |  0.019618 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.042804 | 0.042804 | 0.043117 | 0.043117 | 0.011977 | 1.048304 |  53.428476   80.142713 |       0.0        0.0 |  75.907672 |       0.0 |  8.518019 | 0.012415  0.012423  0.012436  0.012454  0.012472 | 0.011977 |
    | 02/01 08:00 |  0.0 | -20.0 | -20.0 |  3.045538 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.428476   80.142713 |  0.0 |   0.144049 |     0.0 |   3.22827 | 2.324355 | 1.048304 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 |  0.04259 |  0.04259 | 0.042901 | 0.042901 | 0.011917 |      0.0 |  53.428476   80.142713 |       0.0        0.0 |  75.907672 |       0.0 |  8.475429 | 0.012353  0.012361  0.012374  0.012392   0.01241 | 0.011917 |
    | 02/01 09:00 |  0.0 | -20.0 | -20.0 |  1.930758 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.428476   80.142713 |  0.0 |   0.144049 |     0.0 |  2.046603 | 1.473555 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.042377 | 0.042377 | 0.042687 | 0.042687 | 0.011857 |      0.0 |  53.428476   80.142713 |       0.0        0.0 |  75.907672 |       0.0 |  8.433051 | 0.012292  0.012299  0.012312   0.01233  0.012348 | 0.011857 |
    | 02/01 10:00 |  0.4 | -20.0 | -20.0 |  2.461001 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.428476   80.142713 |  0.0 |   0.144049 |  0.5616 |  2.608661 | 1.775661 |   0.5616 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.042165 | 0.042165 | 0.042473 | 0.042473 | 0.011798 |      0.0 |  53.428476   80.142713 |       0.0        0.0 |  75.907672 |       0.0 |  8.390886 |  0.01223  0.012237  0.012251  0.012268  0.012286 | 0.011798 |
    | 02/01 11:00 |  0.1 | -20.0 | -20.0 |  6.215945 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.428476   80.142713 |  0.0 |   0.144049 |  0.1404 |  6.588902 | 4.677869 |   0.1404 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.041954 | 0.041954 | 0.042261 | 0.042261 | 0.011739 |      0.0 |  53.428476   80.142713 |       0.0        0.0 |  75.907672 |       0.0 |  8.348932 | 0.012169  0.012176   0.01219  0.012207  0.012225 | 0.011739 |
    | 02/01 12:00 |  3.6 | -20.0 | -20.0 |  3.374783 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  55.871996   83.807993 |  0.0 |   0.144049 |  5.0544 |   3.57727 | 1.553726 | 1.553726 |    3.0544 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.041745 | 0.041745 | 0.042049 | 0.042049 |  0.01168 | 0.446274 |  55.871996   83.807993 |       0.0        0.0 |  75.907672 |       0.0 |  8.307187 | 0.012108  0.012115  0.012129  0.012146  0.012164 |  0.01168 |
    | 02/01 13:00 |  5.9 | -20.0 | -20.0 |  8.821555 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  61.255895   91.883842 |  0.0 |   0.144049 |  8.2836 |  9.350848 | 2.940569 |      2.0 |  6.729874 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.041536 | 0.041536 | 0.041839 | 0.041839 | 0.011622 |      0.0 |  61.255895   91.883842 |       0.0        0.0 |  75.907672 |       0.0 |  8.265651 | 0.012048  0.012055  0.012068  0.012085  0.012103 | 0.011622 |
    | 02/01 14:00 |  1.1 | -20.0 | -20.0 |  4.046025 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  61.255895   91.883842 |  0.0 |   0.144049 |  1.5444 |  4.288787 | 2.646028 |   1.5444 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.041328 | 0.041328 |  0.04163 |  0.04163 | 0.011564 |      0.0 |  61.255895   91.883842 |       0.0        0.0 |  75.907672 |       0.0 |  8.224323 | 0.011987  0.011995  0.012008  0.012024  0.012042 | 0.011564 |
    | 02/01 15:00 | 20.7 | -20.0 | -20.0 |  2.110757 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  82.906135  124.359202 |  0.0 |   0.144049 | 29.0628 |  2.237402 | 0.088084 | 0.088084 |   27.0628 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.041122 | 0.041122 | 0.041422 | 0.041422 | 0.011506 | 1.911916 |  82.906135  124.359202 |       0.0        0.0 |  75.907672 |       0.0 |  8.183201 | 0.011927  0.011935  0.011948  0.011964  0.011982 | 0.011506 |
    | 02/01 16:00 | 37.9 | -20.0 | -20.0 |  2.239257 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 125.404948  188.107422 |  0.0 |   0.144049 | 53.2116 |  2.373612 | 0.008352 | 0.008352 | 53.123516 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.040916 | 0.040916 | 0.041215 | 0.041215 | 0.011449 | 1.991648 | 125.404948  188.107422 |       0.0        0.0 |  75.907672 |       0.0 |  8.142285 | 0.011868  0.011875  0.011888  0.011904  0.011922 | 0.011449 |
    | 02/01 17:00 |  8.2 | -20.0 | -20.0 |  2.877848 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 134.608506  201.912759 |  0.0 |   0.144049 | 11.5128 |  3.050519 | 0.694563 | 0.694563 | 11.504448 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.040711 | 0.040711 | 0.041009 | 0.041009 | 0.011391 | 1.305437 | 134.608506  201.912759 |       0.0        0.0 |  75.907672 |       0.0 |  8.101574 | 0.011808  0.011816  0.011828  0.011845  0.011863 | 0.011391 |
    | 02/01 18:00 |  3.6 | -20.0 | -20.0 |  1.591452 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 138.096376  207.144564 |  0.0 |   0.144049 |  5.0544 |  1.686939 | 0.732693 | 0.732693 |  4.359837 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.040508 | 0.040508 | 0.040804 | 0.040804 | 0.011334 | 1.267307 | 138.096376  207.144564 |       0.0        0.0 |  75.907672 |       0.0 |  8.061066 | 0.011749  0.011756  0.011769  0.011786  0.011803 | 0.011334 |
    | 02/01 19:00 |  7.5 | -20.0 | -20.0 |  0.291604 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 145.934221  218.901332 |  0.0 |   0.144049 |   10.53 |    0.3091 | 0.077646 | 0.077646 |  9.797307 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.040305 | 0.040305 |   0.0406 |   0.0406 | 0.011278 | 1.922354 | 145.934221  218.901332 |       0.0        0.0 |  75.907672 |       0.0 |  8.020761 | 0.011691  0.011698   0.01171  0.011727  0.011744 | 0.011278 |
    | 02/01 20:00 | 18.5 | -20.0 | -20.0 |  0.092622 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 166.651304  249.976957 |  0.0 |   0.144049 |  25.974 |  0.098179 | 0.005264 | 0.005264 | 25.896354 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.040104 | 0.040104 | 0.040397 | 0.040397 | 0.011221 | 1.994736 | 166.651304  249.976957 |       0.0        0.0 |  75.907672 |       0.0 |  7.980657 | 0.011632  0.011639  0.011652  0.011668  0.011686 | 0.011221 |
    | 02/01 21:00 | 15.4 | -20.0 | -20.0 |  0.092451 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 183.944373   275.91656 |  0.0 |   0.144049 | 21.6216 |  0.097998 |  0.00812 |  0.00812 | 21.616336 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.039903 | 0.039903 | 0.040195 | 0.040195 | 0.011165 |  1.99188 | 183.944373   275.91656 |       0.0        0.0 |  75.907672 |       0.0 |  7.940753 | 0.011574  0.011581  0.011594   0.01161  0.011627 | 0.011165 |
    | 02/01 22:00 |  6.3 | -20.0 | -20.0 |  0.091248 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 191.014038  286.521056 |  0.0 |   0.144049 |  8.8452 |  0.096723 | 0.028755 | 0.028755 |   8.83708 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.039704 | 0.039704 | 0.039994 | 0.039994 | 0.011109 | 1.971245 | 191.014038  286.521056 |       0.0        0.0 |  75.907672 |       0.0 |   7.90105 | 0.011516  0.011523  0.011536  0.011552  0.011569 | 0.011109 |
    | 02/01 23:00 |  1.9 | -20.0 | -20.0 |  0.089683 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 193.125113   289.68767 |  0.0 |   0.144049 |  2.6676 |  0.095064 |  0.05242 |  0.05242 |  2.638845 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.039505 | 0.039505 | 0.039794 | 0.039794 | 0.011054 |  1.94758 | 193.125113   289.68767 |       0.0        0.0 |  75.907672 |       0.0 |  7.861544 | 0.011459  0.011465  0.011478  0.011494  0.011511 | 0.011054 |
    | 03/01 00:00 |  4.9 |  20.0 |  20.0 |  0.089858 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 197.740137  296.610206 |  0.0 |   0.144049 |  5.8212 |  0.095249 | 0.038316 | 0.038316 |   5.76878 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.039308 | 0.039308 | 0.039595 | 0.039595 | 0.010999 | 1.961684 |  184.27879  280.841347 | 13.461347  15.768859 |  75.907672 |       0.0 |  7.822237 | 0.011401  0.011408  0.011421  0.011437  0.011454 | 0.010999 |
    | 03/01 01:00 |  2.7 |  20.0 |  20.0 |  0.089683 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 200.275564  300.413346 |  0.0 |   0.144049 |  3.2076 |  0.095064 | 0.049664 | 0.049664 |  3.169284 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.039111 | 0.039111 | 0.039397 | 0.039397 | 0.010944 | 1.950336 | 175.432468  271.995024 | 24.843097  28.418322 |  75.907672 |       0.0 |  7.783126 | 0.011344  0.011351  0.011363  0.011379  0.011396 | 0.010944 |
    | 03/01 02:00 |  0.5 |  20.0 |  20.0 |  0.088805 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 200.711033  301.066549 |  0.0 |   0.144049 |   0.594 |  0.094133 | 0.063867 | 0.063867 |  0.544336 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.038916 | 0.038916 |   0.0392 |   0.0392 | 0.010889 | 1.936133 | 166.586145  263.148701 | 34.124888  37.917848 |  75.907672 |       0.0 |   7.74421 | 0.011288  0.011294  0.011307  0.011322  0.011339 | 0.010889 |
    | 03/01 03:00 |  0.2 |  20.0 |  20.0 |  0.089157 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 200.850019  301.275028 |  0.0 |   0.144049 |  0.2376 |  0.094506 | 0.066447 | 0.066447 |  0.173733 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.038721 | 0.038721 | 0.039004 | 0.039004 | 0.010834 | 1.933553 | 157.739822  254.302379 | 43.110197   46.97265 |  75.907672 |       0.0 |  7.705489 | 0.011231  0.011238   0.01125  0.011266  0.011283 | 0.010834 |
    | 03/01 04:00 |  0.5 |  20.0 |  20.0 |  0.090207 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 201.272061  301.908092 |  0.0 |   0.144049 |   0.594 |  0.095619 | 0.064876 | 0.064876 |  0.527553 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |       0.0 |      0.0 |      0.0 | 0.0 | 0.038527 | 0.038527 | 0.038809 | 0.038809 |  0.01078 | 1.935124 | 148.893499  245.456056 | 52.378562  56.452036 |  75.907672 |       0.0 |  7.666961 | 0.011175  0.011182  0.011194  0.011209  0.011226 |  0.01078 |
    | 03/01 05:00 |  2.4 |  20.0 |  20.0 |  0.091593 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 196.066047  305.251681 |  0.0 |   0.156619 |  2.8512 |  0.097089 | 0.052562 | 0.052562 |  2.786324 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  3.717537 | 0.535509 | 0.0 |      0.0 | 0.060455 |  0.475054 | 0.078309 | 0.004075 | 0.0 | 0.038726 | 0.042801 | 0.038934 | 0.038934 | 0.010815 | 1.947438 | 140.047176  236.609733 |  56.01887  68.641949 |  79.150155 |   0.39267 |  7.706544 | 0.012371  0.012232  0.011999  0.011737  0.011515 | 0.010815 |
    | 03/01 06:00 |  0.4 |  20.0 |  20.0 |  0.154861 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 183.681195  305.758847 |  0.0 |   0.178815 |  0.4752 |  0.164153 | 0.112705 | 0.112705 |  0.422638 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.361481 | 0.996327 | 0.0 |      0.0 | 0.057742 |  0.938584 | 0.089408 | 0.105621 | 0.0 |  0.03898 | 0.144601 | 0.048257 | 0.048257 | 0.013405 | 1.887295 | 131.200853   227.76341 | 52.480341  77.995437 |  84.573052 |  1.136225 |  7.756972 | 0.040938  0.037612  0.031998    0.0256  0.020051 | 0.013405 |
    | 03/01 07:00 |  0.2 |  20.0 |  20.0 |  0.470369 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 171.296343  305.908721 |  0.0 |   0.201396 |  0.2376 |  0.498591 | 0.350557 | 0.350557 |  0.124895 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.242384 | 1.116232 | 0.0 |      0.0 |  0.05515 |  1.061082 | 0.100698 | 0.396479 | 0.0 | 0.039288 | 0.435768 |  0.12147 |  0.12147 | 0.033742 | 1.649443 | 122.354531  218.917087 | 48.941812  86.991634 |  89.754354 |   1.70013 |  7.818382 | 0.123493  0.113783  0.097063  0.077277  0.058879 | 0.033742 |
    | 03/01 08:00 |  0.0 |  20.0 |  20.0 |  1.173726 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 158.911491   294.09907 |  0.0 |   0.247335 |     0.0 |   1.24415 | 0.895788 | 0.895788 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.097251 | 2.436339 | 0.0 |      0.0 | 0.050292 |  2.386047 | 0.123667 |  1.03346 | 0.0 |  0.03971 |  1.07317 | 0.348196 | 0.348196 | 0.096721 | 0.753656 | 113.508208  210.070764 | 45.403283  84.028306 |  99.465559 |  2.929049 |  7.902339 | 0.304892  0.283475  0.246319  0.201702  0.159081 | 0.096721 |
    | 03/01 09:00 |  0.0 |  20.0 |  20.0 |  4.202296 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 146.526639  281.714218 |  0.0 |   0.296115 |     0.0 |  4.454434 | 3.207192 | 0.753656 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 3.063207 | 0.0 |      0.0 | 0.045606 |    3.0176 | 0.148057 | 1.829676 | 0.0 | 0.040252 | 1.869928 | 0.832176 | 0.832176 |  0.23116 |      0.0 | 104.661885  201.224442 | 41.864754  80.489777 | 108.832811 |  3.968916 |  8.010144 | 0.534278  0.506887  0.458336  0.397729  0.335992 |  0.23116 |
    | 03/01 10:00 |  0.3 |  20.0 |  20.0 |  4.359715 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 134.141787  269.329366 |  0.0 |   0.345694 |  0.3564 |  4.621298 | 3.210837 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 3.667334 | 0.0 |      0.0 | 0.041225 |   3.62611 | 0.172847 | 2.436608 | 0.0 | 0.040915 | 2.477523 | 1.516639 | 1.516639 | 0.421288 |      0.0 |  95.815562  192.378119 | 38.326225  76.951247 | 117.591553 |   4.98557 |  8.142077 | 0.712273  0.690656  0.651107   0.59895   0.54112 | 0.421288 |
    | 03/01 11:00 |  2.6 |  20.0 |  20.0 |  5.305753 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 121.756935  256.944514 |  0.0 |   0.399704 |  3.0888 |  5.624098 | 2.973312 |      2.0 |    1.0888 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 13.473652 | 4.657765 | 0.0 |      0.0 | 0.036796 |  4.620969 | 0.199852 | 3.137838 | 0.0 |  0.04171 | 3.179547 | 2.195192 | 2.195192 | 0.609775 |      0.0 |  86.969239  183.531796 | 34.787696  73.412718 | 126.444236 |  6.268849 |  8.300219 | 0.914998    0.8907  0.847263  0.792009  0.733493 | 0.609775 |
    | 03/01 12:00 |  0.7 |  20.0 |  20.0 |  5.376027 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 109.372083  244.559662 |  0.0 |    0.44831 |  0.8316 |  5.698589 | 3.775582 |   0.8316 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 |  4.95027 | 0.0 |      0.0 | 0.033061 |   4.91721 | 0.224155 | 3.709554 | 0.0 | 0.042622 | 3.752176 | 2.861386 | 2.861386 | 0.794829 |      0.0 |  78.122916  174.685473 | 31.249167  69.874189 | 133.911878 |   7.25235 |  8.481752 | 1.082296  1.062026  1.025091  0.976646  0.923193 | 0.794829 |
    | 03/01 13:00 |  0.3 |  20.0 |  20.0 |  4.658915 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  96.987231   232.17481 |  0.0 |   0.495434 |  0.3564 |   4.93845 | 3.431191 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 |  5.55225 | 0.0 |      0.0 | 0.029628 |  5.522622 | 0.247717 | 4.192861 | 0.0 | 0.043647 | 4.236508 | 3.476939 | 3.476939 | 0.965816 |      0.0 |  69.276594   165.83915 | 27.710637   66.33566 | 140.774108 |  8.334395 |  8.685821 | 1.223657  1.206539  1.175351  1.134381  1.088895 | 0.965816 |
    | 03/01 14:00 |  0.3 |  20.0 |  20.0 |  7.789594 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  84.602379  219.789958 |  0.0 |   0.540589 |  0.3564 |   8.25697 | 5.736869 |   0.3564 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 6.135873 | 0.0 |      0.0 | 0.026488 |  6.109385 | 0.270295 | 4.743723 | 0.0 | 0.044781 | 4.788503 | 4.013836 | 4.013836 | 1.114954 |      0.0 |  60.430271  156.992827 | 24.172108  62.797131 | 147.049575 |  9.429763 |  8.911335 | 1.383105   1.36399   1.32981  1.286322  1.240263 | 1.114954 |
    | 03/01 15:00 |  0.0 |  20.0 |  20.0 |  4.851567 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  72.217527  207.405106 |  0.0 |   0.583413 |     0.0 |  5.142661 | 3.702716 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 |  6.69512 | 0.0 |      0.0 |  0.02363 |   6.67149 | 0.291706 | 5.300698 | 0.0 | 0.046015 | 5.346713 | 4.545821 | 4.545821 | 1.262728 |      0.0 |  51.583948  148.146505 | 20.633579  59.258602 | 152.762937 | 10.508848 |  9.157027 | 1.544903  1.525448  1.490483  1.445675  1.397872 | 1.262728 |
    | 03/01 16:00 |  0.0 |  20.0 |  20.0 |   5.30692 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  59.832675  195.020254 |  0.0 |   0.623653 |     0.0 |  5.625335 | 4.050241 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 7.225482 | 0.0 |      0.0 | 0.021039 |  7.204443 | 0.311826 | 5.846186 | 0.0 | 0.047344 |  5.89353 | 5.095916 | 5.095916 | 1.415532 |      0.0 |  42.737625  139.300182 |  17.09505  55.720073 | 157.943346 | 11.555279 |  9.421509 | 1.703589  1.684484  1.650063  1.605749   1.55811 | 1.415532 |
    | 03/01 17:00 |  0.0 |  20.0 |  20.0 |  3.286036 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  47.447823  182.635402 |  0.0 |   0.661156 |     0.0 |  3.483198 | 2.507903 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 7.723844 | 0.0 |      0.0 | 0.018698 |  7.705146 | 0.330578 | 6.372206 | 0.0 |  0.04876 | 6.420967 | 5.645136 | 5.645136 | 1.568093 |      0.0 |  33.891302  130.453859 | 13.556521  52.181544 | 162.623052 | 12.557641 |  9.703326 | 1.856739  1.838289   1.80501  1.762067  1.715722 | 1.568093 |
    | 03/01 18:00 |  0.0 |  20.0 |  20.0 |  1.506216 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  35.062971   170.25055 |  0.0 |   0.695858 |     0.0 |  1.596589 | 1.149544 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 8.188324 | 0.0 |      0.0 |  0.01659 |  8.171734 | 0.347929 | 6.873745 | 0.0 | 0.050256 | 6.924001 | 6.179956 | 6.179956 | 1.716654 |      0.0 |  25.044979  121.607536 | 10.017992  48.643014 | 166.836169 | 13.507701 | 10.000999 | 2.002864  1.985253   1.95346  1.912374   1.86792 | 1.716654 |
    | 03/01 19:00 |  0.0 |  20.0 |  20.0 |  0.274762 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  22.678119  157.865698 |  0.0 |   0.727759 |     0.0 |  0.291248 | 0.209698 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 8.618094 | 0.0 |      0.0 | 0.014699 |  8.603396 |  0.36388 | 7.347361 | 0.0 | 0.051824 | 7.399186 | 6.693151 | 6.693151 | 1.859209 |      0.0 |  16.198657  112.761213 |  6.479463  45.104485 | 170.617625 | 14.399856 | 10.313054 | 2.140948    2.1243  2.094226  2.055311  2.013121 | 1.859209 |
    | 03/01 20:00 |  0.0 |  20.0 |  20.0 |  0.087565 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  10.293267  145.480847 |  0.0 |    0.75692 |     0.0 |  0.092819 |  0.06683 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 9.013192 | 0.0 |      0.0 | 0.013005 |  9.000186 |  0.37846 | 7.790813 | 0.0 | 0.053458 | 7.844271 | 7.180323 | 7.180323 | 1.994534 |      0.0 |   7.352334   103.91489 |  2.940934  41.565956 | 174.002291 |  15.23077 | 10.638057 | 2.270326  2.254723  2.226518  2.189983  2.150302 | 1.994534 |
    | 03/01 21:00 |  0.0 |  20.0 |  20.0 |  0.085771 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  133.095995 |  0.0 |   0.780903 |     0.0 |  0.090917 |  0.06546 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 7.352334  8.846323 | 0.0   0.0 |  11.33906 |  8.58276 | 0.0 |  0.03273 | 0.011621 |   8.57114 | 0.390452 |  7.84807 | 0.0 | 0.055143 | 7.903212 | 7.613287 | 7.613287 | 2.114802 |      0.0 |        0.0   95.068568 |       0.0  38.027427 | 176.737481 | 15.563388 | 10.973366 | 2.291203   2.28825  2.281461  2.269479  2.251385 | 2.114802 |
    | 03/01 22:00 |  0.0 |  20.0 |  20.0 |  0.084317 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  120.711143 |  0.0 |    0.79275 |     0.0 |  0.089376 | 0.064351 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.846323 | 0.0   0.0 |  6.192426 | 4.835687 | 0.0 | 0.032175 | 0.010953 |  4.824734 | 0.396375 |   6.1996 | 0.0 | 0.056849 | 6.256448 | 7.718499 | 7.718499 | 2.144028 |      0.0 |        0.0   86.222245 |       0.0  34.488898 | 178.072998 | 13.792147 | 11.312892 | 1.830441  1.883911  1.973526  2.074001  2.157849 | 2.144028 |
    | 03/01 23:00 |  0.0 |  20.0 |  20.0 |  0.083215 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  108.326291 |  0.0 |   0.804026 |     0.0 |  0.088208 |  0.06351 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.846323 | 0.0   0.0 |  6.192426 | 4.909045 | 0.0 | 0.031755 | 0.010322 |  4.898723 | 0.402013 | 4.864603 | 0.0 | 0.058575 | 4.923178 |  6.85632 |  6.85632 | 1.904533 |      0.0 |        0.0   77.375922 |       0.0  30.950369 | 179.334946 | 13.424254 |  11.65633 | 1.441295   1.48834  1.573527  1.683408  1.800016 | 1.904533 |
    | 04/01 00:00 |  0.0 |  20.0 |  20.0 |  0.082289 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    95.93582 |  0.0 |   0.814753 |     0.0 |  0.087226 | 0.062803 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 4.981128 | 0.0 | 0.031401 | 0.009725 |  4.971402 | 0.407377 |  4.53539 | 0.0 | 0.060319 | 4.595709 | 5.605758 | 5.605758 | 1.557155 |      0.0 |        0.0   68.525586 |       0.0  27.410234 | 180.527378 | 13.452889 | 12.003388 | 1.336414  1.350193  1.378876  1.424309  1.486745 | 1.557155 |
    | 04/01 01:00 |  0.0 |  20.0 |  20.0 |    0.0845 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   83.545349 |  0.0 |   0.824936 |     0.0 |   0.08957 |  0.06449 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.047589 | 0.0 | 0.032245 | 0.009162 |  5.038426 | 0.412468 | 4.470074 | 0.0 | 0.062079 | 4.532153 | 4.857647 | 4.857647 | 1.349347 |      0.0 |        0.0   59.675249 |       0.0    23.8701 | 181.651942 | 13.608774 | 12.353777 | 1.315069  1.318015  1.324707  1.336801  1.356451 | 1.349347 |
    | 04/01 02:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   71.154878 |  0.0 |   0.834599 |     0.0 |  0.089956 | 0.064768 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.110671 | 0.0 | 0.032384 | 0.008632 |  5.102039 | 0.417299 | 4.489519 | 0.0 | 0.063855 | 4.553375 | 4.602332 | 4.602332 | 1.278426 |      0.0 |        0.0   50.824913 |       0.0  20.329965 | 182.712754 | 13.803994 | 12.707221 | 1.320296  1.319788  1.319321  1.319878  1.322801 | 1.278426 |
    | 04/01 03:00 |  1.3 |  20.0 |  20.0 |  0.083584 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   58.764408 |  0.0 |   0.843935 |  1.5444 |  0.088599 | 0.054662 | 0.054662 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.170536 | 0.0 | 0.013666 | 0.008131 |  5.162404 | 0.421968 | 4.535454 | 0.0 | 0.065646 |   4.6011 | 4.557384 | 4.557384 |  1.26594 | 1.489738 |        0.0   41.974577 |       0.0  16.789831 | 183.731919 | 14.008977 | 13.063543 | 1.333846  1.332254  1.329529  1.326394  1.323778 |  1.26594 |
    | 04/01 04:00 |  0.0 |  20.0 |  20.0 |    0.0834 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   46.373937 |  0.0 |   0.852765 |     0.0 |  0.088404 | 0.063651 | 0.063651 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.228379 | 0.0 | 0.015913 | 0.007651 |  5.220728 | 0.426382 | 4.591621 | 0.0 |  0.06745 | 4.659071 | 4.584044 | 4.584044 | 1.273345 | 1.426087 |        0.0   33.124241 |       0.0  13.249696 | 184.690514 | 14.211702 | 13.422476 | 1.350548  1.348552  1.345013  1.340596  1.336119 | 1.273345 |
    | 04/01 05:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   33.983466 |  0.0 |   0.861125 |     0.0 |  0.089956 | 0.064768 | 0.064768 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.283078 | 0.0 | 0.016192 | 0.007199 |  5.275879 | 0.430563 | 4.649731 | 0.0 | 0.069265 | 4.718996 | 4.634463 | 4.634463 | 1.287351 | 1.361318 |        0.0   24.273904 |       0.0   9.709562 | 185.593678 | 14.407287 | 13.783773 | 1.367904  1.365819  1.362078  1.357305  1.352257 | 1.287351 |
    | 04/01 06:00 |  0.0 |  20.0 |  20.0 |  0.310229 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   21.592995 |  0.0 |   0.868639 |     0.0 |  0.328843 | 0.236767 | 0.236767 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.334874 | 0.0 | 0.059192 | 0.006773 |  5.328101 |  0.43432 | 4.710916 | 0.0 |  0.07109 | 4.782007 | 4.692573 | 4.692573 | 1.303492 | 1.124552 |        0.0   15.423568 |       0.0   6.169427 |  186.40162 | 14.590152 | 14.147002 | 1.386146  1.383955  1.380024  1.375003  1.369669 | 1.303492 |
    | 04/01 07:00 |  0.7 |  20.0 |  20.0 |  1.391958 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    9.202525 |  0.0 |   0.874014 |  0.8316 |  1.475475 | 0.977572 | 0.977572 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.381424 | 0.0 | 0.244393 | 0.006392 |  5.375031 | 0.437007 | 4.784772 | 0.0 |  0.07292 | 4.857692 | 4.754878 | 4.754878 | 1.320799 |  0.97858 |        0.0    6.573232 |       0.0   2.629293 | 186.977431 | 14.743404 | 14.511089 |  1.40797  1.405358  1.400709  1.394843  1.388731 | 1.320799 |
    | 04/01 08:00 |  0.4 |  20.0 |  20.0 |  3.195876 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.864568 |  0.4752 |  3.387629 | 2.325898 |  1.45378 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  6.573232 | 0.0   0.0 |  4.601262 | 4.021568 | 0.0 | 1.599008 | 0.006221 |  4.015346 | 0.432284 | 4.490724 | 0.0 | 0.074717 | 4.565441 | 4.798591 | 4.798591 | 1.332942 |      0.0 |        0.0         0.0 |       0.0        0.0 |  185.96434 | 13.835743 | 14.868656 | 1.326809  1.336157  1.351591  1.368379  1.381526 | 1.332942 |
    | 04/01 09:00 |  0.1 |  20.0 |  20.0 |  5.191651 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.829149 |  0.1188 |   5.50315 | 3.915475 |   0.1188 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 3.856075 | 0.007018 | -0.007018 | 0.414575 |  3.02507 | 0.0 | 0.076416 | 3.101486 | 4.579068 | 4.579068 | 1.271963 |      0.0 |        0.0         0.0 |       0.0        0.0 | 182.115283 |  10.38908 | 15.206815 | 0.913835  0.962164  1.044557  1.140224  1.226098 | 1.271963 |
    | 04/01 10:00 |  0.4 |  20.0 |  20.0 |  7.155036 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.784593 |  0.4752 |  7.584338 | 5.207299 |   0.4752 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 4.969699 | 0.008942 | -0.008942 | 0.392297 | 1.668741 | 0.0 | 0.077996 | 1.746736 | 3.654654 | 3.654654 | 1.015182 |      0.0 |        0.0         0.0 |       0.0        0.0 | 177.154526 |    8.3191 | 15.521116 |  0.52028  0.567665  0.652894  0.761807  0.876315 | 1.015182 |
    | 04/01 11:00 |  0.0 |  20.0 |  20.0 |  8.391432 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.728988 |     0.0 |  8.894918 | 6.404341 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 6.404341 | 0.011423 | -0.011423 | 0.364494 | 1.110617 | 0.0 | 0.079428 | 1.190045 | 2.410755 | 2.410755 | 0.669654 |      0.0 |        0.0         0.0 |       0.0        0.0 | 170.761608 |  6.832566 | 15.806182 | 0.351017  0.372293  0.413615  0.473366   0.54796 | 0.669654 |
    | 04/01 12:00 |  0.0 |  20.0 |  20.0 |  8.391286 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.675454 |     0.0 |  8.894763 | 6.404229 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 6.404229 | 0.014619 | -0.014619 | 0.337727 | 0.790254 | 0.0 |  0.08072 | 0.870974 | 1.532163 | 1.532163 | 0.425601 |      0.0 |        0.0         0.0 |       0.0        0.0 | 164.371998 |  5.689966 | 16.063189 |   0.2559  0.267676  0.290074  0.321851  0.361561 | 0.425601 |
    | 04/01 13:00 |  0.0 |  20.0 |  20.0 | 10.715238 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.610054 |     0.0 | 11.358152 |  8.17787 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |  8.17787 | 0.017814 | -0.017814 | 0.305027 | 0.621287 | 0.0 | 0.081841 | 0.703128 |  1.05606 |  1.05606 |  0.29335 |      0.0 |        0.0         0.0 |       0.0        0.0 | 156.211942 |  4.745838 | 16.286375 | 0.205659  0.211897  0.223808  0.240766  0.261999 |  0.29335 |
    | 04/01 14:00 |  0.0 |  20.0 |  20.0 |  9.383394 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.556822 |     0.0 |  9.946398 | 7.161406 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 6.992837 | 0.021894 | -0.021894 | 0.278411 | 0.476358 | 0.0 | 0.082824 | 0.559182 |  0.79776 |  0.79776 |   0.2216 |      0.0 |        0.0         0.0 |       0.0        0.0 | 149.240999 |  3.969175 | 16.481962 | 0.163629  0.168726  0.178054  0.190466  0.204678 |   0.2216 |
    | 04/01 15:00 |  0.0 |  20.0 |  20.0 |  7.861915 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.516017 |     0.0 |   8.33363 | 6.000214 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 5.597688 |  0.02538 |  -0.02538 | 0.258009 | 0.352578 | 0.0 |   0.0837 | 0.436277 | 0.629573 | 0.629573 | 0.174881 |      0.0 |        0.0         0.0 |       0.0        0.0 |  143.66869 |  3.333209 | 16.656271 | 0.127771  0.132112  0.140017    0.1504  0.161957 | 0.174881 |
    | 04/01 16:00 |  0.0 |  20.0 |  20.0 |  6.298329 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.485668 |     0.0 |  6.676229 | 4.806885 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 4.317089 | 0.028166 | -0.028166 | 0.242834 | 0.252146 | 0.0 | 0.084496 | 0.336642 | 0.495624 | 0.495624 | 0.137673 |      0.0 |        0.0         0.0 |       0.0        0.0 | 139.379767 |  2.810063 |  16.81461 | 0.098642  0.102175   0.10863   0.11715  0.126677 | 0.137673 |
    | 04/01 17:00 |  0.0 |  20.0 |  20.0 |  2.948416 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.472309 |     0.0 |  3.125321 | 2.250231 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 1.960656 |  0.03031 |  -0.03031 | 0.236154 | 0.165423 | 0.0 | 0.085254 | 0.250677 | 0.384885 | 0.384885 | 0.106913 |      0.0 |        0.0         0.0 |       0.0        0.0 | 137.449421 |  2.378176 |  16.96551 | 0.073571  0.076605  0.082126  0.089367  0.097397 | 0.106913 |
    | 04/01 18:00 |  0.0 |  20.0 |  20.0 |  1.309232 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.46664 |     0.0 |  1.387786 | 0.999206 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.858572 | 0.031275 | -0.031275 |  0.23332 | 0.103924 | 0.0 | 0.085994 | 0.189918 | 0.292707 | 0.292707 | 0.081308 |      0.0 |        0.0         0.0 |       0.0        0.0 | 136.622125 |  2.009657 | 17.112836 |   0.0557   0.05788  0.061904  0.067306  0.073487 | 0.081308 |
    | 04/01 19:00 |  0.0 |  20.0 |  20.0 |   0.32955 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.46539 |     0.0 |  0.349323 | 0.251513 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.214813 | 0.031689 | -0.031689 | 0.232695 | 0.062535 | 0.0 | 0.086728 | 0.149263 | 0.221219 | 0.221219 |  0.06145 |      0.0 |        0.0         0.0 |       0.0        0.0 |    136.439 |  1.682737 | 17.258803 | 0.043705  0.045173  0.047901  0.051608  0.055935 |  0.06145 |
    | 04/01 20:00 |  0.0 |  20.0 |  20.0 |  0.089508 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.465209 |     0.0 |  0.094878 | 0.068313 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.058267 |  0.03178 |  -0.03178 | 0.232605 | 0.035896 | 0.0 | 0.087457 | 0.123353 | 0.170771 | 0.170771 | 0.047436 |      0.0 |        0.0         0.0 |       0.0        0.0 | 136.412514 |  1.382456 | 17.403951 |  0.03604   0.03698  0.038737  0.041146  0.043999 | 0.047436 |
    | 04/01 21:00 |  0.0 |  20.0 |  20.0 |  0.085771 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.465045 |     0.0 |  0.090917 |  0.06546 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.055823 | 0.031794 | -0.031794 | 0.232523 | 0.019132 | 0.0 | 0.088182 | 0.107315 | 0.137333 | 0.137333 | 0.038148 |      0.0 |        0.0         0.0 |       0.0        0.0 | 136.388485 |  1.099007 | 17.548291 | 0.031287  0.031871  0.032966  0.034477  0.036282 | 0.038148 |
    | 04/01 22:00 |  0.0 |  20.0 |  20.0 |    0.0845 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.464887 |     0.0 |   0.08957 |  0.06449 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.054986 | 0.031806 | -0.031806 | 0.232444 | 0.008997 | 0.0 | 0.088904 | 0.097901 | 0.116104 | 0.116104 | 0.032251 |      0.0 |        0.0         0.0 |       0.0        0.0 | 136.365304 |  0.825761 | 17.691831 | 0.028489  0.028834  0.029484  0.030389  0.031485 | 0.032251 |
    | 04/01 23:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |   0.464728 |     0.0 |  0.089956 | 0.064768 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.055214 | 0.031817 | -0.031817 | 0.232364 | 0.003401 | 0.0 | 0.089621 | 0.093022 |   0.1032 |   0.1032 | 0.028667 |      0.0 |        0.0         0.0 |       0.0        0.0 | 136.341908 |  0.558178 | 17.834574 | 0.027027  0.027209  0.027556  0.028049  0.028662 | 0.028667 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v2_glacier:

glacier
_______

Besides slight differences in runoff concentration, the following calculation
reproduces the results of the :ref:`hland_v1_glacier` experiment of |hland_v1| exactly:

.. integration-test::

    >>> zonetype(GLACIER)
    >>> test("hland_v2_glacier")
    |        date |    p |     t |    tn |       epn | tmean |    tc | fracrain | rfc | sfc |    cfact |                    swe |     gact | contriarea |      pc |        ep |      epc |  ei |      tf | spl | wcl | spg | wcg |    glmelt |               melt |      refr |       in_ |         r |  sr |  ea |  cf |      inuz | perc |        q0 |  el |       q1 |      inuh |     outuh |        rt |       qt |  ic |                     sp |                   wc |  sm |        uz |        lz |                                               sc |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -20.0 | -20.0 |  0.100707 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  0.106749 |  0.07686 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 |     0.05 |      0.05 |  0.163676 |  0.163676 | 0.045466 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |      9.95 | 0.015696  0.019685  0.026398  0.034007  0.040538 | 0.045466 |
    | 01/01 01:00 |  0.0 | -20.0 | -20.0 |  0.097801 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  0.103669 | 0.074642 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 |  0.04975 |   0.04975 |  0.102365 |  0.102365 | 0.028435 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |   9.90025 |  0.01447  0.014747  0.015662  0.017698  0.021131 | 0.028435 |
    | 01/01 02:00 |  0.0 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |    0.1028 | 0.074016 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.049501 |  0.049501 |  0.059979 |  0.059979 | 0.016661 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.850749 | 0.014359  0.014381  0.014465  0.014714  0.015311 | 0.016661 |
    | 01/01 03:00 |  0.0 | -20.0 | -20.0 |   0.09599 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  0.101749 |  0.07326 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.049254 |  0.049254 |  0.050799 |  0.050799 | 0.014111 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.801495 | 0.014286  0.014295  0.014315  0.014353  0.014436 | 0.014111 |
    | 01/01 04:00 |  0.0 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |    0.1028 | 0.074016 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.049007 |  0.049007 |   0.04947 |   0.04947 | 0.013742 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.752488 | 0.014215  0.014223  0.014239   0.01426  0.014286 | 0.013742 |
    | 01/01 05:00 |  0.0 | -20.0 | -20.0 |  0.102761 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  0.108927 | 0.078427 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.048762 |  0.048762 |  0.049126 |  0.049126 | 0.013646 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.703725 | 0.014144  0.014152  0.014168  0.014187  0.014209 | 0.013646 |
    | 01/01 06:00 |  0.0 | -20.0 | -20.0 |  0.291908 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  0.309422 | 0.222784 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.048519 |  0.048519 |  0.048873 |  0.048873 | 0.013576 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.655206 | 0.014073  0.014081  0.014097  0.014116  0.014138 | 0.013576 |
    | 01/01 07:00 |  0.0 | -20.0 | -20.0 |  1.932875 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  2.048847 |  1.47517 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.048276 |  0.048276 |  0.048629 |  0.048629 | 0.013508 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |   9.60693 | 0.014003  0.014011  0.014026  0.014046  0.014067 | 0.013508 |
    | 01/01 08:00 |  0.0 | -20.0 | -20.0 |  4.369536 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  4.631708 |  3.33483 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.048035 |  0.048035 |  0.048385 |  0.048385 |  0.01344 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.558896 | 0.013932  0.013941  0.013956  0.013976  0.013997 |  0.01344 |
    | 01/01 09:00 |  0.0 | -20.0 | -20.0 |  7.317556 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  7.756609 | 5.584759 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.047794 |  0.047794 |  0.048143 |  0.048143 | 0.013373 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.511101 | 0.013863  0.013871  0.013886  0.013906  0.013927 | 0.013373 |
    | 01/01 10:00 |  0.0 | -20.0 | -20.0 |  8.264362 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  8.760224 | 6.307361 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.047556 |  0.047556 |  0.047903 |  0.047903 | 0.013306 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.463546 | 0.013794  0.013802  0.013817  0.013836  0.013857 | 0.013306 |
    | 01/01 11:00 |  0.0 | -20.0 | -20.0 |  9.369867 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  9.932059 | 7.151082 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.047318 |  0.047318 |  0.047663 |  0.047663 |  0.01324 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.416228 | 0.013725  0.013733  0.013748  0.013767  0.013788 |  0.01324 |
    | 01/01 12:00 |  0.0 | -20.0 | -20.0 |  5.126178 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  5.433749 | 3.912299 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.047081 |  0.047081 |  0.047425 |  0.047425 | 0.013174 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.369147 | 0.013656  0.013664  0.013679  0.013698  0.013719 | 0.013174 |
    | 01/01 13:00 |  0.0 | -20.0 | -20.0 |   6.62503 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  7.022532 | 5.056223 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.046846 |  0.046846 |  0.047188 |  0.047188 | 0.013108 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |  9.322301 | 0.013588  0.013596  0.013611   0.01363   0.01365 | 0.013108 |
    | 01/01 14:00 |  0.0 | -20.0 | -20.0 |  7.397619 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |        1.0 |     0.0 |  7.841476 | 5.645863 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.046612 |  0.046612 |  0.046952 |  0.046952 | 0.013042 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |   9.27569 |  0.01352  0.013528  0.013543  0.013562  0.013582 | 0.013042 |
    | 01/01 15:00 |  0.2 | -20.0 | -20.0 |   2.39151 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    0.22464     0.33696 | 0.901953 |        1.0 |  0.2808 |  2.535001 | 1.774662 | 0.0 |  0.2808 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.046378 |  0.046378 |  0.046717 |  0.046717 | 0.012977 | 0.0 |    0.22464     0.33696 |       0.0        0.0 | 0.0 |       0.0 |  9.229311 | 0.013452   0.01346  0.013475  0.013494  0.013514 | 0.012977 |
    | 01/01 16:00 |  0.0 | -20.0 | -20.0 |  1.829834 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    0.22464     0.33696 | 0.901953 |        1.0 |     0.0 |  1.939624 | 1.396529 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.046147 |  0.046147 |  0.046484 |  0.046484 | 0.012912 | 0.0 |    0.22464     0.33696 |       0.0        0.0 | 0.0 |       0.0 |  9.183165 | 0.013385  0.013393  0.013408  0.013426  0.013446 | 0.012912 |
    | 01/01 17:00 |  0.0 | -20.0 | -20.0 |  1.136569 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    0.22464     0.33696 | 0.901953 |        1.0 |     0.0 |  1.204763 | 0.867429 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.045916 |  0.045916 |  0.046251 |  0.046251 | 0.012848 | 0.0 |    0.22464     0.33696 |       0.0        0.0 | 0.0 |       0.0 |  9.137249 | 0.013318  0.013326   0.01334  0.013359  0.013379 | 0.012848 |
    | 01/01 18:00 |  1.3 | -20.0 | -20.0 |  0.750986 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     1.6848      2.5272 | 0.901953 |        1.0 |  1.8252 |  0.796045 | 0.477532 | 0.0 |  1.8252 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.045686 |  0.045686 |   0.04602 |   0.04602 | 0.012783 | 0.0 |     1.6848      2.5272 |       0.0        0.0 | 0.0 |       0.0 |  9.091563 | 0.013251  0.013259  0.013274  0.013292  0.013312 | 0.012783 |
    | 01/01 19:00 |  5.6 | -20.0 | -20.0 |  0.223895 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    7.97472    11.96208 | 0.901953 |        1.0 |  7.8624 |  0.237329 | 0.077844 | 0.0 |  7.8624 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.045458 |  0.045458 |   0.04579 |   0.04579 | 0.012719 | 0.0 |    7.97472    11.96208 |       0.0        0.0 | 0.0 |       0.0 |  9.046105 | 0.013185  0.013193  0.013207  0.013226  0.013246 | 0.012719 |
    | 01/01 20:00 |  2.9 | -20.0 | -20.0 |  0.099425 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     11.232      16.848 | 0.901953 |        1.0 |  4.0716 |   0.10539 | 0.050502 | 0.0 |  4.0716 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.045231 |  0.045231 |  0.045561 |  0.045561 | 0.012656 | 0.0 |     11.232      16.848 |       0.0        0.0 | 0.0 |       0.0 |  9.000874 | 0.013119  0.013127  0.013141   0.01316  0.013179 | 0.012656 |
    | 01/01 21:00 |  4.9 | -20.0 | -20.0 |  0.098454 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   16.73568    25.10352 | 0.901953 |        1.0 |  6.8796 |  0.104361 | 0.037765 | 0.0 |  6.8796 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.045004 |  0.045004 |  0.045333 |  0.045333 | 0.012593 | 0.0 |   16.73568    25.10352 |       0.0        0.0 | 0.0 |       0.0 |   8.95587 | 0.013054  0.013061  0.013076  0.013094  0.013114 | 0.012593 |
    | 01/01 22:00 | 10.6 | -20.0 | -20.0 |  0.098128 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    28.6416     42.9624 | 0.901953 |        1.0 | 14.8824 |  0.104016 | 0.016908 | 0.0 | 14.8824 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.044779 |  0.044779 |  0.045106 |  0.045106 |  0.01253 | 0.0 |    28.6416     42.9624 |       0.0        0.0 | 0.0 |       0.0 |  8.911091 | 0.012988  0.012996   0.01301  0.013028  0.013048 |  0.01253 |
    | 01/01 23:00 |  0.1 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   28.75392    43.13088 | 0.901953 |        1.0 |  0.1404 |  0.103322 | 0.073355 | 0.0 |  0.1404 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.044555 |  0.044555 |  0.044881 |  0.044881 | 0.012467 | 0.0 |   28.75392    43.13088 |       0.0        0.0 | 0.0 |       0.0 |  8.866535 | 0.012923  0.012931  0.012945  0.012963  0.012983 | 0.012467 |
    | 02/01 00:00 |  0.7 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   29.54016    44.31024 | 0.902305 |        1.0 |  0.9828 |  0.103322 | 0.067429 | 0.0 |  0.9828 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.044333 |  0.044333 |  0.044656 |  0.044656 | 0.012405 | 0.0 |   29.54016    44.31024 |       0.0        0.0 | 0.0 |       0.0 |  8.822202 | 0.012859  0.012867  0.012881  0.012899  0.012918 | 0.012405 |
    | 02/01 01:00 |  3.0 | -20.0 | -20.0 |  0.097474 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   32.90976    49.36464 | 0.902305 |        1.0 |   4.212 |  0.103322 | 0.048821 | 0.0 |   4.212 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.044111 |  0.044111 |  0.044433 |  0.044433 | 0.012343 | 0.0 |   32.90976    49.36464 |       0.0        0.0 | 0.0 |       0.0 |  8.778091 | 0.012794  0.012802  0.012816  0.012834  0.012853 | 0.012343 |
    | 02/01 02:00 |  2.1 | -20.0 | -20.0 |  0.096981 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   35.26848    52.90272 | 0.902305 |        1.0 |  2.9484 |    0.1028 | 0.055116 | 0.0 |  2.9484 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 |  0.04389 |   0.04389 |  0.044211 |  0.044211 | 0.012281 | 0.0 |   35.26848    52.90272 |       0.0        0.0 | 0.0 |       0.0 |  8.734201 |  0.01273  0.012738  0.012752   0.01277  0.012789 | 0.012281 |
    | 02/01 03:00 | 10.4 | -20.0 | -20.0 |  0.096652 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   46.94976    70.42464 | 0.902305 |        1.0 | 14.6016 |  0.102451 | 0.017128 | 0.0 | 14.6016 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.043671 |  0.043671 |   0.04399 |   0.04399 | 0.012219 | 0.0 |   46.94976    70.42464 |       0.0        0.0 | 0.0 |       0.0 |   8.69053 | 0.012667  0.012674  0.012688  0.012706  0.012725 | 0.012219 |
    | 02/01 04:00 |  3.5 | -20.0 | -20.0 |  0.096321 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   50.88096    76.32144 | 0.902305 |        1.0 |   4.914 |    0.1021 | 0.044973 | 0.0 |   4.914 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.043453 |  0.043453 |   0.04377 |   0.04377 | 0.012158 | 0.0 |   50.88096    76.32144 |       0.0        0.0 | 0.0 |       0.0 |  8.647077 | 0.012603  0.012611  0.012625  0.012642  0.012661 | 0.012158 |
    | 02/01 05:00 |  3.4 | -20.0 | -20.0 |   0.09599 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   54.69984    82.04976 | 0.902305 |        1.0 |  4.7736 |  0.101749 | 0.045452 | 0.0 |  4.7736 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.043235 |  0.043235 |  0.043551 |  0.043551 | 0.012098 | 0.0 |   54.69984    82.04976 |       0.0        0.0 | 0.0 |       0.0 |  8.603842 |  0.01254  0.012548  0.012562  0.012579  0.012598 | 0.012098 |
    | 02/01 06:00 |  1.2 | -20.0 | -20.0 |  0.187298 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   56.04768    84.07152 | 0.902305 |        1.0 |  1.6848 |  0.198536 | 0.120782 | 0.0 |  1.6848 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.043019 |  0.043019 |  0.043333 |  0.043333 | 0.012037 | 0.0 |   56.04768    84.07152 |       0.0        0.0 | 0.0 |       0.0 |  8.560823 | 0.012478  0.012485  0.012499  0.012516  0.012535 | 0.012037 |
    | 02/01 07:00 |  0.1 | -20.0 | -20.0 |  1.264612 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |      56.16       84.24 | 0.902305 |        1.0 |  0.1404 |  1.340489 | 0.951696 | 0.0 |  0.1404 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.042804 |  0.042804 |  0.043117 |  0.043117 | 0.011977 | 0.0 |      56.16       84.24 |       0.0        0.0 | 0.0 |       0.0 |  8.518019 | 0.012415  0.012423  0.012436  0.012454  0.012472 | 0.011977 |
    | 02/01 08:00 |  0.0 | -20.0 | -20.0 |  3.045538 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |      56.16       84.24 | 0.902305 |        1.0 |     0.0 |   3.22827 | 2.324355 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 |  0.04259 |   0.04259 |  0.042901 |  0.042901 | 0.011917 | 0.0 |      56.16       84.24 |       0.0        0.0 | 0.0 |       0.0 |  8.475429 | 0.012353  0.012361  0.012374  0.012392   0.01241 | 0.011917 |
    | 02/01 09:00 |  0.0 | -20.0 | -20.0 |  1.930758 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |      56.16       84.24 | 0.902305 |        1.0 |     0.0 |  2.046603 | 1.473555 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.042377 |  0.042377 |  0.042687 |  0.042687 | 0.011857 | 0.0 |      56.16       84.24 |       0.0        0.0 | 0.0 |       0.0 |  8.433051 | 0.012292  0.012299  0.012312   0.01233  0.012348 | 0.011857 |
    | 02/01 10:00 |  0.4 | -20.0 | -20.0 |  2.461001 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   56.60928    84.91392 | 0.902305 |        1.0 |  0.5616 |  2.608661 | 1.775661 | 0.0 |  0.5616 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.042165 |  0.042165 |  0.042473 |  0.042473 | 0.011798 | 0.0 |   56.60928    84.91392 |       0.0        0.0 | 0.0 |       0.0 |  8.390886 |  0.01223  0.012237  0.012251  0.012268  0.012286 | 0.011798 |
    | 02/01 11:00 |  0.1 | -20.0 | -20.0 |  6.215945 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    56.7216     85.0824 | 0.902305 |        1.0 |  0.1404 |  6.588902 | 4.677869 | 0.0 |  0.1404 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.041954 |  0.041954 |  0.042261 |  0.042261 | 0.011739 | 0.0 |    56.7216     85.0824 |       0.0        0.0 | 0.0 |       0.0 |  8.348932 | 0.012169  0.012176   0.01219  0.012207  0.012225 | 0.011739 |
    | 02/01 12:00 |  3.6 | -20.0 | -20.0 |  3.374783 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   60.76512    91.14768 | 0.902305 |        1.0 |  5.0544 |   3.57727 | 1.553726 | 0.0 |  5.0544 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.041745 |  0.041745 |  0.042049 |  0.042049 |  0.01168 | 0.0 |   60.76512    91.14768 |       0.0        0.0 | 0.0 |       0.0 |  8.307187 | 0.012108  0.012115  0.012129  0.012146  0.012164 |  0.01168 |
    | 02/01 13:00 |  5.9 | -20.0 | -20.0 |  8.821555 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |     67.392     101.088 | 0.902305 |        1.0 |  8.2836 |  9.350848 | 2.940569 | 0.0 |  8.2836 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.041536 |  0.041536 |  0.041839 |  0.041839 | 0.011622 | 0.0 |     67.392     101.088 |       0.0        0.0 | 0.0 |       0.0 |  8.265651 | 0.012048  0.012055  0.012068  0.012085  0.012103 | 0.011622 |
    | 02/01 14:00 |  1.1 | -20.0 | -20.0 |  4.046025 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   68.62752   102.94128 | 0.902305 |        1.0 |  1.5444 |  4.288787 | 2.646028 | 0.0 |  1.5444 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.041328 |  0.041328 |   0.04163 |   0.04163 | 0.011564 | 0.0 |   68.62752   102.94128 |       0.0        0.0 | 0.0 |       0.0 |  8.224323 | 0.011987  0.011995  0.012008  0.012024  0.012042 | 0.011564 |
    | 02/01 15:00 | 20.7 | -20.0 | -20.0 |  2.110757 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   91.87776   137.81664 | 0.902305 |        1.0 | 29.0628 |  2.237402 | 0.088084 | 0.0 | 29.0628 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.041122 |  0.041122 |  0.041422 |  0.041422 | 0.011506 | 0.0 |   91.87776   137.81664 |       0.0        0.0 | 0.0 |       0.0 |  8.183201 | 0.011927  0.011935  0.011948  0.011964  0.011982 | 0.011506 |
    | 02/01 16:00 | 37.9 | -20.0 | -20.0 |  2.239257 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  134.44704   201.67056 | 0.902305 |        1.0 | 53.2116 |  2.373612 | 0.008352 | 0.0 | 53.2116 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.040916 |  0.040916 |  0.041215 |  0.041215 | 0.011449 | 0.0 |  134.44704   201.67056 |       0.0        0.0 | 0.0 |       0.0 |  8.142285 | 0.011868  0.011875  0.011888  0.011904  0.011922 | 0.011449 |
    | 02/01 17:00 |  8.2 | -20.0 | -20.0 |  2.877848 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  143.65728   215.48592 | 0.902305 |        1.0 | 11.5128 |  3.050519 | 0.694563 | 0.0 | 11.5128 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.040711 |  0.040711 |  0.041009 |  0.041009 | 0.011391 | 0.0 |  143.65728   215.48592 |       0.0        0.0 | 0.0 |       0.0 |  8.101574 | 0.011808  0.011816  0.011828  0.011845  0.011863 | 0.011391 |
    | 02/01 18:00 |  3.6 | -20.0 | -20.0 |  1.591452 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   147.7008    221.5512 | 0.902305 |        1.0 |  5.0544 |  1.686939 | 0.732693 | 0.0 |  5.0544 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.040508 |  0.040508 |  0.040804 |  0.040804 | 0.011334 | 0.0 |   147.7008    221.5512 |       0.0        0.0 | 0.0 |       0.0 |  8.061066 | 0.011749  0.011756  0.011769  0.011786  0.011803 | 0.011334 |
    | 02/01 19:00 |  7.5 | -20.0 | -20.0 |  0.291604 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   156.1248    234.1872 | 0.902305 |        1.0 |   10.53 |    0.3091 | 0.077646 | 0.0 |   10.53 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.040305 |  0.040305 |    0.0406 |    0.0406 | 0.011278 | 0.0 |   156.1248    234.1872 |       0.0        0.0 | 0.0 |       0.0 |  8.020761 | 0.011691  0.011698   0.01171  0.011727  0.011744 | 0.011278 |
    | 02/01 20:00 | 18.5 | -20.0 | -20.0 |  0.092622 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    176.904     265.356 | 0.902305 |        1.0 |  25.974 |  0.098179 | 0.005264 | 0.0 |  25.974 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.040104 |  0.040104 |  0.040397 |  0.040397 | 0.011221 | 0.0 |    176.904     265.356 |       0.0        0.0 | 0.0 |       0.0 |  7.980657 | 0.011632  0.011639  0.011652  0.011668  0.011686 | 0.011221 |
    | 02/01 21:00 | 15.4 | -20.0 | -20.0 |  0.092451 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  194.20128   291.30192 | 0.902305 |        1.0 | 21.6216 |  0.097998 |  0.00812 | 0.0 | 21.6216 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.039903 |  0.039903 |  0.040195 |  0.040195 | 0.011165 | 0.0 |  194.20128   291.30192 |       0.0        0.0 | 0.0 |       0.0 |  7.940753 | 0.011574  0.011581  0.011594   0.01161  0.011627 | 0.011165 |
    | 02/01 22:00 |  6.3 | -20.0 | -20.0 |  0.091248 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  201.27744   301.91616 | 0.902305 |        1.0 |  8.8452 |  0.096723 | 0.028755 | 0.0 |  8.8452 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.039704 |  0.039704 |  0.039994 |  0.039994 | 0.011109 | 0.0 |  201.27744   301.91616 |       0.0        0.0 | 0.0 |       0.0 |   7.90105 | 0.011516  0.011523  0.011536  0.011552  0.011569 | 0.011109 |
    | 02/01 23:00 |  1.9 | -20.0 | -20.0 |  0.089683 | -19.4 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  203.41152   305.11728 | 0.902305 |        1.0 |  2.6676 |  0.095064 |  0.05242 | 0.0 |  2.6676 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.039505 |  0.039505 |  0.039794 |  0.039794 | 0.011054 | 0.0 |  203.41152   305.11728 |       0.0        0.0 | 0.0 |       0.0 |  7.861544 | 0.011459  0.011465  0.011478  0.011494  0.011511 | 0.011054 |
    | 03/01 00:00 |  4.9 |  20.0 |  20.0 |  0.089858 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  208.06848   312.10272 | 0.902686 |        1.0 |  5.8212 |  0.095249 | 0.038316 | 0.0 |  5.8212 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.039308 |  0.039308 |  0.039595 |  0.039595 | 0.010999 | 0.0 | 194.565197  296.270957 | 13.503283  15.831763 | 0.0 |       0.0 |  7.822237 | 0.011401  0.011408  0.011421  0.011437  0.011454 | 0.010999 |
    | 03/01 01:00 |  2.7 |  20.0 |  20.0 |  0.089683 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  210.63456   315.95184 | 0.902686 |        1.0 |  3.2076 |  0.095064 | 0.049664 | 0.0 |  3.2076 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.039111 |  0.039111 |  0.039397 |  0.039397 | 0.010944 | 0.0 | 185.718874  287.424634 | 24.915686  28.527206 | 0.0 |       0.0 |  7.783126 | 0.011344  0.011351  0.011363  0.011379  0.011396 | 0.010944 |
    | 03/01 02:00 |  0.5 |  20.0 |  20.0 |  0.088805 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  211.10976   316.66464 | 0.902686 |        1.0 |   0.594 |  0.094133 | 0.063867 | 0.0 |   0.594 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.038916 |  0.038916 |    0.0392 |    0.0392 | 0.010889 | 0.0 | 176.872552  278.578312 | 34.237208  38.086328 | 0.0 |       0.0 |   7.74421 | 0.011288  0.011294  0.011307  0.011322  0.011339 | 0.010889 |
    | 03/01 03:00 |  0.2 |  20.0 |  20.0 |  0.089157 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  211.29984   316.94976 | 0.902686 |        1.0 |  0.2376 |  0.094506 | 0.066447 | 0.0 |  0.2376 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.038721 |  0.038721 |  0.039004 |  0.039004 | 0.010834 | 0.0 | 168.026229  269.731989 | 43.273611  47.217771 | 0.0 |       0.0 |  7.705489 | 0.011231  0.011238   0.01125  0.011266  0.011283 | 0.010834 |
    | 03/01 04:00 |  0.5 |  20.0 |  20.0 |  0.090207 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  211.77504   317.66256 | 0.902686 |        1.0 |   0.594 |  0.095619 | 0.064876 | 0.0 |   0.594 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 | 0.0 |       0.0 |  0.0 |       0.0 | 0.0 | 0.038527 |  0.038527 |  0.038809 |  0.038809 |  0.01078 | 0.0 | 159.179906  260.885666 | 52.595134  56.776894 | 0.0 |       0.0 |  7.666961 | 0.011175  0.011182  0.011194  0.011209  0.011226 |  0.01078 |
    | 03/01 05:00 |  2.4 |  20.0 |  20.0 |  0.091593 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 210.467016     321.084 | 0.902686 |        1.0 |  2.8512 |  0.097089 | 0.052562 | 0.0 |  2.8512 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  1.794492 |  1.794492 | 0.0 | 0.0 | 0.0 |  1.794492 |  0.5 |  0.000553 | 0.0 | 0.040835 |  0.041388 |  0.038833 |  0.038833 | 0.010787 | 0.0 | 150.333583  252.039343 | 60.133433  69.044657 | 0.0 |  1.293939 |  8.126127 | 0.011975  0.011882  0.011726  0.011552  0.011406 | 0.010787 |
    | 03/01 06:00 |  0.4 |  20.0 |  20.0 |  0.154861 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 198.082164   321.65424 | 0.902686 |        1.0 |  0.4752 |  0.164153 | 0.112705 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.382506 |  6.382506 | 0.0 | 0.0 | 0.0 |  6.382506 |  0.5 |   0.11148 | 0.0 | 0.043131 |  0.154611 |  0.048264 |  0.048264 | 0.013407 | 0.0 |  141.48726   243.19302 | 56.594904   78.46122 | 0.0 |  7.064965 |  8.582996 |  0.04373  0.040035  0.033804   0.02672    0.0206 | 0.013407 |
    | 03/01 07:00 |  0.2 |  20.0 |  20.0 |  0.470369 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 185.697312   321.93936 | 0.902686 |        1.0 |  0.2376 |  0.498591 | 0.350557 | 0.0 |  0.2376 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.287466 |  6.287466 | 0.0 | 0.0 | 0.0 |  6.287466 |  0.5 |  0.949881 | 0.0 | 0.045415 |  0.995296 |  0.165934 |  0.165934 | 0.046093 | 0.0 | 132.640937  234.346697 | 53.056375  87.592663 | 0.0 |  11.90255 |  9.037581 | 0.280382  0.252743  0.205803  0.151675  0.103648 | 0.046093 |
    | 03/01 08:00 |  0.0 |  20.0 |  20.0 |  1.173726 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  173.31246  315.700524 | 0.902686 |        1.0 |     0.0 |   1.24415 | 0.895788 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  9.311844 |  9.311844 | 0.0 | 0.0 | 0.0 |  9.311844 |  0.5 |  3.446062 | 0.0 | 0.047688 |   3.49375 |  0.804484 |  0.804484 | 0.223468 | 0.0 | 123.794615  225.500375 | 49.517846   90.20015 | 0.0 | 17.268331 |  9.489893 |  0.98849  0.905231   0.76196  0.592574  0.435263 | 0.223468 |
    | 03/01 09:00 |  0.0 |  20.0 |  20.0 |  4.202296 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 160.927608  303.315672 | 0.902686 |        1.0 |     0.0 |  4.454434 | 3.207192 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 |  7.825683 | 0.0 | 0.049949 |  7.875633 |  2.666654 |  2.666654 | 0.740737 | 0.0 | 114.948292  216.654052 | 45.979317  86.661621 | 0.0 |   21.3275 |  9.939944 | 2.240273   2.09193  1.832761   1.51756  1.209972 | 0.740737 |
    | 03/01 10:00 |  0.3 |  20.0 |  20.0 |  4.359715 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 148.542756   290.93082 | 0.902686 |        1.0 |  0.3564 |  4.621298 | 3.210837 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.741252 | 12.741252 | 0.0 | 0.0 | 0.0 | 12.741252 |  0.5 |  10.98364 | 0.0 |   0.0522 |  11.03584 |  6.010904 |  6.010904 | 1.669695 | 0.0 | 106.101969  207.807729 | 42.440788  83.123092 | 0.0 | 22.585111 | 10.387744 | 3.168034   3.05517  2.848197  2.574622  2.271409 | 1.669695 |
    | 03/01 11:00 |  2.6 |  20.0 |  20.0 |  5.305753 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 136.157904  278.545968 | 0.902686 |        1.0 |  3.0888 |  5.624098 | 2.973312 | 0.0 |  3.0888 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 15.473652 | 15.473652 | 0.0 | 0.0 | 0.0 | 15.473652 |  0.5 | 13.386091 | 0.0 | 0.054439 |  13.44053 |   9.46723 |   9.46723 | 2.629786 | 0.0 |  97.255646  198.961406 | 38.902258  79.584562 | 0.0 | 24.172672 | 10.833305 |  3.87316  3.787375   3.62972  3.419503  3.180974 | 2.629786 |
    | 03/01 12:00 |  0.7 |  20.0 |  20.0 |  5.376027 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 123.773052  266.161116 | 0.902686 |        1.0 |  0.8316 |  5.698589 | 3.775582 | 0.0 |  0.8316 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 13.216452 | 13.216452 | 0.0 | 0.0 | 0.0 | 13.216452 |  0.5 | 13.496721 | 0.0 | 0.056667 | 13.553388 | 12.113081 | 12.113081 | 3.364745 | 0.0 |  88.409323  190.115083 | 35.363729  76.046033 | 0.0 | 23.392403 | 11.276639 |  3.92855  3.919219  3.893428  3.840122   3.74972 | 3.364745 |
    | 03/01 13:00 |  0.3 |  20.0 |  20.0 |  4.658915 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 111.388201  253.776265 | 0.902686 |        1.0 |  0.3564 |   4.93845 | 3.431191 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.741252 | 12.741252 | 0.0 | 0.0 | 0.0 | 12.741252 |  0.5 | 12.616985 | 0.0 | 0.058883 | 12.675868 | 13.208693 | 13.208693 | 3.669081 | 0.0 |     79.563   181.26876 |   31.8252  72.507504 | 0.0 |  23.01667 | 11.717755 | 3.684514  3.712572  3.758414  3.806084  3.836632 | 3.669081 |
    | 03/01 14:00 |  0.3 |  20.0 |  20.0 |  7.789594 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  99.003349  241.391413 | 0.902686 |        1.0 |  0.3564 |   8.25697 | 5.736869 | 0.0 |  0.3564 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.741252 | 12.741252 | 0.0 | 0.0 | 0.0 | 12.741252 |  0.5 | 12.315165 | 0.0 | 0.061089 | 12.376254 | 12.994317 | 12.994317 | 3.609532 | 0.0 |  70.716678  172.422438 | 28.286671  68.968975 | 0.0 | 22.942756 | 12.156667 | 3.592329  3.603992  3.626731  3.659241  3.697859 | 3.609532 |
    | 03/01 15:00 |  0.0 |  20.0 |  20.0 |  4.851567 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  86.618497  229.006561 | 0.902686 |        1.0 |     0.0 |  5.142661 | 3.702716 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 12.076198 | 0.0 | 0.063283 | 12.139482 | 12.556889 | 12.556889 | 3.488025 | 0.0 |  61.870355  163.576115 | 24.748142  65.430446 | 0.0 |  22.75141 | 12.593383 | 3.522872  3.531341  3.547011  3.568272   3.59325 | 3.488025 |
    | 03/01 16:00 |  0.0 |  20.0 |  20.0 |   5.30692 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  74.233645  216.621709 | 0.902686 |        1.0 |     0.0 |  5.625335 | 4.050241 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.924071 | 0.0 | 0.065467 | 11.989538 | 12.263174 | 12.263174 | 3.406437 | 0.0 |  53.024032  154.729792 | 21.209613  61.891917 | 0.0 | 22.712191 | 13.027916 | 3.478513  3.483956  3.494117  3.508034  3.524489 | 3.406437 |
    | 03/01 17:00 |  0.0 |  20.0 |  20.0 |  3.286036 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  61.848793  204.236857 | 0.902686 |        1.0 |     0.0 |  3.483198 | 2.507903 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.892956 | 0.0 |  0.06764 | 11.960595 | 12.074888 | 12.074888 | 3.354136 | 0.0 |  44.177709  145.883469 | 17.671084  58.353388 | 0.0 | 22.704088 | 13.460277 | 3.468908  3.470208  3.473042  3.477812  3.484847 | 3.354136 |
    | 03/01 18:00 |  0.0 |  20.0 |  20.0 |  1.506216 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  49.463941  191.852005 | 0.902686 |        1.0 |     0.0 |  1.596589 | 1.149544 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.886529 | 0.0 | 0.069801 | 11.956331 | 11.987704 | 11.987704 | 3.329918 | 0.0 |  35.331386  137.037146 | 14.132554  54.814858 | 0.0 |  22.70241 | 13.890476 | 3.467389  3.467609  3.468148   3.46921  3.471087 | 3.329918 |
    | 03/01 19:00 |  0.0 |  20.0 |  20.0 |  0.274762 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  37.079089  179.467153 | 0.902686 |        1.0 |     0.0 |  0.291248 | 0.209698 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.885199 | 0.0 | 0.071952 | 11.957152 | 11.962291 | 11.962291 | 3.322859 | 0.0 |  26.485063  128.190823 | 10.594025  51.276329 | 0.0 | 22.702063 | 14.318523 | 3.467568  3.467554  3.467562  3.467661  3.467958 | 3.322859 |
    | 03/01 20:00 |  0.0 |  20.0 |  20.0 |  0.087565 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  24.694237  167.082301 | 0.902686 |        1.0 |     0.0 |  0.092819 |  0.06683 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.884924 | 0.0 | 0.074093 | 11.959016 | 11.957712 | 11.957712 | 3.321587 | 0.0 |  17.638741  119.344501 |  7.055496    47.7378 | 0.0 | 22.701991 | 14.744431 | 3.468096  3.468034   3.46793  3.467814  3.467733 | 3.321587 |
    | 03/01 21:00 |  0.0 |  20.0 |  20.0 |  0.085771 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  12.309385  154.697449 | 0.902686 |        1.0 |     0.0 |  0.090917 |  0.06546 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 | 0.0 | 12.384852 |  0.5 | 11.884867 | 0.0 | 0.076222 | 11.961089 | 11.958375 | 11.958375 | 3.321771 | 0.0 |   8.792418  110.498178 |  3.516967  44.199271 | 0.0 | 22.701976 | 15.168208 | 3.468695  3.468623  3.468496  3.468335  3.468173 | 3.321771 |
    | 03/01 22:00 |  0.0 |  20.0 |  20.0 |  0.084317 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  142.312597 | 0.902686 |        1.0 |     0.0 |  0.089376 | 0.064351 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.846323 | 8.792418  8.846323 | 0.0   0.0 | 21.193441 | 21.193441 | 0.0 | 0.0 | 0.0 | 21.193441 |  0.5 | 16.689124 | 0.0 | 0.078341 | 16.767465 | 12.303124 | 12.303124 | 3.417534 | 0.0 |        0.0  101.651855 |       0.0  40.660742 | 0.0 | 26.706293 | 15.589867 | 4.815587   4.65899  4.395378  4.096598  3.840111 | 3.417534 |
    | 03/01 23:00 |  0.0 |  20.0 |  20.0 |  0.083215 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  129.927745 | 0.902686 |        1.0 |     0.0 |  0.088208 |  0.06351 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.846323 |      0.0  8.846323 | 0.0   0.0 | 15.038749 | 15.038749 | 0.0 | 0.0 | 0.0 | 15.038749 |  0.5 |  16.61309 | 0.0 | 0.080449 |  16.69354 | 14.705258 | 14.705258 | 4.084794 | 0.0 |        0.0   92.805532 |       0.0  37.122213 | 0.0 | 24.631952 | 16.009418 | 4.840266  4.832119  4.800811  4.726037  4.595712 | 4.084794 |
    | 04/01 00:00 |  0.0 |  20.0 |  20.0 |  0.082289 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  117.537274 | 0.903096 |        1.0 |     0.0 |  0.087226 | 0.062803 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 | 14.852812 | 0.0 | 0.082547 | 14.935359 | 16.201527 | 16.201527 | 4.500424 | 0.0 |        0.0   83.955196 |       0.0  33.582078 | 0.0 | 24.324711 | 16.426871 |  4.34841   4.40532  4.499611  4.601128  4.674309 | 4.500424 |
    | 04/01 01:00 |  0.0 |  20.0 |  20.0 |    0.0845 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  105.146803 | 0.903096 |        1.0 |     0.0 |   0.08957 |  0.06449 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 | 14.595421 | 0.0 | 0.084634 | 14.680056 | 15.628958 | 15.628958 | 4.341377 | 0.0 |        0.0    75.10486 |       0.0  30.041944 | 0.0 | 24.274862 | 16.842237 |  4.26029  4.272453   4.29927  4.343593   4.40427 | 4.341377 |
    | 04/01 02:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   92.756333 | 0.903096 |        1.0 |     0.0 |  0.089956 | 0.064768 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 | 14.553761 | 0.0 | 0.086711 | 14.640472 | 14.928732 | 14.928732 |  4.14687 | 0.0 |        0.0   66.254523 |       0.0  26.501809 | 0.0 | 24.266672 | 17.255525 | 4.246227  4.248272  4.253295  4.263221  4.280601 |  4.14687 |
    | 04/01 03:00 |  1.3 |  20.0 |  20.0 |  0.083584 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   80.365862 | 0.903096 |        1.0 |  1.5444 |  0.088599 | 0.054662 | 0.0 |  1.5444 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 16.589972 | 16.589972 | 0.0 | 0.0 | 0.0 | 16.589972 |  0.5 | 15.394071 | 0.0 | 0.088778 | 15.482849 | 14.755517 | 14.755517 | 4.098755 | 0.0 |        0.0   57.404187 |       0.0  22.961675 | 0.0 | 24.962573 | 17.666748 | 4.481809  4.454489  4.408782  4.357814  4.316053 | 4.098755 |
    | 04/01 04:00 |  0.0 |  20.0 |  20.0 |    0.0834 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   67.975391 | 0.903096 |        1.0 |     0.0 |  0.088404 | 0.063651 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 | 15.130965 | 0.0 | 0.090834 | 15.221798 | 15.110988 | 15.110988 | 4.197497 | 0.0 |        0.0   48.553851 |       0.0   19.42154 | 0.0 |  24.37718 | 18.075914 | 4.416596  4.423257   4.43141  4.433855  4.424638 | 4.197497 |
    | 04/01 05:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    55.58492 | 0.903096 |        1.0 |     0.0 |  0.089956 | 0.064768 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 | 14.639301 | 0.0 |  0.09288 | 14.732181 | 15.229493 | 15.229493 | 4.230415 | 0.0 |        0.0   39.703515 |       0.0  15.881406 | 0.0 |  24.28345 | 18.483034 | 4.277195  4.293626  4.321932   4.35511  4.384583 | 4.230415 |
    | 04/01 06:00 |  0.0 |  20.0 |  20.0 |  0.310229 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    43.19445 | 0.903096 |        1.0 |     0.0 |  0.328843 | 0.236767 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 0.0 | 15.045572 |  0.5 | 14.560936 | 0.0 | 0.094915 | 14.655852 | 14.949752 | 14.949752 | 4.152709 | 0.0 |        0.0   30.853178 |       0.0  12.341271 | 0.0 | 24.268085 | 18.888119 | 4.251107  4.254694  4.262599  4.275791  4.294354 | 4.152709 |
    | 04/01 07:00 |  0.7 |  20.0 |  20.0 |  1.391958 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   30.803979 | 0.903096 |        1.0 |  0.8316 |  1.475475 | 0.977572 | 0.0 |  0.8316 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.877172 | 15.877172 | 0.0 | 0.0 | 0.0 | 15.877172 |  0.5 | 15.001429 | 0.0 | 0.096941 |  15.09837 | 14.762477 | 14.762477 | 4.100688 | 0.0 |        0.0   22.002842 |       0.0   8.801137 | 0.0 | 24.643828 | 19.291179 | 4.374233  4.360039  4.336611  4.311312  4.292243 | 4.100688 |
    | 04/01 08:00 |  0.4 |  20.0 |  20.0 |  3.195876 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   18.413508 | 0.903096 |        1.0 |  0.4752 |  3.387629 | 2.325898 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.520772 | 15.520772 | 0.0 | 0.0 | 0.0 | 15.520772 |  0.5 | 15.122306 | 0.0 | 0.098956 | 15.221262 | 14.929026 | 14.929026 | 4.146952 | 0.0 |        0.0   13.152506 |       0.0   5.261002 | 0.0 | 24.542293 | 19.692223 |  4.41282  4.407855  4.397919  4.383191  4.364888 | 4.146952 |
    | 04/01 09:00 |  0.1 |  20.0 |  20.0 |  5.191651 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    6.023037 | 0.903096 |        1.0 |  0.1188 |   5.50315 | 3.915475 | 0.0 |  0.1188 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.164372 | 15.164372 | 0.0 | 0.0 | 0.0 | 15.164372 |  0.5 | 14.842181 | 0.0 | 0.100961 | 14.943142 | 15.113338 | 15.113338 |  4.19815 | 0.0 |        0.0     4.30217 |       0.0   1.720868 | 0.0 | 24.364484 | 20.091262 | 4.336184  4.344927  4.359033  4.373479  4.382855 |  4.19815 |
    | 04/01 10:00 |  0.4 |  20.0 |  20.0 |  7.155036 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |  0.4752 |  7.584338 | 5.207299 | 0.0 |  0.4752 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0   4.30217 | 0.0   0.0 | 21.187391 | 21.187391 | 0.0 | 0.0 | 0.0 | 21.187391 |  0.5 | 18.130219 | 0.0 | 0.102956 | 18.233175 | 15.282618 | 15.282618 | 4.245172 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 26.921657 | 20.488305 | 5.255554   5.14896  4.970488  4.770292  4.601741 | 4.245172 |
    | 04/01 11:00 |  0.0 |  20.0 |  20.0 |  8.391432 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  8.894918 | 6.404341 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 18.318022 | 0.0 | 0.104942 | 18.422964 | 16.860328 | 16.860328 | 4.683424 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.804307 | 20.883364 | 5.339724  5.326345  5.291882  5.226257  5.125463 | 4.683424 |
    | 04/01 12:00 |  0.0 |  20.0 |  20.0 |  8.391286 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  8.894763 | 6.404229 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.341911 | 0.0 | 0.106917 | 17.448828 |  18.00226 |  18.00226 | 5.000628 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.663069 | 21.276447 | 5.069582  5.100539  5.150744  5.202049  5.233324 | 5.000628 |
    | 04/01 13:00 |  0.0 |  20.0 |  20.0 | 10.715238 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 | 11.358152 |  8.17787 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.219304 | 0.0 | 0.108882 | 17.328187 | 17.809636 | 17.809636 | 4.947121 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.644437 | 21.667565 | 5.026671  5.032703  5.046271  5.069013   5.10013 | 4.947121 |
    | 04/01 14:00 |  0.0 |  20.0 |  20.0 |  9.383394 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  9.946398 | 7.161406 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.203144 | 0.0 | 0.110838 | 17.313982 | 17.454855 | 17.454855 | 4.848571 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641966 | 22.056727 | 5.021244  5.022078  5.024275  5.028911  5.037406 | 4.848571 |
    | 04/01 15:00 |  0.0 |  20.0 |  20.0 |  7.861915 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |   8.33363 | 6.000214 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.201001 | 0.0 | 0.112784 | 17.313784 | 17.339129 | 17.339129 | 4.816425 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641638 | 22.443943 | 5.021006  5.021062  5.021276   5.02189  5.023336 | 4.816425 |
    | 04/01 16:00 |  0.0 |  20.0 |  20.0 |  6.298329 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  6.676229 | 4.806885 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200716 | 0.0 |  0.11472 | 17.315436 |  17.31714 |  17.31714 | 4.810317 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641594 | 22.829224 | 5.021461   5.02141  5.021334  5.021288  5.021372 | 4.810317 |
    | 04/01 17:00 |  0.0 |  20.0 |  20.0 |  2.948416 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  3.125321 | 2.250231 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200678 | 0.0 | 0.116646 | 17.317324 |  17.31518 |  17.31518 | 4.809772 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641588 | 23.212577 | 5.022005   5.02194  5.021825  5.021685  5.021554 | 4.809772 |
    | 04/01 18:00 |  0.0 |  20.0 |  20.0 |  1.309232 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  1.387786 | 0.999206 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 | 0.118563 | 17.319236 | 17.316541 | 17.316541 |  4.81015 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 23.594015 | 5.022559  5.022493  5.022373   5.02222  5.022059 |  4.81015 |
    | 04/01 19:00 |  0.0 |  20.0 |  20.0 |   0.32955 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  0.349323 | 0.251513 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 |  0.12047 | 17.321143 | 17.318383 | 17.318383 | 4.810662 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 23.973545 | 5.023112  5.023046  5.022926  5.022772  5.022607 | 4.810662 |
    | 04/01 20:00 |  0.0 |  20.0 |  20.0 |  0.089508 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  0.094878 | 0.068313 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 | 0.122368 |  17.32304 | 17.320284 | 17.320284 |  4.81119 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 24.351177 | 5.023662  5.023596  5.023477  5.023324  5.023159 |  4.81119 |
    | 04/01 21:00 |  0.0 |  20.0 |  20.0 |  0.085771 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  0.090917 |  0.06546 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 | 0.124256 | 17.324928 | 17.322185 | 17.322185 | 4.811718 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 24.726921 |  5.02421  5.024144  5.024026  5.023873  5.023709 | 4.811718 |
    | 04/01 22:00 |  0.0 |  20.0 |  20.0 |    0.0845 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |   0.08957 |  0.06449 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 | 0.126135 | 17.326807 | 17.324077 | 17.324077 | 4.812244 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 25.100786 | 5.024755  5.024689  5.024571   5.02442  5.024257 | 4.812244 |
    | 04/01 23:00 |  0.0 |  20.0 |  20.0 |  0.084864 |  20.6 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |        1.0 |     0.0 |  0.089956 | 0.064768 | 0.0 |     0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 0.0 | 17.700673 |  0.5 | 17.200673 | 0.0 | 0.128004 | 17.328676 |  17.32596 |  17.32596 | 4.812767 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 25.641587 | 25.472782 | 5.025297  5.025232  5.025115  5.024964  5.024801 | 4.812767 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v2_multiple_zones:

multiple zones
______________

Besides slight differences in runoff concentration, the following calculation
reproduces the results of the :ref:`hland_v1_multiple_zones` experiment of |hland_v1|
exactly:

.. integration-test::

    >>> ttice(0.0)
    >>> resparea(False)
    >>> name2value = {par.name: par.value for par in control}
    >>> nmbzones(5)
    >>> sclass(1)
    >>> area(15.0)
    >>> zonearea(5.0, 4.0, 3.0, 2.0, 1.0)
    >>> zonetype(FIELD, FOREST, GLACIER, ILAKE, SEALED)
    >>> sfdist(1.0)
    >>> for name, value in name2value.items():
    ...     if name not in ("nmbzones", "sclass", "area", "zonearea", "zonetype", "sfdist"):
    ...         control[name].value = value
    >>> inputs.t.series = t_series
    >>> inputs.tn.series = tn_series
    >>> test("hland_v2_multiple_zones")
    |        date |    p |    t |   tn |       epn | tmean |                           tc |                     fracrain |                     rfc |                     sfc |                                       cfact |                     swe |                          gact | contriarea |                                          pc |                                                    ep |                                              epc |                                     ei |                                                tf |                     spl |                     wcl |                     spg |                     wcg |                           glmelt |                     melt |                     refr |                                                 in_ |                                                   r |                            sr |                                ea |                                cf |      inuz | perc |        q0 |                           el |       q1 |      inuh |     outuh |        rt |        qt |                                     ic |                      sp |                      wc |                                    sm |        uz |        lz |                                               sc |    outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 20.2 |  0.100707 |  21.8 | 21.8  21.8  21.8  21.8  21.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |   0.11682    0.11682    0.11682    0.11682    0.11682 |  0.08411   0.08411   0.08411   0.08411   0.08411 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.760624  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  18.760624      0.0        0.0 |       0.0        0.0  18.760624      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.052569  0.052569  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |  4.690156 |  0.5 |  0.018623 | 0.0  0.0  0.0   0.08411  0.0 | 0.052083 |  0.063509 |   0.16464 |   0.16464 |  0.686001 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.947431   99.947431  0.0  0.0  0.0 |  4.171533 | 10.364473 | 0.019482   0.02303  0.029003  0.035772  0.041582 |  0.686001 |
    | 01/01 01:00 |  0.0 | 19.4 | 18.4 |  0.097801 |  20.0 | 20.0  20.0  20.0  20.0  20.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.113449   0.113449   0.113449   0.113449   0.113449 | 0.081683  0.081683  0.081683  0.081683  0.081683 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.137109  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.137109      0.0        0.0 |       0.0        0.0  17.137109      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.051051  0.051051  0.0  0.0  0.0 | 0.050026  0.050026  0.0  0.0  0.0 |  4.246757 |  0.5 |   0.23242 | 0.0  0.0  0.0  0.081683  0.0 | 0.053907 |  0.236249 |  0.122442 |  0.122442 |  0.510173 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.946406   99.946406  0.0  0.0  0.0 |   7.68587 | 10.727468 |  0.06686  0.061471  0.052799   0.04386  0.037687 |  0.510173 |
    | 01/01 02:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.112498   0.112498   0.112498   0.112498   0.112498 | 0.080999  0.080999  0.080999  0.080999  0.080999 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.686132  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.686132      0.0        0.0 |       0.0        0.0  16.686132      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.050622  0.050622  0.0  0.0  0.0 | 0.050027  0.050027  0.0  0.0  0.0 |  4.134013 |  0.5 |  0.794235 | 0.0  0.0  0.0  0.080999  0.0 | 0.055722 |  0.687395 |  0.203554 |  0.203554 |  0.848142 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.945811   99.945811  0.0  0.0  0.0 | 10.525649 | 11.088746 | 0.194879  0.179814  0.153861  0.123171  0.094792 |  0.848142 |
    | 01/01 03:00 |  0.0 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.111348   0.111348   0.111348   0.111348   0.111348 | 0.080171  0.080171  0.080171  0.080171  0.080171 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   16.14496  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   16.14496      0.0        0.0 |       0.0        0.0   16.14496      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.050105  0.050105  0.0  0.0  0.0 | 0.050027  0.050027  0.0  0.0  0.0 |   3.99872 |  0.5 |  1.564786 | 0.0  0.0  0.0  0.080171  0.0 | 0.057529 |  1.305523 |  0.525212 |  0.525212 |  2.188382 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.945733   99.945733  0.0  0.0  0.0 | 12.459582 | 11.448335 |  0.37241  0.351262   0.31395  0.267774  0.221434 |  2.188382 |
    | 01/01 04:00 |  0.0 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.112498   0.112498   0.112498   0.112498   0.112498 | 0.080999  0.080999  0.080999  0.080999  0.080999 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.686132  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.686132      0.0        0.0 |       0.0        0.0  16.686132      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.050622  0.050622  0.0  0.0  0.0 | 0.050027  0.050027  0.0  0.0  0.0 |  4.134013 |  0.5 |  2.310317 | 0.0  0.0  0.0  0.080999  0.0 | 0.059327 |  1.903625 |  1.044753 |  1.044753 |  4.353137 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.945139   99.945139  0.0  0.0  0.0 | 13.783278 | 11.806009 | 0.545997  0.525102  0.487496  0.439246   0.38786 |  4.353137 |
    | 01/01 05:00 |  0.0 | 22.5 | 21.5 |  0.102761 |  23.1 | 23.1  23.1  23.1  23.1  23.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.119203   0.119203   0.119203   0.119203   0.119203 | 0.085826  0.085826  0.085826  0.085826  0.085826 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.933163  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.933163      0.0        0.0 |       0.0        0.0  19.933163      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.053639  0.053639  0.0  0.0  0.0 | 0.050027  0.050027  0.0  0.0  0.0 |   4.94577 |  0.5 |  3.097316 | 0.0  0.0  0.0  0.085826  0.0 | 0.061112 |   2.53489 |  1.637024 |  1.637024 |  6.820933 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.941527   99.941527  0.0  0.0  0.0 | 15.131732 | 12.161208 | 0.728744  0.706793  0.667413  0.617076  0.563542 |  6.820933 |
    | 01/01 06:00 |  0.0 | 25.1 | 24.1 |  0.291908 |  25.7 | 25.7  25.7  25.7  25.7  25.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.338613   0.338613   0.338613   0.338613   0.338613 | 0.243802  0.243802  0.243802  0.243802  0.243802 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.278241  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.278241      0.0        0.0 |       0.0        0.0  22.278241      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.152363  0.152363  0.0  0.0  0.0 | 0.050029  0.050029  0.0  0.0  0.0 |  5.532038 |  0.5 |  3.941177 | 0.0  0.0  0.0  0.243802  0.0 | 0.062775 |  3.211531 |   2.25712 |   2.25712 |  9.404666 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.839193   99.839193  0.0  0.0  0.0 | 16.222594 | 12.492176 | 0.924516  0.901015  0.858907  0.805212  0.748329 |  9.404666 |
    | 01/01 07:00 |  0.0 | 28.3 | 27.3 |  1.932875 |  28.9 | 28.9  28.9  28.9  28.9  28.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  2.242135   2.242135   2.242135   2.242135   2.242135 | 1.614337  1.614337  1.614337  1.614337  1.614337 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  25.164491  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  25.164491      0.0        0.0 |       0.0        0.0  25.164491      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.007844  1.007844  0.0  0.0  0.0 |  0.05008   0.05008  0.0  0.0  0.0 |  6.253562 |  0.5 |  4.774207 | 0.0  0.0  0.0  1.614337  0.0 | 0.063451 |  3.878586 |  2.910946 |  2.910946 | 12.128941 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   98.88143    98.88143  0.0  0.0  0.0 | 17.201949 | 12.626677 |  1.11804  1.094748  1.052814  0.998905  0.941112 | 12.128941 |
    | 01/01 08:00 |  0.0 | 27.8 | 26.8 |  4.369536 |  28.4 | 28.4  28.4  28.4  28.4  28.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  5.068662   5.068662   5.068662   5.068662   5.068662 | 3.649436  3.649436  3.649436  3.649436  3.649436 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  24.713515  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  24.713515      0.0        0.0 |       0.0        0.0  24.713515      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.256538  2.256538  0.0  0.0  0.0 | 0.050559  0.050559  0.0  0.0  0.0 |  6.140459 |  0.5 |  5.310752 | 0.0  0.0  0.0  3.649436  0.0 |  0.06267 |  4.307093 |     3.561 |     3.561 | 14.837501 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  96.675452   96.675452  0.0  0.0  0.0 | 17.531656 | 12.471231 | 1.244641  1.229137   1.20033  1.161317  1.116287 | 14.837501 |
    | 01/01 09:00 |  0.0 | 31.4 | 30.4 |  7.317556 |  32.0 | 32.0  32.0  32.0  32.0  32.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  8.488365   8.488365   8.488365   8.488365   8.488365 | 6.111623  6.111623  6.111623  6.111623  6.111623 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  27.960546  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  27.960546      0.0        0.0 |       0.0        0.0  27.960546      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.694748  3.694748  0.0  0.0  0.0 | 0.051662  0.051662  0.0  0.0  0.0 |   6.95139 |  0.5 |  5.803413 | 0.0  0.0  0.0  6.111623  0.0 | 0.060134 |  4.698855 |   4.08929 |   4.08929 | 17.038706 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  93.032367   93.032367  0.0  0.0  0.0 | 18.179633 |  11.96658 |  1.35869  1.344908  1.319875  1.287095  1.250711 | 17.038706 |
    | 01/01 10:00 |  0.0 | 32.2 | 31.2 |  8.264362 |  32.8 | 32.8  32.8  32.8  32.8  32.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |   9.58666    9.58666    9.58666    9.58666    9.58666 | 6.902395  6.902395  6.902395  6.902395  6.902395 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  28.682108  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  28.682108      0.0        0.0 |       0.0        0.0  28.682108      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.015721  4.015721  0.0  0.0  0.0 | 0.053484  0.053484  0.0  0.0  0.0 |  7.130414 |  0.5 |  6.279869 | 0.0  0.0  0.0  6.902395  0.0 | 0.057045 |  5.077138 |  4.515625 |  4.515625 | 18.815104 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   89.07013    89.07013  0.0  0.0  0.0 | 18.530178 | 11.352049 | 1.468539  1.455303  1.431412  1.400524  1.367014 | 18.815104 |
    | 01/01 11:00 |  0.0 | 35.2 | 34.2 |  9.369867 |  35.8 | 35.8  35.8  35.8  35.8  35.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 | 10.869046  10.869046  10.869046  10.869046  10.869046 | 7.825713  7.825713  7.825713  7.825713  7.825713 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  31.387967  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  31.387967      0.0        0.0 |       0.0        0.0  31.387967      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.359196  4.359196  0.0  0.0  0.0 | 0.055465  0.055465  0.0  0.0  0.0 |  7.805393 |  0.5 |  6.769495 | 0.0  0.0  0.0  7.825713  0.0 | 0.053313 |  5.465355 |  4.905749 |  4.905749 | 20.440623 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  84.766399   84.766399  0.0  0.0  0.0 | 19.066076 | 10.609348 | 1.581029  1.567505  1.543197  1.512018  1.478647 | 20.440623 |
    | 01/01 12:00 |  0.0 | 37.1 | 36.1 |  5.126178 |  37.7 | 37.7  37.7  37.7  37.7  37.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  5.946366   5.946366   5.946366   5.946366   5.946366 | 4.281384  4.281384  4.281384  4.281384  4.281384 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  33.101678  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  33.101678      0.0        0.0 |       0.0        0.0  33.101678      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.269776  2.269776  0.0  0.0  0.0 | 0.057617  0.057617  0.0  0.0  0.0 |  8.232207 |  0.5 |  7.299567 | 0.0  0.0  0.0  4.281384  0.0 | 0.052131 |   5.88831 |  5.293872 |  5.293872 | 22.057801 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   82.55424    82.55424  0.0  0.0  0.0 | 19.498715 | 10.374162 | 1.703344  1.688667   1.66239  1.628921  1.593512 | 22.057801 |
    | 01/01 13:00 |  0.0 | 31.2 | 30.2 |   6.62503 |  31.8 | 31.8  31.8  31.8  31.8  31.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  7.685035   7.685035   7.685035   7.685035   7.685035 | 5.533225  5.533225  5.533225  5.533225  5.533225 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  27.780155  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  27.780155      0.0        0.0 |       0.0        0.0  27.780155      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.856976  2.856976  0.0  0.0  0.0 | 0.058723  0.058723  0.0  0.0  0.0 |  6.900997 |  0.5 |  7.056489 | 0.0  0.0  0.0  5.533225  0.0 | 0.050061 |  5.691915 |  5.657107 |  5.657107 | 23.571281 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  79.755987   79.755987  0.0  0.0  0.0 | 18.843223 |  9.962211 | 1.652431  1.657856  1.665285  1.669722  1.666347 | 23.571281 |
    | 01/01 14:00 |  0.0 | 24.3 | 23.3 |  7.397619 |  24.9 | 24.9  24.9  24.9  24.9  24.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  8.581238   8.581238   8.581238   8.581238   8.581238 | 6.178491  6.178491  6.178491  6.178491  6.178491 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  21.556679  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  21.556679      0.0        0.0 |       0.0        0.0  21.556679      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.082145  3.082145  0.0  0.0  0.0 | 0.060122  0.060122  0.0  0.0  0.0 |  5.344078 |  0.5 |   6.03778 | 0.0  0.0  0.0  6.178491  0.0 | 0.047541 |  4.874596 |  5.678463 |  5.678463 | 23.660265 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  76.733964   76.733964  0.0  0.0  0.0 | 17.649521 |    9.4606 | 1.421681  1.448692  1.494712  1.547903  1.594786 | 23.660265 |
    | 01/01 15:00 |  0.2 | 25.4 | 24.4 |   2.39151 |  26.0 | 26.0  26.0  26.0  26.0  26.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  0.2376   0.2376   0.2376   0.2376   0.2376 |  2.774152   2.774152   2.774152   2.774152   2.774152 | 1.950491  1.950491  1.950491  1.950491  1.950491 |   0.2376    0.2376  0.0  0.0    0.2376 |       0.0        0.0   0.2376   0.2376        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.548827  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.786427   0.2376        0.0 |       0.0        0.0  22.786427   0.2376        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.936182  0.936182  0.0  0.0  0.0 | 0.061633  0.061633  0.0  0.0  0.0 |  5.650382 |  0.5 |  5.408841 | 0.0  0.0  0.0  1.950491  0.0 | 0.048222 |  4.372081 |  5.200278 |  5.200278 | 21.667826 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  75.859415   75.859415  0.0  0.0  0.0 | 17.391062 |  9.596251 | 1.273086  1.291273   1.32494   1.36997  1.420307 | 21.667826 |
    | 01/01 16:00 |  0.0 | 25.9 | 24.9 |  1.829834 |  26.5 | 26.5  26.5  26.5  26.5  26.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  2.122607   2.122607   2.122607   2.122607   2.122607 | 1.528277  1.528277  1.528277  1.528277  1.528277 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.999804  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.999804      0.0        0.0 |       0.0        0.0  22.999804      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.725182  0.725182  0.0  0.0  0.0 |  0.06207   0.06207  0.0  0.0  0.0 |  5.703398 |  0.5 |  5.271782 | 0.0  0.0  0.0  1.528277  0.0 | 0.049032 |  4.263189 |  4.648565 |  4.648565 | 19.369019 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  75.196304   75.196304  0.0  0.0  0.0 | 17.322678 |  9.757464 | 1.237564  1.242307  1.252437  1.269055  1.292838 | 19.369019 |
    | 01/01 17:00 |  0.0 | 23.7 | 22.7 |  1.136569 |  24.3 | 24.3  24.3  24.3  24.3  24.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |   1.31842    1.31842    1.31842    1.31842    1.31842 | 0.949262  0.949262  0.949262  0.949262  0.949262 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  21.015507  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  21.015507      0.0        0.0 |       0.0        0.0  21.015507      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.446502  0.446502  0.0  0.0  0.0 | 0.062402  0.062402  0.0  0.0  0.0 |  5.207075 |  0.5 |   5.06116 | 0.0  0.0  0.0  0.949262  0.0 | 0.050252 |   4.09583 |  4.347539 |  4.347539 | 18.114744 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  74.812204   74.812204  0.0  0.0  0.0 | 16.968593 | 10.000175 | 1.189468   1.19522  1.205507   1.21877  1.233527 | 18.114744 |
    | 01/01 18:00 |  1.3 | 21.6 | 20.6 |  0.750986 |  22.2 | 22.2  22.2  22.2  22.2  22.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  1.5444   1.5444   1.5444   1.5444   1.5444 |  0.871144   0.871144   0.871144   0.871144   0.871144 | 0.537465  0.537465  0.537465  0.537465  0.537465 | 0.537465  0.537465  0.0  0.0  0.537465 |       0.0        0.0   1.5444   1.5444        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.121405  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  20.665805   1.5444        0.0 |       0.0        0.0  20.665805   1.5444        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.125758  0.125758  0.0  0.0  0.0 | 0.062594  0.062594  0.0  0.0  0.0 |  5.119506 |  0.5 |  4.825479 | 0.0  0.0  0.0  0.537465  0.0 | 0.052863 |  3.909722 |  4.168768 |  4.168768 | 17.369866 | 1.006935  1.006935  0.0  0.0  1.006935 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   74.74904    74.74904  0.0  0.0  0.0 |  16.76262 | 10.519731 | 1.135695  1.142141  1.153658  1.168275  1.183678 | 17.369866 |
    | 01/01 19:00 |  5.6 | 21.2 | 20.2 |  0.223895 |  21.8 | 21.8  21.8  21.8  21.8  21.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  6.6528   6.6528   6.6528   6.6528   6.6528 |  0.259718   0.259718   0.259718   0.259718   0.259718 | 0.096141  0.096141  0.096141  0.096141  0.096141 | 0.096141  0.096141  0.0  0.0  0.096141 |  5.659735   5.659735   6.6528   6.6528   5.659735 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.760624  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.659735   5.659735  25.413424   6.6528   5.659735 |  0.790583   0.790583  25.413424   6.6528   5.659735 | 0.0  0.0  0.0  0.0   5.659735 | 0.023939  0.023939  0.0  0.0  0.0 | 0.060191  0.060191  0.0  0.0  0.0 |   6.90115 |  0.5 |  5.315038 | 0.0  0.0  0.0  0.096141  0.0 | 0.059425 |  4.684809 |  4.060087 |  4.060087 | 16.917031 | 1.903859  1.903859  0.0  0.0  1.903859 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  79.654445   79.654445  0.0  0.0  0.0 | 17.848732 | 11.825543 | 1.351082  1.326258  1.285217  1.240448  1.205163 | 16.917031 |
    | 01/01 20:00 |  2.9 | 20.4 | 19.4 |  0.099425 |  21.0 | 21.0  21.0  21.0  21.0  21.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  3.4452   3.4452   3.4452   3.4452   3.4452 |  0.115333   0.115333   0.115333   0.115333   0.115333 | 0.058839  0.058839  0.058839  0.058839  0.058839 | 0.058839  0.058839  0.0  0.0  0.058839 |  3.349059   3.349059   3.4452   3.4452   3.349059 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.039062  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.349059   3.349059  21.484262   3.4452   3.349059 |   0.53123    0.53123  21.484262   3.4452   3.349059 | 0.0  0.0  0.0  0.0   3.349059 | 0.015175  0.015175  0.0  0.0  0.0 | 0.058764  0.058764  0.0  0.0  0.0 |  5.725415 |  0.5 |  5.556446 | 0.0  0.0  0.0  0.058839  0.0 | 0.063689 |  4.727871 |  4.373377 |  4.373377 | 18.222402 | 1.941161  1.941161  0.0  0.0  1.941161 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  82.515863   82.515863  0.0  0.0  0.0 | 17.517701 | 12.674191 | 1.370409  1.367325   1.35937   1.34427  1.321289 | 18.222402 |
    | 01/01 21:00 |  4.9 | 19.8 | 18.8 |  0.098454 |  20.4 | 20.4  20.4  20.4  20.4  20.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  5.8212   5.8212   5.8212   5.8212   5.8212 |  0.114207   0.114207   0.114207   0.114207   0.114207 | 0.045942  0.045942  0.045942  0.045942  0.045942 | 0.045942  0.045942  0.0  0.0  0.045942 |  5.762361   5.762361   5.8212   5.8212   5.762361 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   17.49789  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.762361   5.762361   23.31909   5.8212   5.762361 |  0.980879   0.980879   23.31909   5.8212   5.762361 | 0.0  0.0  0.0  0.0   5.762361 | 0.012541  0.012541  0.0  0.0  0.0 | 0.056351  0.056351  0.0  0.0  0.0 |  6.523168 |  0.5 |  5.638796 | 0.0  0.0  0.0  0.045942  0.0 | 0.069639 |   4.96019 |  4.663783 |  4.663783 | 19.432428 | 1.954058  1.954058  0.0  0.0  1.954058 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  87.341155   87.341155  0.0  0.0  0.0 | 17.902074 | 13.858161 | 1.436162  1.428413   1.41493   1.39836  1.381205 | 19.432428 |
    | 01/01 22:00 | 10.6 | 19.6 | 18.6 |  0.098128 |  20.2 | 20.2  20.2  20.2  20.2  20.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 | 12.5928  12.5928  12.5928  12.5928  12.5928 |  0.113828   0.113828   0.113828   0.113828   0.113828 | 0.023264  0.023264  0.023264  0.023264  0.023264 | 0.023264  0.023264  0.0  0.0  0.023264 | 12.546858  12.546858  12.5928  12.5928  12.546858 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.317499  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 12.546858  12.546858  29.910299  12.5928  12.546858 |  2.392836   2.392836  29.910299  12.5928  12.546858 | 0.0  0.0  0.0  0.0  12.546858 | 0.007092  0.007092  0.0  0.0  0.0 | 0.051252  0.051252  0.0  0.0  0.0 |  9.233762 |  0.5 |  6.924651 | 0.0  0.0  0.0  0.023264  0.0 | 0.080412 |  6.451229 |  4.956285 |  4.956285 | 20.651188 | 1.976736  1.976736  0.0  0.0  1.976736 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  97.539338   97.539338  0.0  0.0  0.0 | 19.711185 | 16.001968 | 1.856206  1.807109  1.723577  1.626871  1.540252 | 20.651188 |
    | 01/01 23:00 |  0.1 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  0.1188   0.1188   0.1188   0.1188   0.1188 |   0.11307    0.11307    0.11307    0.11307    0.11307 | 0.080449  0.080449  0.080449  0.080449  0.080449 | 0.080449  0.080449  0.0  0.0  0.080449 |  0.095536   0.095536   0.1188   0.1188   0.095536 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.956718  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.095536   0.095536  17.075518   0.1188   0.095536 |  0.022723   0.022723  17.075518   0.1188   0.095536 | 0.0  0.0  0.0  0.0   0.095536 | 0.024553  0.024553  0.0  0.0  0.0 | 0.051194  0.051194  0.0  0.0  0.0 |  4.247526 |  0.5 |  6.190949 | 0.0  0.0  0.0  0.080449  0.0 |  0.08218 |  5.035829 |  5.691843 |  5.691843 | 23.716013 | 1.919551  1.919551  0.0  0.0  1.919551 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  97.638792   97.638792  0.0  0.0  0.0 | 17.267763 | 16.353838 | 1.473731  1.516544  1.583067  1.645973  1.678686 | 23.716013 |
    | 02/01 00:00 |  0.7 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.8316   0.8316   0.8316   0.8316   0.8316 |   0.11307    0.11307    0.11307    0.11307    0.11307 | 0.074914  0.074914  0.074914  0.074914  0.074914 | 0.074914  0.074914  0.0  0.0  0.074914 |  0.751151   0.751151   0.8316   0.8316   0.751151 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.751151   0.751151  17.794937   0.8316   0.751151 |  0.179024   0.179024  17.794937   0.8316   0.751151 | 0.0  0.0  0.0  0.0   0.751151 | 0.023004  0.023004  0.0  0.0  0.0 | 0.050895  0.050895  0.0  0.0  0.0 |  4.544832 |  0.5 |  4.808754 | 0.0  0.0  0.0  0.074914  0.0 | 0.084453 |  3.975902 |  5.440002 |  5.440002 | 22.666673 | 1.925086  1.925086  0.0  0.0  1.925086 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   98.23881    98.23881  0.0  0.0  0.0 | 16.503841 | 16.806055 | 1.163821  1.201295  1.268999  1.355429  1.444357 | 22.666673 |
    | 02/01 01:00 |  3.0 | 19.2 | 18.2 |  0.097474 |  19.8 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |   3.564    3.564    3.564    3.564    3.564 |   0.11307    0.11307    0.11307    0.11307    0.11307 | 0.057003  0.057003  0.057003  0.057003  0.057003 | 0.057003  0.057003  0.0  0.0  0.057003 |  3.489086   3.489086    3.564    3.564   3.489086 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.489086   3.489086  20.527337    3.564   3.489086 |  0.841817   0.841817  20.527337    3.564   3.489086 | 0.0  0.0  0.0  0.0   3.489086 |  0.01798   0.01798  0.0  0.0  0.0 | 0.049557  0.049557  0.0  0.0  0.0 |  5.726029 |  0.5 |  4.759355 | 0.0  0.0  0.0  0.057003  0.0 | 0.088678 |  4.122856 |  4.542518 |  4.542518 | 18.927159 | 1.942997  1.942997  0.0  0.0  1.942997 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 100.917655  100.917655  0.0  0.0  0.0 | 16.970515 | 17.646948 | 1.194556  1.192246  1.192723  1.203511  1.231203 | 18.927159 |
    | 02/01 02:00 |  2.1 | 18.9 | 17.9 |  0.096981 |  19.5 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  2.4948   2.4948   2.4948   2.4948   2.4948 |  0.112498   0.112498   0.112498   0.112498   0.112498 | 0.063115  0.063115  0.063115  0.063115  0.063115 | 0.063115  0.063115  0.0  0.0  0.063115 |  2.437797   2.437797   2.4948   2.4948   2.437797 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.692645  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.437797   2.437797  19.187445   2.4948   2.437797 |  0.620686   0.620686  19.187445   2.4948   2.437797 | 0.0  0.0  0.0  0.0   2.437797 | 0.020272  0.020272  0.0  0.0  0.0 | 0.048633  0.048633  0.0  0.0  0.0 |  5.225901 |  0.5 |  4.861905 | 0.0  0.0  0.0  0.063115  0.0 | 0.092115 |  4.138017 |  4.161304 |  4.161304 | 17.338768 | 1.936885  1.936885  0.0  0.0  1.936885 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 102.763127  102.763127  0.0  0.0  0.0 | 16.834511 | 18.330789 | 1.199841  1.199148  1.197871  1.196679  1.197413 | 17.338768 |
    | 02/01 03:00 | 10.4 | 18.7 | 17.7 |  0.096652 |  19.3 | 19.3  19.3  19.3  19.3  19.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 12.3552  12.3552  12.3552  12.3552  12.3552 |  0.112116   0.112116   0.112116   0.112116   0.112116 | 0.023465  0.023465  0.023465  0.023465  0.023465 | 0.023465  0.023465  0.0  0.0  0.023465 | 12.292085  12.292085  12.3552  12.3552  12.292085 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.512184  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 12.292085  12.292085  28.867384  12.3552  12.292085 |  3.245191   3.245191  28.867384  12.3552  12.292085 | 0.0  0.0  0.0  0.0  12.292085 | 0.008202  0.008202  0.0  0.0  0.0 | 0.044095  0.044095  0.0  0.0  0.0 |  9.617668 |  0.5 |  6.378841 | 0.0  0.0  0.0  0.023465  0.0 | 0.102605 |   6.01831 |  4.264975 |  4.264975 | 17.770728 | 1.976535  1.976535  0.0  0.0  1.976535 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 111.845915  111.845915  0.0  0.0  0.0 | 19.573338 | 20.418431 | 1.726926  1.665622  1.562345  1.445121  1.344274 | 17.770728 |
    | 02/01 04:00 |  3.5 | 18.5 | 17.5 |  0.096321 |  19.1 | 19.1  19.1  19.1  19.1  19.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |   4.158    4.158    4.158    4.158    4.158 |  0.111732   0.111732   0.111732   0.111732   0.111732 |  0.05308   0.05308   0.05308   0.05308   0.05308 |  0.05308   0.05308  0.0  0.0   0.05308 |  4.134535   4.134535    4.158    4.158   4.134535 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.331723  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  4.134535   4.134535  20.489723    4.158   4.134535 |  1.293025   1.293025  20.489723    4.158   4.134535 | 0.0  0.0  0.0  0.0   4.134535 | 0.019031  0.019031  0.0  0.0  0.0 | 0.042656  0.042656  0.0  0.0  0.0 |  6.060207 |  0.5 |  6.777964 | 0.0  0.0  0.0   0.05308  0.0 | 0.107167 |   5.79803 |  5.196277 |  5.196277 | 21.651156 |  1.94692   1.94692  0.0  0.0   1.94692 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  114.71105   114.71105  0.0  0.0  0.0 | 18.355581 | 21.326253 | 1.682962  1.686007  1.684243  1.666848  1.625979 | 21.651156 |
    | 02/01 05:00 |  3.4 | 18.3 | 17.3 |   0.09599 |  18.9 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.0392   4.0392   4.0392   4.0392   4.0392 |  0.111348   0.111348   0.111348   0.111348   0.111348 |  0.05353   0.05353   0.05353   0.05353   0.05353 |  0.05353   0.05353  0.0  0.0   0.05353 |   3.98612    3.98612   4.0392   4.0392    3.98612 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.151262  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   3.98612    3.98612  20.190462   4.0392    3.98612 |  1.311296   1.311296  20.190462   4.0392    3.98612 | 0.0  0.0  0.0  0.0    3.98612 | 0.019643  0.019643  0.0  0.0  0.0 | 0.041307  0.041307  0.0  0.0  0.0 |  6.000108 |  0.5 |  5.969917 | 0.0  0.0  0.0   0.05353  0.0 | 0.111621 |  5.145855 |  5.689788 |  5.689788 | 23.707452 |  1.94647   1.94647  0.0  0.0   1.94647 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 117.407537  117.407537  0.0  0.0  0.0 | 17.885771 | 22.212584 | 1.498724  1.520247  1.556587  1.597243  1.629306 | 23.707452 |
    | 02/01 06:00 |  1.2 | 18.5 | 17.5 |  0.187298 |  19.1 | 19.1  19.1  19.1  19.1  19.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  1.4256   1.4256   1.4256   1.4256   1.4256 |  0.217266   0.217266   0.217266   0.217266   0.217266 | 0.135647  0.135647  0.135647  0.135647  0.135647 | 0.135647  0.135647  0.0  0.0  0.135647 |   1.37207    1.37207   1.4256   1.4256    1.37207 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.331723  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   1.37207    1.37207  17.757323   1.4256    1.37207 |  0.472834   0.472834  17.757323   1.4256    1.37207 | 0.0  0.0  0.0  0.0    1.37207 | 0.050167  0.050167  0.0  0.0  0.0 | 0.040847  0.040847  0.0  0.0  0.0 |  4.763321 |  0.5 |   5.24437 | 0.0  0.0  0.0  0.135647  0.0 | 0.114127 |  4.393486 |  5.371205 |  5.371205 | 22.380019 | 1.864353  1.864353  0.0  0.0  1.864353 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 118.297453  118.297453  0.0  0.0  0.0 | 16.904722 | 22.711308 | 1.281681  1.307641  1.353761  1.411425   1.46988 | 22.380019 |
    | 02/01 07:00 |  0.1 | 18.8 | 17.8 |  1.264612 |  19.4 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.1188   0.1188   0.1188   0.1188   0.1188 |   1.46695    1.46695    1.46695    1.46695    1.46695 |  1.04373   1.04373   1.04373   1.04373   1.04373 |  1.04373   1.04373  0.0  0.0   1.04373 |       0.0        0.0   0.1188   0.1188        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.721215   0.1188        0.0 |       0.0        0.0  16.721215   0.1188        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.385979  0.385979  0.0  0.0  0.0 | 0.040851  0.040851  0.0  0.0  0.0 |  4.149665 |  0.5 |  4.475785 | 0.0  0.0  0.0   1.04373  0.0 | 0.115039 |  3.687997 |  4.719636 |  4.719636 |  19.66515 | 0.939422  0.939422  0.0  0.0  0.939422 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 117.952325  117.952325  0.0  0.0  0.0 | 16.078602 | 22.892707 |  1.07667   1.10138  1.145974   1.20348  1.265245 |  19.66515 |
    | 02/01 08:00 |  0.0 | 18.8 | 17.8 |  3.045538 |  19.4 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  3.532824   3.532824   3.532824   3.532824   3.532824 | 2.543633  2.543633  2.543633  2.543633  2.543633 | 0.939422  0.939422  0.0  0.0  0.939422 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.602415      0.0        0.0 |       0.0        0.0  16.602415      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.740017  1.740017  0.0  0.0  0.0 | 0.041024  0.041024  0.0  0.0  0.0 |  4.119836 |  0.5 |  4.015724 | 0.0  0.0  0.0  2.543633  0.0 |  0.11479 |  3.319716 |  4.032324 |  4.032324 | 16.801351 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 116.253332  116.253332  0.0  0.0  0.0 | 15.682714 | 22.843113 | 0.966558  0.980193  1.006022  1.042066  1.085301 | 16.801351 |
    | 02/01 09:00 |  0.0 | 19.0 | 18.0 |  1.930758 |  19.6 | 19.6  19.6  19.6  19.6  19.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  2.239679   2.239679   2.239679   2.239679   2.239679 | 1.612569  1.612569  1.612569  1.612569  1.612569 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.782876  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.782876      0.0        0.0 |       0.0        0.0  16.782876      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.172088  1.172088  0.0  0.0  0.0 | 0.041873  0.041873  0.0  0.0  0.0 |  4.164314 |  0.5 |  3.820377 | 0.0  0.0  0.0  1.612569  0.0 | 0.115207 |  3.163827 |  3.532487 |  3.532487 | 14.718698 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 115.123117  115.123117  0.0  0.0  0.0 | 15.526652 | 22.926111 | 0.919163  0.925133  0.936811  0.954018  0.976355 | 14.718698 |
    | 02/01 10:00 |  0.4 | 19.2 | 18.2 |  2.461001 |  19.8 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.4752   0.4752   0.4752   0.4752   0.4752 |  2.854761   2.854761   2.854761   2.854761   2.854761 | 1.960038  1.960038  1.960038  1.960038  1.960038 |   0.4752    0.4752  0.0  0.0    0.4752 |       0.0        0.0   0.4752   0.4752        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.438537   0.4752        0.0 |       0.0        0.0  17.438537   0.4752        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.410806  1.410806  0.0  0.0  0.0 | 0.042438  0.042438  0.0  0.0  0.0 |  4.327805 |  0.5 |  3.787265 | 0.0  0.0  0.0  1.960038  0.0 | 0.115713 |  3.137811 |  3.266716 |  3.266716 | 13.611315 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  113.75475   113.75475  0.0  0.0  0.0 | 15.567192 |  23.02685 | 0.910275   0.91151  0.914313  0.919306  0.927172 | 13.611315 |
    | 02/01 11:00 |  0.1 | 19.3 | 18.3 |  6.215945 |  19.9 | 19.9  19.9  19.9  19.9  19.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.1188   0.1188   0.1188   0.1188   0.1188 |  7.210496   7.210496   7.210496   7.210496   7.210496 | 5.130246  5.130246  5.130246  5.130246  5.130246 |   0.1188    0.1188  0.0  0.0    0.1188 |       0.0        0.0   0.1188   0.1188        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.053567  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.172367   0.1188        0.0 |       0.0        0.0  17.172367   0.1188        0.0 | 0.0  0.0  0.0  0.0        0.0 |  3.64882   3.64882  0.0  0.0  0.0 | 0.043123  0.043123  0.0  0.0  0.0 |   4.26075 |  0.5 |  3.788625 | 0.0  0.0  0.0  5.130246  0.0 | 0.113698 |  3.137018 |  3.166026 |  3.166026 | 13.191777 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 110.149053  110.149053  0.0  0.0  0.0 | 15.539317 | 22.625803 | 0.909753  0.909856  0.910191  0.911023  0.912743 | 13.191777 |
    | 02/01 12:00 |  3.6 | 19.0 | 18.0 |  3.374783 |  19.6 | 19.6  19.6  19.6  19.6  19.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.2768   4.2768   4.2768   4.2768   4.2768 |  3.914748   3.914748   3.914748   3.914748   3.914748 | 1.837796  1.837796  1.837796  1.837796  1.837796 | 1.837796  1.837796  0.0  0.0  1.837796 |    2.2768     2.2768   4.2768   4.2768     2.2768 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.782876  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |    2.2768     2.2768  21.059676   4.2768     2.2768 |    0.6906     0.6906  21.059676   4.2768     2.2768 | 0.0  0.0  0.0  0.0     2.2768 | 0.641962  0.641962  0.0  0.0  0.0 | 0.044132  0.044132  0.0  0.0  0.0 |  5.749769 |  0.5 |  4.227527 | 0.0  0.0  0.0  1.837796  0.0 | 0.117014 |  3.643021 |  3.177505 |  3.177505 | 13.239605 | 0.162204  0.162204  0.0  0.0  0.162204 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 111.137424  111.137424  0.0  0.0  0.0 |  16.56156 | 23.285789 | 1.051531  1.035051  1.007326  0.975965  0.949209 | 13.239605 |
    | 02/01 13:00 |  5.9 | 18.8 | 17.8 |  8.821555 |  19.4 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  7.0092   7.0092   7.0092   7.0092   7.0092 | 10.233004  10.233004  10.233004  10.233004  10.233004 | 3.655358  3.655358  3.655358  3.655358  3.655358 |      2.0       2.0  0.0  0.0       2.0 |  5.171404   5.171404   7.0092   7.0092   5.171404 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.171404   5.171404  23.611615   7.0092   5.171404 |  1.596868   1.596868  23.611615   7.0092   5.171404 | 0.0  0.0  0.0  0.0   5.171404 |  2.13852   2.13852  0.0  0.0  0.0 | 0.042644  0.042644  0.0  0.0  0.0 |  7.068572 |  0.5 |  5.253883 | 0.0  0.0  0.0  3.655358  0.0 | 0.120967 |   4.66077 |  3.499616 |  3.499616 | 14.581735 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 112.616083  112.616083  0.0  0.0  0.0 | 17.876249 | 24.072514 | 1.341509   1.30724  1.247702  1.176096  1.107689 | 14.581735 |
    | 02/01 14:00 |  1.1 | 18.7 | 17.7 |  4.046025 |  19.3 | 19.3  19.3  19.3  19.3  19.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  1.3068   1.3068   1.3068   1.3068   1.3068 |  4.693389   4.693389   4.693389   4.693389   4.693389 | 2.965278  2.965278  2.965278  2.965278  2.965278 |   1.3068    1.3068  0.0  0.0    1.3068 |       0.0        0.0   1.3068   1.3068        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.512184  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.818984   1.3068        0.0 |       0.0        0.0  17.818984   1.3068        0.0 | 0.0  0.0  0.0  0.0        0.0 |   1.8732    1.8732  0.0  0.0  0.0 | 0.043692  0.043692  0.0  0.0  0.0 |  4.421977 |  0.5 |  5.123278 | 0.0  0.0  0.0  2.965278  0.0 | 0.121321 |  4.211855 |   4.15134 |   4.15134 | 17.297248 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 110.786575  110.786575  0.0  0.0  0.0 | 16.674948 | 24.142839 | 1.225485  1.237819   1.25467  1.264822  1.257956 | 17.297248 |
    | 02/01 15:00 | 20.7 | 17.8 | 16.8 |  2.110757 |  18.4 | 18.4  18.4  18.4  18.4  18.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 24.5916  24.5916  24.5916  24.5916  24.5916 |  2.448478   2.448478   2.448478   2.448478   2.448478 |  0.15074   0.15074   0.15074   0.15074   0.15074 |  0.15074   0.15074  0.0  0.0   0.15074 |   22.5916    22.5916  24.5916  24.5916    22.5916 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   15.70011  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   22.5916    22.5916   40.29171  24.5916    22.5916 |  6.932043   6.932043   40.29171  24.5916    22.5916 | 0.0  0.0  0.0  0.0    22.5916 | 0.059581  0.059581  0.0  0.0  0.0 | 0.036777  0.036777  0.0  0.0  0.0 | 15.244377 |  0.5 |  8.705784 | 0.0  0.0  0.0   0.15074  0.0 | 0.140315 |  8.601694 |  4.628088 |  4.628088 | 19.283701 |  1.84926   1.84926  0.0  0.0   1.84926 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 126.423327  126.423327  0.0  0.0  0.0 | 22.713541 | 27.922647 | 2.451721  2.309571  2.071538  1.804121  1.577407 | 19.283701 |
    | 02/01 16:00 | 37.9 | 17.4 | 16.4 |  2.239257 |  18.0 | 18.0  18.0  18.0  18.0  18.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 45.0252  45.0252  45.0252  45.0252  45.0252 |  2.597538   2.597538   2.597538   2.597538   2.597538 | 0.020724  0.020724  0.020724  0.020724  0.020724 | 0.020724  0.020724  0.0  0.0  0.020724 |  44.87446   44.87446  45.0252  45.0252   44.87446 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.339188  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  44.87446   44.87446  60.364388  45.0252   44.87446 | 17.930553  17.930553  60.364388  45.0252   44.87446 | 0.0  0.0  0.0  0.0   44.87446 | 0.009934  0.009934  0.0  0.0  0.0 | 0.023316  0.023316  0.0  0.0  0.0 | 28.521524 |  0.5 | 21.215224 | 0.0  0.0  0.0  0.020724  0.0 | 0.173902 | 20.126118 |  7.572683 |  7.572683 | 31.552846 | 1.979276  1.979276  0.0  0.0  1.979276 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 153.380617  153.380617  0.0  0.0  0.0 | 29.519841 | 34.606527 | 5.722493  5.337431  4.673317  3.885207  3.149344 | 31.552846 |
    | 02/01 17:00 |  8.2 | 17.3 | 16.3 |  2.877848 |  17.9 | 17.9  17.9  17.9  17.9  17.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  9.7416   9.7416   9.7416   9.7416   9.7416 |  3.338304   3.338304   3.338304   3.338304   3.338304 | 0.907373  0.907373  0.907373  0.907373  0.907373 | 0.907373  0.907373  0.0  0.0  0.907373 |  9.720876   9.720876   9.7416   9.7416   9.720876 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.248957  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  9.720876   9.720876  24.990557   9.7416   9.720876 |  5.717239   5.717239  24.990557   9.7416   9.720876 | 0.0  0.0  0.0  0.0   9.720876 |  0.44633   0.44633  0.0  0.0  0.0 | 0.021308  0.021308  0.0  0.0  0.0 | 10.519588 |  0.5 | 16.566572 | 0.0  0.0  0.0  0.907373  0.0 | 0.181486 | 14.070702 | 14.402196 | 14.402196 |  60.00915 | 1.092627  1.092627  0.0  0.0  1.092627 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 156.959231  156.959231  0.0  0.0  0.0 | 22.972857 | 36.115645 | 4.135844  4.307335  4.552216  4.730149  4.710755 |  60.00915 |
    | 02/01 18:00 |  3.6 | 16.8 | 15.8 |  1.591452 |  17.4 | 17.4  17.4  17.4  17.4  17.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  4.2768   4.2768   4.2768   4.2768   4.2768 |  1.846084   1.846084   1.846084   1.846084   1.846084 | 0.866652  0.866652  0.866652  0.866652  0.866652 | 0.866652  0.866652  0.0  0.0  0.866652 |  3.369427   3.369427   4.2768   4.2768   3.369427 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.797804  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.369427   3.369427  19.074604   4.2768   3.369427 |  2.075247   2.075247  19.074604   4.2768   3.369427 | 0.0  0.0  0.0  0.0   3.369427 | 0.428652  0.428652  0.0  0.0  0.0 | 0.020873  0.020873  0.0  0.0  0.0 |  6.309431 |  0.5 |  9.257302 | 0.0  0.0  0.0  0.866652  0.0 | 0.185157 |  7.803283 | 15.217663 | 15.217663 | 63.406929 | 1.133348  1.133348  0.0  0.0  1.133348 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 157.845632  157.845632  0.0  0.0  0.0 | 19.524987 | 36.846224 | 2.326075  2.542265  2.923974  3.391328  3.838278 | 63.406929 |
    | 02/01 19:00 |  7.5 | 16.5 | 15.5 |  0.291604 |  17.1 | 17.1  17.1  17.1  17.1  17.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |    8.91     8.91     8.91     8.91     8.91 |  0.338261   0.338261   0.338261   0.338261   0.338261 | 0.099914  0.099914  0.099914  0.099914  0.099914 | 0.099914  0.099914  0.0  0.0  0.099914 |  8.043348   8.043348     8.91     8.91   8.043348 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.527113  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  8.043348   8.043348  23.437113     8.91   8.043348 |  5.010049   5.010049  23.437113     8.91   8.043348 | 0.0  0.0  0.0  0.0   8.043348 | 0.049957  0.049957  0.0  0.0  0.0 | 0.019561  0.019561  0.0  0.0  0.0 |  9.602145 |  0.5 |  8.193251 | 0.0  0.0  0.0  0.099914  0.0 | 0.192667 |  7.270647 | 10.833204 | 10.833204 | 45.138352 | 1.900086  1.900086  0.0  0.0  1.900086 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 160.848534  160.848534  0.0  0.0  0.0 |  20.43388 | 38.340712 | 2.115821  2.147552  2.225854  2.372008  2.598127 | 45.138352 |
    | 02/01 20:00 | 18.5 | 16.3 | 15.3 |  0.092622 |  16.9 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  21.978   21.978   21.978   21.978   21.978 |  0.107442   0.107442   0.107442   0.107442   0.107442 |  0.00859   0.00859   0.00859   0.00859   0.00859 |  0.00859   0.00859  0.0  0.0   0.00859 | 21.878086  21.878086   21.978   21.978  21.878086 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.346652  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 21.878086  21.878086  36.324652   21.978  21.878086 | 14.150883  14.150883  36.324652   21.978  21.878086 | 0.0  0.0  0.0  0.0  21.878086 | 0.004295  0.004295  0.0  0.0  0.0 | 0.015712  0.015712  0.0  0.0  0.0 | 19.682541 |  0.5 | 13.910461 | 0.0  0.0  0.0   0.00859  0.0 | 0.209539 | 12.782477 |  8.459834 |  8.459834 | 35.249308 |  1.99141   1.99141  0.0  0.0   1.99141 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 168.587153  168.587153  0.0  0.0  0.0 |  25.70596 | 41.698232 | 3.653293  3.475611  3.180916  2.859621  2.612564 | 35.249308 |
    | 02/01 21:00 | 15.4 | 16.2 | 15.2 |  0.092451 |  16.8 | 16.8  16.8  16.8  16.8  16.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 18.2952  18.2952  18.2952  18.2952  18.2952 |  0.107243   0.107243   0.107243   0.107243   0.107243 | 0.012392  0.012392  0.012392  0.012392  0.012392 | 0.012392  0.012392  0.0  0.0  0.012392 |  18.28661   18.28661  18.2952  18.2952   18.28661 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.256421  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  18.28661   18.28661  32.551621  18.2952   18.28661 | 12.993381  12.993381  32.551621  18.2952   18.28661 | 0.0  0.0  0.0  0.0   18.28661 | 0.006196  0.006196  0.0  0.0  0.0 |  0.01306   0.01306  0.0  0.0  0.0 | 17.873146 |  0.5 | 17.355859 | 0.0  0.0  0.0  0.012392  0.0 | 0.223693 | 15.312575 | 10.713847 | 10.713847 | 44.641031 | 1.987608  1.987608  0.0  0.0  1.987608 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 173.887246  173.887246  0.0  0.0  0.0 | 25.723248 | 44.514939 |  4.41411  4.319667  4.140873  3.894582    3.6115 | 44.641031 |
    | 02/01 22:00 |  6.3 | 15.5 | 14.5 |  0.091248 |  16.1 | 16.1  16.1  16.1  16.1  16.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  7.4844   7.4844   7.4844   7.4844   7.4844 |  0.105848   0.105848   0.105848   0.105848   0.105848 | 0.036055  0.036055  0.036055  0.036055  0.036055 | 0.036055  0.036055  0.0  0.0  0.036055 |  7.472008   7.472008   7.4844   7.4844   7.472008 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.624808  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  7.472008   7.472008  21.109208   7.4844   7.472008 |  5.648235   5.648235  21.109208   7.4844   7.472008 | 0.0  0.0  0.0  0.0   7.472008 | 0.018028  0.018028  0.0  0.0  0.0 | 0.012144  0.012144  0.0  0.0  0.0 |   9.50437 |  0.5 |  12.87116 | 0.0  0.0  0.0  0.036055  0.0 | 0.230038 | 11.009764 |  13.47705 |  13.47705 | 56.154377 | 1.963945  1.963945  0.0  0.0  1.963945 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 175.705135  175.705135  0.0  0.0  0.0 | 21.856458 | 45.777522 | 3.233993  3.368014  3.582338  3.797846  3.931255 | 56.154377 |
    | 02/01 23:00 |  1.9 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2  15.2  15.2  15.2  15.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  2.2572   2.2572   2.2572   2.2572   2.2572 |  0.104032   0.104032   0.104032   0.104032   0.104032 | 0.059768  0.059768  0.059768  0.059768  0.059768 | 0.059768  0.059768  0.0  0.0  0.059768 |  2.221145   2.221145   2.2572   2.2572   2.221145 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.812733  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.221145   2.221145  15.069933   2.2572   2.221145 |  1.714296   1.714296  15.069933   2.2572   2.221145 | 0.0  0.0  0.0  0.0   2.221145 | 0.029884  0.029884  0.0  0.0  0.0 | 0.011894  0.011894  0.0  0.0  0.0 |  5.044285 |  0.5 |  7.929774 | 0.0  0.0  0.0  0.059768  0.0 |   0.2326 |  6.708989 | 12.281816 | 12.281816 | 51.174232 | 1.940232  1.940232  0.0  0.0  1.940232 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 176.193994  176.193994  0.0  0.0  0.0 | 18.470968 | 46.287412 |  1.98903   2.13829  2.404222  2.737047   3.07203 | 51.174232 |
    | 03/01 00:00 |  4.9 | 14.7 | 13.7 |  0.089858 |  15.3 | 15.3  15.3  15.3  15.3  15.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  5.8212   5.8212   5.8212   5.8212   5.8212 |  0.104235   0.104235   0.104235   0.104235   0.104235 | 0.041931  0.041931  0.041931  0.041931  0.041931 | 0.041931  0.041931  0.0  0.0  0.041931 |  5.761432   5.761432   5.8212   5.8212   5.761432 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   12.90841  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.761432   5.761432   18.72961   5.8212   5.761432 |  4.471494   4.471494   18.72961   5.8212   5.761432 | 0.0  0.0  0.0  0.0   5.761432 | 0.020966  0.020966  0.0  0.0  0.0 | 0.011258  0.011258  0.0  0.0  0.0 |  8.027579 |  0.5 |  6.818281 | 0.0  0.0  0.0  0.041931  0.0 | 0.237708 |  6.060581 |  8.830301 |  8.830301 | 36.792921 | 1.958069  1.958069  0.0  0.0  1.958069 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 177.474225  177.474225  0.0  0.0  0.0 | 19.180266 | 47.303886 | 1.765369  1.796404  1.865908  1.984837  2.158381 | 36.792921 |
    | 03/01 01:00 |  2.7 | 14.6 | 13.6 |  0.089683 |  15.2 | 15.2  15.2  15.2  15.2  15.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  3.2076   3.2076   3.2076   3.2076   3.2076 |  0.104032   0.104032   0.104032   0.104032   0.104032 |  0.05435   0.05435   0.05435   0.05435   0.05435 |  0.05435   0.05435  0.0  0.0   0.05435 |  3.165669   3.165669   3.2076   3.2076   3.165669 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.818141  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.165669   3.165669  16.025741   3.2076   3.165669 |  2.492735   2.492735  16.025741   3.2076   3.165669 | 0.0  0.0  0.0  0.0   3.165669 | 0.027175  0.027175  0.0  0.0  0.0 | 0.010926  0.010926  0.0  0.0  0.0 |  5.867792 |  0.5 |  6.447511 | 0.0  0.0  0.0   0.05435  0.0 | 0.240915 |  5.593907 |  6.700253 |  6.700253 | 27.917722 |  1.94565   1.94565  0.0  0.0   1.94565 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  178.13091   178.13091  0.0  0.0  0.0 | 18.100547 | 47.942007 | 1.627057  1.644184  1.677082  1.725446  1.790784 | 27.917722 |
    | 03/01 02:00 |  0.5 | 14.1 | 13.1 |  0.088805 |  14.7 | 14.7  14.7  14.7  14.7  14.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |   0.594    0.594    0.594    0.594    0.594 |  0.103014   0.103014   0.103014   0.103014   0.103014 | 0.069893  0.069893  0.069893  0.069893  0.069893 | 0.069893  0.069893  0.0  0.0  0.069893 |   0.53965    0.53965    0.594    0.594    0.53965 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.366798  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   0.53965    0.53965  12.960798    0.594    0.53965 |  0.428086   0.428086  12.960798    0.594    0.53965 | 0.0  0.0  0.0  0.0    0.53965 | 0.034946  0.034946  0.0  0.0  0.0 | 0.010879  0.010879  0.0  0.0  0.0 |  3.553105 |  0.5 |  4.967601 | 0.0  0.0  0.0  0.069893  0.0 | 0.242227 |  4.236136 |  5.795639 |  5.795639 | 24.148496 | 1.930107  1.930107  0.0  0.0  1.930107 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 178.218407  178.218407  0.0  0.0  0.0 | 16.186051 | 48.203224 | 1.241913  1.287268  1.365678  1.459669  1.550522 | 24.148496 |
    | 03/01 03:00 |  0.2 | 14.3 | 13.3 |  0.089157 |  14.9 | 14.9  14.9  14.9  14.9  14.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.2376   0.2376   0.2376   0.2376   0.2376 |  0.103422   0.103422   0.103422   0.103422   0.103422 | 0.072716  0.072716  0.072716  0.072716  0.072716 | 0.072716  0.072716  0.0  0.0  0.072716 |  0.167707   0.167707   0.2376   0.2376   0.167707 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.547335  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.167707   0.167707  12.784935   0.2376   0.167707 |  0.133167   0.133167  12.784935   0.2376   0.167707 | 0.0  0.0  0.0  0.0   0.167707 | 0.036358  0.036358  0.0  0.0  0.0 | 0.010874  0.010874  0.0  0.0  0.0 |  3.287954 |  0.5 |  3.827142 | 0.0  0.0  0.0  0.072716  0.0 | 0.243277 |  3.299953 |  4.799205 |  4.799205 | 19.996688 | 1.927284  1.927284  0.0  0.0  1.927284 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 178.227463  178.227463  0.0  0.0  0.0 | 15.146863 | 48.412073 | 0.966589  1.000128   1.06175  1.143372  1.233959 | 19.996688 |
    | 03/01 04:00 |  0.5 | 14.9 | 13.9 |  0.090207 |  15.5 | 15.5  15.5  15.5  15.5  15.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |   0.594    0.594    0.594    0.594    0.594 |   0.10464    0.10464    0.10464    0.10464    0.10464 | 0.070996  0.070996  0.070996  0.070996  0.070996 | 0.070996  0.070996  0.0  0.0  0.070996 |  0.521284   0.521284    0.594    0.594   0.521284 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.088947  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.521284   0.521284  13.682947    0.594   0.521284 |  0.413965   0.413965  13.682947    0.594   0.521284 | 0.0  0.0  0.0  0.0   0.521284 | 0.035498  0.035498  0.0  0.0  0.0 | 0.010833  0.010833  0.0  0.0  0.0 |  3.723086 |  0.5 |  3.422146 | 0.0  0.0  0.0  0.070996  0.0 | 0.244577 |  3.000741 |  3.797043 |  3.797043 | 15.821013 | 1.929004  1.929004  0.0  0.0  1.929004 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 178.310117  178.310117  0.0  0.0  0.0 | 14.947803 | 48.670783 | 0.873463  0.885421  0.909491  0.946163  0.994958 | 15.821013 |
    | 03/01 05:00 |  2.4 | 15.7 | 14.7 |  0.091593 |  16.3 | 16.3  16.3  16.3  16.3  16.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  2.8512   2.8512   2.8512   2.8512   2.8512 |  0.106248   0.106248   0.106248   0.106248   0.106248 | 0.057521  0.057521  0.057521  0.057521  0.057521 | 0.057521  0.057521  0.0  0.0  0.057521 |  2.780204   2.780204   2.8512   2.8512   2.780204 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.811096  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.780204   2.780204  16.662296   2.8512   2.780204 |   2.20988    2.20988  16.662296   2.8512   2.780204 | 0.0  0.0  0.0  0.0   2.780204 | 0.028761  0.028761  0.0  0.0  0.0 |  0.01056   0.01056  0.0  0.0  0.0 |  5.815064 |  0.5 |  3.930692 | 0.0  0.0  0.0  0.057521  0.0 | 0.247492 |  3.560893 |  3.259948 |  3.259948 | 13.583115 | 1.942479  1.942479  0.0  0.0  1.942479 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  178.86224   178.86224  0.0  0.0  0.0 | 16.332175 | 49.250959 | 1.027294  1.009812   0.98186  0.953941  0.937534 | 13.583115 |
    | 03/01 06:00 |  0.4 | 16.0 | 15.0 |  0.154861 |  16.6 | 16.6  16.6  16.6  16.6  16.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.4752   0.4752   0.4752   0.4752   0.4752 |  0.179639   0.179639   0.179639   0.179639   0.179639 | 0.123337  0.123337  0.123337  0.123337  0.123337 | 0.123337  0.123337  0.0  0.0  0.123337 |  0.417679   0.417679   0.4752   0.4752   0.417679 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.081902  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.417679   0.417679  14.557102   0.4752   0.417679 |  0.334057   0.334057  14.557102   0.4752   0.417679 | 0.0  0.0  0.0  0.0   0.417679 | 0.061669  0.061669  0.0  0.0  0.0 | 0.010527  0.010527  0.0  0.0  0.0 |  3.881922 |  0.5 |  4.079736 | 0.0  0.0  0.0  0.123337  0.0 | 0.248649 |  3.523706 |  3.353603 |  3.353603 | 13.973345 | 1.876663  1.876663  0.0  0.0  1.876663 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 178.894721  178.894721  0.0  0.0  0.0 | 15.634362 | 49.481147 | 1.022057  1.022077  1.020196   1.01397  1.002244 | 13.973345 |
    | 03/01 07:00 |  0.2 | 16.7 | 15.7 |  0.470369 |  17.3 | 17.3  17.3  17.3  17.3  17.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.2376   0.2376   0.2376   0.2376   0.2376 |  0.545628   0.545628   0.545628   0.545628   0.545628 | 0.383628  0.383628  0.383628  0.383628  0.383628 | 0.383628  0.383628  0.0  0.0  0.383628 |  0.114263   0.114263   0.2376   0.2376   0.114263 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.713782  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.114263   0.114263  14.951382   0.2376   0.114263 |   0.09142    0.09142  14.951382   0.2376   0.114263 | 0.0  0.0  0.0  0.0   0.114263 | 0.191814  0.191814  0.0  0.0  0.0 | 0.010541  0.010541  0.0  0.0  0.0 |  3.798504 |  0.5 |  3.689107 | 0.0  0.0  0.0  0.383628  0.0 | 0.249444 |  3.191718 |  3.473313 |  3.473313 | 14.472138 | 1.616372  1.616372  0.0  0.0  1.616372 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 178.736291  178.736291  0.0  0.0  0.0 | 15.243759 | 49.639413 | 0.928849  0.939687  0.957867  0.978124  0.994423 | 14.472138 |
    | 03/01 08:00 |  0.0 | 17.1 | 16.1 |  1.173726 |  17.7 | 17.7  17.7  17.7  17.7  17.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  1.361522   1.361522   1.361522   1.361522   1.361522 | 0.980296  0.980296  0.980296  0.980296  0.980296 | 0.980296  0.980296  0.0  0.0  0.980296 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.074856  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.074856      0.0        0.0 |       0.0        0.0  15.074856      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.490148  0.490148  0.0  0.0  0.0 | 0.010632  0.010632  0.0  0.0  0.0 |   3.76074 |  0.5 |  3.480795 | 0.0  0.0  0.0  0.980296  0.0 |  0.24964 |  3.017633 |  3.319318 |  3.319318 | 13.830491 | 0.636076  0.636076  0.0  0.0  0.636076 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 178.256775  178.256775  0.0  0.0  0.0 | 15.023704 | 49.678303 | 0.876925  0.883327  0.895319  0.911611  0.930083 | 13.830491 |
    | 03/01 09:00 |  0.0 | 16.2 | 15.2 |  4.202296 |  16.8 | 16.8  16.8  16.8  16.8  16.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  4.874663   4.874663   4.874663   4.874663   4.874663 | 3.509758  3.509758  3.509758  3.509758  3.509758 | 0.636076  0.636076  0.0  0.0  0.636076 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.262439  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.262439      0.0        0.0 |       0.0        0.0  14.262439      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  3.19172   3.19172  0.0  0.0  0.0 | 0.010872  0.010872  0.0  0.0  0.0 |  3.557456 |  0.5 |  3.316135 | 0.0  0.0  0.0  3.509758  0.0 | 0.248027 |    2.8844 |  3.110023 |  3.110023 | 12.958427 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 175.075927  175.075927  0.0  0.0  0.0 | 14.765026 | 49.357453 | 0.837839  0.842599  0.851372  0.863148  0.876683 | 12.958427 |
    | 03/01 10:00 |  0.3 | 15.9 | 14.9 |  4.359715 |  16.5 | 16.5  16.5  16.5  16.5  16.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.3564   0.3564   0.3564   0.3564   0.3564 |  5.057269   5.057269   5.057269   5.057269   5.057269 | 3.513746  3.513746  3.513746  3.513746  3.513746 |   0.3564    0.3564  0.0  0.0    0.3564 |       0.0        0.0   0.3564   0.3564        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.991633  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.348033   0.3564        0.0 |       0.0        0.0  14.348033   0.3564        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.335546  3.335546  0.0  0.0  0.0 | 0.012462  0.012462  0.0  0.0  0.0 |  3.577662 |  0.5 |  3.196444 | 0.0  0.0  0.0  3.513746  0.0 | 0.246675 |  2.787385 |  2.952129 |  2.952129 | 12.300538 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 171.752844  171.752844  0.0  0.0  0.0 | 14.646244 |   49.0883 | 0.809336   0.81281  0.819219  0.827824  0.837709 | 12.300538 |
    | 03/01 11:00 |  2.6 | 16.3 | 15.3 |  5.305753 |  16.9 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  3.0888   3.0888   3.0888   3.0888   3.0888 |  6.154673   6.154673   6.154673   6.154673   6.154673 | 3.253813  3.253813  3.253813  3.253813  3.253813 |      2.0       2.0  0.0  0.0       2.0 |    1.0888     1.0888   3.0888   3.0888     1.0888 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.352707  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |    1.0888     1.0888  17.441507   3.0888     1.0888 |  0.802964   0.802964  17.441507   3.0888     1.0888 | 0.0  0.0  0.0  0.0     1.0888 | 2.253813  2.253813  0.0  0.0  0.0 | 0.013981  0.013981  0.0  0.0  0.0 |  4.952114 |  0.5 |  3.519341 | 0.0  0.0  0.0  3.253813  0.0 | 0.247466 |  3.119028 |  2.865582 |  2.865582 | 11.939923 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 169.798847  169.798847  0.0  0.0  0.0 | 15.579017 | 49.245832 |  0.90131  0.890734   0.87334  0.854607  0.840353 | 11.939923 |
    | 03/01 12:00 |  0.7 | 16.3 | 15.3 |  5.376027 |  16.9 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.8316   0.8316   0.8316   0.8316   0.8316 |  6.236191   6.236191   6.236191   6.236191   6.236191 | 4.131769  4.131769  4.131769  4.131769  4.131769 |   0.8316    0.8316  0.0  0.0    0.8316 |       0.0        0.0   0.8316   0.8316        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.352707  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.184307   0.8316        0.0 |       0.0        0.0  15.184307   0.8316        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.715969  3.715969  0.0  0.0  0.0 | 0.015101  0.015101  0.0  0.0  0.0 |  3.784751 |  0.5 |  3.656849 | 0.0  0.0  0.0  4.131769  0.0 | 0.246015 |  3.155093 |  2.989185 |  2.989185 | 12.454936 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 166.097979  166.097979  0.0  0.0  0.0 | 15.206919 | 48.956936 | 0.914516  0.912625  0.908266  0.900751  0.890095 | 12.454936 |
    | 03/01 13:00 |  0.3 | 16.4 | 15.4 |  4.658915 |  17.0 | 17.0  17.0  17.0  17.0  17.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.3564   0.3564   0.3564   0.3564   0.3564 |  5.404341   5.404341   5.404341   5.404341   5.404341 | 3.754888  3.754888  3.754888  3.754888  3.754888 |   0.3564    0.3564  0.0  0.0    0.3564 |       0.0        0.0   0.3564   0.3564        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.442976  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.799376   0.3564        0.0 |       0.0        0.0  14.799376   0.3564        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.576688  3.576688  0.0  0.0  0.0 | 0.016951  0.016951  0.0  0.0  0.0 |  3.687131 |  0.5 |  3.441957 | 0.0  0.0  0.0  3.754888  0.0 |   0.2445 |  2.981765 |  3.100885 |  3.100885 | 12.920353 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 162.538242  162.538242  0.0  0.0  0.0 | 14.952093 | 48.655509 | 0.866391  0.871922  0.880981  0.890555  0.897284 | 12.920353 |
    | 03/01 14:00 |  0.3 | 16.5 | 15.5 |  7.789594 |  17.1 | 17.1  17.1  17.1  17.1  17.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.3564   0.3564   0.3564   0.3564   0.3564 |  9.035929   9.035929   9.035929   9.035929   9.035929 | 6.278083  6.278083  6.278083  6.278083  6.278083 |   0.3564    0.3564  0.0  0.0    0.3564 |       0.0        0.0   0.3564   0.3564        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.533245  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.889645   0.3564        0.0 |       0.0        0.0  14.889645   0.3564        0.0 | 0.0  0.0  0.0  0.0        0.0 | 6.099883  6.099883  0.0  0.0  0.0 | 0.018731  0.018731  0.0  0.0  0.0 |  3.708363 |  0.5 |  3.322309 | 0.0  0.0  0.0  6.278083  0.0 | 0.241191 |  2.882958 |  3.042768 |  3.042768 | 12.678199 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  156.45709   156.45709  0.0  0.0  0.0 | 14.838148 | 47.996935 |  0.83708  0.840674  0.847337  0.856225  0.866006 | 12.678199 |
    | 03/01 15:00 |  0.0 | 18.4 | 17.4 |  4.851567 |  19.0 | 19.0  19.0  19.0  19.0  19.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  5.627818   5.627818   5.627818   5.627818   5.627818 | 4.052029  4.052029  4.052029  4.052029  4.052029 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.248348  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.248348      0.0        0.0 |       0.0        0.0  16.248348      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.962855  3.962855  0.0  0.0  0.0 | 0.021771  0.021771  0.0  0.0  0.0 |  4.045758 |  0.5 |  3.359201 | 0.0  0.0  0.0  4.052029  0.0 | 0.239233 |  2.910645 |  2.940947 |  2.940947 | 12.253945 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 152.516006  152.516006  0.0  0.0  0.0 | 15.024705 | 47.607412 | 0.843851  0.843185  0.842488  0.842714  0.844784 | 12.253945 |
    | 03/01 16:00 |  0.0 | 18.3 | 17.3 |   5.30692 |  18.9 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  6.156027   6.156027   6.156027   6.156027   6.156027 |  4.43234   4.43234   4.43234   4.43234   4.43234 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.158079  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.158079      0.0        0.0 |       0.0        0.0  16.158079      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.225675  4.225675  0.0  0.0  0.0 | 0.023742  0.023742  0.0  0.0  0.0 |  4.021713 |  0.5 |  3.444646 | 0.0  0.0  0.0   4.43234  0.0 | 0.237014 |   2.97693 |  2.913345 |  2.913345 | 12.138937 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.314073  148.314073  0.0  0.0  0.0 | 15.101772 | 47.165778 | 0.862654  0.860445  0.856667    0.8523  0.848541 | 12.138937 |
    | 03/01 17:00 |  0.0 | 18.1 | 17.1 |  3.286036 |  18.7 | 18.7  18.7  18.7  18.7  18.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  3.811802   3.811802   3.811802   3.811802   3.811802 | 2.744497  2.744497  2.744497  2.744497  2.744497 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.977542  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.977542      0.0        0.0 |       0.0        0.0  15.977542      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.544491  2.544491  0.0  0.0  0.0 | 0.025843  0.025843  0.0  0.0  0.0 |  3.975003 |  0.5 |  3.469922 | 0.0  0.0  0.0  2.744497  0.0 | 0.236011 |  2.996215 |  2.948752 |  2.948752 | 12.286467 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 145.795425  145.795425  0.0  0.0  0.0 | 15.106853 | 46.966267 | 0.868692  0.867915  0.866358   0.86403  0.861075 | 12.286467 |
    | 03/01 18:00 |  0.0 | 16.7 | 15.7 |  1.506216 |  17.3 | 17.3  17.3  17.3  17.3  17.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  1.747211   1.747211   1.747211   1.747211   1.747211 | 1.257992  1.257992  1.257992  1.257992  1.257992 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.713782  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.713782      0.0        0.0 |       0.0        0.0  14.713782      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.146522  1.146522  0.0  0.0  0.0 | 0.027102  0.027102  0.0  0.0  0.0 |  3.658119 |  0.5 |  3.384447 | 0.0  0.0  0.0  1.257992  0.0 | 0.236076 |  2.927895 |  2.977421 |  2.977421 | 12.405922 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.676006  144.676006  0.0  0.0  0.0 | 14.880525 |  46.97905 |  0.84975  0.851926  0.855494  0.859295  0.862079 | 12.405922 |
    | 03/01 19:00 |  0.0 | 15.2 | 14.2 |  0.274762 |  15.8 | 15.8  15.8  15.8  15.8  15.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.318724   0.318724   0.318724   0.318724   0.318724 | 0.229481  0.229481  0.229481  0.229481  0.229481 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.359753  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  13.359753      0.0        0.0 |       0.0        0.0  13.359753      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.207542  0.207542  0.0  0.0  0.0 | 0.027662  0.027662  0.0  0.0  0.0 |  3.319192 |  0.5 |  3.182896 | 0.0  0.0  0.0  0.229481  0.0 | 0.236874 |  2.767399 |  2.943417 |  2.943417 | 12.264236 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.496125  144.496125  0.0  0.0  0.0 | 14.516821 | 47.137964 | 0.804137  0.809513  0.818805  0.829863  0.840208 | 12.264236 |
    | 03/01 20:00 |  0.0 | 13.4 | 12.4 |  0.087565 |  14.0 | 14.0  14.0  14.0  14.0  14.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.101575   0.101575   0.101575   0.101575   0.101575 | 0.073134  0.073134  0.073134  0.073134  0.073134 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  11.734918  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.734918      0.0        0.0 |       0.0        0.0  11.734918      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.06606   0.06606  0.0  0.0  0.0 | 0.027752  0.027752  0.0  0.0  0.0 |  2.912916 |  0.5 |  2.907373 | 0.0  0.0  0.0  0.073134  0.0 |  0.23778 |  2.547827 |   2.82582 |   2.82582 | 11.774248 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.457817  144.457817  0.0  0.0  0.0 | 14.022363 | 47.318307 |  0.74107  0.748583  0.761843  0.778265  0.794773 | 11.774248 |
    | 03/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0  13.0  13.0  13.0  13.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.099494   0.099494   0.099494   0.099494   0.099494 | 0.071636  0.071636  0.071636  0.071636  0.071636 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.832232  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.832232      0.0        0.0 |       0.0        0.0  10.832232      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.06469   0.06469  0.0  0.0  0.0 | 0.027771  0.027771  0.0  0.0  0.0 |   2.68723 |  0.5 |  2.630798 | 0.0  0.0  0.0  0.071636  0.0 | 0.238683 |  2.327409 |   2.63999 |   2.63999 | 10.999957 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.420898  144.420898  0.0  0.0  0.0 | 13.578795 | 47.497962 | 0.677177  0.684859  0.698654   0.71629  0.734973 | 10.999957 |
    | 03/01 22:00 |  0.0 | 11.6 | 10.6 |  0.084317 |  12.2 | 12.2  12.2  12.2  12.2  12.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.097808   0.097808   0.097808   0.097808   0.097808 | 0.070422  0.070422  0.070422  0.070422  0.070422 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.110083  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.110083      0.0        0.0 |       0.0        0.0  10.110083      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.063577  0.063577  0.0  0.0  0.0 |  0.02779   0.02779  0.0  0.0  0.0 |  2.506679 |  0.5 |  2.399532 | 0.0  0.0  0.0  0.070422  0.0 | 0.239582 |  2.143236 |  2.428387 |  2.428387 | 10.118278 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.385111  144.385111  0.0  0.0  0.0 | 13.185942 |  47.67689 | 0.623414  0.629923  0.641773  0.657295  0.674397 | 10.118278 |
    | 03/01 23:00 |  0.0 | 11.0 | 10.0 |  0.083215 |  11.6 | 11.6  11.6  11.6  11.6  11.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.096529   0.096529   0.096529   0.096529   0.096529 | 0.069501  0.069501  0.069501  0.069501  0.069501 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.568472  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.568472      0.0        0.0 |       0.0        0.0   9.568472      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.06273   0.06273  0.0  0.0  0.0 | 0.027807  0.027807  0.0  0.0  0.0 |  2.371262 |  0.5 |  2.209111 | 0.0  0.0  0.0  0.069501  0.0 | 0.240478 |  1.991735 |  2.231764 |  2.231764 |  9.299015 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.350188  144.350188  0.0  0.0  0.0 | 12.848093 | 47.855055 | 0.579147  0.584513  0.594307  0.607211  0.621595 |  9.299015 |
    | 04/01 00:00 |  0.0 | 10.5 |  9.5 |  0.082289 |  11.1 | 11.1  11.1  11.1  11.1  11.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.095455   0.095455   0.095455   0.095455   0.095455 | 0.068728  0.068728  0.068728  0.068728  0.068728 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.121265  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.121265      0.0        0.0 |       0.0        0.0   9.121265      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.062017  0.062017  0.0  0.0  0.0 | 0.027825  0.027825  0.0  0.0  0.0 |  2.259448 |  0.5 |  2.052834 | 0.0  0.0  0.0  0.068728  0.0 | 0.241369 |  1.867545 |  2.065293 |  2.065293 |  8.605389 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.315995  144.315995  0.0  0.0  0.0 | 12.554706 |  48.03244 | 0.542854  0.547254   0.55529   0.56589  0.577737 |  8.605389 |
    | 04/01 01:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3  12.3  12.3  12.3  12.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |   0.09802    0.09802    0.09802    0.09802    0.09802 | 0.070574  0.070574  0.070574  0.070574  0.070574 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.20498  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.20498      0.0        0.0 |       0.0        0.0   10.20498      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.063669  0.063669  0.0  0.0  0.0 | 0.027842  0.027842  0.0  0.0  0.0 |  2.530363 |  0.5 |  1.996979 | 0.0  0.0  0.0  0.070574  0.0 | 0.242255 |  1.823687 |  1.932097 |  1.932097 |  8.050406 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.280169  144.280169  0.0  0.0  0.0 |  12.58809 | 48.208674 | 0.529341   0.53106   0.53447   0.53956  0.546185 |  8.050406 |
    | 04/01 02:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.098442   0.098442   0.098442   0.098442   0.098442 | 0.070878  0.070878  0.070878  0.070878  0.070878 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599      0.0        0.0 |       0.0        0.0  10.385599      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.063927  0.063927  0.0  0.0  0.0 |  0.02786   0.02786  0.0  0.0  0.0 |  2.575505 |  0.5 |  2.019041 | 0.0  0.0  0.0  0.070878  0.0 | 0.243136 |  1.842159 |  1.855703 |  1.855703 |  7.732096 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.244102  144.244102  0.0  0.0  0.0 | 12.644554 | 48.383985 | 0.534061  0.533571  0.532955  0.532787  0.533697 |  7.732096 |
    | 04/01 03:00 |  1.3 | 11.2 | 10.2 |  0.083584 |  11.8 | 11.8  11.8  11.8  11.8  11.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  1.5444   1.5444   1.5444   1.5444   1.5444 |  0.096957   0.096957   0.096957   0.096957   0.096957 | 0.059819  0.059819  0.059819  0.059819  0.059819 | 0.059819  0.059819  0.0  0.0  0.059819 |       0.0        0.0   1.5444   1.5444        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.753432  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.297832   1.5444        0.0 |       0.0        0.0  11.297832   1.5444        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.02697   0.02697  0.0  0.0  0.0 | 0.027878  0.027878  0.0  0.0  0.0 |  2.803549 |  0.5 |  2.088785 | 0.0  0.0  0.0  0.059819  0.0 | 0.245123 |   1.89981 |   1.84335 |   1.84335 |  7.680625 | 1.484581  1.484581  0.0  0.0  1.484581 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  144.24501   144.24501  0.0  0.0  0.0 | 12.859318 | 48.779516 | 0.550376  0.548463  0.545194  0.541407  0.538092 |  7.680625 |
    | 04/01 04:00 |  0.0 | 11.1 | 10.1 |    0.0834 |  11.7 | 11.7  11.7  11.7  11.7  11.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.096744   0.096744   0.096744   0.096744   0.096744 | 0.069656  0.069656  0.069656  0.069656  0.069656 | 0.069656  0.069656  0.0  0.0  0.069656 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.663122  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.663122      0.0        0.0 |       0.0        0.0   9.663122      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.031404  0.031404  0.0  0.0  0.0 | 0.027877  0.027877  0.0  0.0  0.0 |  2.394872 |  0.5 |  2.085549 | 0.0  0.0  0.0  0.069656  0.0 | 0.245991 |   1.89803 |  1.874025 |  1.874025 |  7.808437 | 1.414925  1.414925  0.0  0.0  1.414925 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.241483  144.241483  0.0  0.0  0.0 | 12.668642 | 48.952146 | 0.550427  0.550357  0.550021  0.549153  0.547578 |  7.808437 |
    | 04/01 05:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.098442   0.098442   0.098442   0.098442   0.098442 | 0.070878  0.070878  0.070878  0.070878  0.070878 | 0.070878  0.070878  0.0  0.0  0.070878 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599      0.0        0.0 |       0.0        0.0  10.385599      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.031955  0.031955  0.0  0.0  0.0 | 0.027879  0.027879  0.0  0.0  0.0 |   2.57549 |  0.5 |  2.049886 | 0.0  0.0  0.0  0.070878  0.0 | 0.246853 |  1.870305 |  1.891904 |  1.891904 |  7.882933 | 1.344047  1.344047  0.0  0.0  1.344047 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.237408  144.237408  0.0  0.0  0.0 | 12.694246 | 49.123739 | 0.542659   0.54356  0.545061  0.546704  0.547954 |  7.882933 |
    | 04/01 06:00 |  0.0 | 12.2 | 11.2 |  0.310229 |  12.8 | 12.8  12.8  12.8  12.8  12.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.359866   0.359866   0.359866   0.359866   0.359866 | 0.259103  0.259103  0.259103  0.259103  0.259103 | 0.259103  0.259103  0.0  0.0  0.259103 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.656527  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.656527      0.0        0.0 |       0.0        0.0  10.656527      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.116811  0.116811  0.0  0.0  0.0 | 0.027881  0.027881  0.0  0.0  0.0 |  2.643221 |  0.5 |  2.074023 | 0.0  0.0  0.0  0.259103  0.0 | 0.247576 |   1.89029 |  1.883171 |  1.883171 |  7.846546 | 1.084943  1.084943  0.0  0.0  1.084943 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 144.148478  144.148478  0.0  0.0  0.0 | 12.763443 | 49.267719 | 0.547998  0.547408  0.546515  0.545725  0.545412 |  7.846546 |
    | 04/01 07:00 |  0.7 | 11.8 | 10.8 |  1.391958 |  12.4 | 12.4  12.4  12.4  12.4  12.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.8316   0.8316   0.8316   0.8316   0.8316 |  1.614671   1.614671   1.614671   1.614671   1.614671 | 1.069795  1.069795  1.069795  1.069795  1.069795 | 1.069795  1.069795  0.0  0.0  1.069795 |       0.0        0.0   0.8316   0.8316        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.295289  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.126889   0.8316        0.0 |       0.0        0.0  11.126889   0.8316        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.481998  0.481998  0.0  0.0  0.0 | 0.027926  0.027926  0.0  0.0  0.0 |  2.760778 |  0.5 |  2.126008 | 0.0  0.0  0.0  1.069795  0.0 | 0.248311 |  1.932564 |  1.887019 |  1.887019 |  7.862581 | 0.846748  0.846748  0.0  0.0  0.846748 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 143.694406  143.694406  0.0  0.0  0.0 | 12.898213 | 49.413951 | 0.560024  0.558606  0.556156   0.55325  0.550566 |  7.862581 |
    | 04/01 08:00 |  0.4 | 11.4 | 10.4 |  3.195876 |  12.0 | 12.0  12.0  12.0  12.0  12.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.4752   0.4752   0.4752   0.4752   0.4752 |  3.707216   3.707216   3.707216   3.707216   3.707216 | 2.545322  2.545322  2.545322  2.545322  2.545322 | 1.321948  1.321948  0.0  0.0  1.321948 |       0.0        0.0   0.4752   0.4752        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.934051  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.409251   0.4752        0.0 |       0.0        0.0  10.409251   0.4752        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.754875  1.754875  0.0  0.0  0.0 | 0.028153  0.028153  0.0  0.0  0.0 |  2.581198 |  0.5 |   2.14036 | 0.0  0.0  0.0  2.545322  0.0 | 0.247734 |  1.943507 |  1.913867 |  1.913867 |  7.974444 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 141.967683  141.967683  0.0  0.0  0.0 | 12.839051 | 49.299057 | 0.563496  0.563044  0.562123  0.560712  0.558866 |  7.974444 |
    | 04/01 09:00 |  0.1 | 11.6 | 10.6 |  5.191651 |  12.2 | 12.2  12.2  12.2  12.2  12.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.1188   0.1188   0.1188   0.1188   0.1188 |  6.022315   6.022315   6.022315   6.022315   6.022315 | 4.284859  4.284859  4.284859  4.284859  4.284859 |   0.1188    0.1188  0.0  0.0    0.1188 |       0.0        0.0   0.1188   0.1188        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.11467  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.23347   0.1188        0.0 |       0.0        0.0   10.23347   0.1188        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.802724  3.802724  0.0  0.0  0.0 | 0.029016  0.029016  0.0  0.0  0.0 |  2.536605 |  0.5 |  2.107668 | 0.0  0.0  0.0  4.284859  0.0 | 0.245662 |  1.915419 |  1.933147 |  1.933147 |  8.054778 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 138.193975  138.193975  0.0  0.0  0.0 | 12.767989 | 48.886815 | 0.555742  0.556628  0.558064  0.559547  0.560532 |  8.054778 |
    | 04/01 10:00 |  0.4 | 13.0 | 12.0 |  7.155036 |  13.6 | 13.6  13.6  13.6  13.6  13.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.4752   0.4752   0.4752   0.4752   0.4752 |  8.299842   8.299842   8.299842   8.299842   8.299842 | 5.698554  5.698554  5.698554  5.698554  5.698554 |   0.4752    0.4752  0.0  0.0    0.4752 |       0.0        0.0   0.4752   0.4752        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  11.379004  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.854204   0.4752        0.0 |       0.0        0.0  11.854204   0.4752        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.923012  4.923012  0.0  0.0  0.0 | 0.030903  0.030903  0.0  0.0  0.0 |  2.940374 |  0.5 |  2.166627 | 0.0  0.0  0.0  5.698554  0.0 | 0.242846 |  1.959958 |  1.929285 |  1.929285 |  8.038689 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 133.301867  133.301867  0.0  0.0  0.0 | 13.041736 | 48.326347 | 0.567962  0.566571  0.564327  0.561996  0.560331 |  8.038689 |
    | 04/01 11:00 |  0.0 | 17.1 | 16.1 |  8.391432 |  17.7 | 17.7  17.7  17.7  17.7  17.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  9.734061   9.734061   9.734061   9.734061   9.734061 | 7.008524  7.008524  7.008524  7.008524  7.008524 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.081695  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.081695      0.0        0.0 |       0.0        0.0  15.081695      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.840519  5.840519  0.0  0.0  0.0 | 0.033349  0.033349  0.0  0.0  0.0 |  3.745412 |  0.5 |  2.461314 | 0.0  0.0  0.0  7.008524  0.0 | 0.238769 |  2.191902 |  1.959638 |  1.959638 |  8.165159 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 127.494697  127.494697  0.0  0.0  0.0 | 13.825834 | 47.514932 |  0.63337  0.625719  0.612686  0.597584  0.584091 |  8.165159 |
    | 04/01 12:00 |  0.0 | 18.2 | 17.2 |  8.391286 |  18.8 | 18.8  18.8  18.8  18.8  18.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  9.733892   9.733892   9.733892   9.733892   9.733892 | 7.008402  7.008402  7.008402  7.008402  7.008402 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.075101  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.075101      0.0        0.0 |       0.0        0.0  16.075101      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.586176  5.586176  0.0  0.0  0.0 | 0.036253  0.036253  0.0  0.0  0.0 |  3.991586 |  0.5 |  2.865841 | 0.0  0.0  0.0  7.008402  0.0 | 0.234712 |  2.511737 |   2.11205 |   2.11205 |  8.800207 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 121.944773  121.944773  0.0  0.0  0.0 | 14.451578 | 46.707592 | 0.725201  0.714266  0.694995  0.671207  0.647468 |  8.800207 |
    | 04/01 13:00 |  0.0 | 22.4 | 21.4 | 10.715238 |  23.0 | 23.0  23.0  23.0  23.0  23.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 | 12.429676  12.429676  12.429676  12.429676  12.429676 | 8.949367  8.949367  8.949367  8.949367  8.949367 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.868102  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.868102      0.0        0.0 |       0.0        0.0  19.868102      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 6.822986  6.822986  0.0  0.0  0.0 | 0.039028  0.039028  0.0  0.0  0.0 |  4.937755 |  0.5 |  3.418262 | 0.0  0.0  0.0  8.949367  0.0 | 0.229288 |  2.948612 |  2.386436 |  2.386436 |  9.943482 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 115.160815  115.160815  0.0  0.0  0.0 | 15.471071 | 45.628394 |  0.85072  0.835758  0.809315  0.776439  0.743082 |  9.943482 |
    | 04/01 14:00 |  0.0 | 21.4 | 20.4 |  9.383394 |  22.0 | 22.0  22.0  22.0  22.0  22.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 | 10.884737  10.884737  10.884737  10.884737  10.884737 | 7.837011  7.837011  7.837011  7.837011  7.837011 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.965006  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  18.965006      0.0        0.0 |       0.0        0.0  18.965006      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.642806  5.642806  0.0  0.0  0.0 |  0.04242   0.04242  0.0  0.0  0.0 |  4.709437 |  0.5 |   3.87128 | 0.0  0.0  0.0  7.837011  0.0 | 0.224687 |  3.306732 |  2.758447 |  2.758447 | 11.493528 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 109.560428  109.560428  0.0  0.0  0.0 | 15.809228 | 44.712705 | 0.955304  0.942641    0.9196  0.889495  0.856559 | 11.493528 |
    | 04/01 15:00 |  0.0 | 21.8 | 20.8 |  7.861915 |  22.4 | 22.4  22.4  22.4  22.4  22.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  9.119821   9.119821   9.119821   9.119821   9.119821 | 6.566271  6.566271  6.566271  6.566271  6.566271 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.326244  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.326244      0.0        0.0 |       0.0        0.0  19.326244      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.498128  4.498128  0.0  0.0  0.0 |  0.04522   0.04522  0.0  0.0  0.0 |  4.797646 |  0.5 |  4.077238 | 0.0  0.0  0.0  6.566271  0.0 | 0.221016 |  3.468072 |  3.125509 |  3.125509 | 13.022954 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  105.10752   105.10752  0.0  0.0  0.0 | 16.029636 | 43.982222 | 1.004041  0.997948   0.98621  0.969402   0.94856 | 13.022954 |
    | 04/01 16:00 |  0.0 | 22.2 | 21.2 |  6.298329 |  22.8 | 22.8  22.8  22.8  22.8  22.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  7.306062   7.306062   7.306062   7.306062   7.306062 | 5.260364  5.260364  5.260364  5.260364  5.260364 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.687483  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.687483      0.0        0.0 |       0.0        0.0  19.687483      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.457209  3.457209  0.0  0.0  0.0 | 0.047446  0.047446  0.0  0.0  0.0 |  4.886286 |  0.5 |  4.223748 | 0.0  0.0  0.0  5.260364  0.0 | 0.218297 |  3.582742 |  3.373024 |  3.373024 | 14.054266 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 101.697758  101.697758  0.0  0.0  0.0 | 16.192174 | 43.441016 | 1.037817  1.033685  1.025994  1.015465  1.002918 | 14.054266 |
    | 04/01 17:00 |  0.0 | 20.1 | 19.1 |  2.948416 |  20.7 | 20.7  20.7  20.7  20.7  20.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  3.420163   3.420163   3.420163   3.420163   3.420163 | 2.462517  2.462517  2.462517  2.462517  2.462517 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.790982  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.790982      0.0        0.0 |       0.0        0.0  17.790982      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.565959  1.565959  0.0  0.0  0.0 | 0.049151  0.049151  0.0  0.0  0.0 |  4.410882 |  0.5 |  4.165073 | 0.0  0.0  0.0  2.462517  0.0 | 0.217589 |  3.535141 |   3.51304 |   3.51304 | 14.637667 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 100.180949  100.180949  0.0  0.0  0.0 | 15.937983 |  43.30021 | 1.025617  1.026896   1.02856  1.029266  1.027643 | 14.637667 |
    | 04/01 18:00 |  0.0 | 17.8 | 16.8 |  1.309232 |  18.4 | 18.4  18.4  18.4  18.4  18.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  1.518709   1.518709   1.518709   1.518709   1.518709 | 1.093471  1.093471  1.093471  1.093471  1.093471 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.713862  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.713862      0.0        0.0 |       0.0        0.0  15.713862      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.684997  0.684997  0.0  0.0  0.0 |  0.04991   0.04991  0.0  0.0  0.0 |  3.891033 |  0.5 |  3.873407 | 0.0  0.0  0.0  1.093471  0.0 | 0.217863 |  3.302065 |  3.527475 |  3.527475 | 14.697811 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.545862   99.545862  0.0  0.0  0.0 | 15.455609 | 43.354709 | 0.959891  0.967576  0.980639   0.99567  1.008796 | 14.697811 |
    | 04/01 19:00 |  0.0 | 15.2 | 14.2 |   0.32955 |  15.8 | 15.8  15.8  15.8  15.8  15.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.382278   0.382278   0.382278   0.382278   0.382278 |  0.27524   0.27524   0.27524   0.27524   0.27524 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.365814  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  13.365814      0.0        0.0 |       0.0        0.0  13.365814      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.17133   0.17133  0.0  0.0  0.0 | 0.050227  0.050227  0.0  0.0  0.0 |  3.303783 |  0.5 |   3.45946 | 0.0  0.0  0.0   0.27524  0.0 |  0.21872 |  2.971706 |   3.38089 |   3.38089 |  14.08704 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.424759   99.424759  0.0  0.0  0.0 | 14.799933 |  43.52524 | 0.865101  0.876381  0.896236  0.920688  0.944983 |  14.08704 |
    | 04/01 20:00 |  0.0 | 14.5 | 13.5 |  0.089508 |  15.1 | 15.1  15.1  15.1  15.1  15.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.103829   0.103829   0.103829   0.103829   0.103829 | 0.074757  0.074757  0.074757  0.074757  0.074757 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.733647  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  12.733647      0.0        0.0 |       0.0        0.0  12.733647      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.046478  0.046478  0.0  0.0  0.0 | 0.050288  0.050288  0.0  0.0  0.0 |  3.145696 |  0.5 |  3.098875 | 0.0  0.0  0.0  0.074757  0.0 | 0.219716 |  2.684168 |  3.112208 |  3.112208 | 12.967532 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.428569   99.428569  0.0  0.0  0.0 | 14.346754 | 43.723416 |  0.78133   0.79145  0.809781   0.83355  0.859236 | 12.967532 |
    | 04/01 21:00 |  0.0 | 12.4 | 11.4 |  0.085771 |  13.0 | 13.0  13.0  13.0  13.0  13.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.099494   0.099494   0.099494   0.099494   0.099494 | 0.071636  0.071636  0.071636  0.071636  0.071636 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.837146  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.837146      0.0        0.0 |       0.0        0.0  10.837146      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.044539  0.044539  0.0  0.0  0.0 | 0.050286  0.050286  0.0  0.0  0.0 |  2.671572 |  0.5 |  2.770382 | 0.0  0.0  0.0  0.071636  0.0 | 0.220709 |    2.4223 |  2.818179 |  2.818179 | 11.742412 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.434315   99.434315  0.0  0.0  0.0 | 13.747944 | 43.921045 | 0.705125  0.714326  0.730995  0.752672  0.776352 | 11.742412 |
    | 04/01 22:00 |  0.0 | 11.7 | 10.7 |    0.0845 |  12.3 | 12.3  12.3  12.3  12.3  12.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |   0.09802    0.09802    0.09802    0.09802    0.09802 | 0.070574  0.070574  0.070574  0.070574  0.070574 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.20498  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.20498      0.0        0.0 |       0.0        0.0   10.20498      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.043882  0.043882  0.0  0.0  0.0 | 0.050283  0.050283  0.0  0.0  0.0 |  2.513533 |  0.5 |  2.472123 | 0.0  0.0  0.0  0.070574  0.0 | 0.221698 |  2.184616 |  2.544619 |  2.544619 | 10.602578 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.440716   99.440716  0.0  0.0  0.0 | 13.289355 | 44.117837 | 0.635951  0.644304  0.659437  0.679124  0.700651 | 10.602578 |
    | 04/01 23:00 |  0.0 | 11.9 | 10.9 |  0.084864 |  12.5 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0      0.0      0.0      0.0      0.0 |  0.098442   0.098442   0.098442   0.098442   0.098442 | 0.070878  0.070878  0.070878  0.070878  0.070878 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0      0.0      0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599      0.0        0.0 |       0.0        0.0  10.385599      0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.044074  0.044074  0.0  0.0  0.0 |  0.05028   0.05028  0.0  0.0  0.0 |   2.55869 |  0.5 |  2.291974 | 0.0  0.0  0.0  0.070878  0.0 | 0.222681 |  2.041415 |  2.300883 |  2.300883 |  9.587013 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.446922   99.446922  0.0  0.0  0.0 |  13.05607 | 44.313602 | 0.593491  0.598709  0.608468  0.621829  0.637502 |  9.587013 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_v2_snow_redistribution:

snow redistribution
___________________

Besides slight differences in runoff concentration, the following calculation
reproduces the results of the :ref:`hland_v1_snow_redistribution_1` experiment of
|hland_v1| exactly:

.. integration-test::

    >>> zonez(0.0, 10.0, 30.0, 0.0, 20.0)
    >>> smax(200.0)
    >>> sred(n_zones=1)
    >>> inputs.t.series = 5.0
    >>> test("hland_v2_snow_redistribution",
    ...      axis1=(states.sp, states.wc), axis2=(factors.tc, fluxes.pc))
    |        date |    p |   t |   tn |       epn | tmean |                         tc |                     fracrain |                      rfc |                      sfc |                                       cfact |                                                 swe |                          gact | contriarea |                                             pc |                                               ep |                                              epc |                                     ei |                                              tf |                                                 spl |                                             wcl |                                                 spg |                                             wcg |                     glmelt |                          melt |                           refr |                                    in_ |                                     r |                      sr |                      ea |                                cf |      inuz |     perc |       q0 |                           el |       q1 |     inuh |    outuh |       rt |       qt |                                     ic |                                                  sp |                                                  wc |                                    sm |       uz |        lz |                                               sc |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 5.0 | 20.2 |  0.100707 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 |     0.05 | 0.046667 | 0.163438 | 0.163438 | 0.680992 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |      9.95 | 0.014762  0.018859  0.025756  0.033571   0.04028 | 0.680992 |
    | 01/01 01:00 |  0.0 | 5.0 | 18.4 |  0.097801 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 |  0.04975 | 0.046433 |  0.10046 |  0.10046 | 0.418582 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.90025 | 0.013509  0.013793  0.014731  0.016822  0.020347 | 0.418582 |
    | 01/01 02:00 |  0.0 | 5.0 | 17.9 |  0.096981 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.049501 | 0.046201 | 0.056933 | 0.056933 | 0.237219 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.850749 | 0.013402  0.013424  0.013509  0.013762  0.014374 | 0.237219 |
    | 01/01 03:00 |  0.0 | 5.0 | 17.3 |   0.09599 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.049254 |  0.04597 | 0.047524 | 0.047524 | 0.198017 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.801495 | 0.013334  0.013342  0.013361  0.013398  0.013481 | 0.198017 |
    | 01/01 04:00 |  0.0 | 5.0 | 17.9 |  0.096981 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.049007 |  0.04574 | 0.046181 | 0.046181 | 0.192422 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.752488 | 0.013267  0.013275   0.01329   0.01331  0.013335 | 0.192422 |
    | 01/01 05:00 |  0.0 | 5.0 | 21.5 |  0.102761 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.048762 | 0.045512 | 0.045852 | 0.045852 | 0.191049 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.703725 | 0.013201  0.013209  0.013223  0.013242  0.013262 | 0.191049 |
    | 01/01 06:00 |  0.0 | 5.0 | 24.1 |  0.291908 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.048519 | 0.045284 | 0.045615 | 0.045615 | 0.190063 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.655206 | 0.013135  0.013143  0.013157  0.013175  0.013195 | 0.190063 |
    | 01/01 07:00 |  0.0 | 5.0 | 27.3 |  1.932875 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.048276 | 0.045058 | 0.045387 | 0.045387 | 0.189111 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.60693 | 0.013069  0.013077  0.013091  0.013109  0.013129 | 0.189111 |
    | 01/01 08:00 |  0.0 | 5.0 | 26.8 |  4.369536 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.048035 | 0.044832 |  0.04516 |  0.04516 | 0.188166 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.558896 | 0.013004  0.013012  0.013026  0.013044  0.013063 | 0.188166 |
    | 01/01 09:00 |  0.0 | 5.0 | 30.4 |  7.317556 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.047794 | 0.044608 | 0.044934 | 0.044934 | 0.187225 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.511101 | 0.012939  0.012946  0.012961  0.012979  0.012998 | 0.187225 |
    | 01/01 10:00 |  0.0 | 5.0 | 31.2 |  8.264362 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.047556 | 0.044385 | 0.044709 | 0.044709 | 0.186289 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.463546 | 0.012874  0.012882  0.012896  0.012914  0.012933 | 0.186289 |
    | 01/01 11:00 |  0.0 | 5.0 | 34.2 |  9.369867 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.047318 | 0.044163 | 0.044486 | 0.044486 | 0.185357 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.416228 |  0.01281  0.012817  0.012831  0.012849  0.012868 | 0.185357 |
    | 01/01 12:00 |  0.0 | 5.0 | 36.1 |  5.126178 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.047081 | 0.043942 | 0.044263 | 0.044263 |  0.18443 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.369147 | 0.012746  0.012753  0.012767  0.012785  0.012804 |  0.18443 |
    | 01/01 13:00 |  0.0 | 5.0 | 30.2 |   6.62503 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046846 | 0.043723 | 0.044042 | 0.044042 | 0.183508 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.322301 | 0.012682  0.012689  0.012703  0.012721   0.01274 | 0.183508 |
    | 01/01 14:00 |  0.0 | 5.0 | 23.3 |  7.397619 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046612 | 0.043504 | 0.043822 | 0.043822 | 0.182591 |      0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.27569 | 0.012618  0.012626   0.01264  0.012657  0.012676 | 0.182591 |
    | 01/01 15:00 |  0.2 | 5.0 | 24.4 |   2.39151 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0      1.1856  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  0.2112   0.50976    1.1856   0.2112    0.8736 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0    1.1856   0.2112       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0   0.2112  0.0 |      0.0       0.0  0.0   0.2112  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046529 | 0.043427 | 0.043613 | 0.043613 |  0.18172 |   0.2112   0.50976  0.0  0.0    0.8736 |        0.0         0.0      1.1856  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.259332 | 0.012595  0.012598  0.012604  0.012613  0.012624 |  0.18172 |
    | 01/01 16:00 |  0.0 | 5.0 | 24.9 |  1.829834 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0      1.1856  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046297 |  0.04321 | 0.043465 | 0.043465 | 0.181105 |   0.2112   0.50976  0.0  0.0    0.8736 |        0.0         0.0      1.1856  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |  9.213035 | 0.012533   0.01254  0.012553  0.012568  0.012583 | 0.181105 |
    | 01/01 17:00 |  0.0 | 5.0 | 22.7 |  1.136569 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0      1.1856  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046065 | 0.042994 | 0.043296 | 0.043296 | 0.180402 |   0.2112   0.50976  0.0  0.0    0.8736 |        0.0         0.0      1.1856  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0 |   9.16697 |  0.01247  0.012478  0.012492  0.012509  0.012527 | 0.180402 |
    | 01/01 18:00 |  1.3 | 5.0 | 20.6 |  0.750986 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    1.180822       8.892  0.0       4.552 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  1.3728   3.31344    7.7064   1.3728    5.6784 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0    1.8232    7.7064   1.3728     4.552 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |      0.0   0.642378  0.0   1.3728  0.0 |      0.0  0.160595  0.0   1.3728  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0  0.049759  0.0  0.0  0.0 |  0.036945 | 0.036945 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.046974 | 0.043842 | 0.043166 | 0.043166 | 0.179859 |    1.584       2.0  0.0  0.0       2.0 |        0.0    0.843444       8.892  0.0       4.552 |        0.0    0.337378         0.0  0.0         0.0 |      100.0  100.531543  0.0  0.0  0.0 |      0.0 |  9.347778 | 0.012706  0.012679  0.012634  0.012585  0.012547 | 0.179859 |
    | 01/01 19:00 |  5.6 | 5.0 | 20.2 |  0.223895 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    10.04271     42.0888  0.0     29.0128 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  5.9136  14.27328   33.1968   5.9136   24.4608 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   5.4976  14.27328   33.1968   5.9136   24.4608 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   5.4976   5.411392  0.0   5.9136  0.0 |   1.3744  1.367268  0.0   5.9136  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047938  0.047712  0.0  0.0  0.0 |  0.992544 |      0.5 |  0.00003 | 0.0  0.0  0.0       0.0  0.0 | 0.053106 |  0.04959 |  0.04391 |  0.04391 | 0.182957 |      2.0       2.0  0.0  0.0       2.0 |        0.0    7.173364     42.0888  0.0     29.0128 |        0.0    2.869346         0.0  0.0         0.0 | 104.171138  104.623379  0.0  0.0  0.0 | 0.492514 | 10.568043 | 0.014325  0.014135  0.013814  0.013443  0.013114 | 0.182957 |
    | 01/01 20:00 |  2.9 | 5.0 | 19.4 |  0.099425 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   14.658902       59.28  0.0       41.68 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  3.0624   7.39152   17.1912   3.0624   12.6672 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   3.0624   7.39152   17.1912   3.0624   12.6672 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   3.0624   2.775328  0.0   3.0624  0.0 |   0.8308  0.759472  0.0   3.0624  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046799   0.04668  0.0  0.0  0.0 |  0.564265 |      0.5 | 0.000145 | 0.0  0.0  0.0       0.0  0.0 | 0.057171 | 0.053475 |  0.04734 |  0.04734 | 0.197248 |      2.0       2.0  0.0  0.0       2.0 |        0.0   10.470644       59.28  0.0       41.68 |        0.0    4.188258         0.0  0.0         0.0 | 106.449537  106.685915  0.0  0.0  0.0 | 0.556633 |  11.37693 | 0.015468  0.015329  0.015073  0.014735  0.014363 | 0.197248 |
    | 01/01 21:00 |  4.9 | 5.0 | 18.8 |  0.098454 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   22.420054     88.3272  0.0     63.0832 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  5.1744  12.48912   29.0472   5.1744   21.4032 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   5.1744  12.48912   29.0472   5.1744   21.4032 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   5.1744   4.727968  0.0   5.1744  0.0 | 1.465843   1.34533  0.0   5.1744  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.044921  0.044966  0.0  0.0  0.0 |  1.025506 |      0.5 | 0.000612 | 0.0  0.0  0.0       0.0  0.0 | 0.062724 | 0.059031 | 0.051725 | 0.051725 | 0.215521 |      2.0       2.0  0.0  0.0       2.0 |        0.0   16.014324     88.3272  0.0     63.0832 |        0.0     6.40573         0.0  0.0         0.0 | 110.203014  110.113519  0.0  0.0  0.0 | 1.081527 | 12.481978 | 0.017063  0.016873  0.016537  0.016116  0.015684 | 0.215521 |
    | 01/01 22:00 | 10.6 | 5.0 | 18.6 |  0.098128 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   39.144342     151.164  0.0     109.384 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 | 11.1936  27.01728   62.8368  11.1936   46.3008 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  11.1936  27.01728   62.8368  11.1936   46.3008 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |  11.1936  10.292992  0.0  11.1936  0.0 | 3.398574   3.12006  0.0  11.1936  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.041001  0.041357  0.0  0.0  0.0 |  2.425223 |      0.5 | 0.010504 | 0.0  0.0  0.0       0.0  0.0 | 0.072548 | 0.076115 | 0.057495 | 0.057495 | 0.239562 |      2.0       2.0  0.0  0.0       2.0 |        0.0   27.960244     151.164  0.0     109.384 |        0.0   11.184098         0.0  0.0         0.0 | 118.039041  117.327808  0.0  0.0  0.0 | 2.996247 | 14.437087 | 0.021904  0.021335  0.020355  0.019193  0.018105 | 0.239562 |
    | 01/01 23:00 |  0.1 | 5.0 | 18.2 |  0.097474 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    39.35759    151.7568  0.0    109.8208 | 0.0  0.0  0.901953  0.0   0.0 |        1.0 |  0.1056   0.25488    0.5928   0.1056    0.4368 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.1056   0.25488    0.5928   0.1056    0.4368 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   0.1056   0.041632  0.0   0.1056  0.0 | 0.036784  0.014327  0.0   0.1056  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.040946  0.041322  0.0  0.0  0.0 | -0.010733 |      0.5 | 0.020476 | 0.0  0.0  0.0       0.0  0.0 | 0.074404 | 0.085825 | 0.068995 | 0.068995 | 0.287477 |      2.0       2.0  0.0  0.0       2.0 |        0.0   28.112564    151.7568  0.0    109.8208 |        0.0   11.245026         0.0  0.0         0.0 | 118.148803  117.396435  0.0  0.0  0.0 | 2.465038 |  14.80634 | 0.024789  0.024434  0.023773  0.022876  0.021852 | 0.287477 |
    | 02/01 00:00 |  0.7 | 5.0 | 18.2 |  0.097474 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   40.514326    155.9064  0.0    112.8784 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.7392   1.78416    4.1496   0.7392    3.0576 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.7392   1.78416    4.1496   0.7392    3.0576 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   0.7392   0.627424  0.0   0.7392  0.0 | 0.257965  0.216178  0.0   0.7392  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.040685  0.041096  0.0  0.0  0.0 |  0.148894 |      0.5 | 0.011948 | 0.0  0.0  0.0       0.0  0.0 | 0.076703 | 0.081148 | 0.079863 | 0.079863 | 0.332762 |      2.0       2.0  0.0  0.0       2.0 |        0.0   28.938804    155.9064  0.0    112.8784 |        0.0   11.575522         0.0  0.0         0.0 | 118.670724  117.848778  0.0  0.0  0.0 | 2.101983 | 15.263809 | 0.023575  0.023704   0.02388  0.023977  0.023872 | 0.332762 |
    | 02/01 01:00 |  3.0 | 5.0 | 18.2 |  0.097474 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   45.287766    173.6904  0.0    125.9824 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |   3.168    7.6464    17.784    3.168    13.104 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |    3.168    7.6464    17.784    3.168    13.104 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |    3.168    2.87296  0.0    3.168  0.0 | 1.115353  0.997516  0.0    3.168  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.039638  0.040138  0.0  0.0  0.0 |   0.76734 |      0.5 | 0.011153 | 0.0  0.0  0.0       0.0  0.0 | 0.080725 | 0.084265 | 0.082396 | 0.082396 | 0.343319 |      2.0       2.0  0.0  0.0       2.0 |        0.0   32.348404    173.6904  0.0    125.9824 |        0.0   12.939362         0.0  0.0         0.0 | 120.763009   119.76436  0.0  0.0  0.0 | 2.358171 | 16.064227 | 0.024408  0.024315  0.024173  0.024036  0.023945 | 0.343319 |
    | 02/01 02:00 |  2.1 | 5.0 | 17.9 |  0.096981 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   48.645974    186.1392  0.0    135.1552 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  2.2176   5.35248   12.4488   2.2176    9.1728 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.2176   5.35248   12.4488   2.2176    9.1728 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   2.2176   1.994272  0.0   2.2176  0.0 | 0.808521  0.715121  0.0   2.2176  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.038914  0.039478  0.0  0.0  0.0 |  0.545884 |      0.5 | 0.013393 | 0.0  0.0  0.0       0.0  0.0 | 0.084048 | 0.089159 | 0.083559 | 0.083559 | 0.348162 |      2.0       2.0  0.0  0.0       2.0 |        0.0   34.747124    186.1392  0.0    135.1552 |        0.0    13.89885         0.0  0.0         0.0 | 122.211002  121.082989  0.0  0.0  0.0 | 2.390661 | 16.725551 | 0.025807  0.025642  0.025352  0.025004  0.024672 | 0.348162 |
    | 02/01 03:00 | 10.4 | 5.0 | 17.7 |  0.096652 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  106.141894       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 10.9824  26.50752   61.6512  10.9824   45.4272 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  10.9824  26.50752   61.6512  10.9824   45.4272 |        0.0         0.0     47.7904  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0     30.9884         0.0  0.0     19.4176 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |  10.9824        0.0  0.0  10.9824  0.0 | 4.100699       0.0  0.0  10.9824  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.035454  0.039459  0.0  0.0  0.0 |  1.680699 |      0.5 | 0.027346 | 0.0  0.0  0.0       0.0  0.0 | 0.093615 | 0.109251 | 0.088277 | 0.088277 | 0.367821 |      2.0       2.0  0.0  0.0       2.0 |        0.0   77.456804       200.0  0.0       200.0 |        0.0    28.68509         0.0  0.0         0.0 | 129.128157  121.122447  0.0  0.0  0.0 | 3.544015 | 18.629421 | 0.031485  0.030819   0.02968  0.028345  0.027123 | 0.367821 |
    | 02/01 04:00 |  3.5 | 5.0 | 17.5 |  0.096321 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  134.445694       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |   3.696    8.9208    20.748    3.696    15.288 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |    3.696    8.9208    20.748    3.696    15.288 |        0.0         0.0      20.748  0.0      15.288 |       0.0        0.0        0.0  0.0        0.0 |        0.0      19.383         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |    3.696        0.0  0.0    3.696  0.0 | 1.540685       0.0  0.0    3.696  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.034358  0.039439  0.0  0.0  0.0 |   0.61449 |      0.5 | 0.045862 | 0.0  0.0  0.0       0.0  0.0 |  0.09793 | 0.128091 | 0.101566 | 0.101566 | 0.423191 |      2.0       2.0  0.0  0.0       2.0 |        0.0  100.811004       200.0  0.0       200.0 |        0.0    33.63469         0.0  0.0         0.0 |  131.31783  121.161886  0.0  0.0  0.0 | 3.612643 | 19.488063 | 0.036956  0.036297  0.035113  0.033601   0.03201 | 0.423191 |
    | 02/01 05:00 |  3.4 | 5.0 | 17.3 |   0.09599 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  161.940814       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  3.5904   8.66592   20.1552   3.5904   14.8512 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   3.5904   8.66592   20.1552   3.5904   14.8512 |        0.0         0.0     20.1552  0.0     14.8512 |       0.0        0.0        0.0  0.0        0.0 |        0.0     18.8292         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   3.5904        0.0  0.0   3.5904  0.0 | 1.547855       0.0  0.0   3.5904  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.03332  0.039419  0.0  0.0  0.0 |  0.617917 |      0.5 | 0.048567 | 0.0  0.0  0.0       0.0  0.0 | 0.102148 | 0.134191 |  0.11875 |  0.11875 | 0.494792 |      2.0       2.0  0.0  0.0       2.0 |        0.0  123.499084       200.0  0.0       200.0 |        0.0    38.44173         0.0  0.0         0.0 | 133.393695  121.201305  0.0  0.0  0.0 | 3.681992 | 20.327401 | 0.038849  0.038607  0.038123  0.037394  0.036445 | 0.494792 |
    | 02/01 06:00 |  1.2 | 5.0 | 17.5 |  0.187298 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  171.644974       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  1.2672   3.05856    7.1136   1.2672    5.2416 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   1.2672   3.05856    7.1136   1.2672    5.2416 |        0.0         0.0      7.1136  0.0      5.2416 |       0.0        0.0        0.0  0.0        0.0 |        0.0      6.6456         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   1.2672        0.0  0.0   1.2672  0.0 |  0.56371       0.0  0.0   1.2672  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032951  0.039399  0.0  0.0  0.0 |  0.208016 |      0.5 | 0.043417 | 0.0  0.0  0.0       0.0  0.0 | 0.104685 | 0.132439 | 0.129696 | 0.129696 | 0.540399 |      2.0       2.0  0.0  0.0       2.0 |        0.0  131.532524       200.0  0.0       200.0 |        0.0    40.11245         0.0  0.0         0.0 | 134.130136  121.240705  0.0  0.0  0.0 | 3.346592 | 20.832316 | 0.038422  0.038464  0.038504  0.038474  0.038298 | 0.540399 |
    | 02/01 07:00 |  0.1 | 5.0 | 17.8 |  1.264612 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  172.453654       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.1056   0.25488    0.5928   0.1056    0.4368 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.1056   0.25488    0.5928   0.1056    0.4368 |        0.0         0.0      0.5928  0.0      0.4368 |       0.0        0.0        0.0  0.0        0.0 |        0.0      0.5538         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   0.1056        0.0  0.0   0.1056  0.0 | 0.047496       0.0  0.0   0.1056  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032906   0.03938  0.0  0.0  0.0 | -0.007047 |      0.5 |  0.02929 | 0.0  0.0  0.0       0.0  0.0 |  0.10638 |  0.12272 |  0.13179 |  0.13179 | 0.549124 |      2.0       2.0  0.0  0.0       2.0 |        0.0  132.238644       200.0  0.0       200.0 |        0.0    40.21501         0.0  0.0         0.0 | 134.221146  121.280084  0.0  0.0  0.0 | 2.810254 | 21.169593 | 0.035684  0.036004  0.036546  0.037165  0.037693 | 0.549124 |
    | 02/01 08:00 |  0.0 | 5.0 | 17.8 |  3.045538 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  172.453654       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032889   0.03936  0.0  0.0  0.0 | -0.026824 |      0.5 | 0.016481 | 0.0  0.0  0.0       0.0  0.0 | 0.107991 | 0.113976 | 0.126324 | 0.126324 | 0.526349 |      2.0       2.0  0.0  0.0       2.0 |        0.0  132.278644       200.0  0.0       200.0 |        0.0    40.17501         0.0  0.0         0.0 | 134.254036  121.319444  0.0  0.0  0.0 | 2.266949 | 21.490174 | 0.033142  0.033448     0.034  0.034706  0.035448 | 0.526349 |
    | 02/01 09:00 |  0.0 | 5.0 | 18.0 |  1.930758 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  172.453654       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032873   0.03934  0.0  0.0  0.0 | -0.026811 |      0.5 | 0.008099 | 0.0  0.0  0.0       0.0  0.0 | 0.109594 | 0.108767 | 0.118162 | 0.118162 |  0.49234 |      2.0       2.0  0.0  0.0       2.0 |        0.0  132.318644       200.0  0.0       200.0 |        0.0    40.13501         0.0  0.0         0.0 | 134.286909  121.358784  0.0  0.0  0.0 | 1.732039 | 21.809151 | 0.031596  0.031786  0.032142  0.032628  0.033197 |  0.49234 |
    | 02/01 10:00 |  0.4 | 5.0 | 18.2 |  2.461001 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  175.688374       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.4224   1.01952    2.3712   0.4224    1.7472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.4224   1.01952    2.3712   0.4224    1.7472 |        0.0         0.0      2.3712  0.0      1.7472 |       0.0        0.0        0.0  0.0        0.0 |        0.0      2.2152         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   0.4224        0.0  0.0   0.4224  0.0 | 0.190428       0.0  0.0   0.4224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032741  0.039321  0.0  0.0  0.0 |  0.052596 |      0.5 | 0.003479 | 0.0  0.0  0.0       0.0  0.0 |  0.11149 | 0.106841 | 0.111678 | 0.111678 | 0.465326 |      2.0       2.0  0.0  0.0       2.0 |        0.0  135.023124       200.0  0.0       200.0 |        0.0    40.66525         0.0  0.0         0.0 | 134.551621  121.398105  0.0  0.0  0.0 | 1.281156 | 22.186575 | 0.031005   0.03108  0.031229  0.031452  0.031747 | 0.465326 |
    | 02/01 11:00 |  0.1 | 5.0 | 18.3 |  6.215945 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  176.497054       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  0.1056   0.25488    0.5928   0.1056    0.4368 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.1056   0.25488    0.5928   0.1056    0.4368 |        0.0         0.0      0.5928  0.0      0.4368 |       0.0        0.0        0.0  0.0        0.0 |        0.0      0.5538         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |   0.1056        0.0  0.0   0.1056  0.0 | 0.047795       0.0  0.0   0.1056  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.032695  0.039301  0.0  0.0  0.0 | -0.006809 |      0.5 | 0.001141 | 0.0  0.0  0.0       0.0  0.0 | 0.113151 | 0.106521 | 0.108167 | 0.108167 | 0.450696 |      2.0       2.0  0.0  0.0       2.0 |        0.0  135.729244       200.0  0.0       200.0 |        0.0    40.76781         0.0  0.0         0.0 | 134.642122  121.437406  0.0  0.0  0.0 | 0.773206 | 22.517081 | 0.030895   0.03091  0.030945  0.031008  0.031108 | 0.450696 |
    | 02/01 12:00 |  3.6 | 5.0 | 18.0 |  3.374783 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |   2.998287       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  3.8016   9.17568   21.3408   3.8016   15.7248 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   3.8016   9.17568   21.3408   3.8016   15.7248 |        0.0         0.0     21.3408  0.0     15.7248 |       0.0        0.0        0.0  0.0        0.0 |   4.487627   14.327266         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |  5.29094        0.0  0.0   3.8016  0.0 |  2.39792       0.0  0.0   3.8016  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.031232  0.039281  0.0  0.0  0.0 |  0.973026 |      0.5 | 0.001092 | 0.0  0.0  0.0       0.0  0.0 | 0.117444 | 0.110488 | 0.107159 | 0.107159 | 0.446497 |      2.0       2.0  0.0  0.0       2.0 |   2.141634   154.14003       200.0  0.0       200.0 |   0.856653    45.85997         0.0  0.0         0.0 | 137.566374  121.476687  0.0  0.0  0.0 | 1.245141 | 23.371295 | 0.032003  0.031874  0.031661  0.031424  0.031233 | 0.446497 |
    | 02/01 13:00 |  5.9 | 5.0 | 17.8 |  8.821555 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  47.398383       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  6.2304  15.03792   34.9752   6.2304   25.7712 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   6.2304  15.03792   34.9752   6.2304   25.7712 |        0.0   11.242668     34.9752  0.0     25.7712 |       0.0   3.795252        0.0  0.0        0.0 |  35.133494         0.0         0.0  0.0         0.0 |  3.036202        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 |      0.0        0.0  0.0   6.2304  0.0 |      0.0       0.0  0.0   6.2304  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.031217  0.039262  0.0  0.0  0.0 | -0.026094 |      0.5 | 0.001006 | 0.0  0.0  0.0       0.0  0.0 |  0.12345 | 0.116024 | 0.109236 | 0.109236 | 0.455148 |      2.0       2.0  0.0  0.0       2.0 |  34.929135  149.564242       200.0  0.0       200.0 |  12.469248   50.435758         0.0  0.0         0.0 |  137.59759  121.515949  0.0  0.0  0.0 | 0.718041 | 24.566474 | 0.033592  0.033403   0.03307  0.032661  0.032258 | 0.455148 |
    | 02/01 14:00 |  1.1 | 5.0 | 17.7 |  4.046025 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |   54.77413       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  1.1616   2.80368    6.5208   1.1616    4.8048 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   1.1616   2.80368    6.5208   1.1616    4.8048 |        0.0    2.084747      6.5208  0.0      4.8048 |       0.0   0.718933        0.0  0.0        0.0 |   6.541237         0.0         0.0  0.0         0.0 |  0.575147        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0   0.0 | 0.902237        0.0  0.0   1.1616  0.0 | 0.427054       0.0  0.0   1.1616  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.030964  0.039242  0.0  0.0  0.0 |  0.151957 |      0.5 | 0.000176 | 0.0  0.0  0.0       0.0  0.0 | 0.125805 | 0.117559 | 0.113441 | 0.113441 | 0.472671 |      2.0       2.0  0.0  0.0       2.0 |  39.124378  148.755016       200.0  0.0       200.0 |  15.649751   51.244984         0.0  0.0         0.0 | 138.103738  121.555191  0.0  0.0  0.0 | 0.369822 | 25.035183 | 0.034075  0.034013  0.033885  0.033691  0.033436 | 0.472671 |
    | 02/01 15:00 | 20.7 | 5.0 | 16.8 |  2.110757 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 204.057976  195.707265  204.057976  0.0  204.057976 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 21.8592  52.76016  122.7096  21.8592   90.4176 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  21.8592  52.76016  122.7096  21.8592   90.4176 |        0.0   35.903699    122.7096  0.0     90.4176 |       0.0  16.856461        0.0  0.0        0.0 | 114.593284    3.649347    3.649347  0.0    3.649347 | 12.831363   0.408629   0.408629  0.0   0.408629 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04  0.408629  0.0  0.34 |      0.0   8.350711  0.0  21.8592  0.0 |      0.0  3.084683  0.0  21.8592  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.030948  0.036589  0.0  0.0  0.0 |  1.003136 |      0.5 | 0.000282 | 0.0  0.0  0.0       0.0  0.0 | 0.142932 | 0.133629 | 0.117552 | 0.117552 | 0.489802 |      2.0       2.0  0.0  0.0       2.0 | 151.371669  139.790904  204.057976  0.0  203.989347 |  52.686307   55.916362         0.0  0.0    0.068629 | 138.134686  126.857809  0.0  0.0  0.0 | 0.872675 | 28.443565 | 0.038595  0.038067  0.037171  0.036136  0.035208 | 0.489802 |
    | 02/01 16:00 | 37.9 | 5.0 | 16.4 |  2.239257 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 302.049358  306.424159  311.186287  0.0  311.186287 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 40.0224  96.59952  224.6712  40.0224  165.5472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  40.0224  96.59952  224.6712  40.0224  165.5472 |  27.337389   57.587039  228.729176  0.0  169.573684 | 16.742987  34.719746        0.0  0.0   0.031492 |  94.061255   94.061255   94.061255  0.0   94.061255 | 17.125032  17.125032  17.125032  0.0  17.125032 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 9.136928   4.762128  0.0  40.0224  0.0 | 4.358587  1.915912  0.0  40.0224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028543  0.035148  0.0  0.0  0.0 |  2.431106 |      0.5 | 0.007994 | 0.0  0.0  0.0       0.0  0.0 | 0.172948 | 0.167814 | 0.129008 | 0.129008 | 0.537534 |      2.0       2.0  0.0  0.0       2.0 | 215.749542  218.874399  294.701255  0.0  294.364119 |  86.299817    87.54976   16.485032  0.0   16.822168 | 142.941571  129.739173  0.0  0.0  0.0 | 2.795787 | 34.416674 | 0.048327  0.047177  0.045183  0.042789  0.040507 | 0.537534 |
    | 02/01 17:00 |  8.2 | 5.0 | 16.3 |  2.877848 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 329.940893  329.940893  329.940893  0.0  329.940893 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  8.6592  20.90016   48.6096   8.6592   35.8176 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   8.6592  20.90016   48.6096   8.6592   35.8176 |  76.873713   88.721551  152.474401  0.0  139.877384 | 33.834845  38.602768   7.321485  0.0   7.126503 | 102.811951  102.811951  102.811951  0.0  102.811951 | 27.128942  27.128942  27.128942  0.0  27.128942 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   8.6592  0.0 |      0.0       0.0  0.0   8.6592  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028529   0.03513  0.0  0.0  0.0 | -0.023597 |      0.5 | 0.016234 | 0.0  0.0  0.0       0.0  0.0 | 0.180411 | 0.181371 | 0.152802 | 0.152802 | 0.636676 |      2.0       2.0  0.0  0.0       2.0 | 239.341786  242.215039  294.288404  0.0  293.456285 |  90.599108   87.725855   35.652489  0.0   36.484608 |   142.9701  129.774303  0.0  0.0  0.0 | 2.255956 | 35.901863 | 0.052454  0.051935  0.050931  0.049494  0.047739 | 0.636676 |
    | 02/01 18:00 |  3.6 | 5.0 | 15.8 |  1.591452 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 340.360733  340.360733  340.360733  0.0  340.360733 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  3.8016   9.17568   21.3408   3.8016   15.7248 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   3.8016   9.17568   21.3408   3.8016   15.7248 |   95.91277  101.023216  135.927722  0.0   130.29085 | 37.829723  38.093357  15.353971  0.0  15.374843 | 109.363903  109.363903  109.363903  0.0  109.363903 | 30.996831  30.996831  30.996831  0.0  30.996831 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   3.8016  0.0 |      0.0       0.0  0.0   3.8016  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028515  0.035113  0.0  0.0  0.0 | -0.023586 |      0.5 | 0.007986 | 0.0  0.0  0.0       0.0  0.0 | 0.184368 | 0.178465 | 0.172373 | 0.172373 |  0.71822 |      2.0       2.0  0.0  0.0       2.0 | 250.446924  254.639245  289.705384  0.0  288.594137 |  89.913809   85.721488   50.655349  0.0   51.766596 | 142.998615  129.809416  0.0  0.0  0.0 | 1.724384 | 36.689152 | 0.051778   0.05184   0.05188   0.05177  0.051377 |  0.71822 |
    | 02/01 19:00 |  7.5 | 5.0 | 15.5 |  0.291604 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 362.068733  362.068733  362.068733  0.0  362.068733 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |    7.92    19.116     44.46     7.92     32.76 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |     7.92    19.116     44.46     7.92     32.76 |  106.62793  116.704262   160.49211  0.0  149.102044 | 41.652803  42.772471  24.328623  0.0  24.018689 | 125.425775  125.425775  125.425775  0.0  125.425775 | 36.642959  36.642959  36.642959  0.0  36.642959 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0     7.92  0.0 |      0.0       0.0  0.0     7.92  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028501  0.035095  0.0  0.0  0.0 | -0.023574 |      0.5 | 0.003201 | 0.0  0.0  0.0       0.0  0.0 | 0.191246 | 0.181057 | 0.178425 | 0.178425 | 0.743438 |      2.0       2.0  0.0  0.0       2.0 | 266.898776  271.824758  299.739049  0.0  298.017868 |  95.169957   90.243976   62.329684  0.0   64.050865 | 143.027116  129.844511  0.0  0.0  0.0 | 1.197609 | 38.057906 | 0.052482  0.052402  0.052273  0.052129   0.05199 | 0.743438 |
    | 02/01 20:00 | 18.5 | 5.0 | 15.3 |  0.092622 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 415.615133  415.615133  415.615133  0.0  415.615133 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  19.536   47.1528   109.668   19.536    80.808 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   19.536   47.1528   109.668   19.536    80.808 | 127.016457  149.598796  235.832672  0.0  207.750786 | 54.588277  59.622737  35.904062  0.0  35.125947 | 165.286636  165.286636  165.286636  0.0  165.286636 | 50.328497  50.328497  50.328497  0.0  50.328497 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   19.536  0.0 |      0.0       0.0  0.0   19.536  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028486  0.035078  0.0  0.0  0.0 | -0.023562 |      0.5 | 0.000875 | 0.0  0.0  0.0       0.0  0.0 | 0.206387 | 0.193328 | 0.180903 | 0.180903 | 0.753762 |      2.0       2.0  0.0  0.0       2.0 | 302.822962  308.331798  339.501014  0.0  336.701718 | 112.792171  107.283336    76.11412  0.0   78.913415 | 143.055602  129.879589  0.0  0.0  0.0 | 0.673172 | 41.070948 | 0.055944  0.055539  0.054848  0.054046  0.053324 | 0.753762 |
    | 02/01 21:00 | 15.4 | 5.0 | 15.2 |  0.092451 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 460.188893  460.188893  460.188893  0.0  460.188893 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 | 16.2624  39.25152   91.2912  16.2624   67.2672 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |  16.2624  39.25152   91.2912  16.2624   67.2672 | 162.587391  182.453457  260.823055  0.0  236.653243 | 69.290142  72.413197  46.083278  0.0   46.22909 | 197.067169  197.067169  197.067169  0.0  197.067169 | 63.121725  63.121725  63.121725  0.0  63.121725 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0  16.2624  0.0 |      0.0       0.0  0.0  16.2624  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028472   0.03506  0.0  0.0  0.0 |  -0.02355 |      0.5 | 0.000096 | 0.0  0.0  0.0       0.0  0.0 | 0.219114 | 0.204583 |  0.18868 |  0.18868 | 0.786167 |      2.0       2.0  0.0  0.0       2.0 | 334.956746   340.28279  367.676327  0.0  364.722844 | 125.232147  119.906104   92.512567  0.0   95.466049 | 143.084074  129.914649  0.0  0.0  0.0 | 0.149526 | 43.603606 | 0.059215  0.058821  0.058112  0.057205  0.056251 | 0.786167 |
    | 02/01 22:00 |  6.3 | 5.0 | 14.5 |  0.091248 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 478.423613  478.423613  478.423613  0.0  478.423613 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  6.6528  16.05744   37.3464   6.6528   27.5184 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   6.6528  16.05744   37.3464   6.6528   27.5184 | 191.457675  201.485302   242.21107  0.0  231.390156 | 75.384018  74.761032  55.324223  0.0  56.317137 |  209.32715   209.32715   209.32715  0.0   209.32715 | 69.096463  69.096463  69.096463  0.0  69.096463 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   6.6528  0.0 |      0.0       0.0  0.0   6.6528  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028458  0.035043  0.0  0.0  0.0 | -0.023538 | 0.125987 |      0.0 | 0.0  0.0  0.0       0.0  0.0 |  0.22331 | 0.208423 | 0.199001 | 0.199001 |  0.82917 |      2.0       2.0  0.0  0.0       2.0 | 350.480227  355.240798  372.778807  0.0  370.518238 | 127.943386  123.182816  105.644807  0.0  107.905375 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.438685 | 0.060401   0.06025   0.05995  0.059502  0.058924 |  0.82917 |
    | 02/01 23:00 |  1.9 | 5.0 | 13.6 |  0.089683 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 483.922973  483.922973  483.922973  0.0  483.922973 | 0.0  0.0  0.902305  0.0   0.0 |        1.0 |  2.0064   4.84272   11.2632   2.0064    8.2992 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.0064   4.84272   11.2632   2.0064    8.2992 | 204.577508  209.475116  227.189914  0.0  223.156997 | 75.852505  73.791217  62.496899  0.0  63.565816 | 212.731903  212.731903  212.731903  0.0  212.731903 |  71.19107   71.19107   71.19107  0.0   71.19107 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   2.0064  0.0 |      0.0       0.0  0.0   2.0064  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.223627 | 0.208718 | 0.205747 | 0.205747 |  0.85728 |      2.0       2.0  0.0  0.0       2.0 | 356.288629  360.671665  370.223996  0.0  368.732344 | 127.634344  123.251308  113.698978  0.0  115.190629 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.501687 | 0.060524  0.060505  0.060453  0.060349  0.060166 |  0.85728 |
    | 03/01 00:00 |  4.9 | 5.0 | 13.7 |  0.089858 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 498.105533  498.105533  498.105533  0.0  498.105533 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  5.1744  12.48912   29.0472   5.1744   21.4032 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   5.1744  12.48912   29.0472   5.1744   21.4032 |  210.59632  218.646568   243.60086  0.0  235.726149 | 78.501053  77.765525  69.369313  0.0  69.600025 | 222.622815  222.622815  222.622815  0.0  222.622815 | 75.482718  75.482718  75.482718  0.0  75.482718 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   5.1744  0.0 |      0.0       0.0  0.0   5.1744  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.226204 | 0.211124 | 0.208326 | 0.208326 | 0.868023 |      2.0       2.0  0.0  0.0       2.0 | 365.968141  370.191592  378.933151  0.0  377.372211 | 132.137392  127.913941  119.172382  0.0  120.733323 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 45.014682 | 0.061202  0.061123  0.060986  0.060823  0.060662 | 0.868023 |
    | 03/01 01:00 |  2.7 | 5.0 | 13.6 |  0.089683 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 505.920413  505.920413  505.920413  0.0  505.920413 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  2.8512   6.88176   16.0056   2.8512   11.7936 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.8512   6.88176   16.0056   2.8512   11.7936 | 219.860457  225.408936  241.299304  0.0  236.521578 | 81.096276  79.578357  72.811829  0.0  73.377556 | 227.796732  227.796732  227.796732  0.0  227.796732 | 78.123681  78.123681  78.123681  0.0  78.123681 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   2.8512  0.0 |      0.0       0.0  0.0   2.8512  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 |  0.22711 | 0.211969 | 0.210083 | 0.210083 | 0.875345 |      2.0       2.0  0.0  0.0       2.0 | 371.557433  375.652029  382.076179  0.0  380.780965 |  134.36298  130.268385  123.844234  0.0  125.139448 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 45.194887 | 0.061462  0.061429  0.061365  0.061271  0.061155 | 0.875345 |
    | 03/01 02:00 |  0.5 | 5.0 | 13.1 |  0.088805 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 507.367613  507.367613  507.367613  0.0  507.367613 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |   0.528    1.2744     2.964    0.528     2.184 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |    0.528    1.2744     2.964    0.528     2.184 | 224.826819  227.862889  233.713014  0.0   232.22234 | 81.621594  79.331924    75.1714  0.0  75.882073 | 228.380541  228.380541  228.380541  0.0  228.380541 | 78.987072  78.987072  78.987072  0.0  78.987072 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0    0.528  0.0 |      0.0       0.0  0.0    0.528  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.226352 | 0.211261 | 0.211351 | 0.211351 | 0.880629 |      2.0       2.0  0.0  0.0       2.0 | 372.764171  376.771281  380.347707  0.0  379.463166 | 134.603442  130.596333  127.019907  0.0  127.904447 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 45.043964 | 0.061272  0.061293  0.061325   0.06135  0.061353 | 0.880629 |
    | 03/01 03:00 |  0.2 | 5.0 | 13.3 |  0.089157 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 507.946493  507.946493  507.946493  0.0  507.946493 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.2112   0.50976    1.1856   0.2112    0.8736 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.2112   0.50976    1.1856   0.2112    0.8736 | 225.884845  228.536493  231.486744  0.0  230.668948 | 81.693969   79.34088  77.066469  0.0  77.572265 | 228.361491  228.361491  228.361491  0.0  228.361491 | 79.585003  79.585003  79.585003  0.0  79.585003 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.2112  0.0 |      0.0       0.0  0.0   0.2112  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.225371 | 0.210346 | 0.211405 | 0.211405 | 0.880855 |      2.0       2.0  0.0  0.0       2.0 | 372.893834  376.860918  379.048053  0.0  378.369308 |  135.05266  131.085575  128.898441  0.0  129.577185 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.848765 |  0.06101  0.061041  0.061096  0.061162  0.061226 | 0.880855 |
    | 03/01 04:00 |  0.5 | 5.0 | 13.9 |  0.090207 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 509.393693  509.393693  509.393693  0.0  509.393693 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |   0.528    1.2744     2.964    0.528     2.184 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |    0.528    1.2744     2.964    0.528     2.184 | 226.222235  229.187234  232.470378  0.0  231.354892 | 82.252258  80.033659  78.440115  0.0  78.775601 | 228.971242  228.971242  228.971242  0.0  228.971242 | 80.422452  80.422452  80.422452  0.0  80.422452 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0    0.528  0.0 |      0.0       0.0  0.0    0.528  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.224621 | 0.209646 | 0.210736 | 0.210736 | 0.878067 |      2.0       2.0  0.0  0.0       2.0 | 373.295856  377.246525  379.152916  0.0  378.509658 | 136.097837  132.147168  130.240778  0.0  130.884036 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.699572 | 0.060805  0.060829  0.060875  0.060935     0.061 | 0.878067 |
    | 03/01 05:00 |  2.4 | 5.0 | 14.7 |  0.091593 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 516.340253  516.340253  516.340253  0.0  516.340253 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  2.5344   6.11712   14.2272   2.5344   10.4832 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.5344   6.11712   14.2272   2.5344   10.4832 |  227.45668   232.53804  243.126327  0.0  239.344792 | 84.471413  82.972773  80.494566  0.0  80.532101 | 233.550718  233.550718  233.550718  0.0  233.550718 | 82.789535  82.789535  82.789535  0.0  82.789535 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   2.5344  0.0 |      0.0       0.0  0.0   2.5344  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.225308 | 0.210288 | 0.210073 | 0.210073 | 0.875305 |      2.0       2.0  0.0  0.0       2.0 |  377.04291  380.994883  384.444507  0.0  383.538784 | 139.297343   135.34537  131.895746  0.0  132.801469 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.836321 | 0.060977  0.060958  0.060929  0.060902  0.060892 | 0.875305 |
    | 03/01 06:00 |  0.4 | 5.0 | 15.0 |  0.154861 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 517.498013  517.498013  517.498013  0.0  517.498013 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.4224   1.01952    2.3712   0.4224    1.7472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.4224   1.01952    2.3712   0.4224    1.7472 | 231.117926  233.986173  237.670858  0.0  236.552027 | 85.644727  83.373601  81.040595  0.0  81.535426 | 233.930686  233.930686  233.930686  0.0  233.930686 | 83.567327  83.567327  83.567327  0.0  83.567327 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.4224  0.0 |      0.0       0.0  0.0   0.4224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.224483 | 0.209518 | 0.210026 | 0.210026 | 0.875107 |      2.0       2.0  0.0  0.0       2.0 | 377.508687  381.428677  383.715535  0.0  383.004643 | 139.989326  136.069336  133.782478  0.0   134.49337 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.672181 | 0.060767  0.060791  0.060829  0.060868  0.060895 | 0.875107 |
    | 03/01 07:00 |  0.2 | 5.0 | 15.7 |  0.470369 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.076893  518.076893  518.076893  0.0  518.076893 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.2112   0.50976    1.1856   0.2112    0.8736 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.2112   0.50976    1.1856   0.2112    0.8736 | 231.670569  234.299035  236.486524  0.0  235.768959 | 86.038644  83.708738   82.19709  0.0  82.602655 | 233.905963  233.905963  233.905963  0.0  233.905963 | 84.170931  84.170931  84.170931  0.0  84.170931 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.2112  0.0 |      0.0       0.0  0.0   0.2112  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.223512 | 0.208611 | 0.209737 | 0.209737 | 0.873904 |      2.0       2.0  0.0  0.0       2.0 | 377.397097  381.300245  382.960574  0.0  382.355247 | 140.679796  136.776649  135.116319  0.0  135.721646 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 |  44.47884 | 0.060506  0.060537  0.060593  0.060661  0.060728 | 0.873904 |
    | 03/01 08:00 |  0.0 | 5.0 | 16.1 |  1.173726 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.076893  518.076893  518.076893  0.0  518.076893 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 231.705559  234.101923  235.121294  0.0  234.749649 | 86.371334   83.97497  82.955599  0.0  83.327244 | 233.465309  233.465309  233.465309  0.0  233.465309 | 84.611584  84.611584  84.611584  0.0  84.611584 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.222394 | 0.207568 | 0.208983 | 0.208983 | 0.870761 |      2.0       2.0  0.0  0.0       2.0 | 376.809864  380.703631  381.944589  0.0  381.410907 | 141.267029  137.373263  136.132304  0.0  136.665986 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.256446 | 0.060205  0.060241  0.060306  0.060387  0.060471 | 0.870761 |
    | 03/01 09:00 |  0.0 | 5.0 | 15.2 |  4.202296 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.076893  518.076893  518.076893  0.0  518.076893 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 231.345023  233.735628  234.497523  0.0  234.169865 |  86.73187  84.341265   83.57937  0.0  83.907028 |  233.02539   233.02539   233.02539  0.0   233.02539 | 85.051504  85.051504  85.051504  0.0  85.051504 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.221282 |  0.20653 | 0.208022 | 0.208022 | 0.866759 |      2.0       2.0  0.0  0.0       2.0 | 376.143247  380.033392  381.112456  0.0  380.606431 | 141.933647  138.043501  136.964438  0.0  137.470462 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.035164 | 0.059904   0.05994  0.060006  0.060089  0.060178 | 0.866759 |
    | 03/01 10:00 |  0.3 | 5.0 | 14.9 |  4.359715 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.945213  518.945213  518.945213  0.0  518.945213 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.3168   0.76464    1.7784   0.3168    1.3104 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.3168   0.76464    1.7784   0.3168    1.3104 | 231.024488  233.747413  235.584149  0.0  234.852459 | 87.369205   85.09412  84.271144  0.0  84.534834 |    233.209     233.209     233.209  0.0     233.209 | 85.736213  85.736213  85.736213  0.0  85.736213 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.3168  0.0 |      0.0       0.0  0.0   0.3168  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.220402 | 0.205709 | 0.207011 | 0.207011 | 0.862547 |      2.0       2.0  0.0  0.0       2.0 | 375.980775  379.871939  381.155706  0.0  380.613372 | 142.964438  139.073275  137.789507  0.0  138.331841 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 43.860019 | 0.059664  0.059693  0.059746  0.059817  0.059895 | 0.862547 |
    | 03/01 11:00 |  2.6 | 5.0 | 15.3 |  5.305753 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 526.470653  526.470653  526.470653  0.0  526.470653 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  2.7456   6.62688   15.4128   2.7456   11.3568 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   2.7456   6.62688   15.4128   2.7456   11.3568 | 231.841463   237.12537  248.140487  0.0  244.141138 |  89.84935  88.446724  86.217526  0.0  86.160875 | 238.174723  238.174723  238.174723  0.0  238.174723 | 88.295931  88.295931  88.295931  0.0  88.295931 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   2.7456  0.0 |      0.0       0.0  0.0   2.7456  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.221261 |  0.20651 | 0.206214 | 0.206214 | 0.859226 |      2.0       2.0  0.0  0.0       2.0 | 379.967051  383.881611  387.242741  0.0  386.343757 | 146.503602  142.589042  139.227912  0.0  140.126897 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 44.030986 |  0.05988  0.059856  0.059819  0.059785   0.05977 | 0.859226 |
    | 03/01 12:00 |  0.7 | 5.0 | 15.3 |  5.376027 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 528.496733  528.496733  528.496733  0.0  528.496733 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.7392   1.78416    4.1496   0.7392    3.0576 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.7392   1.78416    4.1496   0.7392    3.0576 |  235.82443  239.030617  243.869762  0.0  242.326539 | 91.385423  89.224197  86.750491  0.0  87.201714 | 239.167726  239.167726  239.167726  0.0  239.167726 | 89.329007  89.329007  89.329007  0.0  89.329007 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.7392  0.0 |      0.0       0.0  0.0   0.7392  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.220683 | 0.205971 | 0.206208 | 0.206208 |   0.8592 |      2.0       2.0  0.0  0.0       2.0 | 380.963364  384.844961  387.330305  0.0  386.582544 | 147.533369  143.651772  141.166428  0.0  141.914189 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 43.915903 | 0.059737  0.059752  0.059777  0.059799  0.059809 |   0.8592 |
    | 03/01 13:00 |  0.3 | 5.0 | 15.4 |  4.658915 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 529.365053  529.365053  529.365053  0.0  529.365053 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.3168   0.76464    1.7784   0.3168    1.3104 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.3168   0.76464    1.7784   0.3168    1.3104 | 236.881059  239.627403  242.351416  0.0  241.464963 | 91.932474   89.63397  87.923717  0.0   88.34217 | 239.341086  239.341086  239.341086  0.0  239.341086 | 90.023967  90.023967  90.023967  0.0  90.023967 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.3168  0.0 |      0.0       0.0  0.0   0.3168  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.219806 | 0.205152 | 0.206086 | 0.206086 | 0.858691 |      2.0       2.0  0.0  0.0       2.0 | 381.076407  384.935604  386.738375  0.0  386.109067 | 148.288646  144.429449  142.626678  0.0  143.255986 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 43.741355 | 0.059502   0.05953  0.059578  0.059637  0.059692 | 0.858691 |
    | 03/01 14:00 |  0.3 | 5.0 | 15.5 |  7.789594 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 530.233373  530.233373  530.233373  0.0  530.233373 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |  0.3168   0.76464    1.7784   0.3168    1.3104 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.3168   0.76464    1.7784   0.3168    1.3104 | 237.187616  239.922259  242.222296  0.0  241.409523 | 92.494237  90.207435  88.921158  0.0  89.265931 | 239.515656  239.515656  239.515656  0.0  239.515656 | 90.717717  90.717717  90.717717  0.0  90.717717 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0   0.3168  0.0 |      0.0       0.0  0.0   0.3168  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.218933 | 0.204338 | 0.205482 | 0.205482 | 0.856175 |      2.0       2.0  0.0  0.0       2.0 | 381.057463  384.905961  386.450135  0.0    385.8656 |  149.17591  145.327412  143.783238  0.0  144.367773 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 43.567679 | 0.059266  0.059295  0.059345  0.059411  0.059479 | 0.856175 |
    | 03/01 15:00 |  0.0 | 5.0 | 17.4 |  4.851567 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 530.233373  530.233373  530.233373  0.0  530.233373 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 237.325483  239.722357   240.68408  0.0  240.320027 |  92.90789  90.511016  89.549293  0.0  89.913346 | 239.068393  239.068393  239.068393  0.0  239.068393 |  91.16498   91.16498   91.16498  0.0   91.16498 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.217838 | 0.203316 | 0.204686 | 0.204686 | 0.852857 |      2.0       2.0  0.0  0.0       2.0 |  380.45339  384.291998  385.474448  0.0  384.953966 | 149.779984  145.941376  144.758925  0.0  145.279407 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 |  43.34984 | 0.058972  0.059007   0.05907  0.059148  0.059229 | 0.852857 |
    | 03/01 16:00 |  0.0 | 5.0 | 17.3 |   5.30692 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 530.233373  530.233373  530.233373  0.0  530.233373 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 236.949262  239.339976  240.076415  0.0  239.752255 | 93.284112  90.893398  90.156958  0.0  90.481119 | 238.622132  238.622132  238.622132  0.0  238.622132 | 91.611242  91.611242  91.611242  0.0  91.611242 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.216749 | 0.202299 | 0.203756 | 0.203756 | 0.848985 |      2.0       2.0  0.0  0.0       2.0 | 379.779276  383.614153  384.660165  0.0  384.163843 | 150.454097   146.61922  145.573208  0.0   146.06953 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 43.133091 | 0.058677  0.058713  0.058776  0.058858  0.058945 | 0.848985 |
    | 03/01 17:00 |  0.0 | 5.0 | 17.1 |  3.286036 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 530.233373  530.233373  530.233373  0.0  530.233373 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 236.529419  238.917809  239.569273  0.0   239.26016 | 93.703955  91.315564    90.6641  0.0  90.973213 |  238.17587   238.17587   238.17587  0.0   238.17587 | 92.057503  92.057503  92.057503  0.0  92.057503 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |      0.0        0.0  0.0      0.0  0.0 |      0.0       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0 |      0.0 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.215665 | 0.201288 | 0.202755 | 0.202755 | 0.844814 |      2.0       2.0  0.0  0.0       2.0 | 379.078744  382.912214  383.906762  0.0  383.419553 | 151.154629  147.321159  146.326612  0.0   146.81382 | 143.112532  129.949692  0.0  0.0  0.0 |      0.0 | 42.917426 | 0.058384  0.058419  0.058483  0.058564  0.058652 | 0.844814 |
    | 03/01 18:00 |  0.0 | 5.0 | 15.7 |  1.506216 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 529.715546  530.233373  530.233373  0.0  530.233373 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 236.093122  238.480636  239.100048  0.0  238.796611 | 94.140252  91.752737  91.133325  0.0  91.436763 | 237.729608  237.729608  237.729608  0.0  237.729608 | 92.503765  92.503765  92.503765  0.0  92.503765 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 0.517828        0.0  0.0      0.0  0.0 | 0.265143       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028317       0.0  0.0  0.0  0.0 |  0.098677 | 0.098677 |      0.0 | 0.0  0.0  0.0       0.0  0.0 |  0.21501 | 0.200676 | 0.201772 | 0.201772 | 0.840716 |      2.0       2.0  0.0  0.0       2.0 | 378.368247  382.201186  383.176322  0.0  382.692551 | 151.347299  148.032187  147.057051  0.0  147.540823 | 143.393534  129.949692  0.0  0.0  0.0 |      0.0 | 42.786996 | 0.058202  0.058225  0.058266  0.058323  0.058389 | 0.840716 |
    | 03/01 19:00 |  0.0 | 5.0 | 14.2 |  0.274762 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 528.835785  530.034209  530.034209  0.0  530.034209 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 235.511104  238.037803  238.645125  0.0  238.343828 | 94.204442  92.195571  91.588249  0.0  91.889545 | 237.229687  237.229687  237.229687  0.0  237.229687 | 92.804522  92.804522  92.804522  0.0  92.804522 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 1.198424        0.0  0.0      0.0  0.0 | 0.616041       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.028012       0.0  0.0  0.0  0.0 |  0.245012 | 0.245012 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.214985 | 0.200653 | 0.201031 | 0.201031 | 0.837628 |      2.0       2.0  0.0  0.0       2.0 | 377.739846  381.433071  382.400885  0.0   381.91841 | 151.095939  148.601138  147.633324  0.0  148.115799 | 144.003929  129.949692  0.0  0.0  0.0 |      0.0 | 42.782022 |  0.05819  0.058192  0.058198  0.058212  0.058236 | 0.837628 |
    | 03/01 20:00 |  0.0 | 5.0 | 12.4 |  0.087565 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 |  528.03676  529.573277  529.573277  0.0  529.573277 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 234.882703  237.505353  238.107977  0.0  237.807557 | 93.953081  92.528856  91.926231  0.0  92.226652 | 236.658955  236.658955  236.658955  0.0  236.658955 | 92.914321  92.914321  92.914321  0.0  92.914321 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 1.536516        0.0  0.0      0.0  0.0 | 0.796574       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.027628       0.0  0.0  0.0  0.0 |  0.320394 | 0.320394 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.215283 | 0.200931 | 0.200744 | 0.200744 | 0.836433 |      2.0       2.0  0.0  0.0       2.0 | 377.169114  380.626673  381.591862  0.0  381.109808 | 150.867646  148.946603  147.981414  0.0  148.463468 |   144.7715  129.949692  0.0  0.0  0.0 |      0.0 | 42.841362 | 0.058267  0.058258  0.058244  0.058228  0.058217 | 0.836433 |
    | 03/01 21:00 |  0.0 | 5.0 | 11.4 |  0.085771 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 527.188356  528.982309  528.982309  0.0  528.982309 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 234.311972  236.878229  237.478902  0.0  237.178902 | 93.724789  92.695048  92.094374  0.0  92.394375 | 236.052952  236.052952  236.052952  0.0  236.052952 | 92.929356  92.929356  92.929356  0.0  92.929356 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 1.793952        0.0  0.0      0.0  0.0 | 0.939977       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.027187       0.0  0.0  0.0  0.0 |  0.380329 | 0.380329 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.215837 | 0.201448 | 0.200858 | 0.200858 | 0.836907 |      2.0       2.0  0.0  0.0       2.0 | 376.563112  379.841397  380.805913  0.0  380.323859 | 150.625245  149.140912  148.176396  0.0   148.65845 | 145.652663  129.949692  0.0  0.0  0.0 |      0.0 | 42.951521 | 0.058415  0.058397  0.058367   0.05833  0.058295 | 0.836907 |
    | 03/01 22:00 |  0.0 | 5.0 | 10.6 |  0.084317 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 526.302602  528.292327  528.292327  0.0  528.292327 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 233.705969  236.229261  236.829108  0.0  236.529311 | 93.482388  92.753048  92.153201  0.0  92.452998 | 235.420271  235.420271  235.420271  0.0  235.420271 | 92.872056  92.872056  92.872056  0.0  92.872056 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 1.989725        0.0  0.0      0.0  0.0 | 1.055285       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.026706       0.0  0.0  0.0  0.0 |  0.428575 | 0.428575 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.216594 | 0.202155 | 0.201255 | 0.201255 | 0.838562 |      2.0       2.0  0.0  0.0       2.0 |  375.93043  379.072407  380.037076  0.0  379.554818 | 150.372172   149.21992  148.255252  0.0  148.737509 | 146.613809  129.949692  0.0  0.0  0.0 |      0.0 | 43.102276 | 0.058618  0.058594  0.058551  0.058498  0.058444 | 0.838562 |
    | 03/01 23:00 |  0.0 | 5.0 | 10.0 |  0.083215 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 525.388479  527.527048  527.527048  0.0  527.527048 | 0.0  0.0  0.902686  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 | 0.001664  0.001664  0.001664  0.001664  0.001664 | 0.001065  0.002397  0.005059  0.001065  0.003728 | 0.001065  0.002397  0.0  0.0  0.003728 |      0.0       0.0       0.0      0.0       0.0 | 233.073287  235.563828  236.163294  0.0  235.863608 | 93.229315  92.728499  92.129034  0.0  92.428719 | 234.767326  234.767326  234.767326  0.0  234.767326 | 92.759722  92.759722  92.759722  0.0  92.759722 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.138569        0.0  0.0      0.0  0.0 | 1.149246       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.026198       0.0  0.0  0.0  0.0 |  0.467937 | 0.467937 |      0.0 | 0.0  0.0  0.0  0.001065  0.0 | 0.217516 | 0.203015 | 0.201868 | 0.201868 | 0.841117 | 1.998935  1.997603  0.0  0.0  1.996272 | 375.277485  378.315905  379.281108  0.0  378.798536 | 150.110994  149.211143   148.24594  0.0  148.728512 |  147.62933  129.949692  0.0  0.0  0.0 |      0.0 | 43.285697 | 0.058866  0.058836  0.058783  0.058717  0.058649 | 0.841117 |
    | 04/01 00:00 |  0.0 | 5.0 |  9.5 |  0.082289 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 524.451307  526.704522  526.704522  0.0  526.704522 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |  0.00576   0.00576   0.00576   0.00576   0.00576 | 0.003687  0.008295  0.017511  0.003687  0.012903 | 0.003687  0.008295  0.0  0.0  0.012903 |      0.0       0.0       0.0      0.0       0.0 | 232.420342   234.88595  235.485218  0.0  235.185602 | 92.968137  92.641098  92.041831  0.0  92.341446 | 234.098982  234.098982  234.098982  0.0  234.098982 |  92.60554   92.60554   92.60554  0.0   92.60554 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.253215        0.0  0.0      0.0  0.0 | 1.227688       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.025673       0.0  0.0  0.0  0.0 |   0.50084 |      0.5 |      0.0 | 0.0  0.0  0.0  0.003687  0.0 | 0.218569 | 0.203997 | 0.202653 | 0.202653 | 0.844388 | 1.995248  1.989309  0.0  0.0  1.983369 | 374.608076  377.568937  378.534872  0.0  378.051916 | 149.843231  149.135584  148.169649  0.0  148.652606 |  148.68053  129.949692  0.0  0.0  0.0 |  0.00084 | 43.495173 | 0.059149  0.059115  0.059055  0.058978  0.058898 | 0.844388 |
    | 04/01 01:00 |  0.0 | 5.0 | 10.7 |    0.0845 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 523.498661  525.837901  525.837901  0.0  525.837901 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 231.750934  234.198633  234.797784  0.0  234.498215 | 92.700373  92.505888  91.906738  0.0  92.206306 | 233.418521  233.418521  233.418521  0.0  233.418521 |  92.41938   92.41938   92.41938  0.0   92.41938 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.33924        0.0  0.0      0.0  0.0 | 1.292775       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.025137   0.00084  0.0  0.0  0.0 |  0.527903 |      0.5 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.219619 | 0.204977 | 0.203568 | 0.203568 | 0.848202 | 1.995248  1.989309  0.0  0.0  1.983369 | 373.927615  376.828825  377.795609  0.0  377.312221 | 149.571046  149.009076  148.042292  0.0   148.52568 | 149.752131  129.950532  0.0  0.0  0.0 | 0.028742 | 43.704126 | 0.059434  0.059399  0.059338  0.059259  0.059175 | 0.848202 |
    | 04/01 02:00 |  0.0 | 5.0 | 10.9 |  0.084864 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 522.533546  524.938193  524.938193  0.0  524.938193 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 231.070472  233.503734  234.102806  0.0  233.803273 | 92.428189  92.334167  91.735094  0.0  92.034628 | 232.729153  232.729153  232.729153  0.0  232.729153 |  92.20904   92.20904   92.20904  0.0   92.20904 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.404647        0.0  0.0      0.0  0.0 | 1.348147       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.024596  0.028742  0.0  0.0  0.0 |  0.541899 |      0.5 |      0.0 | 0.0  0.0  0.0       0.0  0.0 | 0.220663 | 0.205953 | 0.204537 | 0.204537 | 0.852239 | 1.995248  1.989309  0.0  0.0  1.983369 | 373.238247  376.094244  377.061956  0.0  376.578101 | 149.295299  148.843949  147.876237  0.0  148.360092 | 150.833226  129.979274  0.0  0.0  0.0 | 0.070641 | 43.912033 | 0.059716  0.059682  0.059621  0.059542  0.059458 | 0.852239 |
    | 04/01 03:00 |  1.3 | 5.0 | 10.2 |  0.083584 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 524.784249  527.769652  527.769652  0.0  527.769652 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  1.3728   3.31344    7.7064   1.3728    5.6784 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 | 1.368048  3.302749    7.7064   1.3728  5.661769 | 230.754143  234.603676  240.293649  0.0  238.161507 | 93.147452  93.637266  92.350944  0.0  92.438455 | 234.709836  234.709836  234.709836  0.0  234.709836 | 93.059816  93.059816  93.059816  0.0  93.059816 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.985403        0.0  0.0   1.3728  0.0 | 1.697997       0.0  0.0   1.3728  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.02394   0.03501  0.0  0.0  0.0 |  0.685854 |      0.5 | 0.000006 | 0.0  0.0  0.0       0.0  0.0 | 0.222684 | 0.207843 | 0.205579 | 0.205579 | 0.856578 |      2.0       2.0  0.0  0.0       2.0 | 374.845892  377.695853  379.824543  0.0    379.1282 | 149.938357  150.073799   147.94511  0.0  148.641453 | 152.144572  130.014284  0.0  0.0  0.0 | 0.256489 | 44.314036 | 0.060256  0.060192   0.06008  0.059945  0.059812 | 0.856578 |
    | 04/01 04:00 |  0.0 | 5.0 | 10.1 |    0.0834 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 524.240982   526.62142   526.62142  0.0   526.62142 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 | 0.000834  0.000834  0.000834  0.000834  0.000834 | 0.000534  0.001201  0.002535  0.000534  0.001868 | 0.000534  0.001201  0.0  0.0  0.001868 |      0.0       0.0       0.0      0.0       0.0 | 231.988749    234.5668  235.888816  0.0  235.456354 |   92.7955  93.202852  91.880837  0.0  92.313298 |  233.94875   233.94875   233.94875  0.0   233.94875 | 92.672671  92.672671  92.672671  0.0  92.672671 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.380438        0.0  0.0      0.0  0.0 | 1.377558       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.023426  0.034993  0.0  0.0  0.0 |  0.552557 |      0.5 | 0.000023 | 0.0  0.0  0.0  0.000534  0.0 | 0.223713 | 0.208817 | 0.207012 | 0.207012 | 0.862551 | 1.999466  1.998799  0.0  0.0  1.998132 | 374.457844  377.117803  378.524477  0.0  377.960595 | 149.783138  149.503617  148.096944  0.0  148.660825 | 153.170878  130.049277  0.0  0.0  0.0 | 0.309024 | 44.518818 | 0.060547  0.060511  0.060443  0.060349  0.060239 | 0.862551 |
    | 04/01 05:00 |  0.0 | 5.0 | 10.9 |  0.084864 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 523.269541  525.705867  525.705867  0.0  525.705867 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 231.600701  233.896206  234.768654  0.0  234.418924 | 92.640281  92.725214  91.852766  0.0  92.202497 | 233.254863  233.254863  233.254863  0.0  233.254863 | 92.451004  92.451004  92.451004  0.0  92.451004 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.436327        0.0  0.0      0.0  0.0 | 1.428986       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022911  0.034975  0.0  0.0  0.0 |  0.574206 |      0.5 | 0.000042 | 0.0  0.0  0.0       0.0  0.0 | 0.224737 | 0.209788 | 0.208302 | 0.208302 | 0.867927 | 1.999466  1.998799  0.0  0.0  1.998132 | 373.763958   376.51646  377.650685  0.0  377.136535 | 149.505583  149.189408  148.055182  0.0  148.569332 |  154.20113  130.084253  0.0  0.0  0.0 | 0.383188 | 44.722653 | 0.060829  0.060795  0.060733  0.060653  0.060565 | 0.867927 |
    | 04/01 06:00 |  0.0 | 5.0 | 11.2 |  0.310229 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 522.290279  524.768818  524.768818  0.0  524.768818 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 230.906815  233.274208  233.976928  0.0  233.658382 | 92.362726  92.431659  91.728939  0.0  92.047486 |  232.55539   232.55539   232.55539  0.0   232.55539 | 92.213428  92.213428  92.213428  0.0  92.213428 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.47854        0.0  0.0      0.0  0.0 | 1.473367       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.022397  0.034958  0.0  0.0  0.0 |  0.592918 |      0.5 |  0.00008 | 0.0  0.0  0.0       0.0  0.0 | 0.225756 |  0.21077 | 0.209343 | 0.209343 | 0.872261 | 1.999466  1.998799  0.0  0.0  1.998132 | 373.064485  375.837642  376.869147  0.0  376.373543 | 149.225794  148.931177  147.899671  0.0  148.395275 | 155.228699   130.11921  0.0  0.0  0.0 | 0.476026 | 44.925468 | 0.061113  0.061079  0.061018  0.060938  0.060853 | 0.872261 |
    | 04/01 07:00 |  0.7 | 5.0 | 10.8 |  1.391958 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 523.041495  525.840895  525.840895  0.0  525.840895 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.7392   1.78416    4.1496   0.7392    3.0576 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 | 0.738666  1.782959    4.1496   0.7392  3.055732 | 230.409097   233.57062  236.944072  0.0  235.658291 | 92.619848  92.981158  91.974347  0.0  92.166259 | 233.293729  233.293729  233.293729  0.0  233.293729 | 92.547167  92.547167  92.547167  0.0  92.547167 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |   2.7994        0.0  0.0   0.7392  0.0 | 1.686355       0.0  0.0   0.7392  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.021829   0.03494  0.0  0.0  0.0 |  0.681906 |      0.5 | 0.000188 | 0.0  0.0  0.0       0.0  0.0 | 0.227298 | 0.212295 | 0.210367 | 0.210367 |  0.87653 |      2.0       2.0  0.0  0.0       2.0 | 373.601068  376.386461  378.008404  0.0  377.404713 | 149.440427  149.454435  147.832491  0.0  148.436183 | 156.363574  130.154151  0.0  0.0  0.0 | 0.657744 | 45.232341 |  0.06155  0.061498  0.061407  0.061294   0.06118 |  0.87653 |
    | 04/01 08:00 |  0.4 | 5.0 | 10.4 |  3.195876 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 523.281753  525.921963  525.921963  0.0  525.921963 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.4224   1.01952    2.3712   0.4224    1.7472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.4224   1.01952    2.3712   0.4224    1.7472 | 230.859201  233.786185  236.354275  0.0  235.421637 | 92.604694   93.07423   91.85782  0.0  92.166459 | 233.378862  233.378862  233.378862  0.0  233.378862 | 92.543101  92.543101  92.543101  0.0  92.543101 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.64021        0.0  0.0   0.4224  0.0 |   1.6138       0.0  0.0   0.4224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.021305  0.034923  0.0  0.0  0.0 |  0.651899 |      0.5 |   0.0004 | 0.0  0.0  0.0       0.0  0.0 | 0.228606 | 0.213686 |  0.21165 |  0.21165 | 0.881876 |      2.0       2.0  0.0  0.0       2.0 | 373.772681  376.468418  378.044191  0.0  377.449138 | 149.509072  149.453545  147.877772  0.0  148.472825 | 157.411289  130.189074  0.0  0.0  0.0 | 0.809242 | 45.492649 | 0.061955  0.061906  0.061818  0.061704  0.061582 | 0.881876 |
    | 04/01 09:00 |  0.1 | 5.0 | 10.6 |  5.191651 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 522.680667  525.195938  525.195938  0.0  525.195938 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.1056   0.25488    0.5928   0.1056    0.4368 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.1056   0.25488    0.5928   0.1056    0.4368 | 230.944361  233.442269  234.809309  0.0  234.301005 | 92.442992  92.734574  91.705454  0.0  92.057758 | 232.863063  232.863063  232.863063  0.0  232.863063 | 92.332875  92.332875  92.332875  0.0  92.332875 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.51527        0.0  0.0   0.1056  0.0 | 1.558104       0.0  0.0   0.1056  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.020816  0.034905  0.0  0.0  0.0 |  0.628902 |      0.5 | 0.000671 | 0.0  0.0  0.0       0.0  0.0 | 0.229682 | 0.214906 | 0.213041 | 0.213041 | 0.887672 |      2.0       2.0  0.0  0.0       2.0 | 373.343334  376.041531  377.330745  0.0  376.787995 | 149.337334  149.154406  147.865192  0.0  148.407942 | 158.389271  130.223979  0.0  0.0  0.0 | 0.937472 | 45.706625 |  0.06231  0.062267  0.062189  0.062088  0.061976 | 0.887672 |
    | 04/01 10:00 |  0.4 | 5.0 | 12.0 |  7.155036 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 |  522.72229  525.386286  525.386286  0.0  525.386286 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |  0.4224   1.01952    2.3712   0.4224    1.7472 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |   0.4224   1.01952    2.3712   0.4224    1.7472 | 230.601546  233.397025  235.757443  0.0  234.863073 | 92.501521  92.818433  91.809695  0.0  92.080065 | 232.979326  232.979326  232.979326  0.0  232.979326 |  92.40696   92.40696   92.40696  0.0   92.40696 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.663996        0.0  0.0   0.4224  0.0 | 1.670802       0.0  0.0   0.4224  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.020309  0.034888  0.0  0.0  0.0 |  0.676076 |      0.5 | 0.001088 | 0.0  0.0  0.0       0.0  0.0 | 0.230978 | 0.216449 | 0.214363 | 0.214363 | 0.893181 |      2.0       2.0  0.0  0.0       2.0 | 373.373065  376.113112  377.563828  0.0  376.991448 | 149.349226  149.273174  147.822458  0.0  148.394838 | 159.402773  130.258867  0.0  0.0  0.0 | 1.112461 | 45.964561 | 0.062755  0.062702  0.062607  0.062488  0.062365 | 0.893181 |
    | 04/01 11:00 |  0.0 | 5.0 | 16.1 |  8.391432 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 521.869966  524.361672  524.361672  0.0  524.361672 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 230.515922  232.937273  233.835741  0.0  233.481251 | 92.206369  92.449013  91.550545  0.0  91.905035 | 232.255167  232.255167  232.255167  0.0  232.255167 | 92.106505  92.106505  92.106505  0.0  92.106505 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.491706        0.0  0.0      0.0  0.0 | 1.582809       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.019844  0.034871  0.0  0.0  0.0 |  0.639612 |      0.5 | 0.001658 | 0.0  0.0  0.0       0.0  0.0 | 0.231966 | 0.217828 | 0.215771 | 0.215771 | 0.899048 |      2.0       2.0  0.0  0.0       2.0 | 372.764262  375.471006  376.623254  0.0  376.105365 | 149.105705  148.890666  147.738418  0.0  148.256308 | 160.331514  130.293738  0.0  0.0  0.0 | 1.250415 | 46.161167 | 0.063156  0.063108   0.06302  0.062906  0.062783 | 0.899048 |
    | 04/01 12:00 |  0.0 | 5.0 | 17.2 |  8.391286 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 520.882597  523.403324  523.403324  0.0  523.403324 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 229.907119  232.260308   232.97307  0.0  232.652712 | 91.962848  92.101364  91.388603  0.0   91.70896 | 231.549904  231.549904  231.549904  0.0  231.549904 |  91.85342   91.85342   91.85342  0.0   91.85342 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.520726        0.0  0.0      0.0  0.0 | 1.619957       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.019384  0.034853  0.0  0.0  0.0 |  0.655288 |      0.5 | 0.002348 | 0.0  0.0  0.0       0.0  0.0 | 0.232949 | 0.219298 | 0.217202 | 0.217202 | 0.905007 |      2.0       2.0  0.0  0.0       2.0 | 372.058998  374.800602  375.840088  0.0  375.342557 | 148.823599  148.602722  147.563236  0.0  148.060767 | 161.251667  130.328591  0.0  0.0  0.0 | 1.403354 |  46.35679 | 0.063581   0.06353  0.063439  0.063321  0.063197 | 0.905007 |
    | 04/01 13:00 |  0.0 | 5.0 | 21.4 | 10.715238 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 519.891085  522.433814  522.433814  0.0  522.433814 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 229.201855  231.583857   232.22614  0.0  231.918723 | 91.680742  91.819467  91.177184  0.0  91.484601 | 230.841681  230.841681  230.841681  0.0  230.841681 | 91.592133  91.592133  91.592133  0.0  91.592133 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.542729        0.0  0.0      0.0  0.0 | 1.652907       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.018929  0.034836  0.0  0.0  0.0 |  0.669212 |      0.5 |   0.0033 | 0.0  0.0  0.0       0.0  0.0 | 0.233927 | 0.220972 | 0.218658 | 0.218658 | 0.911074 |      2.0       2.0  0.0  0.0       2.0 | 371.350775  374.098426  375.095628  0.0  374.605514 |  148.54031  148.335388  147.338185  0.0  147.828299 | 162.160418  130.363427  0.0  0.0  0.0 | 1.569267 | 46.551434 | 0.064065  0.064007  0.063904  0.063772  0.063635 | 0.911074 |
    | 04/01 14:00 |  0.0 | 5.0 | 20.4 |  9.383394 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 518.896368  521.455841  521.455841  0.0  521.455841 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 228.493632  230.884715  231.500165  0.0  231.197678 | 91.397453  91.549099  90.933649  0.0  91.236136 | 230.131169  230.131169  230.131169  0.0  230.131169 | 91.324673  91.324673  91.324673  0.0  91.324673 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.559473        0.0  0.0      0.0  0.0 | 1.682598       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.018481  0.034818  0.0  0.0  0.0 |  0.681776 |      0.5 | 0.004578 | 0.0  0.0  0.0       0.0  0.0 |   0.2349 | 0.222903 | 0.220257 | 0.220257 | 0.917736 |      2.0       2.0  0.0  0.0       2.0 | 370.640263  373.384879  374.366632  0.0  373.879005 | 148.256105  148.070962  147.089209  0.0  147.576836 | 163.055775  130.398245  0.0  0.0  0.0 | 1.746464 | 46.745106 | 0.064622  0.064556  0.064437  0.064286  0.064128 | 0.917736 |
    | 04/01 15:00 |  0.0 | 5.0 | 20.8 |  7.861915 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 517.899208  520.471428  520.471428  0.0  520.471428 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 |  227.78312  230.176251  230.781461  0.0  230.480859 | 91.113248   91.27959  90.674381  0.0  90.974982 | 229.418911  229.418911  229.418911  0.0  229.418911 | 91.052517  91.052517  91.052517  0.0  91.052517 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.572221        0.0  0.0      0.0  0.0 | 1.709703       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.018041  0.034801  0.0  0.0  0.0 |  0.693259 |      0.5 | 0.006257 | 0.0  0.0  0.0       0.0  0.0 | 0.235868 | 0.225149 | 0.222083 | 0.222083 | 0.925344 |      2.0       2.0  0.0  0.0       2.0 | 369.928006   372.66754  373.644083  0.0  373.157057 | 147.971202  147.803889  146.827346  0.0  147.314371 | 163.936334  130.433046  0.0  0.0  0.0 | 1.933466 | 46.937809 | 0.065271  0.065193  0.065055   0.06488  0.064697 | 0.925344 |
    | 04/01 16:00 |  0.0 | 5.0 | 21.2 |  6.298329 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 516.900183  519.482113  519.482113  0.0  519.482113 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 227.070863  229.463698  230.064988  0.0   229.76511 | 90.828345   91.00773  90.406441  0.0  90.706318 | 228.705322  228.705322  228.705322  0.0  228.705322 | 90.776791  90.776791  90.776791  0.0  90.776791 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.58193        0.0  0.0      0.0  0.0 | 1.734742       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.017608  0.034783  0.0  0.0  0.0 |  0.703878 |      0.5 | 0.008415 | 0.0  0.0  0.0       0.0  0.0 | 0.236832 | 0.227775 | 0.224198 | 0.224198 | 0.934158 |      2.0       2.0  0.0  0.0       2.0 | 369.214416  371.949163  372.924416  0.0  372.437269 | 147.685766   147.53295  146.557696  0.0  147.044844 |  164.80113  130.467829  0.0  0.0  0.0 | 2.128928 | 47.129548 | 0.066028  0.065938  0.065776  0.065572  0.065359 | 0.934158 |
    | 04/01 17:00 |  0.0 | 5.0 | 19.1 |  2.948416 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 515.899733  518.489063  518.489063  0.0  518.489063 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 226.357273  228.749175  229.348956  0.0   229.04936 | 90.542909  90.732938  90.133156  0.0  90.432753 | 227.990715  227.990715  227.990715  0.0  227.990715 | 90.498348  90.498348  90.498348  0.0  90.498348 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 |  2.58933        0.0  0.0      0.0  0.0 | 1.758117       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.017184  0.034766  0.0  0.0  0.0 |    0.7138 |      0.5 | 0.011137 | 0.0  0.0  0.0       0.0  0.0 | 0.237791 | 0.230847 | 0.226665 | 0.226665 | 0.944436 |      2.0       2.0  0.0  0.0       2.0 | 368.499809  371.230703  372.206175  0.0  371.718624 | 147.399924  147.258359  146.282888  0.0  146.770439 | 165.649527  130.502595  0.0  0.0  0.0 | 2.331592 | 47.320329 | 0.066915  0.066809   0.06662  0.066381  0.066132 | 0.944436 |
    | 04/01 18:00 |  0.0 | 5.0 | 16.8 |  1.309232 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 514.898195  517.493167  517.493167  0.0  517.493167 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 225.642667  228.033583   228.63278  0.0  228.333295 | 90.257067   90.45548  89.856283  0.0  90.155768 | 227.275331  227.275331  227.275331  0.0  227.275331 | 90.217836  90.217836  90.217836  0.0  90.217836 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.594971        0.0  0.0      0.0  0.0 | 1.780135       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.016768  0.034749  0.0  0.0  0.0 |  0.723154 |      0.5 | 0.014507 | 0.0  0.0  0.0       0.0  0.0 | 0.238745 | 0.234434 | 0.229548 | 0.229548 | 0.956449 |      2.0       2.0  0.0  0.0       2.0 | 367.784425  370.512451  371.488726  0.0  371.000659 |  147.11377  146.980716  146.004441  0.0  146.492507 | 166.481131  130.537344  0.0  0.0  0.0 | 2.540238 | 47.510156 |  0.06795  0.067826  0.067605  0.067326  0.067035 | 0.956449 |
    | 04/01 19:00 |  0.0 | 5.0 | 14.2 |   0.32955 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 513.895825  516.495101  516.495101  0.0  516.495101 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 224.927282  227.317342  227.916308  0.0  227.616869 | 89.970913  90.175824  89.576859  0.0  89.876298 | 226.559352  226.559352  226.559352  0.0  226.559352 | 89.935749  89.935749  89.935749  0.0  89.935749 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.599276        0.0  0.0      0.0  0.0 | 1.801036       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.01636  0.034731  0.0  0.0  0.0 |  0.732038 |      0.5 |  0.01861 | 0.0  0.0  0.0       0.0  0.0 | 0.239694 | 0.238602 | 0.232915 | 0.232915 | 0.970478 |      2.0       2.0  0.0  0.0       2.0 | 367.068446   369.79446   370.77177  0.0  370.283143 | 146.827379   146.70064  145.723331  0.0  146.211958 | 167.295731  130.572075  0.0  0.0  0.0 | 2.753666 | 47.699034 | 0.069153  0.069009  0.068752  0.068427  0.068089 | 0.970478 |
    | 04/01 20:00 |  0.0 | 5.0 | 13.5 |  0.089508 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.892817  515.495379  515.495379  0.0  515.495379 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 224.211303  226.600668  227.199539  0.0   226.90012 | 89.684521  89.894432  89.295562  0.0   89.59498 | 225.842917  225.842917  225.842917  0.0  225.842917 | 89.652462  89.652462  89.652462  0.0  89.652462 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.602563        0.0  0.0      0.0  0.0 | 1.821004       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.015961  0.034714  0.0  0.0  0.0 |   0.74053 |      0.5 | 0.023524 | 0.0  0.0  0.0       0.0  0.0 | 0.240638 | 0.243415 | 0.236833 | 0.236833 | 0.986803 |      2.0       2.0  0.0  0.0       2.0 | 366.352012   369.07671  370.055148  0.0   369.56594 | 146.540805   146.41867  145.440231  0.0   145.92944 | 168.093251  130.606789  0.0  0.0  0.0 | 2.970672 | 47.886967 | 0.070542  0.070375  0.070079  0.069704  0.069312 | 0.986803 |
    | 04/01 21:00 |  0.0 | 5.0 | 11.4 |  0.085771 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 511.889319  514.494394  514.494394  0.0  514.494394 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 223.494869  225.883686  226.482514  0.0  226.183107 | 89.397948  89.611693  89.012865  0.0  89.312273 | 225.126134  225.126134  225.126134  0.0  225.126134 |  89.36826   89.36826   89.36826  0.0   89.36826 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.605074        0.0  0.0      0.0  0.0 | 1.840182       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.015571  0.034697  0.0  0.0  0.0 |  0.748689 |      0.5 |  0.02932 | 0.0  0.0  0.0       0.0  0.0 | 0.241578 | 0.248928 | 0.241366 | 0.241366 | 1.005691 |      2.0       2.0  0.0  0.0       2.0 | 365.635228  368.359157  369.338768  0.0  368.848967 | 146.254091  146.135237  145.155625  0.0  145.645427 | 168.873714  130.641486  0.0  0.0  0.0 | 3.190041 | 48.073961 | 0.072134  0.071943  0.071603  0.071172  0.070722 | 1.005691 |
    | 04/01 22:00 |  0.0 | 5.0 | 10.7 |    0.0845 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.885446  513.492442  513.492442  0.0  513.492442 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 222.778085  225.166476  225.765282  0.0  225.465882 | 89.111234  89.327917  88.729111  0.0  89.028512 | 224.409082  224.409082  224.409082  0.0  224.409082 |  89.08336   89.08336   89.08336  0.0   89.08336 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.606996        0.0  0.0      0.0  0.0 | 1.858679       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.015189  0.034679  0.0  0.0  0.0 |  0.756561 |      0.5 | 0.036054 | 0.0  0.0  0.0       0.0  0.0 | 0.242513 | 0.255189 | 0.246573 | 0.246573 | 1.027389 |      2.0       2.0  0.0  0.0       2.0 | 364.918176  367.641762  368.622568  0.0  368.132167 |  145.96727   145.85068  144.869874  0.0  145.360275 |  169.63722  130.676165  0.0  0.0  0.0 | 3.410548 |  48.26002 | 0.073942  0.073725  0.073338  0.072848  0.072336 | 1.027389 |
    | 04/01 23:00 |  0.0 | 5.0 | 10.9 |  0.084864 |   0.2 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 509.881284  512.489751  512.489751  0.0  512.489751 | 0.0  0.0  0.903096  0.0   0.0 |        1.0 |     0.0       0.0       0.0      0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0       0.0       0.0       0.0 |      0.0       0.0  0.0  0.0       0.0 |      0.0       0.0       0.0      0.0       0.0 | 222.061033  224.449095  225.047887  0.0  224.748492 | 88.824413  89.043347  88.444555  0.0   88.74395 | 223.691823  223.691823  223.691823  0.0  223.691823 | 88.797929  88.797929  88.797929  0.0  88.797929 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0  0.34 | 2.608467        0.0  0.0      0.0  0.0 | 1.876583       0.0  0.0      0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.014815  0.034662  0.0  0.0  0.0 |  0.764182 |      0.5 | 0.043768 | 0.0  0.0  0.0       0.0  0.0 | 0.243443 | 0.262228 | 0.252505 | 0.252505 | 1.052104 |      2.0       2.0  0.0  0.0       2.0 | 364.200917   366.92449  367.906504  0.0  367.415498 | 145.680367  145.565261  144.583248  0.0  145.074254 |  170.38392  130.710827  0.0  0.0  0.0 | 3.630962 | 48.445148 | 0.075975  0.075731  0.075296  0.074744  0.074165 | 1.052104 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0
"""
# import...
# from standard-library
from typing import *

# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import masktools
from hydpy.core import modeltools
from hydpy.core.typingtools import *

# ...from hland
from hydpy.models.hland import hland_model
from hydpy.models.hland import hland_masks
from hydpy.models.hland.hland_constants import *


class Model(modeltools.AdHocModel):
    """HBV96-SC version of HydPy-H-Land (|hland_v2|)."""

    INLET_METHODS = ()
    RECEIVER_METHODS = ()
    RUN_METHODS = (
        hland_model.Calc_TC_V1,
        hland_model.Calc_TMean_V1,
        hland_model.Calc_FracRain_V1,
        hland_model.Calc_RFC_SFC_V1,
        hland_model.Calc_PC_V1,
        hland_model.Calc_EP_V1,
        hland_model.Calc_EPC_V1,
        hland_model.Calc_TF_Ic_V1,
        hland_model.Calc_EI_Ic_V1,
        hland_model.Calc_SP_WC_V1,
        hland_model.Calc_SPL_WCL_SP_WC_V1,
        hland_model.Calc_SPG_WCG_SP_WC_V1,
        hland_model.Calc_CFAct_V1,
        hland_model.Calc_Melt_SP_WC_V1,
        hland_model.Calc_Refr_SP_WC_V1,
        hland_model.Calc_In_WC_V1,
        hland_model.Calc_SWE_V1,
        hland_model.Calc_SR_V1,
        hland_model.Calc_GAct_V1,
        hland_model.Calc_GlMelt_In_V1,
        hland_model.Calc_R_SM_V1,
        hland_model.Calc_CF_SM_V1,
        hland_model.Calc_EA_SM_V1,
        hland_model.Calc_InUZ_V1,
        hland_model.Calc_ContriArea_V1,
        hland_model.Calc_Q0_Perc_UZ_V1,
        hland_model.Calc_LZ_V1,
        hland_model.Calc_EL_LZ_V1,
        hland_model.Calc_Q1_LZ_V1,
        hland_model.Calc_InUH_V1,
        hland_model.Calc_OutUH_SC_V1,
        hland_model.Calc_RT_V1,
        hland_model.Calc_QT_V1,
    )
    ADD_METHODS = ()
    OUTLET_METHODS = (hland_model.Pass_Q_v1,)
    SENDER_METHODS = ()
    SUBMODELS = ()

    def check_waterbalance(
        self,
        initial_conditions: Dict[str, Dict[str, ArrayFloat]],
    ) -> float:
        r"""Determine the water balance error of the previous simulation run in mm.

        Method |Model.check_waterbalance| calculates the balance error as follows:

          .. math::
            \sum_{k=1}^{NmbZones} RelZoneAreas^k \cdot \left(
            \sum_{t=t0}^{t1} \big( PC_t^k + GLMelt_t^k - EI_t^k - EA_t^k - EL_t^k \big)
            + \big( IC_{t0}^k - IC_{t1}^k \big)
            + \frac{1}{SClass} \cdot \sum_{c=1}^{SClass}
            \Big( \big( SP_{t0}^{c,k} - SP_{t1}^{c,k} \big)
            + \big( WC_{t0}^{c,k} - WC_{t1}^{c,k} \big) \Big)
            + \big( SM_{t0}^k - SM_{t1}^k \big) \right)
            + RelUpperZoneArea \cdot \big( UZ_{t0}^k - UZ_{t1}^k \big)
            + RelLowerZoneArea \cdot \big( LZ_{t0} - LZ_{t1} \big)
            - \sum_{t=t0}^{t1} RT_t
            + \sum_{i=1}^{NmbStorages} \big( SC_{t0}^i - SC_{t1}^i \big)

        The returned error should always be in scale with numerical precision so that
        it does not affect the simulation results in any relevant manner.

        Pick the required initial conditions before starting the simulation run via
        property |Sequences.conditions|.  See the integration tests of the application
        model |hland_v2| for some examples.
        """
        derived = self.parameters.derived
        fluxes = self.sequences.fluxes
        last = self.sequences.states
        first = initial_conditions["states"]
        areas = derived.relzoneareas.value
        idxs_lake = self.parameters.control.zonetype.values == ILAKE
        idxs_land = ~idxs_lake
        idxs_glac = self.parameters.control.zonetype.values == GLACIER
        idxs_seal = self.parameters.control.zonetype.values == SEALED
        idxs_soil = idxs_land * ~idxs_glac * ~idxs_seal
        return (
            numpy.sum(fluxes.pc.series * areas)
            + numpy.sum((fluxes.glmelt.series * areas)[:, idxs_glac])
            - numpy.sum((fluxes.ei.series * areas)[:, idxs_land])
            - numpy.sum((fluxes.ea.series * areas)[:, idxs_soil])
            - numpy.sum((fluxes.el.series * areas)[:, idxs_lake])
            - numpy.sum(fluxes.rt.series)
            - numpy.sum(((last.ic - first["ic"]) * areas)[idxs_land])
            - numpy.sum((numpy.mean(last.wc - first["wc"], axis=0) * areas)[idxs_land])
            - numpy.sum((numpy.mean(last.sp - first["sp"], axis=0) * areas)[idxs_land])
            - numpy.sum(((last.sm - first["sm"]) * areas)[idxs_soil])
            - (last.uz - first["uz"]) * derived.relupperzonearea
            - (last.lz - first["lz"]) * derived.rellowerzonearea
            - numpy.sum(last.sc - first["sc"])
        )


class Masks(masktools.Masks):
    """Masks applicable to |hland_v2|."""

    CLASSES = hland_masks.Masks.CLASSES


tester = Tester()
cythonizer = Cythonizer()
