# -*- coding: utf-8 -*-
# pylint: disable=line-too-long, unused-wildcard-import
"""
.. _`German Federal Institute of Hydrology (BfG)`: https://www.bafg.de/EN
.. _`LARSIM`: http://www.larsim.de/en/the-model/

One can understand version 3 of the L-Land model as an extended, more complicated
version of the application model |lland_v1|, using precipitation, temperature, air
pressure, wind speed, relative humidity, and actual and possible sunshine duration
as meteorological input parameters.  It relies on the equations after Penman and
Penman-Monteith for calculating evaporation and evapotranspiration, which require much
more input data than the simple Turc-Wendling equation but bring flexibility regarding
the simulation time step (Turc-Wendling is limited to daily time steps, Penman and
Penman-Monteith work for any time step size).  The second main difference is that
|lland_v3| models the energy balance of snow processes in more detail (following Knauf,
2006), taking not only the air temperature but also forcings like the soil heat flux or
global radiation into account explicitly.  The third difference is that |lland_v3|
models the soil temperature to simulate the effect of soil water freezing on runoff
generation.  We created |lland_v3| on behalf of the `German Federal Institute of
Hydrology (BfG)`_ in the context of optimising the control of the Kiel Canal
(Nord-Ostsee-Kanal).

The following list summarises the main components of |lland_v3|:

 * Multiple routines for adjusting the meteorological input data
 * Evaporation from water surfaces after Penman :cite:p:`ref-penman1948natural`
 * Evapotranspiration (interception evaporation) after Penman-Monteith
   :cite:p:`ref-Thompson1981`
 * Actual soil evapotranspiration after Wigmosta
   :cite:p:`ref-wigmosta1994distributed`
 * Surface roughnesses after Quast and Böhm :cite:p:`ref-bohm1998ermittlung`
 * Mixed precipitation within a definable temperature-range
 * Energy balance of the snow-layer after Knauf :cite:p:`ref-LUBW2006a`
 * Dynamical adjustment of the snow albedo
 * Optional refinement of the snow-surface temperature
 * Optional (re)freezing of liquid water within the snow-layer
 * A simple snow retention routine
 * A soil routine based on the Xinanjiang model :cite:p:`ref-zhao1977flood`
 * Frost-sealing of the upper soil-layer
 * One base flow, two interflow and two direct flow components
 * A freely configurable capillary rise routine
 * Options to limit the capacity of the base flow storage
 * Separate linear storages for modelling runoff concentration
 * Additional evaporation from water areas within the subcatchment
 * Optional evaporation from inflowing runoff

Some notes for `LARSIM`_ users: |lland_v3| is similar to many `LARSIM`_ models used for
forecasting (relying on a combination of options equivalent to the features listed
above) but not identical.  Often, we strive for more flexibility. One example is
modifying parameter |KTSchnee| to control if and how |lland_v3| adjusts the snow
surface temperature.  However, there are also differences due to technical reasons.
One relevant difference is that *HydPy* does not include "future values" in the current
simulation timestep. For example, `LARSIM`_ fills nightly gaps regarding cloudiness
with the average value of the present calendar day, while |lland_v3| uses the average
over the 24 hours preceding the current simulation step.  Hence, do not expect the
results of |lland_v3| and `LARSIM`_ to be identical.  When switching from one model
system to the other, the need to fine-tune some parameters via calibration might arise.

Among others, |lland_v3| requires the time series of global radiation and
astronomically possible sunshine duration as input.  You can prepare them during data
preprocessing and provide them via files.  Alternatively, you can connect |lland_v3| to
suitable members of the |meteo| model family.  An obvious choice is |meteo_v003|, which
estimates these meteorological properties according to `LARSIM`_.

.. note::

    Some details of |lland_v3| are still under discussion and might change in the
    future.

Integration tests
=================

.. how_to_understand_integration_tests::

Many of the following integration tests are similar to the integration tests performed
for the application model |lland_v1|.  When there are no substantial differences, we do
not repeat our explanations, so please note the respective sections of the
documentation on |lland_v1|.  Some additional tests cover structural differences
between land-use types that only exist for |lland_v3|, and others show the different
possible snow-related configurations of |lland_v3|.  Furthermore, we perform some daily
and hourly tests to point out some differences and make sure |lland_v3| works well for
different simulation step sizes.

Note that, while our daily simulations always cover an entire month, our hourly
simulations cover only three days, focussing on the "most interesting" part of the
daily simulation.  To ensure the best possible comparability, we use the same input
data (we aggregate the hourly values to daily values), the same parameter values
(possibly adapted to the change in the time step size), and the same initial conditions
(we extract the required initial conditions of the respective hourly simulation from
the longer daily simulation).

daily simulation
________________

The following general setup is identical to the one of |lland_v1|, except that we now
start with a different period and a daily simulation time step:

>>> from hydpy import pub, Timegrid
>>> pub.timegrids = "1997-08-01", "1997-09-01", "1d"
>>> from hydpy.models.lland_v3 import *
>>> parameterstep("1h")
>>> from hydpy import Node, Element
>>> outlet = Node("outlet")
>>> inlet = Node("inlet")
>>> land = Element("land", inlets=inlet, outlets=outlet)
>>> land.model = model
>>> nhru(1)
>>> ft(1.0)
>>> fhru(1.0)
>>> from hydpy import IntegrationTest
>>> IntegrationTest.plotting_options.axis1 = inputs.nied
>>> IntegrationTest.plotting_options.axis2 = fluxes.qah
>>> test = IntegrationTest(land)
>>> test.dateformat = "%Y-%d-%m"

.. _lland_v3_acker_summer_daily:

acre (summer)
-------------

First, we set the values of those parameters also required by the application model
|lland_v1| (see integration test :ref:`lland_v1_acker_summer`):

>>> lnk(ACKER)
>>> kg(0.94)
>>> kt(0.0)
>>> hinz(0.2)
>>> lai.acker_aug = 3.5
>>> tgr(0.0)
>>> trefn(0.0)
>>> tsp(4.0)
>>> pwmax(1.43)
>>> wmax(309.0)
>>> fk(199.0)
>>> pwp(119.4)
>>> kapgrenz(option="0_WMax/10")
>>> kapmax(0.08)
>>> beta(0.001/24)
>>> fbeta(1.0)
>>> rbeta(False)
>>> dmax(r_dmax=0.55)
>>> dmin(r_dmin=10.0)
>>> bsf(0.3)
>>> volbmax(inf)
>>> gsbmax(1.0)
>>> gsbgrad1(inf)
>>> gsbgrad2(inf)
>>> a1(0.0)
>>> a2(0.5)
>>> tind(0.191056)
>>> eqb(35000.0)
>>> eqi1(400.0)
>>> eqi2(400.0)
>>> eqd1(200.0)
>>> eqd2(50.0)

In contrast to the documentation on |lland_v1|, we set parameter |NegQ| to |True| in
most examples.  That means we favour negative discharge estimates over water balance
errors whenever the base flow storage cannot meet the water demand of the capillary
rise of groundwater into the soil:

>>> negq(True)

Next, we set the values of the parameters specific to |lland_v3|, beginning with those
required for calculating the evaporation via Penman and the evapotranspiration via
Penman-Monteith:

>>> measuringheightwindspeed(10.0)
>>> emissivity(0.95)
>>> py(119.4)
>>> cropheight.acker_aug = 0.4
>>> albedo.acker_aug = 0.24
>>> wg2z.aug = -2.9
>>> surfaceresistance.acker = 40.0

The following parameters are related to the energy accounting snow modelling approach
after Knauf :cite:p:`ref-LUBW2006a`:

>>> p1strahl(0.5)
>>> p2strahl(1.0/35.0)
>>> p1wind(0.6)
>>> p2wind(1.0/70.0)
>>> turb0(2.0)
>>> turb1(1.6)
>>> albedo0snow(0.9)
>>> snowagingfactor(0.35)
>>> refreezeflag(True)
>>> ktschnee(inf)

Now we set the values of the parameters controlling the frost sealing of the upper soil
layer:

>>> fvf(0.3)
>>> bsff(2.0)

Due to preparing a simulation with a daily simulation step size, we only need to define
the initial values of the state sequences (and not the values of the log sequences):

>>> test.inits = (
...     (states.inzp, 0.0),
...     (states.wats, 0.0),
...     (states.waes, 0.0),
...     (states.esnow, 0.0),
...     (states.taus, 0.0),
...     (states.ebdn, 350.0),
...     (states.bowa, 72.0),
...     (states.qdgz1, 0.0),
...     (states.qdgz2, 0.0),
...     (states.qigz1, 0.0),
...     (states.qigz2, 0.0),
...     (states.qbgz, 0.0),
...     (states.qdga1, 0.0),
...     (states.qdga2, 0.0),
...     (states.qiga1, 0.0),
...     (states.qiga2, 0.0),
...     (states.qbga, 0.0))

We prepare those input time series usually available as measurements:

>>> inputs.nied.series = (
...     13.3, 0.1, 0.0, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.0, 0.0, 2.7,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 2.0, 0.3, 0.0, 0.1, 10.5, 6.5, 15.0, 9.2, 0.0,
...     0.0)
>>> inputs.teml.series = (
...     16.1, 17.3, 18.9, 17.8, 18.5, 19.9, 20.8, 20.9, 22.3, 23.0, 23.1, 23.2, 23.2,
...     21.7, 20.8, 19.5, 21.0, 21.6, 21.1, 22.4, 22.5, 19.6, 20.3, 24.1, 23.7, 21.1,
...     20.9, 20.2, 16.1, 17.0, 18.2)
>>> inputs.sunshineduration.series = (
...     6.3, 1.7, 4.5, 12.4, 13.9, 13.0, 13.8, 12.3, 13.1, 12.8, 13.1, 13.3, 12.7,
...     10.2, 9.4, 10.3, 11.1, 11.0, 8.5, 11.3, 12.4, 0.1, 6.7, 10.4, 6.5, 4.9, 6.6,
...     0.3, 0.1, 5.0, 3.8)
>>> inputs.windspeed.series = (
...     2.4, 1.6, 1.5, 2.8, 3.0, 3.5, 3.8, 3.3, 1.8, 1.9, 3.0, 2.5, 1.6, 2.7, 3.2, 1.3,
...     1.9, 2.0, 2.6, 3.6, 2.5, 2.1, 1.5, 2.2, 1.9, 2.3, 2.1, 2.6, 2.6, 2.1, 2.1)
>>> inputs.relativehumidity.series = (
...     86.2, 85.8, 82.4, 77.6, 74.9, 77.9, 73.5, 80.1, 72.9, 68.5, 68.6, 66.0, 69.0,
...     75.6, 81.4, 79.0, 75.9, 70.0, 70.3, 69.0, 62.1, 84.5, 83.6, 76.5, 76.6, 84.2,
...     85.8, 86.5, 89.6, 78.3, 78.5)
>>> inputs.atmosphericpressure.series = (
...     1007.0, 1013.0, 1016.0, 1017.0, 1018.0, 1018.0, 1019.0, 1019.0, 1019.0, 1020.0,
...     1021.0, 1022.0, 1019.0, 1015.0, 1016.0, 1019.0, 1019.0, 1018.0, 1019.0, 1019.0,
...     1017.0, 1015.0, 1015.0, 1011.0, 1008.0, 1006.0, 1005.0, 1000.0, 1004.0, 1011.0,
...     1014.0)
>>> inlet.sequences.sim.series = 0.0

The following two input time series stem from the results of |meteo_v003| calculated in
the :ref:`meteo_v003_daily_simulation_summer` example, which addresses the same place
and time:

>>> inputs.possiblesunshineduration.series = (
...     15.69078473, 15.62882066, 15.56613383, 15.50275437, 15.43871163, 15.37403418,
...     15.3087498, 15.24288548, 15.17646742, 15.10952106, 15.04207104, 14.97414122,
...     14.90575471, 14.83693386, 14.76770029, 14.69807488, 14.62807779, 14.5577285,
...     14.48704578, 14.41604776, 14.34475191, 14.27317506, 14.20133345, 14.12924271,
...     14.05691791, 13.98437357, 13.91162366, 13.83868165, 13.76556053, 13.69227282,
...     13.61883057)
>>> inputs.globalradiation.series = (
...     190.2514903, 127.6607822, 164.4012978, 269.0205016, 287.8486672, 274.6742631,
...     284.0478917, 262.9712633, 272.2564135, 267.0192178, 269.6265466, 270.8782909,
...     261.6971638, 227.9621155, 216.3563213, 226.6379036, 235.5285025, 232.8494284,
...     199.7134896, 233.7719021, 246.1212861, 90.58551, 171.9449932, 216.5498126,
...     166.9168296, 145.9856109, 165.5438903, 87.6933456, 84.356885, 142.3092025,
...     126.7220785)

The following results illustrate the behaviour of |lland_v3| for relatively dry and hot
summer condions. Compared to the result table of integration test
:ref:`lland_v1_acker_summer` of the application model |lland_v1|, there are much more
columns because of to the higher number of input, flux, and state sequences.  These are
mainly due to the increased data requirements and the more complex calculations of the
Penman-Monteith equation and the Knauf :cite:p:`ref-LUBW2006a` approach.  Additionally,
|lland_v3| calculates many daily values (containing the averages or sums over the last
24 hours) required for the Penman equation, which we generally apply based on the input
data of the previous 24-hours. (Note that we could save some computation time by not
calculating daily values that are never required.  First, land-use type |ACKER| only
relies on the Penman-Monteith equation.  Second, the values of the "normal" sequences
are already daily values, due to the selected simulation step size of one day.  Maybe
we perform some related optimisations later.):

.. integration-test::

    >>> parameters.update()
    >>> test.reset_inits()
    >>> conditions = sequences.conditions
    >>> conditions_acker_summer = test(
    ...     "lland_v3_acker_summer_daily",
    ...     axis1=(inputs.nied, fluxes.qah), axis2=states.bowa,
    ...     get_conditions="1997-08-03")
    |       date | nied | teml | sunshineduration | possiblesunshineduration | globalradiation | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration | dailypossiblesunshineduration | dailyglobalradiation |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair |    g |        tz |         wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance |   nbes | sbes |     evpo |   evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg |   wada |      qdb | qib1 | qib2 | qbb |     qkap |     qdgz |       qah |        qa | inzp | wats | waes | esnow | taus |       ebdn |      bowa |    qdgz1 | qdgz2 | qigz1 | qigz2 |      qbgz |    qdga1 | qdga2 | qiga1 | qiga2 |      qbga | inlet |    outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |                15.690785 |       190.25149 |             86.2 |       2.4 |              1007.0 | 0.0 | 0.0 |    16.1 |                  86.2 |                   6.3 |                     15.690785 |            190.25149 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            15.797694 |                 15.797694 |     991.202306 |   1.205601 | -2.9 | 11.733333 |      -26.2 |            144.591133 |                144.591133 |                 144.591133 |                 21.732733 |                      0.0 |     122.8584 |              0.0 |          122.8584 |             58.847345 |                 100.0 |               111.309423 |               99.889466 | 11.802 |  0.0 | 3.623832 |   0.7 | 1.720128 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 11.802 | 0.761343 |  0.0 |  0.0 | 0.0 |      0.0 | 0.761343 |  0.196005 |  0.002269 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |      373.3 | 81.320529 | 0.761343 |   0.0 |   0.0 |   0.0 |       0.0 | 0.196005 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.002269 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |                15.628821 |      127.660782 |             85.8 |       1.6 |              1013.0 | 0.0 | 0.0 |    17.3 |                  85.8 |                   1.7 |                     15.628821 |           127.660782 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                19.779139 |                     19.779139 |                          0.0 |                      1.251029 |                           1.251029 |            16.970501 |                 16.970501 |     996.029499 |   1.207284 | -2.9 | 18.443733 |     6.8624 |             97.022194 |                 97.022194 |                  97.022194 |                 11.219797 |                      0.0 |    85.802398 |              0.0 |         85.802398 |             88.271017 |                 100.0 |                 98.30202 |               94.620613 |    0.0 |  0.0 | 2.555581 | 0.094 | 1.697173 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.263667 |  0.003052 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   363.5376 | 79.623355 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.263667 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.003052 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |                15.566134 |      164.401298 |             82.4 |       1.5 |              1016.0 | 0.0 | 0.0 |    18.9 |                  82.4 |                   4.5 |                     15.566134 |           164.401298 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                21.871576 |                     21.871576 |                          0.0 |                      1.365939 |                           1.365939 |            18.022179 |                 18.022179 |     997.977821 |   1.203775 | -2.9 | 15.632162 | -19.607027 |            124.944986 |                124.944986 |                 124.944986 |                 16.044181 |                      0.0 |   108.900806 |              0.0 |        108.900806 |             94.155751 |                 100.0 |               100.628883 |               95.574208 |    0.0 |  0.0 | 3.286385 |   0.0 | 2.329453 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.140696 |  0.001628 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.244627 | 77.293902 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.140696 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.001628 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |                15.502754 |      269.020502 |             77.6 |       2.8 |              1017.0 | 0.0 | 0.0 |    17.8 |                  77.6 |                  12.4 |                     15.502754 |           269.020502 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                20.413369 |                     20.413369 |                          0.0 |                      1.286025 |                           1.286025 |            15.840774 |                 15.840774 |    1001.159226 |    1.21051 | -2.9 | 20.443786 |  15.862716 |            204.455581 |                204.455581 |                 204.455581 |                 35.980407 |                      0.0 |   168.475175 |              0.0 |        168.475175 |             50.440581 |                 100.0 |                103.85044 |               96.853188 |    0.0 |  0.0 | 5.847296 | 0.094 | 3.314337 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.075077 |  0.000869 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 361.481911 | 73.979565 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.075077 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000869 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |                15.438712 |      287.848667 |             74.9 |       3.0 |              1018.0 | 0.0 | 0.0 |    18.5 |                  74.9 |                  13.9 |                     15.438712 |           287.848667 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                21.331125 |                     21.331125 |                          0.0 |                      1.336407 |                           1.336407 |            15.977013 |                 15.977013 |    1002.022987 |   1.208738 | -2.9 | 15.040124 | -20.759257 |            218.764987 |                218.764987 |                 218.764987 |                 39.523937 |                      0.0 |   179.241051 |              0.0 |        179.241051 |             47.077876 |                 100.0 |               108.495906 |                98.61702 |    0.0 |  0.0 | 6.638473 |   0.0 |  3.73228 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.040062 |  0.000464 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.341169 | 70.247285 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.040062 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000464 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |                15.374034 |      274.674263 |             77.9 |       3.5 |              1018.0 | 0.0 | 0.0 |    19.9 |                  77.9 |                  13.0 |                     15.374034 |           274.674263 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                23.275348 |                     23.275348 |                          0.0 |                      1.442189 |                           1.442189 |            18.131496 |                 18.131496 |     999.868504 |   1.201995 | -2.9 |  20.18359 |   1.701539 |             208.75244 |                 208.75244 |                  208.75244 |                 32.936813 |                      0.0 |   175.815627 |              0.0 |        175.815627 |             40.352465 |                 100.0 |               113.827642 |              100.519454 |    0.0 |  0.0 | 6.828968 |   0.0 | 3.652387 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.021377 |  0.000247 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.739629 | 66.594898 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.021377 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000247 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |                 15.30875 |      284.047892 |             73.5 |       3.8 |              1019.0 | 0.0 | 0.0 |    20.8 |                  73.5 |                  13.8 |                      15.30875 |           284.047892 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            18.084931 |                 18.084931 |    1000.915069 |   1.199521 | -2.9 | 18.858347 | -11.649921 |            215.876398 |                215.876398 |                 215.876398 |                  35.31351 |                      0.0 |   180.562888 |              0.0 |        180.562888 |             37.166744 |                 100.0 |               119.167697 |              102.297207 |    0.0 |  0.0 | 7.945133 |   0.0 | 4.128687 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.011407 |  0.000132 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  383.48955 |  62.46621 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.011407 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000132 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |                15.242885 |      262.971263 |             80.1 |       3.3 |              1019.0 | 0.0 | 0.0 |    20.9 |                  80.1 |                  12.3 |                     15.242885 |           262.971263 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            19.830469 |                 19.830469 |     999.169531 |   1.198331 | -2.9 | 21.378324 |   2.869942 |             199.85816 |                 199.85816 |                  199.85816 |                 28.392598 |                      0.0 |   171.465562 |              0.0 |        171.465562 |             42.798069 |                 100.0 |               125.381825 |              104.230888 |    0.0 |  0.0 | 6.427975 |   0.0 | 3.518399 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.006087 |   0.00007 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.719608 | 58.947811 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.006087 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |   0.00007 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |                15.176467 |      272.256414 |             72.9 |       1.8 |              1019.0 | 0.0 | 0.0 |    22.3 |                  72.9 |                  13.1 |                     15.176467 |           272.256414 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 26.96965 |                      26.96965 |                          0.0 |                      1.639967 |                           1.639967 |            19.660875 |                 19.660875 |     999.339125 |   1.192729 | -2.9 |  19.71658 | -15.500518 |            206.914874 |                206.914874 |                 206.914874 |                 31.114889 |                      0.0 |   175.799986 |              0.0 |        175.799986 |             78.463126 |                 100.0 |               130.860307 |              105.804393 |    0.0 |  0.0 | 6.036589 |   0.0 | 4.091259 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.003248 |  0.000038 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 390.320126 | 54.856552 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.003248 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000038 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |                15.109521 |      267.019218 |             68.5 |       1.9 |              1020.0 | 0.0 | 0.0 |    23.0 |                  68.5 |                  12.8 |                     15.109521 |           267.019218 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                28.139149 |                     28.139149 |                          0.0 |                      1.701779 |                           1.701779 |            19.275317 |                 19.275317 |    1000.724683 |   1.191257 | -2.9 |  23.34553 |   2.073177 |            202.934606 |                202.934606 |                 202.934606 |                 32.010643 |                      0.0 |   170.923963 |              0.0 |        170.923963 |             74.333488 |                 100.0 |               137.497212 |               107.59145 |    0.0 |  0.0 | 6.493896 |   0.0 | 4.345753 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.001733 |   0.00002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.346949 | 50.510799 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.001733 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |   0.00002 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |                15.042071 |      269.626547 |             68.6 |       3.0 |              1021.0 | 0.0 | 0.0 |    23.1 |                  68.6 |                  13.1 |                     15.042071 |           269.626547 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                28.309776 |                     28.309776 |                          0.0 |                      1.710768 |                           1.710768 |            19.420506 |                 19.420506 |    1001.579494 |   1.191966 | -2.9 | 21.913255 |  -7.120473 |            204.916175 |                204.916175 |                 204.916175 |                 32.400014 |                      0.0 |   172.516162 |              0.0 |        172.516162 |             47.077876 |                 100.0 |               144.951913 |              109.449121 |    0.0 |  0.0 | 7.914607 |   0.0 | 4.547949 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.000925 |  0.000011 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 389.567421 | 45.962849 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000925 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000011 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |                14.974141 |      270.878291 |             66.0 |       2.5 |              1022.0 | 0.0 | 0.0 |    23.2 |                  66.0 |                  13.3 |                     14.974141 |           270.878291 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |             18.79766 |                  18.79766 |     1003.20234 |   1.193016 | -2.9 | 23.128751 |  -0.427496 |            205.867501 |                205.867501 |                 205.867501 |                 34.565783 |                      0.0 |   171.301718 |              0.0 |        171.301718 |             56.493451 |                 100.0 |               153.358051 |              111.375228 |    0.0 |  0.0 | 7.566564 | 0.094 | 4.554398 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.000494 |  0.000006 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.094917 | 41.408451 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000494 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000006 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |                14.905755 |      261.697164 |             69.0 |       1.6 |              1019.0 | 0.0 | 0.0 |    23.2 |                  69.0 |                  12.7 |                     14.905755 |           261.697164 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |              19.6521 |                   19.6521 |       999.3479 |    1.18911 | -2.9 | 22.416669 |  -4.699983 |            198.889844 |                198.889844 |                 198.889844 |                  31.34697 |                      0.0 |   167.542875 |              0.0 |        167.542875 |             88.271017 |                 100.0 |                162.65311 |               113.31833 |    0.0 |  0.0 | 5.996216 |   0.0 | 4.137046 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.000263 |  0.000003 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   388.8949 | 37.271405 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000263 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000003 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |                14.836934 |      227.962116 |             75.6 |       2.7 |              1015.0 | 0.0 | 0.0 |    21.7 |                  75.6 |                  10.2 |                     14.836934 |           227.962116 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 26.00118 |                      26.00118 |                          0.0 |                        1.5885 |                             1.5885 |            19.656892 |                 19.656892 |     995.343108 |   1.190432 | -2.9 | 22.935065 |   7.410388 |            173.251208 |                173.251208 |                 173.251208 |                 25.918437 |                      0.0 |   147.332771 |              0.0 |        147.332771 |             52.308751 |                 100.0 |                172.21097 |              115.125428 |    0.0 |  0.0 | 5.983775 |   0.0 | 3.428735 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.000141 |  0.000002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 378.584513 |  33.84267 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000141 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000002 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |                  14.7677 |      216.356321 |             81.4 |       3.2 |              1016.0 | 0.0 | 0.0 |    20.8 |                  81.4 |                   9.4 |                       14.7677 |           216.356321 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            20.028753 |                 20.028753 |     995.971247 |   1.195095 | -2.9 | 19.965673 |  -5.005962 |            164.430804 |                164.430804 |                 164.430804 |                 23.397902 |                      0.0 |   141.032902 |              0.0 |        141.032902 |             44.135508 |                 100.0 |               181.322396 |              116.674937 |  1.838 |  0.0 | 5.470835 |   0.7 | 2.505678 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  1.838 | 0.049893 |  0.0 |  0.0 | 0.0 |      0.0 | 0.049893 |   0.01292 |   0.00015 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.690475 | 33.125099 | 0.049893 |   0.0 |   0.0 |   0.0 |       0.0 |  0.01292 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |   0.00015 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |                14.698075 |      226.637904 |             79.0 |       1.3 |              1019.0 | 0.0 | 0.0 |    19.5 |                  79.0 |                  10.3 |                     14.698075 |           226.637904 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                22.704676 |                     22.704676 |                          0.0 |                      1.411269 |                           1.411269 |            17.936694 |                 17.936694 |    1001.063306 |   1.204916 | -2.9 |  20.57219 |   6.433141 |            172.244807 |                172.244807 |                 172.244807 |                 28.794151 |                      0.0 |   143.450656 |              0.0 |        143.450656 |            108.641252 |                 100.0 |               183.411459 |               116.86708 |    0.0 |  0.0 | 4.081206 |   0.0 | 2.836506 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.017319 |    0.0002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 371.357334 | 30.288593 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.017319 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |    0.0002 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |                14.628078 |      235.528503 |             75.9 |       1.9 |              1019.0 | 0.0 | 0.0 |    21.0 |                  75.9 |                  11.1 |                     14.628078 |           235.528503 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                24.909749 |                     24.909749 |                          0.0 |                      1.530181 |                           1.530181 |            18.906499 |                 18.906499 |    1000.093501 |   1.198338 | -2.9 | 17.884246 | -18.694526 |            179.001662 |                179.001662 |                 179.001662 |                 29.177702 |                      0.0 |    149.82396 |              0.0 |         149.82396 |             74.333488 |                 100.0 |               192.482064 |              118.222742 |    0.0 |  0.0 | 5.175279 |   0.0 | 3.276924 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.03799 |      0.0 |  0.009173 |  0.000106 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.151861 | 27.049659 |      0.0 |   0.0 |   0.0 |   0.0 |  -0.03799 | 0.009242 |   0.0 |   0.0 |   0.0 | -0.000068 |   0.0 |  0.000106 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |                14.557728 |      232.849428 |             70.0 |       2.0 |              1018.0 | 0.0 | 0.0 |    21.6 |                  70.0 |                  11.0 |                     14.557728 |           232.849428 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                25.842752 |                     25.842752 |                          0.0 |                      1.580055 |                           1.580055 |            18.089927 |                 18.089927 |     999.910073 |   1.195081 | -2.9 | 22.433069 |   4.998415 |            176.965566 |                176.965566 |                 176.965566 |                 31.281288 |                      0.0 |   145.684278 |              0.0 |        145.684278 |             70.616813 |                 100.0 |               204.991875 |              119.966911 |    0.0 |  0.0 | 5.737858 |   0.0 | 3.581415 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.239244 |      0.0 |  0.004367 |  0.000051 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.253445 | 23.707488 |      0.0 |   0.0 |   0.0 |   0.0 | -0.239244 | 0.004931 |   0.0 |   0.0 |   0.0 | -0.000565 |   0.0 |  0.000051 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |                14.487046 |       199.71349 |             70.3 |       2.6 |              1019.0 | 0.0 | 0.0 |    21.1 |                  70.3 |                   8.5 |                     14.487046 |            199.71349 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            17.619414 |                 17.619414 |    1001.380586 |   1.198506 | -2.9 | 20.158326 |  -5.650046 |            151.782252 |                151.782252 |                 151.782252 |                 26.711296 |                      0.0 |   125.070956 |              0.0 |        125.070956 |             54.320626 |                 100.0 |               221.727637 |              122.069797 |    0.0 |  0.0 | 5.866935 |   0.0 | 3.296835 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.446913 |      0.0 |  0.000839 |   0.00001 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 382.003492 | 20.857567 |      0.0 |   0.0 |   0.0 |   0.0 | -0.446913 | 0.002631 |   0.0 |   0.0 |   0.0 | -0.001792 |   0.0 |   0.00001 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |                14.416048 |      233.771902 |             69.0 |       3.6 |              1019.0 | 0.0 | 0.0 |    22.4 |                  69.0 |                  11.3 |                     14.416048 |           233.771902 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                27.134082 |                     27.134082 |                          0.0 |                      1.648679 |                           1.648679 |            18.722517 |                 18.722517 |    1000.277483 |   1.192743 | -2.9 | 20.950339 |  -8.697966 |            177.666646 |                177.666646 |                 177.666646 |                 31.116622 |                      0.0 |   146.550024 |              0.0 |        146.550024 |             39.231563 |                 100.0 |               241.230014 |              124.199013 |    0.0 |  0.0 |  7.82923 |   0.0 | 3.891994 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.623996 |      0.0 |   -0.0023 | -0.000027 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.801458 | 17.589568 |      0.0 |   0.0 |   0.0 |   0.0 | -0.623996 | 0.001404 |   0.0 |   0.0 |   0.0 | -0.003704 |   0.0 | -0.000027 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |                14.344752 |      246.121286 |             62.1 |       2.5 |              1017.0 | 0.0 | 0.0 |    22.5 |                  62.1 |                  12.4 |                     14.344752 |           246.121286 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                27.299387 |                     27.299387 |                          0.0 |                      1.657431 |                           1.657431 |             16.95292 |                  16.95292 |     1000.04708 |   1.190771 | -2.9 | 22.620153 |    0.72092 |            187.052177 |                187.052177 |                 187.052177 |                 38.297604 |                      0.0 |   148.754573 |              0.0 |        148.754573 |             56.493451 |                 100.0 |               274.853414 |              127.355551 |    0.0 |  0.0 | 7.322689 |   0.0 | 4.183176 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.827056 |      0.0 | -0.005541 | -0.000064 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.180539 | 14.233449 |      0.0 |   0.0 |   0.0 |   0.0 | -0.827056 | 0.000749 |   0.0 |   0.0 |   0.0 |  -0.00629 |   0.0 | -0.000064 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |                14.273175 |        90.58551 |             84.5 |       2.1 |              1015.0 | 0.0 | 0.0 |    19.6 |                  84.5 |                   0.1 |                     14.273175 |             90.58551 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                22.846186 |                     22.846186 |                          0.0 |                      1.418946 |                           1.418946 |            19.305027 |                 19.305027 |     995.694973 |   1.199129 | -2.9 | 21.577328 |  11.863971 |             68.844988 |                 68.844988 |                  68.844988 |                   7.02118 |                      0.0 |    61.823807 |              0.0 |         61.823807 |             67.254108 |                 100.0 |               337.448215 |              131.864309 |   1.18 |  0.0 | 2.567328 |   0.7 | 1.075698 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   1.18 | 0.013305 |  0.0 |  0.0 | 0.0 | 1.035592 | 0.013305 | -0.005779 | -0.000067 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 369.416568 | 15.360037 | 0.013305 |   0.0 |   0.0 |   0.0 | -1.035592 | 0.003825 |   0.0 |   0.0 |   0.0 | -0.009605 |   0.0 | -0.000067 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |                14.201333 |      171.944993 |             83.6 |       1.5 |              1015.0 | 0.0 | 0.0 |    20.3 |                  83.6 |                   6.7 |                     14.201333 |           171.944993 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                23.858503 |                     23.858503 |                          0.0 |                      1.473678 |                           1.473678 |            19.945708 |                 19.945708 |     995.054292 |   1.195981 | -2.9 | 17.325305 | -17.848171 |            130.678195 |                130.678195 |                 130.678195 |                 18.994889 |                      0.0 |   111.683306 |              0.0 |        111.683306 |             94.155751 |                 100.0 |                311.33348 |              129.741425 |    0.0 |  0.0 | 3.374147 | 0.282 | 2.014865 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.96559 |      0.0 | -0.008334 | -0.000096 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.364739 | 14.310761 |      0.0 |   0.0 |   0.0 |   0.0 |  -0.96559 | 0.004821 |   0.0 |   0.0 |   0.0 | -0.013155 |   0.0 | -0.000096 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |                14.129243 |      216.549813 |             76.5 |       2.2 |              1011.0 | 0.0 | 0.0 |    24.1 |                  76.5 |                  10.4 |                     14.129243 |           216.549813 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                30.066256 |                     30.066256 |                          0.0 |                       1.80287 |                            1.80287 |            23.000686 |                 23.000686 |     987.999314 |   1.174651 | -2.9 | 21.630378 | -14.817732 |            164.577858 |                164.577858 |                 164.577858 |                 21.629737 |                      0.0 |   142.948121 |              0.0 |        142.948121 |             64.197103 |                 100.0 |                335.42542 |              131.069494 |    0.0 |  0.0 | 5.573024 |   0.0 | 3.362119 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.030788 |      0.0 | -0.014111 | -0.000163 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  396.28247 |  11.97943 |      0.0 |   0.0 |   0.0 |   0.0 | -1.030788 | 0.002573 |   0.0 |   0.0 |   0.0 | -0.016684 |   0.0 | -0.000163 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |                14.056918 |       166.91683 |             76.6 |       1.9 |              1008.0 | 0.0 | 0.0 |    23.7 |                  76.6 |                   6.5 |                     14.056918 |            166.91683 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                29.352596 |                     29.352596 |                          0.0 |                      1.765541 |                           1.765541 |            22.484088 |                 22.484088 |     985.515912 |   1.172942 | -2.9 | 25.062685 |   8.176109 |             126.85679 |                 126.85679 |                  126.85679 |                 16.222254 |                      0.0 |   110.634536 |              0.0 |        110.634536 |             74.333488 |                 100.0 |               419.576726 |               135.15103 |    0.0 |  0.0 | 4.467668 | 0.094 | 2.719352 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.175647 |      0.0 | -0.019204 | -0.000222 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.206362 | 10.435725 |      0.0 |   0.0 |   0.0 |   0.0 | -1.175647 | 0.001373 |   0.0 |   0.0 |   0.0 | -0.020577 |   0.0 | -0.000222 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |                13.984374 |      145.985611 |             84.2 |       2.3 |              1006.0 | 0.0 | 0.0 |    21.1 |                  84.2 |                   4.9 |                     13.984374 |           145.985611 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            21.103195 |                 21.103195 |     984.896805 |   1.181557 | -2.9 | 21.872765 |   4.636593 |            110.949064 |                110.949064 |                 110.949064 |                 14.602883 |                      0.0 |    96.346181 |              0.0 |         96.346181 |             61.405925 |                 100.0 |               522.087742 |              138.401561 |   9.17 |  0.0 | 3.569826 |   0.7 | 1.597373 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   9.17 | 0.104578 |  0.0 |  0.0 | 0.0 | 1.271567 | 0.104578 |  0.002769 |  0.000032 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.669769 | 19.175341 | 0.104578 |   0.0 |   0.0 |   0.0 | -1.271567 | 0.027656 |   0.0 |   0.0 |   0.0 | -0.024887 |   0.0 |  0.000032 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |                13.911624 |       165.54389 |             85.8 |       2.1 |              1005.0 | 0.0 | 0.0 |    20.9 |                  85.8 |                   6.6 |                     13.911624 |            165.54389 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            21.241626 |                 21.241626 |     983.758374 |   1.181114 | -2.9 | 19.702227 |   -7.18664 |            125.813357 |                125.813357 |                 125.813357 |                 17.349121 |                      0.0 |   108.464235 |              0.0 |        108.464235 |             67.254108 |                 100.0 |               256.482293 |              123.963828 |   5.41 |  0.0 |  3.56693 |   0.7 | 1.722258 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   5.41 | 0.090745 |  0.0 |  0.0 | 0.0 | 0.728523 | 0.090745 |   0.03159 |  0.000366 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 381.956408 | 23.500861 | 0.090745 |   0.0 |   0.0 |   0.0 | -0.728523 |  0.05997 |   0.0 |   0.0 |   0.0 |  -0.02838 |   0.0 |  0.000366 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |                13.838682 |       87.693346 |             86.5 |       2.6 |              1000.0 | 0.0 | 0.0 |    20.2 |                  86.5 |                   0.3 |                     13.838682 |            87.693346 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                23.711531 |                     23.711531 |                          0.0 |                      1.465752 |                           1.465752 |            20.510475 |                 20.510475 |     979.489525 |   1.178323 | -2.9 | 20.936779 |   4.420674 |             66.646943 |                 66.646943 |                  66.646943 |                  6.818024 |                      0.0 |    59.828919 |              0.0 |         59.828919 |             54.320626 |                 100.0 |                222.94421 |               120.11308 |   13.4 |  0.0 | 2.621228 |   0.7 | 1.071953 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   13.4 | 0.312945 |  0.0 |  0.0 | 0.0 | 0.459752 | 0.312945 |  0.101121 |   0.00117 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.635735 | 35.975715 | 0.312945 |   0.0 |   0.0 |   0.0 | -0.459752 | 0.131528 |   0.0 |   0.0 |   0.0 | -0.030407 |   0.0 |   0.00117 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |                13.765561 |       84.356885 |             89.6 |       2.6 |              1004.0 | 0.0 | 0.0 |    16.1 |                  89.6 |                   0.1 |                     13.765561 |            84.356885 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            16.420805 |                 16.420805 |     987.579195 |   1.201704 | -2.9 | 18.828425 |   16.37055 |             64.111233 |                 64.111233 |                  64.111233 |                  8.199212 |                      0.0 |    55.912021 |              0.0 |         55.912021 |             54.320626 |                 100.0 |               175.502292 |              113.152806 |  7.948 |  0.0 | 2.027587 |   0.7 | 0.717164 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  7.948 | 0.249164 |  0.0 |  0.0 | 0.0 |      0.0 | 0.249164 |  0.168598 |  0.001951 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 355.365185 | 42.957387 | 0.249164 |   0.0 |   0.0 |   0.0 |       0.0 | 0.199719 |   0.0 |   0.0 |   0.0 | -0.031121 |   0.0 |  0.001951 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |                13.692273 |      142.309202 |             78.3 |       2.1 |              1011.0 | 0.0 | 0.0 |    17.0 |                  78.3 |                   5.0 |                     13.692273 |           142.309202 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                19.406929 |                     19.406929 |                          0.0 |                      1.230421 |                           1.230421 |            15.195625 |                 15.195625 |     995.804375 |   1.206937 | -2.9 | 13.278507 |  -22.32896 |            108.154994 |                108.154994 |                 108.154994 |                 21.775852 |                      0.0 |    86.379141 |              0.0 |         86.379141 |             67.254108 |                 100.0 |               159.371564 |              110.061251 |    0.0 |  0.0 | 3.393474 |   0.0 | 2.035891 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.127624 |  0.001477 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.794145 | 40.921496 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.158633 |   0.0 |   0.0 |   0.0 | -0.031009 |   0.0 |  0.001477 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |                13.618831 |      126.722078 |             78.5 |       2.1 |              1014.0 | 0.0 | 0.0 |    18.2 |                  78.5 |                   3.8 |                     13.618831 |           126.722078 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                20.933479 |                     20.933479 |                          0.0 |                      1.314615 |                           1.314615 |            16.432781 |                 16.432781 |     997.567219 |   1.204994 | -2.9 | 18.874047 |   4.044283 |              96.30878 |                  96.30878 |                   96.30878 |                 17.491881 |                      0.0 |    78.816899 |              0.0 |         78.816899 |             67.254108 |                 100.0 |               163.714409 |               110.73073 |    0.0 |  0.0 | 3.321963 |   0.0 | 2.018218 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |   0.05375 |  0.000622 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 367.849862 | 38.903278 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.084648 |   0.0 |   0.0 |   0.0 | -0.030898 |   0.0 |  0.000622 |

There is no indication of an error in the water balance:

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_v3_acker_routing_daily:

:ref:`acre (routing) <lland_v1_acker_routing>`
----------------------------------------------

The following calculation shows the possibility to route inflowing runoff, discussed in
the documentation of |lland_v1|:

.. integration-test::

    >>> inlet.sequences.sim.series = 0.02
    >>> test("lland_v3_acker_routing_daily",
    ...      axis1=(inputs.nied, fluxes.qah), axis2=states.bowa)
    |       date | nied | teml | sunshineduration | possiblesunshineduration | globalradiation | relativehumidity | windspeed | atmosphericpressure |   qz |   qzh | temltag | dailyrelativehumidity | dailysunshineduration | dailypossiblesunshineduration | dailyglobalradiation |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair |    g |        tz |         wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance |   nbes | sbes |     evpo |   evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg |   wada |      qdb | qib1 | qib2 | qbb |     qkap |     qdgz |      qah |       qa | inzp | wats | waes | esnow | taus |       ebdn |      bowa |    qdgz1 | qdgz2 | qigz1 | qigz2 |      qbgz |    qdga1 | qdga2 | qiga1 | qiga2 |      qbga | inlet |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |                15.690785 |       190.25149 |             86.2 |       2.4 |              1007.0 | 0.02 | 1.728 |    16.1 |                  86.2 |                   6.3 |                     15.690785 |            190.25149 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            15.797694 |                 15.797694 |     991.202306 |   1.205601 | -2.9 | 11.733333 |      -26.2 |            144.591133 |                144.591133 |                 144.591133 |                 21.732733 |                      0.0 |     122.8584 |              0.0 |          122.8584 |             58.847345 |                 100.0 |               111.309423 |               99.889466 | 11.802 |  0.0 | 3.623832 |   0.7 | 1.720128 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 11.802 | 0.761343 |  0.0 |  0.0 | 0.0 |      0.0 | 0.761343 | 1.924005 | 0.022269 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |      373.3 | 81.320529 | 0.761343 |   0.0 |   0.0 |   0.0 |       0.0 | 0.196005 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.022269 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |                15.628821 |      127.660782 |             85.8 |       1.6 |              1013.0 | 0.02 | 1.728 |    17.3 |                  85.8 |                   1.7 |                     15.628821 |           127.660782 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                19.779139 |                     19.779139 |                          0.0 |                      1.251029 |                           1.251029 |            16.970501 |                 16.970501 |     996.029499 |   1.207284 | -2.9 | 18.443733 |     6.8624 |             97.022194 |                 97.022194 |                  97.022194 |                 11.219797 |                      0.0 |    85.802398 |              0.0 |         85.802398 |             88.271017 |                 100.0 |                 98.30202 |               94.620613 |    0.0 |  0.0 | 2.555581 | 0.094 | 1.697173 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.991667 | 0.023052 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   363.5376 | 79.623355 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.263667 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.023052 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |                15.566134 |      164.401298 |             82.4 |       1.5 |              1016.0 | 0.02 | 1.728 |    18.9 |                  82.4 |                   4.5 |                     15.566134 |           164.401298 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                21.871576 |                     21.871576 |                          0.0 |                      1.365939 |                           1.365939 |            18.022179 |                 18.022179 |     997.977821 |   1.203775 | -2.9 | 15.632162 | -19.607027 |            124.944986 |                124.944986 |                 124.944986 |                 16.044181 |                      0.0 |   108.900806 |              0.0 |        108.900806 |             94.155751 |                 100.0 |               100.628883 |               95.574208 |    0.0 |  0.0 | 3.286385 |   0.0 | 2.329453 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.868696 | 0.021628 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.244627 | 77.293902 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.140696 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.021628 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |                15.502754 |      269.020502 |             77.6 |       2.8 |              1017.0 | 0.02 | 1.728 |    17.8 |                  77.6 |                  12.4 |                     15.502754 |           269.020502 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                20.413369 |                     20.413369 |                          0.0 |                      1.286025 |                           1.286025 |            15.840774 |                 15.840774 |    1001.159226 |    1.21051 | -2.9 | 20.443786 |  15.862716 |            204.455581 |                204.455581 |                 204.455581 |                 35.980407 |                      0.0 |   168.475175 |              0.0 |        168.475175 |             50.440581 |                 100.0 |                103.85044 |               96.853188 |    0.0 |  0.0 | 5.847296 | 0.094 | 3.314337 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.803077 | 0.020869 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 361.481911 | 73.979565 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.075077 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020869 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |                15.438712 |      287.848667 |             74.9 |       3.0 |              1018.0 | 0.02 | 1.728 |    18.5 |                  74.9 |                  13.9 |                     15.438712 |           287.848667 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                21.331125 |                     21.331125 |                          0.0 |                      1.336407 |                           1.336407 |            15.977013 |                 15.977013 |    1002.022987 |   1.208738 | -2.9 | 15.040124 | -20.759257 |            218.764987 |                218.764987 |                 218.764987 |                 39.523937 |                      0.0 |   179.241051 |              0.0 |        179.241051 |             47.077876 |                 100.0 |               108.495906 |                98.61702 |    0.0 |  0.0 | 6.638473 |   0.0 |  3.73228 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.768062 | 0.020464 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.341169 | 70.247285 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.040062 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020464 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |                15.374034 |      274.674263 |             77.9 |       3.5 |              1018.0 | 0.02 | 1.728 |    19.9 |                  77.9 |                  13.0 |                     15.374034 |           274.674263 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                23.275348 |                     23.275348 |                          0.0 |                      1.442189 |                           1.442189 |            18.131496 |                 18.131496 |     999.868504 |   1.201995 | -2.9 |  20.18359 |   1.701539 |             208.75244 |                 208.75244 |                  208.75244 |                 32.936813 |                      0.0 |   175.815627 |              0.0 |        175.815627 |             40.352465 |                 100.0 |               113.827642 |              100.519454 |    0.0 |  0.0 | 6.828968 |   0.0 | 3.652387 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.749377 | 0.020247 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.739629 | 66.594898 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.021377 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020247 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |                 15.30875 |      284.047892 |             73.5 |       3.8 |              1019.0 | 0.02 | 1.728 |    20.8 |                  73.5 |                  13.8 |                      15.30875 |           284.047892 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            18.084931 |                 18.084931 |    1000.915069 |   1.199521 | -2.9 | 18.858347 | -11.649921 |            215.876398 |                215.876398 |                 215.876398 |                  35.31351 |                      0.0 |   180.562888 |              0.0 |        180.562888 |             37.166744 |                 100.0 |               119.167697 |              102.297207 |    0.0 |  0.0 | 7.945133 |   0.0 | 4.128687 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.739407 | 0.020132 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  383.48955 |  62.46621 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.011407 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020132 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |                15.242885 |      262.971263 |             80.1 |       3.3 |              1019.0 | 0.02 | 1.728 |    20.9 |                  80.1 |                  12.3 |                     15.242885 |           262.971263 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            19.830469 |                 19.830469 |     999.169531 |   1.198331 | -2.9 | 21.378324 |   2.869942 |             199.85816 |                 199.85816 |                  199.85816 |                 28.392598 |                      0.0 |   171.465562 |              0.0 |        171.465562 |             42.798069 |                 100.0 |               125.381825 |              104.230888 |    0.0 |  0.0 | 6.427975 |   0.0 | 3.518399 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.734087 |  0.02007 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.719608 | 58.947811 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.006087 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 |  0.02007 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |                15.176467 |      272.256414 |             72.9 |       1.8 |              1019.0 | 0.02 | 1.728 |    22.3 |                  72.9 |                  13.1 |                     15.176467 |           272.256414 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 26.96965 |                      26.96965 |                          0.0 |                      1.639967 |                           1.639967 |            19.660875 |                 19.660875 |     999.339125 |   1.192729 | -2.9 |  19.71658 | -15.500518 |            206.914874 |                206.914874 |                 206.914874 |                 31.114889 |                      0.0 |   175.799986 |              0.0 |        175.799986 |             78.463126 |                 100.0 |               130.860307 |              105.804393 |    0.0 |  0.0 | 6.036589 |   0.0 | 4.091259 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.731248 | 0.020038 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 390.320126 | 54.856552 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.003248 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020038 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |                15.109521 |      267.019218 |             68.5 |       1.9 |              1020.0 | 0.02 | 1.728 |    23.0 |                  68.5 |                  12.8 |                     15.109521 |           267.019218 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                28.139149 |                     28.139149 |                          0.0 |                      1.701779 |                           1.701779 |            19.275317 |                 19.275317 |    1000.724683 |   1.191257 | -2.9 |  23.34553 |   2.073177 |            202.934606 |                202.934606 |                 202.934606 |                 32.010643 |                      0.0 |   170.923963 |              0.0 |        170.923963 |             74.333488 |                 100.0 |               137.497212 |               107.59145 |    0.0 |  0.0 | 6.493896 |   0.0 | 4.345753 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.729733 |  0.02002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.346949 | 50.510799 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.001733 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 |  0.02002 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |                15.042071 |      269.626547 |             68.6 |       3.0 |              1021.0 | 0.02 | 1.728 |    23.1 |                  68.6 |                  13.1 |                     15.042071 |           269.626547 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                28.309776 |                     28.309776 |                          0.0 |                      1.710768 |                           1.710768 |            19.420506 |                 19.420506 |    1001.579494 |   1.191966 | -2.9 | 21.913255 |  -7.120473 |            204.916175 |                204.916175 |                 204.916175 |                 32.400014 |                      0.0 |   172.516162 |              0.0 |        172.516162 |             47.077876 |                 100.0 |               144.951913 |              109.449121 |    0.0 |  0.0 | 7.914607 |   0.0 | 4.547949 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.728925 | 0.020011 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 389.567421 | 45.962849 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000925 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020011 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |                14.974141 |      270.878291 |             66.0 |       2.5 |              1022.0 | 0.02 | 1.728 |    23.2 |                  66.0 |                  13.3 |                     14.974141 |           270.878291 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |             18.79766 |                  18.79766 |     1003.20234 |   1.193016 | -2.9 | 23.128751 |  -0.427496 |            205.867501 |                205.867501 |                 205.867501 |                 34.565783 |                      0.0 |   171.301718 |              0.0 |        171.301718 |             56.493451 |                 100.0 |               153.358051 |              111.375228 |    0.0 |  0.0 | 7.566564 | 0.094 | 4.554398 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.728494 | 0.020006 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.094917 | 41.408451 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000494 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020006 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |                14.905755 |      261.697164 |             69.0 |       1.6 |              1019.0 | 0.02 | 1.728 |    23.2 |                  69.0 |                  12.7 |                     14.905755 |           261.697164 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |              19.6521 |                   19.6521 |       999.3479 |    1.18911 | -2.9 | 22.416669 |  -4.699983 |            198.889844 |                198.889844 |                 198.889844 |                  31.34697 |                      0.0 |   167.542875 |              0.0 |        167.542875 |             88.271017 |                 100.0 |                162.65311 |               113.31833 |    0.0 |  0.0 | 5.996216 |   0.0 | 4.137046 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.728263 | 0.020003 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   388.8949 | 37.271405 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000263 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020003 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |                14.836934 |      227.962116 |             75.6 |       2.7 |              1015.0 | 0.02 | 1.728 |    21.7 |                  75.6 |                  10.2 |                     14.836934 |           227.962116 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 26.00118 |                      26.00118 |                          0.0 |                        1.5885 |                             1.5885 |            19.656892 |                 19.656892 |     995.343108 |   1.190432 | -2.9 | 22.935065 |   7.410388 |            173.251208 |                173.251208 |                 173.251208 |                 25.918437 |                      0.0 |   147.332771 |              0.0 |        147.332771 |             52.308751 |                 100.0 |                172.21097 |              115.125428 |    0.0 |  0.0 | 5.983775 |   0.0 | 3.428735 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.728141 | 0.020002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 378.584513 |  33.84267 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000141 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 | 0.020002 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |                  14.7677 |      216.356321 |             81.4 |       3.2 |              1016.0 | 0.02 | 1.728 |    20.8 |                  81.4 |                   9.4 |                       14.7677 |           216.356321 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            20.028753 |                 20.028753 |     995.971247 |   1.195095 | -2.9 | 19.965673 |  -5.005962 |            164.430804 |                164.430804 |                 164.430804 |                 23.397902 |                      0.0 |   141.032902 |              0.0 |        141.032902 |             44.135508 |                 100.0 |               181.322396 |              116.674937 |  1.838 |  0.0 | 5.470835 |   0.7 | 2.505678 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  1.838 | 0.049893 |  0.0 |  0.0 | 0.0 |      0.0 | 0.049893 |  1.74092 |  0.02015 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.690475 | 33.125099 | 0.049893 |   0.0 |   0.0 |   0.0 |       0.0 |  0.01292 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 |  0.02015 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |                14.698075 |      226.637904 |             79.0 |       1.3 |              1019.0 | 0.02 | 1.728 |    19.5 |                  79.0 |                  10.3 |                     14.698075 |           226.637904 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                22.704676 |                     22.704676 |                          0.0 |                      1.411269 |                           1.411269 |            17.936694 |                 17.936694 |    1001.063306 |   1.204916 | -2.9 |  20.57219 |   6.433141 |            172.244807 |                172.244807 |                 172.244807 |                 28.794151 |                      0.0 |   143.450656 |              0.0 |        143.450656 |            108.641252 |                 100.0 |               183.411459 |               116.86708 |    0.0 |  0.0 | 4.081206 |   0.0 | 2.836506 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.745319 |   0.0202 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 371.357334 | 30.288593 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.017319 |   0.0 |   0.0 |   0.0 |       0.0 |  0.02 |   0.0202 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |                14.628078 |      235.528503 |             75.9 |       1.9 |              1019.0 | 0.02 | 1.728 |    21.0 |                  75.9 |                  11.1 |                     14.628078 |           235.528503 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                24.909749 |                     24.909749 |                          0.0 |                      1.530181 |                           1.530181 |            18.906499 |                 18.906499 |    1000.093501 |   1.198338 | -2.9 | 17.884246 | -18.694526 |            179.001662 |                179.001662 |                 179.001662 |                 29.177702 |                      0.0 |    149.82396 |              0.0 |         149.82396 |             74.333488 |                 100.0 |               192.482064 |              118.222742 |    0.0 |  0.0 | 5.175279 |   0.0 | 3.276924 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.03799 |      0.0 | 1.737173 | 0.020106 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.151861 | 27.049659 |      0.0 |   0.0 |   0.0 |   0.0 |  -0.03799 | 0.009242 |   0.0 |   0.0 |   0.0 | -0.000068 |  0.02 | 0.020106 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |                14.557728 |      232.849428 |             70.0 |       2.0 |              1018.0 | 0.02 | 1.728 |    21.6 |                  70.0 |                  11.0 |                     14.557728 |           232.849428 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                25.842752 |                     25.842752 |                          0.0 |                      1.580055 |                           1.580055 |            18.089927 |                 18.089927 |     999.910073 |   1.195081 | -2.9 | 22.433069 |   4.998415 |            176.965566 |                176.965566 |                 176.965566 |                 31.281288 |                      0.0 |   145.684278 |              0.0 |        145.684278 |             70.616813 |                 100.0 |               204.991875 |              119.966911 |    0.0 |  0.0 | 5.737858 |   0.0 | 3.581415 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.239244 |      0.0 | 1.732367 | 0.020051 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.253445 | 23.707488 |      0.0 |   0.0 |   0.0 |   0.0 | -0.239244 | 0.004931 |   0.0 |   0.0 |   0.0 | -0.000565 |  0.02 | 0.020051 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |                14.487046 |       199.71349 |             70.3 |       2.6 |              1019.0 | 0.02 | 1.728 |    21.1 |                  70.3 |                   8.5 |                     14.487046 |            199.71349 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            17.619414 |                 17.619414 |    1001.380586 |   1.198506 | -2.9 | 20.158326 |  -5.650046 |            151.782252 |                151.782252 |                 151.782252 |                 26.711296 |                      0.0 |   125.070956 |              0.0 |        125.070956 |             54.320626 |                 100.0 |               221.727637 |              122.069797 |    0.0 |  0.0 | 5.866935 |   0.0 | 3.296835 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.446913 |      0.0 | 1.728839 |  0.02001 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 382.003492 | 20.857567 |      0.0 |   0.0 |   0.0 |   0.0 | -0.446913 | 0.002631 |   0.0 |   0.0 |   0.0 | -0.001792 |  0.02 |  0.02001 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |                14.416048 |      233.771902 |             69.0 |       3.6 |              1019.0 | 0.02 | 1.728 |    22.4 |                  69.0 |                  11.3 |                     14.416048 |           233.771902 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                27.134082 |                     27.134082 |                          0.0 |                      1.648679 |                           1.648679 |            18.722517 |                 18.722517 |    1000.277483 |   1.192743 | -2.9 | 20.950339 |  -8.697966 |            177.666646 |                177.666646 |                 177.666646 |                 31.116622 |                      0.0 |   146.550024 |              0.0 |        146.550024 |             39.231563 |                 100.0 |               241.230014 |              124.199013 |    0.0 |  0.0 |  7.82923 |   0.0 | 3.891994 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.623996 |      0.0 |   1.7257 | 0.019973 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.801458 | 17.589568 |      0.0 |   0.0 |   0.0 |   0.0 | -0.623996 | 0.001404 |   0.0 |   0.0 |   0.0 | -0.003704 |  0.02 | 0.019973 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |                14.344752 |      246.121286 |             62.1 |       2.5 |              1017.0 | 0.02 | 1.728 |    22.5 |                  62.1 |                  12.4 |                     14.344752 |           246.121286 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                27.299387 |                     27.299387 |                          0.0 |                      1.657431 |                           1.657431 |             16.95292 |                  16.95292 |     1000.04708 |   1.190771 | -2.9 | 22.620153 |    0.72092 |            187.052177 |                187.052177 |                 187.052177 |                 38.297604 |                      0.0 |   148.754573 |              0.0 |        148.754573 |             56.493451 |                 100.0 |               274.853414 |              127.355551 |    0.0 |  0.0 | 7.322689 |   0.0 | 4.183176 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.827056 |      0.0 | 1.722459 | 0.019936 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.180539 | 14.233449 |      0.0 |   0.0 |   0.0 |   0.0 | -0.827056 | 0.000749 |   0.0 |   0.0 |   0.0 |  -0.00629 |  0.02 | 0.019936 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |                14.273175 |        90.58551 |             84.5 |       2.1 |              1015.0 | 0.02 | 1.728 |    19.6 |                  84.5 |                   0.1 |                     14.273175 |             90.58551 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                22.846186 |                     22.846186 |                          0.0 |                      1.418946 |                           1.418946 |            19.305027 |                 19.305027 |     995.694973 |   1.199129 | -2.9 | 21.577328 |  11.863971 |             68.844988 |                 68.844988 |                  68.844988 |                   7.02118 |                      0.0 |    61.823807 |              0.0 |         61.823807 |             67.254108 |                 100.0 |               337.448215 |              131.864309 |   1.18 |  0.0 | 2.567328 |   0.7 | 1.075698 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   1.18 | 0.013305 |  0.0 |  0.0 | 0.0 | 1.035592 | 0.013305 | 1.722221 | 0.019933 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 369.416568 | 15.360037 | 0.013305 |   0.0 |   0.0 |   0.0 | -1.035592 | 0.003825 |   0.0 |   0.0 |   0.0 | -0.009605 |  0.02 | 0.019933 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |                14.201333 |      171.944993 |             83.6 |       1.5 |              1015.0 | 0.02 | 1.728 |    20.3 |                  83.6 |                   6.7 |                     14.201333 |           171.944993 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                23.858503 |                     23.858503 |                          0.0 |                      1.473678 |                           1.473678 |            19.945708 |                 19.945708 |     995.054292 |   1.195981 | -2.9 | 17.325305 | -17.848171 |            130.678195 |                130.678195 |                 130.678195 |                 18.994889 |                      0.0 |   111.683306 |              0.0 |        111.683306 |             94.155751 |                 100.0 |                311.33348 |              129.741425 |    0.0 |  0.0 | 3.374147 | 0.282 | 2.014865 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.96559 |      0.0 | 1.719666 | 0.019904 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.364739 | 14.310761 |      0.0 |   0.0 |   0.0 |   0.0 |  -0.96559 | 0.004821 |   0.0 |   0.0 |   0.0 | -0.013155 |  0.02 | 0.019904 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |                14.129243 |      216.549813 |             76.5 |       2.2 |              1011.0 | 0.02 | 1.728 |    24.1 |                  76.5 |                  10.4 |                     14.129243 |           216.549813 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                30.066256 |                     30.066256 |                          0.0 |                       1.80287 |                            1.80287 |            23.000686 |                 23.000686 |     987.999314 |   1.174651 | -2.9 | 21.630378 | -14.817732 |            164.577858 |                164.577858 |                 164.577858 |                 21.629737 |                      0.0 |   142.948121 |              0.0 |        142.948121 |             64.197103 |                 100.0 |                335.42542 |              131.069494 |    0.0 |  0.0 | 5.573024 |   0.0 | 3.362119 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.030788 |      0.0 | 1.713889 | 0.019837 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  396.28247 |  11.97943 |      0.0 |   0.0 |   0.0 |   0.0 | -1.030788 | 0.002573 |   0.0 |   0.0 |   0.0 | -0.016684 |  0.02 | 0.019837 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |                14.056918 |       166.91683 |             76.6 |       1.9 |              1008.0 | 0.02 | 1.728 |    23.7 |                  76.6 |                   6.5 |                     14.056918 |            166.91683 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                29.352596 |                     29.352596 |                          0.0 |                      1.765541 |                           1.765541 |            22.484088 |                 22.484088 |     985.515912 |   1.172942 | -2.9 | 25.062685 |   8.176109 |             126.85679 |                 126.85679 |                  126.85679 |                 16.222254 |                      0.0 |   110.634536 |              0.0 |        110.634536 |             74.333488 |                 100.0 |               419.576726 |               135.15103 |    0.0 |  0.0 | 4.467668 | 0.094 | 2.719352 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.175647 |      0.0 | 1.708796 | 0.019778 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.206362 | 10.435725 |      0.0 |   0.0 |   0.0 |   0.0 | -1.175647 | 0.001373 |   0.0 |   0.0 |   0.0 | -0.020577 |  0.02 | 0.019778 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |                13.984374 |      145.985611 |             84.2 |       2.3 |              1006.0 | 0.02 | 1.728 |    21.1 |                  84.2 |                   4.9 |                     13.984374 |           145.985611 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            21.103195 |                 21.103195 |     984.896805 |   1.181557 | -2.9 | 21.872765 |   4.636593 |            110.949064 |                110.949064 |                 110.949064 |                 14.602883 |                      0.0 |    96.346181 |              0.0 |         96.346181 |             61.405925 |                 100.0 |               522.087742 |              138.401561 |   9.17 |  0.0 | 3.569826 |   0.7 | 1.597373 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   9.17 | 0.104578 |  0.0 |  0.0 | 0.0 | 1.271567 | 0.104578 | 1.730769 | 0.020032 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.669769 | 19.175341 | 0.104578 |   0.0 |   0.0 |   0.0 | -1.271567 | 0.027656 |   0.0 |   0.0 |   0.0 | -0.024887 |  0.02 | 0.020032 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |                13.911624 |       165.54389 |             85.8 |       2.1 |              1005.0 | 0.02 | 1.728 |    20.9 |                  85.8 |                   6.6 |                     13.911624 |            165.54389 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            21.241626 |                 21.241626 |     983.758374 |   1.181114 | -2.9 | 19.702227 |   -7.18664 |            125.813357 |                125.813357 |                 125.813357 |                 17.349121 |                      0.0 |   108.464235 |              0.0 |        108.464235 |             67.254108 |                 100.0 |               256.482293 |              123.963828 |   5.41 |  0.0 |  3.56693 |   0.7 | 1.722258 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   5.41 | 0.090745 |  0.0 |  0.0 | 0.0 | 0.728523 | 0.090745 |  1.75959 | 0.020366 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 381.956408 | 23.500861 | 0.090745 |   0.0 |   0.0 |   0.0 | -0.728523 |  0.05997 |   0.0 |   0.0 |   0.0 |  -0.02838 |  0.02 | 0.020366 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |                13.838682 |       87.693346 |             86.5 |       2.6 |              1000.0 | 0.02 | 1.728 |    20.2 |                  86.5 |                   0.3 |                     13.838682 |            87.693346 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                23.711531 |                     23.711531 |                          0.0 |                      1.465752 |                           1.465752 |            20.510475 |                 20.510475 |     979.489525 |   1.178323 | -2.9 | 20.936779 |   4.420674 |             66.646943 |                 66.646943 |                  66.646943 |                  6.818024 |                      0.0 |    59.828919 |              0.0 |         59.828919 |             54.320626 |                 100.0 |                222.94421 |               120.11308 |   13.4 |  0.0 | 2.621228 |   0.7 | 1.071953 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   13.4 | 0.312945 |  0.0 |  0.0 | 0.0 | 0.459752 | 0.312945 | 1.829121 |  0.02117 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.635735 | 35.975715 | 0.312945 |   0.0 |   0.0 |   0.0 | -0.459752 | 0.131528 |   0.0 |   0.0 |   0.0 | -0.030407 |  0.02 |  0.02117 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |                13.765561 |       84.356885 |             89.6 |       2.6 |              1004.0 | 0.02 | 1.728 |    16.1 |                  89.6 |                   0.1 |                     13.765561 |            84.356885 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            16.420805 |                 16.420805 |     987.579195 |   1.201704 | -2.9 | 18.828425 |   16.37055 |             64.111233 |                 64.111233 |                  64.111233 |                  8.199212 |                      0.0 |    55.912021 |              0.0 |         55.912021 |             54.320626 |                 100.0 |               175.502292 |              113.152806 |  7.948 |  0.0 | 2.027587 |   0.7 | 0.717164 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  7.948 | 0.249164 |  0.0 |  0.0 | 0.0 |      0.0 | 0.249164 | 1.896598 | 0.021951 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 355.365185 | 42.957387 | 0.249164 |   0.0 |   0.0 |   0.0 |       0.0 | 0.199719 |   0.0 |   0.0 |   0.0 | -0.031121 |  0.02 | 0.021951 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |                13.692273 |      142.309202 |             78.3 |       2.1 |              1011.0 | 0.02 | 1.728 |    17.0 |                  78.3 |                   5.0 |                     13.692273 |           142.309202 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                19.406929 |                     19.406929 |                          0.0 |                      1.230421 |                           1.230421 |            15.195625 |                 15.195625 |     995.804375 |   1.206937 | -2.9 | 13.278507 |  -22.32896 |            108.154994 |                108.154994 |                 108.154994 |                 21.775852 |                      0.0 |    86.379141 |              0.0 |         86.379141 |             67.254108 |                 100.0 |               159.371564 |              110.061251 |    0.0 |  0.0 | 3.393474 |   0.0 | 2.035891 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 | 1.855624 | 0.021477 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.794145 | 40.921496 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.158633 |   0.0 |   0.0 |   0.0 | -0.031009 |  0.02 | 0.021477 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |                13.618831 |      126.722078 |             78.5 |       2.1 |              1014.0 | 0.02 | 1.728 |    18.2 |                  78.5 |                   3.8 |                     13.618831 |           126.722078 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                20.933479 |                     20.933479 |                          0.0 |                      1.314615 |                           1.314615 |            16.432781 |                 16.432781 |     997.567219 |   1.204994 | -2.9 | 18.874047 |   4.044283 |              96.30878 |                  96.30878 |                   96.30878 |                 17.491881 |                      0.0 |    78.816899 |              0.0 |         78.816899 |             67.254108 |                 100.0 |               163.714409 |               110.73073 |    0.0 |  0.0 | 3.321963 |   0.0 | 2.018218 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  1.78175 | 0.020622 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 367.849862 | 38.903278 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.084648 |   0.0 |   0.0 |   0.0 | -0.030898 |  0.02 | 0.020622 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_v3_acker_heavy_rain_daily:

acre (heavy rain)
-----------------

Integration test :ref:`lland_v3_acker_summer_daily` deals with dry summer conditions.
To show how |lland_v3| works for warm but wet conditions, we set the precipitation
input time series constantly to 20 mm.  Now, the soil water content (|BoWa|) rises from
its initial value of 72 mm and nearly reaches its maximum value of 309 mm (|WMax|),
resulting in a relevant response of all runoff components:

.. integration-test::

    >>> lnk(ACKER)
    >>> inlet.sequences.sim.series = 0.0
    >>> nied = inputs.nied.series.copy()
    >>> inputs.nied.series = 20.0
    >>> conditions_acker_heavy_rain = test(
    ...     "lland_v3_acker_heavy_rain_daily",
    ...     axis1=(inputs.nied, fluxes.qah), axis2=states.bowa,
    ...     get_conditions="1997-08-03")
    |       date | nied | teml | sunshineduration | possiblesunshineduration | globalradiation | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration | dailypossiblesunshineduration | dailyglobalradiation | nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair |    g |        tz |         wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |     evpo | evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada |       qdb |     qib1 |     qib2 |      qbb | qkap |      qdgz |       qah |       qa | inzp | wats | waes | esnow | taus |       ebdn |       bowa |     qdgz1 |    qdgz2 |    qigz1 |    qigz2 |     qbgz |     qdga1 |    qdga2 |    qiga1 |    qiga2 |     qbga | inlet |   outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 20.0 | 16.1 |              6.3 |                15.690785 |       190.25149 |             86.2 |       2.4 |              1007.0 | 0.0 | 0.0 |    16.1 |                  86.2 |                   6.3 |                     15.690785 |            190.25149 | 18.8 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            15.797694 |                 15.797694 |     991.202306 |   1.205601 | -2.9 | 11.733333 |      -26.2 |            144.591133 |                144.591133 |                 144.591133 |                 21.732733 |                      0.0 |     122.8584 |              0.0 |          122.8584 |             58.847345 |                 100.0 |               111.309423 |               99.889466 | 18.1 |  0.0 | 3.623832 | 0.7 | 1.720128 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  1.217805 |      0.0 |      0.0 |      0.0 |  0.0 |  1.217805 |   0.31352 | 0.003629 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |      373.3 |  87.162066 |  1.217805 |      0.0 |      0.0 |      0.0 |      0.0 |   0.31352 |      0.0 |      0.0 |      0.0 |      0.0 |   0.0 | 0.003629 |
    | 1997-02-08 | 20.0 | 17.3 |              1.7 |                15.628821 |      127.660782 |             85.8 |       1.6 |              1013.0 | 0.0 | 0.0 |    17.3 |                  85.8 |                   1.7 |                     15.628821 |           127.660782 | 18.8 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                19.779139 |                     19.779139 |                          0.0 |                      1.251029 |                           1.251029 |            16.970501 |                 16.970501 |     996.029499 |   1.207284 | -2.9 | 18.443733 |     6.8624 |             97.022194 |                 97.022194 |                  97.022194 |                 11.219797 |                      0.0 |    85.802398 |              0.0 |         85.802398 |             88.271017 |                 100.0 |                90.407062 |               91.066356 | 18.1 |  0.0 | 2.555581 | 0.7 | 1.294457 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |   1.48087 |      0.0 |      0.0 |      0.0 |  0.0 |   1.48087 |  0.802994 | 0.009294 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   363.5376 | 102.486739 |   1.48087 |      0.0 |      0.0 |      0.0 |      0.0 |  0.802994 |      0.0 |      0.0 |      0.0 |      0.0 |   0.0 | 0.009294 |
    | 1997-03-08 | 20.0 | 18.9 |              4.5 |                15.566134 |      164.401298 |             82.4 |       1.5 |              1016.0 | 0.0 | 0.0 |    18.9 |                  82.4 |                   4.5 |                     15.566134 |           164.401298 | 18.8 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                21.871576 |                     21.871576 |                          0.0 |                      1.365939 |                           1.365939 |            18.022179 |                 18.022179 |     997.977821 |   1.203775 | -2.9 | 15.632162 | -19.607027 |            124.944986 |                124.944986 |                 124.944986 |                 16.044181 |                      0.0 |   108.900806 |              0.0 |        108.900806 |             94.155751 |                 100.0 |                70.326844 |               80.364888 | 18.1 |  0.0 | 3.286385 | 0.7 | 1.922357 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  1.761525 |      0.0 |      0.0 |      0.0 |  0.0 |  1.761525 |  1.191401 | 0.013789 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.244627 | 116.902858 |  1.761525 |      0.0 |      0.0 |      0.0 |      0.0 |  1.191401 |      0.0 |      0.0 |      0.0 |      0.0 |   0.0 | 0.013789 |
    | 1997-04-08 | 20.0 | 17.8 |             12.4 |                15.502754 |      269.020502 |             77.6 |       2.8 |              1017.0 | 0.0 | 0.0 |    17.8 |                  77.6 |                  12.4 |                     15.502754 |           269.020502 | 18.8 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                20.413369 |                     20.413369 |                          0.0 |                      1.286025 |                           1.286025 |            15.840774 |                 15.840774 |    1001.159226 |    1.21051 | -2.9 | 20.443786 |  15.862716 |            204.455581 |                204.455581 |                 204.455581 |                 35.980407 |                      0.0 |   168.475175 |              0.0 |        168.475175 |             50.440581 |                 100.0 |                51.993251 |               67.804391 | 18.1 |  0.0 | 5.847296 | 0.7 | 3.397171 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  2.040918 |      0.0 |      0.0 |      0.0 |  0.0 |  2.040918 |   1.52923 | 0.017699 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 361.481911 | 129.564768 |  2.040918 |      0.0 |      0.0 |      0.0 |      0.0 |   1.52923 |      0.0 |      0.0 |      0.0 |      0.0 |   0.0 | 0.017699 |
    | 1997-05-08 | 20.0 | 18.5 |             13.9 |                15.438712 |      287.848667 |             74.9 |       3.0 |              1018.0 | 0.0 | 0.0 |    18.5 |                  74.9 |                  13.9 |                     15.438712 |           287.848667 | 18.8 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                21.331125 |                     21.331125 |                          0.0 |                      1.336407 |                           1.336407 |            15.977013 |                 15.977013 |    1002.022987 |   1.208738 | -2.9 | 15.040124 | -20.759257 |            218.764987 |                218.764987 |                 218.764987 |                 39.523937 |                      0.0 |   179.241051 |              0.0 |        179.241051 |             47.077876 |                 100.0 |                 48.85611 |               65.355461 | 18.1 |  0.0 | 6.638473 | 0.7 | 3.917105 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  2.300286 | 0.101438 |      0.0 | 0.010165 |  0.0 |  2.300286 |  1.849051 | 0.021401 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.341169 | 141.335775 |  2.300286 |      0.0 | 0.101438 |      0.0 | 0.010165 |  1.834649 |      0.0 | 0.014384 |      0.0 | 0.000018 |   0.0 | 0.021401 |
    | 1997-06-08 | 20.0 | 19.9 |             13.0 |                15.374034 |      274.674263 |             77.9 |       3.5 |              1018.0 | 0.0 | 0.0 |    19.9 |                  77.9 |                  13.0 |                     15.374034 |           274.674263 | 18.8 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                23.275348 |                     23.275348 |                          0.0 |                      1.442189 |                           1.442189 |            18.131496 |                 18.131496 |     999.868504 |   1.201995 | -2.9 |  20.18359 |   1.701539 |             208.75244 |                 208.75244 |                  208.75244 |                 32.936813 |                      0.0 |   175.815627 |              0.0 |        175.815627 |             40.352465 |                 100.0 |                 48.85611 |               65.425338 | 18.1 |  0.0 | 6.828968 | 0.7 | 3.913569 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  2.554694 | 0.110654 |      0.0 | 0.021936 |  0.0 |  2.554694 |  2.156542 |  0.02496 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.739629 | 152.834923 |  2.554694 |      0.0 | 0.110654 |      0.0 | 0.021936 |  2.117313 |      0.0 | 0.039153 |      0.0 | 0.000076 |   0.0 |  0.02496 |
    | 1997-07-08 | 20.0 | 20.8 |             13.8 |                 15.30875 |      284.047892 |             73.5 |       3.8 |              1019.0 | 0.0 | 0.0 |    20.8 |                  73.5 |                  13.8 |                      15.30875 |           284.047892 | 18.8 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            18.084931 |                 18.084931 |    1000.915069 |   1.199521 | -2.9 | 18.858347 | -11.649921 |            215.876398 |                215.876398 |                 215.876398 |                  35.31351 |                      0.0 |   180.562888 |              0.0 |        180.562888 |             37.166744 |                 100.0 |                 48.85611 |               65.496024 | 18.1 |  0.0 | 7.945133 | 0.7 | 4.551444 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  2.817235 | 0.119656 |      0.0 | 0.033435 |  0.0 |  2.817235 |  2.448767 | 0.028342 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  383.48955 | 163.413153 |  2.817235 |      0.0 | 0.119656 |      0.0 | 0.033435 |  2.388893 |      0.0 |   0.0597 |      0.0 | 0.000175 |   0.0 | 0.028342 |
    | 1997-08-08 | 20.0 | 20.9 |             12.3 |                15.242885 |      262.971263 |             80.1 |       3.3 |              1019.0 | 0.0 | 0.0 |    20.9 |                  80.1 |                  12.3 |                     15.242885 |           262.971263 | 18.8 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            19.830469 |                 19.830469 |     999.169531 |   1.198331 | -2.9 | 21.378324 |   2.869942 |             199.85816 |                 199.85816 |                  199.85816 |                 28.392598 |                      0.0 |   171.465562 |              0.0 |        171.465562 |             42.798069 |                 100.0 |                 48.85611 |               65.567492 | 18.1 |  0.0 | 6.427975 | 0.7 |  3.76789 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.072622 | 0.127938 |      0.0 | 0.044013 |  0.0 |  3.072622 |  2.731761 | 0.031618 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.719608 | 174.500689 |  3.072622 |      0.0 | 0.127938 |      0.0 | 0.044013 |  2.654416 |      0.0 | 0.077033 |      0.0 | 0.000313 |   0.0 | 0.031618 |
    | 1997-09-08 | 20.0 | 22.3 |             13.1 |                15.176467 |      272.256414 |             72.9 |       1.8 |              1019.0 | 0.0 | 0.0 |    22.3 |                  72.9 |                  13.1 |                     15.176467 |           272.256414 | 18.8 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 26.96965 |                      26.96965 |                          0.0 |                      1.639967 |                           1.639967 |            19.660875 |                 19.660875 |     999.339125 |   1.192729 | -2.9 |  19.71658 | -15.500518 |            206.914874 |                206.914874 |                 206.914874 |                 31.114889 |                      0.0 |   175.799986 |              0.0 |        175.799986 |             78.463126 |                 100.0 |                 48.85611 |               65.639719 | 18.1 |  0.0 | 6.036589 | 0.7 | 4.120968 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.356609 | 0.136619 |      0.0 | 0.055101 |  0.0 |  3.356609 |  3.015047 | 0.034896 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 390.320126 | 184.931393 |  3.356609 |      0.0 | 0.136619 |      0.0 | 0.055101 |  2.922574 |      0.0 | 0.091984 |      0.0 | 0.000489 |   0.0 | 0.034896 |
    | 1997-10-08 | 20.0 | 23.0 |             12.8 |                15.109521 |      267.019218 |             68.5 |       1.9 |              1020.0 | 0.0 | 0.0 |    23.0 |                  68.5 |                  12.8 |                     15.109521 |           267.019218 | 18.8 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                28.139149 |                     28.139149 |                          0.0 |                      1.701779 |                           1.701779 |            19.275317 |                 19.275317 |    1000.724683 |   1.191257 | -2.9 |  23.34553 |   2.073177 |            202.934606 |                202.934606 |                 202.934606 |                 32.010643 |                      0.0 |   170.923963 |              0.0 |        170.923963 |             74.333488 |                 100.0 |                 48.85611 |               65.712681 | 18.1 |  0.0 | 6.493896 | 0.7 | 4.450324 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.641281 | 0.144785 |      0.0 | 0.065531 |  0.0 |  3.641281 |  3.304166 | 0.038243 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.346949 | 194.729472 |  3.641281 |      0.0 | 0.144785 |      0.0 | 0.065531 |  3.198291 |      0.0 | 0.105171 |      0.0 | 0.000704 |   0.0 | 0.038243 |
    | 1997-11-08 | 20.0 | 23.1 |             13.1 |                15.042071 |      269.626547 |             68.6 |       3.0 |              1021.0 | 0.0 | 0.0 |    23.1 |                  68.6 |                  13.1 |                     15.042071 |           269.626547 | 18.8 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                28.309776 |                     28.309776 |                          0.0 |                      1.710768 |                           1.710768 |            19.420506 |                 19.420506 |    1001.579494 |   1.191966 | -2.9 | 21.913255 |  -7.120473 |            204.916175 |                204.916175 |                 204.916175 |                 32.400014 |                      0.0 |   172.516162 |              0.0 |        172.516162 |             47.077876 |                 100.0 |                 48.85611 |               65.786357 | 18.1 |  0.0 | 7.914607 | 0.7 | 4.992996 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  3.926597 | 0.152456 |      0.0 | 0.075329 |  0.0 |  3.926597 |  3.596239 | 0.041623 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 389.567421 | 203.682094 |  3.926597 |      0.0 | 0.152456 |      0.0 | 0.075329 |   3.47835 |      0.0 | 0.116936 |      0.0 | 0.000953 |   0.0 | 0.041623 |
    | 1997-12-08 | 20.0 | 23.2 |             13.3 |                14.974141 |      270.878291 |             66.0 |       2.5 |              1022.0 | 0.0 | 0.0 |    23.2 |                  66.0 |                  13.3 |                     14.974141 |           270.878291 | 18.8 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |             18.79766 |                  18.79766 |     1003.20234 |   1.193016 | -2.9 | 23.128751 |  -0.427496 |            205.867501 |                205.867501 |                 205.867501 |                 34.565783 |                      0.0 |   171.301718 |              0.0 |        171.301718 |             56.493451 |                 100.0 |                 48.85611 |               65.860724 | 18.1 |  0.0 | 7.566564 | 0.7 | 4.979765 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  4.204896 | 0.159465 |  0.00956 | 0.084282 |  0.0 |  4.204896 |  3.889149 | 0.045013 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.094917 | 212.344126 |  4.204896 |      0.0 | 0.159465 |  0.00956 | 0.084282 |  3.759055 |      0.0 | 0.127503 | 0.001356 | 0.001236 |   0.0 | 0.045013 |
    | 1997-13-08 | 20.0 | 23.2 |             12.7 |                14.905755 |      261.697164 |             69.0 |       1.6 |              1019.0 | 0.0 | 0.0 |    23.2 |                  69.0 |                  12.7 |                     14.905755 |           261.697164 | 18.8 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |              19.6521 |                   19.6521 |       999.3479 |    1.18911 | -2.9 | 22.416669 |  -4.699983 |            198.889844 |                198.889844 |                 198.889844 |                  31.34697 |                      0.0 |   167.542875 |              0.0 |        167.542875 |             88.271017 |                 100.0 |                 48.85611 |               65.935761 | 18.1 |  0.0 | 5.996216 | 0.7 | 4.198392 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  4.492791 | 0.166247 | 0.045997 | 0.092944 |  0.0 |  4.492791 |  4.188469 | 0.048478 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   388.8949 | 221.447755 |  4.492791 |      0.0 | 0.166247 | 0.045997 | 0.092944 |  4.041108 |      0.0 | 0.137079 | 0.008733 | 0.001549 |   0.0 | 0.048478 |
    | 1997-14-08 | 20.0 | 21.7 |             10.2 |                14.836934 |      227.962116 |             75.6 |       2.7 |              1015.0 | 0.0 | 0.0 |    21.7 |                  75.6 |                  10.2 |                     14.836934 |           227.962116 | 18.8 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 26.00118 |                      26.00118 |                          0.0 |                        1.5885 |                             1.5885 |            19.656892 |                 19.656892 |     995.343108 |   1.190432 | -2.9 | 22.935065 |   7.410388 |            173.251208 |                173.251208 |                 173.251208 |                 25.918437 |                      0.0 |   147.332771 |              0.0 |        147.332771 |             52.308751 |                 100.0 |                 48.85611 |               66.011448 | 18.1 |  0.0 | 5.983775 | 0.7 | 3.701991 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  4.818593 | 0.173374 | 0.100359 | 0.102048 |  0.0 |  4.818593 |  4.509972 | 0.052199 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 378.584513 |  230.65139 |  4.818593 |      0.0 | 0.173374 | 0.100359 | 0.102048 |  4.335644 |      0.0 | 0.145951 | 0.026485 | 0.001893 |   0.0 | 0.052199 |
    | 1997-15-08 | 20.0 | 20.8 |              9.4 |                  14.7677 |      216.356321 |             81.4 |       3.2 |              1016.0 | 0.0 | 0.0 |    20.8 |                  81.4 |                   9.4 |                       14.7677 |           216.356321 | 18.8 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            20.028753 |                 20.028753 |     995.971247 |   1.195095 | -2.9 | 19.965673 |  -5.005962 |            164.430804 |                164.430804 |                 164.430804 |                 23.397902 |                      0.0 |   141.032902 |              0.0 |        141.032902 |             44.135508 |                 100.0 |                 48.85611 |               66.087764 | 18.1 |  0.0 | 5.470835 | 0.7 | 3.155199 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  5.176976 |  0.18058 | 0.168029 | 0.111251 |  0.0 |  5.176976 |  4.865773 | 0.056317 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.690475 | 239.959354 |  5.176976 |      0.0 |  0.18058 | 0.168029 | 0.111251 |  4.653151 |      0.0 | 0.154363 |  0.05599 | 0.002268 |   0.0 | 0.056317 |
    | 1997-16-08 | 20.0 | 19.5 |             10.3 |                14.698075 |      226.637904 |             79.0 |       1.3 |              1019.0 | 0.0 | 0.0 |    19.5 |                  79.0 |                  10.3 |                     14.698075 |           226.637904 | 18.8 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                22.704676 |                     22.704676 |                          0.0 |                      1.411269 |                           1.411269 |            17.936694 |                 17.936694 |    1001.063306 |   1.204916 | -2.9 |  20.57219 |   6.433141 |            172.244807 |                172.244807 |                 172.244807 |                 28.794151 |                      0.0 |   143.450656 |              0.0 |        143.450656 |            108.641252 |                 100.0 |                 48.85611 |                66.16469 | 18.1 |  0.0 | 4.081206 | 0.7 | 2.708353 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  5.575883 | 0.187867 | 0.247357 | 0.120559 |  0.0 |  5.575883 |  5.262727 | 0.060911 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 371.357334 | 249.219334 |  5.575883 |      0.0 | 0.187867 | 0.247357 | 0.120559 |  5.000154 |      0.0 | 0.162462 | 0.097435 | 0.002675 |   0.0 | 0.060911 |
    | 1997-17-08 | 20.0 | 21.0 |             11.1 |                14.628078 |      235.528503 |             75.9 |       1.9 |              1019.0 | 0.0 | 0.0 |    21.0 |                  75.9 |                  11.1 |                     14.628078 |           235.528503 | 18.8 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                24.909749 |                     24.909749 |                          0.0 |                      1.530181 |                           1.530181 |            18.906499 |                 18.906499 |    1000.093501 |   1.198338 | -2.9 | 17.884246 | -18.694526 |            179.001662 |                179.001662 |                 179.001662 |                 29.177702 |                      0.0 |    149.82396 |              0.0 |         149.82396 |             74.333488 |                 100.0 |                 48.85611 |               66.242207 | 18.1 |  0.0 | 5.175279 | 0.7 | 3.378596 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  6.018762 | 0.195117 | 0.335816 | 0.129819 |  0.0 |  6.018762 |  5.706521 | 0.066048 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.151861 | 257.261224 |  6.018762 |      0.0 | 0.195117 | 0.335816 | 0.129819 |  5.382685 |      0.0 | 0.170337 | 0.150384 | 0.003114 |   0.0 | 0.066048 |
    | 1997-18-08 | 20.0 | 21.6 |             11.0 |                14.557728 |      232.849428 |             70.0 |       2.0 |              1018.0 | 0.0 | 0.0 |    21.6 |                  70.0 |                  11.0 |                     14.557728 |           232.849428 | 18.8 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                25.842752 |                     25.842752 |                          0.0 |                      1.580055 |                           1.580055 |            18.089927 |                 18.089927 |     999.910073 |   1.195081 | -2.9 | 22.433069 |   4.998415 |            176.965566 |                176.965566 |                 176.965566 |                 31.281288 |                      0.0 |   145.684278 |              0.0 |        145.684278 |             70.616813 |                 100.0 |                 48.85611 |               66.320298 | 18.1 |  0.0 | 5.737858 | 0.7 | 3.779721 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  6.451575 | 0.201413 | 0.419628 | 0.137861 |  0.0 |  6.451575 |  6.184507 |  0.07158 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.253445 | 264.371025 |  6.451575 |      0.0 | 0.201413 | 0.419628 | 0.137861 |  5.790771 |      0.0 | 0.177909 | 0.212245 | 0.003582 |   0.0 |  0.07158 |
    | 1997-19-08 | 20.0 | 21.1 |              8.5 |                14.487046 |       199.71349 |             70.3 |       2.6 |              1019.0 | 0.0 | 0.0 |    21.1 |                  70.3 |                   8.5 |                     14.487046 |            199.71349 | 18.8 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            17.619414 |                 17.619414 |    1001.380586 |   1.198506 | -2.9 | 20.158326 |  -5.650046 |            151.782252 |                151.782252 |                 151.782252 |                 26.711296 |                      0.0 |   125.070956 |              0.0 |        125.070956 |             54.320626 |                 100.0 |                 48.85611 |               66.398945 | 18.1 |  0.0 | 5.866935 | 0.7 | 3.628366 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  6.883257 | 0.206979 | 0.498739 | 0.144971 |  0.0 |  6.883257 |  6.678562 | 0.077298 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 382.003492 | 271.108713 |  6.883257 |      0.0 | 0.206979 | 0.498739 | 0.144971 |  6.210098 |      0.0 | 0.185033 | 0.279355 | 0.004076 |   0.0 | 0.077298 |
    | 1997-20-08 | 20.0 | 22.4 |             11.3 |                14.416048 |      233.771902 |             69.0 |       3.6 |              1019.0 | 0.0 | 0.0 |    22.4 |                  69.0 |                  11.3 |                     14.416048 |           233.771902 | 18.8 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                27.134082 |                     27.134082 |                          0.0 |                      1.648679 |                           1.648679 |            18.722517 |                 18.722517 |    1000.277483 |   1.192743 | -2.9 | 20.950339 |  -8.697966 |            177.666646 |                177.666646 |                 177.666646 |                 31.116622 |                      0.0 |   146.550024 |              0.0 |        146.550024 |             39.231563 |                 100.0 |                 48.85611 |                66.47813 | 18.1 |  0.0 |  7.82923 | 0.7 | 4.624935 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  7.348961 | 0.212254 | 0.577799 | 0.151709 |  0.0 |  7.348961 |  7.189927 | 0.083217 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.801458 | 276.293054 |  7.348961 |      0.0 | 0.212254 | 0.577799 | 0.151709 |  6.643946 |      0.0 | 0.191696 | 0.349693 | 0.004593 |   0.0 | 0.083217 |
    | 1997-21-08 | 20.0 | 22.5 |             12.4 |                14.344752 |      246.121286 |             62.1 |       2.5 |              1017.0 | 0.0 | 0.0 |    22.5 |                  62.1 |                  12.4 |                     14.344752 |           246.121286 | 18.8 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                27.299387 |                     27.299387 |                          0.0 |                      1.657431 |                           1.657431 |             16.95292 |                  16.95292 |     1000.04708 |   1.190771 | -2.9 | 22.620153 |    0.72092 |            187.052177 |                187.052177 |                 187.052177 |                 38.297604 |                      0.0 |   148.754573 |              0.0 |        148.754573 |             56.493451 |                 100.0 |                 48.85611 |               66.557838 | 18.1 |  0.0 | 7.322689 | 0.7 | 4.756903 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  7.757566 | 0.216313 | 0.641218 | 0.156893 |  0.0 |  7.757566 |  7.701056 | 0.089133 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.180539 | 280.864161 |  7.757566 |      0.0 | 0.216313 | 0.641218 | 0.156893 |  7.077951 |      0.0 | 0.197812 | 0.420163 | 0.005129 |   0.0 | 0.089133 |
    | 1997-22-08 | 20.0 | 19.6 |              0.1 |                14.273175 |        90.58551 |             84.5 |       2.1 |              1015.0 | 0.0 | 0.0 |    19.6 |                  84.5 |                   0.1 |                     14.273175 |             90.58551 | 18.8 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                22.846186 |                     22.846186 |                          0.0 |                      1.418946 |                           1.418946 |            19.305027 |                 19.305027 |     995.694973 |   1.199129 | -2.9 | 21.577328 |  11.863971 |             68.844988 |                 68.844988 |                  68.844988 |                   7.02118 |                      0.0 |    61.823807 |              0.0 |         61.823807 |             67.254108 |                 100.0 |                 48.85611 |               66.638052 | 18.1 |  0.0 | 2.567328 | 0.7 | 1.361124 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  8.166722 | 0.219892 | 0.698934 | 0.161464 |  0.0 |  8.166722 |  8.197163 | 0.094875 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 369.416568 | 288.356024 |  8.166722 |      0.0 | 0.219892 | 0.698934 | 0.161464 |  7.500252 |      0.0 | 0.203306 | 0.487924 | 0.005681 |   0.0 | 0.094875 |
    | 1997-23-08 | 20.0 | 20.3 |              6.7 |                14.201333 |      171.944993 |             83.6 |       1.5 |              1015.0 | 0.0 | 0.0 |    20.3 |                  83.6 |                   6.7 |                     14.201333 |           171.944993 | 18.8 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                23.858503 |                     23.858503 |                          0.0 |                      1.473678 |                           1.473678 |            19.945708 |                 19.945708 |     995.054292 |   1.195981 | -2.9 | 17.325305 | -17.848171 |            130.678195 |                130.678195 |                 130.678195 |                 18.994889 |                      0.0 |   111.683306 |              0.0 |        111.683306 |             94.155751 |                 100.0 |                 48.85611 |               66.718758 | 18.1 |  0.0 | 3.374147 | 0.7 | 2.097454 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  8.984347 | 0.225758 | 0.797042 | 0.168956 |  0.0 |  8.984347 |  8.795147 | 0.101796 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.364739 | 294.182468 |  8.984347 |      0.0 | 0.225758 | 0.797042 | 0.168956 |  8.021581 |      0.0 | 0.208608 | 0.558706 | 0.006253 |   0.0 | 0.101796 |
    | 1997-24-08 | 20.0 | 24.1 |             10.4 |                14.129243 |      216.549813 |             76.5 |       2.2 |              1011.0 | 0.0 | 0.0 |    24.1 |                  76.5 |                  10.4 |                     14.129243 |           216.549813 | 18.8 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                30.066256 |                     30.066256 |                          0.0 |                       1.80287 |                            1.80287 |            23.000686 |                 23.000686 |     987.999314 |   1.174651 | -2.9 | 21.630378 | -14.817732 |            164.577858 |                164.577858 |                 164.577858 |                 21.629737 |                      0.0 |   142.948121 |              0.0 |        142.948121 |             64.197103 |                 100.0 |                 48.85611 |                66.79994 | 18.1 |  0.0 | 5.573024 | 0.7 | 3.649811 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 |  9.835059 | 0.230319 | 0.876255 | 0.174782 |  0.0 |  9.835059 |  9.544513 | 0.110469 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  396.28247 | 297.516241 |  9.835059 |      0.0 | 0.230319 | 0.876255 | 0.174782 |  8.689617 |      0.0 | 0.213877 | 0.634173 | 0.006846 |   0.0 | 0.110469 |
    | 1997-25-08 | 20.0 | 23.7 |              6.5 |                14.056918 |       166.91683 |             76.6 |       1.9 |              1008.0 | 0.0 | 0.0 |    23.7 |                  76.6 |                   6.5 |                     14.056918 |            166.91683 | 18.8 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                29.352596 |                     29.352596 |                          0.0 |                      1.765541 |                           1.765541 |            22.484088 |                 22.484088 |     985.515912 |   1.172942 | -2.9 | 25.062685 |   8.176109 |             126.85679 |                 126.85679 |                  126.85679 |                 16.222254 |                      0.0 |   110.634536 |              0.0 |        110.634536 |             74.333488 |                 100.0 |                 48.85611 |               66.881585 | 18.1 |  0.0 | 4.467668 | 0.7 | 2.895865 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 10.483053 | 0.232929 | 0.922693 | 0.178116 |  0.0 | 10.483053 | 10.322796 | 0.119477 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.206362 | 300.903585 | 10.483053 |      0.0 | 0.232929 | 0.922693 | 0.178116 |  9.390662 |      0.0 | 0.218678 | 0.706002 | 0.007454 |   0.0 | 0.119477 |
    | 1997-26-08 | 20.0 | 21.1 |              4.9 |                13.984374 |      145.985611 |             84.2 |       2.3 |              1006.0 | 0.0 | 0.0 |    21.1 |                  84.2 |                   4.9 |                     13.984374 |           145.985611 | 18.8 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            21.103195 |                 21.103195 |     984.896805 |   1.181557 | -2.9 | 21.872765 |   4.636593 |            110.949064 |                110.949064 |                 110.949064 |                 14.602883 |                      0.0 |    96.346181 |              0.0 |         96.346181 |             61.405925 |                 100.0 |                 48.85611 |               66.963678 | 18.1 |  0.0 | 3.569826 | 0.7 | 2.071449 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 11.394349 | 0.235581 | 0.970688 | 0.181504 |  0.0 | 11.394349 | 11.136926 |   0.1289 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.669769 | 304.150015 | 11.394349 |      0.0 | 0.235581 | 0.970688 | 0.181504 | 10.134751 |      0.0 | 0.222895 | 0.771208 | 0.008071 |   0.0 |   0.1289 |
    | 1997-27-08 | 20.0 | 20.9 |              6.6 |                13.911624 |       165.54389 |             85.8 |       2.1 |              1005.0 | 0.0 | 0.0 |    20.9 |                  85.8 |                   6.6 |                     13.911624 |            165.54389 | 18.8 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            21.241626 |                 21.241626 |     983.758374 |   1.181114 | -2.9 | 19.702227 |   -7.18664 |            125.813357 |                125.813357 |                 125.813357 |                 17.349121 |                      0.0 |   108.464235 |              0.0 |        108.464235 |             67.254108 |                 100.0 |                 48.85611 |               67.046207 | 18.1 |  0.0 |  3.56693 | 0.7 | 2.108859 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 13.250015 | 0.238123 | 1.017441 |  0.18475 |  0.0 | 13.250015 | 12.737917 |  0.14743 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 381.956408 | 305.450827 |      12.0 | 1.250015 | 0.238123 | 1.017441 |  0.18475 | 10.878137 | 0.792807 | 0.226675 |   0.8316 | 0.008698 |   0.0 |  0.14743 |
    | 1997-28-08 | 20.0 | 20.2 |              0.3 |                13.838682 |       87.693346 |             86.5 |       2.6 |              1000.0 | 0.0 | 0.0 |    20.2 |                  86.5 |                   0.3 |                     13.838682 |            87.693346 | 18.8 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                23.711531 |                     23.711531 |                          0.0 |                      1.465752 |                           1.465752 |            20.510475 |                 20.510475 |     979.489525 |   1.178323 | -2.9 | 20.936779 |   4.420674 |             66.646943 |                 66.646943 |                  66.646943 |                  6.818024 |                      0.0 |    59.828919 |              0.0 |         59.828919 |             54.320626 |                 100.0 |                 48.85611 |               67.129158 | 18.1 |  0.0 | 2.621228 | 0.7 |  1.33161 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 14.550827 | 0.239141 |  1.03638 | 0.186051 |  0.0 | 14.550827 |  14.56294 | 0.168553 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.635735 | 306.206818 |      12.0 | 2.550827 | 0.239141 |  1.03638 | 0.186051 | 11.401361 | 2.037971 | 0.229904 | 0.884372 | 0.009332 |   0.0 | 0.168553 |
    | 1997-29-08 | 20.0 | 16.1 |              0.1 |                13.765561 |       84.356885 |             89.6 |       2.6 |              1004.0 | 0.0 | 0.0 |    16.1 |                  89.6 |                   0.1 |                     13.765561 |            84.356885 | 18.8 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            16.420805 |                 16.420805 |     987.579195 |   1.201704 | -2.9 | 18.828425 |   16.37055 |             64.111233 |                 64.111233 |                  64.111233 |                  8.199212 |                      0.0 |    55.912021 |              0.0 |         55.912021 |             54.320626 |                 100.0 |                 48.85611 |               67.212519 | 18.1 |  0.0 | 2.027587 | 0.7 | 0.881773 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 15.306818 | 0.239733 | 1.047439 | 0.186807 |  0.0 | 15.306818 | 15.838636 | 0.183318 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 355.365185 | 306.644248 |      12.0 | 3.306818 | 0.239733 | 1.047439 | 0.186807 |  11.68056 | 2.988724 | 0.232478 | 0.926908 | 0.009966 |   0.0 | 0.183318 |
    | 1997-30-08 | 20.0 | 17.0 |              5.0 |                13.692273 |      142.309202 |             78.3 |       2.1 |              1011.0 | 0.0 | 0.0 |    17.0 |                  78.3 |                   5.0 |                     13.692273 |           142.309202 | 18.8 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                19.406929 |                     19.406929 |                          0.0 |                      1.230421 |                           1.230421 |            15.195625 |                 15.195625 |     995.804375 |   1.206937 | -2.9 | 13.278507 |  -22.32896 |            108.154994 |                108.154994 |                 108.154994 |                 21.775852 |                      0.0 |    86.379141 |              0.0 |         86.379141 |             67.254108 |                 100.0 |                 48.85611 |               67.296277 | 18.1 |  0.0 | 3.393474 | 0.7 | 1.913351 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 15.744248 | 0.240076 | 1.053857 | 0.187244 |  0.0 | 15.744248 | 16.593391 | 0.192053 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.794145 | 305.605473 |      12.0 | 3.744248 | 0.240076 | 1.053857 | 0.187244 | 11.829543 | 3.558462 | 0.234482 | 0.960303 |   0.0106 |   0.0 | 0.192053 |
    | 1997-31-08 | 20.0 | 18.2 |              3.8 |                13.618831 |      126.722078 |             78.5 |       2.1 |              1014.0 | 0.0 | 0.0 |    18.2 |                  78.5 |                   3.8 |                     13.618831 |           126.722078 | 18.8 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                20.933479 |                     20.933479 |                          0.0 |                      1.314615 |                           1.314615 |            16.432781 |                 16.432781 |     997.567219 |   1.204994 | -2.9 | 18.874047 |   4.044283 |              96.30878 |                  96.30878 |                   96.30878 |                 17.491881 |                      0.0 |    78.816899 |              0.0 |         78.816899 |             67.254108 |                 100.0 |                 48.85611 |               67.380422 | 18.1 |  0.0 | 3.321963 | 0.7 | 1.882123 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 18.1 | 14.705473 | 0.239262 | 1.038639 | 0.186205 |  0.0 | 14.705473 | 16.209861 | 0.187614 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 367.849862 | 305.653771 |      12.0 | 2.705473 | 0.239262 | 1.038639 | 0.186205 | 11.909042 | 3.070354 | 0.235874 | 0.983359 | 0.011231 |   0.0 | 0.187614 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

>>> inputs.nied.series = nied

.. _lland_v3_wasser_daily:

water
-----

L-Land defines three types of water areas, |WASSER|, |FLUSS| and |SEE|.  |lland_v3|
calculates their evaporation via the Penman equation based on daily input data.
Besides that, there are no structural differences to |lland_v1|, so please read the
detailed discussions in integration test :ref:`lland_v1_wasser` for better
understanding the following test results for water type |WASSER|:

.. integration-test::

    >>> lnk(WASSER)
    >>> negq(False)
    >>> surfaceresistance.wasser_aug = 0.0
    >>> cropheight.wasser_aug = 0.05
    >>> albedo.wasser_aug = 0.7
    >>> conditions_wasser = test(
    ...     "lland_v3_wasser_daily",
    ...     axis1=(fluxes.nkor, fluxes.evi, fluxes.qah),
    ...     get_conditions="1997-08-03")
    |       date | nied | teml | sunshineduration | possiblesunshineduration | globalradiation | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration | dailypossiblesunshineduration | dailyglobalradiation |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair |   g |  tz |  wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |     evpo |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |       qah |       qa | inzp | wats | waes | esnow | taus | ebdn | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |                15.690785 |       190.25149 |             86.2 |       2.4 |              1007.0 | 0.0 | 0.0 |    16.1 |                  86.2 |                   6.3 |                     15.690785 |            190.25149 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            15.797694 |                 15.797694 |     991.202306 |   1.205601 | 0.0 | nan | 0.0 |             57.075447 |                 57.075447 |                  57.075447 |                 21.732733 |                      0.0 |    35.342714 |              0.0 |         35.342714 |             88.752361 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077197 | 1.077197 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 11.424803 | 0.132232 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.132232 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |                15.628821 |      127.660782 |             85.8 |       1.6 |              1013.0 | 0.0 | 0.0 |    17.3 |                  85.8 |                   1.7 |                     15.628821 |           127.660782 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                19.779139 |                     19.779139 |                          0.0 |                      1.251029 |                           1.251029 |            16.970501 |                 16.970501 |     996.029499 |   1.207284 | 0.0 | nan | 0.0 |             38.298235 |                 38.298235 |                  38.298235 |                 11.219797 |                      0.0 |    27.078438 |              0.0 |         27.078438 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.86545 |    0.094 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |                15.566134 |      164.401298 |             82.4 |       1.5 |              1016.0 | 0.0 | 0.0 |    18.9 |                  82.4 |                   4.5 |                     15.566134 |           164.401298 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                21.871576 |                     21.871576 |                          0.0 |                      1.365939 |                           1.365939 |            18.022179 |                 18.022179 |     997.977821 |   1.203775 | 0.0 | nan | 0.0 |             49.320389 |                 49.320389 |                  49.320389 |                 16.044181 |                      0.0 |    33.276209 |              0.0 |         33.276209 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.092365 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |                15.502754 |      269.020502 |             77.6 |       2.8 |              1017.0 | 0.0 | 0.0 |    17.8 |                  77.6 |                  12.4 |                     15.502754 |           269.020502 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                20.413369 |                     20.413369 |                          0.0 |                      1.286025 |                           1.286025 |            15.840774 |                 15.840774 |    1001.159226 |    1.21051 | 0.0 | nan | 0.0 |              80.70615 |                  80.70615 |                   80.70615 |                 35.980407 |                      0.0 |    44.725744 |              0.0 |         44.725744 |             76.073452 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.565894 |    0.094 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |                15.438712 |      287.848667 |             74.9 |       3.0 |              1018.0 | 0.0 | 0.0 |    18.5 |                  74.9 |                  13.9 |                     15.438712 |           287.848667 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                21.331125 |                     21.331125 |                          0.0 |                      1.336407 |                           1.336407 |            15.977013 |                 15.977013 |    1002.022987 |   1.208738 | 0.0 | nan | 0.0 |               86.3546 |                   86.3546 |                    86.3546 |                 39.523937 |                      0.0 |    46.830664 |              0.0 |         46.830664 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.729733 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |                15.374034 |      274.674263 |             77.9 |       3.5 |              1018.0 | 0.0 | 0.0 |    19.9 |                  77.9 |                  13.0 |                     15.374034 |           274.674263 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                23.275348 |                     23.275348 |                          0.0 |                      1.442189 |                           1.442189 |            18.131496 |                 18.131496 |     999.868504 |   1.201995 | 0.0 | nan | 0.0 |             82.402279 |                 82.402279 |                  82.402279 |                 32.936813 |                      0.0 |    49.465466 |              0.0 |         49.465466 |             60.858762 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826087 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |                 15.30875 |      284.047892 |             73.5 |       3.8 |              1019.0 | 0.0 | 0.0 |    20.8 |                  73.5 |                  13.8 |                      15.30875 |           284.047892 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            18.084931 |                 18.084931 |    1000.915069 |   1.199521 | 0.0 | nan | 0.0 |             85.214368 |                 85.214368 |                  85.214368 |                  35.31351 |                      0.0 |    49.900858 |              0.0 |         49.900858 |             56.054123 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.041956 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |                15.242885 |      262.971263 |             80.1 |       3.3 |              1019.0 | 0.0 | 0.0 |    20.9 |                  80.1 |                  12.3 |                     15.242885 |           262.971263 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            19.830469 |                 19.830469 |     999.169531 |   1.198331 | 0.0 | nan | 0.0 |             78.891379 |                 78.891379 |                  78.891379 |                 28.392598 |                      0.0 |    50.498781 |              0.0 |         50.498781 |             64.547172 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.800053 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |                15.176467 |      272.256414 |             72.9 |       1.8 |              1019.0 | 0.0 | 0.0 |    22.3 |                  72.9 |                  13.1 |                     15.176467 |           272.256414 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 26.96965 |                      26.96965 |                          0.0 |                      1.639967 |                           1.639967 |            19.660875 |                 19.660875 |     999.339125 |   1.192729 | 0.0 | nan | 0.0 |             81.676924 |                 81.676924 |                  81.676924 |                 31.114889 |                      0.0 |    50.562035 |              0.0 |         50.562035 |            118.336481 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.82186 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |                15.109521 |      267.019218 |             68.5 |       1.9 |              1020.0 | 0.0 | 0.0 |    23.0 |                  68.5 |                  12.8 |                     15.109521 |           267.019218 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                28.139149 |                     28.139149 |                          0.0 |                      1.701779 |                           1.701779 |            19.275317 |                 19.275317 |    1000.724683 |   1.191257 | 0.0 | nan | 0.0 |             80.105765 |                 80.105765 |                  80.105765 |                 32.010643 |                      0.0 |    48.095123 |              0.0 |         48.095123 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.891469 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |                15.042071 |      269.626547 |             68.6 |       3.0 |              1021.0 | 0.0 | 0.0 |    23.1 |                  68.6 |                  13.1 |                     15.042071 |           269.626547 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                28.309776 |                     28.309776 |                          0.0 |                      1.710768 |                           1.710768 |            19.420506 |                 19.420506 |    1001.579494 |   1.191966 | 0.0 | nan | 0.0 |             80.887964 |                 80.887964 |                  80.887964 |                 32.400014 |                      0.0 |     48.48795 |              0.0 |          48.48795 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.106575 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |                14.974141 |      270.878291 |             66.0 |       2.5 |              1022.0 | 0.0 | 0.0 |    23.2 |                  66.0 |                  13.3 |                     14.974141 |           270.878291 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |             18.79766 |                  18.79766 |     1003.20234 |   1.193016 | 0.0 | nan | 0.0 |             81.263487 |                 81.263487 |                  81.263487 |                 34.565783 |                      0.0 |    46.697704 |              0.0 |         46.697704 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.036929 |    0.094 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |                14.905755 |      261.697164 |             69.0 |       1.6 |              1019.0 | 0.0 | 0.0 |    23.2 |                  69.0 |                  12.7 |                     14.905755 |           261.697164 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |              19.6521 |                   19.6521 |       999.3479 |    1.18911 | 0.0 | nan | 0.0 |             78.509149 |                 78.509149 |                  78.509149 |                  31.34697 |                      0.0 |    47.162179 |              0.0 |         47.162179 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.808559 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |                14.836934 |      227.962116 |             75.6 |       2.7 |              1015.0 | 0.0 | 0.0 |    21.7 |                  75.6 |                  10.2 |                     14.836934 |           227.962116 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 26.00118 |                      26.00118 |                          0.0 |                        1.5885 |                             1.5885 |            19.656892 |                 19.656892 |     995.343108 |   1.190432 | 0.0 | nan | 0.0 |             68.388635 |                 68.388635 |                  68.388635 |                 25.918437 |                      0.0 |    42.470198 |              0.0 |         42.470198 |             78.890988 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672736 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |                  14.7677 |      216.356321 |             81.4 |       3.2 |              1016.0 | 0.0 | 0.0 |    20.8 |                  81.4 |                   9.4 |                       14.7677 |           216.356321 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            20.028753 |                 20.028753 |     995.971247 |   1.195095 | 0.0 | nan | 0.0 |             64.906896 |                 64.906896 |                  64.906896 |                 23.397902 |                      0.0 |    41.508994 |              0.0 |         41.508994 |             66.564271 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.529542 | 1.529542 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.008458 | 0.011672 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.011672 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |                14.698075 |      226.637904 |             79.0 |       1.3 |              1019.0 | 0.0 | 0.0 |    19.5 |                  79.0 |                  10.3 |                     14.698075 |           226.637904 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                22.704676 |                     22.704676 |                          0.0 |                      1.411269 |                           1.411269 |            17.936694 |                 17.936694 |    1001.063306 |   1.204916 | 0.0 | nan | 0.0 |             67.991371 |                 67.991371 |                  67.991371 |                 28.794151 |                      0.0 |    39.197221 |              0.0 |         39.197221 |            163.850513 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.283901 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |                14.628078 |      235.528503 |             75.9 |       1.9 |              1019.0 | 0.0 | 0.0 |    21.0 |                  75.9 |                  11.1 |                     14.628078 |           235.528503 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                24.909749 |                     24.909749 |                          0.0 |                      1.530181 |                           1.530181 |            18.906499 |                 18.906499 |    1000.093501 |   1.198338 | 0.0 | nan | 0.0 |             70.658551 |                 70.658551 |                  70.658551 |                 29.177702 |                      0.0 |    41.480849 |              0.0 |         41.480849 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.510743 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |                14.557728 |      232.849428 |             70.0 |       2.0 |              1018.0 | 0.0 | 0.0 |    21.6 |                  70.0 |                  11.0 |                     14.557728 |           232.849428 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                25.842752 |                     25.842752 |                          0.0 |                      1.580055 |                           1.580055 |            18.089927 |                 18.089927 |     999.910073 |   1.195081 | 0.0 | nan | 0.0 |             69.854829 |                 69.854829 |                  69.854829 |                 31.281288 |                      0.0 |    38.573541 |              0.0 |         38.573541 |            106.502833 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.594572 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |                14.487046 |       199.71349 |             70.3 |       2.6 |              1019.0 | 0.0 | 0.0 |    21.1 |                  70.3 |                   8.5 |                     14.487046 |            199.71349 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            17.619414 |                 17.619414 |    1001.380586 |   1.198506 | 0.0 | nan | 0.0 |             59.914047 |                 59.914047 |                  59.914047 |                 26.711296 |                      0.0 |    33.202751 |              0.0 |         33.202751 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.541567 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |                14.416048 |      233.771902 |             69.0 |       3.6 |              1019.0 | 0.0 | 0.0 |    22.4 |                  69.0 |                  11.3 |                     14.416048 |           233.771902 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                27.134082 |                     27.134082 |                          0.0 |                      1.648679 |                           1.648679 |            18.722517 |                 18.722517 |    1000.277483 |   1.192743 | 0.0 | nan | 0.0 |             70.131571 |                 70.131571 |                  70.131571 |                 31.116622 |                      0.0 |    39.014949 |              0.0 |         39.014949 |             59.168241 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.939383 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |                14.344752 |      246.121286 |             62.1 |       2.5 |              1017.0 | 0.0 | 0.0 |    22.5 |                  62.1 |                  12.4 |                     14.344752 |           246.121286 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                27.299387 |                     27.299387 |                          0.0 |                      1.657431 |                           1.657431 |             16.95292 |                  16.95292 |     1000.04708 |   1.190771 | 0.0 | nan | 0.0 |             73.836386 |                 73.836386 |                  73.836386 |                 38.297604 |                      0.0 |    35.538782 |              0.0 |         35.538782 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826811 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |                14.273175 |        90.58551 |             84.5 |       2.1 |              1015.0 | 0.0 | 0.0 |    19.6 |                  84.5 |                   0.1 |                     14.273175 |             90.58551 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                22.846186 |                     22.846186 |                          0.0 |                      1.418946 |                           1.418946 |            19.305027 |                 19.305027 |     995.694973 |   1.199129 | 0.0 | nan | 0.0 |             27.175653 |                 27.175653 |                  27.175653 |                   7.02118 |                      0.0 |    20.154473 |              0.0 |         20.154473 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.806188 | 0.806188 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.073812 | 0.012428 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.012428 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |                14.201333 |      171.944993 |             83.6 |       1.5 |              1015.0 | 0.0 | 0.0 |    20.3 |                  83.6 |                   6.7 |                     14.201333 |           171.944993 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                23.858503 |                     23.858503 |                          0.0 |                      1.473678 |                           1.473678 |            19.945708 |                 19.945708 |     995.054292 |   1.195981 | 0.0 | nan | 0.0 |             51.583498 |                 51.583498 |                  51.583498 |                 18.994889 |                      0.0 |    32.588609 |              0.0 |         32.588609 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.08421 |    0.282 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |                14.129243 |      216.549813 |             76.5 |       2.2 |              1011.0 | 0.0 | 0.0 |    24.1 |                  76.5 |                  10.4 |                     14.129243 |           216.549813 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                30.066256 |                     30.066256 |                          0.0 |                       1.80287 |                            1.80287 |            23.000686 |                 23.000686 |     987.999314 |   1.174651 | 0.0 | nan | 0.0 |             64.964944 |                 64.964944 |                  64.964944 |                 21.629737 |                      0.0 |    43.335207 |              0.0 |         43.335207 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672232 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |                14.056918 |       166.91683 |             76.6 |       1.9 |              1008.0 | 0.0 | 0.0 |    23.7 |                  76.6 |                   6.5 |                     14.056918 |            166.91683 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                29.352596 |                     29.352596 |                          0.0 |                      1.765541 |                           1.765541 |            22.484088 |                 22.484088 |     985.515912 |   1.172942 | 0.0 | nan | 0.0 |             50.075049 |                 50.075049 |                  50.075049 |                 16.222254 |                      0.0 |    33.852795 |              0.0 |         33.852795 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.374102 |    0.094 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |                13.984374 |      145.985611 |             84.2 |       2.3 |              1006.0 | 0.0 | 0.0 |    21.1 |                  84.2 |                   4.9 |                     13.984374 |           145.985611 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            21.103195 |                 21.103195 |     984.896805 |   1.181557 | 0.0 | nan | 0.0 |             43.795683 |                 43.795683 |                  43.795683 |                 14.602883 |                      0.0 |      29.1928 |              0.0 |           29.1928 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077096 | 1.077096 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  8.792904 |  0.10177 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.10177 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |                13.911624 |       165.54389 |             85.8 |       2.1 |              1005.0 | 0.0 | 0.0 |    20.9 |                  85.8 |                   6.6 |                     13.911624 |            165.54389 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            21.241626 |                 21.241626 |     983.758374 |   1.181114 | 0.0 | nan | 0.0 |             49.663167 |                 49.663167 |                  49.663167 |                 17.349121 |                      0.0 |    32.314046 |              0.0 |         32.314046 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.09756 |  1.09756 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   5.01244 | 0.058014 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.058014 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |                13.838682 |       87.693346 |             86.5 |       2.6 |              1000.0 | 0.0 | 0.0 |    20.2 |                  86.5 |                   0.3 |                     13.838682 |            87.693346 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                23.711531 |                     23.711531 |                          0.0 |                      1.465752 |                           1.465752 |            20.510475 |                 20.510475 |     979.489525 |   1.178323 | 0.0 | nan | 0.0 |             26.308004 |                 26.308004 |                  26.308004 |                  6.818024 |                      0.0 |     19.48998 |              0.0 |          19.48998 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.794859 | 0.794859 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 13.305141 | 0.153995 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.153995 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |                13.765561 |       84.356885 |             89.6 |       2.6 |              1004.0 | 0.0 | 0.0 |    16.1 |                  89.6 |                   0.1 |                     13.765561 |            84.356885 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            16.420805 |                 16.420805 |     987.579195 |   1.201704 | 0.0 | nan | 0.0 |             25.307066 |                 25.307066 |                  25.307066 |                  8.199212 |                      0.0 |    17.107854 |              0.0 |         17.107854 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.607776 | 0.607776 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  8.040224 | 0.093058 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.093058 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |                13.692273 |      142.309202 |             78.3 |       2.1 |              1011.0 | 0.0 | 0.0 |    17.0 |                  78.3 |                   5.0 |                     13.692273 |           142.309202 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                19.406929 |                     19.406929 |                          0.0 |                      1.230421 |                           1.230421 |            15.195625 |                 15.195625 |     995.804375 |   1.206937 | 0.0 | nan | 0.0 |             42.692761 |                 42.692761 |                  42.692761 |                 21.775852 |                      0.0 |    20.916908 |              0.0 |         20.916908 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.90065 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |                13.618831 |      126.722078 |             78.5 |       2.1 |              1014.0 | 0.0 | 0.0 |    18.2 |                  78.5 |                   3.8 |                     13.618831 |           126.722078 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                20.933479 |                     20.933479 |                          0.0 |                      1.314615 |                           1.314615 |            16.432781 |                 16.432781 |     997.567219 |   1.204994 | 0.0 | nan | 0.0 |             38.016624 |                 38.016624 |                  38.016624 |                 17.491881 |                      0.0 |    20.524743 |              0.0 |         20.524743 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.912078 |      0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_v3_wasser_routing_daily:

water (routing)
---------------

The following calculation shows the possibility of subtracting evaporation from
inflowing runoff, discussed in the integration test :ref:`lland_v1_wasser_routing` of
|lland_v1|:

.. integration-test::

    >>> inlet.sequences.sim.series = 0.02
    >>> test("lland_v3_wasser_routing_daily",
    ...      axis1=(fluxes.nkor, fluxes.evi, fluxes.qah))
    |       date | nied | teml | sunshineduration | possiblesunshineduration | globalradiation | relativehumidity | windspeed | atmosphericpressure |   qz |   qzh | temltag | dailyrelativehumidity | dailysunshineduration | dailypossiblesunshineduration | dailyglobalradiation |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair |   g |  tz |  wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |     evpo |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |       qah |       qa | inzp | wats | waes | esnow | taus | ebdn | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |                15.690785 |       190.25149 |             86.2 |       2.4 |              1007.0 | 0.02 | 1.728 |    16.1 |                  86.2 |                   6.3 |                     15.690785 |            190.25149 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            15.797694 |                 15.797694 |     991.202306 |   1.205601 | 0.0 | nan | 0.0 |             57.075447 |                 57.075447 |                  57.075447 |                 21.732733 |                      0.0 |    35.342714 |              0.0 |         35.342714 |             88.752361 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077197 | 1.077197 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 13.152803 | 0.152232 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.152232 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |                15.628821 |      127.660782 |             85.8 |       1.6 |              1013.0 | 0.02 | 1.728 |    17.3 |                  85.8 |                   1.7 |                     15.628821 |           127.660782 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                19.779139 |                     19.779139 |                          0.0 |                      1.251029 |                           1.251029 |            16.970501 |                 16.970501 |     996.029499 |   1.207284 | 0.0 | nan | 0.0 |             38.298235 |                 38.298235 |                  38.298235 |                 11.219797 |                      0.0 |    27.078438 |              0.0 |         27.078438 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.86545 |  0.86545 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.95655 | 0.011071 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.011071 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |                15.566134 |      164.401298 |             82.4 |       1.5 |              1016.0 | 0.02 | 1.728 |    18.9 |                  82.4 |                   4.5 |                     15.566134 |           164.401298 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                21.871576 |                     21.871576 |                          0.0 |                      1.365939 |                           1.365939 |            18.022179 |                 18.022179 |     997.977821 |   1.203775 | 0.0 | nan | 0.0 |             49.320389 |                 49.320389 |                  49.320389 |                 16.044181 |                      0.0 |    33.276209 |              0.0 |         33.276209 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.092365 | 1.092365 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.635635 | 0.007357 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.007357 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |                15.502754 |      269.020502 |             77.6 |       2.8 |              1017.0 | 0.02 | 1.728 |    17.8 |                  77.6 |                  12.4 |                     15.502754 |           269.020502 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                20.413369 |                     20.413369 |                          0.0 |                      1.286025 |                           1.286025 |            15.840774 |                 15.840774 |    1001.159226 |    1.21051 | 0.0 | nan | 0.0 |              80.70615 |                  80.70615 |                   80.70615 |                 35.980407 |                      0.0 |    44.725744 |              0.0 |         44.725744 |             76.073452 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.565894 | 1.565894 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.256106 | 0.002964 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.002964 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |                15.438712 |      287.848667 |             74.9 |       3.0 |              1018.0 | 0.02 | 1.728 |    18.5 |                  74.9 |                  13.9 |                     15.438712 |           287.848667 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                21.331125 |                     21.331125 |                          0.0 |                      1.336407 |                           1.336407 |            15.977013 |                 15.977013 |    1002.022987 |   1.208738 | 0.0 | nan | 0.0 |               86.3546 |                   86.3546 |                    86.3546 |                 39.523937 |                      0.0 |    46.830664 |              0.0 |         46.830664 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.729733 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |                15.374034 |      274.674263 |             77.9 |       3.5 |              1018.0 | 0.02 | 1.728 |    19.9 |                  77.9 |                  13.0 |                     15.374034 |           274.674263 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                23.275348 |                     23.275348 |                          0.0 |                      1.442189 |                           1.442189 |            18.131496 |                 18.131496 |     999.868504 |   1.201995 | 0.0 | nan | 0.0 |             82.402279 |                 82.402279 |                  82.402279 |                 32.936813 |                      0.0 |    49.465466 |              0.0 |         49.465466 |             60.858762 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826087 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |                 15.30875 |      284.047892 |             73.5 |       3.8 |              1019.0 | 0.02 | 1.728 |    20.8 |                  73.5 |                  13.8 |                      15.30875 |           284.047892 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            18.084931 |                 18.084931 |    1000.915069 |   1.199521 | 0.0 | nan | 0.0 |             85.214368 |                 85.214368 |                  85.214368 |                  35.31351 |                      0.0 |    49.900858 |              0.0 |         49.900858 |             56.054123 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.041956 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |                15.242885 |      262.971263 |             80.1 |       3.3 |              1019.0 | 0.02 | 1.728 |    20.9 |                  80.1 |                  12.3 |                     15.242885 |           262.971263 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            19.830469 |                 19.830469 |     999.169531 |   1.198331 | 0.0 | nan | 0.0 |             78.891379 |                 78.891379 |                  78.891379 |                 28.392598 |                      0.0 |    50.498781 |              0.0 |         50.498781 |             64.547172 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.800053 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |                15.176467 |      272.256414 |             72.9 |       1.8 |              1019.0 | 0.02 | 1.728 |    22.3 |                  72.9 |                  13.1 |                     15.176467 |           272.256414 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 26.96965 |                      26.96965 |                          0.0 |                      1.639967 |                           1.639967 |            19.660875 |                 19.660875 |     999.339125 |   1.192729 | 0.0 | nan | 0.0 |             81.676924 |                 81.676924 |                  81.676924 |                 31.114889 |                      0.0 |    50.562035 |              0.0 |         50.562035 |            118.336481 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.82186 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |                15.109521 |      267.019218 |             68.5 |       1.9 |              1020.0 | 0.02 | 1.728 |    23.0 |                  68.5 |                  12.8 |                     15.109521 |           267.019218 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                28.139149 |                     28.139149 |                          0.0 |                      1.701779 |                           1.701779 |            19.275317 |                 19.275317 |    1000.724683 |   1.191257 | 0.0 | nan | 0.0 |             80.105765 |                 80.105765 |                  80.105765 |                 32.010643 |                      0.0 |    48.095123 |              0.0 |         48.095123 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.891469 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |                15.042071 |      269.626547 |             68.6 |       3.0 |              1021.0 | 0.02 | 1.728 |    23.1 |                  68.6 |                  13.1 |                     15.042071 |           269.626547 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                28.309776 |                     28.309776 |                          0.0 |                      1.710768 |                           1.710768 |            19.420506 |                 19.420506 |    1001.579494 |   1.191966 | 0.0 | nan | 0.0 |             80.887964 |                 80.887964 |                  80.887964 |                 32.400014 |                      0.0 |     48.48795 |              0.0 |          48.48795 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.106575 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |                14.974141 |      270.878291 |             66.0 |       2.5 |              1022.0 | 0.02 | 1.728 |    23.2 |                  66.0 |                  13.3 |                     14.974141 |           270.878291 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |             18.79766 |                  18.79766 |     1003.20234 |   1.193016 | 0.0 | nan | 0.0 |             81.263487 |                 81.263487 |                  81.263487 |                 34.565783 |                      0.0 |    46.697704 |              0.0 |         46.697704 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.036929 |    1.822 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |                14.905755 |      261.697164 |             69.0 |       1.6 |              1019.0 | 0.02 | 1.728 |    23.2 |                  69.0 |                  12.7 |                     14.905755 |           261.697164 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |              19.6521 |                   19.6521 |       999.3479 |    1.18911 | 0.0 | nan | 0.0 |             78.509149 |                 78.509149 |                  78.509149 |                  31.34697 |                      0.0 |    47.162179 |              0.0 |         47.162179 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.808559 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |                14.836934 |      227.962116 |             75.6 |       2.7 |              1015.0 | 0.02 | 1.728 |    21.7 |                  75.6 |                  10.2 |                     14.836934 |           227.962116 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 26.00118 |                      26.00118 |                          0.0 |                        1.5885 |                             1.5885 |            19.656892 |                 19.656892 |     995.343108 |   1.190432 | 0.0 | nan | 0.0 |             68.388635 |                 68.388635 |                  68.388635 |                 25.918437 |                      0.0 |    42.470198 |              0.0 |         42.470198 |             78.890988 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672736 | 1.672736 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.055264 |  0.00064 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |  0.00064 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |                  14.7677 |      216.356321 |             81.4 |       3.2 |              1016.0 | 0.02 | 1.728 |    20.8 |                  81.4 |                   9.4 |                       14.7677 |           216.356321 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            20.028753 |                 20.028753 |     995.971247 |   1.195095 | 0.0 | nan | 0.0 |             64.906896 |                 64.906896 |                  64.906896 |                 23.397902 |                      0.0 |    41.508994 |              0.0 |         41.508994 |             66.564271 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.529542 | 1.529542 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.736458 | 0.031672 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.031672 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |                14.698075 |      226.637904 |             79.0 |       1.3 |              1019.0 | 0.02 | 1.728 |    19.5 |                  79.0 |                  10.3 |                     14.698075 |           226.637904 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                22.704676 |                     22.704676 |                          0.0 |                      1.411269 |                           1.411269 |            17.936694 |                 17.936694 |    1001.063306 |   1.204916 | 0.0 | nan | 0.0 |             67.991371 |                 67.991371 |                  67.991371 |                 28.794151 |                      0.0 |    39.197221 |              0.0 |         39.197221 |            163.850513 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.283901 | 1.283901 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.444099 |  0.00514 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |  0.00514 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |                14.628078 |      235.528503 |             75.9 |       1.9 |              1019.0 | 0.02 | 1.728 |    21.0 |                  75.9 |                  11.1 |                     14.628078 |           235.528503 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                24.909749 |                     24.909749 |                          0.0 |                      1.530181 |                           1.530181 |            18.906499 |                 18.906499 |    1000.093501 |   1.198338 | 0.0 | nan | 0.0 |             70.658551 |                 70.658551 |                  70.658551 |                 29.177702 |                      0.0 |    41.480849 |              0.0 |         41.480849 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.510743 | 1.510743 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.217257 | 0.002515 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.002515 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |                14.557728 |      232.849428 |             70.0 |       2.0 |              1018.0 | 0.02 | 1.728 |    21.6 |                  70.0 |                  11.0 |                     14.557728 |           232.849428 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                25.842752 |                     25.842752 |                          0.0 |                      1.580055 |                           1.580055 |            18.089927 |                 18.089927 |     999.910073 |   1.195081 | 0.0 | nan | 0.0 |             69.854829 |                 69.854829 |                  69.854829 |                 31.281288 |                      0.0 |    38.573541 |              0.0 |         38.573541 |            106.502833 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.594572 | 1.594572 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.133428 | 0.001544 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.001544 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |                14.487046 |       199.71349 |             70.3 |       2.6 |              1019.0 | 0.02 | 1.728 |    21.1 |                  70.3 |                   8.5 |                     14.487046 |            199.71349 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            17.619414 |                 17.619414 |    1001.380586 |   1.198506 | 0.0 | nan | 0.0 |             59.914047 |                 59.914047 |                  59.914047 |                 26.711296 |                      0.0 |    33.202751 |              0.0 |         33.202751 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.541567 | 1.541567 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.186433 | 0.002158 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.002158 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |                14.416048 |      233.771902 |             69.0 |       3.6 |              1019.0 | 0.02 | 1.728 |    22.4 |                  69.0 |                  11.3 |                     14.416048 |           233.771902 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                27.134082 |                     27.134082 |                          0.0 |                      1.648679 |                           1.648679 |            18.722517 |                 18.722517 |    1000.277483 |   1.192743 | 0.0 | nan | 0.0 |             70.131571 |                 70.131571 |                  70.131571 |                 31.116622 |                      0.0 |    39.014949 |              0.0 |         39.014949 |             59.168241 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.939383 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |                14.344752 |      246.121286 |             62.1 |       2.5 |              1017.0 | 0.02 | 1.728 |    22.5 |                  62.1 |                  12.4 |                     14.344752 |           246.121286 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                27.299387 |                     27.299387 |                          0.0 |                      1.657431 |                           1.657431 |             16.95292 |                  16.95292 |     1000.04708 |   1.190771 | 0.0 | nan | 0.0 |             73.836386 |                 73.836386 |                  73.836386 |                 38.297604 |                      0.0 |    35.538782 |              0.0 |         35.538782 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826811 |    1.728 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |       0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |                14.273175 |        90.58551 |             84.5 |       2.1 |              1015.0 | 0.02 | 1.728 |    19.6 |                  84.5 |                   0.1 |                     14.273175 |             90.58551 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                22.846186 |                     22.846186 |                          0.0 |                      1.418946 |                           1.418946 |            19.305027 |                 19.305027 |     995.694973 |   1.199129 | 0.0 | nan | 0.0 |             27.175653 |                 27.175653 |                  27.175653 |                   7.02118 |                      0.0 |    20.154473 |              0.0 |         20.154473 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.806188 | 0.806188 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  2.801812 | 0.032428 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.032428 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |                14.201333 |      171.944993 |             83.6 |       1.5 |              1015.0 | 0.02 | 1.728 |    20.3 |                  83.6 |                   6.7 |                     14.201333 |           171.944993 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                23.858503 |                     23.858503 |                          0.0 |                      1.473678 |                           1.473678 |            19.945708 |                 19.945708 |     995.054292 |   1.195981 | 0.0 | nan | 0.0 |             51.583498 |                 51.583498 |                  51.583498 |                 18.994889 |                      0.0 |    32.588609 |              0.0 |         32.588609 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.08421 |  1.08421 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.92579 | 0.010715 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.010715 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |                14.129243 |      216.549813 |             76.5 |       2.2 |              1011.0 | 0.02 | 1.728 |    24.1 |                  76.5 |                  10.4 |                     14.129243 |           216.549813 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                30.066256 |                     30.066256 |                          0.0 |                       1.80287 |                            1.80287 |            23.000686 |                 23.000686 |     987.999314 |   1.174651 | 0.0 | nan | 0.0 |             64.964944 |                 64.964944 |                  64.964944 |                 21.629737 |                      0.0 |    43.335207 |              0.0 |         43.335207 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672232 | 1.672232 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.055768 | 0.000645 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.000645 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |                14.056918 |       166.91683 |             76.6 |       1.9 |              1008.0 | 0.02 | 1.728 |    23.7 |                  76.6 |                   6.5 |                     14.056918 |            166.91683 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                29.352596 |                     29.352596 |                          0.0 |                      1.765541 |                           1.765541 |            22.484088 |                 22.484088 |     985.515912 |   1.172942 | 0.0 | nan | 0.0 |             50.075049 |                 50.075049 |                  50.075049 |                 16.222254 |                      0.0 |    33.852795 |              0.0 |         33.852795 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.374102 | 1.374102 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.447898 | 0.005184 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.005184 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |                13.984374 |      145.985611 |             84.2 |       2.3 |              1006.0 | 0.02 | 1.728 |    21.1 |                  84.2 |                   4.9 |                     13.984374 |           145.985611 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            21.103195 |                 21.103195 |     984.896805 |   1.181557 | 0.0 | nan | 0.0 |             43.795683 |                 43.795683 |                  43.795683 |                 14.602883 |                      0.0 |      29.1928 |              0.0 |           29.1928 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077096 | 1.077096 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 10.520904 |  0.12177 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 |  0.12177 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |                13.911624 |       165.54389 |             85.8 |       2.1 |              1005.0 | 0.02 | 1.728 |    20.9 |                  85.8 |                   6.6 |                     13.911624 |            165.54389 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            21.241626 |                 21.241626 |     983.758374 |   1.181114 | 0.0 | nan | 0.0 |             49.663167 |                 49.663167 |                  49.663167 |                 17.349121 |                      0.0 |    32.314046 |              0.0 |         32.314046 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.09756 |  1.09756 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   6.74044 | 0.078014 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.078014 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |                13.838682 |       87.693346 |             86.5 |       2.6 |              1000.0 | 0.02 | 1.728 |    20.2 |                  86.5 |                   0.3 |                     13.838682 |            87.693346 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                23.711531 |                     23.711531 |                          0.0 |                      1.465752 |                           1.465752 |            20.510475 |                 20.510475 |     979.489525 |   1.178323 | 0.0 | nan | 0.0 |             26.308004 |                 26.308004 |                  26.308004 |                  6.818024 |                      0.0 |     19.48998 |              0.0 |          19.48998 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.794859 | 0.794859 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 15.033141 | 0.173995 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.173995 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |                13.765561 |       84.356885 |             89.6 |       2.6 |              1004.0 | 0.02 | 1.728 |    16.1 |                  89.6 |                   0.1 |                     13.765561 |            84.356885 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            16.420805 |                 16.420805 |     987.579195 |   1.201704 | 0.0 | nan | 0.0 |             25.307066 |                 25.307066 |                  25.307066 |                  8.199212 |                      0.0 |    17.107854 |              0.0 |         17.107854 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.607776 | 0.607776 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  9.768224 | 0.113058 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.113058 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |                13.692273 |      142.309202 |             78.3 |       2.1 |              1011.0 | 0.02 | 1.728 |    17.0 |                  78.3 |                   5.0 |                     13.692273 |           142.309202 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                19.406929 |                     19.406929 |                          0.0 |                      1.230421 |                           1.230421 |            15.195625 |                 15.195625 |     995.804375 |   1.206937 | 0.0 | nan | 0.0 |             42.692761 |                 42.692761 |                  42.692761 |                 21.775852 |                      0.0 |    20.916908 |              0.0 |         20.916908 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.90065 |  0.90065 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.82735 | 0.009576 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.009576 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |                13.618831 |      126.722078 |             78.5 |       2.1 |              1014.0 | 0.02 | 1.728 |    18.2 |                  78.5 |                   3.8 |                     13.618831 |           126.722078 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                20.933479 |                     20.933479 |                          0.0 |                      1.314615 |                           1.314615 |            16.432781 |                 16.432781 |     997.567219 |   1.204994 | 0.0 | nan | 0.0 |             38.016624 |                 38.016624 |                  38.016624 |                 17.491881 |                      0.0 |    20.524743 |              0.0 |         20.524743 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.912078 | 0.912078 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.815922 | 0.009444 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.02 | 0.009444 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_v3_wasser_negq_daily:

water (negative runoff)
-----------------------

The following calculation shows the possibility to calculate negative discharge values,
discussed in the integration test :ref:`lland_v1_wasser_negq` of |lland_v1|:

.. integration-test::

    >>> negq(True)
    >>> inlet.sequences.sim.series = 0.0
    >>> test("lland_v3_wasser_negq_daily",
    ...      axis1=(fluxes.nkor, fluxes.evi, fluxes.qah))
    |       date | nied | teml | sunshineduration | possiblesunshineduration | globalradiation | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration | dailypossiblesunshineduration | dailyglobalradiation |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair |   g |  tz |  wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |     evpo |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |       qah |        qa | inzp | wats | waes | esnow | taus | ebdn | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |    outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |                15.690785 |       190.25149 |             86.2 |       2.4 |              1007.0 | 0.0 | 0.0 |    16.1 |                  86.2 |                   6.3 |                     15.690785 |            190.25149 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            15.797694 |                 15.797694 |     991.202306 |   1.205601 | 0.0 | nan | 0.0 |             57.075447 |                 57.075447 |                  57.075447 |                 21.732733 |                      0.0 |    35.342714 |              0.0 |         35.342714 |             88.752361 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077197 | 1.077197 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 11.424803 |  0.132232 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.132232 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |                15.628821 |      127.660782 |             85.8 |       1.6 |              1013.0 | 0.0 | 0.0 |    17.3 |                  85.8 |                   1.7 |                     15.628821 |           127.660782 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                19.779139 |                     19.779139 |                          0.0 |                      1.251029 |                           1.251029 |            16.970501 |                 16.970501 |     996.029499 |   1.207284 | 0.0 | nan | 0.0 |             38.298235 |                 38.298235 |                  38.298235 |                 11.219797 |                      0.0 |    27.078438 |              0.0 |         27.078438 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.86545 |  0.86545 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  -0.77145 | -0.008929 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.008929 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |                15.566134 |      164.401298 |             82.4 |       1.5 |              1016.0 | 0.0 | 0.0 |    18.9 |                  82.4 |                   4.5 |                     15.566134 |           164.401298 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                21.871576 |                     21.871576 |                          0.0 |                      1.365939 |                           1.365939 |            18.022179 |                 18.022179 |     997.977821 |   1.203775 | 0.0 | nan | 0.0 |             49.320389 |                 49.320389 |                  49.320389 |                 16.044181 |                      0.0 |    33.276209 |              0.0 |         33.276209 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.092365 | 1.092365 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.092365 | -0.012643 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.012643 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |                15.502754 |      269.020502 |             77.6 |       2.8 |              1017.0 | 0.0 | 0.0 |    17.8 |                  77.6 |                  12.4 |                     15.502754 |           269.020502 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                20.413369 |                     20.413369 |                          0.0 |                      1.286025 |                           1.286025 |            15.840774 |                 15.840774 |    1001.159226 |    1.21051 | 0.0 | nan | 0.0 |              80.70615 |                  80.70615 |                   80.70615 |                 35.980407 |                      0.0 |    44.725744 |              0.0 |         44.725744 |             76.073452 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.565894 | 1.565894 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.471894 | -0.017036 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.017036 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |                15.438712 |      287.848667 |             74.9 |       3.0 |              1018.0 | 0.0 | 0.0 |    18.5 |                  74.9 |                  13.9 |                     15.438712 |           287.848667 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                21.331125 |                     21.331125 |                          0.0 |                      1.336407 |                           1.336407 |            15.977013 |                 15.977013 |    1002.022987 |   1.208738 | 0.0 | nan | 0.0 |               86.3546 |                   86.3546 |                    86.3546 |                 39.523937 |                      0.0 |    46.830664 |              0.0 |         46.830664 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.729733 | 1.729733 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.729733 |  -0.02002 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  -0.02002 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |                15.374034 |      274.674263 |             77.9 |       3.5 |              1018.0 | 0.0 | 0.0 |    19.9 |                  77.9 |                  13.0 |                     15.374034 |           274.674263 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                23.275348 |                     23.275348 |                          0.0 |                      1.442189 |                           1.442189 |            18.131496 |                 18.131496 |     999.868504 |   1.201995 | 0.0 | nan | 0.0 |             82.402279 |                 82.402279 |                  82.402279 |                 32.936813 |                      0.0 |    49.465466 |              0.0 |         49.465466 |             60.858762 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826087 | 1.826087 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.826087 | -0.021135 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.021135 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |                 15.30875 |      284.047892 |             73.5 |       3.8 |              1019.0 | 0.0 | 0.0 |    20.8 |                  73.5 |                  13.8 |                      15.30875 |           284.047892 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            18.084931 |                 18.084931 |    1000.915069 |   1.199521 | 0.0 | nan | 0.0 |             85.214368 |                 85.214368 |                  85.214368 |                  35.31351 |                      0.0 |    49.900858 |              0.0 |         49.900858 |             56.054123 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.041956 | 2.041956 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -2.041956 | -0.023634 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.023634 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |                15.242885 |      262.971263 |             80.1 |       3.3 |              1019.0 | 0.0 | 0.0 |    20.9 |                  80.1 |                  12.3 |                     15.242885 |           262.971263 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            19.830469 |                 19.830469 |     999.169531 |   1.198331 | 0.0 | nan | 0.0 |             78.891379 |                 78.891379 |                  78.891379 |                 28.392598 |                      0.0 |    50.498781 |              0.0 |         50.498781 |             64.547172 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.800053 | 1.800053 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.800053 | -0.020834 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.020834 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |                15.176467 |      272.256414 |             72.9 |       1.8 |              1019.0 | 0.0 | 0.0 |    22.3 |                  72.9 |                  13.1 |                     15.176467 |           272.256414 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 26.96965 |                      26.96965 |                          0.0 |                      1.639967 |                           1.639967 |            19.660875 |                 19.660875 |     999.339125 |   1.192729 | 0.0 | nan | 0.0 |             81.676924 |                 81.676924 |                  81.676924 |                 31.114889 |                      0.0 |    50.562035 |              0.0 |         50.562035 |            118.336481 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.82186 |  1.82186 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  -1.82186 | -0.021086 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.021086 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |                15.109521 |      267.019218 |             68.5 |       1.9 |              1020.0 | 0.0 | 0.0 |    23.0 |                  68.5 |                  12.8 |                     15.109521 |           267.019218 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                28.139149 |                     28.139149 |                          0.0 |                      1.701779 |                           1.701779 |            19.275317 |                 19.275317 |    1000.724683 |   1.191257 | 0.0 | nan | 0.0 |             80.105765 |                 80.105765 |                  80.105765 |                 32.010643 |                      0.0 |    48.095123 |              0.0 |         48.095123 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.891469 | 1.891469 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.891469 | -0.021892 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.021892 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |                15.042071 |      269.626547 |             68.6 |       3.0 |              1021.0 | 0.0 | 0.0 |    23.1 |                  68.6 |                  13.1 |                     15.042071 |           269.626547 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                28.309776 |                     28.309776 |                          0.0 |                      1.710768 |                           1.710768 |            19.420506 |                 19.420506 |    1001.579494 |   1.191966 | 0.0 | nan | 0.0 |             80.887964 |                 80.887964 |                  80.887964 |                 32.400014 |                      0.0 |     48.48795 |              0.0 |          48.48795 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.106575 | 2.106575 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -2.106575 | -0.024382 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.024382 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |                14.974141 |      270.878291 |             66.0 |       2.5 |              1022.0 | 0.0 | 0.0 |    23.2 |                  66.0 |                  13.3 |                     14.974141 |           270.878291 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |             18.79766 |                  18.79766 |     1003.20234 |   1.193016 | 0.0 | nan | 0.0 |             81.263487 |                 81.263487 |                  81.263487 |                 34.565783 |                      0.0 |    46.697704 |              0.0 |         46.697704 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.036929 | 2.036929 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.942929 | -0.022488 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.022488 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |                14.905755 |      261.697164 |             69.0 |       1.6 |              1019.0 | 0.0 | 0.0 |    23.2 |                  69.0 |                  12.7 |                     14.905755 |           261.697164 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |              19.6521 |                   19.6521 |       999.3479 |    1.18911 | 0.0 | nan | 0.0 |             78.509149 |                 78.509149 |                  78.509149 |                  31.34697 |                      0.0 |    47.162179 |              0.0 |         47.162179 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.808559 | 1.808559 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.808559 | -0.020932 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.020932 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |                14.836934 |      227.962116 |             75.6 |       2.7 |              1015.0 | 0.0 | 0.0 |    21.7 |                  75.6 |                  10.2 |                     14.836934 |           227.962116 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 26.00118 |                      26.00118 |                          0.0 |                        1.5885 |                             1.5885 |            19.656892 |                 19.656892 |     995.343108 |   1.190432 | 0.0 | nan | 0.0 |             68.388635 |                 68.388635 |                  68.388635 |                 25.918437 |                      0.0 |    42.470198 |              0.0 |         42.470198 |             78.890988 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672736 | 1.672736 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.672736 |  -0.01936 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  -0.01936 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |                  14.7677 |      216.356321 |             81.4 |       3.2 |              1016.0 | 0.0 | 0.0 |    20.8 |                  81.4 |                   9.4 |                       14.7677 |           216.356321 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            20.028753 |                 20.028753 |     995.971247 |   1.195095 | 0.0 | nan | 0.0 |             64.906896 |                 64.906896 |                  64.906896 |                 23.397902 |                      0.0 |    41.508994 |              0.0 |         41.508994 |             66.564271 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.529542 | 1.529542 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.008458 |  0.011672 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.011672 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |                14.698075 |      226.637904 |             79.0 |       1.3 |              1019.0 | 0.0 | 0.0 |    19.5 |                  79.0 |                  10.3 |                     14.698075 |           226.637904 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                22.704676 |                     22.704676 |                          0.0 |                      1.411269 |                           1.411269 |            17.936694 |                 17.936694 |    1001.063306 |   1.204916 | 0.0 | nan | 0.0 |             67.991371 |                 67.991371 |                  67.991371 |                 28.794151 |                      0.0 |    39.197221 |              0.0 |         39.197221 |            163.850513 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.283901 | 1.283901 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.283901 |  -0.01486 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  -0.01486 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |                14.628078 |      235.528503 |             75.9 |       1.9 |              1019.0 | 0.0 | 0.0 |    21.0 |                  75.9 |                  11.1 |                     14.628078 |           235.528503 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                24.909749 |                     24.909749 |                          0.0 |                      1.530181 |                           1.530181 |            18.906499 |                 18.906499 |    1000.093501 |   1.198338 | 0.0 | nan | 0.0 |             70.658551 |                 70.658551 |                  70.658551 |                 29.177702 |                      0.0 |    41.480849 |              0.0 |         41.480849 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.510743 | 1.510743 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.510743 | -0.017485 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.017485 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |                14.557728 |      232.849428 |             70.0 |       2.0 |              1018.0 | 0.0 | 0.0 |    21.6 |                  70.0 |                  11.0 |                     14.557728 |           232.849428 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                25.842752 |                     25.842752 |                          0.0 |                      1.580055 |                           1.580055 |            18.089927 |                 18.089927 |     999.910073 |   1.195081 | 0.0 | nan | 0.0 |             69.854829 |                 69.854829 |                  69.854829 |                 31.281288 |                      0.0 |    38.573541 |              0.0 |         38.573541 |            106.502833 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.594572 | 1.594572 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.594572 | -0.018456 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.018456 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |                14.487046 |       199.71349 |             70.3 |       2.6 |              1019.0 | 0.0 | 0.0 |    21.1 |                  70.3 |                   8.5 |                     14.487046 |            199.71349 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            17.619414 |                 17.619414 |    1001.380586 |   1.198506 | 0.0 | nan | 0.0 |             59.914047 |                 59.914047 |                  59.914047 |                 26.711296 |                      0.0 |    33.202751 |              0.0 |         33.202751 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.541567 | 1.541567 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.541567 | -0.017842 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.017842 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |                14.416048 |      233.771902 |             69.0 |       3.6 |              1019.0 | 0.0 | 0.0 |    22.4 |                  69.0 |                  11.3 |                     14.416048 |           233.771902 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                27.134082 |                     27.134082 |                          0.0 |                      1.648679 |                           1.648679 |            18.722517 |                 18.722517 |    1000.277483 |   1.192743 | 0.0 | nan | 0.0 |             70.131571 |                 70.131571 |                  70.131571 |                 31.116622 |                      0.0 |    39.014949 |              0.0 |         39.014949 |             59.168241 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.939383 | 1.939383 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.939383 | -0.022447 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.022447 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |                14.344752 |      246.121286 |             62.1 |       2.5 |              1017.0 | 0.0 | 0.0 |    22.5 |                  62.1 |                  12.4 |                     14.344752 |           246.121286 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                27.299387 |                     27.299387 |                          0.0 |                      1.657431 |                           1.657431 |             16.95292 |                  16.95292 |     1000.04708 |   1.190771 | 0.0 | nan | 0.0 |             73.836386 |                 73.836386 |                  73.836386 |                 38.297604 |                      0.0 |    35.538782 |              0.0 |         35.538782 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826811 | 1.826811 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.826811 | -0.021144 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.021144 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |                14.273175 |        90.58551 |             84.5 |       2.1 |              1015.0 | 0.0 | 0.0 |    19.6 |                  84.5 |                   0.1 |                     14.273175 |             90.58551 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                22.846186 |                     22.846186 |                          0.0 |                      1.418946 |                           1.418946 |            19.305027 |                 19.305027 |     995.694973 |   1.199129 | 0.0 | nan | 0.0 |             27.175653 |                 27.175653 |                  27.175653 |                   7.02118 |                      0.0 |    20.154473 |              0.0 |         20.154473 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.806188 | 0.806188 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  1.073812 |  0.012428 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.012428 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |                14.201333 |      171.944993 |             83.6 |       1.5 |              1015.0 | 0.0 | 0.0 |    20.3 |                  83.6 |                   6.7 |                     14.201333 |           171.944993 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                23.858503 |                     23.858503 |                          0.0 |                      1.473678 |                           1.473678 |            19.945708 |                 19.945708 |     995.054292 |   1.195981 | 0.0 | nan | 0.0 |             51.583498 |                 51.583498 |                  51.583498 |                 18.994889 |                      0.0 |    32.588609 |              0.0 |         32.588609 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.08421 |  1.08421 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  -0.80221 | -0.009285 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.009285 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |                14.129243 |      216.549813 |             76.5 |       2.2 |              1011.0 | 0.0 | 0.0 |    24.1 |                  76.5 |                  10.4 |                     14.129243 |           216.549813 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                30.066256 |                     30.066256 |                          0.0 |                       1.80287 |                            1.80287 |            23.000686 |                 23.000686 |     987.999314 |   1.174651 | 0.0 | nan | 0.0 |             64.964944 |                 64.964944 |                  64.964944 |                 21.629737 |                      0.0 |    43.335207 |              0.0 |         43.335207 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672232 | 1.672232 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.672232 | -0.019355 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.019355 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |                14.056918 |       166.91683 |             76.6 |       1.9 |              1008.0 | 0.0 | 0.0 |    23.7 |                  76.6 |                   6.5 |                     14.056918 |            166.91683 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                29.352596 |                     29.352596 |                          0.0 |                      1.765541 |                           1.765541 |            22.484088 |                 22.484088 |     985.515912 |   1.172942 | 0.0 | nan | 0.0 |             50.075049 |                 50.075049 |                  50.075049 |                 16.222254 |                      0.0 |    33.852795 |              0.0 |         33.852795 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.374102 | 1.374102 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -1.280102 | -0.014816 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.014816 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |                13.984374 |      145.985611 |             84.2 |       2.3 |              1006.0 | 0.0 | 0.0 |    21.1 |                  84.2 |                   4.9 |                     13.984374 |           145.985611 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            21.103195 |                 21.103195 |     984.896805 |   1.181557 | 0.0 | nan | 0.0 |             43.795683 |                 43.795683 |                  43.795683 |                 14.602883 |                      0.0 |      29.1928 |              0.0 |           29.1928 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077096 | 1.077096 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  8.792904 |   0.10177 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.10177 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |                13.911624 |       165.54389 |             85.8 |       2.1 |              1005.0 | 0.0 | 0.0 |    20.9 |                  85.8 |                   6.6 |                     13.911624 |            165.54389 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            21.241626 |                 21.241626 |     983.758374 |   1.181114 | 0.0 | nan | 0.0 |             49.663167 |                 49.663167 |                  49.663167 |                 17.349121 |                      0.0 |    32.314046 |              0.0 |         32.314046 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.09756 |  1.09756 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   5.01244 |  0.058014 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.058014 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |                13.838682 |       87.693346 |             86.5 |       2.6 |              1000.0 | 0.0 | 0.0 |    20.2 |                  86.5 |                   0.3 |                     13.838682 |            87.693346 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                23.711531 |                     23.711531 |                          0.0 |                      1.465752 |                           1.465752 |            20.510475 |                 20.510475 |     979.489525 |   1.178323 | 0.0 | nan | 0.0 |             26.308004 |                 26.308004 |                  26.308004 |                  6.818024 |                      0.0 |     19.48998 |              0.0 |          19.48998 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.794859 | 0.794859 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 13.305141 |  0.153995 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.153995 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |                13.765561 |       84.356885 |             89.6 |       2.6 |              1004.0 | 0.0 | 0.0 |    16.1 |                  89.6 |                   0.1 |                     13.765561 |            84.356885 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            16.420805 |                 16.420805 |     987.579195 |   1.201704 | 0.0 | nan | 0.0 |             25.307066 |                 25.307066 |                  25.307066 |                  8.199212 |                      0.0 |    17.107854 |              0.0 |         17.107854 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.607776 | 0.607776 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  8.040224 |  0.093058 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.093058 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |                13.692273 |      142.309202 |             78.3 |       2.1 |              1011.0 | 0.0 | 0.0 |    17.0 |                  78.3 |                   5.0 |                     13.692273 |           142.309202 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                19.406929 |                     19.406929 |                          0.0 |                      1.230421 |                           1.230421 |            15.195625 |                 15.195625 |     995.804375 |   1.206937 | 0.0 | nan | 0.0 |             42.692761 |                 42.692761 |                  42.692761 |                 21.775852 |                      0.0 |    20.916908 |              0.0 |         20.916908 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.90065 |  0.90065 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  -0.90065 | -0.010424 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.010424 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |                13.618831 |      126.722078 |             78.5 |       2.1 |              1014.0 | 0.0 | 0.0 |    18.2 |                  78.5 |                   3.8 |                     13.618831 |           126.722078 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                20.933479 |                     20.933479 |                          0.0 |                      1.314615 |                           1.314615 |            16.432781 |                 16.432781 |     997.567219 |   1.204994 | 0.0 | nan | 0.0 |             38.016624 |                 38.016624 |                  38.016624 |                 17.491881 |                      0.0 |    20.524743 |              0.0 |         20.524743 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.912078 | 0.912078 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | -0.912078 | -0.010556 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | -0.010556 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_v3_see_daily:

lakes
-----

The following example focuses on water-type |SEE| (for further information, see
integration test :ref:`lland_v1_see` of |lland_v1|):

.. integration-test::

    >>> lnk(SEE)
    >>> surfaceresistance.see_aug = 0.0
    >>> cropheight.see_aug = 0.05
    >>> albedo.see_aug = 0.7
    >>> negq(False)
    >>> test("lland_v3_see_daily",
    ...      axis1=(fluxes.nkor, fluxes.evi, fluxes.qah))
    |       date | nied | teml | sunshineduration | possiblesunshineduration | globalradiation | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration | dailypossiblesunshineduration | dailyglobalradiation |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair |   g |  tz |  wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |     evpo |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |      qah |       qa | inzp | wats | waes | esnow | taus | ebdn | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 |      qbgz | qdga1 | qdga2 | qiga1 | qiga2 |      qbga | inlet |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |                15.690785 |       190.25149 |             86.2 |       2.4 |              1007.0 | 0.0 | 0.0 |    16.1 |                  86.2 |                   6.3 |                     15.690785 |            190.25149 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            15.797694 |                 15.797694 |     991.202306 |   1.205601 | 0.0 | nan | 0.0 |             57.075447 |                 57.075447 |                  57.075447 |                 21.732733 |                      0.0 |    35.342714 |              0.0 |         35.342714 |             88.752361 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077197 | 1.077197 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.020478 | 0.000237 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 11.424803 |   0.0 |   0.0 |   0.0 |   0.0 |  0.020478 |   0.0 | 0.000237 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |                15.628821 |      127.660782 |             85.8 |       1.6 |              1013.0 | 0.0 | 0.0 |    17.3 |                  85.8 |                   1.7 |                     15.628821 |           127.660782 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                19.779139 |                     19.779139 |                          0.0 |                      1.251029 |                           1.251029 |            16.970501 |                 16.970501 |     996.029499 |   1.207284 | 0.0 | nan | 0.0 |             38.298235 |                 38.298235 |                  38.298235 |                 11.219797 |                      0.0 |    27.078438 |              0.0 |         27.078438 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.86545 |  0.86545 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.039475 | 0.000457 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.77145 |   0.0 |   0.0 |   0.0 |   0.0 |  0.039475 |   0.0 | 0.000457 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |                15.566134 |      164.401298 |             82.4 |       1.5 |              1016.0 | 0.0 | 0.0 |    18.9 |                  82.4 |                   4.5 |                     15.566134 |           164.401298 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                21.871576 |                     21.871576 |                          0.0 |                      1.365939 |                           1.365939 |            18.022179 |                 18.022179 |     997.977821 |   1.203775 | 0.0 | nan | 0.0 |             49.320389 |                 49.320389 |                  49.320389 |                 16.044181 |                      0.0 |    33.276209 |              0.0 |         33.276209 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.092365 | 1.092365 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.035994 | 0.000417 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.092365 |   0.0 |   0.0 |   0.0 |   0.0 |  0.035994 |   0.0 | 0.000417 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |                15.502754 |      269.020502 |             77.6 |       2.8 |              1017.0 | 0.0 | 0.0 |    17.8 |                  77.6 |                  12.4 |                     15.502754 |           269.020502 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                20.413369 |                     20.413369 |                          0.0 |                      1.286025 |                           1.286025 |            15.840774 |                 15.840774 |    1001.159226 |    1.21051 | 0.0 | nan | 0.0 |              80.70615 |                  80.70615 |                   80.70615 |                 35.980407 |                      0.0 |    44.725744 |              0.0 |         44.725744 |             76.073452 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.565894 | 1.565894 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.031272 | 0.000362 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.471894 |   0.0 |   0.0 |   0.0 |   0.0 |  0.031272 |   0.0 | 0.000362 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |                15.438712 |      287.848667 |             74.9 |       3.0 |              1018.0 | 0.0 | 0.0 |    18.5 |                  74.9 |                  13.9 |                     15.438712 |           287.848667 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                21.331125 |                     21.331125 |                          0.0 |                      1.336407 |                           1.336407 |            15.977013 |                 15.977013 |    1002.022987 |   1.208738 | 0.0 | nan | 0.0 |               86.3546 |                   86.3546 |                    86.3546 |                 39.523937 |                      0.0 |    46.830664 |              0.0 |         46.830664 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.729733 | 1.729733 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.025424 | 0.000294 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.729733 |   0.0 |   0.0 |   0.0 |   0.0 |  0.025424 |   0.0 | 0.000294 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |                15.374034 |      274.674263 |             77.9 |       3.5 |              1018.0 | 0.0 | 0.0 |    19.9 |                  77.9 |                  13.0 |                     15.374034 |           274.674263 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                23.275348 |                     23.275348 |                          0.0 |                      1.442189 |                           1.442189 |            18.131496 |                 18.131496 |     999.868504 |   1.201995 | 0.0 | nan | 0.0 |             82.402279 |                 82.402279 |                  82.402279 |                 32.936813 |                      0.0 |    49.465466 |              0.0 |         49.465466 |             60.858762 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826087 | 1.826087 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.018963 | 0.000219 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.826087 |   0.0 |   0.0 |   0.0 |   0.0 |  0.018963 |   0.0 | 0.000219 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |                 15.30875 |      284.047892 |             73.5 |       3.8 |              1019.0 | 0.0 | 0.0 |    20.8 |                  73.5 |                  13.8 |                      15.30875 |           284.047892 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            18.084931 |                 18.084931 |    1000.915069 |   1.199521 | 0.0 | nan | 0.0 |             85.214368 |                 85.214368 |                  85.214368 |                  35.31351 |                      0.0 |    49.900858 |              0.0 |         49.900858 |             56.054123 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.041956 | 2.041956 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.011966 | 0.000138 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -2.041956 |   0.0 |   0.0 |   0.0 |   0.0 |  0.011966 |   0.0 | 0.000138 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |                15.242885 |      262.971263 |             80.1 |       3.3 |              1019.0 | 0.0 | 0.0 |    20.9 |                  80.1 |                  12.3 |                     15.242885 |           262.971263 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            19.830469 |                 19.830469 |     999.169531 |   1.198331 | 0.0 | nan | 0.0 |             78.891379 |                 78.891379 |                  78.891379 |                 28.392598 |                      0.0 |    50.498781 |              0.0 |         50.498781 |             64.547172 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.800053 | 1.800053 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.005041 | 0.000058 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.800053 |   0.0 |   0.0 |   0.0 |   0.0 |  0.005041 |   0.0 | 0.000058 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |                15.176467 |      272.256414 |             72.9 |       1.8 |              1019.0 | 0.0 | 0.0 |    22.3 |                  72.9 |                  13.1 |                     15.176467 |           272.256414 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 26.96965 |                      26.96965 |                          0.0 |                      1.639967 |                           1.639967 |            19.660875 |                 19.660875 |     999.339125 |   1.192729 | 0.0 | nan | 0.0 |             81.676924 |                 81.676924 |                  81.676924 |                 31.114889 |                      0.0 |    50.562035 |              0.0 |         50.562035 |            118.336481 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.82186 | 1.820396 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -1.82186 |   0.0 |   0.0 |   0.0 |   0.0 | -0.001465 |   0.0 |      0.0 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |                15.109521 |      267.019218 |             68.5 |       1.9 |              1020.0 | 0.0 | 0.0 |    23.0 |                  68.5 |                  12.8 |                     15.109521 |           267.019218 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                28.139149 |                     28.139149 |                          0.0 |                      1.701779 |                           1.701779 |            19.275317 |                 19.275317 |    1000.724683 |   1.191257 | 0.0 | nan | 0.0 |             80.105765 |                 80.105765 |                  80.105765 |                 32.010643 |                      0.0 |    48.095123 |              0.0 |         48.095123 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.891469 | 1.883357 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.891469 |   0.0 |   0.0 |   0.0 |   0.0 | -0.008111 |   0.0 |      0.0 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |                15.042071 |      269.626547 |             68.6 |       3.0 |              1021.0 | 0.0 | 0.0 |    23.1 |                  68.6 |                  13.1 |                     15.042071 |           269.626547 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                28.309776 |                     28.309776 |                          0.0 |                      1.710768 |                           1.710768 |            19.420506 |                 19.420506 |    1001.579494 |   1.191966 | 0.0 | nan | 0.0 |             80.887964 |                 80.887964 |                  80.887964 |                 32.400014 |                      0.0 |     48.48795 |              0.0 |          48.48795 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.106575 | 2.091331 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -2.106575 |   0.0 |   0.0 |   0.0 |   0.0 | -0.015244 |   0.0 |      0.0 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |                14.974141 |      270.878291 |             66.0 |       2.5 |              1022.0 | 0.0 | 0.0 |    23.2 |                  66.0 |                  13.3 |                     14.974141 |           270.878291 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |             18.79766 |                  18.79766 |     1003.20234 |   1.193016 | 0.0 | nan | 0.0 |             81.263487 |                 81.263487 |                  81.263487 |                 34.565783 |                      0.0 |    46.697704 |              0.0 |         46.697704 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.036929 | 2.014486 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.942929 |   0.0 |   0.0 |   0.0 |   0.0 | -0.022443 |   0.0 |      0.0 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |                14.905755 |      261.697164 |             69.0 |       1.6 |              1019.0 | 0.0 | 0.0 |    23.2 |                  69.0 |                  12.7 |                     14.905755 |           261.697164 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |              19.6521 |                   19.6521 |       999.3479 |    1.18911 | 0.0 | nan | 0.0 |             78.509149 |                 78.509149 |                  78.509149 |                  31.34697 |                      0.0 |    47.162179 |              0.0 |         47.162179 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.808559 | 1.779476 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.808559 |   0.0 |   0.0 |   0.0 |   0.0 | -0.029083 |   0.0 |      0.0 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |                14.836934 |      227.962116 |             75.6 |       2.7 |              1015.0 | 0.0 | 0.0 |    21.7 |                  75.6 |                  10.2 |                     14.836934 |           227.962116 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 26.00118 |                      26.00118 |                          0.0 |                        1.5885 |                             1.5885 |            19.656892 |                 19.656892 |     995.343108 |   1.190432 | 0.0 | nan | 0.0 |             68.388635 |                 68.388635 |                  68.388635 |                 25.918437 |                      0.0 |    42.470198 |              0.0 |         42.470198 |             78.890988 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672736 | 1.637521 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.672736 |   0.0 |   0.0 |   0.0 |   0.0 | -0.035215 |   0.0 |      0.0 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |                  14.7677 |      216.356321 |             81.4 |       3.2 |              1016.0 | 0.0 | 0.0 |    20.8 |                  81.4 |                   9.4 |                       14.7677 |           216.356321 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            20.028753 |                 20.028753 |     995.971247 |   1.195095 | 0.0 | nan | 0.0 |             64.906896 |                 64.906896 |                  64.906896 |                 23.397902 |                      0.0 |    41.508994 |              0.0 |         41.508994 |             66.564271 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.529542 | 1.493266 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  1.008458 |   0.0 |   0.0 |   0.0 |   0.0 | -0.036276 |   0.0 |      0.0 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |                14.698075 |      226.637904 |             79.0 |       1.3 |              1019.0 | 0.0 | 0.0 |    19.5 |                  79.0 |                  10.3 |                     14.698075 |           226.637904 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                22.704676 |                     22.704676 |                          0.0 |                      1.411269 |                           1.411269 |            17.936694 |                 17.936694 |    1001.063306 |   1.204916 | 0.0 | nan | 0.0 |             67.991371 |                 67.991371 |                  67.991371 |                 28.794151 |                      0.0 |    39.197221 |              0.0 |         39.197221 |            163.850513 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.283901 | 1.247259 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.283901 |   0.0 |   0.0 |   0.0 |   0.0 | -0.036642 |   0.0 |      0.0 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |                14.628078 |      235.528503 |             75.9 |       1.9 |              1019.0 | 0.0 | 0.0 |    21.0 |                  75.9 |                  11.1 |                     14.628078 |           235.528503 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                24.909749 |                     24.909749 |                          0.0 |                      1.530181 |                           1.530181 |            18.906499 |                 18.906499 |    1000.093501 |   1.198338 | 0.0 | nan | 0.0 |             70.658551 |                 70.658551 |                  70.658551 |                 29.177702 |                      0.0 |    41.480849 |              0.0 |         41.480849 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.510743 | 1.469226 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.510743 |   0.0 |   0.0 |   0.0 |   0.0 | -0.041517 |   0.0 |      0.0 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |                14.557728 |      232.849428 |             70.0 |       2.0 |              1018.0 | 0.0 | 0.0 |    21.6 |                  70.0 |                  11.0 |                     14.557728 |           232.849428 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                25.842752 |                     25.842752 |                          0.0 |                      1.580055 |                           1.580055 |            18.089927 |                 18.089927 |     999.910073 |   1.195081 | 0.0 | nan | 0.0 |             69.854829 |                 69.854829 |                  69.854829 |                 31.281288 |                      0.0 |    38.573541 |              0.0 |         38.573541 |            106.502833 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.594572 | 1.547641 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.594572 |   0.0 |   0.0 |   0.0 |   0.0 | -0.046931 |   0.0 |      0.0 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |                14.487046 |       199.71349 |             70.3 |       2.6 |              1019.0 | 0.0 | 0.0 |    21.1 |                  70.3 |                   8.5 |                     14.487046 |            199.71349 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            17.619414 |                 17.619414 |    1001.380586 |   1.198506 | 0.0 | nan | 0.0 |             59.914047 |                 59.914047 |                  59.914047 |                 26.711296 |                      0.0 |    33.202751 |              0.0 |         33.202751 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.541567 | 1.489186 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.541567 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.05238 |   0.0 |      0.0 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |                14.416048 |      233.771902 |             69.0 |       3.6 |              1019.0 | 0.0 | 0.0 |    22.4 |                  69.0 |                  11.3 |                     14.416048 |           233.771902 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                27.134082 |                     27.134082 |                          0.0 |                      1.648679 |                           1.648679 |            18.722517 |                 18.722517 |    1000.277483 |   1.192743 | 0.0 | nan | 0.0 |             70.131571 |                 70.131571 |                  70.131571 |                 31.116622 |                      0.0 |    39.014949 |              0.0 |         39.014949 |             59.168241 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.939383 | 1.880954 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.939383 |   0.0 |   0.0 |   0.0 |   0.0 | -0.058428 |   0.0 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |                14.344752 |      246.121286 |             62.1 |       2.5 |              1017.0 | 0.0 | 0.0 |    22.5 |                  62.1 |                  12.4 |                     14.344752 |           246.121286 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                27.299387 |                     27.299387 |                          0.0 |                      1.657431 |                           1.657431 |             16.95292 |                  16.95292 |     1000.04708 |   1.190771 | 0.0 | nan | 0.0 |             73.836386 |                 73.836386 |                  73.836386 |                 38.297604 |                      0.0 |    35.538782 |              0.0 |         35.538782 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826811 | 1.761846 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.826811 |   0.0 |   0.0 |   0.0 |   0.0 | -0.064965 |   0.0 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |                14.273175 |        90.58551 |             84.5 |       2.1 |              1015.0 | 0.0 | 0.0 |    19.6 |                  84.5 |                   0.1 |                     14.273175 |             90.58551 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                22.846186 |                     22.846186 |                          0.0 |                      1.418946 |                           1.418946 |            19.305027 |                 19.305027 |     995.694973 |   1.199129 | 0.0 | nan | 0.0 |             27.175653 |                 27.175653 |                  27.175653 |                   7.02118 |                      0.0 |    20.154473 |              0.0 |         20.154473 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.806188 | 0.740109 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  1.073812 |   0.0 |   0.0 |   0.0 |   0.0 | -0.066078 |   0.0 |      0.0 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |                14.201333 |      171.944993 |             83.6 |       1.5 |              1015.0 | 0.0 | 0.0 |    20.3 |                  83.6 |                   6.7 |                     14.201333 |           171.944993 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                23.858503 |                     23.858503 |                          0.0 |                      1.473678 |                           1.473678 |            19.945708 |                 19.945708 |     995.054292 |   1.195981 | 0.0 | nan | 0.0 |             51.583498 |                 51.583498 |                  51.583498 |                 18.994889 |                      0.0 |    32.588609 |              0.0 |         32.588609 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.08421 | 1.018853 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.80221 |   0.0 |   0.0 |   0.0 |   0.0 | -0.065357 |   0.0 |      0.0 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |                14.129243 |      216.549813 |             76.5 |       2.2 |              1011.0 | 0.0 | 0.0 |    24.1 |                  76.5 |                  10.4 |                     14.129243 |           216.549813 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                30.066256 |                     30.066256 |                          0.0 |                       1.80287 |                            1.80287 |            23.000686 |                 23.000686 |     987.999314 |   1.174651 | 0.0 | nan | 0.0 |             64.964944 |                 64.964944 |                  64.964944 |                 21.629737 |                      0.0 |    43.335207 |              0.0 |         43.335207 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672232 | 1.602675 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.672232 |   0.0 |   0.0 |   0.0 |   0.0 | -0.069557 |   0.0 |      0.0 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |                14.056918 |       166.91683 |             76.6 |       1.9 |              1008.0 | 0.0 | 0.0 |    23.7 |                  76.6 |                   6.5 |                     14.056918 |            166.91683 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                29.352596 |                     29.352596 |                          0.0 |                      1.765541 |                           1.765541 |            22.484088 |                 22.484088 |     985.515912 |   1.172942 | 0.0 | nan | 0.0 |             50.075049 |                 50.075049 |                  50.075049 |                 16.222254 |                      0.0 |    33.852795 |              0.0 |         33.852795 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.374102 | 1.299507 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -1.280102 |   0.0 |   0.0 |   0.0 |   0.0 | -0.074595 |   0.0 |      0.0 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |                13.984374 |      145.985611 |             84.2 |       2.3 |              1006.0 | 0.0 | 0.0 |    21.1 |                  84.2 |                   4.9 |                     13.984374 |           145.985611 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            21.103195 |                 21.103195 |     984.896805 |   1.181557 | 0.0 | nan | 0.0 |             43.795683 |                 43.795683 |                  43.795683 |                 14.602883 |                      0.0 |      29.1928 |              0.0 |           29.1928 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077096 | 1.016237 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  8.792904 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.06086 |   0.0 |      0.0 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |                13.911624 |       165.54389 |             85.8 |       2.1 |              1005.0 | 0.0 | 0.0 |    20.9 |                  85.8 |                   6.6 |                     13.911624 |            165.54389 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            21.241626 |                 21.241626 |     983.758374 |   1.181114 | 0.0 | nan | 0.0 |             49.663167 |                 49.663167 |                  49.663167 |                 17.349121 |                      0.0 |    32.314046 |              0.0 |         32.314046 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.09756 | 1.061645 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |   5.01244 |   0.0 |   0.0 |   0.0 |   0.0 | -0.035916 |   0.0 |      0.0 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |                13.838682 |       87.693346 |             86.5 |       2.6 |              1000.0 | 0.0 | 0.0 |    20.2 |                  86.5 |                   0.3 |                     13.838682 |            87.693346 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                23.711531 |                     23.711531 |                          0.0 |                      1.465752 |                           1.465752 |            20.510475 |                 20.510475 |     979.489525 |   1.178323 | 0.0 | nan | 0.0 |             26.308004 |                 26.308004 |                  26.308004 |                  6.818024 |                      0.0 |     19.48998 |              0.0 |          19.48998 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.794859 | 0.791893 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | 13.305141 |   0.0 |   0.0 |   0.0 |   0.0 | -0.002966 |   0.0 |      0.0 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |                13.765561 |       84.356885 |             89.6 |       2.6 |              1004.0 | 0.0 | 0.0 |    16.1 |                  89.6 |                   0.1 |                     13.765561 |            84.356885 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            16.420805 |                 16.420805 |     987.579195 |   1.201704 | 0.0 | nan | 0.0 |             25.307066 |                 25.307066 |                  25.307066 |                  8.199212 |                      0.0 |    17.107854 |              0.0 |         17.107854 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.607776 | 0.607776 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.035276 | 0.000408 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  8.040224 |   0.0 |   0.0 |   0.0 |   0.0 |  0.035276 |   0.0 | 0.000408 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |                13.692273 |      142.309202 |             78.3 |       2.1 |              1011.0 | 0.0 | 0.0 |    17.0 |                  78.3 |                   5.0 |                     13.692273 |           142.309202 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                19.406929 |                     19.406929 |                          0.0 |                      1.230421 |                           1.230421 |            15.195625 |                 15.195625 |     995.804375 |   1.206937 | 0.0 | nan | 0.0 |             42.692761 |                 42.692761 |                  42.692761 |                 21.775852 |                      0.0 |    20.916908 |              0.0 |         20.916908 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.90065 |  0.90065 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.047929 | 0.000555 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  -0.90065 |   0.0 |   0.0 |   0.0 |   0.0 |  0.047929 |   0.0 | 0.000555 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |                13.618831 |      126.722078 |             78.5 |       2.1 |              1014.0 | 0.0 | 0.0 |    18.2 |                  78.5 |                   3.8 |                     13.618831 |           126.722078 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                20.933479 |                     20.933479 |                          0.0 |                      1.314615 |                           1.314615 |            16.432781 |                 16.432781 |     997.567219 |   1.204994 | 0.0 | nan | 0.0 |             38.016624 |                 38.016624 |                  38.016624 |                 17.491881 |                      0.0 |    20.524743 |              0.0 |         20.524743 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.912078 | 0.912078 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.04451 | 0.000515 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 | -0.912078 |   0.0 |   0.0 |   0.0 |   0.0 |   0.04451 |   0.0 | 0.000515 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_v3_fluss_daily:

streams
-------

The following example focuses on water-type |FLUSS| (for further information, see
integration test :ref:`lland_v1_fluss` of |lland_v1|):

.. integration-test::

    >>> lnk(FLUSS)
    >>> surfaceresistance.fluss_aug = 0.0
    >>> cropheight.fluss_aug = 0.05
    >>> albedo.fluss_aug = 0.7
    >>> test("lland_v3_fluss_daily",
    ...       axis1=(fluxes.nkor, fluxes.evi, fluxes.qah))
    |       date | nied | teml | sunshineduration | possiblesunshineduration | globalradiation | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration | dailypossiblesunshineduration | dailyglobalradiation |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair |   g |  tz |  wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |     evpo |       evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap |      qdgz |      qah |       qa | inzp | wats | waes | esnow | taus | ebdn | bowa |     qdgz1 |    qdgz2 | qigz1 | qigz2 | qbgz |     qdga1 |    qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |                15.690785 |       190.25149 |             86.2 |       2.4 |              1007.0 | 0.0 | 0.0 |    16.1 |                  86.2 |                   6.3 |                     15.690785 |            190.25149 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            15.797694 |                 15.797694 |     991.202306 |   1.205601 | 0.0 | nan | 0.0 |             57.075447 |                 57.075447 |                  57.075447 |                 21.732733 |                      0.0 |    35.342714 |              0.0 |         35.342714 |             88.752361 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077197 |  1.077197 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 11.424803 | 2.941276 | 0.034043 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | 11.424803 |      0.0 |   0.0 |   0.0 |  0.0 |  2.941276 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.034043 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |                15.628821 |      127.660782 |             85.8 |       1.6 |              1013.0 | 0.0 | 0.0 |    17.3 |                  85.8 |                   1.7 |                     15.628821 |           127.660782 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                19.779139 |                     19.779139 |                          0.0 |                      1.251029 |                           1.251029 |            16.970501 |                 16.970501 |     996.029499 |   1.207284 | 0.0 | nan | 0.0 |             38.298235 |                 38.298235 |                  38.298235 |                 11.219797 |                      0.0 |    27.078438 |              0.0 |         27.078438 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.86545 |   0.86545 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.77145 | 3.758016 | 0.043496 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  -0.77145 |      0.0 |   0.0 |   0.0 |  0.0 |  3.758016 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.043496 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |                15.566134 |      164.401298 |             82.4 |       1.5 |              1016.0 | 0.0 | 0.0 |    18.9 |                  82.4 |                   4.5 |                     15.566134 |           164.401298 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                21.871576 |                     21.871576 |                          0.0 |                      1.365939 |                           1.365939 |            18.022179 |                 18.022179 |     997.977821 |   1.203775 | 0.0 | nan | 0.0 |             49.320389 |                 49.320389 |                  49.320389 |                 16.044181 |                      0.0 |    33.276209 |              0.0 |         33.276209 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.092365 |  1.092365 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.092365 | 1.562905 | 0.018089 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.092365 |      0.0 |   0.0 |   0.0 |  0.0 |  1.562905 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.018089 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |                15.502754 |      269.020502 |             77.6 |       2.8 |              1017.0 | 0.0 | 0.0 |    17.8 |                  77.6 |                  12.4 |                     15.502754 |           269.020502 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                20.413369 |                     20.413369 |                          0.0 |                      1.286025 |                           1.286025 |            15.840774 |                 15.840774 |    1001.159226 |    1.21051 | 0.0 | nan | 0.0 |              80.70615 |                  80.70615 |                   80.70615 |                 35.980407 |                      0.0 |    44.725744 |              0.0 |         44.725744 |             76.073452 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.565894 |  1.565894 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.471894 | 0.226808 | 0.002625 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.471894 |      0.0 |   0.0 |   0.0 |  0.0 |  0.226808 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.002625 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |                15.438712 |      287.848667 |             74.9 |       3.0 |              1018.0 | 0.0 | 0.0 |    18.5 |                  74.9 |                  13.9 |                     15.438712 |           287.848667 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                21.331125 |                     21.331125 |                          0.0 |                      1.336407 |                           1.336407 |            15.977013 |                 15.977013 |    1002.022987 |   1.208738 | 0.0 | nan | 0.0 |               86.3546 |                   86.3546 |                    86.3546 |                 39.523937 |                      0.0 |    46.830664 |              0.0 |         46.830664 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.729733 |  1.097905 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.729733 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.729733 |      0.0 |   0.0 |   0.0 |  0.0 | -0.631827 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |                15.374034 |      274.674263 |             77.9 |       3.5 |              1018.0 | 0.0 | 0.0 |    19.9 |                  77.9 |                  13.0 |                     15.374034 |           274.674263 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                23.275348 |                     23.275348 |                          0.0 |                      1.442189 |                           1.442189 |            18.131496 |                 18.131496 |     999.868504 |   1.201995 | 0.0 | nan | 0.0 |             82.402279 |                 82.402279 |                  82.402279 |                 32.936813 |                      0.0 |    49.465466 |              0.0 |         49.465466 |             60.858762 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826087 |  0.657403 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.826087 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.826087 |      0.0 |   0.0 |   0.0 |  0.0 | -1.168684 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |                 15.30875 |      284.047892 |             73.5 |       3.8 |              1019.0 | 0.0 | 0.0 |    20.8 |                  73.5 |                  13.8 |                      15.30875 |           284.047892 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            18.084931 |                 18.084931 |    1000.915069 |   1.199521 | 0.0 | nan | 0.0 |             85.214368 |                 85.214368 |                  85.214368 |                  35.31351 |                      0.0 |    49.900858 |              0.0 |         49.900858 |             56.054123 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.041956 |  0.511091 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -2.041956 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.041956 |      0.0 |   0.0 |   0.0 |  0.0 | -1.530864 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |                15.242885 |      262.971263 |             80.1 |       3.3 |              1019.0 | 0.0 | 0.0 |    20.9 |                  80.1 |                  12.3 |                     15.242885 |           262.971263 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            19.830469 |                 19.830469 |     999.169531 |   1.198331 | 0.0 | nan | 0.0 |             78.891379 |                 78.891379 |                  78.891379 |                 28.392598 |                      0.0 |    50.498781 |              0.0 |         50.498781 |             64.547172 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.800053 |  0.093098 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.800053 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.800053 |      0.0 |   0.0 |   0.0 |  0.0 | -1.706955 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |                15.176467 |      272.256414 |             72.9 |       1.8 |              1019.0 | 0.0 | 0.0 |    22.3 |                  72.9 |                  13.1 |                     15.176467 |           272.256414 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 26.96965 |                      26.96965 |                          0.0 |                      1.639967 |                           1.639967 |            19.660875 |                 19.660875 |     999.339125 |   1.192729 | 0.0 | nan | 0.0 |             81.676924 |                 81.676924 |                  81.676924 |                 31.114889 |                      0.0 |    50.562035 |              0.0 |         50.562035 |            118.336481 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.82186 |  0.065871 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -1.82186 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  -1.82186 |      0.0 |   0.0 |   0.0 |  0.0 | -1.755989 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |                15.109521 |      267.019218 |             68.5 |       1.9 |              1020.0 | 0.0 | 0.0 |    23.0 |                  68.5 |                  12.8 |                     15.109521 |           267.019218 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                28.139149 |                     28.139149 |                          0.0 |                      1.701779 |                           1.701779 |            19.275317 |                 19.275317 |    1000.724683 |   1.191257 | 0.0 | nan | 0.0 |             80.105765 |                 80.105765 |                  80.105765 |                 32.010643 |                      0.0 |    48.095123 |              0.0 |         48.095123 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.891469 |  0.086838 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.891469 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.891469 |      0.0 |   0.0 |   0.0 |  0.0 | -1.804631 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |                15.042071 |      269.626547 |             68.6 |       3.0 |              1021.0 | 0.0 | 0.0 |    23.1 |                  68.6 |                  13.1 |                     15.042071 |           269.626547 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                28.309776 |                     28.309776 |                          0.0 |                      1.710768 |                           1.710768 |            19.420506 |                 19.420506 |    1001.579494 |   1.191966 | 0.0 | nan | 0.0 |             80.887964 |                 80.887964 |                  80.887964 |                 32.400014 |                      0.0 |     48.48795 |              0.0 |          48.48795 |             71.001889 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.106575 |  0.206065 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -2.106575 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -2.106575 |      0.0 |   0.0 |   0.0 |  0.0 | -1.900509 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |                14.974141 |      270.878291 |             66.0 |       2.5 |              1022.0 | 0.0 | 0.0 |    23.2 |                  66.0 |                  13.3 |                     14.974141 |           270.878291 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |             18.79766 |                  18.79766 |     1003.20234 |   1.193016 | 0.0 | nan | 0.0 |             81.263487 |                 81.263487 |                  81.263487 |                 34.565783 |                      0.0 |    46.697704 |              0.0 |         46.697704 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 2.036929 |  0.082443 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.942929 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.942929 |      0.0 |   0.0 |   0.0 |  0.0 | -1.954486 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |                14.905755 |      261.697164 |             69.0 |       1.6 |              1019.0 | 0.0 | 0.0 |    23.2 |                  69.0 |                  12.7 |                     14.905755 |           261.697164 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |              19.6521 |                   19.6521 |       999.3479 |    1.18911 | 0.0 | nan | 0.0 |             78.509149 |                 78.509149 |                  78.509149 |                  31.34697 |                      0.0 |    47.162179 |              0.0 |         47.162179 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.808559 | -0.105944 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.808559 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.808559 |      0.0 |   0.0 |   0.0 |  0.0 | -1.914503 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |                14.836934 |      227.962116 |             75.6 |       2.7 |              1015.0 | 0.0 | 0.0 |    21.7 |                  75.6 |                  10.2 |                     14.836934 |           227.962116 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 26.00118 |                      26.00118 |                          0.0 |                        1.5885 |                             1.5885 |            19.656892 |                 19.656892 |     995.343108 |   1.190432 | 0.0 | nan | 0.0 |             68.388635 |                 68.388635 |                  68.388635 |                 25.918437 |                      0.0 |    42.470198 |              0.0 |         42.470198 |             78.890988 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672736 | -0.157388 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.672736 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.672736 |      0.0 |   0.0 |   0.0 |  0.0 | -1.830125 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |                  14.7677 |      216.356321 |             81.4 |       3.2 |              1016.0 | 0.0 | 0.0 |    20.8 |                  81.4 |                   9.4 |                       14.7677 |           216.356321 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            20.028753 |                 20.028753 |     995.971247 |   1.195095 | 0.0 | nan | 0.0 |             64.906896 |                 64.906896 |                  64.906896 |                 23.397902 |                      0.0 |    41.508994 |              0.0 |         41.508994 |             66.564271 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.529542 |  0.463086 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.008458 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  1.008458 |      0.0 |   0.0 |   0.0 |  0.0 | -1.066456 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |                14.698075 |      226.637904 |             79.0 |       1.3 |              1019.0 | 0.0 | 0.0 |    19.5 |                  79.0 |                  10.3 |                     14.698075 |           226.637904 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                22.704676 |                     22.704676 |                          0.0 |                      1.411269 |                           1.411269 |            17.936694 |                 17.936694 |    1001.063306 |   1.204916 | 0.0 | nan | 0.0 |             67.991371 |                 67.991371 |                  67.991371 |                 28.794151 |                      0.0 |    39.197221 |              0.0 |         39.197221 |            163.850513 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.283901 |  0.595002 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.283901 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.283901 |      0.0 |   0.0 |   0.0 |  0.0 | -0.688899 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |                14.628078 |      235.528503 |             75.9 |       1.9 |              1019.0 | 0.0 | 0.0 |    21.0 |                  75.9 |                  11.1 |                     14.628078 |           235.528503 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                24.909749 |                     24.909749 |                          0.0 |                      1.530181 |                           1.530181 |            18.906499 |                 18.906499 |    1000.093501 |   1.198338 | 0.0 | nan | 0.0 |             70.658551 |                 70.658551 |                  70.658551 |                 29.177702 |                      0.0 |    41.480849 |              0.0 |         41.480849 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.510743 |  0.485942 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.510743 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.510743 |      0.0 |   0.0 |   0.0 |  0.0 | -1.024801 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |                14.557728 |      232.849428 |             70.0 |       2.0 |              1018.0 | 0.0 | 0.0 |    21.6 |                  70.0 |                  11.0 |                     14.557728 |           232.849428 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                25.842752 |                     25.842752 |                          0.0 |                      1.580055 |                           1.580055 |            18.089927 |                 18.089927 |     999.910073 |   1.195081 | 0.0 | nan | 0.0 |             69.854829 |                 69.854829 |                  69.854829 |                 31.281288 |                      0.0 |    38.573541 |              0.0 |         38.573541 |            106.502833 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.594572 |  0.321551 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.594572 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.594572 |      0.0 |   0.0 |   0.0 |  0.0 |  -1.27302 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |                14.487046 |       199.71349 |             70.3 |       2.6 |              1019.0 | 0.0 | 0.0 |    21.1 |                  70.3 |                   8.5 |                     14.487046 |            199.71349 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            17.619414 |                 17.619414 |    1001.380586 |   1.198506 | 0.0 | nan | 0.0 |             59.914047 |                 59.914047 |                  59.914047 |                 26.711296 |                      0.0 |    33.202751 |              0.0 |         33.202751 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.541567 |  0.132224 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.541567 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.541567 |      0.0 |   0.0 |   0.0 |  0.0 | -1.409342 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |                14.416048 |      233.771902 |             69.0 |       3.6 |              1019.0 | 0.0 | 0.0 |    22.4 |                  69.0 |                  11.3 |                     14.416048 |           233.771902 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                27.134082 |                     27.134082 |                          0.0 |                      1.648679 |                           1.648679 |            18.722517 |                 18.722517 |    1000.277483 |   1.192743 | 0.0 | nan | 0.0 |             70.131571 |                 70.131571 |                  70.131571 |                 31.116622 |                      0.0 |    39.014949 |              0.0 |         39.014949 |             59.168241 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.939383 |  0.365956 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.939383 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.939383 |      0.0 |   0.0 |   0.0 |  0.0 | -1.573427 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |                14.344752 |      246.121286 |             62.1 |       2.5 |              1017.0 | 0.0 | 0.0 |    22.5 |                  62.1 |                  12.4 |                     14.344752 |           246.121286 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                27.299387 |                     27.299387 |                          0.0 |                      1.657431 |                           1.657431 |             16.95292 |                  16.95292 |     1000.04708 |   1.190771 | 0.0 | nan | 0.0 |             73.836386 |                 73.836386 |                  73.836386 |                 38.297604 |                      0.0 |    35.538782 |              0.0 |         35.538782 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.826811 |  0.111688 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.826811 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.826811 |      0.0 |   0.0 |   0.0 |  0.0 | -1.715123 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |                14.273175 |        90.58551 |             84.5 |       2.1 |              1015.0 | 0.0 | 0.0 |    19.6 |                  84.5 |                   0.1 |                     14.273175 |             90.58551 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                22.846186 |                     22.846186 |                          0.0 |                      1.418946 |                           1.418946 |            19.305027 |                 19.305027 |     995.694973 |   1.199129 | 0.0 | nan | 0.0 |             27.175653 |                 27.175653 |                  27.175653 |                   7.02118 |                      0.0 |    20.154473 |              0.0 |         20.154473 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.806188 |  -0.21427 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  1.073812 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  1.073812 |      0.0 |   0.0 |   0.0 |  0.0 | -1.020458 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |                14.201333 |      171.944993 |             83.6 |       1.5 |              1015.0 | 0.0 | 0.0 |    20.3 |                  83.6 |                   6.7 |                     14.201333 |           171.944993 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                23.858503 |                     23.858503 |                          0.0 |                      1.473678 |                           1.473678 |            19.945708 |                 19.945708 |     995.054292 |   1.195981 | 0.0 | nan | 0.0 |             51.583498 |                 51.583498 |                  51.583498 |                 18.994889 |                      0.0 |    32.588609 |              0.0 |         32.588609 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.08421 |  0.557521 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.80221 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  -0.80221 |      0.0 |   0.0 |   0.0 |  0.0 | -0.526689 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |                14.129243 |      216.549813 |             76.5 |       2.2 |              1011.0 | 0.0 | 0.0 |    24.1 |                  76.5 |                  10.4 |                     14.129243 |           216.549813 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                30.066256 |                     30.066256 |                          0.0 |                       1.80287 |                            1.80287 |            23.000686 |                 23.000686 |     987.999314 |   1.174651 | 0.0 | nan | 0.0 |             64.964944 |                 64.964944 |                  64.964944 |                 21.629737 |                      0.0 |    43.335207 |              0.0 |         43.335207 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.672232 |  0.793059 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.672232 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.672232 |      0.0 |   0.0 |   0.0 |  0.0 | -0.879173 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |                14.056918 |       166.91683 |             76.6 |       1.9 |              1008.0 | 0.0 | 0.0 |    23.7 |                  76.6 |                   6.5 |                     14.056918 |            166.91683 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                29.352596 |                     29.352596 |                          0.0 |                      1.765541 |                           1.765541 |            22.484088 |                 22.484088 |     985.515912 |   1.172942 | 0.0 | nan | 0.0 |             50.075049 |                 50.075049 |                  50.075049 |                 16.222254 |                      0.0 |    33.852795 |              0.0 |         33.852795 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.374102 |  0.226008 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -1.280102 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -1.280102 |      0.0 |   0.0 |   0.0 |  0.0 | -1.148095 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |                13.984374 |      145.985611 |             84.2 |       2.3 |              1006.0 | 0.0 | 0.0 |    21.1 |                  84.2 |                   4.9 |                     13.984374 |           145.985611 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            21.103195 |                 21.103195 |     984.896805 |   1.181557 | 0.0 | nan | 0.0 |             43.795683 |                 43.795683 |                  43.795683 |                 14.602883 |                      0.0 |      29.1928 |              0.0 |           29.1928 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 1.077096 |  1.077096 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  8.792904 |   1.3836 | 0.016014 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  8.792904 |      0.0 |   0.0 |   0.0 |  0.0 |    1.3836 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.016014 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |                13.911624 |       165.54389 |             85.8 |       2.1 |              1005.0 | 0.0 | 0.0 |    20.9 |                  85.8 |                   6.6 |                     13.911624 |            165.54389 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            21.241626 |                 21.241626 |     983.758374 |   1.181114 | 0.0 | nan | 0.0 |             49.663167 |                 49.663167 |                  49.663167 |                 17.349121 |                      0.0 |    32.314046 |              0.0 |         32.314046 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  1.09756 |   1.09756 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |   5.01244 |  3.86595 | 0.044745 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   5.01244 |      0.0 |   0.0 |   0.0 |  0.0 |   3.86595 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.044745 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |                13.838682 |       87.693346 |             86.5 |       2.6 |              1000.0 | 0.0 | 0.0 |    20.2 |                  86.5 |                   0.3 |                     13.838682 |            87.693346 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                23.711531 |                     23.711531 |                          0.0 |                      1.465752 |                           1.465752 |            20.510475 |                 20.510475 |     979.489525 |   1.178323 | 0.0 | nan | 0.0 |             26.308004 |                 26.308004 |                  26.308004 |                  6.818024 |                      0.0 |     19.48998 |              0.0 |          19.48998 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.794859 |  0.794859 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | 13.305141 | 7.027354 | 0.081335 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |      12.0 | 1.305141 |   0.0 |   0.0 |  0.0 |  6.199583 |  0.82777 |   0.0 |   0.0 |  0.0 |   0.0 | 0.081335 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |                13.765561 |       84.356885 |             89.6 |       2.6 |              1004.0 | 0.0 | 0.0 |    16.1 |                  89.6 |                   0.1 |                     13.765561 |            84.356885 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            16.420805 |                 16.420805 |     987.579195 |   1.201704 | 0.0 | nan | 0.0 |             25.307066 |                 25.307066 |                  25.307066 |                  8.199212 |                      0.0 |    17.107854 |              0.0 |         17.107854 |             81.925256 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.607776 |  0.607776 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  8.040224 |  8.32407 | 0.096343 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  8.040224 |      0.0 |   0.0 |   0.0 |  0.0 |  7.885403 | 0.438667 |   0.0 |   0.0 |  0.0 |   0.0 | 0.096343 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |                13.692273 |      142.309202 |             78.3 |       2.1 |              1011.0 | 0.0 | 0.0 |    17.0 |                  78.3 |                   5.0 |                     13.692273 |           142.309202 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                19.406929 |                     19.406929 |                          0.0 |                      1.230421 |                           1.230421 |            15.195625 |                 15.195625 |     995.804375 |   1.206937 | 0.0 | nan | 0.0 |             42.692761 |                 42.692761 |                  42.692761 |                 21.775852 |                      0.0 |    20.916908 |              0.0 |         20.916908 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.90065 |   0.90065 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  -0.90065 | 5.691378 | 0.065872 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |  -0.90065 |      0.0 |   0.0 |   0.0 |  0.0 |  5.655812 | 0.035566 |   0.0 |   0.0 |  0.0 |   0.0 | 0.065872 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |                13.618831 |      126.722078 |             78.5 |       2.1 |              1014.0 | 0.0 | 0.0 |    18.2 |                  78.5 |                   3.8 |                     13.618831 |           126.722078 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                20.933479 |                     20.933479 |                          0.0 |                      1.314615 |                           1.314615 |            16.432781 |                 16.432781 |     997.567219 |   1.204994 | 0.0 | nan | 0.0 |             38.016624 |                 38.016624 |                  38.016624 |                 17.491881 |                      0.0 |    20.524743 |              0.0 |         20.524743 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.912078 |  0.912078 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 | -0.912078 | 2.597892 | 0.030068 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 | -0.912078 |      0.0 |   0.0 |   0.0 |  0.0 |  2.595008 | 0.002884 |   0.0 |   0.0 |  0.0 |   0.0 | 0.030068 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_v3_vers_daily:

sealed surfaces
---------------

As also shown in the integration test :ref:`lland_v1_vers` of |lland_v1|, sealed
surfaces route the not intercepted water to the linear storages for direct discharge
immediately:

.. integration-test::

    >>> lnk(VERS)
    >>> negq(True)
    >>> surfaceresistance.vers_aug = 500
    >>> cropheight.vers_aug = 5.0
    >>> albedo.vers_aug = 0.10
    >>> lai.vers_aug = 10.0
    >>> test("lland_v3_vers_daily",
    ...      axis1=(inputs.nied, fluxes.qah, states.bowa))
    |       date | nied | teml | sunshineduration | possiblesunshineduration | globalradiation | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration | dailypossiblesunshineduration | dailyglobalradiation |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair |    g |        tz |         wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance |   nbes | sbes |      evpo |   evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg |   wada |    qdb | qib1 | qib2 | qbb | qkap |   qdgz |      qah |       qa | inzp | wats | waes | esnow | taus |       ebdn | bowa |  qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 |    qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |                15.690785 |       190.25149 |             86.2 |       2.4 |              1007.0 | 0.0 | 0.0 |    16.1 |                  86.2 |                   6.3 |                     15.690785 |            190.25149 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            15.797694 |                 15.797694 |     991.202306 |   1.205601 | -2.9 | 11.733333 |      -26.2 |            171.226341 |                171.226341 |                 171.226341 |                 21.732733 |                      0.0 |   149.493608 |              0.0 |        149.493608 |             16.038785 |                   nan |                    500.0 |                   500.0 | 10.502 |  0.0 |  7.049443 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 10.502 | 10.502 |  0.0 |  0.0 | 0.0 |  0.0 | 10.502 | 2.703704 | 0.031293 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |      373.3 |  0.0 | 10.502 |   0.0 |   0.0 |   0.0 |  0.0 | 2.703704 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.031293 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |                15.628821 |      127.660782 |             85.8 |       1.6 |              1013.0 | 0.0 | 0.0 |    17.3 |                  85.8 |                   1.7 |                     15.628821 |           127.660782 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                19.779139 |                     19.779139 |                          0.0 |                      1.251029 |                           1.251029 |            16.970501 |                 16.970501 |     996.029499 |   1.207284 | -2.9 | 18.443733 |     6.8624 |            114.894704 |                114.894704 |                 114.894704 |                 11.219797 |                      0.0 |   103.674907 |              0.0 |        103.674907 |             24.058177 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |   5.02075 | 0.094 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 3.637039 | 0.042095 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   363.5376 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 3.637039 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.042095 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |                15.566134 |      164.401298 |             82.4 |       1.5 |              1016.0 | 0.0 | 0.0 |    18.9 |                  82.4 |                   4.5 |                     15.566134 |           164.401298 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                21.871576 |                     21.871576 |                          0.0 |                      1.365939 |                           1.365939 |            18.022179 |                 18.022179 |     997.977821 |   1.203775 | -2.9 | 15.632162 | -19.607027 |            147.961168 |                147.961168 |                 147.961168 |                 16.044181 |                      0.0 |   131.916987 |              0.0 |        131.916987 |             25.662055 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  6.335399 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 1.940764 | 0.022463 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.244627 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.940764 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.022463 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |                15.502754 |      269.020502 |             77.6 |       2.8 |              1017.0 | 0.0 | 0.0 |    17.8 |                  77.6 |                  12.4 |                     15.502754 |           269.020502 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                20.413369 |                     20.413369 |                          0.0 |                      1.286025 |                           1.286025 |            15.840774 |                 15.840774 |    1001.159226 |    1.21051 | -2.9 | 20.443786 |  15.862716 |            242.118451 |                242.118451 |                 242.118451 |                 35.980407 |                      0.0 |   206.138045 |              0.0 |        206.138045 |              13.74753 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  12.23059 | 0.094 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 1.035613 | 0.011986 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 361.481911 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.035613 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.011986 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |                15.438712 |      287.848667 |             74.9 |       3.0 |              1018.0 | 0.0 | 0.0 |    18.5 |                  74.9 |                  13.9 |                     15.438712 |           287.848667 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                21.331125 |                     21.331125 |                          0.0 |                      1.336407 |                           1.336407 |            15.977013 |                 15.977013 |    1002.022987 |   1.208738 | -2.9 | 15.040124 | -20.759257 |              259.0638 |                  259.0638 |                   259.0638 |                 39.523937 |                      0.0 |   219.539864 |              0.0 |        219.539864 |             12.831028 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 14.271812 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.552615 | 0.006396 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.341169 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.552615 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.006396 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |                15.374034 |      274.674263 |             77.9 |       3.5 |              1018.0 | 0.0 | 0.0 |    19.9 |                  77.9 |                  13.0 |                     15.374034 |           274.674263 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           2.805569 |          3.5 |                23.275348 |                     23.275348 |                          0.0 |                      1.442189 |                           1.442189 |            18.131496 |                 18.131496 |     999.868504 |   1.201995 | -2.9 |  20.18359 |   1.701539 |            247.206837 |                247.206837 |                 247.206837 |                 32.936813 |                      0.0 |   214.270023 |              0.0 |        214.270023 |             10.998024 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 14.791542 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.294881 | 0.003413 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.739629 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.294881 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003413 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |                 15.30875 |      284.047892 |             73.5 |       3.8 |              1019.0 | 0.0 | 0.0 |    20.8 |                  73.5 |                  13.8 |                      15.30875 |           284.047892 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            18.084931 |                 18.084931 |    1000.915069 |   1.199521 | -2.9 | 18.858347 | -11.649921 |            255.643103 |                255.643103 |                 255.643103 |                  35.31351 |                      0.0 |   220.329593 |              0.0 |        220.329593 |             10.129759 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 18.203372 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.157352 | 0.001821 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  383.48955 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.157352 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001821 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |                15.242885 |      262.971263 |             80.1 |       3.3 |              1019.0 | 0.0 | 0.0 |    20.9 |                  80.1 |                  12.3 |                     15.242885 |           262.971263 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           2.645251 |          3.3 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            19.830469 |                 19.830469 |     999.169531 |   1.198331 | -2.9 | 21.378324 |   2.869942 |            236.674137 |                236.674137 |                 236.674137 |                 28.392598 |                      0.0 |   208.281539 |              0.0 |        208.281539 |             11.664571 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 13.456656 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.083965 | 0.000972 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.719608 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.083965 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000972 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |                15.176467 |      272.256414 |             72.9 |       1.8 |              1019.0 | 0.0 | 0.0 |    22.3 |                  72.9 |                  13.1 |                     15.176467 |           272.256414 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 26.96965 |                      26.96965 |                          0.0 |                      1.639967 |                           1.639967 |            19.660875 |                 19.660875 |     999.339125 |   1.192729 | -2.9 |  19.71658 | -15.500518 |            245.030772 |                245.030772 |                 245.030772 |                 31.114889 |                      0.0 |   213.915883 |              0.0 |        213.915883 |             21.385046 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 11.840372 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.044804 | 0.000519 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 390.320126 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.044804 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000519 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |                15.109521 |      267.019218 |             68.5 |       1.9 |              1020.0 | 0.0 | 0.0 |    23.0 |                  68.5 |                  12.8 |                     15.109521 |           267.019218 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                28.139149 |                     28.139149 |                          0.0 |                      1.701779 |                           1.701779 |            19.275317 |                 19.275317 |    1000.724683 |   1.191257 | -2.9 |  23.34553 |   2.073177 |            240.317296 |                240.317296 |                 240.317296 |                 32.010643 |                      0.0 |   208.306653 |              0.0 |        208.306653 |             20.259517 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 13.388635 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.023908 | 0.000277 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.346949 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.023908 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000277 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |                15.042071 |      269.626547 |             68.6 |       3.0 |              1021.0 | 0.0 | 0.0 |    23.1 |                  68.6 |                  13.1 |                     15.042071 |           269.626547 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                28.309776 |                     28.309776 |                          0.0 |                      1.710768 |                           1.710768 |            19.420506 |                 19.420506 |    1001.579494 |   1.191966 | -2.9 | 21.913255 |  -7.120473 |            242.663892 |                242.663892 |                 242.663892 |                 32.400014 |                      0.0 |   210.263878 |              0.0 |        210.263878 |             12.831028 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 18.009707 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.012758 | 0.000148 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 389.567421 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.012758 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000148 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |                14.974141 |      270.878291 |             66.0 |       2.5 |              1022.0 | 0.0 | 0.0 |    23.2 |                  66.0 |                  13.3 |                     14.974141 |           270.878291 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |             18.79766 |                  18.79766 |     1003.20234 |   1.193016 | -2.9 | 23.128751 |  -0.427496 |            243.790462 |                243.790462 |                 243.790462 |                 34.565783 |                      0.0 |   209.224678 |              0.0 |        209.224678 |             15.397233 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 16.851676 | 0.094 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.006808 | 0.000079 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.094917 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006808 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000079 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |                14.905755 |      261.697164 |             69.0 |       1.6 |              1019.0 | 0.0 | 0.0 |    23.2 |                  69.0 |                  12.7 |                     14.905755 |           261.697164 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           1.282546 |          1.6 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |              19.6521 |                   19.6521 |       999.3479 |    1.18911 | -2.9 | 22.416669 |  -4.699983 |            235.527447 |                235.527447 |                 235.527447 |                  31.34697 |                      0.0 |   204.180477 |              0.0 |        204.180477 |             24.058177 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 11.953889 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.003633 | 0.000042 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   388.8949 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003633 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000042 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |                14.836934 |      227.962116 |             75.6 |       2.7 |              1015.0 | 0.0 | 0.0 |    21.7 |                  75.6 |                  10.2 |                     14.836934 |           227.962116 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 26.00118 |                      26.00118 |                          0.0 |                        1.5885 |                             1.5885 |            19.656892 |                 19.656892 |     995.343108 |   1.190432 | -2.9 | 22.935065 |   7.410388 |            205.165904 |                205.165904 |                 205.165904 |                 25.918437 |                      0.0 |   179.247467 |              0.0 |        179.247467 |             14.256697 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 13.005745 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.001938 | 0.000022 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 378.584513 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001938 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000022 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |                  14.7677 |      216.356321 |             81.4 |       3.2 |              1016.0 | 0.0 | 0.0 |    20.8 |                  81.4 |                   9.4 |                       14.7677 |           216.356321 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           2.565092 |          3.2 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            20.028753 |                 20.028753 |     995.971247 |   1.195095 | -2.9 | 19.965673 |  -5.005962 |            194.720689 |                194.720689 |                 194.720689 |                 23.397902 |                      0.0 |   171.322787 |              0.0 |        171.322787 |             12.029088 |                   nan |                    500.0 |                   500.0 |  0.538 |  0.0 | 11.728612 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.538 |  0.538 |  0.0 |  0.0 | 0.0 |  0.0 |  0.538 | 0.139541 | 0.001615 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.690475 |  0.0 |  0.538 |   0.0 |   0.0 |   0.0 |  0.0 | 0.139541 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001615 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |                14.698075 |      226.637904 |             79.0 |       1.3 |              1019.0 | 0.0 | 0.0 |    19.5 |                  79.0 |                  10.3 |                     14.698075 |           226.637904 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                22.704676 |                     22.704676 |                          0.0 |                      1.411269 |                           1.411269 |            17.936694 |                 17.936694 |    1001.063306 |   1.204916 | -2.9 |  20.57219 |   6.433141 |            203.974113 |                203.974113 |                 203.974113 |                 28.794151 |                      0.0 |   175.179963 |              0.0 |        175.179963 |             29.610064 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  7.560661 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.186871 | 0.002163 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 371.357334 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.186871 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.002163 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |                14.628078 |      235.528503 |             75.9 |       1.9 |              1019.0 | 0.0 | 0.0 |    21.0 |                  75.9 |                  11.1 |                     14.628078 |           235.528503 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                24.909749 |                     24.909749 |                          0.0 |                      1.530181 |                           1.530181 |            18.906499 |                 18.906499 |    1000.093501 |   1.198338 | -2.9 | 17.884246 | -18.694526 |            211.975652 |                211.975652 |                 211.975652 |                 29.177702 |                      0.0 |    182.79795 |              0.0 |         182.79795 |             20.259517 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 10.391615 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.099717 | 0.001154 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.151861 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.099717 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001154 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |                14.557728 |      232.849428 |             70.0 |       2.0 |              1018.0 | 0.0 | 0.0 |    21.6 |                  70.0 |                  11.0 |                     14.557728 |           232.849428 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                25.842752 |                     25.842752 |                          0.0 |                      1.580055 |                           1.580055 |            18.089927 |                 18.089927 |     999.910073 |   1.195081 | -2.9 | 22.433069 |   4.998415 |            209.564486 |                209.564486 |                 209.564486 |                 31.281288 |                      0.0 |   178.283198 |              0.0 |        178.283198 |             19.246541 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 12.298412 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 |  0.05321 | 0.000616 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.253445 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.05321 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000616 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |                14.487046 |       199.71349 |             70.3 |       2.6 |              1019.0 | 0.0 | 0.0 |    21.1 |                  70.3 |                   8.5 |                     14.487046 |            199.71349 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            17.619414 |                 17.619414 |    1001.380586 |   1.198506 | -2.9 | 20.158326 |  -5.650046 |            179.742141 |                179.742141 |                 179.742141 |                 26.711296 |                      0.0 |   153.030845 |              0.0 |        153.030845 |             14.805032 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 13.758946 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.028393 | 0.000329 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 382.003492 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.028393 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000329 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |                14.416048 |      233.771902 |             69.0 |       3.6 |              1019.0 | 0.0 | 0.0 |    22.4 |                  69.0 |                  11.3 |                     14.416048 |           233.771902 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           2.885728 |          3.6 |                27.134082 |                     27.134082 |                          0.0 |                      1.648679 |                           1.648679 |            18.722517 |                 18.722517 |    1000.277483 |   1.192743 | -2.9 | 20.950339 |  -8.697966 |            210.394712 |                210.394712 |                 210.394712 |                 31.116622 |                      0.0 |    179.27809 |              0.0 |         179.27809 |             10.692523 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 19.233312 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.015151 | 0.000175 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.801458 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.015151 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000175 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |                14.344752 |      246.121286 |             62.1 |       2.5 |              1017.0 | 0.0 | 0.0 |    22.5 |                  62.1 |                  12.4 |                     14.344752 |           246.121286 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           2.003978 |          2.5 |                27.299387 |                     27.299387 |                          0.0 |                      1.657431 |                           1.657431 |             16.95292 |                  16.95292 |     1000.04708 |   1.190771 | -2.9 | 22.620153 |    0.72092 |            221.509157 |                221.509157 |                 221.509157 |                 38.297604 |                      0.0 |   183.211553 |              0.0 |        183.211553 |             15.397233 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  17.26053 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.008085 | 0.000094 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.180539 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.008085 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000094 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |                14.273175 |        90.58551 |             84.5 |       2.1 |              1015.0 | 0.0 | 0.0 |    19.6 |                  84.5 |                   0.1 |                     14.273175 |             90.58551 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                22.846186 |                     22.846186 |                          0.0 |                      1.418946 |                           1.418946 |            19.305027 |                 19.305027 |     995.694973 |   1.199129 | -2.9 | 21.577328 |  11.863971 |             81.526959 |                 81.526959 |                  81.526959 |                   7.02118 |                      0.0 |    74.505779 |              0.0 |         74.505779 |             18.330039 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  5.830318 |  1.88 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.004314 |  0.00005 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 369.416568 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.004314 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.00005 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |                14.201333 |      171.944993 |             83.6 |       1.5 |              1015.0 | 0.0 | 0.0 |    20.3 |                  83.6 |                   6.7 |                     14.201333 |           171.944993 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                23.858503 |                     23.858503 |                          0.0 |                      1.473678 |                           1.473678 |            19.945708 |                 19.945708 |     995.054292 |   1.195981 | -2.9 | 17.325305 | -17.848171 |            154.750494 |                154.750494 |                 154.750494 |                 18.994889 |                      0.0 |   135.755605 |              0.0 |        135.755605 |             25.662055 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  6.371341 | 0.282 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.002302 | 0.000027 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.364739 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002302 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000027 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |                14.129243 |      216.549813 |             76.5 |       2.2 |              1011.0 | 0.0 | 0.0 |    24.1 |                  76.5 |                  10.4 |                     14.129243 |           216.549813 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                30.066256 |                     30.066256 |                          0.0 |                       1.80287 |                            1.80287 |            23.000686 |                 23.000686 |     987.999314 |   1.174651 | -2.9 | 21.630378 | -14.817732 |            194.894831 |                194.894831 |                 194.894831 |                 21.629737 |                      0.0 |   173.265095 |              0.0 |        173.265095 |             17.496856 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 | 11.503853 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.001228 | 0.000014 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  396.28247 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001228 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000014 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |                14.056918 |       166.91683 |             76.6 |       1.9 |              1008.0 | 0.0 | 0.0 |    23.7 |                  76.6 |                   6.5 |                     14.056918 |            166.91683 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                29.352596 |                     29.352596 |                          0.0 |                      1.765541 |                           1.765541 |            22.484088 |                 22.484088 |     985.515912 |   1.172942 | -2.9 | 25.062685 |   8.176109 |            150.225147 |                150.225147 |                 150.225147 |                 16.222254 |                      0.0 |   134.002892 |              0.0 |        134.002892 |             20.259517 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  9.461664 | 0.094 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 0.000655 | 0.000008 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.206362 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000655 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000008 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |                13.984374 |      145.985611 |             84.2 |       2.3 |              1006.0 | 0.0 | 0.0 |    21.1 |                  84.2 |                   4.9 |                     13.984374 |           145.985611 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            21.103195 |                 21.103195 |     984.896805 |   1.181557 | -2.9 | 21.872765 |   4.636593 |             131.38705 |                 131.38705 |                  131.38705 |                 14.602883 |                      0.0 |   116.784167 |              0.0 |        116.784167 |             16.736123 |                   nan |                    500.0 |                   500.0 |   7.87 |  0.0 |  7.472407 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   7.87 |   7.87 |  0.0 |  0.0 | 0.0 |  0.0 |   7.87 | 2.026454 | 0.023454 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.669769 |  0.0 |   7.87 |   0.0 |   0.0 |   0.0 |  0.0 | 2.026454 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.023454 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |                13.911624 |       165.54389 |             85.8 |       2.1 |              1005.0 | 0.0 | 0.0 |    20.9 |                  85.8 |                   6.6 |                     13.911624 |            165.54389 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            21.241626 |                 21.241626 |     983.758374 |   1.181114 | -2.9 | 19.702227 |   -7.18664 |            148.989501 |                148.989501 |                 148.989501 |                 17.349121 |                      0.0 |    131.64038 |              0.0 |         131.64038 |             18.330039 |                   nan |                    500.0 |                   500.0 |   4.11 |  0.0 |   6.96061 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   4.11 |   4.11 |  0.0 |  0.0 | 0.0 |  0.0 |   4.11 |  3.78382 | 0.043794 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 381.956408 |  0.0 |   4.11 |   0.0 |   0.0 |   0.0 |  0.0 |  3.78382 |      0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.043794 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |                13.838682 |       87.693346 |             86.5 |       2.6 |              1000.0 | 0.0 | 0.0 |    20.2 |                  86.5 |                   0.3 |                     13.838682 |            87.693346 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                23.711531 |                     23.711531 |                          0.0 |                      1.465752 |                           1.465752 |            20.510475 |                 20.510475 |     979.489525 |   1.178323 | -2.9 | 20.936779 |   4.420674 |             78.924011 |                 78.924011 |                  78.924011 |                  6.818024 |                      0.0 |    72.105987 |              0.0 |         72.105987 |             14.805032 |                   nan |                    500.0 |                   500.0 |   12.1 |  0.0 |  6.075004 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   12.1 |   12.1 |  0.0 |  0.0 | 0.0 |  0.0 |   12.1 | 6.030624 | 0.069799 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.635735 |  0.0 |   12.0 |   0.1 |   0.0 |   0.0 |  0.0 | 5.967201 | 0.063424 |   0.0 |   0.0 |  0.0 |   0.0 | 0.069799 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |                13.765561 |       84.356885 |             89.6 |       2.6 |              1004.0 | 0.0 | 0.0 |    16.1 |                  89.6 |                   0.1 |                     13.765561 |            84.356885 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           2.084137 |          2.6 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            16.420805 |                 16.420805 |     987.579195 |   1.201704 | -2.9 | 18.828425 |   16.37055 |             75.921197 |                 75.921197 |                  75.921197 |                  8.199212 |                      0.0 |    67.721985 |              0.0 |         67.721985 |             14.805032 |                   nan |                    500.0 |                   500.0 |  6.648 |  0.0 |  4.534094 |   2.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  6.648 |  6.648 |  0.0 |  0.0 | 0.0 |  0.0 |  6.648 | 7.436588 | 0.086072 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 355.365185 |  0.0 |  6.648 |   0.0 |   0.0 |   0.0 |  0.0 | 7.402977 | 0.033611 |   0.0 |   0.0 |  0.0 |   0.0 | 0.086072 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |                13.692273 |      142.309202 |             78.3 |       2.1 |              1011.0 | 0.0 | 0.0 |    17.0 |                  78.3 |                   5.0 |                     13.692273 |           142.309202 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                19.406929 |                     19.406929 |                          0.0 |                      1.230421 |                           1.230421 |            15.195625 |                 15.195625 |     995.804375 |   1.206937 | -2.9 | 13.278507 |  -22.32896 |            128.078282 |                128.078282 |                 128.078282 |                 21.775852 |                      0.0 |    106.30243 |              0.0 |         106.30243 |             18.330039 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  7.752223 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 5.342084 |  0.06183 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.794145 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 5.339359 | 0.002725 |   0.0 |   0.0 |  0.0 |   0.0 |  0.06183 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |                13.618831 |      126.722078 |             78.5 |       2.1 |              1014.0 | 0.0 | 0.0 |    18.2 |                  78.5 |                   3.8 |                     13.618831 |           126.722078 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |           1.683342 |          2.1 |                20.933479 |                     20.933479 |                          0.0 |                      1.314615 |                           1.314615 |            16.432781 |                 16.432781 |     997.567219 |   1.204994 | -2.9 | 18.874047 |   4.044283 |            114.049871 |                114.049871 |                 114.049871 |                 17.491881 |                      0.0 |     96.55799 |              0.0 |          96.55799 |             18.330039 |                   nan |                    500.0 |                   500.0 |    0.0 |  0.0 |  7.710948 |   0.0 | 0.0 | 0.0 |   0.0 |          nan |          0.1 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |    0.0 |  0.0 |  0.0 | 0.0 |  0.0 |    0.0 | 2.849362 | 0.032979 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 367.849862 |  0.0 |    0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 2.849141 | 0.000221 |   0.0 |   0.0 |  0.0 |   0.0 | 0.032979 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_v3_nadelw_summer_daily:

conifers (summer)
-----------------

The following tests focus on functionalities of |lland_v3| not implemented by
application model |lland_v1|.  We start with the land-use type |NADELW|.  For
|lland_v1|, there is no structural difference between |NADELW| and, for example,
|ACKER|.  Different simulation results only are based solely on different
parameterisations.  In contrast, |lland_v3| handles |NADELW| unlike all other land use
types (even unlike |LAUBW| and |MISCHW|) regarding the calculation of the surface
resistance, relevant for the Penman-Monteith equation (see method
|Calc_LanduseSurfaceResistance_V1| for more details).  The calculated soil
evapotranspiration (|EvB|) is similar to the one calculated with the same input data
for land-use type |ACKER| in the integration test :ref:`lland_v3_acker_summer_daily`:

.. integration-test::

    >>> lnk(NADELW)
    >>> surfaceresistance.nadelw = 56.0
    >>> cropheight.nadelw = 10.0
    >>> albedo.nadelw = 0.12
    >>> lai.nadelw = 11.0
    >>> conditions_nadelw_summer = test(
    ...     "lland_v3_nadelw_summer_daily",
    ...     axis1=(inputs.nied, fluxes.qah, states.bowa),
    ...     get_conditions="1997-08-03")
    |       date | nied | teml | sunshineduration | possiblesunshineduration | globalradiation | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration | dailypossiblesunshineduration | dailyglobalradiation |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair |    g |        tz |         wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance |   nbes | sbes |      evpo |   evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg |   wada |      qdb | qib1 | qib2 | qbb |     qkap |     qdgz |       qah |        qa | inzp | wats | waes | esnow | taus |       ebdn |      bowa |    qdgz1 | qdgz2 | qigz1 | qigz2 |      qbgz |    qdga1 | qdga2 | qiga1 | qiga2 |      qbga | inlet |    outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-01-08 | 13.3 | 16.1 |              6.3 |                15.690785 |       190.25149 |             86.2 |       2.4 |              1007.0 | 0.0 | 0.0 |    16.1 |                  86.2 |                   6.3 |                     15.690785 |            190.25149 | 12.502 | 16.1 |    16.1 |    1.923819 |         1.923819 |           0.851977 |          2.4 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            15.797694 |                 15.797694 |     991.202306 |   1.205601 | -2.9 | 11.733333 |      -26.2 |            167.421311 |                 31.092529 |                 167.421311 |                 21.732733 |                      0.0 |   145.688578 |              0.0 |        145.688578 |             39.166667 |                 100.0 |               211.364559 |              122.747678 | 10.302 |  0.0 |   4.68629 |   2.2 | 1.110342 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 10.302 | 0.657815 |  0.0 |  0.0 | 0.0 |      0.0 | 0.657815 |  0.169352 |   0.00196 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |      373.3 | 80.533843 | 0.657815 |   0.0 |   0.0 |   0.0 |       0.0 | 0.169352 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |   0.00196 |
    | 1997-02-08 |  0.1 | 17.3 |              1.7 |                15.628821 |      127.660782 |             85.8 |       1.6 |              1013.0 | 0.0 | 0.0 |    17.3 |                  85.8 |                   1.7 |                     15.628821 |           127.660782 |  0.094 | 17.3 |    17.3 |    1.282546 |         1.282546 |           0.567985 |          1.6 |                19.779139 |                     19.779139 |                          0.0 |                      1.251029 |                           1.251029 |            16.970501 |                 16.970501 |     996.029499 |   1.207284 | -2.9 | 18.443733 |     6.8624 |            112.341488 |                 20.863419 |                 112.341488 |                 11.219797 |                      0.0 |   101.121692 |              0.0 |        101.121692 |                 58.75 |                 100.0 |               181.457771 |               115.35051 |    0.0 |  0.0 |  3.311865 | 0.094 | 1.809024 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.227814 |  0.002637 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   363.5376 | 78.724819 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.227814 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.002637 |
    | 1997-03-08 |  0.0 | 18.9 |              4.5 |                15.566134 |      164.401298 |             82.4 |       1.5 |              1016.0 | 0.0 | 0.0 |    18.9 |                  82.4 |                   4.5 |                     15.566134 |           164.401298 |    0.0 | 18.9 |    18.9 |    1.202387 |         1.202387 |           0.532486 |          1.5 |                21.871576 |                     21.871576 |                          0.0 |                      1.365939 |                           1.365939 |            18.022179 |                 18.022179 |     997.977821 |   1.203775 | -2.9 | 15.632162 | -19.607027 |            144.673142 |                 26.867869 |                 144.673142 |                 16.044181 |                      0.0 |   128.628961 |              0.0 |        128.628961 |             62.666667 |                 100.0 |               184.734223 |              115.873858 |    0.0 |  0.0 |  4.247909 |   0.0 | 2.494966 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.121564 |  0.001407 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.244627 | 76.229853 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.121564 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.001407 |
    | 1997-04-08 |  0.1 | 17.8 |             12.4 |                15.502754 |      269.020502 |             77.6 |       2.8 |              1017.0 | 0.0 | 0.0 |    17.8 |                  77.6 |                  12.4 |                     15.502754 |           269.020502 |  0.094 | 17.8 |    17.8 |    2.244455 |         2.244455 |           0.993973 |          2.8 |                20.413369 |                     20.413369 |                          0.0 |                      1.286025 |                           1.286025 |            15.840774 |                 15.840774 |    1001.159226 |    1.21051 | -2.9 | 20.443786 |  15.862716 |            236.738041 |                 43.965636 |                 236.738041 |                 35.980407 |                      0.0 |   200.757635 |              0.0 |        200.757635 |             33.571429 |                 100.0 |               209.622013 |              121.261638 |    0.0 |  0.0 |  7.655384 | 0.094 | 3.243031 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.064868 |  0.000751 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 361.481911 | 72.986823 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.064868 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000751 |
    | 1997-05-08 |  0.0 | 18.5 |             13.9 |                15.438712 |      287.848667 |             74.9 |       3.0 |              1018.0 | 0.0 | 0.0 |    18.5 |                  74.9 |                  13.9 |                     15.438712 |           287.848667 |    0.0 | 18.5 |    18.5 |    2.404774 |         2.404774 |           1.064971 |          3.0 |                21.331125 |                     21.331125 |                          0.0 |                      1.336407 |                           1.336407 |            15.977013 |                 15.977013 |    1002.022987 |   1.208738 | -2.9 | 15.040124 | -20.759257 |            253.306827 |                 47.042696 |                 253.306827 |                 39.523937 |                      0.0 |   213.782891 |              0.0 |        213.782891 |             31.333333 |                 100.0 |               223.523903 |               123.68636 |    0.0 |  0.0 |  8.731356 |   0.0 | 3.618047 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.034614 |  0.000401 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.341169 | 69.368775 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.034614 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000401 |
    | 1997-06-08 |  0.0 | 19.9 |             13.0 |                15.374034 |      274.674263 |             77.9 |       3.5 |              1018.0 | 0.0 | 0.0 |    19.9 |                  77.9 |                  13.0 |                     15.374034 |           274.674263 |    0.0 | 19.9 |    19.9 |    2.805569 |         2.805569 |           1.242466 |          3.5 |                23.275348 |                     23.275348 |                          0.0 |                      1.442189 |                           1.442189 |            18.131496 |                 18.131496 |     999.868504 |   1.201995 | -2.9 |  20.18359 |   1.701539 |            241.713352 |                 44.889622 |                 241.713352 |                 32.936813 |                      0.0 |   208.776538 |              0.0 |        208.776538 |             26.857143 |                 100.0 |               217.805293 |              122.172051 |    0.0 |  0.0 |   8.96754 |   0.0 | 3.538533 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.018471 |  0.000214 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.739629 | 65.830242 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.018471 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000214 |
    | 1997-07-08 |  0.0 | 20.8 |             13.8 |                 15.30875 |      284.047892 |             73.5 |       3.8 |              1019.0 | 0.0 | 0.0 |    20.8 |                  73.5 |                  13.8 |                      15.30875 |           284.047892 |    0.0 | 20.8 |    20.8 |    3.046047 |         3.046047 |           1.348964 |          3.8 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            18.084931 |                 18.084931 |    1000.915069 |   1.199521 | -2.9 | 18.858347 | -11.649921 |            249.962145 |                 46.421541 |                 249.962145 |                  35.31351 |                      0.0 |   214.648635 |              0.0 |        214.648635 |             24.736842 |                 100.0 |               249.895771 |              127.503409 |    0.0 |  0.0 | 10.546373 |   0.0 | 3.943085 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.009856 |  0.000114 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  383.48955 | 61.887157 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.009856 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000114 |
    | 1997-08-08 |  0.0 | 20.9 |             12.3 |                15.242885 |      262.971263 |             80.1 |       3.3 |              1019.0 | 0.0 | 0.0 |    20.9 |                  80.1 |                  12.3 |                     15.242885 |           262.971263 |    0.0 | 20.9 |    20.9 |    2.645251 |         2.645251 |           1.171468 |          3.3 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            19.830469 |                 19.830469 |     999.169531 |   1.198331 | -2.9 | 21.378324 |   2.869942 |            231.414712 |                 42.977018 |                 231.414712 |                 28.392598 |                      0.0 |   203.022114 |              0.0 |        203.022114 |             28.484848 |                 100.0 |               234.558928 |              124.470477 |    0.0 |  0.0 |  8.374876 |   0.0 | 3.447369 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.005259 |  0.000061 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.719608 | 58.439788 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.005259 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000061 |
    | 1997-09-08 |  0.0 | 22.3 |             13.1 |                15.176467 |      272.256414 |             72.9 |       1.8 |              1019.0 | 0.0 | 0.0 |    22.3 |                  72.9 |                  13.1 |                     15.176467 |           272.256414 |    0.0 | 22.3 |    22.3 |    1.442864 |         1.442864 |           0.638983 |          1.8 |                 26.96965 |                      26.96965 |                          0.0 |                      1.639967 |                           1.639967 |            19.660875 |                 19.660875 |     999.339125 |   1.192729 | -2.9 |  19.71658 | -15.500518 |            239.585644 |                 44.494477 |                 239.585644 |                 31.114889 |                      0.0 |   208.470755 |              0.0 |        208.470755 |             52.222222 |                 100.0 |               290.491428 |              132.398577 |    0.0 |  0.0 |  7.801219 |   0.0 | 4.238236 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.002806 |  0.000032 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 390.320126 | 54.201552 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.002806 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000032 |
    | 1997-10-08 |  0.0 | 23.0 |             12.8 |                15.109521 |      267.019218 |             68.5 |       1.9 |              1020.0 | 0.0 | 0.0 |    23.0 |                  68.5 |                  12.8 |                     15.109521 |           267.019218 |    0.0 | 23.0 |    23.0 |    1.523023 |         1.523023 |           0.674482 |          1.9 |                28.139149 |                     28.139149 |                          0.0 |                      1.701779 |                           1.701779 |            19.275317 |                 19.275317 |    1000.724683 |   1.191257 | -2.9 |  23.34553 |   2.073177 |            234.976912 |                 43.638569 |                 234.976912 |                 32.010643 |                      0.0 |   202.966269 |              0.0 |        202.966269 |             49.473684 |                 100.0 |               348.464183 |              138.317772 |    0.0 |  0.0 |  8.457253 |   0.0 | 4.453704 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.001498 |  0.000017 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.346949 | 49.747848 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.001498 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000017 |
    | 1997-11-08 |  0.0 | 23.1 |             13.1 |                15.042071 |      269.626547 |             68.6 |       3.0 |              1021.0 | 0.0 | 0.0 |    23.1 |                  68.6 |                  13.1 |                     15.042071 |           269.626547 |    0.0 | 23.1 |    23.1 |    2.404774 |         2.404774 |           1.064971 |          3.0 |                28.309776 |                     28.309776 |                          0.0 |                      1.710768 |                           1.710768 |            19.420506 |                 19.420506 |    1001.579494 |   1.191966 | -2.9 | 21.913255 |  -7.120473 |            237.271361 |                 44.064681 |                 237.271361 |                 32.400014 |                      0.0 |   204.871347 |              0.0 |        204.871347 |             31.333333 |                 100.0 |               368.722677 |              139.569846 |    0.0 |  0.0 | 10.478603 |   0.0 | 4.442272 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.000799 |  0.000009 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 389.567421 | 45.305576 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000799 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000009 |
    | 1997-12-08 |  0.1 | 23.2 |             13.3 |                14.974141 |      270.878291 |             66.0 |       2.5 |              1022.0 | 0.0 | 0.0 |    23.2 |                  66.0 |                  13.3 |                     14.974141 |           270.878291 |  0.094 | 23.2 |    23.2 |    2.003978 |         2.003978 |           0.887476 |          2.5 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |             18.79766 |                  18.79766 |     1003.20234 |   1.193016 | -2.9 | 23.128751 |  -0.427496 |            238.372896 |                 44.269252 |                 238.372896 |                 34.565783 |                      0.0 |   203.807113 |              0.0 |        203.807113 |                  37.6 |                 100.0 |               419.707371 |              142.937668 |    0.0 |  0.0 |  9.985446 | 0.094 |  4.54598 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.000426 |  0.000005 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.094917 | 40.759596 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000426 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000005 |
    | 1997-13-08 |  0.0 | 23.2 |             12.7 |                14.905755 |      261.697164 |             69.0 |       1.6 |              1019.0 | 0.0 | 0.0 |    23.2 |                  69.0 |                  12.7 |                     14.905755 |           261.697164 |    0.0 | 23.2 |    23.2 |    1.282546 |         1.282546 |           0.567985 |          1.6 |                28.481304 |                     28.481304 |                          0.0 |                      1.719796 |                           1.719796 |              19.6521 |                   19.6521 |       999.3479 |    1.18911 | -2.9 | 22.416669 |  -4.699983 |            230.293504 |                 42.768794 |                 230.293504 |                  31.34697 |                      0.0 |   198.946534 |              0.0 |        198.946534 |                 58.75 |                 100.0 |               411.248888 |              141.648997 |    0.0 |  0.0 |  7.775446 |   0.0 | 4.347763 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.000228 |  0.000003 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |   388.8949 | 36.411834 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000228 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000003 |
    | 1997-14-08 |  0.0 | 21.7 |             10.2 |                14.836934 |      227.962116 |             75.6 |       2.7 |              1015.0 | 0.0 | 0.0 |    21.7 |                  75.6 |                  10.2 |                     14.836934 |           227.962116 |    0.0 | 21.7 |    21.7 |    2.164296 |         2.164296 |           0.958474 |          2.7 |                 26.00118 |                      26.00118 |                          0.0 |                        1.5885 |                             1.5885 |            19.656892 |                 19.656892 |     995.343108 |   1.190432 | -2.9 | 22.935065 |   7.410388 |            200.606662 |                 37.255523 |                 200.606662 |                 25.918437 |                      0.0 |   174.688225 |              0.0 |        174.688225 |             34.814815 |                 100.0 |               357.544238 |               136.70684 |    0.0 |  0.0 |  7.856911 |   0.0 | 3.459634 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.000121 |  0.000001 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 378.584513 | 32.952199 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.000121 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000001 |
    | 1997-15-08 |  2.7 | 20.8 |              9.4 |                  14.7677 |      216.356321 |             81.4 |       3.2 |              1016.0 | 0.0 | 0.0 |    20.8 |                  81.4 |                   9.4 |                       14.7677 |           216.356321 |  2.538 | 20.8 |    20.8 |    2.565092 |         2.565092 |           1.135969 |          3.2 |                24.605348 |                     24.605348 |                          0.0 |                      1.513854 |                           1.513854 |            20.028753 |                 20.028753 |     995.971247 |   1.195095 | -2.9 | 19.965673 |  -5.005962 |            190.393563 |                 35.358805 |                 190.393563 |                 23.397902 |                      0.0 |   166.995661 |              0.0 |        166.995661 |                29.375 |                 100.0 |               333.903254 |              133.915021 |  0.338 |  0.0 |  7.161899 |   2.2 | 1.984444 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.338 | 0.008728 |  0.0 |  0.0 | 0.0 |      0.0 | 0.008728 |  0.002312 |  0.000027 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 380.690475 | 31.297027 | 0.008728 |   0.0 |   0.0 |   0.0 |       0.0 | 0.002312 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000027 |
    | 1997-16-08 |  0.0 | 19.5 |             10.3 |                14.698075 |      226.637904 |             79.0 |       1.3 |              1019.0 | 0.0 | 0.0 |    19.5 |                  79.0 |                  10.3 |                     14.698075 |           226.637904 |    0.0 | 19.5 |    19.5 |    1.042069 |         1.042069 |           0.461488 |          1.3 |                22.704676 |                     22.704676 |                          0.0 |                      1.411269 |                           1.411269 |            17.936694 |                 17.936694 |    1001.063306 |   1.204916 | -2.9 |  20.57219 |   6.433141 |            199.441355 |                 37.039109 |                 199.441355 |                 28.794151 |                      0.0 |   170.647205 |              0.0 |        170.647205 |             72.307692 |                 100.0 |               354.689619 |              135.231167 |    0.0 |  0.0 |  5.274133 |   0.0 |  3.08192 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.003057 |  0.000035 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 371.357334 | 28.215107 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.003057 |   0.0 |   0.0 |   0.0 |       0.0 |   0.0 |  0.000035 |
    | 1997-17-08 |  0.0 | 21.0 |             11.1 |                14.628078 |      235.528503 |             75.9 |       1.9 |              1019.0 | 0.0 | 0.0 |    21.0 |                  75.9 |                  11.1 |                     14.628078 |           235.528503 |    0.0 | 21.0 |    21.0 |    1.523023 |         1.523023 |           0.674482 |          1.9 |                24.909749 |                     24.909749 |                          0.0 |                      1.530181 |                           1.530181 |            18.906499 |                 18.906499 |    1000.093501 |   1.198338 | -2.9 | 17.884246 | -18.694526 |            207.265082 |                 38.492087 |                 207.265082 |                 29.177702 |                      0.0 |    178.08738 |              0.0 |         178.08738 |             49.473684 |                 100.0 |               400.417627 |              138.202815 |    0.0 |  0.0 |  6.725408 |   0.0 | 3.429476 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.166828 |      0.0 |  0.001332 |  0.000015 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.151861 |  24.95246 |      0.0 |   0.0 |   0.0 |   0.0 | -0.166828 | 0.001631 |   0.0 |   0.0 |   0.0 | -0.000299 |   0.0 |  0.000015 |
    | 1997-18-08 |  0.0 | 21.6 |             11.0 |                14.557728 |      232.849428 |             70.0 |       2.0 |              1018.0 | 0.0 | 0.0 |    21.6 |                  70.0 |                  11.0 |                     14.557728 |           232.849428 |    0.0 | 21.6 |    21.6 |    1.603182 |         1.603182 |           0.709981 |          2.0 |                25.842752 |                     25.842752 |                          0.0 |                      1.580055 |                           1.580055 |            18.089927 |                 18.089927 |     999.910073 |   1.195081 | -2.9 | 22.433069 |   4.998415 |            204.907497 |                 38.054249 |                 204.907497 |                 31.281288 |                      0.0 |   173.626209 |              0.0 |        173.626209 |                  47.0 |                 100.0 |               491.312196 |              142.949327 |    0.0 |  0.0 |  7.539559 |   0.0 | 3.733791 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.369556 |      0.0 | -0.000389 | -0.000004 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 379.253445 | 21.588224 |      0.0 |   0.0 |   0.0 |   0.0 | -0.369556 |  0.00087 |   0.0 |   0.0 |   0.0 | -0.001259 |   0.0 | -0.000004 |
    | 1997-19-08 |  0.0 | 21.1 |              8.5 |                14.487046 |       199.71349 |             70.3 |       2.6 |              1019.0 | 0.0 | 0.0 |    21.1 |                  70.3 |                   8.5 |                     14.487046 |            199.71349 |    0.0 | 21.1 |    21.1 |    2.084137 |         2.084137 |           0.922975 |          2.6 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            17.619414 |                 17.619414 |    1001.380586 |   1.198506 | -2.9 | 20.158326 |  -5.650046 |            175.747871 |                  32.63889 |                 175.747871 |                 26.711296 |                      0.0 |   149.036575 |              0.0 |        149.036575 |             36.153846 |                 100.0 |               525.371793 |               143.81531 |    0.0 |  0.0 |  7.831862 |   0.0 | 3.377664 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.578596 |      0.0 | -0.002489 | -0.000029 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 382.003492 | 18.789156 |      0.0 |   0.0 |   0.0 |   0.0 | -0.578596 | 0.000465 |   0.0 |   0.0 |   0.0 | -0.002953 |   0.0 | -0.000029 |
    | 1997-20-08 |  0.0 | 22.4 |             11.3 |                14.416048 |      233.771902 |             69.0 |       3.6 |              1019.0 | 0.0 | 0.0 |    22.4 |                  69.0 |                  11.3 |                     14.416048 |           233.771902 |    0.0 | 22.4 |    22.4 |    2.885728 |         2.885728 |           1.277965 |          3.6 |                27.134082 |                     27.134082 |                          0.0 |                      1.648679 |                           1.648679 |            18.722517 |                 18.722517 |    1000.277483 |   1.192743 | -2.9 | 20.950339 |  -8.697966 |            205.719274 |                 38.205008 |                 205.719274 |                 31.116622 |                      0.0 |   174.602652 |              0.0 |        174.602652 |             26.111111 |                 100.0 |               629.512602 |              146.962511 |    0.0 |  0.0 | 10.542746 |   0.0 | 3.855538 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.752518 |      0.0 | -0.005079 | -0.000059 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 387.801458 | 15.686136 |      0.0 |   0.0 |   0.0 |   0.0 | -0.752518 | 0.000248 |   0.0 |   0.0 |   0.0 | -0.005327 |   0.0 | -0.000059 |
    | 1997-21-08 |  0.0 | 22.5 |             12.4 |                14.344752 |      246.121286 |             62.1 |       2.5 |              1017.0 | 0.0 | 0.0 |    22.5 |                  62.1 |                  12.4 |                     14.344752 |           246.121286 |    0.0 | 22.5 |    22.5 |    2.003978 |         2.003978 |           0.887476 |          2.5 |                27.299387 |                     27.299387 |                          0.0 |                      1.657431 |                           1.657431 |             16.95292 |                  16.95292 |     1000.04708 |   1.190771 | -2.9 | 22.620153 |    0.72092 |            216.586732 |                  40.22325 |                 216.586732 |                 38.297604 |                      0.0 |   178.289128 |              0.0 |        178.289128 |                  37.6 |                 100.0 |               884.445242 |              152.119959 |    0.0 |  0.0 |  9.796296 |   0.0 | 4.291217 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 0.945327 |      0.0 | -0.008217 | -0.000095 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.180539 | 12.340246 |      0.0 |   0.0 |   0.0 |   0.0 | -0.945327 | 0.000132 |   0.0 |   0.0 |   0.0 |  -0.00835 |   0.0 | -0.000095 |
    | 1997-22-08 |  2.0 | 19.6 |              0.1 |                14.273175 |        90.58551 |             84.5 |       2.1 |              1015.0 | 0.0 | 0.0 |    19.6 |                  84.5 |                   0.1 |                     14.273175 |             90.58551 |   1.88 | 19.6 |    19.6 |    1.683342 |         1.683342 |            0.74548 |          2.1 |                22.846186 |                     22.846186 |                          0.0 |                      1.418946 |                           1.418946 |            19.305027 |                 19.305027 |     995.694973 |   1.199129 | -2.9 | 21.577328 |  11.863971 |             79.715249 |                  14.80426 |                  79.715249 |                   7.02118 |                      0.0 |    72.694069 |              0.0 |         72.694069 |             44.761905 |                 100.0 |               695.922288 |              147.245828 |    0.0 |  0.0 |  3.396916 |  1.88 | 0.698036 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.153227 |      0.0 | -0.012009 | -0.000139 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 369.416568 | 12.795437 |      0.0 |   0.0 |   0.0 |   0.0 | -1.153227 | 0.000071 |   0.0 |   0.0 |   0.0 | -0.012079 |   0.0 | -0.000139 |
    | 1997-23-08 |  0.3 | 20.3 |              6.7 |                14.201333 |      171.944993 |             83.6 |       1.5 |              1015.0 | 0.0 | 0.0 |    20.3 |                  83.6 |                   6.7 |                     14.201333 |           171.944993 |  0.282 | 20.3 |    20.3 |    1.202387 |         1.202387 |           0.532486 |          1.5 |                23.858503 |                     23.858503 |                          0.0 |                      1.473678 |                           1.473678 |            19.945708 |                 19.945708 |     995.054292 |   1.195981 | -2.9 | 17.325305 | -17.848171 |            151.311594 |                 28.100725 |                 151.311594 |                 18.994889 |                      0.0 |   132.316705 |              0.0 |        132.316705 |             62.666667 |                 100.0 |               667.605003 |              145.666152 |    0.0 |  0.0 |  4.351769 | 0.282 | 2.206422 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.124944 |      0.0 | -0.016079 | -0.000186 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 384.364739 | 11.713959 |      0.0 |   0.0 |   0.0 |   0.0 | -1.124944 | 0.000038 |   0.0 |   0.0 |   0.0 | -0.016117 |   0.0 | -0.000186 |
    | 1997-24-08 |  0.0 | 24.1 |             10.4 |                14.129243 |      216.549813 |             76.5 |       2.2 |              1011.0 | 0.0 | 0.0 |    24.1 |                  76.5 |                  10.4 |                     14.129243 |           216.549813 |    0.0 | 24.1 |    24.1 |    1.763501 |         1.763501 |           0.780979 |          2.2 |                30.066256 |                     30.066256 |                          0.0 |                       1.80287 |                            1.80287 |            23.000686 |                 23.000686 |     987.999314 |   1.174651 | -2.9 | 21.630378 | -14.817732 |            190.563835 |                 35.390427 |                 190.563835 |                 21.629737 |                      0.0 |   168.934098 |              0.0 |        168.934098 |             42.727273 |                 100.0 |               938.332509 |              150.288557 |    0.0 |  0.0 |  7.235935 |   0.0 | 3.494124 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.192142 |      0.0 |  -0.02019 | -0.000234 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  396.28247 |  9.411977 |      0.0 |   0.0 |   0.0 |   0.0 | -1.192142 |  0.00002 |   0.0 |   0.0 |   0.0 |  -0.02021 |   0.0 | -0.000234 |
    | 1997-25-08 |  0.1 | 23.7 |              6.5 |                14.056918 |       166.91683 |             76.6 |       1.9 |              1008.0 | 0.0 | 0.0 |    23.7 |                  76.6 |                   6.5 |                     14.056918 |            166.91683 |  0.094 | 23.7 |    23.7 |    1.523023 |         1.523023 |           0.674482 |          1.9 |                29.352596 |                     29.352596 |                          0.0 |                      1.765541 |                           1.765541 |            22.484088 |                 22.484088 |     985.515912 |   1.172942 | -2.9 | 25.062685 |   8.176109 |             146.88681 |                 27.278979 |                  146.88681 |                 16.222254 |                      0.0 |   130.664556 |              0.0 |        130.664556 |             49.473684 |                 100.0 |               1353.41939 |              153.707473 |    0.0 |  0.0 |  5.830169 | 0.094 | 2.901389 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 | 1.335178 |      0.0 | -0.024654 | -0.000285 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 385.206362 |  7.845767 |      0.0 |   0.0 |   0.0 |   0.0 | -1.335178 | 0.000011 |   0.0 |   0.0 |   0.0 | -0.024665 |   0.0 | -0.000285 |
    | 1997-26-08 | 10.5 | 21.1 |              4.9 |                13.984374 |      145.985611 |             84.2 |       2.3 |              1006.0 | 0.0 | 0.0 |    21.1 |                  84.2 |                   4.9 |                     13.984374 |           145.985611 |   9.87 | 21.1 |    21.1 |     1.84366 |          1.84366 |           0.816478 |          2.3 |                25.063177 |                     25.063177 |                          0.0 |                        1.5384 |                             1.5384 |            21.103195 |                 21.103195 |     984.896805 |   1.181557 | -2.9 | 21.872765 |   4.636593 |            128.467338 |                  23.85822 |                 128.467338 |                 14.602883 |                      0.0 |   113.864455 |              0.0 |        113.864455 |             40.869565 |                 100.0 |              1693.426996 |              154.753913 |   7.67 |  0.0 |  4.656397 |   2.2 | 1.080885 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   7.67 | 0.067763 |  0.0 |  0.0 | 0.0 | 1.432496 | 0.067763 | -0.012083 |  -0.00014 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 377.669769 | 15.799615 | 0.067763 |   0.0 |   0.0 |   0.0 | -1.432496 | 0.017451 |   0.0 |   0.0 |   0.0 | -0.029534 |   0.0 |  -0.00014 |
    | 1997-27-08 |  6.5 | 20.9 |              6.6 |                13.911624 |       165.54389 |             85.8 |       2.1 |              1005.0 | 0.0 | 0.0 |    20.9 |                  85.8 |                   6.6 |                     13.911624 |            165.54389 |   6.11 | 20.9 |    20.9 |    1.683342 |         1.683342 |            0.74548 |          2.1 |                 24.75714 |                      24.75714 |                          0.0 |                      1.521999 |                           1.521999 |            21.241626 |                 21.241626 |     983.758374 |   1.181114 | -2.9 | 19.702227 |   -7.18664 |            145.678623 |                 27.054602 |                 145.678623 |                 17.349121 |                      0.0 |   128.329502 |              0.0 |        128.329502 |             44.761905 |                 100.0 |               514.330605 |              137.506981 |   3.91 |  0.0 |  4.603714 |   2.2 | 1.172574 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   3.91 | 0.052958 |  0.0 |  0.0 | 0.0 | 0.938276 | 0.052958 |   0.00343 |   0.00004 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 381.956408 | 19.422358 | 0.052958 |   0.0 |   0.0 |   0.0 | -0.938276 | 0.037105 |   0.0 |   0.0 |   0.0 | -0.033675 |   0.0 |   0.00004 |
    | 1997-28-08 | 15.0 | 20.2 |              0.3 |                13.838682 |       87.693346 |             86.5 |       2.6 |              1000.0 | 0.0 | 0.0 |    20.2 |                  86.5 |                   0.3 |                     13.838682 |            87.693346 |   14.1 | 20.2 |    20.2 |    2.084137 |         2.084137 |           0.922975 |          2.6 |                23.711531 |                     23.711531 |                          0.0 |                      1.465752 |                           1.465752 |            20.510475 |                 20.510475 |     979.489525 |   1.178323 | -2.9 | 20.936779 |   4.420674 |             77.170144 |                 14.331598 |                  77.170144 |                  6.818024 |                      0.0 |     70.35212 |              0.0 |          70.35212 |             36.153846 |                 100.0 |               423.374916 |              132.523752 |   11.9 |  0.0 |  3.480087 |   2.2 | 0.567488 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   11.9 | 0.232109 |  0.0 |  0.0 | 0.0 | 0.713174 | 0.232109 |  0.054108 |  0.000626 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.635735 | 31.235935 | 0.232109 |   0.0 |   0.0 |   0.0 | -0.713174 |  0.09062 |   0.0 |   0.0 |   0.0 | -0.036512 |   0.0 |  0.000626 |
    | 1997-29-08 |  9.2 | 16.1 |              0.1 |                13.765561 |       84.356885 |             89.6 |       2.6 |              1004.0 | 0.0 | 0.0 |    16.1 |                  89.6 |                   0.1 |                     13.765561 |            84.356885 |  8.648 | 16.1 |    16.1 |    2.084137 |         2.084137 |           0.922975 |          2.6 |                18.326791 |                     18.326791 |                          0.0 |                      1.170311 |                           1.170311 |            16.420805 |                 16.420805 |     987.579195 |   1.201704 | -2.9 | 18.828425 |   16.37055 |             74.234059 |                 13.786325 |                  74.234059 |                  8.199212 |                      0.0 |    66.034847 |              0.0 |         66.034847 |             36.153846 |                 100.0 |               347.066882 |              127.000167 |  6.448 |  0.0 |  2.684886 |   2.2 | 0.203734 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  6.448 | 0.173151 |  0.0 |  0.0 | 0.0 |      0.0 | 0.173151 |  0.103772 |  0.001201 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 355.365185 | 37.307051 | 0.173151 |   0.0 |   0.0 |   0.0 |       0.0 |  0.14143 |   0.0 |   0.0 |   0.0 | -0.037658 |   0.0 |  0.001201 |
    | 1997-30-08 |  0.0 | 17.0 |              5.0 |                13.692273 |      142.309202 |             78.3 |       2.1 |              1011.0 | 0.0 | 0.0 |    17.0 |                  78.3 |                   5.0 |                     13.692273 |           142.309202 |    0.0 | 17.0 |    17.0 |    1.683342 |         1.683342 |            0.74548 |          2.1 |                19.406929 |                     19.406929 |                          0.0 |                      1.230421 |                           1.230421 |            15.195625 |                 15.195625 |     995.804375 |   1.206937 | -2.9 | 13.278507 |  -22.32896 |            125.232098 |                  23.25739 |                 125.232098 |                 21.775852 |                      0.0 |   103.456246 |              0.0 |        103.456246 |             44.761905 |                 100.0 |               346.870428 |              126.427648 |    0.0 |  0.0 |   4.53294 |   0.0 | 2.160883 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.074124 |  0.000858 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 374.794145 | 35.146168 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.111647 |   0.0 |   0.0 |   0.0 | -0.037523 |   0.0 |  0.000858 |
    | 1997-31-08 |  0.0 | 18.2 |              3.8 |                13.618831 |      126.722078 |             78.5 |       2.1 |              1014.0 | 0.0 | 0.0 |    18.2 |                  78.5 |                   3.8 |                     13.618831 |           126.722078 |    0.0 | 18.2 |    18.2 |    1.683342 |         1.683342 |            0.74548 |          2.1 |                20.933479 |                     20.933479 |                          0.0 |                      1.314615 |                           1.314615 |            16.432781 |                 16.432781 |     997.567219 |   1.204994 | -2.9 | 18.874047 |   4.044283 |            111.515429 |                 20.710008 |                 111.515429 |                 17.491881 |                      0.0 |    94.023548 |              0.0 |         94.023548 |             44.761905 |                 100.0 |               345.930896 |              125.804674 |    0.0 |  0.0 |  4.439668 |   0.0 | 2.165731 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |    0.0 |      0.0 |  0.0 |  0.0 | 0.0 |      0.0 |      0.0 |  0.022187 |  0.000257 |  0.0 |  0.0 |  0.0 |   0.0 |  nan | 367.849862 | 32.980437 |      0.0 |   0.0 |   0.0 |   0.0 |       0.0 | 0.059576 |   0.0 |   0.0 |   0.0 | -0.037389 |   0.0 |  0.000257 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_v3_acker_winter_daily:

acre (winter)
-------------

None of the examples discussed so far dealt with snow processes.  Thus, we now switch
from a warm August to a snowy December and adapt the initial conditions and the input
time series accordingly (those of global radiation and the astronomically possible
sunshine duration stem from the example :ref:`meteo_v003_daily_simulation_winter` of
|meteo_v003|):

>>> pub.timegrids = "2010-12-01", "2011-01-01", "1d"
>>> lnk(ACKER)
>>> lai.acker_dec = 0.3
>>> cropheight.acker_dec = 0.05
>>> albedo.acker_dec = 0.1
>>> wg2z.dec = 8.6
>>> test = IntegrationTest(land)
>>> test.dateformat = "%Y-%d-%m"

>>> test.inits = (
...     (states.inzp, 0.0),
...     (states.wats, 0.0),
...     (states.waes, 0.0),
...     (states.esnow, 0.0),
...     (states.taus, 0.0),
...     (states.ebdn, 0.0),
...     (states.bowa, 72.0),
...     (states.qdgz1, 0.0),
...     (states.qdgz2, 0.0),
...     (states.qigz1, 0.0),
...     (states.qigz2, 0.0),
...     (states.qbgz, 0.0),
...     (states.qdga1, 0.0),
...     (states.qdga2, 0.0),
...     (states.qiga1, 0.0),
...     (states.qiga2, 0.0),
...     (states.qbga, 0.0))

>>> inputs.nied.series = (
...     0.0, 11.1, 0.4, 1.2, 6.4, 0.0, 0.0, 0.0, 0.7, 3.3, 9.4, 0.2, 2.9, 0.0, 0.0,
...     6.7, 0.0, 0.1, 0.0, 0.2, 0.2, 2.4, 7.0, 0.2, 0.0, 1.2, 0.9, 0.0, 0.0, 0.2, 0.9)
>>> inputs.teml.series = (
...     -5.8, -3.0, -5.6, -3.8, 0.2, -1.1, -2.9, -2.9, -2.1, -3.1, 5.1, 0.2, -3.5,
...     -3.9, -6.3, -3.5, -7.3, -5.8, -7.9, -5.7, -5.6, -4.8, -1.1, -2.9, -9.1, -8.6,
...     -5.3, -7.7, -9.2, -7.4, 1.7)
>>> inputs.sunshineduration.series = (
...     3.5, 0.0, 0.1, 3.7, 0.0, 0.1, 0.0, 0.0, 4.4, 0.1, 0.2, 0.7, 0.0, 1.2, 3.0, 0.0,
...     0.0, 0.0, 0.0, 0.2, 0.8, 0.0, 0.0, 0.0, 3.2, 4.3, 0.0, 3.4, 0.0, 0.0, 4.0)
>>> inputs.windspeed.series = (
...     6.9, 5.2, 2.0, 4.3, 4.0, 2.4, 1.9, 1.8, 2.7, 3.0, 7.4, 5.7, 2.2, 3.1, 2.8, 3.7,
...     0.9, 1.3, 1.9, 2.4, 2.3, 4.6, 8.6, 7.6, 3.8, 1.5, 1.3, 3.0, 2.7, 1.4, 4.2)
>>> inputs.relativehumidity.series = (
...     70.3, 85.0, 95.9, 90.6, 96.5, 99.0, 99.2, 99.1, 96.9, 95.0, 90.6, 78.7, 90.5,
...     88.1, 92.8, 98.1, 96.6, 97.7, 95.8, 97.5, 95.0, 93.8, 94.9, 85.4, 87.2, 92.7,
...     97.8, 95.4, 94.9, 96.0, 96.5)
>>> inputs.atmosphericpressure.series = (
...     1015.0, 1007.0, 1005.0, 1008.0, 996.0, 998.0, 1004.0, 1006.0, 1009.0, 1018.0,
...     1004.0, 1012.0, 1020.0, 1025.0, 1023.0, 1000.0, 991.0, 997.0, 997.0, 997.0,
...     1004.0, 1007.0, 1004.0, 1008.0, 1018.0, 1019.0, 1017.0, 1022.0, 1024.0, 1021.0,
...     1016.0)
>>> inlet.sequences.sim.series = 0.0

>>> inputs.possiblesunshineduration.series = (
...     7.64102946, 7.6096444, 7.57982377, 7.55159603, 7.52498861, 7.50002776,
...     7.47673851, 7.45514453, 7.43526809, 7.41712991, 7.40074913, 7.38614323,
...     7.3733279, 7.36231706, 7.35312273, 7.34575502, 7.34022207, 7.33653002,
...     7.33468296, 7.33468296, 7.33653002, 7.34022207, 7.34575502, 7.35312273,
...     7.36231706, 7.3733279, 7.38614323, 7.40074913, 7.41712991, 7.43526809,
...     7.45514453)
>>> inputs.globalradiation.series = (
...     31.9427981, 9.8738984, 16.6834998, 31.8074275, 9.5325105, 16.1414844,
...     9.3414287, 9.2567644, 33.405815, 15.5911584, 15.8915442, 17.8235189, 8.941324,
...     19.6680679, 26.8315815, 8.8377384, 8.817416, 8.8041854, 8.7980439, 15.4647459,
...     17.8795412, 8.8221612, 8.8443961, 8.8737425, 27.7596417, 32.3287553, 9.0045823,
...     28.9823209, 9.1276454, 9.1999895, 32.0622345)

When comparing the following results with those of integration test
:ref:`lland_v1_acker_winter` on |lland_v1|, it is not immediately clear what
differences are due to the much higher complexities of the snow module of |lland_v3|.
But, at least, one can see that |lland_v3| allows for the (re)freezing of liquid water
within the snow layer.  Refreezing  occurs around December 7th, where the frozen water
equivalent (|WATS|) grows until it reaches to total water equivalent (|WAeS|) (a note
for `LARSIM`_ users: currently, `LARSIM`_ does not implement such a freezing feature;
you can disable it in |lland_v3| through setting |RefreezeFlag| to |False|):

.. integration-test::

    >>> test.reset_inits()
    >>> conditions = sequences.conditions
    >>> conditions_acker_winter = test(
    ...     "lland_v3_acker_winter_daily",
    ...     axis1=(inputs.nied, fluxes.wada), axis2=(states.waes, states.wats),
    ...     get_conditions="2010-12-10")
    |       date | nied | teml | sunshineduration | possiblesunshineduration | globalradiation | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration | dailypossiblesunshineduration | dailyglobalradiation |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair |   g |         tz |         wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance |   nbes |    sbes |      evpo | evi |      evb |       evs |     wnied | tempssurface | actualalbedo |   schmpot |      schm |  gefrpot |     gefr |   wlatsnow |  wsenssnow |       wsurf |      sff |      fvg |      wada |      qdb | qib1 | qib2 | qbb | qkap |     qdgz |      qah |       qa | inzp |      wats |      waes |     esnow |     taus |       ebdn |      bowa |    qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-01-12 |  0.0 | -5.8 |              3.5 |                 7.641029 |       31.942798 |             70.3 |       6.9 |              1015.0 | 0.0 | 0.0 |    -5.8 |                  70.3 |                   3.5 |                      7.641029 |            31.942798 |    0.0 | -5.8 |    -5.8 |     5.53098 |          5.53098 |            5.53098 |          6.9 |                 3.958114 |                      3.958114 |                          0.0 |                      0.303557 |                           0.303557 |             2.782554 |                  2.782554 |    1012.217446 |   1.321192 | 8.6 |        0.0 |       34.8 |             28.748518 |                 28.748518 |                  28.748518 |                 46.472798 |                      0.0 |    -17.72428 |              0.0 |         -17.72428 |             30.870386 |                 100.0 |               111.309423 |               99.512695 |    0.0 |     0.0 |  1.808353 | 0.0 | 0.553793 |       0.0 |       0.0 |          nan |          0.1 |       0.0 |       0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |  0.0 |       0.0 |       0.0 |       0.0 |      nan |      -26.2 | 71.446207 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-02-12 | 11.1 | -3.0 |              0.0 |                 7.609644 |        9.873898 |             85.0 |       5.2 |              1007.0 | 0.0 | 0.0 |    -3.0 |                  85.0 |                   0.0 |                      7.609644 |             9.873898 | 10.434 | -3.0 |    -3.0 |    4.168274 |         4.168274 |           4.168274 |          5.2 |                 4.893489 |                      4.893489 |                     3.819164 |                      0.366257 |                           0.366257 |             4.159465 |                  4.159465 |    1002.840535 |   1.296511 | 8.6 |    -7.5456 |  -7.683496 |              1.229506 |                  1.229506 |                   1.229506 |                  14.78976 |                27.862843 |   -13.560254 |       -26.633337 |        -13.560254 |             40.962628 |                 100.0 |               112.102221 |               99.530053 | 10.374 |  10.374 |  0.780099 | 0.0 |      0.0 |  -0.16818 | -0.752835 |    -6.265017 |     0.875479 |       0.0 |       0.0 | 0.406695 |      0.0 |  -5.192276 | -28.305216 |   -6.864155 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 |  10.54218 |  10.54218 | -1.572176 | 0.084415 |  -9.916504 | 71.446207 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-03-12 |  0.4 | -5.6 |              0.1 |                 7.579824 |         16.6835 |             95.9 |       2.0 |              1005.0 | 0.0 | 0.0 |    -5.6 |                  95.9 |                   0.1 |                      7.579824 |              16.6835 |  0.376 | -5.6 |    -5.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                 4.019239 |                      4.019239 |                     3.571122 |                      0.307706 |                           0.307706 |              3.85445 |                   3.85445 |     1001.14555 |   1.306656 | 8.6 |  -2.855953 |  25.666656 |              2.321028 |                  2.321028 |                   2.321028 |                 15.387201 |                37.526691 |   -13.066174 |       -35.205663 |        -13.066174 |            106.502833 |                 100.0 |               112.102221 |               99.527228 |  0.376 |   0.376 |  0.039709 | 0.0 |      0.0 | -0.073734 | -0.050934 |    -7.133729 |     0.860879 |       0.0 |       0.0 | 0.487379 |      0.0 |  -2.276422 |  -7.001615 |   25.927626 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 | 10.991915 | 10.991915 |  -1.88408 | 0.141806 |  -26.98316 | 71.446207 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-04-12 |  1.2 | -3.8 |              3.7 |                 7.551596 |       31.807427 |             90.6 |       4.3 |              1008.0 | 0.0 | 0.0 |    -3.8 |                  90.6 |                   3.7 |                      7.551596 |            31.807427 |  1.128 | -3.8 |    -3.8 |    3.446842 |         3.446842 |           3.446842 |          4.3 |                 4.608124 |                      4.608124 |                     3.236431 |                      0.347298 |                           0.347298 |              4.17496 |                   4.17496 |     1003.82504 |   1.301648 | 8.6 |   -7.77115 |   3.742499 |              4.733805 |                  4.733805 |                   4.733805 |                  43.14024 |                55.893456 |   -38.406434 |       -51.159651 |        -38.406434 |             49.536202 |                 100.0 |               112.102221 |               99.524553 |  1.128 |   1.128 |  0.092292 | 0.0 |      0.0 | -0.402072 | -0.103687 |      -8.3949 |     0.851173 |       0.0 |       0.0 | 0.636671 |      0.0 | -12.413282 | -34.530433 |    4.215936 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 | 12.521987 | 12.521987 | -2.461204 | 0.183441 | -22.125659 | 71.446207 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-05-12 |  6.4 |  0.2 |              0.0 |                 7.524989 |        9.532511 |             96.5 |       4.0 |               996.0 | 0.0 | 0.0 |     0.2 |                  96.5 |                   0.0 |                      7.524989 |             9.532511 |  6.016 |  0.2 |     0.2 |    3.206365 |         3.206365 |           3.206365 |          4.0 |                 6.197477 |                      6.197477 |                     4.912922 |                      0.451275 |                           0.451275 |             5.980565 |                  5.980565 |     990.019435 |   1.266438 | 8.6 |   -6.37219 | -20.550929 |               1.46096 |                   1.46096 |                    1.46096 |                 13.205056 |                15.889112 |   -11.744097 |       -14.428152 |        -11.744097 |             53.251417 |                 100.0 |               112.102221 |               99.522033 |  6.016 |  2.7072 |  0.134993 | 0.0 |      0.0 | -0.433968 |  0.045165 |    -2.947035 |     0.846739 |       0.0 |       0.0 | 0.403076 | 0.403076 | -13.397994 | -22.438939 |  -21.408781 | 0.977284 | 0.286525 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 | 15.998208 | 18.971955 |       0.0 | 0.203488 |   7.025269 | 71.446207 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-06-12 |  0.0 | -1.1 |              0.1 |                 7.500028 |       16.141484 |             99.0 |       2.4 |               998.0 | 0.0 | 0.0 |    -1.1 |                  99.0 |                   0.1 |                      7.500028 |            16.141484 |    0.0 | -1.1 |    -1.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                 5.634754 |                      5.634754 |                     5.151568 |                       0.41489 |                            0.41489 |             5.578406 |                  5.578406 |     992.421594 |   1.275245 | 8.6 |        0.0 |  13.867614 |              3.011254 |                  3.011254 |                   3.011254 |                 14.093918 |                 28.07219 |   -11.082664 |       -25.060937 |        -11.082664 |             88.752361 |                 100.0 |               112.102221 |               99.519668 |    0.0 |     0.0 |  0.000778 | 0.0 |      0.0 | -0.123565 |       0.0 |    -2.311269 |     0.813446 |       0.0 |       0.0 | 0.317535 | 0.317535 |  -3.814858 |  -6.150958 |   15.095121 | 0.994317 |   0.2966 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 | 16.422008 |  19.09552 |       0.0 |  0.37888 |   1.757655 | 71.446207 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-07-12 |  0.0 | -2.9 |              0.0 |                 7.476739 |        9.341429 |             99.2 |       1.9 |              1004.0 | 0.0 | 0.0 |    -2.9 |                  99.2 |                   0.0 |                      7.476739 |             9.341429 |    0.0 | -2.9 |    -2.9 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 4.930236 |                      4.930236 |                     4.623647 |                      0.368688 |                           0.368688 |             4.890794 |                  4.890794 |     999.109206 |   1.291808 | 8.6 |        0.0 |  22.532218 |              1.983491 |                  1.983491 |                   1.983491 |                 13.815559 |                32.375277 |   -11.832067 |       -30.391785 |        -11.832067 |            112.108246 |                 100.0 |               112.102221 |               99.517462 |    0.0 |     0.0 |  -0.01574 | 0.0 |      0.0 |  -0.06757 |       0.0 |     -3.75537 |     0.787667 |       0.0 |       0.0 | 0.511759 | 0.511759 |  -2.086105 |  -3.795136 |   24.510544 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 | 16.993687 |  19.16309 |       0.0 | 0.554272 | -12.174563 | 71.446207 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-08-12 |  0.0 | -2.9 |              0.0 |                 7.455145 |        9.256764 |             99.1 |       1.8 |              1006.0 | 0.0 | 0.0 |    -2.9 |                  99.1 |                   0.0 |                      7.455145 |             9.256764 |    0.0 | -2.9 |    -2.9 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 4.930236 |                      4.930236 |                     4.225088 |                      0.368688 |                           0.368688 |             4.885864 |                  4.885864 |    1001.114136 |   1.294388 | 8.6 |  -3.506274 |   8.636691 |              2.155749 |                  2.155749 |                   2.155749 |                 13.821847 |                27.170321 |   -11.666098 |       -25.014571 |        -11.666098 |            118.336481 |                 100.0 |               112.102221 |               99.515416 |    0.0 |     0.0 | -0.012472 | 0.0 |      0.0 |   -0.1623 |       0.0 |    -4.945723 |     0.767116 |       0.0 |       0.0 | 0.660412 | 0.660412 |  -5.010735 |  -8.814165 |   11.189672 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 | 17.803619 |  19.32539 |       0.0 | 0.729664 | -12.211254 | 71.446207 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-09-12 |  0.7 | -2.1 |              4.4 |                 7.435268 |       33.405815 |             96.9 |       2.7 |              1009.0 | 0.0 | 0.0 |    -2.1 |                  96.9 |                   4.4 |                      7.435268 |            33.405815 |  0.658 | -2.1 |    -2.1 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 5.233109 |                      5.233109 |                     3.897456 |                      0.388644 |                           0.388644 |             5.070883 |                  5.070883 |    1003.929117 |   1.294334 | 8.6 |  -3.516841 |  14.906113 |              8.184701 |                  8.184701 |                   8.184701 |                 46.364476 |                58.773789 |   -38.179775 |       -50.589088 |        -38.179775 |             78.890988 |                 100.0 |               112.102221 |               99.513534 |  0.658 |   0.658 | -0.234819 | 0.0 |      0.0 | -0.365433 | -0.033425 |    -6.001193 |     0.754992 |       0.0 |       0.0 | 0.807756 | 0.807756 | -11.282099 | -21.311727 |   17.995262 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 |  19.62175 | 20.348822 |       0.0 | 0.853031 | -18.517366 | 71.446207 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-10-12 |  3.3 | -3.1 |              0.1 |                  7.41713 |       15.591158 |             95.0 |       3.0 |              1018.0 | 0.0 | 0.0 |    -3.1 |                  95.0 |                   0.1 |                       7.41713 |            15.591158 |  3.102 | -3.1 |    -3.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 4.856984 |                      4.856984 |                      4.02737 |                       0.36384 |                            0.36384 |             4.614135 |                  4.614135 |    1013.385865 |   1.310959 | 8.6 |  -5.333001 |   1.443584 |              3.649898 |                  3.649898 |                   3.649898 |                 14.892721 |                28.624554 |   -11.242823 |       -24.974656 |        -11.242823 |             71.001889 |                 100.0 |               112.102221 |               99.511816 |  3.102 |   3.102 |  0.137371 | 0.0 |      0.0 | -0.195603 | -0.232614 |    -5.573599 |     0.765899 |       0.0 |       0.0 | 0.843328 | 0.727072 |  -6.038888 |  -14.46471 |    4.471057 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.06 | 23.646425 | 23.646425 | -0.449415 | 0.741299 |  -11.36095 | 71.446207 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-11-12 |  9.4 |  5.1 |              0.2 |                 7.400749 |       15.891544 |             90.6 |       7.4 |              1004.0 | 0.0 | 0.0 |     5.1 |                  90.6 |                   0.2 |                      7.400749 |            15.891544 |  8.836 |  5.1 |     5.1 |    5.931775 |         5.931775 |           5.931775 |          7.4 |                 8.790172 |                      8.790172 |                       6.1078 |                      0.614118 |                           0.614118 |             7.963896 |                  7.963896 |     996.036104 |   1.253213 | 8.6 |  -3.271954 | -19.631722 |              3.997614 |                  3.997614 |                   3.997614 |                 13.525343 |                -0.754591 |    -9.527729 |         4.752205 |         -9.527729 |              28.78455 |                 100.0 |               112.102221 |               99.510264 |  8.836 |     0.0 |  1.033274 | 0.0 |      0.0 | -1.215856 |  2.183707 |          0.0 |     0.748444 | 21.469498 | 21.469498 |      0.0 |      0.0 | -37.537452 | -58.603285 | -100.892942 | 0.945448 | 0.268161 | 30.468752 | 9.782248 |  0.0 |  0.0 | 0.0 |  0.0 | 9.782248 | 2.518406 | 0.029148 | 0.06 |  2.258412 |  3.229529 |       0.0 | 0.927266 |  16.870772 |  92.13271 | 9.782248 |   0.0 |   0.0 |   0.0 |  0.0 | 2.518406 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.029148 |
    | 2010-12-12 |  0.2 |  0.2 |              0.7 |                 7.386143 |       17.823519 |             78.7 |       5.7 |              1012.0 | 0.0 | 0.0 |     0.2 |                  78.7 |                   0.7 |                      7.386143 |            17.823519 |  0.188 |  0.2 |     0.2 |     4.56907 |          4.56907 |            4.56907 |          5.7 |                 6.197477 |                      6.197477 |                     5.176439 |                      0.451275 |                           0.451275 |             4.877414 |                  4.877414 |    1007.122586 |    1.28736 | 8.6 |        0.0 |   13.47911 |              4.716666 |                  4.716666 |                   4.716666 |                 20.069431 |                36.311921 |   -15.352764 |       -31.595255 |        -15.352764 |             37.369415 |                 100.0 |                83.806965 |               98.586162 |  0.188 |  0.0846 |  1.429571 | 0.0 |      0.0 |  0.158712 |  0.000762 |    -2.246518 |     0.735368 |       0.0 |       0.0 | 0.061309 | 0.061309 |   4.899959 | -22.778338 |   13.716876 | 0.961225 | 0.277186 |  0.067022 |  0.02238 |  0.0 |  0.0 | 0.0 |  0.0 |  0.02238 | 3.393537 | 0.039277 | 0.06 |   2.29043 |  3.191795 |       0.0 | 1.094855 |  11.991662 | 92.177353 |  0.02238 |   0.0 |   0.0 |   0.0 |  0.0 | 3.393537 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.039277 |
    | 2010-13-12 |  2.9 | -3.5 |              0.0 |                 7.373328 |        8.941324 |             90.5 |       2.2 |              1020.0 | 0.0 | 0.0 |    -3.5 |                  90.5 |                   0.0 |                      7.373328 |             8.941324 |  2.726 | -3.5 |    -3.5 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                 4.713362 |                      4.713362 |                     4.320033 |                      0.354306 |                           0.354306 |             4.265593 |                  4.265593 |    1015.734407 |   1.315658 | 8.6 |        0.0 |  27.920468 |              2.237038 |                  2.237038 |                   2.237038 |                 14.516908 |                 35.79395 |    -12.27987 |       -33.556911 |         -12.27987 |             96.820757 |                 100.0 |                83.748115 |               98.584016 |  2.726 |   2.726 |  0.194059 | 0.0 |      0.0 |  0.014964 | -0.230795 |    -4.653411 |     0.749809 |       0.0 |       0.0 |  0.19865 |  0.19865 |   0.461981 |  -5.561289 |   28.457603 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 1.815505 | 0.021013 | 0.06 |  5.201893 |  5.902832 |       0.0 | 0.911303 |  -7.328805 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.815505 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.021013 |
    | 2010-14-12 |  0.0 | -3.9 |              1.2 |                 7.362317 |       19.668068 |             88.1 |       3.1 |              1025.0 | 0.0 | 0.0 |    -3.9 |                  88.1 |                   1.2 |                      7.362317 |            19.668068 |    0.0 | -3.9 |    -3.9 |    2.484933 |         2.484933 |           2.484933 |          3.1 |                  4.57351 |                       4.57351 |                     3.839218 |                      0.344989 |                           0.344989 |             4.029262 |                  4.029262 |    1020.970738 |   1.324197 | 8.6 |  -2.110696 |  24.517738 |              5.193227 |                  5.193227 |                   5.193227 |                 24.390008 |                46.426577 |   -19.196781 |        -41.23335 |        -19.196781 |             68.711505 |                 100.0 |                83.748115 |               98.584359 |    0.0 |     0.0 |  0.281823 | 0.0 |      0.0 | -0.064742 |       0.0 |    -6.196986 |     0.735956 |       0.0 |       0.0 | 0.256166 | 0.256166 |    -1.9988 | -13.726539 |   25.508011 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.968774 | 0.011213 | 0.06 |  5.517923 |  5.967574 |       0.0 | 1.086695 | -23.246543 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.968774 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.011213 |
    | 2010-15-12 |  0.0 | -6.3 |              3.0 |                 7.353123 |       26.831581 |             92.8 |       2.8 |              1023.0 | 0.0 | 0.0 |    -6.3 |                  92.8 |                   3.0 |                      7.353123 |            26.831581 |    0.0 | -6.3 |    -6.3 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                 3.808888 |                      3.808888 |                     2.826211 |                      0.293396 |                           0.293396 |             3.534648 |                  3.534648 |    1019.465352 |    1.33374 | 8.6 |  -6.695004 |  20.484563 |              7.398757 |                  7.398757 |                   7.398757 |                 39.328283 |                57.721383 |   -31.929526 |       -50.322626 |        -31.929526 |             76.073452 |                 100.0 |                83.748115 |               98.584646 |    0.0 |     0.0 | -0.028371 | 0.0 |      0.0 | -0.225804 |       0.0 |   -10.109098 |     0.724252 |       0.0 |       0.0 |  0.40603 |  0.40603 |  -6.971296 | -21.297159 |   22.054171 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.516948 | 0.005983 | 0.06 |  6.148107 |  6.193377 |       0.0 | 1.262087 | -35.131106 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.516948 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.005983 |
    | 2010-16-12 |  6.7 | -3.5 |              0.0 |                 7.345755 |        8.837738 |             98.1 |       3.7 |              1000.0 | 0.0 | 0.0 |    -3.5 |                  98.1 |                   0.0 |                      7.345755 |             8.837738 |  6.298 | -3.5 |    -3.5 |    2.965888 |         2.965888 |           2.965888 |          3.7 |                 4.713362 |                      4.713362 |                     3.578876 |                      0.354306 |                           0.354306 |             4.623808 |                  4.623808 |     995.376192 |   1.289645 | 8.6 | -10.117759 | -18.071922 |              1.897806 |                  1.897806 |                   1.897806 |                   14.0252 |                22.175395 |   -12.127394 |       -20.277589 |        -12.127394 |             57.569099 |                 100.0 |                83.748115 |               98.584875 |  6.298 |   6.298 |  0.035476 | 0.0 |      0.0 | -0.401816 | -0.533216 |    -7.105772 |     0.785261 |       0.0 |       0.0 | 0.557439 | 0.045271 | -12.405366 | -24.322444 |  -16.450222 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.275849 | 0.003193 | 0.06 | 12.893193 | 12.893193 | -1.979911 | 0.572947 |  -8.459184 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.275849 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003193 |
    | 2010-17-12 |  0.0 | -7.3 |              0.0 |                 7.340222 |        8.817416 |             96.6 |       0.9 |               991.0 | 0.0 | 0.0 |    -7.3 |                  96.6 |                   0.0 |                      7.340222 |             8.817416 |    0.0 | -7.3 |    -7.3 |    0.721432 |         0.721432 |           0.721432 |          0.9 |                  3.52531 |                       3.52531 |                     3.261432 |                      0.273951 |                           0.273951 |             3.405449 |                  3.405449 |     987.594551 |   1.296889 | 8.6 |  -2.436245 |  35.163253 |              1.957901 |                  1.957901 |                   1.957901 |                 14.906398 |                41.672557 |   -12.948497 |       -39.714657 |        -12.948497 |            236.672963 |                 100.0 |                83.748115 |               98.585048 |    0.0 |     0.0 | -0.001143 | 0.0 |      0.0 | -0.025897 |       0.0 |    -8.296787 |     0.777951 |       0.0 |       0.0 | 0.669376 |      0.0 |  -0.799522 |  -3.144158 |   35.770977 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.147196 | 0.001704 | 0.06 |  12.91909 |  12.91909 | -2.587635 | 0.632541 | -35.022438 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.147196 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001704 |
    | 2010-18-12 |  0.1 | -5.8 |              0.0 |                  7.33653 |        8.804185 |             97.7 |       1.3 |               997.0 | 0.0 | 0.0 |    -5.8 |                  97.7 |                   0.0 |                       7.33653 |             8.804185 |  0.094 | -5.8 |    -5.8 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                 3.958114 |                      3.958114 |                     2.837467 |                      0.303557 |                           0.303557 |             3.867077 |                  3.867077 |     993.132923 |   1.297204 | 8.6 | -10.086462 |  -0.163598 |              2.001268 |                  2.001268 |                   2.001268 |                 14.549007 |                24.668697 |   -12.547739 |       -22.667429 |        -12.547739 |            163.850513 |                 100.0 |                83.748115 |               98.585163 |  0.094 |   0.094 | -0.002683 | 0.0 |      0.0 | -0.215254 | -0.013188 |   -10.059196 |     0.772691 |       0.0 |       0.0 | 0.819112 |      0.0 |  -6.645584 |  -15.61979 |    0.402055 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.078546 | 0.000909 | 0.06 | 13.228344 | 13.228344 | -3.166476 | 0.678288 |  -26.25884 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.078546 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000909 |
    | 2010-19-12 |  0.0 | -7.9 |              0.0 |                 7.334683 |        8.798044 |             95.8 |       1.9 |               997.0 | 0.0 | 0.0 |    -7.9 |                  95.8 |                   0.0 |                      7.334683 |             8.798044 |    0.0 | -7.9 |    -7.9 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 3.364296 |                      3.364296 |                      2.72315 |                      0.262827 |                           0.262827 |             3.222996 |                  3.222996 |     993.777004 |   1.307793 | 8.6 |  -7.562546 |  18.071092 |              2.044019 |                  2.044019 |                   2.044019 |                 15.062045 |                36.101415 |   -13.018026 |       -34.057396 |        -13.018026 |            112.108246 |                 100.0 |                83.748115 |                98.58522 |    0.0 |     0.0 |  0.032664 | 0.0 |      0.0 | -0.126427 |       0.0 |   -10.574395 |     0.767673 |       0.0 |       0.0 | 0.875308 |      0.0 |  -3.903213 | -11.865853 |   18.288329 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.041913 | 0.000485 | 0.06 | 13.354771 | 13.354771 | -3.383713 | 0.724389 | -35.729932 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.041913 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000485 |
    | 2010-20-12 |  0.2 | -5.7 |              0.2 |                 7.334683 |       15.464746 |             97.5 |       2.4 |               997.0 | 0.0 | 0.0 |    -5.7 |                  97.5 |                   0.2 |                      7.334683 |            15.464746 |  0.188 | -5.7 |    -5.7 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                 3.988573 |                      3.988573 |                     2.930765 |                      0.305625 |                           0.305625 |             3.888858 |                  3.888858 |     993.111142 |   1.296708 | 8.6 |  -10.29022 |  -3.828076 |              3.642564 |                  3.642564 |                   3.642564 |                  16.12634 |                 28.19957 |   -12.483776 |       -24.557007 |        -12.483776 |             88.752361 |                 100.0 |                83.748115 |                98.58522 |  0.188 |   0.188 |  0.021098 | 0.0 |      0.0 | -0.277358 | -0.025922 |    -9.652208 |      0.76446 |       0.0 |       0.0 | 0.817964 |      0.0 |  -8.562936 | -20.069747 |   -4.075676 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.022365 | 0.000259 | 0.06 | 13.820128 | 13.820128 | -3.162035 | 0.755264 | -23.301856 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.022365 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000259 |
    | 2010-21-12 |  0.2 | -5.6 |              0.8 |                  7.33653 |       17.879541 |             95.0 |       2.3 |              1004.0 | 0.0 | 0.0 |    -5.6 |                  95.0 |                   0.8 |                       7.33653 |            17.879541 |  0.188 | -5.6 |    -5.6 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                 4.019239 |                      4.019239 |                     3.128794 |                      0.307706 |                           0.307706 |             3.818277 |                  3.818277 |    1000.181723 |   1.305372 | 8.6 |  -6.710934 |   12.68525 |              4.271427 |                  4.271427 |                   4.271427 |                 21.069426 |                38.730209 |   -16.797999 |       -34.458782 |        -16.797999 |             92.611159 |                 100.0 |                83.748115 |               98.585163 |  0.188 |   0.188 |  0.035086 | 0.0 |      0.0 | -0.194557 | -0.025467 |    -8.825143 |       0.7611 |       0.0 |       0.0 | 0.773574 |      0.0 |  -6.006607 | -15.963992 |   12.488183 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.011934 | 0.000138 | 0.06 | 14.202685 | 14.202685 | -2.990435 | 0.788759 | -27.387106 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.011934 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000138 |
    | 2010-22-12 |  2.4 | -4.8 |              0.0 |                 7.340222 |        8.822161 |             93.8 |       4.6 |              1007.0 | 0.0 | 0.0 |    -4.8 |                  93.8 |                   0.0 |                      7.340222 |             8.822161 |  2.256 | -4.8 |    -4.8 |     3.68732 |          3.68732 |            3.68732 |          4.6 |                 4.272186 |                      4.272186 |                     3.442548 |                      0.324793 |                           0.324793 |             4.007311 |                  4.007311 |    1002.992689 |   1.305282 | 8.6 |  -7.887486 |   -1.69314 |              1.989984 |                  1.989984 |                   1.989984 |                 14.580265 |                30.085589 |   -12.590281 |       -28.095605 |        -12.590281 |              46.30558 |                 100.0 |                83.748115 |               98.585048 |  2.256 |   2.256 |  0.238381 | 0.0 |      0.0 | -0.254336 | -0.261947 |    -7.605297 |     0.774434 |       0.0 |       0.0 | 0.783269 |      0.0 |  -7.852177 | -22.161033 |   -1.917606 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.006368 | 0.000074 | 0.06 | 16.713021 | 16.713021 | -3.027916 | 0.662853 | -17.093966 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.006368 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000074 |
    | 2010-23-12 |  7.0 | -1.1 |              0.0 |                 7.345755 |        8.844396 |             94.9 |       8.6 |              1004.0 | 0.0 | 0.0 |    -1.1 |                  94.9 |                   0.0 |                      7.345755 |             8.844396 |   6.58 | -1.1 |    -1.1 |    6.893685 |         6.893685 |           6.893685 |          8.6 |                 5.634754 |                      5.634754 |                     4.876829 |                       0.41489 |                            0.41489 |             5.347381 |                  5.347381 |     998.652619 |    1.28304 | 8.6 |  -4.923062 | -11.265049 |              1.665037 |                  1.665037 |                   1.665037 |                 13.655186 |                25.124268 |    -11.99015 |       -23.459231 |         -11.99015 |             24.768101 |                 100.0 |                83.748115 |               98.584875 |   6.58 |  5.0995 |  0.461849 | 0.0 |      0.0 | -0.349525 | -0.214608 |    -3.045554 |     0.811741 |       0.0 |       0.0 | 0.472213 | 0.472213 | -10.790985 | -25.350365 |  -12.682119 |  0.99104 | 0.294648 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.003398 | 0.000039 | 0.06 |  22.61913 | 23.642546 |       0.0 |  0.38925 |   2.771083 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003398 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000039 |
    | 2010-24-12 |  0.2 | -2.9 |              0.0 |                 7.353123 |        8.873743 |             85.4 |       7.6 |              1008.0 | 0.0 | 0.0 |    -2.9 |                  85.4 |                   0.0 |                      7.353123 |             8.873743 |  0.188 | -2.9 |    -2.9 |    6.092093 |         6.092093 |           6.092093 |          7.6 |                 4.930236 |                      4.930236 |                     4.510204 |                      0.368688 |                           0.368688 |             4.210422 |                  4.210422 |    1003.789578 |   1.297296 | 8.6 |        0.0 |  24.507813 |               1.88611 |                   1.88611 |                    1.88611 |                 14.739672 |                36.555032 |   -12.853562 |       -34.668922 |        -12.853562 |             28.027061 |                 100.0 |                83.748115 |               98.584646 |  0.188 |   0.188 |  1.132637 | 0.0 |      0.0 |  0.200759 | -0.013188 |    -4.084636 |      0.78745 |       0.0 |       0.0 | 0.635341 | 0.635341 |   6.198095 | -13.916328 |   26.950689 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.001813 | 0.000021 | 0.06 | 23.244981 | 23.629787 |       0.0 | 0.555937 |  -13.13673 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001813 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000021 |
    | 2010-25-12 |  0.0 | -9.1 |              3.2 |                 7.362317 |       27.759642 |             87.2 |       3.8 |              1018.0 | 0.0 | 0.0 |    -9.1 |                  87.2 |                   3.2 |                      7.362317 |            27.759642 |    0.0 | -9.1 |    -9.1 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                 3.061705 |                      3.061705 |                     2.608657 |                      0.241745 |                           0.241745 |             2.669806 |                  2.669806 |    1015.330194 |   1.341718 | 8.6 |  -3.783378 |  43.959797 |              6.469629 |                  6.469629 |                   6.469629 |                 43.173862 |                71.439587 |   -36.704233 |       -64.969959 |        -36.704233 |             56.054123 |                 100.0 |                83.748115 |               98.584359 |    0.0 |     0.0 |  0.146249 | 0.0 |      0.0 | -0.023961 |       0.0 |   -11.110011 |     0.766941 |       0.0 |       0.0 | 1.669599 | 0.384806 |  -0.739762 | -13.816162 |   50.414034 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000968 | 0.000011 | 0.06 | 23.653748 | 23.653748 | -4.966676 | 0.731329 | -48.496527 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000968 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000011 |
    | 2010-26-12 |  1.2 | -8.6 |              4.3 |                 7.373328 |       32.328755 |             92.7 |       1.5 |              1019.0 | 0.0 | 0.0 |    -8.6 |                  92.7 |                   4.3 |                      7.373328 |            32.328755 |  1.128 | -8.6 |    -8.6 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                 3.184716 |                      3.184716 |                     1.819251 |                      0.250344 |                           0.250344 |             2.952232 |                  2.952232 |    1016.047768 |   1.340358 | 8.6 |    -13.967 |   9.228637 |              7.422691 |                  7.422691 |                   7.422691 |                 51.194924 |                55.663436 |   -43.772233 |       -48.240745 |        -43.772233 |            142.003778 |                 100.0 |                83.748115 |               98.584016 |  1.128 |   1.128 | -0.168556 | 0.0 |      0.0 | -0.253432 | -0.234661 |   -15.505106 |       0.7704 |       0.0 |       0.0 | 2.404399 |      0.0 |  -7.824276 | -27.094386 |   13.322083 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000516 | 0.000006 | 0.06 |  25.03518 |  25.03518 | -9.294782 | 0.699032 | -49.125164 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000516 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000006 |
    | 2010-27-12 |  0.9 | -5.3 |              0.0 |                 7.386143 |        9.004582 |             97.8 |       1.3 |              1017.0 | 0.0 | 0.0 |    -5.3 |                  97.8 |                   0.0 |                      7.386143 |             9.004582 |  0.846 | -5.3 |    -5.3 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                 4.112496 |                      4.112496 |                     2.537766 |                       0.31402 |                            0.31402 |             4.022021 |                  4.022021 |    1012.977979 |   1.320718 | 8.6 | -14.148047 | -16.174968 |              2.040737 |                  2.040737 |                   2.040737 |                 14.439522 |                15.774476 |   -12.398785 |       -13.733739 |        -12.398785 |            163.850513 |                 100.0 |                83.748115 |               98.583617 |  0.846 |   0.846 | -0.002298 | 0.0 |      0.0 | -0.310303 | -0.108462 |   -11.452219 |     0.773367 |       0.0 |       0.0 |   1.8547 |      0.0 |  -9.580069 | -22.562094 |  -18.408425 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000276 | 0.000003 | 0.06 | 26.191484 | 26.191484 | -7.169788 | 0.672269 | -24.350197 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000276 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000003 |
    | 2010-28-12 |  0.0 | -7.7 |              3.4 |                 7.400749 |       28.982321 |             95.4 |       3.0 |              1022.0 | 0.0 | 0.0 |    -7.7 |                  95.4 |                   3.4 |                      7.400749 |            28.982321 |    0.0 | -7.7 |    -7.7 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 3.417227 |                      3.417227 |                      2.55336 |                      0.266491 |                           0.266491 |             3.260035 |                  3.260035 |    1018.739965 |   1.339598 | 8.6 |  -7.012857 |  26.180199 |              6.701461 |                  6.701461 |                   6.701461 |                 42.703706 |                61.689685 |   -36.002245 |       -54.988224 |        -36.002245 |             71.001889 |                 100.0 |                83.748115 |               98.583162 |    0.0 |     0.0 | -0.122009 | 0.0 |      0.0 | -0.235576 |       0.0 |   -11.376223 |     0.768774 |       0.0 |       0.0 | 1.864481 |      0.0 |  -7.272988 | -21.497223 |   26.218013 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000147 | 0.000002 | 0.06 | 26.427059 | 26.427059 | -7.207601 |  0.71406 | -41.930396 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000147 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000002 |
    | 2010-29-12 |  0.0 | -9.2 |              0.0 |                  7.41713 |        9.127645 |             94.9 |       2.7 |              1024.0 | 0.0 | 0.0 |    -9.2 |                  94.9 |                   0.0 |                       7.41713 |             9.127645 |    0.0 | -9.2 |    -9.2 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 3.037615 |                      3.037615 |                     2.253172 |                      0.240056 |                           0.240056 |             2.882696 |                  2.882696 |    1021.117304 |   1.350039 | 8.6 | -12.075954 |   5.050047 |              2.150584 |                  2.150584 |                   2.150584 |                 15.336383 |                 34.61242 |     -13.1858 |       -32.461836 |          -13.1858 |             78.890988 |                 100.0 |                83.748115 |               98.582652 |    0.0 |     0.0 |  0.073039 | 0.0 |      0.0 | -0.196048 |       0.0 |   -12.917629 |     0.764388 |       0.0 |       0.0 | 2.136149 |      0.0 |  -6.052657 | -20.308937 |    6.100242 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000078 | 0.000001 | 0.06 | 26.623108 | 26.623108 | -8.257797 | 0.755971 | -38.380443 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000078 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000001 |
    | 2010-30-12 |  0.2 | -7.4 |              0.0 |                 7.435268 |        9.199989 |             96.0 |       1.4 |              1021.0 | 0.0 | 0.0 |    -7.4 |                  96.0 |                   0.0 |                      7.435268 |             9.199989 |  0.188 | -7.4 |    -7.4 |    1.122228 |         1.122228 |           1.122228 |          1.4 |                 3.498009 |                      3.498009 |                     2.475816 |                      0.272069 |                           0.272069 |             3.358088 |                  3.358088 |    1017.641912 |   1.336727 | 8.6 | -11.053568 |   4.228358 |              2.189479 |                  2.189479 |                   2.189479 |                 14.959543 |                28.188233 |   -12.770065 |       -25.998754 |        -12.770065 |            152.146905 |                 100.0 |                83.748115 |               98.582087 |  0.188 |   0.188 |  0.019065 | 0.0 |      0.0 | -0.190901 | -0.033653 |   -11.758294 |     0.762013 |       0.0 |       0.0 |  1.97269 |      0.0 |   -5.89375 | -16.542186 |    3.562818 |      1.0 |      0.3 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000042 |      0.0 | 0.06 | 27.002009 | 27.002009 | -7.625909 | 0.779531 | -34.008802 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000042 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-31-12 |  0.9 |  1.7 |              4.0 |                 7.455145 |       32.062235 |             96.5 |       4.2 |              1016.0 | 0.0 | 0.0 |     1.7 |                  96.5 |                   4.0 |                      7.455145 |            32.062235 |  0.846 |  1.7 |     1.7 |    3.366683 |         3.366683 |           3.366683 |          4.2 |                 6.907942 |                      6.907942 |                     4.470314 |                      0.496631 |                           0.496631 |             6.666164 |                  6.666164 |    1009.333836 |   1.284547 | 8.6 |  -9.794535 | -33.554262 |              7.749127 |                  7.749127 |                   7.749127 |                 40.225917 |                41.691328 |   -32.476791 |       -33.942201 |        -32.476791 |             50.715635 |                 100.0 |                83.748115 |               98.581468 |  0.846 | 0.06345 | -0.128022 | 0.0 |      0.0 | -0.924662 |  0.067075 |    -4.202158 |      0.75831 |       0.0 |       0.0 | 0.752906 | 0.752906 |  -28.54733 | -43.597429 |  -38.202558 | 0.973661 | 0.284405 |       0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000022 |      0.0 | 0.06 | 28.742042 | 28.772671 |       0.0 | 0.817554 |   8.145461 | 92.177353 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000022 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_v3_nadelw_winter_daily:

conifers (winter)
-----------------

Next, we repeat the integration test :ref:`lland_v3_acker_winter_daily` but select the
land use type |NADELW| instead of |ACKER|.  Like for |LAUBW| and |MISCHW|, |lland_v3|
now modifies the global radiation, the net longwave radiation, and the wind speed to
take the shadowing effects of the tree-canopies into account.  Here, the net effect of
these modifications is that the total water equivalent's peak amounts are two to three
times smaller than for land use type |ACKER|.  Also, snow evaporation (|EvS|) coincides
with interception (|EvI|) and soil evaporation (|EvB|), which never happens non-forest
sites:

.. integration-test::

    >>> lnk(NADELW)
    >>> conditions_nadelw_winter = test(
    ...     "lland_v3_nadelw_winter_daily",
    ...     axis1=(inputs.nied, fluxes.wada), axis2=(states.waes, states.wats),
    ...     get_conditions="2010-12-10")
    |       date | nied | teml | sunshineduration | possiblesunshineduration | globalradiation | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration | dailypossiblesunshineduration | dailyglobalradiation |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair |   g |        tz |         wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance |     nbes |     sbes |      evpo |       evi |      evb |       evs |     wnied | tempssurface | actualalbedo |   schmpot |      schm |  gefrpot |     gefr |   wlatsnow |  wsenssnow |      wsurf |      sff |      fvg |     wada |      qdb | qib1 | qib2 | qbb | qkap |     qdgz |      qah |       qa |     inzp |      wats |      waes |     esnow |     taus |       ebdn |      bowa |    qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-01-12 |  0.0 | -5.8 |              3.5 |                 7.641029 |       31.942798 |             70.3 |       6.9 |              1015.0 | 0.0 | 0.0 |    -5.8 |                  70.3 |                   3.5 |                      7.641029 |            31.942798 |    0.0 | -5.8 |    -5.8 |     5.53098 |          5.53098 |           2.449434 |          6.9 |                 3.958114 |                      3.958114 |                          0.0 |                      0.303557 |                           0.303557 |             2.782554 |                  2.782554 |    1012.217446 |   1.321192 | 8.6 |       0.0 |       34.8 |             28.109662 |                  5.220366 |                  28.109662 |                 46.472798 |                      0.0 |   -18.363136 |              0.0 |        -18.363136 |             13.623188 |                 100.0 |             27827.355778 |              101.116829 |      0.0 |      0.0 |  4.143156 |       0.0 | 0.675021 |       0.0 |       0.0 |          nan |         0.12 |       0.0 |       0.0 |      0.0 |      0.0 |        0.0 |        0.0 |        0.0 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |      0.0 |       0.0 |       0.0 |       0.0 |      nan |      -26.2 | 71.324979 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-02-12 | 11.1 | -3.0 |              0.0 |                 7.609644 |        9.873898 |             85.0 |       5.2 |              1007.0 | 0.0 | 0.0 |    -3.0 |                  85.0 |                   0.0 |                      7.609644 |             9.873898 | 10.434 | -3.0 |    -3.0 |    4.168274 |         4.168274 |            1.84595 |          5.2 |                 4.893489 |                      4.893489 |                     4.009414 |                      0.366257 |                           0.366257 |             4.159465 |                  4.159465 |    1002.840535 |   1.296511 | 8.6 |   -7.5456 | -11.481814 |              1.229506 |                  0.228337 |                   1.229506 |                  14.78976 |                 3.616419 |   -13.560254 |        -3.388082 |        -13.560254 |             18.076923 |                 100.0 |              2039.691073 |                99.50958 |    8.234 |    8.234 |  1.794683 |  1.794683 |      0.0 | -0.042372 | -0.597537 |    -5.631964 |     0.875479 |       0.0 |       0.0 | 0.290182 |      0.0 |  -1.308176 | -13.037488 | -10.957582 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.405317 |  8.276372 |  8.276372 | -1.121769 | 0.084415 |  -6.118186 | 71.324979 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-03-12 |  0.4 | -5.6 |              0.1 |                 7.579824 |         16.6835 |             95.9 |       2.0 |              1005.0 | 0.0 | 0.0 |    -5.6 |                  95.9 |                   0.1 |                      7.579824 |              16.6835 |  0.376 | -5.6 |    -5.6 |    1.603182 |         1.603182 |           0.709981 |          2.0 |                 4.019239 |                      4.019239 |                     4.166471 |                      0.307706 |                           0.307706 |              3.85445 |                   3.85445 |     1001.14555 |   1.306656 | 8.6 | -1.762038 |  20.202401 |              2.344922 |                  0.435485 |                   2.344922 |                 15.387201 |                17.344132 |    -13.04228 |       -16.908647 |         -13.04228 |                  47.0 |                 100.0 |             28069.029042 |              100.749738 |      0.0 |      0.0 |  0.129738 |  0.129738 |      0.0 |  0.055781 |       0.0 |    -5.129104 |     0.859447 |       0.0 |       0.0 | 0.265633 |      0.0 |   1.722139 |   1.476714 |    20.1075 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 0.651579 |  8.220592 |  8.220592 | -1.026867 | 0.147764 | -17.720587 | 71.324979 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-04-12 |  1.2 | -3.8 |              3.7 |                 7.551596 |       31.807427 |             90.6 |       4.3 |              1008.0 | 0.0 | 0.0 |    -3.8 |                  90.6 |                   3.7 |                      7.551596 |            31.807427 |  1.128 | -3.8 |    -3.8 |    3.446842 |         3.446842 |           1.526459 |          4.3 |                 4.608124 |                      4.608124 |                     4.014615 |                      0.347298 |                           0.347298 |              4.17496 |                   4.17496 |     1003.82504 |   1.301648 | 8.6 | -5.103529 |   3.069047 |              4.944298 |                  0.918227 |                   4.944298 |                  43.14024 |                13.393648 |   -38.195942 |       -12.475422 |        -38.195942 |             21.860465 |                 100.0 |              3347.214643 |                99.76651 |      0.0 |      0.0 |  0.584565 |  0.584565 |      0.0 | -0.040607 |       0.0 |    -5.615037 |     0.844555 |       0.0 |       0.0 | 0.288839 |      0.0 |  -1.253665 |  -8.063001 |   3.158756 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 1.195014 |  8.261198 |  8.261198 | -1.116576 | 0.213614 | -12.189634 | 71.324979 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-05-12 |  6.4 |  0.2 |              0.0 |                 7.524989 |        9.532511 |             96.5 |       4.0 |               996.0 | 0.0 | 0.0 |     0.2 |                  96.5 |                   0.0 |                      7.524989 |             9.532511 |  6.016 |  0.2 |     0.2 |    3.206365 |         3.206365 |           1.419962 |          4.0 |                 6.197477 |                      6.197477 |                     5.323469 |                      0.451275 |                           0.451275 |             5.980565 |                  5.980565 |     990.019435 |   1.266438 | 8.6 | -3.510615 |  -9.848154 |              1.559077 |                  0.289543 |                   1.559077 |                 13.205056 |                 3.867776 |    -11.64598 |        -3.578233 |         -11.64598 |                  23.5 |                 100.0 |               763.990552 |               96.624999 | 5.011014 | 2.254956 |  0.342799 |  0.342799 |      0.0 | -0.160024 |   0.03762 |    -1.869256 |     0.836446 |       0.0 |       0.0 | 0.187585 | 0.187585 |  -4.940454 |  -8.839733 | -10.201955 | 0.979763 | 0.287981 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 1.857201 | 10.832795 | 13.432236 |       0.0 | 0.252752 |    6.25852 | 71.324979 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-06-12 |  0.0 | -1.1 |              0.1 |                 7.500028 |       16.141484 |             99.0 |       2.4 |               998.0 | 0.0 | 0.0 |    -1.1 |                  99.0 |                   0.1 |                      7.500028 |            16.141484 |    0.0 | -1.1 |    -1.1 |    1.923819 |         1.923819 |           0.851977 |          2.4 |                 5.634754 |                      5.634754 |                     5.474524 |                       0.41489 |                            0.41489 |             5.578406 |                  5.578406 |     992.421594 |   1.275245 | 8.6 |       0.0 |   8.946827 |              3.138454 |                  0.582856 |                   3.138454 |                 14.093918 |                12.038612 |   -10.955464 |       -11.455757 |        -10.955464 |             39.166667 |                 100.0 |              1010.452998 |               97.419094 |      0.0 |      0.0 |  0.033701 |  0.033701 |      0.0 | -0.019917 |       0.0 |    -1.491138 |     0.805566 |       0.0 |       0.0 | 0.149667 | 0.149667 |  -0.614896 |   -1.31546 |     9.5254 | 0.980884 |  0.28864 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 |   1.8235 | 10.998746 | 13.452153 |       0.0 | 0.428144 |   5.911693 | 71.324979 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-07-12 |  0.0 | -2.9 |              0.0 |                 7.476739 |        9.341429 |             99.2 |       1.9 |              1004.0 | 0.0 | 0.0 |    -2.9 |                  99.2 |                   0.0 |                      7.476739 |             9.341429 |    0.0 | -2.9 |    -2.9 |    1.523023 |         1.523023 |           0.674482 |          1.9 |                 4.930236 |                      4.930236 |                     5.036854 |                      0.368688 |                           0.368688 |             4.890794 |                  4.890794 |     999.109206 |   1.291808 | 8.6 |       0.0 |  15.681283 |              2.041655 |                  0.379164 |                   2.041655 |                 13.815559 |                15.392937 |   -11.773904 |       -15.013773 |        -11.773904 |             49.473684 |                 100.0 |              1874.966213 |               98.634659 |      0.0 |      0.0 |  0.001011 |  0.001011 |      0.0 |  0.025639 |       0.0 |    -2.613547 |     0.781441 |       0.0 |       0.0 | 0.260254 | 0.260254 |   0.791547 |   0.882037 |  16.687357 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 1.822489 | 11.237542 | 13.426514 |       0.0 | 0.603536 |  -1.169591 | 71.324979 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-08-12 |  0.0 | -2.9 |              0.0 |                 7.455145 |        9.256764 |             99.1 |       1.8 |              1006.0 | 0.0 | 0.0 |    -2.9 |                  99.1 |                   0.0 |                      7.455145 |             9.256764 |    0.0 | -2.9 |    -2.9 |    1.442864 |         1.442864 |           0.638983 |          1.8 |                 4.930236 |                      4.930236 |                      4.98944 |                      0.368688 |                           0.368688 |             4.885864 |                  4.885864 |    1001.114136 |   1.294388 | 8.6 | -0.336842 |  14.420543 |              2.202435 |                  0.409024 |                   2.202435 |                 13.821847 |                14.831397 |   -11.619412 |       -14.422374 |        -11.619412 |             52.222222 |                 100.0 |              1875.429442 |               98.515451 |      0.0 |      0.0 |  0.005587 |  0.005587 |      0.0 |  0.017846 |       0.0 |    -2.740266 |     0.762073 |       0.0 |       0.0 | 0.267884 | 0.267884 |   0.550961 |   0.482776 |  15.456111 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 1.816902 | 11.490133 | 13.408669 |       0.0 | 0.778928 |  -6.990133 | 71.324979 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-09-12 |  0.7 | -2.1 |              4.4 |                 7.435268 |       33.405815 |             96.9 |       2.7 |              1009.0 | 0.0 | 0.0 |    -2.1 |                  96.9 |                   4.4 |                      7.435268 |            33.405815 |  0.658 | -2.1 |    -2.1 |    2.164296 |         2.164296 |           0.958474 |          2.7 |                 5.233109 |                      5.233109 |                     4.791747 |                      0.388644 |                           0.388644 |             5.070883 |                  5.070883 |    1003.929117 |   1.294334 | 8.6 | -2.013158 |    7.60331 |              8.412801 |                  1.562377 |                   8.412801 |                 46.364476 |                16.297167 |   -37.951675 |       -14.734789 |        -37.951675 |             34.814815 |                 100.0 |              1366.137754 |               97.824519 | 0.274902 | 0.274902 | -0.145824 | -0.145824 |      0.0 | -0.056229 | -0.013965 |    -3.280377 |     0.748164 |       0.0 |       0.0 | 0.320391 | 0.320391 |  -1.735965 |   -4.17093 |   8.827894 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 2.345824 | 12.135087 | 13.739799 |       0.0 | 0.930578 |  -5.993443 | 71.324979 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-10-12 |  3.3 | -3.1 |              0.1 |                  7.41713 |       15.591158 |             95.0 |       3.0 |              1018.0 | 0.0 | 0.0 |    -3.1 |                  95.0 |                   0.1 |                       7.41713 |            15.591158 |  3.102 | -3.1 |    -3.1 |    2.404774 |         2.404774 |           1.064971 |          3.0 |                 4.856984 |                      4.856984 |                     4.725379 |                       0.36384 |                            0.36384 |             4.614135 |                  4.614135 |    1013.385865 |   1.310959 | 8.6 | -1.726112 |  10.440055 |              3.702385 |                  0.687586 |                   3.702385 |                 14.892721 |                13.074297 |   -11.190336 |       -12.386711 |        -11.190336 |             31.333333 |                 100.0 |              2093.666507 |               98.468592 | 3.247824 | 3.247824 |  0.333091 |  0.333091 |      0.0 |  0.023489 | -0.243549 |    -3.466121 |     0.762533 |       0.0 |       0.0 | 0.403366 | 0.403366 |   0.725197 |  -1.356094 |  11.755814 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |      0.0 |      0.0 | 1.866909 | 15.764449 | 16.964134 |       0.0 | 0.774318 |  -7.833498 | 71.324979 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-11-12 |  9.4 |  5.1 |              0.2 |                 7.400749 |       15.891544 |             90.6 |       7.4 |              1004.0 | 0.0 | 0.0 |     5.1 |                  90.6 |                   0.2 |                      7.400749 |            15.891544 |  8.836 |  5.1 |     5.1 |    5.931775 |         5.931775 |           2.626929 |          7.4 |                 8.790172 |                      8.790172 |                       6.1078 |                      0.614118 |                           0.614118 |             7.963896 |                  7.963896 |     996.036104 |   1.253213 | 8.6 | -2.256047 | -13.536284 |              4.043116 |                  0.750864 |                   4.043116 |                 13.525343 |               -11.576952 |    -9.482227 |        12.327816 |         -9.482227 |             12.702703 |                 100.0 |               405.842551 |               92.890931 | 8.502909 |      0.0 |  2.300498 |       2.2 | 0.020689 | -0.656354 |  1.378787 |          0.0 |     0.745581 | 13.469568 | 13.469568 |      0.0 |      0.0 | -20.263798 | -31.635741 | -64.227355 | 0.953751 | 0.272893 | 22.74617 | 7.343664 |  0.0 |  0.0 | 0.0 |  0.0 | 7.343664 | 1.890601 | 0.021882 |      0.0 |  2.361698 |  3.377227 |       0.0 | 0.961631 |  14.302786 | 86.706796 | 7.343664 |   0.0 |   0.0 |   0.0 |  0.0 | 1.890601 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.021882 |
    | 2010-12-12 |  0.2 |  0.2 |              0.7 |                 7.386143 |       17.823519 |             78.7 |       5.7 |              1012.0 | 0.0 | 0.0 |     0.2 |                  78.7 |                   0.7 |                      7.386143 |            17.823519 |  0.188 |  0.2 |     0.2 |     4.56907 |          4.56907 |           2.023445 |          5.7 |                 6.197477 |                      6.197477 |                     5.568922 |                      0.451275 |                           0.451275 |             4.877414 |                  4.877414 |    1007.122586 |    1.28736 | 8.6 |       0.0 |   7.556989 |               4.76955 |                  0.885774 |                    4.76955 |                 20.069431 |                 9.846507 |   -15.299881 |        -8.960734 |        -15.299881 |             16.491228 |                 100.0 |               655.903496 |               95.337074 |      0.0 |      0.0 |  3.264658 |     0.188 | 0.683908 |  0.206468 |       0.0 |    -1.259498 |     0.732401 |       0.0 |       0.0 | 0.034647 | 0.034647 |   6.374329 |  -7.644139 |   7.690923 | 0.950379 | 0.270966 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 2.543248 | 0.029436 |      0.0 |  2.249843 |   3.17076 |       0.0 | 1.137023 |  15.345798 | 86.022888 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 2.543248 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.029436 |
    | 2010-13-12 |  2.9 | -3.5 |              0.0 |                 7.373328 |        8.941324 |             90.5 |       2.2 |              1020.0 | 0.0 | 0.0 |    -3.5 |                  90.5 |                   0.0 |                      7.373328 |             8.941324 |  2.726 | -3.5 |    -3.5 |    1.763501 |         1.763501 |           0.780979 |          2.2 |                 4.713362 |                      4.713362 |                     4.810831 |                      0.354306 |                           0.354306 |             4.265593 |                  4.265593 |    1015.734407 |   1.315658 | 8.6 |       0.0 |  19.364316 |              2.447669 |                  0.454567 |                   2.447669 |                 14.516908 |                16.130859 |   -12.069239 |       -15.676292 |        -12.069239 |             42.727273 |                 100.0 |              2194.252498 |               98.289809 |    0.526 |    0.526 |  0.463667 |  0.463667 |      0.0 |  0.101005 | -0.044533 |    -3.227386 |     0.726252 |       0.0 |       0.0 | 0.093316 | 0.093316 |   3.118348 |   0.885877 |  19.680517 | 0.985186 | 0.291177 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 1.357105 | 0.015707 | 1.736333 |  2.790766 |  3.595755 |       0.0 | 1.230056 |   4.581482 | 86.022888 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.357105 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.015707 |
    | 2010-14-12 |  0.0 | -3.9 |              1.2 |                 7.362317 |       19.668068 |             88.1 |       3.1 |              1025.0 | 0.0 | 0.0 |    -3.9 |                  88.1 |                   1.2 |                      7.362317 |            19.668068 |    0.0 | -3.9 |    -3.9 |    2.484933 |         2.484933 |            1.10047 |          3.1 |                  4.57351 |                       4.57351 |                     4.646441 |                      0.344989 |                           0.344989 |             4.029262 |                  4.029262 |    1020.970738 |   1.324197 | 8.6 |       0.0 |  22.140455 |              5.586661 |                  1.037523 |                   5.586661 |                 24.390008 |                  18.6965 |   -18.803347 |       -17.658977 |        -18.803347 |             30.322581 |                 100.0 |              3007.000202 |               98.587575 |      0.0 |      0.0 |   0.74798 |   0.74798 |      0.0 |  0.132317 |       0.0 |    -3.690076 |     0.715953 |       0.0 |       0.0 | 0.101676 | 0.101676 |   4.085059 |   0.789473 |  22.533509 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.724166 | 0.008382 | 0.988353 |  2.786006 |  3.463438 |       0.0 | 1.405448 |  -8.958973 | 86.022888 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.724166 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.008382 |
    | 2010-15-12 |  0.0 | -6.3 |              3.0 |                 7.353123 |       26.831581 |             92.8 |       2.8 |              1023.0 | 0.0 | 0.0 |    -6.3 |                  92.8 |                   3.0 |                      7.353123 |            26.831581 |    0.0 | -6.3 |    -6.3 |    2.244455 |         2.244455 |           0.993973 |          2.8 |                 3.808888 |                      3.808888 |                       3.8451 |                      0.293396 |                           0.293396 |             3.534648 |                  3.534648 |    1019.465352 |    1.33374 | 8.6 | -2.580184 |  21.581439 |              7.860224 |                  1.459756 |                   7.860224 |                 39.328283 |                21.257217 |   -31.468059 |       -19.797461 |        -31.468059 |             33.571429 |                 100.0 |             22983.498688 |               99.386047 |      0.0 |      0.0 |  0.185292 |  0.185292 |      0.0 |  0.063542 |       0.0 |    -6.177091 |     0.707053 |       0.0 |       0.0 | 0.160142 | 0.160142 |   1.961759 |   0.441288 |  22.200508 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.386423 | 0.004472 | 0.803061 |  2.892097 |  3.399896 |       0.0 |  1.58084 | -21.940412 | 86.022888 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.386423 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.004472 |
    | 2010-16-12 |  6.7 | -3.5 |              0.0 |                 7.345755 |        8.837738 |             98.1 |       3.7 |              1000.0 | 0.0 | 0.0 |    -3.5 |                  98.1 |                   0.0 |                      7.345755 |             8.837738 |  6.298 | -3.5 |    -3.5 |    2.965888 |         2.965888 |           1.313465 |          3.7 |                 4.713362 |                      4.713362 |                     4.106014 |                      0.354306 |                           0.354306 |             4.623808 |                  4.623808 |     995.376192 |   1.289645 | 8.6 | -6.318839 |  -5.989098 |              2.208317 |                  0.410116 |                   2.208317 |                   14.0252 |                  6.34533 |   -11.816883 |        -5.935214 |        -11.816883 |             25.405405 |                 100.0 |              2154.774967 |               98.113252 | 4.901061 | 4.901061 |  0.125607 |  0.125607 |      0.0 | -0.121069 | -0.414945 |    -5.320656 |     0.750126 |       0.0 |       0.0 | 0.293333 | 0.293333 |   -3.73781 |  -7.467498 |  -5.270094 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |   0.2062 | 0.002387 | 2.074393 |  8.204433 |  8.422026 |       0.0 | 0.907629 |  -7.351314 | 86.022888 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.2062 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.002387 |
    | 2010-17-12 |  0.0 | -7.3 |              0.0 |                 7.340222 |        8.817416 |             96.6 |       0.9 |               991.0 | 0.0 | 0.0 |    -7.3 |                  96.6 |                   0.0 |                      7.340222 |             8.817416 |    0.0 | -7.3 |    -7.3 |    0.721432 |         0.721432 |           0.319491 |          0.9 |                  3.52531 |                       3.52531 |                     3.867045 |                      0.273951 |                           0.273951 |             3.405449 |                  3.405449 |     987.594551 |   1.296889 | 8.6 | -2.117178 |  23.915569 |              2.325835 |                  0.431941 |                   2.325835 |                 14.906398 |                20.577376 |   -12.580563 |       -20.145435 |        -12.580563 |            104.444444 |                 100.0 |             22983.498688 |               99.310924 |      0.0 |      0.0 |  0.027374 |  0.027374 |      0.0 |   0.06608 |       0.0 |    -6.103107 |     0.736223 |       0.0 |       0.0 | 0.329975 | 0.217594 |    2.04011 |   3.005622 |  25.191167 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |  0.11003 | 0.001274 | 2.047019 |  8.355946 |  8.355946 | -0.434437 | 1.083021 | -22.666883 | 86.022888 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.11003 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001274 |
    | 2010-18-12 |  0.1 | -5.8 |              0.0 |                  7.33653 |        8.804185 |             97.7 |       1.3 |               997.0 | 0.0 | 0.0 |    -5.8 |                  97.7 |                   0.0 |                       7.33653 |             8.804185 |  0.094 | -5.8 |    -5.8 |    1.042069 |         1.042069 |           0.461488 |          1.3 |                 3.958114 |                      3.958114 |                     3.588204 |                      0.303557 |                           0.303557 |             3.867077 |                  3.867077 |     993.132923 |   1.297204 | 8.6 | -6.528062 |   3.264901 |              2.381905 |                  0.442354 |                   2.381905 |                 14.549007 |                 9.530158 |   -12.167102 |        -9.087804 |        -12.167102 |             72.307692 |                 100.0 |             22983.498688 |               99.289445 |      0.0 |      0.0 |  0.029554 |  0.029554 |      0.0 | -0.043534 |       0.0 |    -7.072213 |     0.729458 |       0.0 |       0.0 | 0.369786 |      0.0 |  -1.344041 |  -3.483801 |   4.259962 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.058713 |  0.00068 | 2.111465 |   8.39948 |   8.39948 | -1.429498 | 1.180569 | -17.331784 | 86.022888 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.058713 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.00068 |
    | 2010-19-12 |  0.0 | -7.9 |              0.0 |                 7.334683 |        8.798044 |             95.8 |       1.9 |               997.0 | 0.0 | 0.0 |    -7.9 |                  95.8 |                   0.0 |                      7.334683 |             8.798044 |    0.0 | -7.9 |    -7.9 |    1.523023 |         1.523023 |           0.674482 |          1.9 |                 3.364296 |                      3.364296 |                     3.393896 |                      0.262827 |                           0.262827 |             3.222996 |                  3.222996 |     993.777004 |   1.307793 | 8.6 | -4.991554 |  16.777576 |              2.412358 |                  0.448009 |                   2.412358 |                 15.062045 |                16.106832 |   -12.649688 |       -15.658823 |        -12.649688 |             49.473684 |                 100.0 |             22983.498688 |               99.278703 |      0.0 |      0.0 |  0.112396 |  0.112396 |      0.0 |  0.029999 |       0.0 |    -7.787816 |     0.725807 |       0.0 |       0.0 | 0.409325 |      0.0 |   0.926166 |   0.345432 |  16.930421 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |  0.03133 | 0.000363 | 1.999069 |  8.369481 |  8.369481 | -1.582343 | 1.237097 |  -25.50936 | 86.022888 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.03133 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000363 |
    | 2010-20-12 |  0.2 | -5.7 |              0.2 |                 7.334683 |       15.464746 |             97.5 |       2.4 |               997.0 | 0.0 | 0.0 |    -5.7 |                  97.5 |                   0.2 |                      7.334683 |            15.464746 |  0.188 | -5.7 |    -5.7 |    1.923819 |         1.923819 |           0.851977 |          2.4 |                 3.988573 |                      3.988573 |                     3.514448 |                      0.305625 |                           0.305625 |             3.888858 |                  3.888858 |     993.111142 |   1.296708 | 8.6 | -7.346696 |  -0.041947 |              4.291062 |                  0.796911 |                   4.291062 |                  16.12634 |                 8.389385 |   -11.835278 |        -7.592473 |        -11.835278 |             39.166667 |                 100.0 |             22983.498688 |               99.278703 |      0.0 |      0.0 |  0.092286 |  0.092286 |      0.0 | -0.071784 |       0.0 |    -7.339705 |     0.722526 |       0.0 |       0.0 | 0.384394 |      0.0 |  -2.216201 |  -5.514594 |  -0.138321 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.016718 | 0.000193 | 2.094783 |  8.441265 |  8.441265 | -1.485969 | 1.290472 | -16.867413 | 86.022888 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.016718 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000193 |
    | 2010-21-12 |  0.2 | -5.6 |              0.8 |                  7.33653 |       17.879541 |             95.0 |       2.3 |              1004.0 | 0.0 | 0.0 |    -5.6 |                  95.0 |                   0.8 |                       7.33653 |            17.879541 |  0.188 | -5.6 |    -5.6 |     1.84366 |          1.84366 |           0.816478 |          2.3 |                 4.019239 |                      4.019239 |                     3.770073 |                      0.307706 |                           0.307706 |             3.818277 |                  3.818277 |    1000.181723 |   1.305372 | 8.6 | -4.857815 |   9.450477 |              5.008382 |                  0.930128 |                   5.008382 |                 21.069426 |                13.244216 |   -16.061044 |       -12.314088 |        -16.061044 |             40.869565 |                 100.0 |             22983.498688 |               99.289445 | 0.082783 | 0.082783 |  0.167051 |  0.167051 |      0.0 | -0.009086 | -0.011214 |    -6.432895 |     0.719882 |       0.0 |       0.0 | 0.343125 |      0.0 |  -0.280508 |  -2.753853 |   9.279726 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.008921 | 0.000103 | 2.032949 |  8.533134 |  8.533134 | -1.326432 | 1.335375 |  -17.71789 | 86.022888 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.008921 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000103 |
    | 2010-22-12 |  2.4 | -4.8 |              0.0 |                 7.340222 |        8.822161 |             93.8 |       4.6 |              1007.0 | 0.0 | 0.0 |    -4.8 |                  93.8 |                   0.0 |                      7.340222 |             8.822161 |  2.256 | -4.8 |    -4.8 |     3.68732 |          3.68732 |           1.632956 |          4.6 |                 4.272186 |                      4.272186 |                     3.939138 |                      0.324793 |                           0.324793 |             4.007311 |                  4.007311 |    1002.992689 |   1.305282 | 8.6 | -5.102752 |   4.559358 |               2.34861 |                   0.43617 |                    2.34861 |                 14.580265 |                10.388081 |   -12.231655 |        -9.951911 |        -12.231655 |             20.434783 |                 100.0 |             16304.380661 |               99.258251 | 2.088949 | 2.088949 |  0.580986 |  0.580986 |      0.0 | -0.017927 |  -0.24255 |    -5.862645 |     0.733783 |       0.0 |       0.0 | 0.389676 |      0.0 |  -0.553455 |  -4.901695 |    4.49676 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |  0.00476 | 0.000055 | 1.619014 |  10.64001 |  10.64001 | -1.506385 | 1.117179 | -13.677248 | 86.022888 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00476 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000055 |
    | 2010-23-12 |  7.0 | -1.1 |              0.0 |                 7.345755 |        8.844396 |             94.9 |       8.6 |              1004.0 | 0.0 | 0.0 |    -1.1 |                  94.9 |                   0.0 |                      7.345755 |             8.844396 |   6.58 | -1.1 |    -1.1 |    6.893685 |         6.893685 |           3.052917 |          8.6 |                 5.634754 |                      5.634754 |                     5.020182 |                       0.41489 |                            0.41489 |             5.347381 |                  5.347381 |     998.652619 |    1.28304 | 8.6 | -3.939047 |  -7.686385 |              2.004812 |                  0.372322 |                   2.004812 |                 13.655186 |                 6.831659 |   -11.650374 |        -6.459337 |        -11.650374 |             10.930233 |                 100.0 |               837.076304 |               96.047796 | 5.999014 | 4.649236 |  1.086744 |  1.086744 |      0.0 | -0.128418 | -0.195659 |    -2.657983 |     0.773324 |       0.0 |       0.0 | 0.299268 | 0.299268 |  -3.964676 | -10.726197 |  -8.231536 | 0.991563 | 0.294959 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |  0.00254 | 0.000029 | 1.113256 | 15.708823 | 16.767441 |       0.0 | 0.672649 |   2.609137 | 86.022888 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00254 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000029 |
    | 2010-24-12 |  0.2 | -2.9 |              0.0 |                 7.353123 |        8.873743 |             85.4 |       7.6 |              1008.0 | 0.0 | 0.0 |    -2.9 |                  85.4 |                   0.0 |                      7.353123 |             8.873743 |  0.188 | -2.9 |    -2.9 |    6.092093 |         6.092093 |           2.697927 |          7.6 |                 4.930236 |                      4.930236 |                     4.859622 |                      0.368688 |                           0.368688 |             4.210422 |                  4.210422 |    1003.789578 |   1.297296 | 8.6 |       0.0 |  18.556505 |              2.170067 |                  0.403012 |                   2.170067 |                 14.739672 |                14.293546 |   -12.569605 |       -13.890533 |        -12.569605 |             12.368421 |                 100.0 |              1589.438278 |               97.678607 |      0.0 |      0.0 |  2.588686 |  1.301256 | 0.210203 |  0.233775 |       0.0 |    -3.092751 |     0.755451 |       0.0 |       0.0 | 0.345052 | 0.345052 |   7.217399 |  -1.217546 |  19.890385 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.001355 | 0.000016 |      0.0 | 15.830048 | 16.533666 |       0.0 | 0.848041 |  -7.347368 | 85.812685 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001355 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000016 |
    | 2010-25-12 |  0.0 | -9.1 |              3.2 |                 7.362317 |       27.759642 |             87.2 |       3.8 |              1018.0 | 0.0 | 0.0 |    -9.1 |                  87.2 |                   3.2 |                      7.362317 |            27.759642 |    0.0 | -9.1 |    -9.1 |    3.046047 |         3.046047 |           1.348964 |          3.8 |                 3.061705 |                      3.061705 |                     3.406529 |                      0.241745 |                           0.241745 |             2.669806 |                  2.669806 |    1015.330194 |   1.341718 | 8.6 | -2.116042 |  33.744938 |              7.198741 |                  1.336909 |                   7.198741 |                 43.173862 |                 27.26332 |   -35.975121 |       -25.926411 |        -35.975121 |             24.736842 |                 100.0 |             23054.094894 |               99.440053 |      0.0 |      0.0 |  0.605575 |       0.0 | 0.151738 |  0.174644 |       0.0 |    -7.740198 |     0.740676 |       0.0 |       0.0 | 0.834984 | 0.703618 |   5.391841 |   5.654519 |  36.972772 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000723 | 0.000008 |      0.0 | 16.359022 | 16.359022 | -0.507831 | 1.023433 | -32.492307 | 85.660946 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000723 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000008 |
    | 2010-26-12 |  1.2 | -8.6 |              4.3 |                 7.373328 |       32.328755 |             92.7 |       1.5 |              1019.0 | 0.0 | 0.0 |    -8.6 |                  92.7 |                   4.3 |                      7.373328 |            32.328755 |  1.128 | -8.6 |    -8.6 |    1.202387 |         1.202387 |           0.532486 |          1.5 |                 3.184716 |                      3.184716 |                     2.803445 |                      0.250344 |                           0.250344 |             2.952232 |                  2.952232 |    1016.047768 |   1.340358 | 8.6 | -9.357784 |   5.116691 |              8.661508 |                  1.608566 |                   8.661508 |                 51.194924 |                15.598098 |   -42.533416 |       -13.989532 |        -42.533416 |             62.666667 |                 100.0 |             23105.086914 |               99.504617 |      0.0 |      0.0 |  -0.08782 |  -0.08782 |      0.0 |  -0.02419 |       0.0 |   -10.210566 |      0.73208 |       0.0 |       0.0 | 1.045218 |      0.0 |  -0.746832 |  -4.593297 |   8.649402 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000386 | 0.000004 |  1.21582 | 16.383212 | 16.383212 | -4.040543 | 1.141686 | -29.008997 | 85.660946 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000386 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000004 |
    | 2010-27-12 |  0.9 | -5.3 |              0.0 |                 7.386143 |        9.004582 |             97.8 |       1.3 |              1017.0 | 0.0 | 0.0 |    -5.3 |                  97.8 |                   0.0 |                      7.386143 |             9.004582 |  0.846 | -5.3 |    -5.3 |    1.042069 |         1.042069 |           0.461488 |          1.3 |                 4.112496 |                      4.112496 |                     3.425855 |                       0.31402 |                            0.31402 |             4.022021 |                  4.022021 |    1012.977979 |   1.320718 | 8.6 | -8.354591 |  -4.121585 |               2.43985 |                  0.453115 |                    2.43985 |                 14.439522 |                 4.686537 |   -11.999672 |        -4.233422 |        -11.999672 |             72.307692 |                 100.0 |             23105.086914 |               99.579536 |      0.0 |      0.0 |  0.030409 |  0.030409 |      0.0 | -0.093066 |       0.0 |     -7.66766 |     0.729044 |       0.0 |       0.0 | 0.786071 |      0.0 |  -2.873249 |  -6.483554 |  -5.123381 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000206 | 0.000002 | 2.031411 | 16.476278 | 16.476278 | -3.038746 | 1.186839 | -16.287412 | 85.660946 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000206 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000002 |
    | 2010-28-12 |  0.0 | -7.7 |              3.4 |                 7.400749 |       28.982321 |             95.4 |       3.0 |              1022.0 | 0.0 | 0.0 |    -7.7 |                  95.4 |                   3.4 |                      7.400749 |            28.982321 |    0.0 | -7.7 |    -7.7 |    2.404774 |         2.404774 |           1.064971 |          3.0 |                 3.417227 |                      3.417227 |                     3.373812 |                      0.266491 |                           0.266491 |             3.260035 |                  3.260035 |    1018.739965 |   1.339598 | 8.6 | -4.690775 |   19.03839 |              7.953772 |                  1.477129 |                   7.953772 |                 42.703706 |                20.476117 |   -34.749934 |       -18.998988 |        -34.749934 |             31.333333 |                 100.0 |             23105.086914 |                99.66506 |      0.0 |      0.0 |  0.014952 |  0.014952 |      0.0 |  0.024024 |       0.0 |     -7.86384 |     0.725565 |       0.0 |       0.0 | 0.810762 |      0.0 |   0.741708 |  -0.606855 |  19.133841 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 |  0.00011 | 0.000001 | 2.016459 | 16.452253 | 16.452253 | -3.134197 | 1.240961 | -26.725803 | 85.660946 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.00011 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000001 |
    | 2010-29-12 |  0.0 | -9.2 |              0.0 |                  7.41713 |        9.127645 |             94.9 |       2.7 |              1024.0 | 0.0 | 0.0 |    -9.2 |                  94.9 |                   0.0 |                       7.41713 |             9.127645 |    0.0 | -9.2 |    -9.2 |    2.164296 |         2.164296 |           0.958474 |          2.7 |                 3.037615 |                      3.037615 |                     2.932544 |                      0.240056 |                           0.240056 |             2.882696 |                  2.882696 |    1021.117304 |   1.350039 | 8.6 | -7.697031 |  11.685178 |              2.534671 |                  0.470725 |                   2.534671 |                 15.336383 |                14.120895 |   -12.801712 |       -13.650171 |        -12.801712 |             34.814815 |                 100.0 |             23105.086914 |               99.761152 |      0.0 |      0.0 |  0.205881 |  0.205881 |      0.0 |  0.010041 |       0.0 |    -9.644561 |     0.722308 |       0.0 |       0.0 | 0.992905 |      0.0 |   0.310004 |  -1.570882 |  12.389293 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000059 | 0.000001 | 1.810578 | 16.442212 | 16.442212 | -3.838313 | 1.294107 | -29.810981 | 85.660946 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000059 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000001 |
    | 2010-30-12 |  0.2 | -7.4 |              0.0 |                 7.435268 |        9.199989 |             96.0 |       1.4 |              1021.0 | 0.0 | 0.0 |    -7.4 |                  96.0 |                   0.0 |                      7.435268 |             9.199989 |  0.188 | -7.4 |    -7.4 |    1.122228 |         1.122228 |           0.496987 |          1.4 |                 3.498009 |                      3.498009 |                        3.082 |                      0.272069 |                           0.272069 |             3.358088 |                  3.358088 |    1017.641912 |   1.336727 | 8.6 | -8.585562 |   2.584381 |              2.580064 |                  0.479155 |                   2.580064 |                 14.959543 |                 8.687356 |   -12.379479 |        -8.208202 |        -12.379479 |             67.142857 |                 100.0 |             23105.086914 |               99.867769 |      0.0 |      0.0 |  0.077402 |  0.077402 |      0.0 | -0.043993 |       0.0 |    -9.016293 |     0.719558 |       0.0 |       0.0 | 0.927658 |      0.0 |  -1.358221 |  -4.517826 |   2.332154 |      1.0 |      0.3 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000031 |      0.0 | 1.921177 | 16.486206 | 16.486206 | -3.586087 | 1.340999 | -23.795361 | 85.660946 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000031 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-31-12 |  0.9 |  1.7 |              4.0 |                 7.455145 |       32.062235 |             96.5 |       4.2 |              1016.0 | 0.0 | 0.0 |     1.7 |                  96.5 |                   4.0 |                      7.455145 |            32.062235 |  0.846 |  1.7 |     1.7 |    3.366683 |         3.366683 |            1.49096 |          4.2 |                 6.907942 |                      6.907942 |                     5.130481 |                      0.496631 |                           0.496631 |             6.666164 |                  6.666164 |    1009.333836 |   1.284547 | 8.6 | -6.853064 | -26.920086 |              9.074296 |                  1.685226 |                   9.074296 |                 40.225917 |                 1.916828 |   -31.151622 |        -0.231602 |        -31.151622 |             22.380952 |                 100.0 |               488.700714 |               94.238334 | 0.567177 | 0.042538 |  0.111701 |  0.111701 |      0.0 | -0.383932 |  0.044969 |    -2.366383 |     0.716979 |       0.0 |       0.0 | 0.260314 | 0.260314 | -11.853233 | -17.833268 | -29.454898 | 0.962088 | 0.277684 |      0.0 |      0.0 |  0.0 |  0.0 | 0.0 |  0.0 |      0.0 | 0.000017 |      0.0 | 2.088299 | 17.167039 | 17.437315 |       0.0 | 1.386749 |  11.724724 | 85.660946 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000017 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _lland_v3_acker_winter_ktschnee_daily:

acre (snow surface temperature)
-------------------------------

In integration test :ref:`lland_v3_acker_winter_daily`, |lland_v3| assumes that the
snow-layer has a constant temperature over its complete depth.  For less simplified
modelling of the processes within the snow layer, one can enable an iterative
adjustment of the snow surface temperature (see method |Return_TempSSurface_V1|).
Therefore, one must set the thermal conductivity of the top snow layer (|KTSchnee|) to
a value smaller than |numpy.inf|.  The lower the conductivity, the higher the possible
differences between the bulk and the surface temperature of the snow layer. Setting it
to zero (which is not advisable) would prevent any thermal exchange.  Setting it to
|numpy.inf| results in an infinitely fast exchange, avoiding any temperature
differences within the snow layer.

We show what happens when we use the default |KTSchnee| value used by `LARSIM`_.  In
integration test :ref:`lland_v3_acker_winter_daily`, with :math:`KTSchnee = inf`,
nearly the complete frozen water equivalent melts on December 11th, where the average
air temperature is 5 °C.  In our next test, with :math:`KTSchnee = 5 W/m²/K`, the
limited thermal conductivity of the top snow layer weakens the energy flux into the
snow layer and fewer snow melts.  The snow surface temperature rises faster than the
bulk temperature, and the higher surface temperature results in smaller sensible
(|WSensSnow|) and latent (|WLatSnow|) heat fluxes from the atmosphere into the snow
layer and to larger longwave radiation losses (|NetLongwaveRadiationSnow|):

.. integration-test::

    >>> lnk(ACKER)
    >>> ktschnee(5.0)
    >>> conditions_acker_winter_ktschnee = test(
    ...     "lland_v3_acker_winter_ktschnee_daily",
    ...     axis1=(inputs.nied, fluxes.wada), axis2=(states.waes, states.wats),
    ...     get_conditions="2010-12-10")
    |       date | nied | teml | sunshineduration | possiblesunshineduration | globalradiation | relativehumidity | windspeed | atmosphericpressure |  qz | qzh | temltag | dailyrelativehumidity | dailysunshineduration | dailypossiblesunshineduration | dailyglobalradiation |   nkor | tkor | tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair |   g |        tz |         wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance |   nbes |    sbes |      evpo |  evi |      evb |       evs |     wnied | tempssurface | actualalbedo |   schmpot |     schm |  gefrpot |     gefr |   wlatsnow |  wsenssnow |       wsurf |      sff |      fvg |      wada |       qdb | qib1 | qib2 | qbb | qkap |      qdgz |      qah |       qa | inzp |      wats |      waes |     esnow |     taus |       ebdn |      bowa |     qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2010-01-12 |  0.0 | -5.8 |              3.5 |                 7.641029 |       31.942798 |             70.3 |       6.9 |              1015.0 | 0.0 | 0.0 |    -5.8 |                  70.3 |                   3.5 |                      7.641029 |            31.942798 |    0.0 | -5.8 |    -5.8 |     5.53098 |          5.53098 |            5.53098 |          6.9 |                 3.958114 |                      3.958114 |                          0.0 |                      0.303557 |                           0.303557 |             2.782554 |                  2.782554 |    1012.217446 |   1.321192 | 8.6 |       0.0 |       34.8 |             28.748518 |                 28.748518 |                  28.748518 |                 46.472798 |                      0.0 |    -17.72428 |              0.0 |         -17.72428 |             30.870386 |                 100.0 |               111.309423 |               99.512695 |    0.0 |     0.0 |  1.808353 |  0.0 | 0.553793 |       0.0 |       0.0 |          nan |          0.1 |       0.0 |      0.0 |      0.0 |      0.0 |        0.0 |        0.0 |         0.0 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 |  0.0 |       0.0 |       0.0 |       0.0 |      nan |      -26.2 | 71.446207 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-02-12 | 11.1 | -3.0 |              0.0 |                 7.609644 |        9.873898 |             85.0 |       5.2 |              1007.0 | 0.0 | 0.0 |    -3.0 |                  85.0 |                   0.0 |                      7.609644 |             9.873898 | 10.434 | -3.0 |    -3.0 |    4.168274 |         4.168274 |           4.168274 |          5.2 |                 4.893489 |                      4.893489 |                     3.874827 |                      0.366257 |                           0.366257 |             4.159465 |                  4.159465 |    1002.840535 |   1.296511 | 8.6 |   -7.5456 |  -4.524364 |              1.229506 |                  1.229506 |                   1.229506 |                  14.78976 |                28.674485 |   -13.560254 |       -27.444979 |        -13.560254 |             40.962628 |                 100.0 |               112.102221 |               99.530053 | 10.374 |  10.374 |  0.780099 |  0.0 |      0.0 | -0.140671 | -0.752835 |     -6.07696 |     0.875479 |       0.0 |      0.0 | 0.440874 |      0.0 |  -4.342973 | -26.674903 |   -3.572896 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 | 10.514671 | 10.514671 | -1.704304 | 0.084415 | -13.075636 | 71.446207 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-03-12 |  0.4 | -5.6 |              0.1 |                 7.579824 |         16.6835 |             95.9 |       2.0 |              1005.0 | 0.0 | 0.0 |    -5.6 |                  95.9 |                   0.1 |                      7.579824 |              16.6835 |  0.376 | -5.6 |    -5.6 |    1.603182 |         1.603182 |           1.603182 |          2.0 |                 4.019239 |                      4.019239 |                     3.243675 |                      0.307706 |                           0.307706 |              3.85445 |                   3.85445 |     1001.14555 |   1.306656 | 8.6 | -3.765783 |  12.660208 |              2.311282 |                  2.311282 |                   2.311282 |                 15.387201 |                32.300381 |   -13.075919 |       -29.989098 |        -13.075919 |            106.502833 |                 100.0 |               112.102221 |               99.527228 |  0.376 |   0.376 |   0.03962 |  0.0 |      0.0 |  -0.15895 | -0.050934 |    -8.366401 |     0.861463 |       0.0 |      0.0 | 0.400426 |      0.0 |   -4.90731 | -12.628873 |   12.452915 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 | 11.049621 | 11.049621 | -1.547945 | 0.139393 | -17.135843 | 71.446207 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-04-12 |  1.2 | -3.8 |              3.7 |                 7.551596 |       31.807427 |             90.6 |       4.3 |              1008.0 | 0.0 | 0.0 |    -3.8 |                  90.6 |                   3.7 |                      7.551596 |            31.807427 |  1.128 | -3.8 |    -3.8 |    3.446842 |         3.446842 |           3.446842 |          4.3 |                 4.608124 |                      4.608124 |                     3.313259 |                      0.347298 |                           0.347298 |              4.17496 |                   4.17496 |     1003.82504 |   1.301648 | 8.6 | -4.935123 |   8.499536 |              4.759177 |                  4.759177 |                   4.759177 |                  43.14024 |                57.155746 |   -38.381062 |       -52.396569 |        -38.381062 |             49.536202 |                 100.0 |               112.102221 |               99.524553 |  1.128 |   1.128 |  0.092572 |  0.0 |      0.0 | -0.369159 | -0.103687 |    -8.095503 |     0.850375 |       0.0 |      0.0 | 0.484009 |      0.0 | -11.397129 | -32.280484 |    8.718956 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 |  12.54678 |  12.54678 | -1.871052 | 0.186998 |  -17.03538 | 71.446207 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-05-12 |  6.4 |  0.2 |              0.0 |                 7.524989 |        9.532511 |             96.5 |       4.0 |               996.0 | 0.0 | 0.0 |     0.2 |                  96.5 |                   0.0 |                      7.524989 |             9.532511 |  6.016 |  0.2 |     0.2 |    3.206365 |         3.206365 |           3.206365 |          4.0 |                 6.197477 |                      6.197477 |                     5.232243 |                      0.451275 |                           0.451275 |             5.980565 |                  5.980565 |     990.019435 |   1.266438 | 8.6 | -4.906189 |  -7.697363 |              1.485911 |                  1.485911 |                   1.485911 |                 13.205056 |                19.686689 |   -11.719146 |       -18.200779 |        -11.719146 |             53.251417 |                 100.0 |               112.102221 |               99.522033 |  6.016 |  2.7072 |  0.135315 |  0.0 |      0.0 | -0.304172 |  0.045165 |    -2.102229 |     0.844122 |       0.0 |      0.0 | 0.496132 | 0.496132 |  -9.390793 | -16.415317 |   -7.605332 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 | 16.008195 | 18.866952 |       0.0 | 0.215645 |  -0.738016 | 71.446207 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-06-12 |  0.0 | -1.1 |              0.1 |                 7.500028 |       16.141484 |             99.0 |       2.4 |               998.0 | 0.0 | 0.0 |    -1.1 |                  99.0 |                   0.1 |                      7.500028 |            16.141484 |    0.0 | -1.1 |    -1.1 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                 5.634754 |                      5.634754 |                     4.935069 |                       0.41489 |                            0.41489 |             5.578406 |                  5.578406 |     992.421594 |   1.275245 | 8.6 | -0.212549 |    6.90279 |               3.04348 |                   3.04348 |                    3.04348 |                 14.093918 |                25.486763 |   -11.050438 |       -22.443283 |        -11.050438 |             88.752361 |                 100.0 |               112.102221 |               99.519668 |    0.0 |     0.0 |  0.001149 |  0.0 |      0.0 | -0.186239 |       0.0 |    -2.886897 |      0.81145 |       0.0 |      0.0 | 0.185379 | 0.185379 |  -5.749809 |  -9.074059 |    7.619416 | 0.996898 | 0.298142 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 | 16.353424 | 19.053191 |       0.0 | 0.391037 |   0.959194 | 71.446207 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-07-12 |  0.0 | -2.9 |              0.0 |                 7.476739 |        9.341429 |             99.2 |       1.9 |              1004.0 | 0.0 | 0.0 |    -2.9 |                  99.2 |                   0.0 |                      7.476739 |             9.341429 |    0.0 | -2.9 |    -2.9 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 4.930236 |                      4.930236 |                      4.29691 |                      0.368688 |                           0.368688 |             4.890794 |                  4.890794 |     999.109206 |   1.291808 | 8.6 |       0.0 |   12.29543 |              1.998184 |                  1.998184 |                   1.998184 |                 13.815559 |                28.103018 |   -11.817375 |       -26.104834 |        -11.817375 |            112.108246 |                 100.0 |               112.102221 |               99.517462 |    0.0 |     0.0 |  -0.01559 |  0.0 |      0.0 | -0.150212 |       0.0 |    -4.724073 |     0.786094 |       0.0 |      0.0 | 0.279053 | 0.279053 |  -4.637544 |  -8.093116 |   13.374174 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 | 16.763604 | 19.203403 |       0.0 | 0.566429 |  -2.736236 | 71.446207 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-08-12 |  0.0 | -2.9 |              0.0 |                 7.455145 |        9.256764 |             99.1 |       1.8 |              1006.0 | 0.0 | 0.0 |    -2.9 |                  99.1 |                   0.0 |                      7.455145 |             9.256764 |    0.0 | -2.9 |    -2.9 |    1.442864 |         1.442864 |           1.442864 |          1.8 |                 4.930236 |                      4.930236 |                     4.242856 |                      0.368688 |                           0.368688 |             4.885864 |                  4.885864 |    1001.114136 |   1.294388 | 8.6 | -0.788036 |  10.465352 |              2.167515 |                  2.167515 |                   2.167515 |                 13.821847 |                 27.41168 |   -11.654332 |       -25.244165 |        -11.654332 |            118.336481 |                 100.0 |               112.102221 |               99.515416 |    0.0 |     0.0 | -0.012353 |  0.0 |      0.0 | -0.157936 |       0.0 |     -4.89058 |     0.765845 |       0.0 |      0.0 | 0.343075 | 0.343075 |  -4.875997 |  -8.576577 |   11.791592 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 | 17.247371 | 19.361339 |       0.0 | 0.741821 |  -4.601588 | 71.446207 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-09-12 |  0.7 | -2.1 |              4.4 |                 7.435268 |       33.405815 |             96.9 |       2.7 |              1009.0 | 0.0 | 0.0 |    -2.1 |                  96.9 |                   4.4 |                      7.435268 |            33.405815 |  0.658 | -2.1 |    -2.1 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 5.233109 |                      5.233109 |                     3.812675 |                      0.388644 |                           0.388644 |             5.070883 |                  5.070883 |    1003.929117 |   1.294334 | 8.6 | -1.325257 |   12.55224 |              8.219632 |                  8.219632 |                   8.219632 |                 46.364476 |                57.539469 |   -38.144844 |       -49.319838 |        -38.144844 |             78.890988 |                 100.0 |               112.102221 |               99.513534 |  0.658 |   0.658 | -0.234427 |  0.0 |      0.0 | -0.391835 | -0.033425 |    -6.287098 |     0.753946 |       0.0 |      0.0 | 0.473439 | 0.473439 |  -12.09724 | -22.873591 |   14.349006 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 | 18.738535 | 20.411175 |       0.0 |   0.8645 |  -8.553827 | 71.446207 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-10-12 |  3.3 | -3.1 |              0.1 |                  7.41713 |       15.591158 |             95.0 |       3.0 |              1018.0 | 0.0 | 0.0 |    -3.1 |                  95.0 |                   0.1 |                       7.41713 |            15.591158 |  3.102 | -3.1 |    -3.1 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 4.856984 |                      4.856984 |                     4.122087 |                       0.36384 |                            0.36384 |             4.614135 |                  4.614135 |    1013.385865 |   1.310959 | 8.6 | -2.463502 |   6.595367 |              3.663944 |                  3.663944 |                   3.663944 |                 14.892721 |                29.948168 |   -11.228777 |       -26.284224 |        -11.228777 |             71.001889 |                 100.0 |               112.102221 |               99.511816 |  3.102 |   3.102 |  0.137524 |  0.0 |      0.0 | -0.164028 | -0.232614 |    -5.269486 |     0.764999 |       0.0 |      0.0 | 0.561607 | 0.561607 |  -5.064075 | -12.686369 |     8.53378 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |      0.0 |      0.0 | 0.06 |  22.55842 | 23.677203 |       0.0 | 0.750014 |  -6.549194 | 71.446207 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |      0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 2010-11-12 |  9.4 |  5.1 |              0.2 |                 7.400749 |       15.891544 |             90.6 |       7.4 |              1004.0 | 0.0 | 0.0 |     5.1 |                  90.6 |                   0.2 |                      7.400749 |            15.891544 |  8.836 |  5.1 |     5.1 |    5.931775 |         5.931775 |           5.931775 |          7.4 |                 8.790172 |                      8.790172 |                       6.1078 |                      0.614118 |                           0.614118 |             7.963896 |                  7.963896 |     996.036104 |   1.253213 | 8.6 | -1.886168 | -11.317008 |              4.012368 |                  4.012368 |                   4.012368 |                 13.525343 |                -0.754591 |    -9.512976 |         4.766959 |         -9.512976 |              28.78455 |                 100.0 |               112.102221 |               99.510264 |  8.836 |     0.0 |  1.033512 | 0.06 | 0.339079 | -1.215856 |   2.12077 |          0.0 |     0.747516 | 23.724162 | 22.55842 |      0.0 |      0.0 | -37.537452 | -58.603285 | -100.907696 | 0.956775 | 0.274625 | 33.729059 | 11.067249 |  0.0 |  0.0 | 0.0 |  0.0 | 11.067249 | 2.849225 | 0.032977 |  0.0 |       0.0 |       0.0 |       0.0 | 0.938276 |  13.367813 | 93.768939 | 11.067249 |   0.0 |   0.0 |   0.0 |  0.0 | 2.849225 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.032977 |
    | 2010-12-12 |  0.2 |  0.2 |              0.7 |                 7.386143 |       17.823519 |             78.7 |       5.7 |              1012.0 | 0.0 | 0.0 |     0.2 |                  78.7 |                   0.7 |                      7.386143 |            17.823519 |  0.188 |  0.2 |     0.2 |     4.56907 |          4.56907 |            4.56907 |          5.7 |                 6.197477 |                      6.197477 |                     5.049903 |                      0.451275 |                           0.451275 |             4.877414 |                  4.877414 |    1007.122586 |    1.28736 | 8.6 |       0.0 |   7.031305 |              4.735182 |                  4.735182 |                   4.735182 |                 20.069431 |                 34.81614 |   -15.334248 |       -30.080958 |        -15.334248 |             37.369415 |                 100.0 |                81.654529 |               98.490783 |  0.128 |  0.0576 |  1.429818 | 0.06 | 0.519712 |  0.082368 |  0.000519 |    -2.578857 |      0.73433 |       0.0 |      0.0 | 0.000786 | 0.000786 |   2.826483 | -25.872578 |    7.034863 | 0.951702 | 0.271721 |  0.045632 |  0.015061 |  0.0 |  0.0 | 0.0 |  0.0 |  0.015061 | 3.836673 | 0.044406 |  0.0 |       0.0 |       0.0 |       0.0 | 1.109428 |  14.936509 | 93.279798 |  0.015061 |   0.0 |   0.0 |   0.0 |  0.0 | 3.836673 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.044406 |
    | 2010-13-12 |  2.9 | -3.5 |              0.0 |                 7.373328 |        8.941324 |             90.5 |       2.2 |              1020.0 | 0.0 | 0.0 |    -3.5 |                  90.5 |                   0.0 |                      7.373328 |             8.941324 |  2.726 | -3.5 |    -3.5 |    1.763501 |         1.763501 |           1.763501 |          2.2 |                 4.713362 |                      4.713362 |                     3.977478 |                      0.354306 |                           0.354306 |             4.265593 |                  4.265593 |    1015.734407 |   1.315658 | 8.6 |       0.0 |  15.675795 |              2.222322 |                  2.222322 |                   2.222322 |                 14.516908 |                31.068606 |   -12.294587 |       -28.846284 |        -12.294587 |             96.820757 |                 100.0 |                82.297017 |               98.520265 |  2.666 |   2.666 |  0.193908 |  0.0 |      0.0 | -0.079193 | -0.225715 |    -5.736346 |     0.751455 |       0.0 |      0.0 | 0.043585 |      0.0 |  -2.444945 |  -10.78277 |   15.618569 | 0.974582 | 0.284943 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 2.050438 | 0.023732 | 0.06 |  2.745193 |  2.745193 | -0.168489 | 0.892404 |   7.860714 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 2.050438 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.023732 |
    | 2010-14-12 |  0.0 | -3.9 |              1.2 |                 7.362317 |       19.668068 |             88.1 |       3.1 |              1025.0 | 0.0 | 0.0 |    -3.9 |                  88.1 |                   1.2 |                      7.362317 |            19.668068 |    0.0 | -3.9 |    -3.9 |    2.484933 |         2.484933 |           2.484933 |          3.1 |                  4.57351 |                       4.57351 |                      3.68122 |                      0.344989 |                           0.344989 |             4.029262 |                  4.029262 |    1020.970738 |   1.324197 | 8.6 |       0.0 |  18.367586 |              5.038442 |                  5.038442 |                   5.038442 |                 24.390008 |                44.083324 |   -19.351566 |       -39.044881 |        -19.351566 |             68.711505 |                 100.0 |                82.297017 |               98.520703 |    0.0 |     0.0 |  0.280183 |  0.0 |      0.0 | -0.118567 |       0.0 |    -6.741741 |     0.743826 |       0.0 |      0.0 | 0.052586 |      0.0 |  -3.660562 | -16.981937 |   18.402382 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 1.094136 | 0.012664 | 0.06 |   2.86376 |   2.86376 | -0.203285 | 0.983299 |  -1.906872 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 1.094136 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.012664 |
    | 2010-15-12 |  0.0 | -6.3 |              3.0 |                 7.353123 |       26.831581 |             92.8 |       2.8 |              1023.0 | 0.0 | 0.0 |    -6.3 |                  92.8 |                   3.0 |                      7.353123 |            26.831581 |    0.0 | -6.3 |    -6.3 |    2.244455 |         2.244455 |           2.244455 |          2.8 |                 3.808888 |                      3.808888 |                     2.883853 |                      0.293396 |                           0.293396 |             3.534648 |                  3.534648 |    1019.465352 |    1.33374 | 8.6 | -0.549179 |  25.311295 |              7.049178 |                  7.049178 |                   7.049178 |                 39.328283 |                58.770158 |   -32.279104 |        -51.72098 |        -32.279104 |             76.073452 |                 100.0 |                82.297017 |               98.521069 |    0.0 |     0.0 | -0.031641 |  0.0 |      0.0 | -0.207431 |       0.0 |    -9.855386 |     0.737281 |       0.0 |      0.0 | 0.085437 |      0.0 |   -6.40407 | -19.878621 |   25.438289 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.583843 | 0.006757 | 0.06 |  3.071191 |  3.071191 | -0.330279 |  1.06855 | -18.618167 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.583843 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.006757 |
    | 2010-16-12 |  6.7 | -3.5 |              0.0 |                 7.345755 |        8.837738 |             98.1 |       3.7 |              1000.0 | 0.0 | 0.0 |    -3.5 |                  98.1 |                   0.0 |                      7.345755 |             8.837738 |  6.298 | -3.5 |    -3.5 |    2.965888 |         2.965888 |           2.965888 |          3.7 |                 4.713362 |                      4.713362 |                     3.936783 |                      0.354306 |                           0.354306 |             4.623808 |                  4.623808 |     995.376192 |   1.289645 | 8.6 | -5.362032 |   1.170579 |              1.775591 |                  1.775591 |                   1.775591 |                   14.0252 |                27.488061 |   -12.249609 |       -25.712469 |        -12.249609 |             57.569099 |                 100.0 |                82.297017 |               98.521361 |  6.298 |   6.298 |  0.034134 |  0.0 |      0.0 | -0.264187 | -0.533216 |    -5.870438 |      0.79909 |       0.0 |      0.0 | 0.325801 |      0.0 |  -8.156322 | -15.989601 |    1.566547 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.311545 | 0.003606 | 0.06 |  9.633378 |  9.633378 | -1.259462 | 0.471345 | -11.188747 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.311545 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003606 |
    | 2010-17-12 |  0.0 | -7.3 |              0.0 |                 7.340222 |        8.817416 |             96.6 |       0.9 |               991.0 | 0.0 | 0.0 |    -7.3 |                  96.6 |                   0.0 |                      7.340222 |             8.817416 |    0.0 | -7.3 |    -7.3 |    0.721432 |         0.721432 |           0.721432 |          0.9 |                  3.52531 |                       3.52531 |                     2.800277 |                      0.273951 |                           0.273951 |             3.405449 |                  3.405449 |     987.594551 |   1.296889 | 8.6 | -3.222359 |  18.972885 |              1.850724 |                  1.850724 |                   1.850724 |                 14.906398 |                33.637078 |   -13.055673 |       -31.786354 |        -13.055673 |            236.672963 |                 100.0 |                82.297017 |               98.521581 |    0.0 |     0.0 | -0.001874 |  0.0 |      0.0 | -0.108821 |       0.0 |   -10.224747 |     0.790106 |       0.0 |      0.0 | 0.384863 |      0.0 |  -3.359648 |  -9.225504 |   19.201202 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.166244 | 0.001924 | 0.06 |  9.742199 |  9.742199 | -1.487779 | 0.535793 | -21.561632 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.166244 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001924 |
    | 2010-18-12 |  0.1 | -5.8 |              0.0 |                  7.33653 |        8.804185 |             97.7 |       1.3 |               997.0 | 0.0 | 0.0 |    -5.8 |                  97.7 |                   0.0 |                       7.33653 |             8.804185 |  0.094 | -5.8 |    -5.8 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                 3.958114 |                      3.958114 |                     3.033407 |                      0.303557 |                           0.303557 |             3.867077 |                  3.867077 |     993.132923 |   1.297204 | 8.6 |  -6.20975 |   8.041899 |              1.910103 |                  1.910103 |                   1.910103 |                 14.549007 |                28.161536 |   -12.638904 |       -26.251433 |        -12.638904 |            163.850513 |                 100.0 |                82.297017 |               98.521728 |  0.094 |   0.094 | -0.003431 |  0.0 |      0.0 |  -0.17429 | -0.013188 |    -9.217539 |     0.783046 |       0.0 |      0.0 | 0.464706 |      0.0 |  -5.380896 | -12.533174 |    8.337363 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 |  0.08871 | 0.001027 | 0.06 | 10.010489 | 10.010489 | -1.796431 | 0.590539 | -21.003531 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.08871 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001027 |
    | 2010-19-12 |  0.0 | -7.9 |              0.0 |                 7.334683 |        8.798044 |             95.8 |       1.9 |               997.0 | 0.0 | 0.0 |    -7.9 |                  95.8 |                   0.0 |                      7.334683 |             8.798044 |    0.0 | -7.9 |    -7.9 |    1.523023 |         1.523023 |           1.523023 |          1.9 |                 3.364296 |                      3.364296 |                     2.624919 |                      0.262827 |                           0.262827 |             3.222996 |                  3.222996 |     993.777004 |   1.307793 | 8.6 | -6.049017 |  13.476044 |               1.96695 |                   1.96695 |                    1.96695 |                 15.062045 |                34.224679 |   -13.095096 |        -32.25773 |        -13.095096 |            112.108246 |                 100.0 |                82.297017 |               98.521802 |    0.0 |     0.0 |  0.032042 |  0.0 |      0.0 | -0.151273 |       0.0 |   -11.032678 |     0.776433 |       0.0 |      0.0 | 0.519604 |      0.0 |   -4.67028 | -13.899182 |   13.688268 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.047336 | 0.000548 | 0.06 | 10.161762 | 10.161762 | -2.008656 | 0.645482 | -25.879575 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.047336 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000548 |
    | 2010-20-12 |  0.2 | -5.7 |              0.2 |                 7.334683 |       15.464746 |             97.5 |       2.4 |               997.0 | 0.0 | 0.0 |    -5.7 |                  97.5 |                   0.2 |                      7.334683 |            15.464746 |  0.188 | -5.7 |    -5.7 |    1.923819 |         1.923819 |           1.923819 |          2.4 |                 3.988573 |                      3.988573 |                     3.099124 |                      0.305625 |                           0.305625 |             3.888858 |                  3.888858 |     993.111142 |   1.296708 | 8.6 | -7.453318 |   4.070387 |              3.527764 |                  3.527764 |                   3.527764 |                  16.12634 |                31.141438 |   -12.598576 |       -27.613674 |        -12.598576 |             88.752361 |                 100.0 |                82.297017 |               98.521802 |  0.188 |   0.188 |  0.020022 |  0.0 |      0.0 |  -0.22862 | -0.025922 |    -8.946047 |     0.771883 |       0.0 |      0.0 | 0.526638 |      0.0 |  -7.058235 | -16.483783 |    4.071656 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.025259 | 0.000292 | 0.06 | 10.578381 | 10.578381 | -2.035847 | 0.685542 | -21.349962 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.025259 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000292 |
    | 2010-21-12 |  0.2 | -5.6 |              0.8 |                  7.33653 |       17.879541 |             95.0 |       2.3 |              1004.0 | 0.0 | 0.0 |    -5.6 |                  95.0 |                   0.8 |                       7.33653 |            17.879541 |  0.188 | -5.6 |    -5.6 |     1.84366 |          1.84366 |            1.84366 |          2.3 |                 4.019239 |                      4.019239 |                     3.035743 |                      0.307706 |                           0.307706 |             3.818277 |                  3.818277 |    1000.181723 |   1.305372 | 8.6 | -6.148789 |   8.394877 |              4.156278 |                  4.156278 |                   4.156278 |                 21.069426 |                37.130933 |   -16.913148 |       -32.974655 |        -16.913148 |             92.611159 |                 100.0 |                82.297017 |               98.521728 |  0.188 |   0.188 |  0.034008 |  0.0 |      0.0 | -0.220814 | -0.025467 |      -9.2078 |      0.76754 |       0.0 |      0.0 | 0.508508 |      0.0 |  -6.817243 | -17.858089 |    8.299323 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.013479 | 0.000156 | 0.06 | 10.987195 | 10.987195 |  -1.96576 | 0.725649 | -21.144839 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.013479 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000156 |
    | 2010-22-12 |  2.4 | -4.8 |              0.0 |                 7.340222 |        8.822161 |             93.8 |       4.6 |              1007.0 | 0.0 | 0.0 |    -4.8 |                  93.8 |                   0.0 |                      7.340222 |             8.822161 |  2.256 | -4.8 |    -4.8 |     3.68732 |          3.68732 |            3.68732 |          4.6 |                 4.272186 |                      4.272186 |                     3.529214 |                      0.324793 |                           0.324793 |             4.007311 |                  4.007311 |    1002.992689 |   1.305282 | 8.6 | -6.089714 |   3.316366 |              1.944525 |                  1.944525 |                   1.944525 |                 14.580265 |                31.445051 |    -12.63574 |       -29.500526 |         -12.63574 |              46.30558 |                 100.0 |                82.297017 |               98.521581 |  2.256 |   2.256 |  0.237899 |  0.0 |      0.0 | -0.215306 | -0.261947 |    -7.285755 |     0.779586 |       0.0 |      0.0 | 0.550453 |      0.0 |   -6.64721 | -19.636748 |    3.216569 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.007192 | 0.000083 | 0.06 | 13.458501 | 13.458501 | -2.127909 | 0.618819 | -15.861205 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.007192 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000083 |
    | 2010-23-12 |  7.0 | -1.1 |              0.0 |                 7.345755 |        8.844396 |             94.9 |       8.6 |              1004.0 | 0.0 | 0.0 |    -1.1 |                  94.9 |                   0.0 |                      7.345755 |             8.844396 |   6.58 | -1.1 |    -1.1 |    6.893685 |         6.893685 |           6.893685 |          8.6 |                 5.634754 |                      5.634754 |                     4.983961 |                       0.41489 |                            0.41489 |             5.347381 |                  5.347381 |     998.652619 |    1.28304 | 8.6 | -4.568027 |  -4.739886 |              1.641983 |                  1.641983 |                   1.641983 |                 13.655186 |                26.425029 |   -12.013203 |       -24.783046 |        -12.013203 |             24.768101 |                 100.0 |                82.297017 |               98.521361 |   6.58 |  5.0995 |  0.461551 |  0.0 |      0.0 | -0.269948 | -0.214608 |    -2.754979 |     0.814348 |       0.0 |      0.0 | 0.508846 | 0.508846 |  -8.334182 |   -21.5642 |   -5.115337 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.003838 | 0.000044 | 0.06 | 19.323706 |  20.30845 |       0.0 | 0.373461 |  -2.521319 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.003838 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000044 |
    | 2010-24-12 |  0.2 | -2.9 |              0.0 |                 7.353123 |        8.873743 |             85.4 |       7.6 |              1008.0 | 0.0 | 0.0 |    -2.9 |                  85.4 |                   0.0 |                      7.353123 |             8.873743 |  0.188 | -2.9 |    -2.9 |    6.092093 |         6.092093 |           6.092093 |          7.6 |                 4.930236 |                      4.930236 |                     4.287255 |                      0.368688 |                           0.368688 |             4.210422 |                  4.210422 |    1003.789578 |   1.297296 | 8.6 |  -0.72614 |  10.298965 |              1.867788 |                  1.867788 |                   1.867788 |                 14.739672 |                 33.61027 |   -12.871884 |       -31.742482 |        -12.871884 |             28.027061 |                 100.0 |                82.297017 |               98.521069 |  0.188 |   0.188 |  1.132418 |  0.0 |      0.0 |  0.051454 | -0.013188 |    -4.753678 |     0.789515 |       0.0 |      0.0 | 0.328384 | 0.328384 |   1.588548 | -21.775808 |   11.555222 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.002048 | 0.000024 | 0.06 | 19.790283 | 20.444996 |       0.0 | 0.540228 |  -4.220284 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.002048 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000024 |
    | 2010-25-12 |  0.0 | -9.1 |              3.2 |                 7.362317 |       27.759642 |             87.2 |       3.8 |              1018.0 | 0.0 | 0.0 |    -9.1 |                  87.2 |                   3.2 |                      7.362317 |            27.759642 |    0.0 | -9.1 |    -9.1 |    3.046047 |         3.046047 |           3.046047 |          3.8 |                 3.061705 |                      3.061705 |                     2.334322 |                      0.241745 |                           0.241745 |             2.669806 |                  2.669806 |    1015.330194 |   1.341718 | 8.6 | -1.215442 |  29.045464 |              6.423384 |                  6.423384 |                   6.423384 |                 43.173862 |                65.877607 |   -36.750477 |       -59.454222 |        -36.750477 |             56.054123 |                 100.0 |                82.297017 |               98.520703 |    0.0 |     0.0 |  0.145858 |  0.0 |      0.0 |  -0.13146 |       0.0 |   -12.483738 |     0.768607 |       0.0 |      0.0 | 0.799708 | 0.654712 |  -4.058584 | -23.258712 |   32.136926 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.001093 | 0.000013 | 0.06 | 20.576455 | 20.576455 | -0.560514 |  0.71562 | -24.665748 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.001093 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000013 |
    | 2010-26-12 |  1.2 | -8.6 |              4.3 |                 7.373328 |       32.328755 |             92.7 |       1.5 |              1019.0 | 0.0 | 0.0 |    -8.6 |                  92.7 |                   4.3 |                      7.373328 |            32.328755 |  1.128 | -8.6 |    -8.6 |    1.202387 |         1.202387 |           1.202387 |          1.5 |                 3.184716 |                      3.184716 |                     1.979308 |                      0.250344 |                           0.250344 |             2.952232 |                  2.952232 |    1016.047768 |   1.340358 | 8.6 | -7.103735 |  17.705249 |              7.586074 |                  7.586074 |                   7.586074 |                 51.194924 |                59.614784 |    -43.60885 |        -52.02871 |         -43.60885 |            142.003778 |                 100.0 |                82.297017 |               98.520265 |  1.128 |   1.128 | -0.167341 |  0.0 |      0.0 |  -0.21763 | -0.234661 |   -14.492413 |     0.765346 |       0.0 |      0.0 |  1.36557 |      0.0 |  -6.718935 | -23.120761 |   22.189013 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.000583 | 0.000007 | 0.06 | 21.922085 | 21.922085 | -5.278939 | 0.746643 | -33.770997 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000583 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000007 |
    | 2010-27-12 |  0.9 | -5.3 |              0.0 |                 7.386143 |        9.004582 |             97.8 |       1.3 |              1017.0 | 0.0 | 0.0 |    -5.3 |                  97.8 |                   0.0 |                      7.386143 |             9.004582 |  0.846 | -5.3 |    -5.3 |    1.042069 |         1.042069 |           1.042069 |          1.3 |                 4.112496 |                      4.112496 |                     2.963655 |                       0.31402 |                            0.31402 |             4.022021 |                  4.022021 |    1012.977979 |   1.320718 | 8.6 | -9.726047 |   -0.54134 |              2.097285 |                  2.097285 |                   2.097285 |                 14.439522 |                23.751486 |   -12.342237 |       -21.654201 |        -12.342237 |            163.850513 |                 100.0 |                82.297017 |               98.519756 |  0.846 |   0.846 |  -0.00182 |  0.0 |      0.0 | -0.221265 | -0.108462 |    -9.511473 |     0.767087 |       0.0 |      0.0 | 1.372825 |      0.0 |  -6.831183 | -15.444774 |   -0.621756 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.000311 | 0.000004 | 0.06 |  22.98935 |  22.98935 | -5.306985 | 0.729943 | -24.629657 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000311 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000004 |
    | 2010-28-12 |  0.0 | -7.7 |              3.4 |                 7.400749 |       28.982321 |             95.4 |       3.0 |              1022.0 | 0.0 | 0.0 |    -7.7 |                  95.4 |                   3.4 |                      7.400749 |            28.982321 |    0.0 | -7.7 |    -7.7 |    2.404774 |         2.404774 |           2.404774 |          3.0 |                 3.417227 |                      3.417227 |                     2.358788 |                      0.266491 |                           0.266491 |             3.260035 |                  3.260035 |    1018.739965 |   1.339598 | 8.6 | -7.093341 |  14.368183 |              6.890648 |                  6.890648 |                   6.890648 |                 42.703706 |                57.727703 |   -35.813057 |       -50.837054 |        -35.813057 |             71.001889 |                 100.0 |                82.297017 |               98.519175 |    0.0 |     0.0 | -0.120342 |  0.0 |      0.0 | -0.300438 |       0.0 |   -12.355553 |     0.762246 |       0.0 |      0.0 | 1.364907 |      0.0 |  -9.275493 | -27.223988 |   14.337573 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.000166 | 0.000002 | 0.06 | 23.289788 | 23.289788 | -5.276375 | 0.777186 |  -30.39784 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000166 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000002 |
    | 2010-29-12 |  0.0 | -9.2 |              0.0 |                  7.41713 |        9.127645 |             94.9 |       2.7 |              1024.0 | 0.0 | 0.0 |    -9.2 |                  94.9 |                   0.0 |                       7.41713 |             9.127645 |    0.0 | -9.2 |    -9.2 |    2.164296 |         2.164296 |           2.164296 |          2.7 |                 3.037615 |                      3.037615 |                     2.327708 |                      0.240056 |                           0.240056 |             2.882696 |                  2.882696 |    1021.117304 |   1.350039 | 8.6 | -8.754578 |   9.943979 |              2.212531 |                  2.212531 |                   2.212531 |                 15.336383 |                36.211017 |   -13.123853 |       -33.998487 |        -13.123853 |             78.890988 |                 100.0 |                82.297017 |               98.518524 |    0.0 |     0.0 |  0.073538 |  0.0 |      0.0 | -0.172836 |       0.0 |   -12.518592 |     0.757601 |       0.0 |      0.0 | 1.517385 |      0.0 |  -5.336019 | -18.129048 |   10.533419 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.000089 | 0.000001 | 0.06 | 23.462624 | 23.462624 | -5.865815 | 0.825017 | -31.741819 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000089 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000001 |
    | 2010-30-12 |  0.2 | -7.4 |              0.0 |                 7.435268 |        9.199989 |             96.0 |       1.4 |              1021.0 | 0.0 | 0.0 |    -7.4 |                  96.0 |                   0.0 |                      7.435268 |             9.199989 |  0.188 | -7.4 |    -7.4 |    1.122228 |         1.122228 |           1.122228 |          1.4 |                 3.498009 |                      3.498009 |                     2.537041 |                      0.272069 |                           0.272069 |             3.358088 |                  3.358088 |    1017.641912 |   1.336727 | 8.6 | -9.141644 |   6.346345 |              2.255141 |                  2.255141 |                   2.255141 |                 14.959543 |                29.415788 |   -12.704402 |       -27.160647 |        -12.704402 |            152.146905 |                 100.0 |                82.297017 |               98.517803 |  0.188 |   0.188 |  0.019576 |  0.0 |      0.0 | -0.177654 | -0.033653 |   -11.455764 |     0.754876 |       0.0 |      0.0 | 1.509439 |      0.0 |  -5.484757 | -15.393912 |    6.281978 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.000047 | 0.000001 | 0.06 | 23.828278 | 23.828278 | -5.835101 | 0.854297 | -29.488164 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000047 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000001 |
    | 2010-31-12 |  0.9 |  1.7 |              4.0 |                 7.455145 |       32.062235 |             96.5 |       4.2 |              1016.0 | 0.0 | 0.0 |     1.7 |                  96.5 |                   4.0 |                      7.455145 |            32.062235 |  0.846 |  1.7 |     1.7 |    3.366683 |         3.366683 |           3.366683 |          4.2 |                 6.907942 |                      6.907942 |                     4.947106 |                      0.496631 |                           0.496631 |             6.666164 |                  6.666164 |    1009.333836 |   1.284547 | 8.6 | -8.492591 |  -14.28406 |              7.981016 |                  7.981016 |                   7.981016 |                 40.225917 |                47.683043 |   -32.244902 |       -39.702027 |        -32.244902 |             50.715635 |                 100.0 |                82.297017 |               98.517013 |  0.846 | 0.06345 | -0.124832 |  0.0 |      0.0 | -0.723887 |  0.067075 |    -2.854311 |     0.751077 |       0.0 |      0.0 | 0.973699 |  0.78255 | -22.348747 | -33.641298 |  -16.288018 |      1.0 |      0.3 |       0.0 |       0.0 |  0.0 |  0.0 | 0.0 |  0.0 |       0.0 | 0.000025 |      0.0 | 0.06 | 25.398165 | 25.398165 | -0.738933 | 0.896706 |  -6.604104 | 93.279798 |       0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.000025 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

>>> ktschnee(inf)

hourly simulation
_________________

We change the simulation time step size from one day to one hour. Consequently, we need
to change the values of all time-dependent control parameters by the factor 1/24 (or,
in single cases, 24).  Also, we need to |FixedParameter.restore| all time-dependent
fixed parameter values:

>>> pub.timegrids = "1997-08-03", "1997-08-06", "1h"
>>> for parameter in control:
...     if parameter.TIME == True:
...         parameter.value = parameter.value/24
...     if parameter.TIME == False:
...         parameter.value = parameter.value*24
>>> for parameter in fixed:
...     parameter.restore()
>>> test = IntegrationTest(land)

.. _lland_v3_acker_summer_hourly:

acre (summer)
-------------

The following input values agree with the ones of the respective sub-period
of the daily integration test :ref:`lland_v3_acker_summer_daily` (global radiation and
possible sunshine duration stem from :ref:`meteo_v003_hourly_simulation_summer`):

>>> inputs.nied.series = 0.0
>>> inputs.teml.series = (
...     16.9, 16.6, 16.4, 16.3, 16.0, 15.9, 16.0, 16.6, 17.4, 19.0, 20.3, 21.4, 21.3,
...     21.8, 22.9, 22.7, 22.5, 21.9, 21.4, 20.7, 19.4, 17.8, 17.0, 16.4, 15.9, 15.4,
...     15.2, 14.9, 14.3, 14.1, 14.3, 15.4, 16.8, 18.2, 19.5, 20.3, 21.0, 21.8, 21.9,
...     21.8, 22.2, 21.8, 20.9, 19.7, 17.9, 15.8, 14.8, 14.1, 13.4, 12.8, 12.3, 12.0,
...     11.7, 11.6, 12.6, 14.8, 17.1, 19.4, 20.6, 21.4, 22.5, 23.1, 23.4, 23.8, 24.0,
...     24.0, 23.7, 22.8, 21.3, 19.3, 18.3, 17.4)
>>> inputs.sunshineduration.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.5, 0.7, 0.8, 0.5, 0.4, 0.5,
...     0.5, 0.3, 0.1, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.1, 0.9,
...     1.0, 1.0, 0.9, 0.8, 0.9, 0.8, 0.9, 0.9, 0.9, 1.0, 1.0, 1.0, 0.3, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.6, 0.9, 1.0, 0.9, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
...     1.0, 1.0, 1.0, 1.0, 0.5, 0.0, 0.0, 0.0)
>>> inputs.windspeed.series = (
...     0.8, 0.8, 0.8, 0.8, 0.8, 0.6, 0.9, 0.9, 0.9, 1.3, 1.5, 1.2, 1.3, 1.5, 1.9, 1.9,
...     2.3, 2.4, 2.5, 2.5, 2.2, 1.7, 1.7, 2.3, 2.3, 2.2, 2.3, 2.0, 2.3, 2.5, 2.4, 2.5,
...     2.5, 2.9, 3.1, 3.3, 3.3, 3.2, 2.5, 2.9, 3.6, 4.2, 4.2, 3.6, 3.2, 2.2, 2.2, 2.1,
...     1.8, 1.8, 1.6, 1.2, 1.9, 1.3, 1.6, 1.9, 3.2, 4.0, 4.6, 4.1, 4.1, 4.4, 4.6, 4.5,
...     4.7, 4.5, 4.1, 3.8, 2.5, 1.9, 2.2, 2.7)
>>> inputs.relativehumidity.series = (
...     95.1, 94.9, 95.9, 96.7, 97.2, 97.5, 97.7, 97.4, 96.8, 86.1, 76.8, 71.8, 67.5,
...     66.1, 63.4, 62.4, 61.1, 62.1, 67.0, 74.5, 81.2, 86.9, 90.1, 90.9, 88.0, 89.2,
...     88.8, 89.7, 93.0, 93.6, 94.6, 93.3, 86.9, 78.9, 75.8, 73.4, 68.3, 61.3, 60.6,
...     58.8, 57.2, 57.2, 59.9, 66.1, 71.7, 80.6, 85.9, 89.3, 91.4, 93.1, 94.6, 95.6,
...     96.2, 95.7, 95.1, 96.3, 89.9, 79.2, 73.4, 67.3, 62.2, 59.6, 55.4, 52.1, 49.5,
...     48.1, 49.4, 57.0, 65.9, 73.0, 76.7, 81.8)
>>> inputs.atmosphericpressure.series = (
...     1015.0, 1015.0, 1015.0, 1015.0, 1015.0, 1015.0, 1015.0, 1015.0, 1016.0, 1016.0,
...     1016.0, 1016.0, 1016.0, 1016.0, 1016.0, 1016.0, 1016.0, 1016.0, 1016.0, 1016.0,
...     1016.0, 1016.0, 1017.0, 1017.0, 1017.0, 1016.0, 1016.0, 1016.0, 1016.0, 1017.0,
...     1017.0, 1017.0, 1017.0, 1017.0, 1017.0, 1018.0, 1018.0, 1018.0, 1018.0, 1017.0,
...     1017.0, 1017.0, 1017.0, 1017.0, 1017.0, 1018.0, 1018.0, 1018.0, 1018.0, 1018.0,
...     1018.0, 1018.0, 1018.0, 1018.0, 1018.0, 1019.0, 1019.0, 1019.0, 1019.0, 1019.0,
...     1019.0, 1019.0, 1019.0, 1019.0, 1018.0, 1018.0, 1018.0, 1018.0, 1018.0, 1018.0,
...     1018.0, 1018.0)
>>> inlet.sequences.sim.series = 0.0

>>> inputs.possiblesunshineduration.series = (
...     0.0, 0.0, 0.0, 0.0, 0.42973358, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
...     1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 0.13640025, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.39804385, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
...     1.0, 1.0, 0.10471052, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.36602248, 1.0, 1.0,
...     1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 0.07268915,
...     0.0, 0.0, 0.0)
>>> inputs.globalradiation.series = (
...     0.0, 0.0, 0.0, 0.0, 1.9436861, 21.9324417, 57.2561861, 109.3328444,
...     170.9491528, 311.762625, 501.5833, 615.0187278, 626.544325, 496.1334167,
...     419.5209944, 387.8873528, 278.4968722, 137.1386083, 51.0807139, 13.6328167,
...     0.1859444, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 1.5905167, 19.9509028,
...     62.985775, 285.7763889, 475.2252472, 614.0067194, 643.8065083, 600.2629806,
...     620.5141389, 599.4413861, 613.5512222, 517.1070472, 377.8366167, 252.2046389,
...     128.5767139, 42.0725861, 0.8128972, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     1.4823472, 21.1355667, 125.88345, 305.9549444, 511.104575, 629.5192389,
...     765.8165889, 767.1458417, 736.2638528, 768.0061833, 735.9293694, 620.3749528,
...     450.0952639, 273.6141417, 133.3681056, 40.3265028, 0.8671222, 0.0, 0.0, 0.0)

In this and the following "summer" examples, we use the initial conditions of August
3rd, calculated by the corresponding daily integration tests.  However, this is
possible for most state sequences, but not for the (now higher resolved) log sequences.
Hence, we need to define them manually:

>>> test.inits = (
...     (logs.loggedsunshineduration,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.2, 0.1, 0.2, 0.1, 0.2, 0.2, 0.3, 0.0,
...       0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0]),
...     (logs.loggedpossiblesunshineduration,
...      [0.0, 0.0, 0.0, 0.0, 0.5, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0, 1.0,
...       1.0, 1.0, 1.0, 1.0, 1.0, 0.2, 0.0, 0.0, 0.0]),
...     (logs.loggedglobalradiation,
...      [0.0, 0.0, 0.0, 0.0, 0.0, 27.8, 55.6, 138.9, 222.2, 305.6, 333.3, 388.9,
...       527.8, 444.4, 250.0, 222.2, 166.7, 111.1, 55.6, 27.8, 0.0, 0.0, 0.0, 0.0]),
...     (logs.loggedteml,
...      [13.2, 13.2, 13.1, 12.6, 12.7, 13.0, 13.5, 14.8, 16.2, 17.7, 18.8, 19.4, 20.4,
...       21.0, 21.5, 21.2, 20.4, 20.7, 20.2, 19.7, 19.0, 18.0, 17.5, 17.1]),
...     (logs.loggedrelativehumidity,
...      [95.1, 94.5, 94.8, 96.4, 96.6, 97.1, 97.1, 96.7, 92.2, 88.5, 81.1, 76.5, 75.1,
...       70.8, 68.9, 69.2, 75.0, 74.0, 77.4, 81.4, 85.3, 90.1, 92.3, 93.8]),
...     (logs.loggedwindspeed2m,
...      [0.8, 1.0, 1.2, 1.3, 0.9, 1.1, 1.3, 1.3, 1.9, 2.2, 1.8, 2.3, 2.4, 2.5, 2.4,
...       2.5, 2.1, 2.2, 1.7, 1.7, 1.3, 1.3, 0.7, 0.8]))

|ESnow| and |EBdn| are the only state sequences with values depending on the current
simulation step size.  According to their unit WT/m², we must multiply their conditions
saved for August 3rd and December 10th (referring to Wd/m²) by 24 (to adapt them to
Wh/m²).  We do so by trick to ensure we do not miss one of the relevant condition
dictionaries:

>>> for key, value in locals().copy().items():
...     if key.startswith("conditions_") and "states" in value:
...         value["states"]["esnow"] *= 24
...         value["states"]["ebdn"] *= 24

This integration test deals with a dry situation.  Hence, the soil water content
(|BoWa|) shows a pronounced decline.  This decline is sharpest around noon when
evapotranspiration from the soil (|EvB|) reaches its maximum:

.. integration-test::

    >>> test("lland_v3_acker_summer_hourly",
    ...      axis1=(fluxes.evb, fluxes.qah), axis2=states.bowa,
    ...      use_conditions=conditions_acker_summer)
    |                date | nied | teml | sunshineduration | possiblesunshineduration | globalradiation | relativehumidity | windspeed | atmosphericpressure |  qz | qzh |   temltag | dailyrelativehumidity | dailysunshineduration | dailypossiblesunshineduration | dailyglobalradiation | nkor | tkor |   tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair |    g |        tz |         wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |      evpo |       evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |      qah |       qa |     inzp | wats | waes | esnow | taus |        ebdn |      bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-08-03 00:00:00 |  0.0 | 16.9 |              0.0 |                      0.0 |             0.0 |             95.1 |       0.8 |              1015.0 | 0.0 | 0.0 | 17.279167 |             85.883333 |                   1.3 |                          15.7 |           136.579167 |  0.0 | 16.9 | 17.279167 |    0.641273 |         1.605886 |           0.641273 |          0.8 |                19.284227 |                     19.753091 |                          0.0 |                      1.223615 |                           1.249589 |              18.3393 |                 16.964613 |       996.6607 |    1.21073 | -2.9 | 15.632162 |  -7.607027 |                   0.0 |                       0.0 |                 103.800167 |                 10.408237 |                      0.0 |   -10.408237 |              0.0 |          93.39193 |            176.542034 |                 100.0 |               100.628883 |               87.719298 |  0.0 |  0.0 | -0.002953 | -0.002953 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.256857 | 0.071349 | 0.002953 |  0.0 |  0.0 |   0.0 |  nan | 8729.609427 | 79.623355 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.256857 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.071349 |
    | 1997-08-03 01:00:00 |  0.0 | 16.6 |              0.0 |                      0.0 |             0.0 |             94.9 |       0.8 |              1015.0 | 0.0 | 0.0 | 17.241667 |             85.991667 |                   1.3 |                          15.7 |           136.579167 |  0.0 | 16.6 | 17.241667 |    0.641273 |         1.603439 |           0.641273 |          0.8 |                18.920184 |                      19.70628 |                          0.0 |                       1.20339 |                           1.246999 |            17.955254 |                 16.945759 |     997.044746 |   1.212158 | -2.9 | 15.688646 |  -5.468121 |                   0.0 |                       0.0 |                 103.800167 |                  10.41572 |                      0.0 |    -10.41572 |              0.0 |         93.384446 |            176.542034 |                 100.0 |               100.628883 |               87.719298 |  0.0 |  0.0 | -0.002673 | -0.002673 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.250222 | 0.069506 | 0.005626 |  0.0 |  0.0 |   0.0 |  nan | 8732.177548 | 79.623355 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.250222 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.069506 |
    | 1997-08-03 02:00:00 |  0.0 | 16.4 |              0.0 |                      0.0 |             0.0 |             95.9 |       0.8 |              1015.0 | 0.0 | 0.0 |    17.175 |             86.233333 |                   1.3 |                          15.7 |           136.579167 |  0.0 | 16.4 |    17.175 |    0.641273 |         1.575992 |           0.641273 |          0.8 |                 18.68084 |                       19.6233 |                          0.0 |                      1.190065 |                           1.242407 |            17.914926 |                 16.921826 |     997.085074 |   1.213014 | -2.9 | 15.719464 |  -4.083217 |                   0.0 |                       0.0 |                 103.800167 |                 10.421627 |                      0.0 |   -10.421627 |              0.0 |         93.378539 |            176.542034 |                 100.0 |               100.628883 |               87.719298 |  0.0 |  0.0 | -0.004096 | -0.004096 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.243758 | 0.067711 | 0.009722 |  0.0 |  0.0 |   0.0 |  nan | 8733.360765 | 79.623355 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.243758 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.067711 |
    | 1997-08-03 03:00:00 |  0.0 | 16.3 |              0.0 |                      0.0 |             0.0 |             96.7 |       0.8 |              1015.0 | 0.0 | 0.0 |   17.0625 |             86.708333 |                   1.3 |                          15.5 |           136.579167 |  0.0 | 16.3 |   17.0625 |    0.641273 |         1.548545 |           0.641273 |          0.8 |                18.562165 |                     19.483964 |                          0.0 |                      1.183449 |                            1.23469 |            17.949613 |                  16.89422 |     997.050387 |   1.213417 | -2.9 | 15.733663 |  -3.398025 |                   0.0 |                       0.0 |                 103.800167 |                 10.455184 |                      0.0 |   -10.455184 |              0.0 |         93.344983 |            176.542034 |                 100.0 |               100.628883 |               87.719298 |  0.0 |  0.0 | -0.005245 | -0.005245 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.237462 | 0.065962 | 0.014967 |  0.0 |  0.0 |   0.0 |  nan |  8733.85879 | 79.623355 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.237462 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.065962 |
    | 1997-08-03 04:00:00 |  0.0 | 16.0 |              0.0 |                 0.429734 |        1.943686 |             97.2 |       0.8 |              1015.0 | 0.0 | 0.0 | 16.908333 |             87.366667 |                   1.3 |                     14.929734 |            135.50182 |  0.0 | 16.0 | 16.908333 |    0.641273 |         1.504432 |           0.641273 |          0.8 |                18.210086 |                     19.294427 |                          0.0 |                      1.163788 |                           1.224181 |            17.700204 |                 16.856897 |     997.299796 |    1.21479 | -2.9 | 15.739639 |  -1.562167 |              1.477201 |                  1.477201 |                 102.981383 |                 10.555242 |                      0.0 |    -9.078041 |              0.0 |         92.426141 |            176.542034 |                 100.0 |               100.628883 |               92.771059 |  0.0 |  0.0 | -0.004905 | -0.004905 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.231328 | 0.064258 | 0.019872 |  0.0 |  0.0 |   0.0 |  nan | 8732.520957 | 79.623355 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.231328 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.064258 |
    | 1997-08-03 05:00:00 |  0.0 | 15.9 |              0.0 |                      1.0 |       21.932442 |             97.5 |       0.6 |              1015.0 | 0.0 | 0.0 | 16.729167 |             88.204167 |                   1.3 |                     14.929734 |           134.099005 |  0.0 | 15.9 | 16.729167 |    0.480955 |         1.453638 |           0.480955 |          0.6 |                18.094032 |                     19.076181 |                          0.0 |                      1.157296 |                           1.212063 |            17.641681 |                 16.825987 |     997.358319 |   1.215237 | -2.9 | 15.723585 |  -1.058491 |             16.668656 |                 16.668656 |                 101.915244 |                 10.545193 |                      0.0 |     6.123463 |              0.0 |         91.370051 |            235.389378 |                 100.0 |               100.628883 |              100.447602 |  0.0 |  0.0 |  0.004992 |  0.004992 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.225353 | 0.062598 |  0.01488 |  0.0 |  0.0 |   0.0 |  nan | 8730.679448 | 79.623355 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.225353 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.062598 |
    | 1997-08-03 06:00:00 |  0.0 | 16.0 |              0.0 |                      1.0 |       57.256186 |             97.7 |       0.9 |              1015.0 | 0.0 | 0.0 | 16.533333 |             89.191667 |                   1.3 |                     14.929734 |           131.855513 |  0.0 | 16.0 | 16.533333 |    0.721432 |         1.392031 |           0.721432 |          0.9 |                18.210086 |                     18.840106 |                          0.0 |                      1.163788 |                           1.198935 |            17.791254 |                 16.803805 |     997.208746 |   1.214748 | -2.9 | 15.701487 |   -1.79108 |             43.514701 |                 43.514701 |                  100.21019 |                  10.52516 |                      0.0 |    32.989541 |              0.0 |          89.68503 |            156.926252 |                 100.0 |               100.628883 |              100.447602 |  0.0 |  0.0 |  0.026203 |   0.01488 | 0.008643 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.219531 | 0.060981 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8729.570528 | 79.614713 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.219531 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.060981 |
    | 1997-08-03 07:00:00 |  0.0 | 16.6 |              0.0 |                      1.0 |      109.332844 |             97.4 |       0.9 |              1015.0 | 0.0 | 0.0 |    16.375 |                90.125 |                   1.3 |                     14.929734 |           129.465215 |  0.0 | 16.6 |    16.375 |    0.721432 |         1.334591 |           0.721432 |          0.9 |                18.920184 |                     18.651109 |                          0.0 |                       1.20339 |                           1.188408 |            18.428259 |                 16.809312 |     996.571741 |   1.211943 | -2.9 |  15.68818 |  -5.470922 |             83.092962 |                 83.092962 |                  98.393563 |                  10.49073 |                      0.0 |    72.602232 |              0.0 |         87.902834 |            156.926252 |                 100.0 |               100.640774 |               100.45605 |  0.0 |  0.0 |  0.057366 |       0.0 | 0.043948 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.213861 | 0.059406 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  8732.14145 | 79.570765 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.213861 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.059406 |
    | 1997-08-03 08:00:00 |  0.0 | 17.4 |              0.0 |                      1.0 |      170.949153 |             96.8 |       0.9 |              1016.0 | 0.0 | 0.0 | 16.216667 |                91.275 |                   1.3 |                     14.929734 |           127.329763 |  0.0 | 17.4 | 16.216667 |    0.721432 |         1.260484 |           0.721432 |          0.9 |                19.904589 |                     18.463773 |                          0.0 |                      1.257963 |                           1.177959 |            19.267642 |                 16.852809 |     996.732358 |   1.209425 | -2.9 | 15.719031 | -10.085816 |            129.921356 |                129.921356 |                   96.77062 |                 10.426918 |                      0.0 |   119.494439 |              0.0 |         86.343702 |            156.926252 |                 100.0 |                100.70125 |              100.499005 |  0.0 |  0.0 |  0.096008 |       0.0 | 0.073907 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.208337 | 0.057871 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8739.327265 | 79.496858 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.208337 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.057871 |
    | 1997-08-03 09:00:00 |  0.0 | 19.0 |              0.2 |                      1.0 |      311.762625 |             86.1 |       1.3 |              1016.0 | 0.0 | 0.0 |   16.1125 |             91.991667 |                   1.5 |                     14.929734 |           129.903206 |  0.0 | 19.0 |   16.1125 |    1.042069 |         1.203904 |           1.042069 |          1.3 |                22.008543 |                     18.341425 |                          0.0 |                      1.373407 |                           1.171128 |            18.949356 |                 16.872582 |     997.050644 |   1.202945 | -2.9 | 15.805261 | -19.168437 |            236.939595 |                236.939595 |                  98.726436 |                 10.804921 |                      0.0 |   226.134674 |              0.0 |         87.921515 |            108.641252 |                 100.0 |               100.802976 |              100.571226 |  0.0 |  0.0 |  0.222649 |       0.0 | 0.160936 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.202955 | 0.056376 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8755.595702 | 79.335922 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.202955 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.056376 |
    | 1997-08-03 10:00:00 |  0.0 | 20.3 |              0.5 |                      1.0 |        501.5833 |             76.8 |       1.5 |              1016.0 | 0.0 | 0.0 | 16.083333 |             92.241667 |                   1.7 |                     14.929734 |           132.285843 |  0.0 | 20.3 | 16.083333 |    1.202387 |         1.149836 |           1.202387 |          1.5 |                23.858503 |                     18.307295 |                          0.0 |                      1.473678 |                           1.169221 |             18.32333 |                 16.886954 |      997.67667 |   1.197896 | -2.9 | 16.000482 | -25.797109 |            381.203308 |                381.203308 |                 100.537241 |                 11.199993 |                      0.0 |   370.003315 |              0.0 |         89.337248 |             94.155751 |                 100.0 |               101.024603 |              100.728426 |  0.0 |  0.0 |  0.389789 |       0.0 | 0.275473 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.197713 |  0.05492 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8778.492812 | 79.060449 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.197713 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.05492 |
    | 1997-08-03 11:00:00 |  0.0 | 21.4 |              0.7 |                      1.0 |      615.018728 |             71.8 |       1.2 |              1016.0 | 0.0 | 0.0 |    16.125 |             92.104167 |                   2.2 |                     14.929734 |           135.919957 |  0.0 | 21.4 |    16.125 |    0.961909 |         1.089916 |           0.961909 |          1.2 |                25.528421 |                     18.356069 |                          0.0 |                      1.563281 |                           1.171946 |            18.329406 |                 16.906704 |     997.670594 |    1.19342 | -2.9 | 16.275247 | -30.748518 |            467.414233 |                467.414233 |                 103.299167 |                 12.222123 |                      0.0 |    455.19211 |              0.0 |         91.077044 |            117.694689 |                 100.0 |               101.404317 |              100.997295 |  0.0 |  0.0 |   0.46611 |       0.0 | 0.348731 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.192605 | 0.053502 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8806.341329 | 78.711718 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.192605 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.053502 |
    | 1997-08-03 12:00:00 |  0.0 | 21.3 |              0.8 |                      1.0 |      626.544325 |             67.5 |       1.3 |              1016.0 | 0.0 | 0.0 | 16.204167 |             91.729167 |                   2.8 |                     14.929734 |           145.821804 |  0.0 | 21.3 | 16.204167 |    1.042069 |         1.037502 |           1.042069 |          1.3 |                 25.37251 |                     18.449053 |                          0.0 |                       1.55495 |                           1.177138 |            17.126444 |                 16.923163 |     998.873556 |   1.194363 | -2.9 | 16.609429 | -28.143424 |            476.173687 |                476.173687 |                 110.824571 |                 13.461134 |                      0.0 |   462.712553 |              0.0 |         97.363437 |            108.641252 |                 100.0 |               101.885667 |              101.337292 |  0.0 |  0.0 |  0.490849 |       0.0 | 0.360693 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.18763 |  0.05212 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8831.584754 | 78.351026 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.18763 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.05212 |
    | 1997-08-03 13:00:00 |  0.0 | 21.8 |              0.5 |                      1.0 |      496.133417 |             66.1 |       1.5 |              1016.0 | 0.0 | 0.0 | 16.329167 |             91.104167 |                   3.2 |                     14.929734 |           152.606529 |  0.0 | 21.8 | 16.329167 |    1.202387 |         1.012602 |           1.202387 |          1.5 |                26.160453 |                      18.59671 |                          0.0 |                      1.596982 |                           1.185375 |             17.29206 |                 16.942378 |      998.70794 |   1.192265 | -2.9 |  16.91235 | -29.325898 |            377.061397 |                377.061397 |                 115.980962 |                 14.297428 |                      0.0 |   362.763969 |              0.0 |        101.683534 |             94.155751 |                 100.0 |               102.384309 |               101.68852 |  0.0 |  0.0 |  0.424547 |       0.0 | 0.303239 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.182784 | 0.050773 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8858.010651 | 78.047787 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.182784 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.050773 |
    | 1997-08-03 14:00:00 |  0.0 | 22.9 |              0.4 |                      1.0 |      419.520994 |             63.4 |       1.9 |              1016.0 | 0.0 | 0.0 | 16.545833 |             90.058333 |                   3.4 |                     14.929734 |           157.353237 |  0.0 | 22.9 | 16.545833 |    1.523023 |         0.984394 |           1.523023 |          1.9 |                27.969419 |                     18.855098 |                          0.0 |                      1.692831 |                           1.199769 |            17.732611 |                 16.980587 |     998.267389 |   1.187639 | -2.9 | 17.229461 | -34.023233 |            318.835956 |                318.835956 |                  119.58846 |                 14.726001 |                      0.0 |   304.109955 |              0.0 |         104.86246 |             74.333488 |                 100.0 |               102.804149 |              101.983469 |  0.0 |  0.0 |  0.415176 |       0.0 | 0.282369 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.178062 | 0.049462 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8889.133884 | 77.765418 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.178062 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.049462 |
    | 1997-08-03 15:00:00 |  0.0 | 22.7 |              0.5 |                      1.0 |      387.887353 |             62.4 |       1.9 |              1016.0 | 0.0 | 0.0 | 16.816667 |             88.816667 |                   3.8 |                     14.929734 |           164.256877 |  0.0 | 22.7 | 16.816667 |    1.523023 |         0.968687 |           1.523023 |          1.9 |                27.632633 |                     19.182495 |                          0.0 |                      1.675052 |                           1.217969 |            17.242763 |                 17.037252 |     998.757237 |    1.18866 | -2.9 |  17.60294 |  -30.58236 |            294.794388 |                294.794388 |                 124.835227 |                 15.562805 |                      0.0 |   279.231583 |              0.0 |        109.272421 |             74.333488 |                 100.0 |               103.195613 |              102.257848 |  0.0 |  0.0 |  0.393194 |       0.0 | 0.266636 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.173463 | 0.048184 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8916.816245 | 77.498782 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.173463 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.048184 |
    | 1997-08-03 16:00:00 |  0.0 | 22.5 |              0.5 |                      1.0 |      278.496872 |             61.1 |       2.3 |              1016.0 | 0.0 | 0.0 |   17.1375 |             87.333333 |                   4.1 |                     14.929734 |           170.073414 |  0.0 | 22.5 |   17.1375 |     1.84366 |         0.991339 |            1.84366 |          2.3 |                27.299387 |                     19.576758 |                          0.0 |                      1.657431 |                            1.23983 |            16.679926 |                 17.097035 |     999.320074 |   1.189715 | -2.9 | 17.935128 |  -27.38923 |            211.657623 |                211.657623 |                 129.255794 |                 16.204965 |                      0.0 |   195.452657 |              0.0 |        113.050829 |             61.405925 |                 100.0 |               103.565733 |              102.516706 |  0.0 |  0.0 |   0.34217 |       0.0 | 0.218982 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.168982 | 0.046939 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8941.305475 | 77.279799 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.168982 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.046939 |
    | 1997-08-03 17:00:00 |  0.0 | 21.9 |              0.3 |                      1.0 |      137.138608 |             62.1 |       2.4 |              1016.0 | 0.0 | 0.0 |   17.4875 |                85.875 |                   4.4 |                     14.929734 |           173.470856 |  0.0 | 21.9 |   17.4875 |    1.923819 |         1.017332 |           1.923819 |          2.4 |                26.320577 |                     20.014927 |                          0.0 |                      1.605502 |                           1.264057 |            16.345078 |                 17.187818 |     999.654922 |   1.192283 | -2.9 | 18.228999 | -22.026006 |            104.225342 |                104.225342 |                  131.83785 |                 16.818923 |                      0.0 |     87.40642 |              0.0 |        115.018928 |             58.847345 |                 100.0 |               103.870047 |              102.729133 |  0.0 |  0.0 |  0.236219 |       0.0 | 0.147975 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.164617 | 0.045727 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8960.431481 | 77.131824 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.164617 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.045727 |
    | 1997-08-03 18:00:00 |  0.0 | 21.4 |              0.1 |                      1.0 |       51.080714 |             67.0 |       2.5 |              1016.0 | 0.0 | 0.0 |   17.8375 |             84.620833 |                   4.5 |                     14.929734 |           174.440885 |  0.0 | 21.4 |   17.8375 |    2.003978 |         1.054998 |           2.003978 |          2.5 |                25.528421 |                     20.461645 |                          0.0 |                      1.563281 |                           1.288683 |            17.104042 |                 17.314814 |     998.895958 |   1.193968 | -2.9 | 18.458511 | -17.648933 |             38.821343 |                 38.821343 |                 132.575073 |                 16.973018 |                      0.0 |    21.848325 |              0.0 |        115.602055 |             56.493451 |                 100.0 |                104.07586 |              102.872593 |  0.0 |  0.0 |  0.156077 |       0.0 | 0.095816 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.160365 | 0.044546 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8975.180414 | 77.036008 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.160365 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.044546 |
    | 1997-08-03 19:00:00 |  0.0 | 20.7 |              0.0 |                      1.0 |       13.632817 |             74.5 |       2.5 |              1016.0 | 0.0 | 0.0 | 18.170833 |                  83.7 |                   4.5 |                     15.429734 |           175.008919 |  0.0 | 20.7 | 18.170833 |    2.003978 |         1.100997 |           2.003978 |          2.5 |                24.454368 |                     20.895167 |                          0.0 |                      1.505746 |                           1.312512 |            18.218504 |                 17.489255 |     997.781496 |   1.196313 | -2.9 | 18.635498 |  -12.38701 |             10.360941 |                 10.360941 |                 133.006779 |                 16.556776 |                      0.0 |    -6.195835 |              0.0 |        116.450003 |             56.493451 |                 100.0 |               104.209204 |              102.965449 |  0.0 |  0.0 |   0.09611 |       0.0 | 0.058452 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.156222 | 0.043395 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8984.667424 | 76.977556 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.156222 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.043395 |
    | 1997-08-03 20:00:00 |  0.0 | 19.4 |              0.0 |                   0.1364 |        0.185944 |             81.2 |       2.2 |              1016.0 | 0.0 | 0.0 | 18.454167 |             83.066667 |                   4.5 |                     15.566134 |           175.016667 |  0.0 | 19.4 | 18.454167 |    1.763501 |         1.120309 |           1.763501 |          2.2 |                22.563931 |                      21.26995 |                          0.0 |                      1.403627 |                           1.333058 |            18.321912 |                 17.668238 |     997.678088 |   1.201582 | -2.9 | 18.749342 |  -3.903945 |              0.141318 |                  0.141318 |                 133.012667 |                 16.343497 |                      0.0 |    -16.20218 |              0.0 |        116.669169 |             64.197103 |                 100.0 |               104.290579 |               89.533309 |  0.0 |  0.0 |  0.049374 |       0.0 | 0.032413 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.152187 | 0.042274 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  8985.67137 | 76.945143 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.152187 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.042274 |
    | 1997-08-03 21:00:00 |  0.0 | 17.8 |              0.0 |                      0.0 |             0.0 |             86.9 |       1.7 |              1016.0 | 0.0 | 0.0 |     18.65 |               82.7375 |                   4.5 |                     15.566134 |           175.016667 |  0.0 | 17.8 |     18.65 |    1.362705 |         1.127089 |           1.362705 |          1.7 |                20.413369 |                     21.532411 |                          0.0 |                      1.286025 |                           1.347418 |            17.739217 |                 17.815378 |     998.260783 |   1.208454 | -2.9 |  18.76139 |   5.768339 |                   0.0 |                       0.0 |                 133.012667 |                 16.221809 |                      0.0 |   -16.221809 |              0.0 |        116.790858 |             83.078604 |                 100.0 |               104.335713 |               87.719298 |  0.0 |  0.0 |  0.019253 |       0.0 | 0.013439 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.148256 | 0.041182 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8977.003031 | 76.931705 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.148256 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.041182 |
    | 1997-08-03 22:00:00 |  0.0 | 17.0 |              0.0 |                      0.0 |             0.0 |             90.1 |       1.7 |              1017.0 | 0.0 | 0.0 | 18.808333 |             82.554167 |                   4.5 |                     15.566134 |           175.016667 |  0.0 | 17.0 | 18.808333 |    1.362705 |         1.142201 |           1.362705 |          1.7 |                19.406929 |                     21.746678 |                          0.0 |                      1.230421 |                           1.359123 |            17.485643 |                 17.952788 |     999.514357 |   1.213101 | -2.9 |  18.65737 |   9.944218 |                   0.0 |                       0.0 |                 133.012667 |                 16.101364 |                      0.0 |   -16.101364 |              0.0 |        116.911302 |             83.078604 |                 100.0 |               104.354428 |               87.719298 |  0.0 |  0.0 |  0.010292 |       0.0 | 0.007132 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.144426 | 0.040118 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8964.158813 | 76.924573 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.144426 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.040118 |
    | 1997-08-03 23:00:00 |  0.0 | 16.4 |              0.0 |                      0.0 |             0.0 |             90.9 |       2.3 |              1017.0 | 0.0 | 0.0 | 18.941667 |             82.379167 |                   4.5 |                     15.566134 |           175.016667 |  0.0 | 16.4 | 18.941667 |     1.84366 |         1.185687 |            1.84366 |          2.3 |                 18.68084 |                     21.928555 |                          0.0 |                      1.190065 |                           1.369047 |            16.980884 |                 18.064561 |    1000.019116 |   1.215845 | -2.9 | 18.503239 |  12.619435 |                   0.0 |                       0.0 |                 133.012667 |                 16.005095 |                      0.0 |   -16.005095 |              0.0 |        117.007571 |             61.405925 |                 100.0 |                104.36436 |               87.719298 |  0.0 |  0.0 |  0.014655 |       0.0 | 0.009242 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.140696 | 0.039082 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8948.639378 | 76.915331 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.140696 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.039082 |
    | 1997-08-04 00:00:00 |  0.0 | 15.9 |              0.0 |                      0.0 |             0.0 |             88.0 |       2.3 |              1017.0 | 0.0 | 0.0 |      18.9 |             82.083333 |                   4.5 |                     15.566134 |           175.016667 |  0.0 | 15.9 |      18.9 |     1.84366 |         1.235786 |            1.84366 |          2.3 |                18.094032 |                     21.871576 |                          0.0 |                      1.157296 |                           1.365939 |            15.922748 |                 17.952919 |    1001.077252 |    1.21843 | -2.9 | 18.317006 |  14.502035 |                   0.0 |                       0.0 |                 133.012667 |                 16.128293 |                      0.0 |   -16.128293 |              0.0 |        116.884374 |             61.405925 |                 100.0 |               104.377231 |               87.719298 |  0.0 |  0.0 |  0.024014 |       0.0 | 0.015057 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.137061 | 0.038073 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8931.237343 | 76.900275 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.137061 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.038073 |
    | 1997-08-04 01:00:00 |  0.0 | 15.4 |              0.0 |                      0.0 |             0.0 |             89.2 |       2.2 |              1016.0 | 0.0 | 0.0 |     18.85 |             81.845833 |                   4.5 |                     15.566134 |           175.016667 |  0.0 | 15.4 |     18.85 |    1.763501 |         1.282546 |           1.763501 |          2.2 |                17.523416 |                     21.803372 |                          0.0 |                      1.125295 |                           1.362218 |            15.630887 |                 17.845152 |    1000.369113 |   1.219467 | -2.9 | 18.108181 |  16.249089 |                   0.0 |                       0.0 |                 133.012667 |                 16.244859 |                      0.0 |   -16.244859 |              0.0 |        116.767808 |             64.197103 |                 100.0 |               104.398203 |               87.719298 |  0.0 |  0.0 |  0.018207 |       0.0 | 0.011515 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.133521 | 0.037089 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8912.088255 | 76.888759 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.133521 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.037089 |
    | 1997-08-04 02:00:00 |  0.0 | 15.2 |              0.0 |                      0.0 |             0.0 |             88.8 |       2.3 |              1016.0 | 0.0 | 0.0 |      18.8 |                 81.55 |                   4.5 |                     15.566134 |           175.016667 |  0.0 | 15.2 |      18.8 |     1.84366 |         1.332645 |            1.84366 |          2.3 |                17.299618 |                     21.735354 |                          0.0 |                      1.112706 |                           1.358505 |            15.362061 |                 17.725181 |    1000.637939 |   1.220436 | -2.9 | 17.878392 |  16.070354 |                   0.0 |                       0.0 |                 133.012667 |                 16.376926 |                      0.0 |   -16.376926 |              0.0 |        116.635741 |             61.405925 |                 100.0 |               104.414243 |               87.719298 |  0.0 |  0.0 |  0.020574 |       0.0 | 0.012795 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.130072 | 0.036131 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |   8893.1179 | 76.875964 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.130072 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.036131 |
    | 1997-08-04 03:00:00 |  0.0 | 14.9 |              0.0 |                      0.0 |             0.0 |             89.7 |       2.0 |              1016.0 | 0.0 | 0.0 | 18.741667 |             81.258333 |                   4.5 |                     15.566134 |           175.016667 |  0.0 | 14.9 | 18.741667 |    1.603182 |         1.372725 |           1.603182 |          2.0 |                16.968612 |                     21.656234 |                          0.0 |                      1.094046 |                           1.354184 |            15.220845 |                 17.597495 |    1000.779155 |   1.221771 | -2.9 | 17.650748 |  16.504489 |                   0.0 |                       0.0 |                 133.012667 |                 16.516685 |                      0.0 |   -16.516685 |              0.0 |        116.495982 |             70.616813 |                 100.0 |               104.432066 |               87.719298 |  0.0 |  0.0 |  0.013655 |       0.0 | 0.008851 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.126712 | 0.035198 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8873.713411 | 76.867113 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.126712 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.035198 |
    | 1997-08-04 04:00:00 |  0.0 | 14.3 |              0.0 |                 0.398044 |        1.590517 |             93.0 |       2.3 |              1016.0 | 0.0 | 0.0 | 18.670833 |             81.083333 |                   4.5 |                     15.534444 |           175.001952 |  0.0 | 14.3 | 18.670833 |     1.84366 |         1.422824 |            1.84366 |          2.3 |                16.323194 |                     21.560498 |                          0.0 |                      1.057522 |                           1.348953 |             15.18057 |                  17.48197 |     1000.81943 |    1.22434 | -2.9 | 17.417894 |  18.707366 |              1.208793 |                  1.208793 |                 133.001483 |                  16.65652 |                      0.0 |   -15.447727 |              0.0 |        116.344963 |             61.405925 |                 100.0 |               104.444396 |               93.266466 |  0.0 |  0.0 |  0.007264 |       0.0 | 0.004365 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.123439 | 0.034289 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8852.106046 | 76.862748 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.123439 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.034289 |
    | 1997-08-04 05:00:00 |  0.0 | 14.1 |              0.0 |                      1.0 |       19.950903 |             93.6 |       2.5 |              1017.0 | 0.0 | 0.0 | 18.595833 |             80.920833 |                   4.5 |                     15.534444 |           174.919387 |  0.0 | 14.1 | 18.595833 |    2.003978 |         1.486284 |           2.003978 |          2.5 |                16.112886 |                     21.459534 |                          0.0 |                      1.045579 |                           1.343433 |            15.081661 |                 17.365233 |    1001.918339 |   1.226451 | -2.9 | 17.158606 |  18.351635 |             15.162686 |                 15.162686 |                 132.938735 |                 16.779096 |                      0.0 |     -1.61641 |              0.0 |        116.159638 |             56.493451 |                 100.0 |               104.450477 |              103.133286 |  0.0 |  0.0 |  0.018172 |       0.0 | 0.010118 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.120251 | 0.033403 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  8830.85441 |  76.85263 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.120251 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.033403 |
    | 1997-08-04 06:00:00 |  0.0 | 14.3 |              0.1 |                      1.0 |       62.985775 |             94.6 |       2.4 |              1017.0 | 0.0 | 0.0 |    18.525 |             80.791667 |                   4.6 |                     15.534444 |            175.15812 |  0.0 | 14.3 |    18.525 |    1.923819 |         1.536383 |           1.923819 |          2.4 |                16.323194 |                     21.364558 |                          0.0 |                      1.057522 |                           1.338237 |            15.441741 |                 17.260782 |    1001.558259 |   1.225432 | -2.9 | 16.903586 |  15.621518 |             47.869189 |                 47.869189 |                 133.120171 |                 17.089561 |                      0.0 |    30.779628 |              0.0 |         116.03061 |             58.847345 |                 100.0 |               104.464573 |              103.143085 |  0.0 |  0.0 |   0.04094 |       0.0 | 0.023223 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.117144 |  0.03254 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8812.332893 | 76.829407 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.117144 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.03254 |
    | 1997-08-04 07:00:00 |  0.0 | 15.4 |              0.9 |                      1.0 |      285.776389 |             93.3 |       2.5 |              1017.0 | 0.0 | 0.0 |    18.475 |             80.620833 |                   5.5 |                     15.534444 |           182.509935 |  0.0 | 15.4 |    18.475 |    2.003978 |         1.589823 |           2.003978 |          2.5 |                17.523416 |                     21.297737 |                          0.0 |                      1.125295 |                            1.33458 |            16.349347 |                 17.170413 |    1000.650653 |   1.220347 | -2.9 | 16.681328 |   7.687968 |            217.190056 |                217.190056 |                  138.70755 |                 19.012403 |                      0.0 |   198.177653 |              0.0 |        119.695148 |             56.493451 |                 100.0 |               104.496928 |              103.165573 |  0.0 |  0.0 |  0.189733 |       0.0 |   0.1075 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.114118 |   0.0317 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8801.744925 | 76.721908 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.114118 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0317 |
    | 1997-08-04 08:00:00 |  0.0 | 16.8 |              1.0 |                      1.0 |      475.225247 |             86.9 |       2.5 |              1017.0 | 0.0 | 0.0 |     18.45 |             80.208333 |                   6.5 |                     15.534444 |           195.188105 |  0.0 | 16.8 |     18.45 |    2.003978 |         1.643262 |           2.003978 |          2.5 |                19.162205 |                     21.264396 |                          0.0 |                      1.216842 |                           1.332754 |            16.651956 |                 17.055817 |    1000.348044 |   1.214317 | -2.9 | 16.554272 |  -1.474365 |            361.171188 |                361.171188 |                  148.34296 |                 21.208047 |                      0.0 |   339.963141 |              0.0 |        127.134913 |             56.493451 |                 100.0 |               104.646751 |              103.269652 |  0.0 |  0.0 |  0.343753 |       0.0 | 0.198377 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.111171 | 0.030881 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  8800.31929 | 76.523531 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.111171 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.030881 |
    | 1997-08-04 09:00:00 |  0.0 | 18.2 |              1.0 |                      1.0 |      614.006719 |             78.9 |       2.9 |              1017.0 | 0.0 | 0.0 | 18.416667 |             79.908333 |                   7.3 |                     15.534444 |           207.781609 |  0.0 | 18.2 | 18.416667 |    2.324615 |         1.696701 |           2.324615 |          2.9 |                20.933479 |                     21.220011 |                          0.0 |                      1.314615 |                           1.330323 |            16.516515 |                 16.956557 |    1000.483485 |   1.208543 | -2.9 | 16.537165 |  -9.977011 |            466.645107 |                466.645107 |                 157.914023 |                 22.996116 |                      0.0 |   443.648991 |              0.0 |        134.917907 |             48.701251 |                 100.0 |               104.923431 |              103.461623 |  0.0 |  0.0 |  0.494205 |       0.0 | 0.274475 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.108299 | 0.030083 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8807.396301 | 76.249056 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.108299 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.030083 |
    | 1997-08-04 10:00:00 |  0.0 | 19.5 |              0.9 |                      1.0 |      643.806508 |             75.8 |       3.1 |              1017.0 | 0.0 | 0.0 | 18.383333 |             79.866667 |                   7.7 |                     15.534444 |           213.707576 |  0.0 | 19.5 | 18.383333 |    2.484933 |         1.750141 |           2.484933 |          3.1 |                22.704676 |                     21.175708 |                          0.0 |                      1.411269 |                           1.327896 |            17.210144 |                 16.912332 |     999.789856 |   1.202862 | -2.9 | 16.622089 | -17.267466 |            489.292946 |                489.292946 |                 162.417758 |                 23.882018 |                      0.0 |   465.410928 |              0.0 |         138.53574 |             45.559235 |                 100.0 |               105.306685 |              103.727044 |  0.0 |  0.0 |  0.550473 |       0.0 |  0.30313 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.105502 | 0.029306 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8821.763767 | 75.945926 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.105502 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.029306 |
    | 1997-08-04 11:00:00 |  0.0 | 20.3 |              0.8 |                      1.0 |      600.262981 |             73.4 |       3.3 |              1018.0 | 0.0 | 0.0 |   18.3375 |             79.933333 |                   7.8 |                     15.534444 |           213.092753 |  0.0 | 20.3 |   18.3375 |    2.645251 |          1.82028 |           2.645251 |          3.3 |                23.858503 |                     21.114922 |                          0.0 |                      1.473678 |                           1.324565 |            17.512141 |                 16.877861 |    1000.487859 |   1.200635 | -2.9 | 16.794499 | -21.033009 |            456.199865 |                456.199865 |                 161.950493 |                   24.1295 |                      0.0 |   432.070365 |              0.0 |        137.820992 |             42.798069 |                 100.0 |               105.730548 |              104.019919 |  0.0 |  0.0 |  0.548409 |       0.0 | 0.297686 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.102776 | 0.028549 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8839.896776 |  75.64824 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.102776 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.028549 |
    | 1997-08-04 12:00:00 |  0.0 | 21.0 |              0.9 |                      1.0 |      620.514139 |             68.3 |       3.3 |              1018.0 | 0.0 | 0.0 |    18.325 |             79.966667 |                   7.9 |                     15.534444 |           212.841496 |  0.0 | 21.0 |    18.325 |    2.645251 |         1.887079 |           2.645251 |          3.3 |                24.909749 |                     21.098371 |                          0.0 |                      1.530181 |                           1.323658 |            17.013358 |                 16.871664 |    1000.986642 |   1.198001 | -2.9 | 17.012095 | -23.927432 |            471.590746 |                471.590746 |                 161.759537 |                 24.341655 |                      0.0 |    447.24909 |              0.0 |        137.417882 |             42.798069 |                 100.0 |               106.147416 |              104.307278 |  0.0 |  0.0 |  0.597585 |       0.0 | 0.327488 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.100122 | 0.027812 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8860.924208 | 75.320752 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.100122 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.027812 |
    | 1997-08-04 13:00:00 |  0.0 | 21.8 |              0.8 |                      1.0 |      599.441386 |             61.3 |       3.2 |              1018.0 | 0.0 | 0.0 |    18.325 |             79.766667 |                   8.2 |                     15.534444 |           217.145994 |  0.0 | 21.8 |    18.325 |    2.565092 |         1.943859 |           2.565092 |          3.2 |                26.160453 |                     21.098371 |                          0.0 |                      1.596982 |                           1.323658 |            16.036358 |                 16.829467 |    1001.963642 |   1.195188 | -2.9 | 17.264424 | -27.213457 |            455.575453 |                455.575453 |                 165.030956 |                 25.038979 |                      0.0 |   430.536475 |              0.0 |        139.991977 |             44.135508 |                 100.0 |               106.606735 |              104.623118 |  0.0 |  0.0 |  0.621828 |       0.0 | 0.348705 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.097535 | 0.027093 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8885.237665 | 74.972048 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.097535 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.027093 |
    | 1997-08-04 14:00:00 |  0.0 | 21.9 |              0.9 |                      1.0 |      613.551222 |             60.6 |       2.5 |              1018.0 | 0.0 | 0.0 | 18.283333 |                 79.65 |                   8.7 |                     15.534444 |           225.230587 |  0.0 | 21.9 | 18.283333 |    2.003978 |         1.963899 |           2.003978 |          2.5 |                26.320577 |                     21.043281 |                          0.0 |                      1.605502 |                           1.320638 |             15.95027 |                 16.760973 |     1002.04973 |   1.194821 | -2.9 | 17.556185 | -26.062888 |            466.298929 |                466.298929 |                 171.175246 |                 26.187015 |                      0.0 |   440.111914 |              0.0 |        144.988231 |             56.493451 |                 100.0 |                107.09665 |              104.959099 |  0.0 |  0.0 |  0.589213 |       0.0 | 0.361278 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.095016 | 0.026393 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8908.400553 | 74.610769 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.095016 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.026393 |
    | 1997-08-04 15:00:00 |  0.0 | 21.8 |              0.9 |                      1.0 |      517.107047 |             58.8 |       2.9 |              1017.0 | 0.0 | 0.0 | 18.245833 |                  79.5 |                   9.1 |                     15.534444 |           230.614741 |  0.0 | 21.8 | 18.245833 |    2.324615 |         1.997298 |           2.324615 |          2.9 |                26.160453 |                     20.993808 |                          0.0 |                      1.596982 |                           1.317924 |            15.382347 |                 16.690077 |    1001.617653 |   1.194298 | -2.9 |  17.83414 |  -23.79516 |            393.001356 |                393.001356 |                 175.267203 |                 27.142101 |                      0.0 |   365.859255 |              0.0 |        148.125102 |             48.701251 |                 100.0 |               107.605159 |              105.306852 |  0.0 |  0.0 |  0.551455 |       0.0 | 0.320086 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.092562 | 0.025712 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8929.295714 | 74.290683 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.092562 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.025712 |
    | 1997-08-04 16:00:00 |  0.0 | 22.2 |              0.9 |                      1.0 |      377.836617 |             57.2 |       3.6 |              1017.0 | 0.0 | 0.0 | 18.233333 |               79.3375 |                   9.5 |                     15.534444 |           234.753897 |  0.0 | 22.2 | 18.233333 |    2.885728 |         2.040718 |           2.885728 |          3.6 |                26.806088 |                      20.97734 |                          0.0 |                      1.631293 |                           1.317021 |            15.333082 |                 16.642897 |    1001.666918 |   1.192703 | -2.9 | 18.084882 | -24.690709 |            287.155829 |                287.155829 |                 178.412962 |                 28.068928 |                      0.0 |   259.086901 |              0.0 |        150.344034 |             39.231563 |                 100.0 |               108.056488 |              105.614671 |  0.0 |  0.0 |  0.505795 |       0.0 | 0.271017 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.090171 | 0.025047 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8951.086422 | 74.019667 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.090171 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.025047 |
    | 1997-08-04 17:00:00 |  0.0 | 21.8 |              1.0 |                      1.0 |      252.204639 |             57.2 |       4.2 |              1017.0 | 0.0 | 0.0 | 18.229167 |             79.133333 |                  10.2 |                     15.534444 |           239.548315 |  0.0 | 21.8 | 18.229167 |    3.366683 |         2.100837 |           3.366683 |          4.2 |                26.160453 |                     20.971853 |                          0.0 |                      1.596982 |                            1.31672 |            14.963779 |                 16.595726 |    1002.036221 |   1.194485 | -2.9 |  18.34637 | -20.721778 |            191.675526 |                191.675526 |                  182.05672 |                 29.636995 |                      0.0 |    162.03853 |              0.0 |        152.419724 |             33.627054 |                 100.0 |               108.439226 |              105.875099 |  0.0 |  0.0 |   0.44464 |       0.0 | 0.220995 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.087841 |   0.0244 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |   8968.9082 | 73.798672 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.087841 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0244 |
    | 1997-08-04 18:00:00 |  0.0 | 20.9 |              1.0 |                      1.0 |      128.576714 |             59.9 |       4.2 |              1017.0 | 0.0 | 0.0 | 18.208333 |               78.8375 |                  11.1 |                     15.534444 |           242.777315 |  0.0 | 20.9 | 18.208333 |    3.366683 |         2.157616 |           3.366683 |          4.2 |                 24.75714 |                     20.944437 |                          0.0 |                      1.521999 |                           1.315216 |            14.829527 |                  16.51207 |    1002.170473 |   1.198201 | -2.9 | 18.560232 |  -14.03861 |             97.718303 |                 97.718303 |                  184.51076 |                 31.712576 |                      0.0 |    66.005727 |              0.0 |        152.798184 |             33.627054 |                 100.0 |               108.751732 |              106.087324 |  0.0 |  0.0 |   0.32482 |       0.0 | 0.158739 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.085572 |  0.02377 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8980.046809 | 73.639933 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.085572 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.02377 |
    | 1997-08-04 19:00:00 |  0.0 | 19.7 |              1.0 |                      1.0 |       42.072586 |             66.1 |       3.6 |              1017.0 | 0.0 | 0.0 | 18.166667 |               78.4875 |                  12.1 |                     15.534444 |           243.962306 |  0.0 | 19.7 | 18.166667 |    2.885728 |         2.194356 |           2.885728 |          3.6 |                22.988466 |                     20.889699 |                          0.0 |                      1.426658 |                           1.312212 |            15.195376 |                 16.395802 |    1001.804624 |   1.202947 | -2.9 | 18.693895 |   -6.03663 |             31.975165 |                 31.975165 |                 185.411352 |                 34.091963 |                      0.0 |    -2.116797 |              0.0 |        151.319389 |             39.231563 |                 100.0 |               108.976433 |              106.239691 |  0.0 |  0.0 |  0.183135 |       0.0 | 0.093951 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.083362 | 0.023156 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8983.183439 | 73.545981 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.083362 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.023156 |
    | 1997-08-04 20:00:00 |  0.0 | 17.9 |              0.3 |                 0.104711 |        0.812897 |             71.7 |       3.2 |              1017.0 | 0.0 | 0.0 | 18.104167 |             78.091667 |                  12.4 |                     15.502754 |           243.988429 |  0.0 | 17.9 | 18.104167 |    2.565092 |         2.227756 |           2.565092 |          3.2 |                20.542326 |                     20.807826 |                          0.0 |                      1.293123 |                           1.307717 |            14.728848 |                 16.249178 |    1002.271152 |   1.210597 | -2.9 | 18.731535 |   4.989208 |              0.617802 |                  0.617802 |                 185.431206 |                 35.120093 |                      0.0 |   -34.502291 |              0.0 |        150.311113 |             44.135508 |                 100.0 |               109.109515 |               89.356954 |  0.0 |  0.0 |  0.100625 |       0.0 | 0.057093 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.081209 | 0.022558 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8975.294232 | 73.488889 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.081209 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.022558 |
    | 1997-08-04 21:00:00 |  0.0 | 15.8 |              0.0 |                      0.0 |             0.0 |             80.6 |       2.2 |              1018.0 | 0.0 | 0.0 | 18.020833 |             77.829167 |                  12.4 |                     15.502754 |           243.988429 |  0.0 | 15.8 | 18.020833 |    1.763501 |         2.244455 |           1.763501 |          2.2 |                17.978626 |                     20.699099 |                          0.0 |                      1.150835 |                           1.301743 |            14.490772 |                 16.109936 |    1003.509228 |    1.22071 | -2.9 | 18.636864 |  17.021185 |                   0.0 |                       0.0 |                 185.431206 |                 35.424509 |                      0.0 |   -35.424509 |              0.0 |        150.006697 |             64.197103 |                 100.0 |                109.19042 |               87.719298 |  0.0 |  0.0 |  0.029975 |       0.0 | 0.019046 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.079111 | 0.021975 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8955.373047 | 73.469843 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.079111 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.021975 |
    | 1997-08-04 22:00:00 |  0.0 | 14.8 |              0.0 |                      0.0 |             0.0 |             85.9 |       2.2 |              1018.0 | 0.0 | 0.0 | 17.929167 |             77.654167 |                  12.4 |                     15.502754 |           243.988429 |  0.0 | 14.8 | 17.929167 |    1.763501 |         2.261155 |           1.763501 |          2.2 |                16.859515 |                     20.580072 |                          0.0 |                      1.087886 |                           1.295199 |            14.482324 |                 15.981284 |    1003.517676 |   1.224953 | -2.9 |  18.39781 |  21.586859 |                   0.0 |                       0.0 |                 185.431206 |                 35.698198 |                      0.0 |   -35.698198 |              0.0 |        149.733007 |             64.197103 |                 100.0 |               109.217416 |               87.719298 |  0.0 |  0.0 |  0.012059 |       0.0 | 0.007576 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.077068 | 0.021408 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8930.886187 | 73.462266 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.077068 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.021408 |
    | 1997-08-04 23:00:00 |  0.0 | 14.1 |              0.0 |                      0.0 |             0.0 |             89.3 |       2.1 |              1018.0 | 0.0 | 0.0 | 17.833333 |               77.5875 |                  12.4 |                     15.502754 |           243.988429 |  0.0 | 14.1 | 17.833333 |    1.683342 |         2.254475 |           1.683342 |          2.1 |                16.112886 |                     20.456276 |                          0.0 |                      1.045579 |                           1.288387 |            14.388807 |                 15.871513 |    1003.611193 |   1.227981 | -2.9 | 18.103968 |  24.023805 |                   0.0 |                       0.0 |                 185.431206 |                 35.921753 |                      0.0 |   -35.921753 |              0.0 |        149.509453 |             67.254108 |                 100.0 |               109.228155 |               87.719298 |  0.0 |  0.0 | -0.000116 | -0.000116 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.075077 | 0.020855 | 0.000116 |  0.0 |  0.0 |   0.0 |  nan | 8903.962382 | 73.462266 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.075077 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.020855 |
    | 1997-08-05 00:00:00 |  0.0 | 13.4 |              0.0 |                      0.0 |             0.0 |             91.4 |       1.8 |              1018.0 | 0.0 | 0.0 | 17.729167 |             77.729167 |                  12.4 |                     15.502754 |           243.988429 |  0.0 | 13.4 | 17.729167 |    1.442864 |         2.237776 |           1.442864 |          1.8 |                15.395377 |                     20.322453 |                          0.0 |                      1.004677 |                           1.281018 |            14.071374 |                 15.796473 |    1003.928626 |   1.231127 | -2.9 | 17.780882 |  26.285292 |                   0.0 |                       0.0 |                 185.431206 |                 36.050541 |                      0.0 |   -36.050541 |              0.0 |        149.380664 |             78.463126 |                 100.0 |               109.228155 |               87.719298 |  0.0 |  0.0 | -0.009023 | -0.009023 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.073138 | 0.020316 | 0.009139 |  0.0 |  0.0 |   0.0 |  nan |  8874.77709 | 73.462266 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.073138 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.020316 |
    | 1997-08-05 01:00:00 |  0.0 | 12.8 |              0.0 |                      0.0 |             0.0 |             93.1 |       1.8 |              1018.0 | 0.0 | 0.0 | 17.620833 |             77.891667 |                  12.4 |                     15.502754 |           243.988429 |  0.0 | 12.8 | 17.620833 |    1.442864 |         2.224416 |           1.442864 |          1.8 |                14.802811 |                     20.184089 |                          0.0 |                      0.970706 |                           1.273391 |            13.781417 |                 15.721724 |    1004.218583 |   1.233843 | -2.9 | 17.430658 |  27.783951 |                   0.0 |                       0.0 |                 185.431206 |                 36.176259 |                      0.0 |   -36.176259 |              0.0 |        149.254947 |             78.463126 |                 100.0 |               109.228155 |               87.719298 |  0.0 |  0.0 | -0.013264 | -0.013264 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.071248 | 0.019791 | 0.022403 |  0.0 |  0.0 |   0.0 |  nan |  8844.09314 | 73.462266 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.071248 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.019791 |
    | 1997-08-05 02:00:00 |  0.0 | 12.3 |              0.0 |                      0.0 |             0.0 |             94.6 |       1.6 |              1018.0 | 0.0 | 0.0 |      17.5 |             78.133333 |                  12.4 |                     15.502754 |           243.988429 |  0.0 | 12.3 |      17.5 |    1.282546 |         2.201036 |           1.282546 |          1.6 |                14.324376 |                     20.030733 |                          0.0 |                      0.943147 |                            1.26493 |            13.550859 |                 15.650679 |    1004.449141 |   1.236111 | -2.9 | 17.062451 |  28.574706 |                   0.0 |                       0.0 |                 185.431206 |                 36.284611 |                      0.0 |   -36.284611 |              0.0 |        149.146595 |             88.271017 |                 100.0 |               109.228155 |               87.719298 |  0.0 |  0.0 | -0.017718 | -0.017718 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.069408 |  0.01928 | 0.040121 |  0.0 |  0.0 |   0.0 |  nan | 8812.618434 | 73.462266 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.069408 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.01928 |
    | 1997-08-05 03:00:00 |  0.0 | 12.0 |              0.0 |                      0.0 |             0.0 |             95.6 |       1.2 |              1018.0 | 0.0 | 0.0 | 17.379167 |             78.379167 |                  12.4 |                     15.502754 |           243.988429 |  0.0 | 12.0 | 17.379167 |    0.961909 |         2.174316 |           0.961909 |          1.2 |                 14.04387 |                     19.878396 |                          0.0 |                      0.926933 |                           1.256516 |            13.425939 |                 15.580521 |    1004.574061 |   1.237469 | -2.9 | 16.684755 |  28.108527 |                   0.0 |                       0.0 |                 185.431206 |                 36.390891 |                      0.0 |   -36.390891 |              0.0 |        149.040314 |            117.694689 |                 100.0 |               109.228155 |               87.719298 |  0.0 |  0.0 | -0.020716 | -0.020716 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.067615 | 0.018782 | 0.060837 |  0.0 |  0.0 |   0.0 |  nan | 8781.609907 | 73.462266 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.067615 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.018782 |
    | 1997-08-05 04:00:00 |  0.0 | 11.7 |              0.0 |                 0.366022 |        1.482347 |             96.2 |       1.9 |              1018.0 | 0.0 | 0.0 | 17.270833 |               78.5125 |                  12.4 |                     15.470733 |           243.983922 |  0.0 | 11.7 | 17.270833 |    1.523023 |         2.160956 |           1.523023 |          1.9 |                13.768192 |                      19.74268 |                          0.0 |                      0.910957 |                           1.249013 |            13.245001 |                 15.500472 |    1004.754999 |   1.238856 | -2.9 | 16.312652 |  27.675913 |              1.126584 |                  1.126584 |                  185.42778 |                 36.589063 |                      0.0 |   -35.462479 |              0.0 |        148.838717 |             74.333488 |                 100.0 |               109.228155 |               93.746399 |  0.0 |  0.0 | -0.019578 | -0.019578 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.065868 | 0.018297 | 0.080415 |  0.0 |  0.0 |   0.0 |  nan | 8751.033993 | 73.462266 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.065868 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.018297 |
    | 1997-08-05 05:00:00 |  0.0 | 11.6 |              0.0 |                      1.0 |       21.135567 |             95.7 |       1.3 |              1018.0 | 0.0 | 0.0 | 17.166667 |                  78.6 |                  12.4 |                     15.470733 |           244.033283 |  0.0 | 11.6 | 17.166667 |    1.042069 |         2.120877 |           1.042069 |          1.3 |                 13.67736 |                     19.612949 |                          0.0 |                      0.905683 |                           1.241834 |            13.089234 |                 15.415778 |    1004.910766 |   1.239363 | -2.9 | 15.945741 |  26.074448 |             16.063031 |                 16.063031 |                 185.465295 |                  36.74524 |                      0.0 |   -20.682209 |              0.0 |        148.720055 |            108.641252 |                 100.0 |               109.228155 |              106.410154 |  0.0 |  0.0 | -0.009263 | -0.009263 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.064167 | 0.017824 | 0.089679 |  0.0 |  0.0 |   0.0 |  nan | 8722.059546 | 73.462266 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.064167 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.017824 |
    | 1997-08-05 06:00:00 |  0.0 | 12.6 |              0.6 |                      1.0 |       125.88345 |             95.1 |       1.6 |              1018.0 | 0.0 | 0.0 | 17.095833 |             78.620833 |                  12.9 |                     15.470733 |           246.654019 |  0.0 | 12.6 | 17.095833 |    1.282546 |         2.094157 |           1.282546 |          1.6 |                14.609782 |                     19.525158 |                          0.0 |                      0.959602 |                           1.236973 |            13.893903 |                 15.350842 |    1004.106097 |   1.234655 | -2.9 | 15.598048 |  17.988287 |             95.671422 |                 95.671422 |                 187.457054 |                  38.00446 |                      0.0 |    57.666962 |              0.0 |        149.452594 |             88.271017 |                 100.0 |               109.228155 |              106.410154 |  0.0 |  0.0 |  0.052361 |  0.052361 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.06251 | 0.017364 | 0.037317 |  0.0 |  0.0 |   0.0 |  nan | 8701.171259 | 73.462266 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.06251 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.017364 |
    | 1997-08-05 07:00:00 |  0.0 | 14.8 |              0.9 |                      1.0 |      305.954944 |             96.3 |       1.9 |              1019.0 | 0.0 | 0.0 | 17.070833 |             78.745833 |                  12.9 |                     15.470733 |           247.494792 |  0.0 | 14.8 | 17.070833 |    1.523023 |         2.074117 |           1.523023 |          1.9 |                16.859515 |                     19.494255 |                          0.0 |                      1.087886 |                           1.235261 |            16.235713 |                 15.350914 |    1002.764287 |   1.225361 | -2.9 | 15.347388 |   3.284331 |            232.525758 |                232.525758 |                 188.096042 |                 37.987562 |                      0.0 |   194.538195 |              0.0 |         150.10848 |             74.333488 |                 100.0 |               109.228155 |              106.410154 |  0.0 |  0.0 |  0.166779 |  0.037317 | 0.079358 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.060895 | 0.016915 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8694.986928 | 73.382908 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.060895 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.016915 |
    | 1997-08-05 08:00:00 |  0.0 | 17.1 |              1.0 |                      1.0 |      511.104575 |             89.9 |       3.2 |              1019.0 | 0.0 | 0.0 | 17.083333 |             78.870833 |                  12.9 |                     15.470733 |           248.989764 |  0.0 | 17.1 | 17.083333 |    2.565092 |         2.097497 |           2.565092 |          3.2 |                19.530313 |                     19.509701 |                          0.0 |                      1.237258 |                           1.236116 |            17.557751 |                 15.387464 |    1001.442249 |   1.215051 | -2.9 | 15.273176 | -10.960941 |            388.439477 |                388.439477 |                 189.232221 |                 37.896217 |                      0.0 |    350.54326 |              0.0 |        151.336004 |             44.135508 |                 100.0 |                109.34067 |              106.486271 |  0.0 |  0.0 |  0.358863 |       0.0 | 0.185679 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.059322 | 0.016478 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8703.047869 | 73.197229 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.059322 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.016478 |
    | 1997-08-05 09:00:00 |  0.0 | 19.4 |              0.9 |                      1.0 |      629.519239 |             79.2 |       4.0 |              1019.0 | 0.0 | 0.0 | 17.133333 |             78.883333 |                  12.8 |                     15.470733 |           249.636119 |  0.0 | 19.4 | 17.133333 |    3.206365 |         2.134237 |           3.206365 |          4.0 |                22.563931 |                     19.571593 |                          0.0 |                      1.403627 |                           1.239544 |            17.870634 |                 15.438725 |    1001.129366 |   1.205358 | -2.9 | 15.369908 | -24.180553 |            478.434622 |                478.434622 |                 189.723451 |                  37.56462 |                      0.0 |   440.870002 |              0.0 |        152.158831 |             35.308407 |                 100.0 |               109.604122 |              106.664308 |  0.0 |  0.0 |  0.541806 |       0.0 |   0.2631 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.05779 | 0.016053 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8724.328423 |  72.93413 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.05779 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.016053 |
    | 1997-08-05 10:00:00 |  0.0 | 20.6 |              1.0 |                      1.0 |      765.816589 |             73.4 |       4.6 |              1019.0 | 0.0 | 0.0 | 17.179167 |             78.783333 |                  12.9 |                     15.470733 |           254.719872 |  0.0 | 20.6 | 17.179167 |     3.68732 |         2.184336 |            3.68732 |          4.6 |                24.304197 |                     19.628477 |                          0.0 |                      1.497674 |                           1.242694 |            17.839281 |                 15.463969 |    1001.160719 |   1.200448 | -2.9 | 15.625274 | -29.848354 |            582.020608 |                582.020608 |                 193.587103 |                 37.751868 |                      0.0 |    544.26874 |              0.0 |        155.835235 |             30.702962 |                 100.0 |               109.977887 |              106.916443 |  0.0 |  0.0 |  0.716842 |       0.0 |   0.3322 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.056297 | 0.015638 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8751.276776 |  72.60193 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.056297 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.015638 |
    | 1997-08-05 11:00:00 |  0.0 | 21.4 |              1.0 |                      1.0 |      767.145842 |             67.3 |       4.1 |              1019.0 | 0.0 | 0.0 |    17.225 |             78.529167 |                  13.1 |                     15.470733 |           261.673325 |  0.0 | 21.4 |    17.225 |    3.286524 |         2.211056 |           3.286524 |          4.1 |                25.528421 |                     19.685506 |                          0.0 |                      1.563281 |                            1.24585 |            17.180627 |                 15.458864 |    1001.819373 |   1.197482 | -2.9 | 15.948655 | -32.708072 |             583.03084 |                 583.03084 |                 198.871727 |                 38.247049 |                      0.0 |   544.783791 |              0.0 |        160.624678 |             34.447226 |                 100.0 |               110.450603 |              107.234579 |  0.0 |  0.0 |  0.746089 |       0.0 | 0.369947 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.054843 | 0.015234 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8781.084848 | 72.231982 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.054843 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.015234 |
    | 1997-08-05 12:00:00 |  0.0 | 22.5 |              1.0 |                      1.0 |      736.263853 |             62.2 |       4.1 |              1019.0 | 0.0 | 0.0 |   17.2875 |                78.275 |                  13.2 |                     15.470733 |            266.49623 |  0.0 | 22.5 |   17.2875 |    3.286524 |         2.237776 |           3.286524 |          4.1 |                27.299387 |                     19.763507 |                          0.0 |                      1.657431 |                           1.250165 |            16.980219 |                 15.469885 |    1002.019781 |   1.193116 | -2.9 | 16.306352 | -37.161891 |            559.560528 |                559.560528 |                 202.537135 |                  38.48429 |                      0.0 |   521.076238 |              0.0 |        164.052844 |             34.447226 |                 100.0 |               110.978084 |              107.588582 |  0.0 |  0.0 |  0.771457 |       0.0 | 0.389289 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.053426 | 0.014841 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8815.346739 | 71.842694 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.053426 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.014841 |
    | 1997-08-05 13:00:00 |  0.0 | 23.1 |              1.0 |                      1.0 |      768.006183 |             59.6 |       4.4 |              1019.0 | 0.0 | 0.0 | 17.341667 |             78.204167 |                  13.4 |                     15.470733 |           273.519763 |  0.0 | 23.1 | 17.341667 |    3.527001 |         2.277855 |           3.527001 |          4.4 |                28.309776 |                     19.831326 |                          0.0 |                      1.710768 |                           1.253914 |            16.872626 |                 15.508923 |    1002.127374 |   1.190747 | -2.9 | 16.717494 | -38.295035 |            583.684699 |                583.684699 |                  207.87502 |                  38.86309 |                      0.0 |   544.821609 |              0.0 |         169.01193 |             32.098552 |                 100.0 |               111.534357 |              107.960785 |  0.0 |  0.0 |  0.841038 |       0.0 |  0.41464 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.052046 | 0.014457 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8850.741774 | 71.428054 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.052046 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.014457 |
    | 1997-08-05 14:00:00 |  0.0 | 23.4 |              1.0 |                      1.0 |      735.929369 |             55.4 |       4.6 |              1019.0 | 0.0 | 0.0 | 17.404167 |               77.9875 |                  13.5 |                     15.470733 |           278.618852 |  0.0 | 23.4 | 17.404167 |     3.68732 |         2.347994 |            3.68732 |          4.6 |                28.827078 |                     19.909831 |                          0.0 |                      1.737973 |                           1.258253 |            15.970201 |                 15.527179 |    1003.029799 |   1.189943 | -2.9 | 17.142235 | -37.546592 |            559.306321 |                559.306321 |                 211.750328 |                 39.079824 |                      0.0 |   520.226497 |              0.0 |        172.670504 |             30.702962 |                 100.0 |               112.128253 |              108.356894 |  0.0 |  0.0 |  0.865229 |       0.0 | 0.419328 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.050702 | 0.014084 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8885.388366 | 71.008726 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.050702 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.014084 |
    | 1997-08-05 15:00:00 |  0.0 | 23.8 |              1.0 |                      1.0 |      620.374953 |             52.1 |       4.5 |              1019.0 | 0.0 | 0.0 |   17.4875 |             77.708333 |                  13.6 |                     15.470733 |           282.921682 |  0.0 | 23.8 |   17.4875 |    3.607161 |         2.401434 |           3.607161 |          4.5 |                29.529613 |                     20.014927 |                          0.0 |                      1.774812 |                           1.264057 |            15.384928 |                 15.553266 |    1003.615072 |     1.1886 | -2.9 | 17.557994 | -37.452038 |            471.484964 |                471.484964 |                 215.020478 |                 39.288738 |                      0.0 |   432.196226 |              0.0 |         175.73174 |              31.38525 |                 100.0 |               112.730358 |              108.757148 |  0.0 |  0.0 |   0.79847 |       0.0 |  0.39307 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.049392 |  0.01372 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8919.940404 | 70.615656 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.049392 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.01372 |
    | 1997-08-05 16:00:00 |  0.0 | 24.0 |              1.0 |                      1.0 |      450.095264 |             49.5 |       4.7 |              1018.0 | 0.0 | 0.0 |   17.5625 |               77.3875 |                  13.7 |                     15.470733 |           285.932459 |  0.0 | 24.0 |   17.5625 |    3.767479 |         2.438173 |           3.767479 |          4.7 |                29.886439 |                     20.109928 |                          0.0 |                      1.793476 |                             1.2693 |            14.793787 |                  15.56257 |    1003.206213 |   1.186889 | -2.9 | 17.972618 | -36.164291 |            342.072401 |                342.072401 |                 217.308669 |                 39.539407 |                      0.0 |   302.532993 |              0.0 |        177.769261 |             30.049708 |                 100.0 |               113.296152 |              109.132049 |  0.0 |  0.0 |   0.70277 |       0.0 |  0.33964 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.048116 | 0.013366 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8953.204695 | 70.276016 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.048116 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.013366 |
    | 1997-08-05 17:00:00 |  0.0 | 24.0 |              1.0 |                      1.0 |      273.614142 |             48.1 |       4.5 |              1018.0 | 0.0 | 0.0 | 17.654167 |             77.008333 |                  13.7 |                     15.470733 |           286.824521 |  0.0 | 24.0 | 17.654167 |    3.607161 |         2.448193 |           3.607161 |          4.5 |                29.886439 |                     20.226575 |                          0.0 |                      1.793476 |                           1.275734 |            14.375377 |                 15.576148 |    1003.624623 |   1.187075 | -2.9 |  18.37179 | -33.769262 |            207.946748 |                207.946748 |                 217.986636 |                 39.564612 |                      0.0 |   168.382136 |              0.0 |        178.422024 |              31.38525 |                 100.0 |               113.786144 |              109.455777 |  0.0 |  0.0 |  0.559369 |       0.0 | 0.275457 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.046873 |  0.01302 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8984.073957 | 70.000559 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.046873 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.01302 |
    | 1997-08-05 18:00:00 |  0.0 | 23.7 |              1.0 |                      1.0 |      133.368106 |             49.4 |       4.1 |              1018.0 | 0.0 | 0.0 | 17.770833 |             76.570833 |                  13.7 |                     15.470733 |           287.024163 |  0.0 | 23.7 | 17.770833 |    3.286524 |         2.444853 |           3.286524 |          4.1 |                29.352596 |                      20.37589 |                          0.0 |                      1.765541 |                           1.283961 |            14.500182 |                 15.601989 |    1003.499818 |   1.188219 | -2.9 | 18.742221 | -29.746675 |             101.35976 |                 101.35976 |                 218.138364 |                 39.572333 |                      0.0 |    61.787427 |              0.0 |        178.566031 |             34.447226 |                 100.0 |               114.184306 |              109.718191 |  0.0 |  0.0 |  0.406457 |       0.0 | 0.207398 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.045662 | 0.012684 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 9010.920632 | 69.793161 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.045662 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.012684 |
    | 1997-08-05 19:00:00 |  0.0 | 22.8 |              1.0 |                      1.0 |       40.326503 |             57.0 |       3.8 |              1018.0 | 0.0 | 0.0 |      17.9 |             76.191667 |                  13.7 |                     15.470733 |           286.951409 |  0.0 | 22.8 |      17.9 |    3.046047 |         2.451533 |           3.046047 |          3.8 |                27.800581 |                     20.542326 |                          0.0 |                      1.683922 |                           1.293123 |            15.846331 |                 15.651541 |    1002.153669 |   1.191234 | -2.9 | 19.064381 | -22.413715 |             30.648142 |                 30.648142 |                 218.083071 |                 39.521285 |                      0.0 |    -8.873143 |              0.0 |        178.561786 |             37.166744 |                 100.0 |               114.484549 |               109.91569 |  0.0 |  0.0 |  0.257526 |       0.0 | 0.133656 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.044483 | 0.012356 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 9030.434346 | 69.659505 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.044483 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.012356 |
    | 1997-08-05 20:00:00 |  0.0 | 21.3 |              0.5 |                 0.072689 |        0.867122 |             65.9 |       2.5 |              1018.0 | 0.0 | 0.0 | 18.041667 |                 75.95 |                  13.9 |                     15.438712 |           286.953669 |  0.0 | 21.3 | 18.041667 |    2.003978 |         2.428153 |           2.003978 |          2.5 |                 25.37251 |                     20.726234 |                          0.0 |                       1.55495 |                           1.303234 |            16.720484 |                 15.741575 |    1001.279516 |   1.196911 | -2.9 | 19.298545 | -12.008727 |              0.659013 |                  0.659013 |                 218.084788 |                 39.876809 |                      0.0 |   -39.217796 |              0.0 |        178.207979 |             56.493451 |                 100.0 |                114.67825 |               89.032161 |  0.0 |  0.0 |  0.102206 |       0.0 | 0.066112 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.043334 | 0.012037 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 9039.543073 | 69.593394 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.043334 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.012037 |
    | 1997-08-05 21:00:00 |  0.0 | 19.3 |              0.0 |                      0.0 |             0.0 |             73.0 |       1.9 |              1018.0 | 0.0 | 0.0 |   18.1875 |             75.633333 |                  13.9 |                     15.438712 |           286.953669 |  0.0 | 19.3 |   18.1875 |    1.523023 |         2.418134 |           1.523023 |          1.9 |                22.423949 |                     20.917052 |                          0.0 |                       1.39602 |                           1.313713 |            16.369483 |                 15.820264 |    1001.630517 |   1.205255 | -2.9 |  19.40785 |   0.647101 |                   0.0 |                       0.0 |                 218.084788 |                 39.753451 |                      0.0 |   -39.753451 |              0.0 |        178.331337 |             74.333488 |                 100.0 |               114.774124 |               87.719298 |  0.0 |  0.0 |  0.046254 |       0.0 |  0.03182 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.042215 | 0.011726 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 9035.995972 | 69.561574 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.042215 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.011726 |
    | 1997-08-05 22:00:00 |  0.0 | 18.3 |              0.0 |                      0.0 |             0.0 |             76.7 |       2.2 |              1018.0 | 0.0 | 0.0 | 18.333333 |                 75.25 |                  13.9 |                     15.438712 |           286.953669 |  0.0 | 18.3 | 18.333333 |    1.763501 |         2.418134 |           1.763501 |          2.2 |                21.065302 |                     21.109404 |                          0.0 |                      1.321845 |                           1.324263 |            16.157086 |                 15.884826 |    1001.842914 |   1.209486 | -2.9 | 19.365285 |    6.39171 |                   0.0 |                       0.0 |                 218.084788 |                 39.670785 |                      0.0 |   -39.670785 |              0.0 |        178.414003 |             64.197103 |                 100.0 |               114.820283 |               87.719298 |  0.0 |  0.0 |  0.042508 |       0.0 |  0.02777 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.041124 | 0.011423 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 9026.704262 | 69.533804 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.041124 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.011423 |
    | 1997-08-05 23:00:00 |  0.0 | 17.4 |              0.0 |                      0.0 |             0.0 |             81.8 |       2.7 |              1018.0 | 0.0 | 0.0 | 18.470833 |               74.9375 |                  13.9 |                     15.438712 |           286.953669 |  0.0 | 17.4 | 18.470833 |    2.164296 |         2.438173 |           2.164296 |          2.7 |                19.904589 |                     21.292177 |                          0.0 |                      1.257963 |                           1.334275 |            16.281953 |                 15.955825 |    1001.718047 |   1.213176 | -2.9 | 19.253784 |  11.122707 |                   0.0 |                       0.0 |                 218.084788 |                 39.564038 |                      0.0 |   -39.564038 |              0.0 |        178.520751 |             52.308751 |                 100.0 |               114.860575 |               87.719298 |  0.0 |  0.0 |  0.035577 |       0.0 | 0.021514 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.040062 | 0.011128 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 9012.681555 |  69.51229 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.040062 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.011128 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions_acker_summer))
0.0

.. _lland_v3_acker_heavy_rain_hourly:

acre (heavy rain)
-----------------

When comparing the soil moisture content (|BoWa|) on August 6th, 00:00, we notice only
minor differences between the daily (:ref:`lland_v3_acker_summer_daily`, 70.2 mm) and
the hourly (:ref:`lland_v3_acker_summer_daily`, 69.5) simulation results for a dry
situation.  However, when we compare the differences between our daily
(:ref:`lland_v3_acker_heavy_rain_daily`, 152.8 mm) and our hourly test for a wet
situation, we see much more pronounced differences, which are mainly due to the
prioritisation of throughfall (|NBes|) over interception evaporation (|EvI|):

.. integration-test::

    >>> inputs.nied.series = 20.0 / 24.0
    >>> control.negq(False)
    >>> test("lland_v3_acker_heavy_rain_hourly",
    ...      axis1=(inputs.nied, fluxes.qah), axis2=states.bowa,
    ...      use_conditions=conditions_acker_heavy_rain)
    |                date |     nied | teml | sunshineduration | possiblesunshineduration | globalradiation | relativehumidity | windspeed | atmosphericpressure |  qz | qzh |   temltag | dailyrelativehumidity | dailysunshineduration | dailypossiblesunshineduration | dailyglobalradiation |     nkor | tkor |   tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair |    g |        tz |         wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance |     nbes | sbes |      evpo |       evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg |     wada |      qdb |     qib1 | qib2 |      qbb | qkap |     qdgz |      qah |       qa |     inzp | wats | waes | esnow | taus |        ebdn |       bowa |    qdgz1 | qdgz2 |    qigz1 | qigz2 |     qbgz |    qdga1 | qdga2 |    qiga1 | qiga2 |     qbga | inlet |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-08-03 00:00:00 | 0.833333 | 16.9 |              0.0 |                      0.0 |             0.0 |             95.1 |       0.8 |              1015.0 | 0.0 | 0.0 | 17.279167 |             85.883333 |                   1.3 |                          15.7 |           136.579167 | 0.783333 | 16.9 | 17.279167 |    0.641273 |         1.605886 |           0.641273 |          0.8 |                19.284227 |                     19.753091 |                          0.0 |                      1.223615 |                           1.249589 |              18.3393 |                 16.964613 |       996.6607 |    1.21073 | -2.9 | 15.632162 |  -7.607027 |                   0.0 |                       0.0 |                 103.800167 |                 10.408237 |                      0.0 |   -10.408237 |              0.0 |          93.39193 |            176.542034 |                 100.0 |                70.326844 |               87.719298 | 0.083333 |  0.0 | -0.002953 | -0.002953 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.007403 |      0.0 |  0.0 |      0.0 |  0.0 | 0.007403 |  0.80139 | 0.222608 | 0.702953 |  0.0 |  0.0 |   0.0 |  nan | 8729.609427 | 102.562669 | 0.007403 |   0.0 |      0.0 |   0.0 |      0.0 |  0.80139 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.222608 |
    | 1997-08-03 01:00:00 | 0.833333 | 16.6 |              0.0 |                      0.0 |             0.0 |             94.9 |       0.8 |              1015.0 | 0.0 | 0.0 | 17.241667 |             85.991667 |                   1.3 |                          15.7 |           136.579167 | 0.783333 | 16.6 | 17.241667 |    0.641273 |         1.603439 |           0.641273 |          0.8 |                18.920184 |                      19.70628 |                          0.0 |                       1.20339 |                           1.246999 |            17.955254 |                 16.945759 |     997.044746 |   1.212158 | -2.9 | 15.688646 |  -5.468121 |                   0.0 |                       0.0 |                 103.800167 |                  10.41572 |                      0.0 |    -10.41572 |              0.0 |         93.384446 |            176.542034 |                 100.0 |                70.229104 |               87.719298 | 0.786286 |  0.0 | -0.002673 | -0.002673 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.786286 | 0.070171 |      0.0 |  0.0 |      0.0 |  0.0 | 0.070171 | 0.781695 | 0.217138 | 0.702673 |  0.0 |  0.0 |   0.0 |  nan | 8732.177548 | 103.278784 | 0.070171 |   0.0 |      0.0 |   0.0 |      0.0 | 0.781695 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.217138 |
    | 1997-08-03 02:00:00 | 0.833333 | 16.4 |              0.0 |                      0.0 |             0.0 |             95.9 |       0.8 |              1015.0 | 0.0 | 0.0 |    17.175 |             86.233333 |                   1.3 |                          15.7 |           136.579167 | 0.783333 | 16.4 |    17.175 |    0.641273 |         1.575992 |           0.641273 |          0.8 |                 18.68084 |                       19.6233 |                          0.0 |                      1.190065 |                           1.242407 |            17.914926 |                 16.921826 |     997.085074 |   1.213014 | -2.9 | 15.719464 |  -4.083217 |                   0.0 |                       0.0 |                 103.800167 |                 10.421627 |                      0.0 |   -10.421627 |              0.0 |         93.378539 |            176.542034 |                 100.0 |                69.307997 |               87.719298 | 0.786007 |  0.0 | -0.004096 | -0.004096 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.786007 | 0.070721 |      0.0 |  0.0 |      0.0 |  0.0 | 0.070721 | 0.763323 | 0.212034 | 0.704096 |  0.0 |  0.0 |   0.0 |  nan | 8733.360765 |  103.99407 | 0.070721 |   0.0 |      0.0 |   0.0 |      0.0 | 0.763323 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.212034 |
    | 1997-08-03 03:00:00 | 0.833333 | 16.3 |              0.0 |                      0.0 |             0.0 |             96.7 |       0.8 |              1015.0 | 0.0 | 0.0 |   17.0625 |             86.708333 |                   1.3 |                          15.5 |           136.579167 | 0.783333 | 16.3 |   17.0625 |    0.641273 |         1.548545 |           0.641273 |          0.8 |                18.562165 |                     19.483964 |                          0.0 |                      1.183449 |                            1.23469 |            17.949613 |                  16.89422 |     997.050387 |   1.213417 | -2.9 | 15.733663 |  -3.398025 |                   0.0 |                       0.0 |                 103.800167 |                 10.455184 |                      0.0 |   -10.455184 |              0.0 |         93.344983 |            176.542034 |                 100.0 |                68.389204 |               87.719298 | 0.787429 |  0.0 | -0.005245 | -0.005245 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.787429 | 0.071426 |      0.0 |  0.0 |      0.0 |  0.0 | 0.071426 | 0.745442 | 0.207067 | 0.705245 |  0.0 |  0.0 |   0.0 |  nan |  8733.85879 | 104.710073 | 0.071426 |   0.0 |      0.0 |   0.0 |      0.0 | 0.745442 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.207067 |
    | 1997-08-03 04:00:00 | 0.833333 | 16.0 |              0.0 |                 0.429734 |        1.943686 |             97.2 |       0.8 |              1015.0 | 0.0 | 0.0 | 16.908333 |             87.366667 |                   1.3 |                     14.929734 |            135.50182 | 0.783333 | 16.0 | 16.908333 |    0.641273 |         1.504432 |           0.641273 |          0.8 |                18.210086 |                     19.294427 |                          0.0 |                      1.163788 |                           1.224181 |            17.700204 |                 16.856897 |     997.299796 |    1.21479 | -2.9 | 15.739639 |  -1.562167 |              1.477201 |                  1.477201 |                 102.981383 |                 10.555242 |                      0.0 |    -9.078041 |              0.0 |         92.426141 |            176.542034 |                 100.0 |                67.470711 |               81.462042 | 0.788579 |  0.0 | -0.004905 | -0.004905 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.788579 |  0.07211 |      0.0 |  0.0 |      0.0 |  0.0 |  0.07211 |  0.72804 | 0.202233 | 0.704905 |  0.0 |  0.0 |   0.0 |  nan | 8732.520957 | 105.426542 |  0.07211 |   0.0 |      0.0 |   0.0 |      0.0 |  0.72804 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.202233 |
    | 1997-08-03 05:00:00 | 0.833333 | 15.9 |              0.0 |                      1.0 |       21.932442 |             97.5 |       0.6 |              1015.0 | 0.0 | 0.0 | 16.729167 |             88.204167 |                   1.3 |                     14.929734 |           134.099005 | 0.783333 | 15.9 | 16.729167 |    0.480955 |         1.453638 |           0.480955 |          0.6 |                18.094032 |                     19.076181 |                          0.0 |                      1.157296 |                           1.212063 |            17.641681 |                 16.825987 |     997.358319 |   1.215237 | -2.9 | 15.723585 |  -1.058491 |             16.668656 |                 16.668656 |                 101.915244 |                 10.545193 |                      0.0 |     6.123463 |              0.0 |         91.370051 |            235.389378 |                 100.0 |                66.552817 |               73.619139 | 0.788239 |  0.0 |  0.004992 |  0.004992 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.788239 |  0.07266 |      0.0 |  0.0 |      0.0 |  0.0 |  0.07266 | 0.711104 | 0.197529 | 0.695008 |  0.0 |  0.0 |   0.0 |  nan | 8730.679448 |  106.14212 |  0.07266 |   0.0 |      0.0 |   0.0 |      0.0 | 0.711104 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.197529 |
    | 1997-08-03 06:00:00 | 0.833333 | 16.0 |              0.0 |                      1.0 |       57.256186 |             97.7 |       0.9 |              1015.0 | 0.0 | 0.0 | 16.533333 |             89.191667 |                   1.3 |                     14.929734 |           131.855513 | 0.783333 | 16.0 | 16.533333 |    0.721432 |         1.392031 |           0.721432 |          0.9 |                18.210086 |                     18.840106 |                          0.0 |                      1.163788 |                           1.198935 |            17.791254 |                 16.803805 |     997.208746 |   1.214748 | -2.9 | 15.701487 |   -1.79108 |             43.514701 |                 43.514701 |                  100.21019 |                  10.52516 |                      0.0 |    32.989541 |              0.0 |          89.68503 |            156.926252 |                 100.0 |                65.637228 |               72.817981 | 0.778341 |  0.0 |  0.026203 |  0.026203 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.778341 | 0.072319 |      0.0 |  0.0 |      0.0 |  0.0 | 0.072319 | 0.694608 | 0.192947 | 0.673797 |  0.0 |  0.0 |   0.0 |  nan | 8729.570528 | 106.848143 | 0.072319 |   0.0 |      0.0 |   0.0 |      0.0 | 0.694608 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.192947 |
    | 1997-08-03 07:00:00 | 0.833333 | 16.6 |              0.0 |                      1.0 |      109.332844 |             97.4 |       0.9 |              1015.0 | 0.0 | 0.0 |    16.375 |                90.125 |                   1.3 |                     14.929734 |           129.465215 | 0.783333 | 16.6 |    16.375 |    0.721432 |         1.334591 |           0.721432 |          0.9 |                18.920184 |                     18.651109 |                          0.0 |                       1.20339 |                           1.188408 |            18.428259 |                 16.809312 |     996.571741 |   1.211943 | -2.9 |  15.68818 |  -5.470922 |             83.092962 |                 83.092962 |                  98.393563 |                  10.49073 |                      0.0 |    72.602232 |              0.0 |         87.902834 |            156.926252 |                 100.0 |                64.734984 |               72.023917 |  0.75713 |  0.0 |  0.057366 |  0.057366 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.75713 | 0.070894 |      0.0 |  0.0 |      0.0 |  0.0 | 0.070894 | 0.678515 | 0.188476 | 0.642634 |  0.0 |  0.0 |   0.0 |  nan |  8732.14145 |  107.53438 | 0.070894 |   0.0 |      0.0 |   0.0 |      0.0 | 0.678515 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.188476 |
    | 1997-08-03 08:00:00 | 0.833333 | 17.4 |              0.0 |                      1.0 |      170.949153 |             96.8 |       0.9 |              1016.0 | 0.0 | 0.0 | 16.216667 |                91.275 |                   1.3 |                     14.929734 |           127.329763 | 0.783333 | 17.4 | 16.216667 |    0.721432 |         1.260484 |           0.721432 |          0.9 |                19.904589 |                     18.463773 |                          0.0 |                      1.257963 |                           1.177959 |            19.267642 |                 16.852809 |     996.732358 |   1.209425 | -2.9 | 15.719031 | -10.085816 |            129.921356 |                129.921356 |                   96.77062 |                 10.426918 |                      0.0 |   119.494439 |              0.0 |         86.343702 |            156.926252 |                 100.0 |                63.859065 |               71.248633 | 0.725967 |  0.0 |  0.096008 |  0.096008 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.725967 | 0.068482 |      0.0 |  0.0 |      0.0 |  0.0 | 0.068482 | 0.662789 | 0.184108 | 0.603992 |  0.0 |  0.0 |   0.0 |  nan | 8739.327265 | 108.191865 | 0.068482 |   0.0 |      0.0 |   0.0 |      0.0 | 0.662789 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.184108 |
    | 1997-08-03 09:00:00 | 0.833333 | 19.0 |              0.2 |                      1.0 |      311.762625 |             86.1 |       1.3 |              1016.0 | 0.0 | 0.0 |   16.1125 |             91.991667 |                   1.5 |                     14.929734 |           129.903206 | 0.783333 | 19.0 |   16.1125 |    1.042069 |         1.203904 |           1.042069 |          1.3 |                22.008543 |                     18.341425 |                          0.0 |                      1.373407 |                           1.171128 |            18.949356 |                 16.872582 |     997.050644 |   1.202945 | -2.9 | 15.805261 | -19.168437 |            236.939595 |                236.939595 |                  98.726436 |                 10.804921 |                      0.0 |   226.134674 |              0.0 |         87.921515 |            108.641252 |                 100.0 |                63.020785 |               70.502581 | 0.687326 |  0.0 |  0.222649 |  0.222649 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.687326 | 0.065294 |      0.0 |  0.0 |      0.0 |  0.0 | 0.065294 | 0.647396 | 0.179832 | 0.477351 |  0.0 |  0.0 |   0.0 |  nan | 8755.595702 | 108.813897 | 0.065294 |   0.0 |      0.0 |   0.0 |      0.0 | 0.647396 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.179832 |
    | 1997-08-03 10:00:00 | 0.833333 | 20.3 |              0.5 |                      1.0 |        501.5833 |             76.8 |       1.5 |              1016.0 | 0.0 | 0.0 | 16.083333 |             92.241667 |                   1.7 |                     14.929734 |           132.285843 | 0.783333 | 20.3 | 16.083333 |    1.202387 |         1.149836 |           1.202387 |          1.5 |                23.858503 |                     18.307295 |                          0.0 |                      1.473678 |                           1.169221 |             18.32333 |                 16.886954 |      997.67667 |   1.197896 | -2.9 | 16.000482 | -25.797109 |            381.203308 |                381.203308 |                 100.537241 |                 11.199993 |                      0.0 |   370.003315 |              0.0 |         89.337248 |             94.155751 |                 100.0 |                62.228541 |               69.793799 | 0.560684 |  0.0 |  0.389789 |  0.389789 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.560684 | 0.053594 |      0.0 |  0.0 |      0.0 |  0.0 | 0.053594 | 0.632208 | 0.175613 | 0.310211 |  0.0 |  0.0 |   0.0 |  nan | 8778.492812 | 109.320987 | 0.053594 |   0.0 |      0.0 |   0.0 |      0.0 | 0.632208 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.175613 |
    | 1997-08-03 11:00:00 | 0.833333 | 21.4 |              0.7 |                      1.0 |      615.018728 |             71.8 |       1.2 |              1016.0 | 0.0 | 0.0 |    16.125 |             92.104167 |                   2.2 |                     14.929734 |           135.919957 | 0.783333 | 21.4 |    16.125 |    0.961909 |         1.089916 |           0.961909 |          1.2 |                25.528421 |                     18.356069 |                          0.0 |                      1.563281 |                           1.171946 |            18.329406 |                 16.906704 |     997.670594 |    1.19342 | -2.9 | 16.275247 | -30.748518 |            467.414233 |                467.414233 |                 103.299167 |                 12.222123 |                      0.0 |    455.19211 |              0.0 |         91.077044 |            117.694689 |                 100.0 |                61.583277 |               69.213837 | 0.393544 |  0.0 |   0.46611 |   0.46611 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.393544 | 0.037795 |      0.0 |  0.0 |      0.0 |  0.0 | 0.037795 | 0.617057 | 0.171405 |  0.23389 |  0.0 |  0.0 |   0.0 |  nan | 8806.341329 | 109.676737 | 0.037795 |   0.0 |      0.0 |   0.0 |      0.0 | 0.617057 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.171405 |
    | 1997-08-03 12:00:00 | 0.833333 | 21.3 |              0.8 |                      1.0 |      626.544325 |             67.5 |       1.3 |              1016.0 | 0.0 | 0.0 | 16.204167 |             91.729167 |                   2.8 |                     14.929734 |           145.821804 | 0.783333 | 21.3 | 16.204167 |    1.042069 |         1.037502 |           1.042069 |          1.3 |                 25.37251 |                     18.449053 |                          0.0 |                       1.55495 |                           1.177138 |            17.126444 |                 16.923163 |     998.873556 |   1.194363 | -2.9 | 16.609429 | -28.143424 |            476.173687 |                476.173687 |                 110.824571 |                 13.461134 |                      0.0 |   462.712553 |              0.0 |         97.363437 |            108.641252 |                 100.0 |                61.130903 |               68.805802 | 0.317224 |  0.0 |  0.490849 |  0.490849 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.317224 | 0.030572 |      0.0 |  0.0 |      0.0 |  0.0 | 0.030572 |    0.602 | 0.167222 | 0.209151 |  0.0 |  0.0 |   0.0 |  nan | 8831.584754 | 109.963389 | 0.030572 |   0.0 |      0.0 |   0.0 |      0.0 |    0.602 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.167222 |
    | 1997-08-03 13:00:00 | 0.833333 | 21.8 |              0.5 |                      1.0 |      496.133417 |             66.1 |       1.5 |              1016.0 | 0.0 | 0.0 | 16.329167 |             91.104167 |                   3.2 |                     14.929734 |           152.606529 | 0.783333 | 21.8 | 16.329167 |    1.202387 |         1.012602 |           1.202387 |          1.5 |                26.160453 |                      18.59671 |                          0.0 |                      1.596982 |                           1.185375 |             17.29206 |                 16.942378 |      998.70794 |   1.192265 | -2.9 |  16.91235 | -29.325898 |            377.061397 |                377.061397 |                 115.980962 |                 14.297428 |                      0.0 |   362.763969 |              0.0 |        101.683534 |             94.155751 |                 100.0 |                60.766577 |               68.476317 | 0.292485 |  0.0 |  0.424547 |  0.424547 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.292485 | 0.028272 |      0.0 |  0.0 |      0.0 |  0.0 | 0.028272 |  0.58721 | 0.163114 | 0.275453 |  0.0 |  0.0 |   0.0 |  nan | 8858.010651 | 110.227602 | 0.028272 |   0.0 |      0.0 |   0.0 |      0.0 |  0.58721 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.163114 |
    | 1997-08-03 14:00:00 | 0.833333 | 22.9 |              0.4 |                      1.0 |      419.520994 |             63.4 |       1.9 |              1016.0 | 0.0 | 0.0 | 16.545833 |             90.058333 |                   3.4 |                     14.929734 |           157.353237 | 0.783333 | 22.9 | 16.545833 |    1.523023 |         0.984394 |           1.523023 |          1.9 |                27.969419 |                     18.855098 |                          0.0 |                      1.692831 |                           1.199769 |            17.732611 |                 16.980587 |     998.267389 |   1.187639 | -2.9 | 17.229461 | -34.023233 |            318.835956 |                318.835956 |                  119.58846 |                 14.726001 |                      0.0 |   304.109955 |              0.0 |         104.86246 |             74.333488 |                 100.0 |                60.430915 |               68.172068 | 0.358786 |  0.0 |  0.415176 |  0.415176 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.358786 | 0.034791 |      0.0 |  0.0 |      0.0 |  0.0 | 0.034791 | 0.572857 | 0.159127 | 0.284824 |  0.0 |  0.0 |   0.0 |  nan | 8889.133884 | 110.551596 | 0.034791 |   0.0 |      0.0 |   0.0 |      0.0 | 0.572857 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.159127 |
    | 1997-08-03 15:00:00 | 0.833333 | 22.7 |              0.5 |                      1.0 |      387.887353 |             62.4 |       1.9 |              1016.0 | 0.0 | 0.0 | 16.816667 |             88.816667 |                   3.8 |                     14.929734 |           164.256877 | 0.783333 | 22.7 | 16.816667 |    1.523023 |         0.968687 |           1.523023 |          1.9 |                27.632633 |                     19.182495 |                          0.0 |                      1.675052 |                           1.217969 |            17.242763 |                 17.037252 |     998.757237 |    1.18866 | -2.9 |  17.60294 |  -30.58236 |            294.794388 |                294.794388 |                 124.835227 |                 15.562805 |                      0.0 |   279.231583 |              0.0 |        109.272421 |             74.333488 |                 100.0 |                60.019492 |               67.798243 | 0.368157 |  0.0 |  0.393194 |  0.393194 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.368157 | 0.035827 |      0.0 |  0.0 |      0.0 |  0.0 | 0.035827 | 0.558971 |  0.15527 | 0.306806 |  0.0 |  0.0 |   0.0 |  nan | 8916.816245 | 110.883927 | 0.035827 |   0.0 |      0.0 |   0.0 |      0.0 | 0.558971 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 |  0.15527 |
    | 1997-08-03 16:00:00 | 0.833333 | 22.5 |              0.5 |                      1.0 |      278.496872 |             61.1 |       2.3 |              1016.0 | 0.0 | 0.0 |   17.1375 |             87.333333 |                   4.1 |                     14.929734 |           170.073414 | 0.783333 | 22.5 |   17.1375 |     1.84366 |         0.991339 |            1.84366 |          2.3 |                27.299387 |                     19.576758 |                          0.0 |                      1.657431 |                            1.23983 |            16.679926 |                 17.097035 |     999.320074 |   1.189715 | -2.9 | 17.935128 |  -27.38923 |            211.657623 |                211.657623 |                 129.255794 |                 16.204965 |                      0.0 |   195.452657 |              0.0 |        113.050829 |             61.405925 |                 100.0 |                59.597696 |               67.413957 | 0.390139 |  0.0 |   0.34217 |   0.34217 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.390139 | 0.038106 |      0.0 |  0.0 |      0.0 |  0.0 | 0.038106 | 0.545488 | 0.151524 |  0.35783 |  0.0 |  0.0 |   0.0 |  nan | 8941.305475 | 111.235959 | 0.038106 |   0.0 |      0.0 |   0.0 |      0.0 | 0.545488 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.151524 |
    | 1997-08-03 17:00:00 | 0.833333 | 21.9 |              0.3 |                      1.0 |      137.138608 |             62.1 |       2.4 |              1016.0 | 0.0 | 0.0 |   17.4875 |                85.875 |                   4.4 |                     14.929734 |           173.470856 | 0.783333 | 21.9 |   17.4875 |    1.923819 |         1.017332 |           1.923819 |          2.4 |                26.320577 |                     20.014927 |                          0.0 |                      1.605502 |                           1.264057 |            16.345078 |                 17.187818 |     999.654922 |   1.192283 | -2.9 | 18.228999 | -22.026006 |            104.225342 |                104.225342 |                  131.83785 |                 16.818923 |                      0.0 |     87.40642 |              0.0 |        115.018928 |             58.847345 |                 100.0 |                59.151125 |               67.005952 | 0.441163 |  0.0 |  0.236219 |  0.236219 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.441163 | 0.043264 |      0.0 |  0.0 |      0.0 |  0.0 | 0.043264 | 0.532448 | 0.147902 | 0.463781 |  0.0 |  0.0 |   0.0 |  nan | 8960.431481 | 111.633858 | 0.043264 |   0.0 |      0.0 |   0.0 |      0.0 | 0.532448 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.147902 |
    | 1997-08-03 18:00:00 | 0.833333 | 21.4 |              0.1 |                      1.0 |       51.080714 |             67.0 |       2.5 |              1016.0 | 0.0 | 0.0 |   17.8375 |             84.620833 |                   4.5 |                     14.929734 |           174.440885 | 0.783333 | 21.4 |   17.8375 |    2.003978 |         1.054998 |           2.003978 |          2.5 |                25.528421 |                     20.461645 |                          0.0 |                      1.563281 |                           1.288683 |            17.104042 |                 17.314814 |     998.895958 |   1.193968 | -2.9 | 18.458511 | -17.648933 |             38.821343 |                 38.821343 |                 132.575073 |                 16.973018 |                      0.0 |    21.848325 |              0.0 |        115.602055 |             56.493451 |                 100.0 |                58.646654 |               66.543619 | 0.547114 |  0.0 |  0.156077 |  0.156077 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.547114 | 0.053912 |      0.0 |  0.0 |      0.0 |  0.0 | 0.053912 | 0.519951 | 0.144431 | 0.543923 |  0.0 |  0.0 |   0.0 |  nan | 8975.180414 | 112.127061 | 0.053912 |   0.0 |      0.0 |   0.0 |      0.0 | 0.519951 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.144431 |
    | 1997-08-03 19:00:00 | 0.833333 | 20.7 |              0.0 |                      1.0 |       13.632817 |             74.5 |       2.5 |              1016.0 | 0.0 | 0.0 | 18.170833 |                  83.7 |                   4.5 |                     15.429734 |           175.008919 | 0.783333 | 20.7 | 18.170833 |    2.003978 |         1.100997 |           2.003978 |          2.5 |                24.454368 |                     20.895167 |                          0.0 |                      1.505746 |                           1.312512 |            18.218504 |                 17.489255 |     997.781496 |   1.196313 | -2.9 | 18.635498 |  -12.38701 |             10.360941 |                 10.360941 |                 133.006779 |                 16.556776 |                      0.0 |    -6.195835 |              0.0 |        116.450003 |             56.493451 |                 100.0 |                58.021768 |               65.968819 | 0.627256 |  0.0 |   0.09611 |   0.09611 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.627256 |  0.06216 |      0.0 |  0.0 |      0.0 |  0.0 |  0.06216 | 0.508019 | 0.141116 |  0.60389 |  0.0 |  0.0 |   0.0 |  nan | 8984.667424 | 112.692157 |  0.06216 |   0.0 |      0.0 |   0.0 |      0.0 | 0.508019 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.141116 |
    | 1997-08-03 20:00:00 | 0.833333 | 19.4 |              0.0 |                   0.1364 |        0.185944 |             81.2 |       2.2 |              1016.0 | 0.0 | 0.0 | 18.454167 |             83.066667 |                   4.5 |                     15.566134 |           175.016667 | 0.783333 | 19.4 | 18.454167 |    1.763501 |         1.120309 |           1.763501 |          2.2 |                22.563931 |                      21.26995 |                          0.0 |                      1.403627 |                           1.333058 |            18.321912 |                 17.668238 |     997.678088 |   1.201582 | -2.9 | 18.749342 |  -3.903945 |              0.141318 |                  0.141318 |                 133.012667 |                 16.343497 |                      0.0 |    -16.20218 |              0.0 |        116.669169 |             64.197103 |                 100.0 |                57.306347 |               83.796936 | 0.687223 |  0.0 |  0.049374 |  0.049374 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.687223 | 0.068533 |      0.0 |  0.0 |      0.0 |  0.0 | 0.068533 | 0.496585 |  0.13794 | 0.650626 |  0.0 |  0.0 |   0.0 |  nan |  8985.67137 | 113.310848 | 0.068533 |   0.0 |      0.0 |   0.0 |      0.0 | 0.496585 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 |  0.13794 |
    | 1997-08-03 21:00:00 | 0.833333 | 17.8 |              0.0 |                      0.0 |             0.0 |             86.9 |       1.7 |              1016.0 | 0.0 | 0.0 |     18.65 |               82.7375 |                   4.5 |                     15.566134 |           175.016667 | 0.783333 | 17.8 |     18.65 |    1.362705 |         1.127089 |           1.362705 |          1.7 |                20.413369 |                     21.532411 |                          0.0 |                      1.286025 |                           1.347418 |            17.739217 |                 17.815378 |     998.260783 |   1.208454 | -2.9 |  18.76139 |   5.768339 |                   0.0 |                       0.0 |                 133.012667 |                 16.221809 |                      0.0 |   -16.221809 |              0.0 |        116.790858 |             83.078604 |                 100.0 |                56.523742 |               87.719298 | 0.733959 |  0.0 |  0.019253 |  0.019253 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.733959 | 0.073692 |      0.0 |  0.0 |      0.0 |  0.0 | 0.073692 | 0.485595 | 0.134888 | 0.680747 |  0.0 |  0.0 |   0.0 |  nan | 8977.003031 | 113.971116 | 0.073692 |   0.0 |      0.0 |   0.0 |      0.0 | 0.485595 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.134888 |
    | 1997-08-03 22:00:00 | 0.833333 | 17.0 |              0.0 |                      0.0 |             0.0 |             90.1 |       1.7 |              1017.0 | 0.0 | 0.0 | 18.808333 |             82.554167 |                   4.5 |                     15.566134 |           175.016667 | 0.783333 | 17.0 | 18.808333 |    1.362705 |         1.142201 |           1.362705 |          1.7 |                19.406929 |                     21.746678 |                          0.0 |                      1.230421 |                           1.359123 |            17.485643 |                 17.952788 |     999.514357 |   1.213101 | -2.9 |  18.65737 |   9.944218 |                   0.0 |                       0.0 |                 133.012667 |                 16.101364 |                      0.0 |   -16.101364 |              0.0 |        116.911302 |             83.078604 |                 100.0 |                55.689302 |               87.719298 | 0.764081 |  0.0 |  0.010292 |  0.010292 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.764081 | 0.077264 |      0.0 |  0.0 |      0.0 |  0.0 | 0.077264 | 0.475002 | 0.131945 | 0.689708 |  0.0 |  0.0 |   0.0 |  nan | 8964.158813 | 114.657933 | 0.077264 |   0.0 |      0.0 |   0.0 |      0.0 | 0.475002 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.131945 |
    | 1997-08-03 23:00:00 | 0.833333 | 16.4 |              0.0 |                      0.0 |             0.0 |             90.9 |       2.3 |              1017.0 | 0.0 | 0.0 | 18.941667 |             82.379167 |                   4.5 |                     15.566134 |           175.016667 | 0.783333 | 16.4 | 18.941667 |     1.84366 |         1.185687 |            1.84366 |          2.3 |                 18.68084 |                     21.928555 |                          0.0 |                      1.190065 |                           1.369047 |            16.980884 |                 18.064561 |    1000.019116 |   1.215845 | -2.9 | 18.503239 |  12.619435 |                   0.0 |                       0.0 |                 133.012667 |                 16.005095 |                      0.0 |   -16.005095 |              0.0 |        117.007571 |             61.405925 |                 100.0 |                54.822121 |               87.719298 | 0.773041 |  0.0 |  0.014655 |  0.014655 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.773041 | 0.078739 |      0.0 |  0.0 |      0.0 |  0.0 | 0.078739 | 0.464747 | 0.129096 | 0.685345 |  0.0 |  0.0 |   0.0 |  nan | 8948.639378 | 115.352235 | 0.078739 |   0.0 |      0.0 |   0.0 |      0.0 | 0.464747 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.129096 |
    | 1997-08-04 00:00:00 | 0.833333 | 15.9 |              0.0 |                      0.0 |             0.0 |             88.0 |       2.3 |              1017.0 | 0.0 | 0.0 |      18.9 |             82.083333 |                   4.5 |                     15.566134 |           175.016667 | 0.783333 | 15.9 |      18.9 |     1.84366 |         1.235786 |            1.84366 |          2.3 |                18.094032 |                     21.871576 |                          0.0 |                      1.157296 |                           1.365939 |            15.922748 |                 17.952919 |    1001.077252 |    1.21843 | -2.9 | 18.317006 |  14.502035 |                   0.0 |                       0.0 |                 133.012667 |                 16.128293 |                      0.0 |   -16.128293 |              0.0 |        116.884374 |             61.405925 |                 100.0 |                53.946317 |               87.719298 | 0.768678 |  0.0 |  0.024014 |  0.024014 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.768678 | 0.078864 |      0.0 |  0.0 |      0.0 |  0.0 | 0.078864 | 0.454777 | 0.126327 | 0.675986 |  0.0 |  0.0 |   0.0 |  nan | 8931.237343 | 116.042049 | 0.078864 |   0.0 |      0.0 |   0.0 |      0.0 | 0.454777 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.126327 |
    | 1997-08-04 01:00:00 | 0.833333 | 15.4 |              0.0 |                      0.0 |             0.0 |             89.2 |       2.2 |              1016.0 | 0.0 | 0.0 |     18.85 |             81.845833 |                   4.5 |                     15.566134 |           175.016667 | 0.783333 | 15.4 |     18.85 |    1.763501 |         1.282546 |           1.763501 |          2.2 |                17.523416 |                     21.803372 |                          0.0 |                      1.125295 |                           1.362218 |            15.630887 |                 17.845152 |    1000.369113 |   1.219467 | -2.9 | 18.108181 |  16.249089 |                   0.0 |                       0.0 |                 133.012667 |                 16.244859 |                      0.0 |   -16.244859 |              0.0 |        116.767808 |             64.197103 |                 100.0 |                 53.07698 |               87.719298 | 0.759319 |  0.0 |  0.018207 |  0.018207 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.759319 | 0.078462 |      0.0 |  0.0 |      0.0 |  0.0 | 0.078462 | 0.445062 | 0.123628 | 0.681793 |  0.0 |  0.0 |   0.0 |  nan | 8912.088255 | 116.722906 | 0.078462 |   0.0 |      0.0 |   0.0 |      0.0 | 0.445062 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.123628 |
    | 1997-08-04 02:00:00 | 0.833333 | 15.2 |              0.0 |                      0.0 |             0.0 |             88.8 |       2.3 |              1016.0 | 0.0 | 0.0 |      18.8 |                 81.55 |                   4.5 |                     15.566134 |           175.016667 | 0.783333 | 15.2 |      18.8 |     1.84366 |         1.332645 |            1.84366 |          2.3 |                17.299618 |                     21.735354 |                          0.0 |                      1.112706 |                           1.358505 |            15.362061 |                 17.725181 |    1000.637939 |   1.220436 | -2.9 | 17.878392 |  16.070354 |                   0.0 |                       0.0 |                 133.012667 |                 16.376926 |                      0.0 |   -16.376926 |              0.0 |        116.635741 |             61.405925 |                 100.0 |                52.219703 |               87.719298 | 0.765127 |  0.0 |  0.020574 |  0.020574 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.765127 | 0.079624 |      0.0 |  0.0 |      0.0 |  0.0 | 0.079624 | 0.435608 | 0.121002 | 0.679426 |  0.0 |  0.0 |   0.0 |  nan |   8893.1179 | 117.408409 | 0.079624 |   0.0 |      0.0 |   0.0 |      0.0 | 0.435608 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.121002 |
    | 1997-08-04 03:00:00 | 0.833333 | 14.9 |              0.0 |                      0.0 |             0.0 |             89.7 |       2.0 |              1016.0 | 0.0 | 0.0 | 18.741667 |             81.258333 |                   4.5 |                     15.566134 |           175.016667 | 0.783333 | 14.9 | 18.741667 |    1.603182 |         1.372725 |           1.603182 |          2.0 |                16.968612 |                     21.656234 |                          0.0 |                      1.094046 |                           1.354184 |            15.220845 |                 17.597495 |    1000.779155 |   1.221771 | -2.9 | 17.650748 |  16.504489 |                   0.0 |                       0.0 |                 133.012667 |                 16.516685 |                      0.0 |   -16.516685 |              0.0 |        116.495982 |             70.616813 |                 100.0 |                 51.35734 |               87.719298 | 0.762759 |  0.0 |  0.013655 |  0.013655 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.762759 | 0.079941 |      0.0 |  0.0 |      0.0 |  0.0 | 0.079941 | 0.426416 | 0.118449 | 0.686345 |  0.0 |  0.0 |   0.0 |  nan | 8873.713411 | 118.091227 | 0.079941 |   0.0 |      0.0 |   0.0 |      0.0 | 0.426416 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.118449 |
    | 1997-08-04 04:00:00 | 0.833333 | 14.3 |              0.0 |                 0.398044 |        1.590517 |             93.0 |       2.3 |              1016.0 | 0.0 | 0.0 | 18.670833 |             81.083333 |                   4.5 |                     15.534444 |           175.001952 | 0.783333 | 14.3 | 18.670833 |     1.84366 |         1.422824 |            1.84366 |          2.3 |                16.323194 |                     21.560498 |                          0.0 |                      1.057522 |                           1.348953 |             15.18057 |                  17.48197 |     1000.81943 |    1.22434 | -2.9 | 17.417894 |  18.707366 |              1.208793 |                  1.208793 |                 133.001483 |                  16.65652 |                      0.0 |   -15.447727 |              0.0 |        116.344963 |             61.405925 |                 100.0 |                50.499098 |               73.395613 | 0.769679 |  0.0 |  0.007264 |  0.007264 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.769679 | 0.081237 |      0.0 |  0.0 |      0.0 |  0.0 | 0.081237 | 0.417483 | 0.115968 | 0.692736 |  0.0 |  0.0 |   0.0 |  nan | 8852.106046 | 118.779668 | 0.081237 |   0.0 |      0.0 |   0.0 |      0.0 | 0.417483 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.115968 |
    | 1997-08-04 05:00:00 | 0.833333 | 14.1 |              0.0 |                      1.0 |       19.950903 |             93.6 |       2.5 |              1017.0 | 0.0 | 0.0 | 18.595833 |             80.920833 |                   4.5 |                     15.534444 |           174.919387 | 0.783333 | 14.1 | 18.595833 |    2.003978 |         1.486284 |           2.003978 |          2.5 |                16.112886 |                     21.459534 |                          0.0 |                      1.045579 |                           1.343433 |            15.081661 |                 17.365233 |    1001.918339 |   1.226451 | -2.9 | 17.158606 |  18.351635 |             15.162686 |                 15.162686 |                 132.938735 |                 16.779096 |                      0.0 |     -1.61641 |              0.0 |        116.159638 |             56.493451 |                 100.0 |                49.634525 |                58.02059 | 0.776069 |  0.0 |  0.018172 |  0.018172 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.776069 | 0.082493 |      0.0 |  0.0 |      0.0 |  0.0 | 0.082493 | 0.408814 | 0.113559 | 0.681828 |  0.0 |  0.0 |   0.0 |  nan |  8830.85441 | 119.473244 | 0.082493 |   0.0 |      0.0 |   0.0 |      0.0 | 0.408814 |   0.0 |      0.0 |   0.0 |      0.0 |   0.0 | 0.113559 |
    | 1997-08-04 06:00:00 | 0.833333 | 14.3 |              0.1 |                      1.0 |       62.985775 |             94.6 |       2.4 |              1017.0 | 0.0 | 0.0 |    18.525 |             80.791667 |                   4.6 |                     15.534444 |            175.15812 | 0.783333 | 14.3 |    18.525 |    1.923819 |         1.536383 |           1.923819 |          2.4 |                16.323194 |                     21.364558 |                          0.0 |                      1.057522 |                           1.338237 |            15.441741 |                 17.260782 |    1001.558259 |   1.225432 | -2.9 | 16.903586 |  15.621518 |             47.869189 |                 47.869189 |                 133.120171 |                 17.089561 |                      0.0 |    30.779628 |              0.0 |         116.03061 |             58.847345 |                 100.0 |                 48.85611 |               57.260179 | 0.765161 |  0.0 |   0.04094 |   0.04094 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.765161 | 0.081907 | 0.003897 |  0.0 | 0.000003 |  0.0 | 0.081907 | 0.400403 | 0.111223 |  0.65906 |  0.0 |  0.0 |   0.0 |  nan | 8812.332893 | 120.152598 | 0.081907 |   0.0 | 0.003897 |   0.0 | 0.000003 | 0.400377 |   0.0 | 0.000025 |   0.0 |      0.0 |   0.0 | 0.111223 |
    | 1997-08-04 07:00:00 | 0.833333 | 15.4 |              0.9 |                      1.0 |      285.776389 |             93.3 |       2.5 |              1017.0 | 0.0 | 0.0 |    18.475 |             80.620833 |                   5.5 |                     15.534444 |           182.509935 | 0.783333 | 15.4 |    18.475 |    2.003978 |         1.589823 |           2.003978 |          2.5 |                17.523416 |                     21.297737 |                          0.0 |                      1.125295 |                            1.33458 |            16.349347 |                 17.170413 |    1000.650653 |   1.220347 | -2.9 | 16.681328 |   7.687968 |            217.190056 |                217.190056 |                  138.70755 |                 19.012403 |                      0.0 |   198.177653 |              0.0 |        119.695148 |             56.493451 |                 100.0 |                 48.85611 |               57.260179 | 0.742393 |  0.0 |  0.189733 |  0.189733 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.742393 | 0.080011 |  0.00392 |  0.0 | 0.000031 |  0.0 | 0.080011 | 0.392202 | 0.108945 | 0.510267 |  0.0 |  0.0 |   0.0 |  nan | 8801.744925 | 120.811029 | 0.080011 |   0.0 |  0.00392 |   0.0 | 0.000031 | 0.392126 |   0.0 | 0.000076 |   0.0 |      0.0 |   0.0 | 0.108945 |
    | 1997-08-04 08:00:00 | 0.833333 | 16.8 |              1.0 |                      1.0 |      475.225247 |             86.9 |       2.5 |              1017.0 | 0.0 | 0.0 |     18.45 |             80.208333 |                   6.5 |                     15.534444 |           195.188105 | 0.783333 | 16.8 |     18.45 |    2.003978 |         1.643262 |           2.003978 |          2.5 |                19.162205 |                     21.264396 |                          0.0 |                      1.216842 |                           1.332754 |            16.651956 |                 17.055817 |    1000.348044 |   1.214317 | -2.9 | 16.554272 |  -1.474365 |            361.171188 |                361.171188 |                  148.34296 |                 21.208047 |                      0.0 |   339.963141 |              0.0 |        127.134913 |             56.493451 |                 100.0 |                 48.85611 |               57.260179 |   0.5936 |  0.0 |  0.343753 |  0.343753 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |   0.5936 | 0.064359 | 0.003941 |  0.0 | 0.000059 |  0.0 | 0.064359 | 0.383987 | 0.106663 | 0.356247 |  0.0 |  0.0 |   0.0 |  nan |  8800.31929 |  121.33627 | 0.064359 |   0.0 | 0.003941 |   0.0 | 0.000059 | 0.383861 |   0.0 | 0.000126 |   0.0 |      0.0 |   0.0 | 0.106663 |
    | 1997-08-04 09:00:00 | 0.833333 | 18.2 |              1.0 |                      1.0 |      614.006719 |             78.9 |       2.9 |              1017.0 | 0.0 | 0.0 | 18.416667 |             79.908333 |                   7.3 |                     15.534444 |           207.781609 | 0.783333 | 18.2 | 18.416667 |    2.324615 |         1.696701 |           2.324615 |          2.9 |                20.933479 |                     21.220011 |                          0.0 |                      1.314615 |                           1.330323 |            16.516515 |                 16.956557 |    1000.483485 |   1.208543 | -2.9 | 16.537165 |  -9.977011 |            466.645107 |                466.645107 |                 157.914023 |                 22.996116 |                      0.0 |   443.648991 |              0.0 |        134.917907 |             48.701251 |                 100.0 |                 48.85611 |               57.260179 | 0.439581 |  0.0 |  0.494205 |  0.494205 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.439581 |  0.04788 | 0.003958 |  0.0 | 0.000081 |  0.0 |  0.04788 |  0.37557 | 0.104325 | 0.205795 |  0.0 |  0.0 |   0.0 |  nan | 8807.396301 | 121.723932 |  0.04788 |   0.0 | 0.003958 |   0.0 | 0.000081 | 0.375394 |   0.0 | 0.000176 |   0.0 |      0.0 |   0.0 | 0.104325 |
    | 1997-08-04 10:00:00 | 0.833333 | 19.5 |              0.9 |                      1.0 |      643.806508 |             75.8 |       3.1 |              1017.0 | 0.0 | 0.0 | 18.383333 |             79.866667 |                   7.7 |                     15.534444 |           213.707576 | 0.783333 | 19.5 | 18.383333 |    2.484933 |         1.750141 |           2.484933 |          3.1 |                22.704676 |                     21.175708 |                          0.0 |                      1.411269 |                           1.327896 |            17.210144 |                 16.912332 |     999.789856 |   1.202862 | -2.9 | 16.622089 | -17.267466 |            489.292946 |                489.292946 |                 162.417758 |                 23.882018 |                      0.0 |   465.410928 |              0.0 |         138.53574 |             45.559235 |                 100.0 |                 48.85611 |               57.260179 | 0.289128 |  0.0 |  0.550473 |  0.550473 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.289128 | 0.031594 | 0.003971 |  0.0 | 0.000097 |  0.0 | 0.031594 | 0.366948 |  0.10193 | 0.149527 |  0.0 |  0.0 |   0.0 |  nan | 8821.763767 | 121.977399 | 0.031594 |   0.0 | 0.003971 |   0.0 | 0.000097 | 0.366723 |   0.0 | 0.000225 |   0.0 |      0.0 |   0.0 |  0.10193 |
    | 1997-08-04 11:00:00 | 0.833333 | 20.3 |              0.8 |                      1.0 |      600.262981 |             73.4 |       3.3 |              1018.0 | 0.0 | 0.0 |   18.3375 |             79.933333 |                   7.8 |                     15.534444 |           213.092753 | 0.783333 | 20.3 |   18.3375 |    2.645251 |          1.82028 |           2.645251 |          3.3 |                23.858503 |                     21.114922 |                          0.0 |                      1.473678 |                           1.324565 |            17.512141 |                 16.877861 |    1000.487859 |   1.200635 | -2.9 | 16.794499 | -21.033009 |            456.199865 |                456.199865 |                 161.950493 |                   24.1295 |                      0.0 |   432.070365 |              0.0 |        137.820992 |             42.798069 |                 100.0 |                 48.85611 |               57.260179 |  0.23286 |  0.0 |  0.548409 |  0.548409 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.23286 | 0.025504 | 0.003979 |  0.0 | 0.000107 |  0.0 | 0.025504 | 0.358261 | 0.099517 | 0.151591 |  0.0 |  0.0 |   0.0 |  nan | 8839.896776 | 122.180668 | 0.025504 |   0.0 | 0.003979 |   0.0 | 0.000107 | 0.357987 |   0.0 | 0.000274 |   0.0 |      0.0 |   0.0 | 0.099517 |
    | 1997-08-04 12:00:00 | 0.833333 | 21.0 |              0.9 |                      1.0 |      620.514139 |             68.3 |       3.3 |              1018.0 | 0.0 | 0.0 |    18.325 |             79.966667 |                   7.9 |                     15.534444 |           212.841496 | 0.783333 | 21.0 |    18.325 |    2.645251 |         1.887079 |           2.645251 |          3.3 |                24.909749 |                     21.098371 |                          0.0 |                      1.530181 |                           1.323658 |            17.013358 |                 16.871664 |    1000.986642 |   1.198001 | -2.9 | 17.012095 | -23.927432 |            471.590746 |                471.590746 |                 161.759537 |                 24.341655 |                      0.0 |    447.24909 |              0.0 |        137.417882 |             42.798069 |                 100.0 |                 48.85611 |               57.260179 | 0.234924 |  0.0 |  0.597585 |  0.597585 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.234924 | 0.025783 | 0.003986 |  0.0 | 0.000116 |  0.0 | 0.025783 | 0.349725 | 0.097146 | 0.102415 |  0.0 |  0.0 |   0.0 |  nan | 8860.924208 | 122.385708 | 0.025783 |   0.0 | 0.003986 |   0.0 | 0.000116 | 0.349403 |   0.0 | 0.000322 |   0.0 |      0.0 |   0.0 | 0.097146 |
    | 1997-08-04 13:00:00 | 0.833333 | 21.8 |              0.8 |                      1.0 |      599.441386 |             61.3 |       3.2 |              1018.0 | 0.0 | 0.0 |    18.325 |             79.766667 |                   8.2 |                     15.534444 |           217.145994 | 0.783333 | 21.8 |    18.325 |    2.565092 |         1.943859 |           2.565092 |          3.2 |                26.160453 |                     21.098371 |                          0.0 |                      1.596982 |                           1.323658 |            16.036358 |                 16.829467 |    1001.963642 |   1.195188 | -2.9 | 17.264424 | -27.213457 |            455.575453 |                455.575453 |                 165.030956 |                 25.038979 |                      0.0 |   430.536475 |              0.0 |        139.991977 |             44.135508 |                 100.0 |                 48.85611 |               57.260179 | 0.185749 |  0.0 |  0.621828 |  0.621828 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.185749 | 0.020423 | 0.003992 |  0.0 | 0.000124 |  0.0 | 0.020423 | 0.341344 | 0.094818 | 0.078172 |  0.0 |  0.0 |   0.0 |  nan | 8885.237665 | 122.546917 | 0.020423 |   0.0 | 0.003992 |   0.0 | 0.000124 | 0.340974 |   0.0 |  0.00037 |   0.0 |      0.0 |   0.0 | 0.094818 |
    | 1997-08-04 14:00:00 | 0.833333 | 21.9 |              0.9 |                      1.0 |      613.551222 |             60.6 |       2.5 |              1018.0 | 0.0 | 0.0 | 18.283333 |                 79.65 |                   8.7 |                     15.534444 |           225.230587 | 0.783333 | 21.9 | 18.283333 |    2.003978 |         1.963899 |           2.003978 |          2.5 |                26.320577 |                     21.043281 |                          0.0 |                      1.605502 |                           1.320638 |             15.95027 |                 16.760973 |     1002.04973 |   1.194821 | -2.9 | 17.556185 | -26.062888 |            466.298929 |                466.298929 |                 171.175246 |                 26.187015 |                      0.0 |   440.111914 |              0.0 |        144.988231 |             56.493451 |                 100.0 |                 48.85611 |               57.260179 | 0.161505 |  0.0 |  0.589213 |  0.589213 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.161505 | 0.017784 | 0.003998 |  0.0 | 0.000131 |  0.0 | 0.017784 | 0.333076 | 0.092521 | 0.110787 |  0.0 |  0.0 |   0.0 |  nan | 8908.400553 | 122.686509 | 0.017784 |   0.0 | 0.003998 |   0.0 | 0.000131 |  0.33266 |   0.0 | 0.000417 |   0.0 |      0.0 |   0.0 | 0.092521 |
    | 1997-08-04 15:00:00 | 0.833333 | 21.8 |              0.9 |                      1.0 |      517.107047 |             58.8 |       2.9 |              1017.0 | 0.0 | 0.0 | 18.245833 |                  79.5 |                   9.1 |                     15.534444 |           230.614741 | 0.783333 | 21.8 | 18.245833 |    2.324615 |         1.997298 |           2.324615 |          2.9 |                26.160453 |                     20.993808 |                          0.0 |                      1.596982 |                           1.317924 |            15.382347 |                 16.690077 |    1001.617653 |   1.194298 | -2.9 |  17.83414 |  -23.79516 |            393.001356 |                393.001356 |                 175.267203 |                 27.142101 |                      0.0 |   365.859255 |              0.0 |        148.125102 |             48.701251 |                 100.0 |                 48.85611 |               57.260179 | 0.194121 |  0.0 |  0.551455 |  0.551455 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.194121 | 0.021409 | 0.004002 |  0.0 | 0.000137 |  0.0 | 0.021409 | 0.325036 | 0.090288 | 0.148545 |  0.0 |  0.0 |   0.0 |  nan | 8929.295714 | 122.855082 | 0.021409 |   0.0 | 0.004002 |   0.0 | 0.000137 | 0.324573 |   0.0 | 0.000463 |   0.0 |      0.0 |   0.0 | 0.090288 |
    | 1997-08-04 16:00:00 | 0.833333 | 22.2 |              0.9 |                      1.0 |      377.836617 |             57.2 |       3.6 |              1017.0 | 0.0 | 0.0 | 18.233333 |               79.3375 |                   9.5 |                     15.534444 |           234.753897 | 0.783333 | 22.2 | 18.233333 |    2.885728 |         2.040718 |           2.885728 |          3.6 |                26.806088 |                      20.97734 |                          0.0 |                      1.631293 |                           1.317021 |            15.333082 |                 16.642897 |    1001.666918 |   1.192703 | -2.9 | 18.084882 | -24.690709 |            287.155829 |                287.155829 |                 178.412962 |                 28.068928 |                      0.0 |   259.086901 |              0.0 |        150.344034 |             39.231563 |                 100.0 |                 48.85611 |               57.260179 | 0.231878 |  0.0 |  0.505795 |  0.505795 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.231878 |  0.02562 | 0.004008 |  0.0 | 0.000144 |  0.0 |  0.02562 | 0.317306 | 0.088141 | 0.194205 |  0.0 |  0.0 |   0.0 |  nan | 8951.086422 | 123.057187 |  0.02562 |   0.0 | 0.004008 |   0.0 | 0.000144 | 0.316797 |   0.0 | 0.000509 |   0.0 |      0.0 |   0.0 | 0.088141 |
    | 1997-08-04 17:00:00 | 0.833333 | 21.8 |              1.0 |                      1.0 |      252.204639 |             57.2 |       4.2 |              1017.0 | 0.0 | 0.0 | 18.229167 |             79.133333 |                  10.2 |                     15.534444 |           239.548315 | 0.783333 | 21.8 | 18.229167 |    3.366683 |         2.100837 |           3.366683 |          4.2 |                26.160453 |                     20.971853 |                          0.0 |                      1.596982 |                            1.31672 |            14.963779 |                 16.595726 |    1002.036221 |   1.194485 | -2.9 |  18.34637 | -20.721778 |            191.675526 |                191.675526 |                  182.05672 |                 29.636995 |                      0.0 |    162.03853 |              0.0 |        152.419724 |             33.627054 |                 100.0 |                 48.85611 |               57.260179 | 0.277538 |  0.0 |   0.44464 |   0.44464 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.277538 | 0.030734 | 0.004014 |  0.0 | 0.000152 |  0.0 | 0.030734 | 0.309897 | 0.086082 |  0.25536 |  0.0 |  0.0 |   0.0 |  nan |   8968.9082 | 123.299825 | 0.030734 |   0.0 | 0.004014 |   0.0 | 0.000152 | 0.309342 |   0.0 | 0.000555 |   0.0 |      0.0 |   0.0 | 0.086082 |
    | 1997-08-04 18:00:00 | 0.833333 | 20.9 |              1.0 |                      1.0 |      128.576714 |             59.9 |       4.2 |              1017.0 | 0.0 | 0.0 | 18.208333 |               78.8375 |                  11.1 |                     15.534444 |           242.777315 | 0.783333 | 20.9 | 18.208333 |    3.366683 |         2.157616 |           3.366683 |          4.2 |                 24.75714 |                     20.944437 |                          0.0 |                      1.521999 |                           1.315216 |            14.829527 |                  16.51207 |    1002.170473 |   1.198201 | -2.9 | 18.560232 |  -14.03861 |             97.718303 |                 97.718303 |                  184.51076 |                 31.712576 |                      0.0 |    66.005727 |              0.0 |        152.798184 |             33.627054 |                 100.0 |                 48.85611 |               57.260179 | 0.338693 |  0.0 |   0.32482 |   0.32482 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.338693 | 0.037607 | 0.004022 |  0.0 | 0.000162 |  0.0 | 0.037607 | 0.302834 | 0.084121 |  0.37518 |  0.0 |  0.0 |   0.0 |  nan | 8980.046809 | 123.596727 | 0.037607 |   0.0 | 0.004022 |   0.0 | 0.000162 | 0.302234 |   0.0 |   0.0006 |   0.0 |      0.0 |   0.0 | 0.084121 |
    | 1997-08-04 19:00:00 | 0.833333 | 19.7 |              1.0 |                      1.0 |       42.072586 |             66.1 |       3.6 |              1017.0 | 0.0 | 0.0 | 18.166667 |               78.4875 |                  12.1 |                     15.534444 |           243.962306 | 0.783333 | 19.7 | 18.166667 |    2.885728 |         2.194356 |           2.885728 |          3.6 |                22.988466 |                     20.889699 |                          0.0 |                      1.426658 |                           1.312212 |            15.195376 |                 16.395802 |    1001.804624 |   1.202947 | -2.9 | 18.693895 |   -6.03663 |             31.975165 |                 31.975165 |                 185.411352 |                 34.091963 |                      0.0 |    -2.116797 |              0.0 |        151.319389 |             39.231563 |                 100.0 |                 48.85611 |               57.260179 | 0.458513 |  0.0 |  0.183135 |  0.183135 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.458513 | 0.051088 | 0.004032 |  0.0 | 0.000175 |  0.0 | 0.051088 | 0.296218 | 0.082283 | 0.516865 |  0.0 |  0.0 |   0.0 |  nan | 8983.183439 | 123.999945 | 0.051088 |   0.0 | 0.004032 |   0.0 | 0.000175 | 0.295573 |   0.0 | 0.000644 |   0.0 |      0.0 |   0.0 | 0.082283 |
    | 1997-08-04 20:00:00 | 0.833333 | 17.9 |              0.3 |                 0.104711 |        0.812897 |             71.7 |       3.2 |              1017.0 | 0.0 | 0.0 | 18.104167 |             78.091667 |                  12.4 |                     15.502754 |           243.988429 | 0.783333 | 17.9 | 18.104167 |    2.565092 |         2.227756 |           2.565092 |          3.2 |                20.542326 |                     20.807826 |                          0.0 |                      1.293123 |                           1.307717 |            14.728848 |                 16.249178 |    1002.271152 |   1.210597 | -2.9 | 18.731535 |   4.989208 |              0.617802 |                  0.617802 |                 185.431206 |                 35.120093 |                      0.0 |   -34.502291 |              0.0 |        150.311113 |             44.135508 |                 100.0 |                 48.85611 |               83.091125 | 0.600198 |  0.0 |  0.100625 |  0.100625 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.600198 | 0.067185 | 0.004045 |  0.0 | 0.000192 |  0.0 | 0.067185 | 0.290156 | 0.080599 | 0.599375 |  0.0 |  0.0 |   0.0 |  nan | 8975.294232 | 124.528721 | 0.067185 |   0.0 | 0.004045 |   0.0 | 0.000192 | 0.289467 |   0.0 | 0.000689 |   0.0 |      0.0 |   0.0 | 0.080599 |
    | 1997-08-04 21:00:00 | 0.833333 | 15.8 |              0.0 |                      0.0 |             0.0 |             80.6 |       2.2 |              1018.0 | 0.0 | 0.0 | 18.020833 |             77.829167 |                  12.4 |                     15.502754 |           243.988429 | 0.783333 | 15.8 | 18.020833 |    1.763501 |         2.244455 |           1.763501 |          2.2 |                17.978626 |                     20.699099 |                          0.0 |                      1.150835 |                           1.301743 |            14.490772 |                 16.109936 |    1003.509228 |    1.22071 | -2.9 | 18.636864 |  17.021185 |                   0.0 |                       0.0 |                 185.431206 |                 35.424509 |                      0.0 |   -35.424509 |              0.0 |        150.006697 |             64.197103 |                 100.0 |                 48.85611 |               87.719298 | 0.682708 |  0.0 |  0.029975 |  0.029975 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.682708 |  0.07685 | 0.004062 |  0.0 | 0.000214 |  0.0 |  0.07685 | 0.284583 | 0.079051 | 0.670025 |  0.0 |  0.0 |   0.0 |  nan | 8955.373047 | 125.130304 |  0.07685 |   0.0 | 0.004062 |   0.0 | 0.000214 | 0.283851 |   0.0 | 0.000732 |   0.0 |      0.0 |   0.0 | 0.079051 |
    | 1997-08-04 22:00:00 | 0.833333 | 14.8 |              0.0 |                      0.0 |             0.0 |             85.9 |       2.2 |              1018.0 | 0.0 | 0.0 | 17.929167 |             77.654167 |                  12.4 |                     15.502754 |           243.988429 | 0.783333 | 14.8 | 17.929167 |    1.763501 |         2.261155 |           1.763501 |          2.2 |                16.859515 |                     20.580072 |                          0.0 |                      1.087886 |                           1.295199 |            14.482324 |                 15.981284 |    1003.517676 |   1.224953 | -2.9 |  18.39781 |  21.586859 |                   0.0 |                       0.0 |                 185.431206 |                 35.698198 |                      0.0 |   -35.698198 |              0.0 |        149.733007 |             64.197103 |                 100.0 |                 48.85611 |               87.719298 | 0.753358 |  0.0 |  0.012059 |  0.012059 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.753358 | 0.085333 | 0.004082 |  0.0 | 0.000239 |  0.0 | 0.085333 |  0.27939 | 0.077608 | 0.687941 |  0.0 |  0.0 |   0.0 |  nan | 8930.886187 | 125.794008 | 0.085333 |   0.0 | 0.004082 |   0.0 | 0.000239 | 0.278614 |   0.0 | 0.000776 |   0.0 |      0.0 |   0.0 | 0.077608 |
    | 1997-08-04 23:00:00 | 0.833333 | 14.1 |              0.0 |                      0.0 |             0.0 |             89.3 |       2.1 |              1018.0 | 0.0 | 0.0 | 17.833333 |               77.5875 |                  12.4 |                     15.502754 |           243.988429 | 0.783333 | 14.1 | 17.833333 |    1.683342 |         2.254475 |           1.683342 |          2.1 |                16.112886 |                     20.456276 |                          0.0 |                      1.045579 |                           1.288387 |            14.388807 |                 15.871513 |    1003.611193 |   1.227981 | -2.9 | 18.103968 |  24.023805 |                   0.0 |                       0.0 |                 185.431206 |                 35.921753 |                      0.0 |   -35.921753 |              0.0 |        149.509453 |             67.254108 |                 100.0 |                 48.85611 |               87.719298 | 0.771275 |  0.0 | -0.000116 | -0.000116 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.771275 | 0.087941 | 0.004104 |  0.0 | 0.000266 |  0.0 | 0.087941 | 0.274474 | 0.076243 | 0.700116 |  0.0 |  0.0 |   0.0 |  nan | 8903.962382 | 126.472972 | 0.087941 |   0.0 | 0.004104 |   0.0 | 0.000266 | 0.273655 |   0.0 | 0.000819 |   0.0 |      0.0 |   0.0 | 0.076243 |
    | 1997-08-05 00:00:00 | 0.833333 | 13.4 |              0.0 |                      0.0 |             0.0 |             91.4 |       1.8 |              1018.0 | 0.0 | 0.0 | 17.729167 |             77.729167 |                  12.4 |                     15.502754 |           243.988429 | 0.783333 | 13.4 | 17.729167 |    1.442864 |         2.237776 |           1.442864 |          1.8 |                15.395377 |                     20.322453 |                          0.0 |                      1.004677 |                           1.281018 |            14.071374 |                 15.796473 |    1003.928626 |   1.231127 | -2.9 | 17.780882 |  26.285292 |                   0.0 |                       0.0 |                 185.431206 |                 36.050541 |                      0.0 |   -36.050541 |              0.0 |        149.380664 |             78.463126 |                 100.0 |                 48.85611 |               87.719298 |  0.78345 |  0.0 | -0.009023 | -0.009023 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.78345 | 0.089929 | 0.004126 |  0.0 | 0.000295 |  0.0 | 0.089929 | 0.269746 | 0.074929 | 0.709023 |  0.0 |  0.0 |   0.0 |  nan |  8874.77709 | 127.162072 | 0.089929 |   0.0 | 0.004126 |   0.0 | 0.000295 | 0.268884 |   0.0 | 0.000862 |   0.0 |      0.0 |   0.0 | 0.074929 |
    | 1997-08-05 01:00:00 | 0.833333 | 12.8 |              0.0 |                      0.0 |             0.0 |             93.1 |       1.8 |              1018.0 | 0.0 | 0.0 | 17.620833 |             77.891667 |                  12.4 |                     15.502754 |           243.988429 | 0.783333 | 12.8 | 17.620833 |    1.442864 |         2.224416 |           1.442864 |          1.8 |                14.802811 |                     20.184089 |                          0.0 |                      0.970706 |                           1.273391 |            13.781417 |                 15.721724 |    1004.218583 |   1.233843 | -2.9 | 17.430658 |  27.783951 |                   0.0 |                       0.0 |                 185.431206 |                 36.176259 |                      0.0 |   -36.176259 |              0.0 |        149.254947 |             78.463126 |                 100.0 |                 48.85611 |               87.719298 | 0.792356 |  0.0 | -0.013264 | -0.013264 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.792356 | 0.091568 | 0.004148 |  0.0 | 0.000323 |  0.0 | 0.091568 | 0.265187 | 0.073663 | 0.713264 |  0.0 |  0.0 |   0.0 |  nan |  8844.09314 | 127.858389 | 0.091568 |   0.0 | 0.004148 |   0.0 | 0.000323 | 0.264282 |   0.0 | 0.000904 |   0.0 |      0.0 |   0.0 | 0.073663 |
    | 1997-08-05 02:00:00 | 0.833333 | 12.3 |              0.0 |                      0.0 |             0.0 |             94.6 |       1.6 |              1018.0 | 0.0 | 0.0 |      17.5 |             78.133333 |                  12.4 |                     15.502754 |           243.988429 | 0.783333 | 12.3 |      17.5 |    1.282546 |         2.201036 |           1.282546 |          1.6 |                14.324376 |                     20.030733 |                          0.0 |                      0.943147 |                            1.26493 |            13.550859 |                 15.650679 |    1004.449141 |   1.236111 | -2.9 | 17.062451 |  28.574706 |                   0.0 |                       0.0 |                 185.431206 |                 36.284611 |                      0.0 |   -36.284611 |              0.0 |        149.146595 |             88.271017 |                 100.0 |                 48.85611 |               87.719298 | 0.796598 |  0.0 | -0.017718 | -0.017718 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.796598 | 0.092684 | 0.004171 |  0.0 | 0.000352 |  0.0 | 0.092684 | 0.260782 |  0.07244 | 0.717718 |  0.0 |  0.0 |   0.0 |  nan | 8812.618434 | 128.557779 | 0.092684 |   0.0 | 0.004171 |   0.0 | 0.000352 | 0.259835 |   0.0 | 0.000947 |   0.0 |      0.0 |   0.0 |  0.07244 |
    | 1997-08-05 03:00:00 | 0.833333 | 12.0 |              0.0 |                      0.0 |             0.0 |             95.6 |       1.2 |              1018.0 | 0.0 | 0.0 | 17.379167 |             78.379167 |                  12.4 |                     15.502754 |           243.988429 | 0.783333 | 12.0 | 17.379167 |    0.961909 |         2.174316 |           0.961909 |          1.2 |                 14.04387 |                     19.878396 |                          0.0 |                      0.926933 |                           1.256516 |            13.425939 |                 15.580521 |    1004.574061 |   1.237469 | -2.9 | 16.684755 |  28.108527 |                   0.0 |                       0.0 |                 185.431206 |                 36.390891 |                      0.0 |   -36.390891 |              0.0 |        149.040314 |            117.694689 |                 100.0 |                 48.85611 |               87.719298 | 0.801051 |  0.0 | -0.020716 | -0.020716 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.801051 | 0.093837 | 0.004194 |  0.0 | 0.000382 |  0.0 | 0.093837 | 0.256522 | 0.071256 | 0.720716 |  0.0 |  0.0 |   0.0 |  nan | 8781.609907 | 129.260418 | 0.093837 |   0.0 | 0.004194 |   0.0 | 0.000382 | 0.255533 |   0.0 | 0.000989 |   0.0 | 0.000001 |   0.0 | 0.071256 |
    | 1997-08-05 04:00:00 | 0.833333 | 11.7 |              0.0 |                 0.366022 |        1.482347 |             96.2 |       1.9 |              1018.0 | 0.0 | 0.0 | 17.270833 |               78.5125 |                  12.4 |                     15.470733 |           243.983922 | 0.783333 | 11.7 | 17.270833 |    1.523023 |         2.160956 |           1.523023 |          1.9 |                13.768192 |                      19.74268 |                          0.0 |                      0.910957 |                           1.249013 |            13.245001 |                 15.500472 |    1004.754999 |   1.238856 | -2.9 | 16.312652 |  27.675913 |              1.126584 |                  1.126584 |                  185.42778 |                 36.589063 |                      0.0 |   -35.462479 |              0.0 |        148.838717 |             74.333488 |                 100.0 |                 48.85611 |               73.423526 | 0.804049 |  0.0 | -0.019578 | -0.019578 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.804049 | 0.094829 | 0.004217 |  0.0 | 0.000411 |  0.0 | 0.094829 |   0.2524 | 0.070111 | 0.719578 |  0.0 |  0.0 |   0.0 |  nan | 8751.033993 | 129.965011 | 0.094829 |   0.0 | 0.004217 |   0.0 | 0.000411 | 0.251369 |   0.0 |  0.00103 |   0.0 | 0.000001 |   0.0 | 0.070111 |
    | 1997-08-05 05:00:00 | 0.833333 | 11.6 |              0.0 |                      1.0 |       21.135567 |             95.7 |       1.3 |              1018.0 | 0.0 | 0.0 | 17.166667 |                  78.6 |                  12.4 |                     15.470733 |           244.033283 | 0.783333 | 11.6 | 17.166667 |    1.042069 |         2.120877 |           1.042069 |          1.3 |                 13.67736 |                     19.612949 |                          0.0 |                      0.905683 |                           1.241834 |            13.089234 |                 15.415778 |    1004.910766 |   1.239363 | -2.9 | 15.945741 |  26.074448 |             16.063031 |                 16.063031 |                 185.465295 |                  36.74524 |                      0.0 |   -20.682209 |              0.0 |        148.720055 |            108.641252 |                 100.0 |                 48.85611 |               57.260179 | 0.802911 |  0.0 | -0.009263 | -0.009263 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.802911 | 0.095337 |  0.00424 |  0.0 |  0.00044 |  0.0 | 0.095337 | 0.248404 | 0.069001 | 0.709263 |  0.0 |  0.0 |   0.0 |  nan | 8722.059546 | 130.667906 | 0.095337 |   0.0 |  0.00424 |   0.0 |  0.00044 | 0.247332 |   0.0 | 0.001072 |   0.0 | 0.000001 |   0.0 | 0.069001 |
    | 1997-08-05 06:00:00 | 0.833333 | 12.6 |              0.6 |                      1.0 |       125.88345 |             95.1 |       1.6 |              1018.0 | 0.0 | 0.0 | 17.095833 |             78.620833 |                  12.9 |                     15.470733 |           246.654019 | 0.783333 | 12.6 | 17.095833 |    1.282546 |         2.094157 |           1.282546 |          1.6 |                14.609782 |                     19.525158 |                          0.0 |                      0.959602 |                           1.236973 |            13.893903 |                 15.350842 |    1004.106097 |   1.234655 | -2.9 | 15.598048 |  17.988287 |             95.671422 |                 95.671422 |                 187.457054 |                  38.00446 |                      0.0 |    57.666962 |              0.0 |        149.452594 |             88.271017 |                 100.0 |                 48.85611 |               57.260179 | 0.792597 |  0.0 |  0.052361 |  0.052361 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.792597 | 0.094743 | 0.004263 |  0.0 | 0.000469 |  0.0 | 0.094743 | 0.244512 |  0.06792 | 0.647639 |  0.0 |  0.0 |   0.0 |  nan | 8701.171259 | 131.361028 | 0.094743 |   0.0 | 0.004263 |   0.0 | 0.000469 | 0.243398 |   0.0 | 0.001113 |   0.0 | 0.000001 |   0.0 |  0.06792 |
    | 1997-08-05 07:00:00 | 0.833333 | 14.8 |              0.9 |                      1.0 |      305.954944 |             96.3 |       1.9 |              1019.0 | 0.0 | 0.0 | 17.070833 |             78.745833 |                  12.9 |                     15.470733 |           247.494792 | 0.783333 | 14.8 | 17.070833 |    1.523023 |         2.074117 |           1.523023 |          1.9 |                16.859515 |                     19.494255 |                          0.0 |                      1.087886 |                           1.235261 |            16.235713 |                 15.350914 |    1002.764287 |   1.225361 | -2.9 | 15.347388 |   3.284331 |            232.525758 |                232.525758 |                 188.096042 |                 37.987562 |                      0.0 |   194.538195 |              0.0 |         150.10848 |             74.333488 |                 100.0 |                 48.85611 |               57.260179 | 0.730972 |  0.0 |  0.166779 |  0.166779 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.730972 | 0.087933 | 0.004285 |  0.0 | 0.000498 |  0.0 | 0.087933 | 0.240625 |  0.06684 | 0.533221 |  0.0 |  0.0 |   0.0 |  nan | 8694.986928 | 131.999284 | 0.087933 |   0.0 | 0.004285 |   0.0 | 0.000498 |  0.23947 |   0.0 | 0.001154 |   0.0 | 0.000001 |   0.0 |  0.06684 |
    | 1997-08-05 08:00:00 | 0.833333 | 17.1 |              1.0 |                      1.0 |      511.104575 |             89.9 |       3.2 |              1019.0 | 0.0 | 0.0 | 17.083333 |             78.870833 |                  12.9 |                     15.470733 |           248.989764 | 0.783333 | 17.1 | 17.083333 |    2.565092 |         2.097497 |           2.565092 |          3.2 |                19.530313 |                     19.509701 |                          0.0 |                      1.237258 |                           1.236116 |            17.557751 |                 15.387464 |    1001.442249 |   1.215051 | -2.9 | 15.273176 | -10.960941 |            388.439477 |                388.439477 |                 189.232221 |                 37.896217 |                      0.0 |    350.54326 |              0.0 |        151.336004 |             44.135508 |                 100.0 |                 48.85611 |               57.260179 | 0.616554 |  0.0 |  0.358863 |  0.358863 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.616554 | 0.074584 | 0.004306 |  0.0 | 0.000525 |  0.0 | 0.074584 | 0.236578 | 0.065716 | 0.341137 |  0.0 |  0.0 |   0.0 |  nan | 8703.047869 | 132.536423 | 0.074584 |   0.0 | 0.004306 |   0.0 | 0.000525 | 0.235382 |   0.0 | 0.001195 |   0.0 | 0.000001 |   0.0 | 0.065716 |
    | 1997-08-05 09:00:00 | 0.833333 | 19.4 |              0.9 |                      1.0 |      629.519239 |             79.2 |       4.0 |              1019.0 | 0.0 | 0.0 | 17.133333 |             78.883333 |                  12.8 |                     15.470733 |           249.636119 | 0.783333 | 19.4 | 17.133333 |    3.206365 |         2.134237 |           3.206365 |          4.0 |                22.563931 |                     19.571593 |                          0.0 |                      1.403627 |                           1.239544 |            17.870634 |                 15.438725 |    1001.129366 |   1.205358 | -2.9 | 15.369908 | -24.180553 |            478.434622 |                478.434622 |                 189.723451 |                  37.56462 |                      0.0 |   440.870002 |              0.0 |        152.158831 |             35.308407 |                 100.0 |                 48.85611 |               57.260179 | 0.424471 |  0.0 |  0.541806 |  0.541806 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.424471 | 0.051569 | 0.004324 |  0.0 | 0.000547 |  0.0 | 0.051569 | 0.232167 | 0.064491 | 0.158194 |  0.0 |  0.0 |   0.0 |  nan | 8724.328423 | 132.904454 | 0.051569 |   0.0 | 0.004324 |   0.0 | 0.000547 |  0.23093 |   0.0 | 0.001236 |   0.0 | 0.000001 |   0.0 | 0.064491 |
    | 1997-08-05 10:00:00 | 0.833333 | 20.6 |              1.0 |                      1.0 |      765.816589 |             73.4 |       4.6 |              1019.0 | 0.0 | 0.0 | 17.179167 |             78.783333 |                  12.9 |                     15.470733 |           254.719872 | 0.783333 | 20.6 | 17.179167 |     3.68732 |         2.184336 |            3.68732 |          4.6 |                24.304197 |                     19.628477 |                          0.0 |                      1.497674 |                           1.242694 |            17.839281 |                 15.463969 |    1001.160719 |   1.200448 | -2.9 | 15.625274 | -29.848354 |            582.020608 |                582.020608 |                 193.587103 |                 37.751868 |                      0.0 |    544.26874 |              0.0 |        155.835235 |             30.702962 |                 100.0 |                 48.85611 |               57.260179 | 0.241527 |  0.0 |  0.716842 |       0.7 | 0.010395 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.241527 | 0.029423 | 0.004336 |  0.0 | 0.000563 |  0.0 | 0.029423 | 0.227287 | 0.063135 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8751.276776 | 133.101265 | 0.029423 |   0.0 | 0.004336 |   0.0 | 0.000563 |  0.22601 |   0.0 | 0.001276 |   0.0 | 0.000001 |   0.0 | 0.063135 |
    | 1997-08-05 11:00:00 | 0.833333 | 21.4 |              1.0 |                      1.0 |      767.145842 |             67.3 |       4.1 |              1019.0 | 0.0 | 0.0 |    17.225 |             78.529167 |                  13.1 |                     15.470733 |           261.673325 | 0.783333 | 21.4 |    17.225 |    3.286524 |         2.211056 |           3.286524 |          4.1 |                25.528421 |                     19.685506 |                          0.0 |                      1.563281 |                            1.24585 |            17.180627 |                 15.458864 |    1001.819373 |   1.197482 | -2.9 | 15.948655 | -32.708072 |             583.03084 |                 583.03084 |                 198.871727 |                 38.247049 |                      0.0 |   544.783791 |              0.0 |        160.624678 |             34.447226 |                 100.0 |                 48.85611 |               57.260179 | 0.083333 |  0.0 |  0.746089 |       0.7 | 0.029871 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010164 | 0.004342 |  0.0 | 0.000571 |  0.0 | 0.010164 | 0.221999 | 0.061666 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8781.084848 |  133.13965 | 0.010164 |   0.0 | 0.004342 |   0.0 | 0.000571 | 0.220682 |   0.0 | 0.001316 |   0.0 | 0.000001 |   0.0 | 0.061666 |
    | 1997-08-05 12:00:00 | 0.833333 | 22.5 |              1.0 |                      1.0 |      736.263853 |             62.2 |       4.1 |              1019.0 | 0.0 | 0.0 |   17.2875 |                78.275 |                  13.2 |                     15.470733 |            266.49623 | 0.783333 | 22.5 |   17.2875 |    3.286524 |         2.237776 |           3.286524 |          4.1 |                27.299387 |                     19.763507 |                          0.0 |                      1.657431 |                           1.250165 |            16.980219 |                 15.469885 |    1002.019781 |   1.193116 | -2.9 | 16.306352 | -37.161891 |            559.560528 |                559.560528 |                 202.537135 |                  38.48429 |                      0.0 |   521.076238 |              0.0 |        164.052844 |             34.447226 |                 100.0 |                 48.85611 |               57.260179 | 0.083333 |  0.0 |  0.771457 |       0.7 | 0.046934 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010168 | 0.004343 |  0.0 | 0.000572 |  0.0 | 0.010168 | 0.216601 | 0.060167 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8815.346739 | 133.160966 | 0.010168 |   0.0 | 0.004343 |   0.0 | 0.000572 | 0.215244 |   0.0 | 0.001355 |   0.0 | 0.000001 |   0.0 | 0.060167 |
    | 1997-08-05 13:00:00 | 0.833333 | 23.1 |              1.0 |                      1.0 |      768.006183 |             59.6 |       4.4 |              1019.0 | 0.0 | 0.0 | 17.341667 |             78.204167 |                  13.4 |                     15.470733 |           273.519763 | 0.783333 | 23.1 | 17.341667 |    3.527001 |         2.277855 |           3.527001 |          4.4 |                28.309776 |                     19.831326 |                          0.0 |                      1.710768 |                           1.253914 |            16.872626 |                 15.508923 |    1002.127374 |   1.190747 | -2.9 | 16.717494 | -38.295035 |            583.684699 |                583.684699 |                  207.87502 |                  38.86309 |                      0.0 |   544.821609 |              0.0 |         169.01193 |             32.098552 |                 100.0 |                 48.85611 |               57.260179 | 0.083333 |  0.0 |  0.841038 |       0.7 | 0.091262 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 |  0.01017 | 0.004344 |  0.0 | 0.000573 |  0.0 |  0.01017 | 0.211342 | 0.058706 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8850.741774 |  133.13795 |  0.01017 |   0.0 | 0.004344 |   0.0 | 0.000573 | 0.209947 |   0.0 | 0.001394 |   0.0 | 0.000001 |   0.0 | 0.058706 |
    | 1997-08-05 14:00:00 | 0.833333 | 23.4 |              1.0 |                      1.0 |      735.929369 |             55.4 |       4.6 |              1019.0 | 0.0 | 0.0 | 17.404167 |               77.9875 |                  13.5 |                     15.470733 |           278.618852 | 0.783333 | 23.4 | 17.404167 |     3.68732 |         2.347994 |            3.68732 |          4.6 |                28.827078 |                     19.909831 |                          0.0 |                      1.737973 |                           1.258253 |            15.970201 |                 15.527179 |    1003.029799 |   1.189943 | -2.9 | 17.142235 | -37.546592 |            559.306321 |                559.306321 |                 211.750328 |                 39.079824 |                      0.0 |   520.226497 |              0.0 |        172.670504 |             30.702962 |                 100.0 |                 48.85611 |               57.260179 | 0.083333 |  0.0 |  0.865229 |       0.7 | 0.105785 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010167 | 0.004343 |  0.0 | 0.000572 |  0.0 | 0.010167 |  0.20622 | 0.057283 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8885.388366 | 133.100415 | 0.010167 |   0.0 | 0.004343 |   0.0 | 0.000572 | 0.204786 |   0.0 | 0.001432 |   0.0 | 0.000001 |   0.0 | 0.057283 |
    | 1997-08-05 15:00:00 | 0.833333 | 23.8 |              1.0 |                      1.0 |      620.374953 |             52.1 |       4.5 |              1019.0 | 0.0 | 0.0 |   17.4875 |             77.708333 |                  13.6 |                     15.470733 |           282.921682 | 0.783333 | 23.8 |   17.4875 |    3.607161 |         2.401434 |           3.607161 |          4.5 |                29.529613 |                     20.014927 |                          0.0 |                      1.774812 |                           1.264057 |            15.384928 |                 15.553266 |    1003.615072 |     1.1886 | -2.9 | 17.557994 | -37.452038 |            471.484964 |                471.484964 |                 215.020478 |                 39.288738 |                      0.0 |   432.196226 |              0.0 |         175.73174 |              31.38525 |                 100.0 |                 48.85611 |               57.260179 | 0.083333 |  0.0 |   0.79847 |       0.7 | 0.063817 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010164 | 0.004342 |  0.0 | 0.000571 |  0.0 | 0.010164 | 0.201231 | 0.055897 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8919.940404 | 133.104855 | 0.010164 |   0.0 | 0.004342 |   0.0 | 0.000571 | 0.199759 |   0.0 |  0.00147 |   0.0 | 0.000001 |   0.0 | 0.055897 |
    | 1997-08-05 16:00:00 | 0.833333 | 24.0 |              1.0 |                      1.0 |      450.095264 |             49.5 |       4.7 |              1018.0 | 0.0 | 0.0 |   17.5625 |               77.3875 |                  13.7 |                     15.470733 |           285.932459 | 0.783333 | 24.0 |   17.5625 |    3.767479 |         2.438173 |           3.767479 |          4.7 |                29.886439 |                     20.109928 |                          0.0 |                      1.793476 |                             1.2693 |            14.793787 |                  15.56257 |    1003.206213 |   1.186889 | -2.9 | 17.972618 | -36.164291 |            342.072401 |                342.072401 |                 217.308669 |                 39.539407 |                      0.0 |   302.532993 |              0.0 |        177.769261 |             30.049708 |                 100.0 |                 48.85611 |               57.260179 | 0.083333 |  0.0 |   0.70277 |       0.7 | 0.001774 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010164 | 0.004342 |  0.0 | 0.000571 |  0.0 | 0.010164 | 0.196371 | 0.054547 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8953.204695 | 133.171337 | 0.010164 |   0.0 | 0.004342 |   0.0 | 0.000571 | 0.194862 |   0.0 | 0.001508 |   0.0 | 0.000002 |   0.0 | 0.054547 |
    | 1997-08-05 17:00:00 | 0.833333 | 24.0 |              1.0 |                      1.0 |      273.614142 |             48.1 |       4.5 |              1018.0 | 0.0 | 0.0 | 17.654167 |             77.008333 |                  13.7 |                     15.470733 |           286.824521 | 0.783333 | 24.0 | 17.654167 |    3.607161 |         2.448193 |           3.607161 |          4.5 |                29.886439 |                     20.226575 |                          0.0 |                      1.793476 |                           1.275734 |            14.375377 |                 15.576148 |    1003.624623 |   1.187075 | -2.9 |  18.37179 | -33.769262 |            207.946748 |                207.946748 |                 217.986636 |                 39.564612 |                      0.0 |   168.382136 |              0.0 |        178.422024 |              31.38525 |                 100.0 |                 48.85611 |               57.260179 | 0.083333 |  0.0 |  0.559369 |  0.559369 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.083333 | 0.010171 | 0.004344 |  0.0 | 0.000574 |  0.0 | 0.010171 | 0.191637 | 0.053233 | 0.140631 |  0.0 |  0.0 |   0.0 |  nan | 8984.073957 | 133.239582 | 0.010171 |   0.0 | 0.004344 |   0.0 | 0.000574 | 0.190091 |   0.0 | 0.001544 |   0.0 | 0.000002 |   0.0 | 0.053233 |
    | 1997-08-05 18:00:00 | 0.833333 | 23.7 |              1.0 |                      1.0 |      133.368106 |             49.4 |       4.1 |              1018.0 | 0.0 | 0.0 | 17.770833 |             76.570833 |                  13.7 |                     15.470733 |           287.024163 | 0.783333 | 23.7 | 17.770833 |    3.286524 |         2.444853 |           3.286524 |          4.1 |                29.352596 |                      20.37589 |                          0.0 |                      1.765541 |                           1.283961 |            14.500182 |                 15.601989 |    1003.499818 |   1.188219 | -2.9 | 18.742221 | -29.746675 |             101.35976 |                 101.35976 |                 218.138364 |                 39.572333 |                      0.0 |    61.787427 |              0.0 |        178.566031 |             34.447226 |                 100.0 |                 48.85611 |               57.260179 | 0.223964 |  0.0 |  0.406457 |  0.406457 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.223964 | 0.027368 | 0.004346 |  0.0 | 0.000577 |  0.0 | 0.027368 | 0.187249 | 0.052014 | 0.293543 |  0.0 |  0.0 |   0.0 |  nan | 9010.920632 | 133.431255 | 0.027368 |   0.0 | 0.004346 |   0.0 | 0.000577 | 0.185667 |   0.0 | 0.001581 |   0.0 | 0.000002 |   0.0 | 0.052014 |
    | 1997-08-05 19:00:00 | 0.833333 | 22.8 |              1.0 |                      1.0 |       40.326503 |             57.0 |       3.8 |              1018.0 | 0.0 | 0.0 |      17.9 |             76.191667 |                  13.7 |                     15.470733 |           286.951409 | 0.783333 | 22.8 |      17.9 |    3.046047 |         2.451533 |           3.046047 |          3.8 |                27.800581 |                     20.542326 |                          0.0 |                      1.683922 |                           1.293123 |            15.846331 |                 15.651541 |    1002.153669 |   1.191234 | -2.9 | 19.064381 | -22.413715 |             30.648142 |                 30.648142 |                 218.083071 |                 39.521285 |                      0.0 |    -8.873143 |              0.0 |        178.561786 |             37.166744 |                 100.0 |                 48.85611 |               57.260179 | 0.376876 |  0.0 |  0.257526 |  0.257526 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.376876 | 0.046166 | 0.004353 |  0.0 | 0.000585 |  0.0 | 0.046166 |  0.18344 | 0.050956 | 0.442474 |  0.0 |  0.0 |   0.0 |  nan | 9030.434346 | 133.757028 | 0.046166 |   0.0 | 0.004353 |   0.0 | 0.000585 | 0.181821 |   0.0 | 0.001617 |   0.0 | 0.000002 |   0.0 | 0.050956 |
    | 1997-08-05 20:00:00 | 0.833333 | 21.3 |              0.5 |                 0.072689 |        0.867122 |             65.9 |       2.5 |              1018.0 | 0.0 | 0.0 | 18.041667 |                 75.95 |                  13.9 |                     15.438712 |           286.953669 | 0.783333 | 21.3 | 18.041667 |    2.003978 |         2.428153 |           2.003978 |          2.5 |                 25.37251 |                     20.726234 |                          0.0 |                       1.55495 |                           1.303234 |            16.720484 |                 15.741575 |    1001.279516 |   1.196911 | -2.9 | 19.298545 | -12.008727 |              0.659013 |                  0.659013 |                 218.084788 |                 39.876809 |                      0.0 |   -39.217796 |              0.0 |        178.207979 |             56.493451 |                 100.0 |                 48.85611 |               84.453771 | 0.525807 |  0.0 |  0.102206 |  0.102206 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.525807 | 0.064647 | 0.004363 |  0.0 | 0.000598 |  0.0 | 0.064647 | 0.180211 | 0.050059 | 0.597794 |  0.0 |  0.0 |   0.0 |  nan | 9039.543073 | 134.213226 | 0.064647 |   0.0 | 0.004363 |   0.0 | 0.000598 | 0.178557 |   0.0 | 0.001652 |   0.0 | 0.000002 |   0.0 | 0.050059 |
    | 1997-08-05 21:00:00 | 0.833333 | 19.3 |              0.0 |                      0.0 |             0.0 |             73.0 |       1.9 |              1018.0 | 0.0 | 0.0 |   18.1875 |             75.633333 |                  13.9 |                     15.438712 |           286.953669 | 0.783333 | 19.3 |   18.1875 |    1.523023 |         2.418134 |           1.523023 |          1.9 |                22.423949 |                     20.917052 |                          0.0 |                       1.39602 |                           1.313713 |            16.369483 |                 15.820264 |    1001.630517 |   1.205255 | -2.9 |  19.40785 |   0.647101 |                   0.0 |                       0.0 |                 218.084788 |                 39.753451 |                      0.0 |   -39.753451 |              0.0 |        178.331337 |             74.333488 |                 100.0 |                 48.85611 |               87.719298 | 0.681127 |  0.0 |  0.046254 |  0.046254 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.681127 | 0.084156 | 0.004378 |  0.0 | 0.000617 |  0.0 | 0.084156 | 0.177557 | 0.049322 | 0.653746 |  0.0 |  0.0 |   0.0 |  nan | 9035.995972 | 134.805202 | 0.084156 |   0.0 | 0.004378 |   0.0 | 0.000617 | 0.175868 |   0.0 | 0.001688 |   0.0 | 0.000002 |   0.0 | 0.049322 |
    | 1997-08-05 22:00:00 | 0.833333 | 18.3 |              0.0 |                      0.0 |             0.0 |             76.7 |       2.2 |              1018.0 | 0.0 | 0.0 | 18.333333 |                 75.25 |                  13.9 |                     15.438712 |           286.953669 | 0.783333 | 18.3 | 18.333333 |    1.763501 |         2.418134 |           1.763501 |          2.2 |                21.065302 |                     21.109404 |                          0.0 |                      1.321845 |                           1.324263 |            16.157086 |                 15.884826 |    1001.842914 |   1.209486 | -2.9 | 19.365285 |    6.39171 |                   0.0 |                       0.0 |                 218.084788 |                 39.670785 |                      0.0 |   -39.670785 |              0.0 |        178.414003 |             64.197103 |                 100.0 |                 48.85611 |               87.719298 |  0.73708 |  0.0 |  0.042508 |  0.042508 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.73708 | 0.091597 | 0.004398 |  0.0 | 0.000642 |  0.0 | 0.091597 |  0.17532 |   0.0487 | 0.657492 |  0.0 |  0.0 |   0.0 |  nan | 9026.704262 | 135.445645 | 0.091597 |   0.0 | 0.004398 |   0.0 | 0.000642 | 0.173595 |   0.0 | 0.001723 |   0.0 | 0.000002 |   0.0 |   0.0487 |
    | 1997-08-05 23:00:00 | 0.833333 | 17.4 |              0.0 |                      0.0 |             0.0 |             81.8 |       2.7 |              1018.0 | 0.0 | 0.0 | 18.470833 |               74.9375 |                  13.9 |                     15.438712 |           286.953669 | 0.783333 | 17.4 | 18.470833 |    2.164296 |         2.438173 |           2.164296 |          2.7 |                19.904589 |                     21.292177 |                          0.0 |                      1.257963 |                           1.334275 |            16.281953 |                 15.955825 |    1001.718047 |   1.213176 | -2.9 | 19.253784 |  11.122707 |                   0.0 |                       0.0 |                 218.084788 |                 39.564038 |                      0.0 |   -39.564038 |              0.0 |        178.520751 |             52.308751 |                 100.0 |                 48.85611 |               87.719298 | 0.740825 |  0.0 |  0.035577 |  0.035577 |      0.0 | 0.0 |   0.0 |          nan |         0.24 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 | 0.740825 | 0.092616 | 0.004418 |  0.0 | 0.000669 |  0.0 | 0.092616 |  0.17325 | 0.048125 | 0.664423 |  0.0 |  0.0 |   0.0 |  nan | 9012.681555 | 136.088768 | 0.092616 |   0.0 | 0.004418 |   0.0 | 0.000669 |  0.17149 |   0.0 | 0.001758 |   0.0 | 0.000002 |   0.0 | 0.048125 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions_acker_heavy_rain))
0.0

.. _lland_v3_water_hourly:

water
-----

For our hourly integration test on water type |WASSER|, we deviate from the input data
of the daily simulation (:ref:`lland_v3_wasser_daily`) in setting precipitation to the
constant value of 0.05 mm/h.  Neither the actual (|EvI|) or the potential evaporation
(|EvPo|) does show a diurnal pattern due to method |Return_Penman_V1| relying on daily
aggregated input data (e.g. |DailyNetRadiation|) only (the same holds for water types
|FLUSS| and |SEE|):

.. integration-test::

    >>> lnk(WASSER)
    >>> inputs.nied.series = 0.05
    >>> test("lland_v3_water_hourly",
    ...      axis1=(fluxes.nkor, fluxes.evpo, fluxes.evi, fluxes.qah),
    ...      use_conditions=conditions_wasser)
    |                date | nied | teml | sunshineduration | possiblesunshineduration | globalradiation | relativehumidity | windspeed | atmosphericpressure |  qz | qzh |   temltag | dailyrelativehumidity | dailysunshineduration | dailypossiblesunshineduration | dailyglobalradiation |  nkor | tkor |   tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair |   g |  tz |  wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |     evpo |      evi | evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |      qah |       qa | inzp | wats | waes | esnow | taus | ebdn | bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz | qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-08-03 00:00:00 | 0.05 | 16.9 |              0.0 |                      0.0 |             0.0 |             95.1 |       0.8 |              1015.0 | 0.0 | 0.0 | 17.279167 |             85.883333 |                   1.3 |                          15.7 |           136.579167 | 0.047 | 16.9 | 17.279167 |    0.641273 |         1.605886 |           0.641273 |          0.8 |                19.284227 |                     19.753091 |                          0.0 |                      1.223615 |                           1.249589 |              18.3393 |                 16.964613 |       996.6607 |    1.21073 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   40.97375 |                 10.408237 |                      0.0 |   -10.408237 |              0.0 |         30.565513 |            266.257083 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.040545 | 0.040545 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.006455 | 0.001793 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001793 |
    | 1997-08-03 01:00:00 | 0.05 | 16.6 |              0.0 |                      0.0 |             0.0 |             94.9 |       0.8 |              1015.0 | 0.0 | 0.0 | 17.241667 |             85.991667 |                   1.3 |                          15.7 |           136.579167 | 0.047 | 16.6 | 17.241667 |    0.641273 |         1.603439 |           0.641273 |          0.8 |                18.920184 |                      19.70628 |                          0.0 |                       1.20339 |                           1.246999 |            17.955254 |                 16.945759 |     997.044746 |   1.212158 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   40.97375 |                  10.41572 |                      0.0 |    -10.41572 |              0.0 |          30.55803 |            266.257083 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.04041 |  0.04041 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.00659 | 0.001831 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001831 |
    | 1997-08-03 02:00:00 | 0.05 | 16.4 |              0.0 |                      0.0 |             0.0 |             95.9 |       0.8 |              1015.0 | 0.0 | 0.0 |    17.175 |             86.233333 |                   1.3 |                          15.7 |           136.579167 | 0.047 | 16.4 |    17.175 |    0.641273 |         1.575992 |           0.641273 |          0.8 |                 18.68084 |                       19.6233 |                          0.0 |                      1.190065 |                           1.242407 |            17.914926 |                 16.921826 |     997.085074 |   1.213014 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   40.97375 |                 10.421627 |                      0.0 |   -10.421627 |              0.0 |         30.552123 |            266.257083 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.040055 | 0.040055 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.006945 | 0.001929 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001929 |
    | 1997-08-03 03:00:00 | 0.05 | 16.3 |              0.0 |                      0.0 |             0.0 |             96.7 |       0.8 |              1015.0 | 0.0 | 0.0 |   17.0625 |             86.708333 |                   1.3 |                          15.5 |           136.579167 | 0.047 | 16.3 |   17.0625 |    0.641273 |         1.548545 |           0.641273 |          0.8 |                18.562165 |                     19.483964 |                          0.0 |                      1.183449 |                            1.23469 |            17.949613 |                  16.89422 |     997.050387 |   1.213417 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                   40.97375 |                 10.455184 |                      0.0 |   -10.455184 |              0.0 |         30.518566 |            266.257083 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.039459 | 0.039459 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.007541 | 0.002095 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.002095 |
    | 1997-08-03 04:00:00 | 0.05 | 16.0 |              0.0 |                 0.429734 |        1.943686 |             97.2 |       0.8 |              1015.0 | 0.0 | 0.0 | 16.908333 |             87.366667 |                   1.3 |                     14.929734 |            135.50182 | 0.047 | 16.0 | 16.908333 |    0.641273 |         1.504432 |           0.641273 |          0.8 |                18.210086 |                     19.294427 |                          0.0 |                      1.163788 |                           1.224181 |            17.700204 |                 16.856897 |     997.299796 |    1.21479 | 0.0 | nan | 0.0 |              0.583106 |                  0.583106 |                  40.650546 |                 10.555242 |                      0.0 |    -9.972137 |              0.0 |         30.095304 |            266.257083 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.038271 | 0.038271 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.008729 | 0.002425 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.002425 |
    | 1997-08-03 05:00:00 | 0.05 | 15.9 |              0.0 |                      1.0 |       21.932442 |             97.5 |       0.6 |              1015.0 | 0.0 | 0.0 | 16.729167 |             88.204167 |                   1.3 |                     14.929734 |           134.099005 | 0.047 | 15.9 | 16.729167 |    0.480955 |         1.453638 |           0.480955 |          0.6 |                18.094032 |                     19.076181 |                          0.0 |                      1.157296 |                           1.212063 |            17.641681 |                 16.825987 |     997.358319 |   1.215237 | 0.0 | nan | 0.0 |              6.579733 |                  6.579733 |                  40.229702 |                 10.545193 |                      0.0 |    -3.965461 |              0.0 |         29.684509 |            355.009444 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.036944 | 0.036944 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.010056 | 0.002793 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.002793 |
    | 1997-08-03 06:00:00 | 0.05 | 16.0 |              0.0 |                      1.0 |       57.256186 |             97.7 |       0.9 |              1015.0 | 0.0 | 0.0 | 16.533333 |             89.191667 |                   1.3 |                     14.929734 |           131.855513 | 0.047 | 16.0 | 16.533333 |    0.721432 |         1.392031 |           0.721432 |          0.9 |                18.210086 |                     18.840106 |                          0.0 |                      1.163788 |                           1.198935 |            17.791254 |                 16.803805 |     997.208746 |   1.214748 | 0.0 | nan | 0.0 |             17.176856 |                 17.176856 |                  39.556654 |                  10.52516 |                      0.0 |     6.651696 |              0.0 |         29.031494 |            236.672963 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.035268 | 0.035268 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.011732 | 0.003259 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003259 |
    | 1997-08-03 07:00:00 | 0.05 | 16.6 |              0.0 |                      1.0 |      109.332844 |             97.4 |       0.9 |              1015.0 | 0.0 | 0.0 |    16.375 |                90.125 |                   1.3 |                     14.929734 |           129.465215 | 0.047 | 16.6 |    16.375 |    0.721432 |         1.334591 |           0.721432 |          0.9 |                18.920184 |                     18.651109 |                          0.0 |                       1.20339 |                           1.188408 |            18.428259 |                 16.809312 |     996.571741 |   1.211943 | 0.0 | nan | 0.0 |             32.799853 |                 32.799853 |                  38.839564 |                  10.49073 |                      0.0 |    22.309124 |              0.0 |         28.348835 |            236.672963 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.033685 | 0.033685 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.013315 | 0.003699 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003699 |
    | 1997-08-03 08:00:00 | 0.05 | 17.4 |              0.0 |                      1.0 |      170.949153 |             96.8 |       0.9 |              1016.0 | 0.0 | 0.0 | 16.216667 |                91.275 |                   1.3 |                     14.929734 |           127.329763 | 0.047 | 17.4 | 16.216667 |    0.721432 |         1.260484 |           0.721432 |          0.9 |                19.904589 |                     18.463773 |                          0.0 |                      1.257963 |                           1.177959 |            19.267642 |                 16.852809 |     996.732358 |   1.209425 | 0.0 | nan | 0.0 |             51.284746 |                 51.284746 |                  38.198929 |                 10.426918 |                      0.0 |    40.857828 |              0.0 |         27.772011 |            236.672963 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.032054 | 0.032054 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.014946 | 0.004152 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.004152 |
    | 1997-08-03 09:00:00 | 0.05 | 19.0 |              0.2 |                      1.0 |      311.762625 |             86.1 |       1.3 |              1016.0 | 0.0 | 0.0 |   16.1125 |             91.991667 |                   1.5 |                     14.929734 |           129.903206 | 0.047 | 19.0 |   16.1125 |    1.042069 |         1.203904 |           1.042069 |          1.3 |                22.008543 |                     18.341425 |                          0.0 |                      1.373407 |                           1.171128 |            18.949356 |                 16.872582 |     997.050644 |   1.202945 | 0.0 | nan | 0.0 |             93.528788 |                 93.528788 |                  38.970962 |                 10.804921 |                      0.0 |    82.723866 |              0.0 |         28.166041 |            163.850513 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.031746 | 0.031746 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.015254 | 0.004237 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.004237 |
    | 1997-08-03 10:00:00 | 0.05 | 20.3 |              0.5 |                      1.0 |        501.5833 |             76.8 |       1.5 |              1016.0 | 0.0 | 0.0 | 16.083333 |             92.241667 |                   1.7 |                     14.929734 |           132.285843 | 0.047 | 20.3 | 16.083333 |    1.202387 |         1.149836 |           1.202387 |          1.5 |                23.858503 |                     18.307295 |                          0.0 |                      1.473678 |                           1.169221 |             18.32333 |                 16.886954 |      997.67667 |   1.197896 | 0.0 | nan | 0.0 |             150.47499 |                 150.47499 |                  39.685753 |                 11.199993 |                      0.0 |   139.274997 |              0.0 |          28.48576 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.031752 | 0.031752 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.015248 | 0.004236 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.004236 |
    | 1997-08-03 11:00:00 | 0.05 | 21.4 |              0.7 |                      1.0 |      615.018728 |             71.8 |       1.2 |              1016.0 | 0.0 | 0.0 |    16.125 |             92.104167 |                   2.2 |                     14.929734 |           135.919957 | 0.047 | 21.4 |    16.125 |    0.961909 |         1.089916 |           0.961909 |          1.2 |                25.528421 |                     18.356069 |                          0.0 |                      1.563281 |                           1.171946 |            18.329406 |                 16.906704 |     997.670594 |    1.19342 | 0.0 | nan | 0.0 |            184.505618 |                184.505618 |                  40.775987 |                 12.222123 |                      0.0 |   172.283495 |              0.0 |         28.553864 |            177.504722 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.031812 | 0.031812 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.015188 | 0.004219 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.004219 |
    | 1997-08-03 12:00:00 | 0.05 | 21.3 |              0.8 |                      1.0 |      626.544325 |             67.5 |       1.3 |              1016.0 | 0.0 | 0.0 | 16.204167 |             91.729167 |                   2.8 |                     14.929734 |           145.821804 | 0.047 | 21.3 | 16.204167 |    1.042069 |         1.037502 |           1.042069 |          1.3 |                 25.37251 |                     18.449053 |                          0.0 |                       1.55495 |                           1.177138 |            17.126444 |                 16.923163 |     998.873556 |   1.194363 | 0.0 | nan | 0.0 |            187.963298 |                187.963298 |                  43.746541 |                 13.461134 |                      0.0 |   174.502163 |              0.0 |         30.285407 |            163.850513 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.033619 | 0.033619 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.013381 | 0.003717 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003717 |
    | 1997-08-03 13:00:00 | 0.05 | 21.8 |              0.5 |                      1.0 |      496.133417 |             66.1 |       1.5 |              1016.0 | 0.0 | 0.0 | 16.329167 |             91.104167 |                   3.2 |                     14.929734 |           152.606529 | 0.047 | 21.8 | 16.329167 |    1.202387 |         1.012602 |           1.202387 |          1.5 |                26.160453 |                      18.59671 |                          0.0 |                      1.596982 |                           1.185375 |             17.29206 |                 16.942378 |      998.70794 |   1.192265 | 0.0 | nan | 0.0 |            148.840025 |                148.840025 |                  45.781959 |                 14.297428 |                      0.0 |   134.542597 |              0.0 |         31.484531 |            142.003778 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.035172 | 0.035172 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.011828 | 0.003286 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.003286 |
    | 1997-08-03 14:00:00 | 0.05 | 22.9 |              0.4 |                      1.0 |      419.520994 |             63.4 |       1.9 |              1016.0 | 0.0 | 0.0 | 16.545833 |             90.058333 |                   3.4 |                     14.929734 |           157.353237 | 0.047 | 22.9 | 16.545833 |    1.523023 |         0.984394 |           1.523023 |          1.9 |                27.969419 |                     18.855098 |                          0.0 |                      1.692831 |                           1.199769 |            17.732611 |                 16.980587 |     998.267389 |   1.187639 | 0.0 | nan | 0.0 |            125.856298 |                125.856298 |                  47.205971 |                 14.726001 |                      0.0 |   111.130298 |              0.0 |         32.479971 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.036854 | 0.036854 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.010146 | 0.002818 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.002818 |
    | 1997-08-03 15:00:00 | 0.05 | 22.7 |              0.5 |                      1.0 |      387.887353 |             62.4 |       1.9 |              1016.0 | 0.0 | 0.0 | 16.816667 |             88.816667 |                   3.8 |                     14.929734 |           164.256877 | 0.047 | 22.7 | 16.816667 |    1.523023 |         0.968687 |           1.523023 |          1.9 |                27.632633 |                     19.182495 |                          0.0 |                      1.675052 |                           1.217969 |            17.242763 |                 17.037252 |     998.757237 |    1.18866 | 0.0 | nan | 0.0 |            116.366206 |                116.366206 |                  49.277063 |                 15.562805 |                      0.0 |     100.8034 |              0.0 |         33.714258 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.038963 | 0.038963 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.008037 | 0.002233 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.002233 |
    | 1997-08-03 16:00:00 | 0.05 | 22.5 |              0.5 |                      1.0 |      278.496872 |             61.1 |       2.3 |              1016.0 | 0.0 | 0.0 |   17.1375 |             87.333333 |                   4.1 |                     14.929734 |           170.073414 | 0.047 | 22.5 |   17.1375 |     1.84366 |         0.991339 |            1.84366 |          2.3 |                27.299387 |                     19.576758 |                          0.0 |                      1.657431 |                            1.23983 |            16.679926 |                 17.097035 |     999.320074 |   1.189715 | 0.0 | nan | 0.0 |             83.549062 |                 83.549062 |                  51.022024 |                 16.204965 |                      0.0 |    67.344096 |              0.0 |         34.817059 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.041278 | 0.041278 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.005722 |  0.00159 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.00159 |
    | 1997-08-03 17:00:00 | 0.05 | 21.9 |              0.3 |                      1.0 |      137.138608 |             62.1 |       2.4 |              1016.0 | 0.0 | 0.0 |   17.4875 |                85.875 |                   4.4 |                     14.929734 |           173.470856 | 0.047 | 21.9 |   17.4875 |    1.923819 |         1.017332 |           1.923819 |          2.4 |                26.320577 |                     20.014927 |                          0.0 |                      1.605502 |                           1.264057 |            16.345078 |                 17.187818 |     999.654922 |   1.192283 | 0.0 | nan | 0.0 |             41.141582 |                 41.141582 |                  52.041257 |                 16.818923 |                      0.0 |     24.32266 |              0.0 |         35.222334 |             88.752361 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.04299 |  0.04299 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.00401 | 0.001114 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.001114 |
    | 1997-08-03 18:00:00 | 0.05 | 21.4 |              0.1 |                      1.0 |       51.080714 |             67.0 |       2.5 |              1016.0 | 0.0 | 0.0 |   17.8375 |             84.620833 |                   4.5 |                     14.929734 |           174.440885 | 0.047 | 21.4 |   17.8375 |    2.003978 |         1.054998 |           2.003978 |          2.5 |                25.528421 |                     20.461645 |                          0.0 |                      1.563281 |                           1.288683 |            17.104042 |                 17.314814 |     998.895958 |   1.193968 | 0.0 | nan | 0.0 |             15.324214 |                 15.324214 |                  52.332266 |                 16.973018 |                      0.0 |    -1.648804 |              0.0 |         35.359248 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |   0.0444 |   0.0444 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |   0.0026 | 0.000722 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000722 |
    | 1997-08-03 19:00:00 | 0.05 | 20.7 |              0.0 |                      1.0 |       13.632817 |             74.5 |       2.5 |              1016.0 | 0.0 | 0.0 | 18.170833 |                  83.7 |                   4.5 |                     15.429734 |           175.008919 | 0.047 | 20.7 | 18.170833 |    2.003978 |         1.100997 |           2.003978 |          2.5 |                24.454368 |                     20.895167 |                          0.0 |                      1.505746 |                           1.312512 |            18.218504 |                 17.489255 |     997.781496 |   1.196313 | 0.0 | nan | 0.0 |              4.089845 |                  4.089845 |                  52.502676 |                 16.556776 |                      0.0 |   -12.466931 |              0.0 |           35.9459 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.046089 | 0.046089 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.000911 | 0.000253 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.000253 |
    | 1997-08-03 20:00:00 | 0.05 | 19.4 |              0.0 |                   0.1364 |        0.185944 |             81.2 |       2.2 |              1016.0 | 0.0 | 0.0 | 18.454167 |             83.066667 |                   4.5 |                     15.566134 |           175.016667 | 0.047 | 19.4 | 18.454167 |    1.763501 |         1.120309 |           1.763501 |          2.2 |                22.563931 |                      21.26995 |                          0.0 |                      1.403627 |                           1.333058 |            18.321912 |                 17.668238 |     997.678088 |   1.201582 | 0.0 | nan | 0.0 |              0.055783 |                  0.055783 |                     52.505 |                 16.343497 |                      0.0 |   -16.287714 |              0.0 |         36.161503 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.047084 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-03 21:00:00 | 0.05 | 17.8 |              0.0 |                      0.0 |             0.0 |             86.9 |       1.7 |              1016.0 | 0.0 | 0.0 |     18.65 |               82.7375 |                   4.5 |                     15.566134 |           175.016667 | 0.047 | 17.8 |     18.65 |    1.362705 |         1.127089 |           1.362705 |          1.7 |                20.413369 |                     21.532411 |                          0.0 |                      1.286025 |                           1.347418 |            17.739217 |                 17.815378 |     998.260783 |   1.208454 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                     52.505 |                 16.221809 |                      0.0 |   -16.221809 |              0.0 |         36.283192 |            125.297451 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.047647 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-03 22:00:00 | 0.05 | 17.0 |              0.0 |                      0.0 |             0.0 |             90.1 |       1.7 |              1017.0 | 0.0 | 0.0 | 18.808333 |             82.554167 |                   4.5 |                     15.566134 |           175.016667 | 0.047 | 17.0 | 18.808333 |    1.362705 |         1.142201 |           1.362705 |          1.7 |                19.406929 |                     21.746678 |                          0.0 |                      1.230421 |                           1.359123 |            17.485643 |                 17.952788 |     999.514357 |   1.213101 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                     52.505 |                 16.101364 |                      0.0 |   -16.101364 |              0.0 |         36.403636 |            125.297451 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.048116 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-03 23:00:00 | 0.05 | 16.4 |              0.0 |                      0.0 |             0.0 |             90.9 |       2.3 |              1017.0 | 0.0 | 0.0 | 18.941667 |             82.379167 |                   4.5 |                     15.566134 |           175.016667 | 0.047 | 16.4 | 18.941667 |     1.84366 |         1.185687 |            1.84366 |          2.3 |                 18.68084 |                     21.928555 |                          0.0 |                      1.190065 |                           1.369047 |            16.980884 |                 18.064561 |    1000.019116 |   1.215845 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                     52.505 |                 16.005095 |                      0.0 |   -16.005095 |              0.0 |         36.499905 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.048674 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 00:00:00 | 0.05 | 15.9 |              0.0 |                      0.0 |             0.0 |             88.0 |       2.3 |              1017.0 | 0.0 | 0.0 |      18.9 |             82.083333 |                   4.5 |                     15.566134 |           175.016667 | 0.047 | 15.9 |      18.9 |     1.84366 |         1.235786 |            1.84366 |          2.3 |                18.094032 |                     21.871576 |                          0.0 |                      1.157296 |                           1.365939 |            15.922748 |                 17.952919 |    1001.077252 |    1.21843 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                     52.505 |                 16.128293 |                      0.0 |   -16.128293 |              0.0 |         36.376707 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.048972 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 01:00:00 | 0.05 | 15.4 |              0.0 |                      0.0 |             0.0 |             89.2 |       2.2 |              1016.0 | 0.0 | 0.0 |     18.85 |             81.845833 |                   4.5 |                     15.566134 |           175.016667 | 0.047 | 15.4 |     18.85 |    1.763501 |         1.282546 |           1.763501 |          2.2 |                17.523416 |                     21.803372 |                          0.0 |                      1.125295 |                           1.362218 |            15.630887 |                 17.845152 |    1000.369113 |   1.219467 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                     52.505 |                 16.244859 |                      0.0 |   -16.244859 |              0.0 |         36.260141 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.049217 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 02:00:00 | 0.05 | 15.2 |              0.0 |                      0.0 |             0.0 |             88.8 |       2.3 |              1016.0 | 0.0 | 0.0 |      18.8 |                 81.55 |                   4.5 |                     15.566134 |           175.016667 | 0.047 | 15.2 |      18.8 |     1.84366 |         1.332645 |            1.84366 |          2.3 |                17.299618 |                     21.735354 |                          0.0 |                      1.112706 |                           1.358505 |            15.362061 |                 17.725181 |    1000.637939 |   1.220436 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                     52.505 |                 16.376926 |                      0.0 |   -16.376926 |              0.0 |         36.128074 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.049512 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 03:00:00 | 0.05 | 14.9 |              0.0 |                      0.0 |             0.0 |             89.7 |       2.0 |              1016.0 | 0.0 | 0.0 | 18.741667 |             81.258333 |                   4.5 |                     15.566134 |           175.016667 | 0.047 | 14.9 | 18.741667 |    1.603182 |         1.372725 |           1.603182 |          2.0 |                16.968612 |                     21.656234 |                          0.0 |                      1.094046 |                           1.354184 |            15.220845 |                 17.597495 |    1000.779155 |   1.221771 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                     52.505 |                 16.516685 |                      0.0 |   -16.516685 |              0.0 |         35.988315 |            106.502833 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.049743 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 04:00:00 | 0.05 | 14.3 |              0.0 |                 0.398044 |        1.590517 |             93.0 |       2.3 |              1016.0 | 0.0 | 0.0 | 18.670833 |             81.083333 |                   4.5 |                     15.534444 |           175.001952 | 0.047 | 14.3 | 18.670833 |     1.84366 |         1.422824 |            1.84366 |          2.3 |                16.323194 |                     21.560498 |                          0.0 |                      1.057522 |                           1.348953 |             15.18057 |                  17.48197 |     1000.81943 |    1.22434 | 0.0 | nan | 0.0 |              0.477155 |                  0.477155 |                  52.500585 |                  16.65652 |                      0.0 |   -16.179365 |              0.0 |         35.844065 |             92.611159 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.049925 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 05:00:00 | 0.05 | 14.1 |              0.0 |                      1.0 |       19.950903 |             93.6 |       2.5 |              1017.0 | 0.0 | 0.0 | 18.595833 |             80.920833 |                   4.5 |                     15.534444 |           174.919387 | 0.047 | 14.1 | 18.595833 |    2.003978 |         1.486284 |           2.003978 |          2.5 |                16.112886 |                     21.459534 |                          0.0 |                      1.045579 |                           1.343433 |            15.081661 |                 17.365233 |    1001.918339 |   1.226451 | 0.0 | nan | 0.0 |              5.985271 |                  5.985271 |                  52.475816 |                 16.779096 |                      0.0 |   -10.793826 |              0.0 |          35.69672 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.050164 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 06:00:00 | 0.05 | 14.3 |              0.1 |                      1.0 |       62.985775 |             94.6 |       2.4 |              1017.0 | 0.0 | 0.0 |    18.525 |             80.791667 |                   4.6 |                     15.534444 |            175.15812 | 0.047 | 14.3 |    18.525 |    1.923819 |         1.536383 |           1.923819 |          2.4 |                16.323194 |                     21.364558 |                          0.0 |                      1.057522 |                           1.338237 |            15.441741 |                 17.260782 |    1001.558259 |   1.225432 | 0.0 | nan | 0.0 |             18.895733 |                 18.895733 |                  52.547436 |                 17.089561 |                      0.0 |     1.806172 |              0.0 |         35.457875 |             88.752361 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.050224 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 07:00:00 | 0.05 | 15.4 |              0.9 |                      1.0 |      285.776389 |             93.3 |       2.5 |              1017.0 | 0.0 | 0.0 |    18.475 |             80.620833 |                   5.5 |                     15.534444 |           182.509935 | 0.047 | 15.4 |    18.475 |    2.003978 |         1.589823 |           2.003978 |          2.5 |                17.523416 |                     21.297737 |                          0.0 |                      1.125295 |                            1.33458 |            16.349347 |                 17.170413 |    1000.650653 |   1.220347 | 0.0 | nan | 0.0 |             85.732917 |                 85.732917 |                   54.75298 |                 19.012403 |                      0.0 |    66.720514 |              0.0 |         35.740578 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.050871 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 08:00:00 | 0.05 | 16.8 |              1.0 |                      1.0 |      475.225247 |             86.9 |       2.5 |              1017.0 | 0.0 | 0.0 |     18.45 |             80.208333 |                   6.5 |                     15.534444 |           195.188105 | 0.047 | 16.8 |     18.45 |    2.003978 |         1.643262 |           2.003978 |          2.5 |                19.162205 |                     21.264396 |                          0.0 |                      1.216842 |                           1.332754 |            16.651956 |                 17.055817 |    1000.348044 |   1.214317 | 0.0 | nan | 0.0 |            142.567574 |                142.567574 |                  58.556432 |                 21.208047 |                      0.0 |   121.359527 |              0.0 |         37.348385 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.053048 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 09:00:00 | 0.05 | 18.2 |              1.0 |                      1.0 |      614.006719 |             78.9 |       2.9 |              1017.0 | 0.0 | 0.0 | 18.416667 |             79.908333 |                   7.3 |                     15.534444 |           207.781609 | 0.047 | 18.2 | 18.416667 |    2.324615 |         1.696701 |           2.324615 |          2.9 |                20.933479 |                     21.220011 |                          0.0 |                      1.314615 |                           1.330323 |            16.516515 |                 16.956557 |    1000.483485 |   1.208543 | 0.0 | nan | 0.0 |            184.202016 |                184.202016 |                  62.334483 |                 22.996116 |                      0.0 |     161.2059 |              0.0 |         39.338367 |              73.45023 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.055504 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 10:00:00 | 0.05 | 19.5 |              0.9 |                      1.0 |      643.806508 |             75.8 |       3.1 |              1017.0 | 0.0 | 0.0 | 18.383333 |             79.866667 |                   7.7 |                     15.534444 |           213.707576 | 0.047 | 19.5 | 18.383333 |    2.484933 |         1.750141 |           2.484933 |          3.1 |                22.704676 |                     21.175708 |                          0.0 |                      1.411269 |                           1.327896 |            17.210144 |                 16.912332 |     999.789856 |   1.202862 | 0.0 | nan | 0.0 |            193.141952 |                193.141952 |                  64.112273 |                 23.882018 |                      0.0 |   169.259935 |              0.0 |         40.230255 |             68.711505 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.05667 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 11:00:00 | 0.05 | 20.3 |              0.8 |                      1.0 |      600.262981 |             73.4 |       3.3 |              1018.0 | 0.0 | 0.0 |   18.3375 |             79.933333 |                   7.8 |                     15.534444 |           213.092753 | 0.047 | 20.3 |   18.3375 |    2.645251 |          1.82028 |           2.645251 |          3.3 |                23.858503 |                     21.114922 |                          0.0 |                      1.473678 |                           1.324565 |            17.512141 |                 16.877861 |    1000.487859 |   1.200635 | 0.0 | nan | 0.0 |            180.078894 |                180.078894 |                  63.927826 |                   24.1295 |                      0.0 |   155.949394 |              0.0 |         39.798326 |             64.547172 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.056522 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 12:00:00 | 0.05 | 21.0 |              0.9 |                      1.0 |      620.514139 |             68.3 |       3.3 |              1018.0 | 0.0 | 0.0 |    18.325 |             79.966667 |                   7.9 |                     15.534444 |           212.841496 | 0.047 | 21.0 |    18.325 |    2.645251 |         1.887079 |           2.645251 |          3.3 |                24.909749 |                     21.098371 |                          0.0 |                      1.530181 |                           1.323658 |            17.013358 |                 16.871664 |    1000.986642 |   1.198001 | 0.0 | nan | 0.0 |            186.154242 |                186.154242 |                  63.852449 |                 24.341655 |                      0.0 |   161.812586 |              0.0 |         39.510794 |             64.547172 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.056563 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 13:00:00 | 0.05 | 21.8 |              0.8 |                      1.0 |      599.441386 |             61.3 |       3.2 |              1018.0 | 0.0 | 0.0 |    18.325 |             79.766667 |                   8.2 |                     15.534444 |           217.145994 | 0.047 | 21.8 |    18.325 |    2.565092 |         1.943859 |           2.565092 |          3.2 |                26.160453 |                     21.098371 |                          0.0 |                      1.596982 |                           1.323658 |            16.036358 |                 16.829467 |    1001.963642 |   1.195188 | 0.0 | nan | 0.0 |            179.832416 |                179.832416 |                  65.143798 |                 25.038979 |                      0.0 |   154.793437 |              0.0 |          40.10482 |             66.564271 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.057637 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 14:00:00 | 0.05 | 21.9 |              0.9 |                      1.0 |      613.551222 |             60.6 |       2.5 |              1018.0 | 0.0 | 0.0 | 18.283333 |                 79.65 |                   8.7 |                     15.534444 |           225.230587 | 0.047 | 21.9 | 18.283333 |    2.003978 |         1.963899 |           2.003978 |          2.5 |                26.320577 |                     21.043281 |                          0.0 |                      1.605502 |                           1.320638 |             15.95027 |                 16.760973 |     1002.04973 |   1.194821 | 0.0 | nan | 0.0 |            184.065367 |                184.065367 |                  67.569176 |                 26.187015 |                      0.0 |   157.878352 |              0.0 |         41.382161 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.059053 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 15:00:00 | 0.05 | 21.8 |              0.9 |                      1.0 |      517.107047 |             58.8 |       2.9 |              1017.0 | 0.0 | 0.0 | 18.245833 |                  79.5 |                   9.1 |                     15.534444 |           230.614741 | 0.047 | 21.8 | 18.245833 |    2.324615 |         1.997298 |           2.324615 |          2.9 |                26.160453 |                     20.993808 |                          0.0 |                      1.596982 |                           1.317924 |            15.382347 |                 16.690077 |    1001.617653 |   1.194298 | 0.0 | nan | 0.0 |            155.132114 |                155.132114 |                  69.184422 |                 27.142101 |                      0.0 |   127.990013 |              0.0 |         42.042321 |              73.45023 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.059976 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 16:00:00 | 0.05 | 22.2 |              0.9 |                      1.0 |      377.836617 |             57.2 |       3.6 |              1017.0 | 0.0 | 0.0 | 18.233333 |               79.3375 |                   9.5 |                     15.534444 |           234.753897 | 0.047 | 22.2 | 18.233333 |    2.885728 |         2.040718 |           2.885728 |          3.6 |                26.806088 |                      20.97734 |                          0.0 |                      1.631293 |                           1.317021 |            15.333082 |                 16.642897 |    1001.666918 |   1.192703 | 0.0 | nan | 0.0 |            113.350985 |                113.350985 |                  70.426169 |                 28.068928 |                      0.0 |    85.282057 |              0.0 |         42.357241 |             59.168241 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.060663 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 17:00:00 | 0.05 | 21.8 |              1.0 |                      1.0 |      252.204639 |             57.2 |       4.2 |              1017.0 | 0.0 | 0.0 | 18.229167 |             79.133333 |                  10.2 |                     15.534444 |           239.548315 | 0.047 | 21.8 | 18.229167 |    3.366683 |         2.100837 |           3.366683 |          4.2 |                26.160453 |                     20.971853 |                          0.0 |                      1.596982 |                            1.31672 |            14.963779 |                 16.595726 |    1002.036221 |   1.194485 | 0.0 | nan | 0.0 |             75.661392 |                 75.661392 |                  71.864495 |                 29.636995 |                      0.0 |    46.024396 |              0.0 |         42.227499 |             50.715635 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.061064 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 18:00:00 | 0.05 | 20.9 |              1.0 |                      1.0 |      128.576714 |             59.9 |       4.2 |              1017.0 | 0.0 | 0.0 | 18.208333 |               78.8375 |                  11.1 |                     15.534444 |           242.777315 | 0.047 | 20.9 | 18.208333 |    3.366683 |         2.157616 |           3.366683 |          4.2 |                 24.75714 |                     20.944437 |                          0.0 |                      1.521999 |                           1.315216 |            14.829527 |                  16.51207 |    1002.170473 |   1.198201 | 0.0 | nan | 0.0 |             38.573014 |                 38.573014 |                  72.833195 |                 31.712576 |                      0.0 |     6.860438 |              0.0 |         41.120619 |             50.715635 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.060566 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 19:00:00 | 0.05 | 19.7 |              1.0 |                      1.0 |       42.072586 |             66.1 |       3.6 |              1017.0 | 0.0 | 0.0 | 18.166667 |               78.4875 |                  12.1 |                     15.534444 |           243.962306 | 0.047 | 19.7 | 18.166667 |    2.885728 |         2.194356 |           2.885728 |          3.6 |                22.988466 |                     20.889699 |                          0.0 |                      1.426658 |                           1.312212 |            15.195376 |                 16.395802 |    1001.804624 |   1.202947 | 0.0 | nan | 0.0 |             12.621776 |                 12.621776 |                  73.188692 |                 34.091963 |                      0.0 |   -21.470187 |              0.0 |         39.096729 |             59.168241 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.059092 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 20:00:00 | 0.05 | 17.9 |              0.3 |                 0.104711 |        0.812897 |             71.7 |       3.2 |              1017.0 | 0.0 | 0.0 | 18.104167 |             78.091667 |                  12.4 |                     15.502754 |           243.988429 | 0.047 | 17.9 | 18.104167 |    2.565092 |         2.227756 |           2.565092 |          3.2 |                20.542326 |                     20.807826 |                          0.0 |                      1.293123 |                           1.307717 |            14.728848 |                 16.249178 |    1002.271152 |   1.210597 | 0.0 | nan | 0.0 |              0.243869 |                  0.243869 |                  73.196529 |                 35.120093 |                      0.0 |   -34.876224 |              0.0 |         38.076436 |             66.564271 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.058604 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 21:00:00 | 0.05 | 15.8 |              0.0 |                      0.0 |             0.0 |             80.6 |       2.2 |              1018.0 | 0.0 | 0.0 | 18.020833 |             77.829167 |                  12.4 |                     15.502754 |           243.988429 | 0.047 | 15.8 | 18.020833 |    1.763501 |         2.244455 |           1.763501 |          2.2 |                17.978626 |                     20.699099 |                          0.0 |                      1.150835 |                           1.301743 |            14.490772 |                 16.109936 |    1003.509228 |    1.22071 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                  73.196529 |                 35.424509 |                      0.0 |   -35.424509 |              0.0 |          37.77202 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.058562 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 22:00:00 | 0.05 | 14.8 |              0.0 |                      0.0 |             0.0 |             85.9 |       2.2 |              1018.0 | 0.0 | 0.0 | 17.929167 |             77.654167 |                  12.4 |                     15.502754 |           243.988429 | 0.047 | 14.8 | 17.929167 |    1.763501 |         2.261155 |           1.763501 |          2.2 |                16.859515 |                     20.580072 |                          0.0 |                      1.087886 |                           1.295199 |            14.482324 |                 15.981284 |    1003.517676 |   1.224953 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                  73.196529 |                 35.698198 |                      0.0 |   -35.698198 |              0.0 |          37.49833 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.058455 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-04 23:00:00 | 0.05 | 14.1 |              0.0 |                      0.0 |             0.0 |             89.3 |       2.1 |              1018.0 | 0.0 | 0.0 | 17.833333 |               77.5875 |                  12.4 |                     15.502754 |           243.988429 | 0.047 | 14.1 | 17.833333 |    1.683342 |         2.254475 |           1.683342 |          2.1 |                16.112886 |                     20.456276 |                          0.0 |                      1.045579 |                           1.288387 |            14.388807 |                 15.871513 |    1003.611193 |   1.227981 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                  73.196529 |                 35.921753 |                      0.0 |   -35.921753 |              0.0 |         37.274776 |             101.43127 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.058143 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 00:00:00 | 0.05 | 13.4 |              0.0 |                      0.0 |             0.0 |             91.4 |       1.8 |              1018.0 | 0.0 | 0.0 | 17.729167 |             77.729167 |                  12.4 |                     15.502754 |           243.988429 | 0.047 | 13.4 | 17.729167 |    1.442864 |         2.237776 |           1.442864 |          1.8 |                15.395377 |                     20.322453 |                          0.0 |                      1.004677 |                           1.281018 |            14.071374 |                 15.796473 |    1003.928626 |   1.231127 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                  73.196529 |                 36.050541 |                      0.0 |   -36.050541 |              0.0 |         37.145987 |            118.336481 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.057649 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 01:00:00 | 0.05 | 12.8 |              0.0 |                      0.0 |             0.0 |             93.1 |       1.8 |              1018.0 | 0.0 | 0.0 | 17.620833 |             77.891667 |                  12.4 |                     15.502754 |           243.988429 | 0.047 | 12.8 | 17.620833 |    1.442864 |         2.224416 |           1.442864 |          1.8 |                14.802811 |                     20.184089 |                          0.0 |                      0.970706 |                           1.273391 |            13.781417 |                 15.721724 |    1004.218583 |   1.233843 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                  73.196529 |                 36.176259 |                      0.0 |   -36.176259 |              0.0 |          37.02027 |            118.336481 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.057155 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 02:00:00 | 0.05 | 12.3 |              0.0 |                      0.0 |             0.0 |             94.6 |       1.6 |              1018.0 | 0.0 | 0.0 |      17.5 |             78.133333 |                  12.4 |                     15.502754 |           243.988429 | 0.047 | 12.3 |      17.5 |    1.282546 |         2.201036 |           1.282546 |          1.6 |                14.324376 |                     20.030733 |                          0.0 |                      0.943147 |                            1.26493 |            13.550859 |                 15.650679 |    1004.449141 |   1.236111 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                  73.196529 |                 36.284611 |                      0.0 |   -36.284611 |              0.0 |         36.911917 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.05653 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 03:00:00 | 0.05 | 12.0 |              0.0 |                      0.0 |             0.0 |             95.6 |       1.2 |              1018.0 | 0.0 | 0.0 | 17.379167 |             78.379167 |                  12.4 |                     15.502754 |           243.988429 | 0.047 | 12.0 | 17.379167 |    0.961909 |         2.174316 |           0.961909 |          1.2 |                 14.04387 |                     19.878396 |                          0.0 |                      0.926933 |                           1.256516 |            13.425939 |                 15.580521 |    1004.574061 |   1.237469 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                  73.196529 |                 36.390891 |                      0.0 |   -36.390891 |              0.0 |         36.805637 |            177.504722 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 |  0.05589 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 04:00:00 | 0.05 | 11.7 |              0.0 |                 0.366022 |        1.482347 |             96.2 |       1.9 |              1018.0 | 0.0 | 0.0 | 17.270833 |               78.5125 |                  12.4 |                     15.470733 |           243.983922 | 0.047 | 11.7 | 17.270833 |    1.523023 |         2.160956 |           1.523023 |          1.9 |                13.768192 |                      19.74268 |                          0.0 |                      0.910957 |                           1.249013 |            13.245001 |                 15.500472 |    1004.754999 |   1.238856 | 0.0 | nan | 0.0 |              0.444704 |                  0.444704 |                  73.195176 |                 36.589063 |                      0.0 |   -36.144359 |              0.0 |         36.606113 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.055364 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 05:00:00 | 0.05 | 11.6 |              0.0 |                      1.0 |       21.135567 |             95.7 |       1.3 |              1018.0 | 0.0 | 0.0 | 17.166667 |                  78.6 |                  12.4 |                     15.470733 |           244.033283 | 0.047 | 11.6 | 17.166667 |    1.042069 |         2.120877 |           1.042069 |          1.3 |                 13.67736 |                     19.612949 |                          0.0 |                      0.905683 |                           1.241834 |            13.089234 |                 15.415778 |    1004.910766 |   1.239363 | 0.0 | nan | 0.0 |               6.34067 |                   6.34067 |                  73.209985 |                  36.74524 |                      0.0 |    -30.40457 |              0.0 |         36.464745 |            163.850513 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.054792 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 06:00:00 | 0.05 | 12.6 |              0.6 |                      1.0 |       125.88345 |             95.1 |       1.6 |              1018.0 | 0.0 | 0.0 | 17.095833 |             78.620833 |                  12.9 |                     15.470733 |           246.654019 | 0.047 | 12.6 | 17.095833 |    1.282546 |         2.094157 |           1.282546 |          1.6 |                14.609782 |                     19.525158 |                          0.0 |                      0.959602 |                           1.236973 |            13.893903 |                 15.350842 |    1004.106097 |   1.234655 | 0.0 | nan | 0.0 |             37.765035 |                 37.765035 |                  73.996206 |                  38.00446 |                      0.0 |    -0.239425 |              0.0 |         35.991746 |            133.128542 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.054084 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 07:00:00 | 0.05 | 14.8 |              0.9 |                      1.0 |      305.954944 |             96.3 |       1.9 |              1019.0 | 0.0 | 0.0 | 17.070833 |             78.745833 |                  12.9 |                     15.470733 |           247.494792 | 0.047 | 14.8 | 17.070833 |    1.523023 |         2.074117 |           1.523023 |          1.9 |                16.859515 |                     19.494255 |                          0.0 |                      1.087886 |                           1.235261 |            16.235713 |                 15.350914 |    1002.764287 |   1.225361 | 0.0 | nan | 0.0 |             91.786483 |                 91.786483 |                  74.248438 |                 37.987562 |                      0.0 |    53.798921 |              0.0 |         36.260875 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.054083 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 08:00:00 | 0.05 | 17.1 |              1.0 |                      1.0 |      511.104575 |             89.9 |       3.2 |              1019.0 | 0.0 | 0.0 | 17.083333 |             78.870833 |                  12.9 |                     15.470733 |           248.989764 | 0.047 | 17.1 | 17.083333 |    2.565092 |         2.097497 |           2.565092 |          3.2 |                19.530313 |                     19.509701 |                          0.0 |                      1.237258 |                           1.236116 |            17.557751 |                 15.387464 |    1001.442249 |   1.215051 | 0.0 | nan | 0.0 |            153.331373 |                153.331373 |                  74.696929 |                 37.896217 |                      0.0 |   115.435156 |              0.0 |         36.800712 |             66.564271 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.054631 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 09:00:00 | 0.05 | 19.4 |              0.9 |                      1.0 |      629.519239 |             79.2 |       4.0 |              1019.0 | 0.0 | 0.0 | 17.133333 |             78.883333 |                  12.8 |                     15.470733 |           249.636119 | 0.047 | 19.4 | 17.133333 |    3.206365 |         2.134237 |           3.206365 |          4.0 |                22.563931 |                     19.571593 |                          0.0 |                      1.403627 |                           1.239544 |            17.870634 |                 15.438725 |    1001.129366 |   1.205358 | 0.0 | nan | 0.0 |            188.855772 |                188.855772 |                  74.890836 |                  37.56462 |                      0.0 |   151.291152 |              0.0 |         37.326216 |             53.251417 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.055387 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 10:00:00 | 0.05 | 20.6 |              1.0 |                      1.0 |      765.816589 |             73.4 |       4.6 |              1019.0 | 0.0 | 0.0 | 17.179167 |             78.783333 |                  12.9 |                     15.470733 |           254.719872 | 0.047 | 20.6 | 17.179167 |     3.68732 |         2.184336 |            3.68732 |          4.6 |                24.304197 |                     19.628477 |                          0.0 |                      1.497674 |                           1.242694 |            17.839281 |                 15.463969 |    1001.160719 |   1.200448 | 0.0 | nan | 0.0 |            229.744977 |                229.744977 |                  76.415962 |                 37.751868 |                      0.0 |   191.993109 |              0.0 |         38.664094 |              46.30558 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.057099 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 11:00:00 | 0.05 | 21.4 |              1.0 |                      1.0 |      767.145842 |             67.3 |       4.1 |              1019.0 | 0.0 | 0.0 |    17.225 |             78.529167 |                  13.1 |                     15.470733 |           261.673325 | 0.047 | 21.4 |    17.225 |    3.286524 |         2.211056 |           3.286524 |          4.1 |                25.528421 |                     19.685506 |                          0.0 |                      1.563281 |                            1.24585 |            17.180627 |                 15.458864 |    1001.819373 |   1.197482 | 0.0 | nan | 0.0 |            230.143753 |                230.143753 |                  78.501998 |                 38.247049 |                      0.0 |   191.896704 |              0.0 |         40.254949 |             51.952602 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.059074 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 12:00:00 | 0.05 | 22.5 |              1.0 |                      1.0 |      736.263853 |             62.2 |       4.1 |              1019.0 | 0.0 | 0.0 |   17.2875 |                78.275 |                  13.2 |                     15.470733 |            266.49623 | 0.047 | 22.5 |   17.2875 |    3.286524 |         2.237776 |           3.286524 |          4.1 |                27.299387 |                     19.763507 |                          0.0 |                      1.657431 |                           1.250165 |            16.980219 |                 15.469885 |    1002.019781 |   1.193116 | 0.0 | nan | 0.0 |            220.879156 |                220.879156 |                  79.948869 |                  38.48429 |                      0.0 |   182.394866 |              0.0 |         41.464579 |             51.952602 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.060713 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 13:00:00 | 0.05 | 23.1 |              1.0 |                      1.0 |      768.006183 |             59.6 |       4.4 |              1019.0 | 0.0 | 0.0 | 17.341667 |             78.204167 |                  13.4 |                     15.470733 |           273.519763 | 0.047 | 23.1 | 17.341667 |    3.527001 |         2.277855 |           3.527001 |          4.4 |                28.309776 |                     19.831326 |                          0.0 |                      1.710768 |                           1.253914 |            16.872626 |                 15.508923 |    1002.127374 |   1.190747 | 0.0 | nan | 0.0 |            230.401855 |                230.401855 |                  82.055929 |                  38.86309 |                      0.0 |   191.538765 |              0.0 |         43.192839 |             48.410379 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.062745 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 14:00:00 | 0.05 | 23.4 |              1.0 |                      1.0 |      735.929369 |             55.4 |       4.6 |              1019.0 | 0.0 | 0.0 | 17.404167 |               77.9875 |                  13.5 |                     15.470733 |           278.618852 | 0.047 | 23.4 | 17.404167 |     3.68732 |         2.347994 |            3.68732 |          4.6 |                28.827078 |                     19.909831 |                          0.0 |                      1.737973 |                           1.258253 |            15.970201 |                 15.527179 |    1003.029799 |   1.189943 | 0.0 | nan | 0.0 |            220.778811 |                220.778811 |                  83.585656 |                 39.079824 |                      0.0 |   181.698987 |              0.0 |         44.505832 |              46.30558 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.064717 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 15:00:00 | 0.05 | 23.8 |              1.0 |                      1.0 |      620.374953 |             52.1 |       4.5 |              1019.0 | 0.0 | 0.0 |   17.4875 |             77.708333 |                  13.6 |                     15.470733 |           282.921682 | 0.047 | 23.8 |   17.4875 |    3.607161 |         2.401434 |           3.607161 |          4.5 |                29.529613 |                     20.014927 |                          0.0 |                      1.774812 |                           1.264057 |            15.384928 |                 15.553266 |    1003.615072 |     1.1886 | 0.0 | nan | 0.0 |            186.112486 |                186.112486 |                  84.876505 |                 39.288738 |                      0.0 |   146.823747 |              0.0 |         45.587766 |             47.334593 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.066472 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 16:00:00 | 0.05 | 24.0 |              1.0 |                      1.0 |      450.095264 |             49.5 |       4.7 |              1018.0 | 0.0 | 0.0 |   17.5625 |               77.3875 |                  13.7 |                     15.470733 |           285.932459 | 0.047 | 24.0 |   17.5625 |    3.767479 |         2.438173 |           3.767479 |          4.7 |                29.886439 |                     20.109928 |                          0.0 |                      1.793476 |                             1.2693 |            14.793787 |                  15.56257 |    1003.206213 |   1.186889 | 0.0 | nan | 0.0 |            135.028579 |                135.028579 |                  85.779738 |                 39.539407 |                      0.0 |    95.489172 |              0.0 |          46.24033 |             45.320355 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.067757 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 17:00:00 | 0.05 | 24.0 |              1.0 |                      1.0 |      273.614142 |             48.1 |       4.5 |              1018.0 | 0.0 | 0.0 | 17.654167 |             77.008333 |                  13.7 |                     15.470733 |           286.824521 | 0.047 | 24.0 | 17.654167 |    3.607161 |         2.448193 |           3.607161 |          4.5 |                29.886439 |                     20.226575 |                          0.0 |                      1.793476 |                           1.275734 |            14.375377 |                 15.576148 |    1003.624623 |   1.187075 | 0.0 | nan | 0.0 |             82.084243 |                 82.084243 |                  86.047356 |                 39.564612 |                      0.0 |    42.519631 |              0.0 |         46.482744 |             47.334593 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.068576 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 18:00:00 | 0.05 | 23.7 |              1.0 |                      1.0 |      133.368106 |             49.4 |       4.1 |              1018.0 | 0.0 | 0.0 | 17.770833 |             76.570833 |                  13.7 |                     15.470733 |           287.024163 | 0.047 | 23.7 | 17.770833 |    3.286524 |         2.444853 |           3.286524 |          4.1 |                29.352596 |                      20.37589 |                          0.0 |                      1.765541 |                           1.283961 |            14.500182 |                 15.601989 |    1003.499818 |   1.188219 | 0.0 | nan | 0.0 |             40.010432 |                 40.010432 |                  86.107249 |                 39.572333 |                      0.0 |     0.438099 |              0.0 |         46.534916 |             51.952602 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.069229 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 19:00:00 | 0.05 | 22.8 |              1.0 |                      1.0 |       40.326503 |             57.0 |       3.8 |              1018.0 | 0.0 | 0.0 |      17.9 |             76.191667 |                  13.7 |                     15.470733 |           286.951409 | 0.047 | 22.8 |      17.9 |    3.046047 |         2.451533 |           3.046047 |          3.8 |                27.800581 |                     20.542326 |                          0.0 |                      1.683922 |                           1.293123 |            15.846331 |                 15.651541 |    1002.153669 |   1.191234 | 0.0 | nan | 0.0 |             12.097951 |                 12.097951 |                  86.085423 |                 39.521285 |                      0.0 |   -27.423334 |              0.0 |         46.564138 |             56.054123 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.069884 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 20:00:00 | 0.05 | 21.3 |              0.5 |                 0.072689 |        0.867122 |             65.9 |       2.5 |              1018.0 | 0.0 | 0.0 | 18.041667 |                 75.95 |                  13.9 |                     15.438712 |           286.953669 | 0.047 | 21.3 | 18.041667 |    2.003978 |         2.428153 |           2.003978 |          2.5 |                 25.37251 |                     20.726234 |                          0.0 |                       1.55495 |                           1.303234 |            16.720484 |                 15.741575 |    1001.279516 |   1.196911 | 0.0 | nan | 0.0 |              0.260137 |                  0.260137 |                  86.086101 |                 39.876809 |                      0.0 |   -39.616673 |              0.0 |         46.209291 |             85.202267 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.069848 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 21:00:00 | 0.05 | 19.3 |              0.0 |                      0.0 |             0.0 |             73.0 |       1.9 |              1018.0 | 0.0 | 0.0 |   18.1875 |             75.633333 |                  13.9 |                     15.438712 |           286.953669 | 0.047 | 19.3 |   18.1875 |    1.523023 |         2.418134 |           1.523023 |          1.9 |                22.423949 |                     20.917052 |                          0.0 |                       1.39602 |                           1.313713 |            16.369483 |                 15.820264 |    1001.630517 |   1.205255 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                  86.086101 |                 39.753451 |                      0.0 |   -39.753451 |              0.0 |          46.33265 |            112.108246 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.070447 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 22:00:00 | 0.05 | 18.3 |              0.0 |                      0.0 |             0.0 |             76.7 |       2.2 |              1018.0 | 0.0 | 0.0 | 18.333333 |                 75.25 |                  13.9 |                     15.438712 |           286.953669 | 0.047 | 18.3 | 18.333333 |    1.763501 |         2.418134 |           1.763501 |          2.2 |                21.065302 |                     21.109404 |                          0.0 |                      1.321845 |                           1.324263 |            16.157086 |                 15.884826 |    1001.842914 |   1.209486 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                  86.086101 |                 39.670785 |                      0.0 |   -39.670785 |              0.0 |         46.415315 |             96.820757 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.071143 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |
    | 1997-08-05 23:00:00 | 0.05 | 17.4 |              0.0 |                      0.0 |             0.0 |             81.8 |       2.7 |              1018.0 | 0.0 | 0.0 | 18.470833 |               74.9375 |                  13.9 |                     15.438712 |           286.953669 | 0.047 | 17.4 | 18.470833 |    2.164296 |         2.438173 |           2.164296 |          2.7 |                19.904589 |                     21.292177 |                          0.0 |                      1.257963 |                           1.334275 |            16.281953 |                 15.955825 |    1001.718047 |   1.213176 | 0.0 | nan | 0.0 |                   0.0 |                       0.0 |                  86.086101 |                 39.564038 |                      0.0 |   -39.564038 |              0.0 |         46.522063 |             78.890988 |                   nan |                      0.0 |                     0.0 |  0.0 |  0.0 | 0.071917 |    0.047 | 0.0 | 0.0 |   0.0 |          nan |          0.7 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |      0.0 |      0.0 |  0.0 |  0.0 |  0.0 |   0.0 |  nan |  0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |      0.0 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

    >>> inputs.nied.series = 0.0

.. _lland_v3_nadelw_summer_hourly:

conifers (summer)
-----------------

Now we repeat the daily integration test :ref:`lland_v3_nadelw_summer_daily` on an
hourly basis. In contrast to the calculation for water areas (see
:ref:`lland_v3_water_hourly`), both the interception evaporation (|EvI|) and the soil
evapotranspiration (|EvB|) rely on hourly input data (see method
|Return_PenmanMonteith_V1|) and thus show a diurnal pattern.  |lland_v3| always adds
condensed water to the interception storage (|Inzp|).  Hence, |EvI| is sometimes
negative at night, but |Inzp| dries out as soon as the sun starts to rise:

.. integration-test::

    >>> lnk(NADELW)
    >>> test("lland_v3_nadelw_summer_hourly",
    ...      axis1=(fluxes.evi, fluxes.evb), axis2=states.inzp,
    ...      use_conditions=conditions_nadelw_summer)
    |                date | nied | teml | sunshineduration | possiblesunshineduration | globalradiation | relativehumidity | windspeed | atmosphericpressure |  qz | qzh |   temltag | dailyrelativehumidity | dailysunshineduration | dailypossiblesunshineduration | dailyglobalradiation | nkor | tkor |   tkortag | windspeed2m | dailywindspeed2m | reducedwindspeed2m | windspeed10m | saturationvapourpressure | dailysaturationvapourpressure | saturationvapourpressuresnow | saturationvapourpressureslope | dailysaturationvapourpressureslope | actualvapourpressure | dailyactualvapourpressure | dryairpressure | densityair |    g |        tz |         wg | netshortwaveradiation | netshortwaveradiationsnow | dailynetshortwaveradiation | dailynetlongwaveradiation | netlongwaveradiationsnow | netradiation | netradiationsnow | dailynetradiation | aerodynamicresistance | soilsurfaceresistance | landusesurfaceresistance | actualsurfaceresistance | nbes | sbes |      evpo |       evi |      evb | evs | wnied | tempssurface | actualalbedo | schmpot | schm | gefrpot | gefr | wlatsnow | wsenssnow | wsurf | sff | fvg | wada | qdb | qib1 | qib2 | qbb | qkap | qdgz |      qah |       qa |     inzp | wats | waes | esnow | taus |        ebdn |      bowa | qdgz1 | qdgz2 | qigz1 | qigz2 | qbgz |    qdga1 | qdga2 | qiga1 | qiga2 | qbga | inlet |   outlet |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 1997-08-03 00:00:00 |  0.0 | 16.9 |              0.0 |                      0.0 |             0.0 |             95.1 |       0.8 |              1015.0 | 0.0 | 0.0 | 17.279167 |             85.883333 |                   1.3 |                          15.7 |           136.579167 |  0.0 | 16.9 | 17.279167 |    0.641273 |         1.605886 |           0.283992 |          0.8 |                19.284227 |                     19.753091 |                          0.0 |                      1.223615 |                           1.249589 |              18.3393 |                 16.964613 |       996.6607 |    1.21073 | -2.9 | 15.632162 |  -7.607027 |                   0.0 |                       0.0 |                 120.189667 |                 10.408237 |                      0.0 |   -10.408237 |              0.0 |         109.78143 |                 117.5 |                 100.0 |               170.875289 |               69.444444 |  0.0 |  0.0 | -0.001012 | -0.001012 |      0.0 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.221929 | 0.061647 | 0.001012 |  0.0 |  0.0 |   0.0 |  nan | 8729.609427 | 78.724819 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.221929 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.061647 |
    | 1997-08-03 01:00:00 |  0.0 | 16.6 |              0.0 |                      0.0 |             0.0 |             94.9 |       0.8 |              1015.0 | 0.0 | 0.0 | 17.241667 |             85.991667 |                   1.3 |                          15.7 |           136.579167 |  0.0 | 16.6 | 17.241667 |    0.641273 |         1.603439 |           0.283992 |          0.8 |                18.920184 |                      19.70628 |                          0.0 |                       1.20339 |                           1.246999 |            17.955254 |                 16.945759 |     997.044746 |   1.212158 | -2.9 | 15.688646 |  -5.468121 |                   0.0 |                       0.0 |                 120.189667 |                  10.41572 |                      0.0 |    -10.41572 |              0.0 |        109.773946 |                 117.5 |                 100.0 |               173.430608 |               69.444444 |  0.0 |  0.0 | -0.000643 | -0.000643 |      0.0 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.216196 | 0.060055 | 0.001655 |  0.0 |  0.0 |   0.0 |  nan | 8732.177548 | 78.724819 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.216196 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.060055 |
    | 1997-08-03 02:00:00 |  0.0 | 16.4 |              0.0 |                      0.0 |             0.0 |             95.9 |       0.8 |              1015.0 | 0.0 | 0.0 |    17.175 |             86.233333 |                   1.3 |                          15.7 |           136.579167 |  0.0 | 16.4 |    17.175 |    0.641273 |         1.575992 |           0.283992 |          0.8 |                 18.68084 |                       19.6233 |                          0.0 |                      1.190065 |                           1.242407 |            17.914926 |                 16.921826 |     997.085074 |   1.213014 | -2.9 | 15.719464 |  -4.083217 |                   0.0 |                       0.0 |                 120.189667 |                 10.421627 |                      0.0 |   -10.421627 |              0.0 |        109.768039 |                 117.5 |                 100.0 |               173.240199 |               69.444444 |  0.0 |  0.0 | -0.002622 | -0.002622 |      0.0 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.210612 | 0.058503 | 0.004277 |  0.0 |  0.0 |   0.0 |  nan | 8733.360765 | 78.724819 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.210612 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.058503 |
    | 1997-08-03 03:00:00 |  0.0 | 16.3 |              0.0 |                      0.0 |             0.0 |             96.7 |       0.8 |              1015.0 | 0.0 | 0.0 |   17.0625 |             86.708333 |                   1.3 |                          15.5 |           136.579167 |  0.0 | 16.3 |   17.0625 |    0.641273 |         1.548545 |           0.283992 |          0.8 |                18.562165 |                     19.483964 |                          0.0 |                      1.183449 |                            1.23469 |            17.949613 |                  16.89422 |     997.050387 |   1.213417 | -2.9 | 15.733663 |  -3.398025 |                   0.0 |                       0.0 |                 120.189667 |                 10.455184 |                      0.0 |   -10.455184 |              0.0 |        109.734483 |                 117.5 |                 100.0 |               172.676695 |               69.444444 |  0.0 |  0.0 | -0.004212 | -0.004212 |      0.0 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.205172 | 0.056992 | 0.008489 |  0.0 |  0.0 |   0.0 |  nan |  8733.85879 | 78.724819 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.205172 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.056992 |
    | 1997-08-03 04:00:00 |  0.0 | 16.0 |              0.0 |                 0.429734 |        1.943686 |             97.2 |       0.8 |              1015.0 | 0.0 | 0.0 | 16.908333 |             87.366667 |                   1.3 |                     14.929734 |            135.50182 |  0.0 | 16.0 | 16.908333 |    0.641273 |         1.504432 |           0.283992 |          0.8 |                18.210086 |                     19.294427 |                          0.0 |                      1.163788 |                           1.224181 |            17.700204 |                 16.856897 |     997.299796 |    1.21479 | -2.9 | 15.739639 |  -1.562167 |              1.710444 |                  0.317654 |                 119.241602 |                 10.555242 |                      0.0 |    -8.844799 |              0.0 |        108.686359 |                 117.5 |                 100.0 |               174.220893 |               93.330265 |  0.0 |  0.0 | -0.003891 | -0.003891 |      0.0 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.199872 |  0.05552 |  0.01238 |  0.0 |  0.0 |   0.0 |  nan | 8732.520957 | 78.724819 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.199872 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.05552 |
    | 1997-08-03 05:00:00 |  0.0 | 15.9 |              0.0 |                      1.0 |       21.932442 |             97.5 |       0.6 |              1015.0 | 0.0 | 0.0 | 16.729167 |             88.204167 |                   1.3 |                     14.929734 |           134.099005 |  0.0 | 15.9 | 16.729167 |    0.480955 |         1.453638 |           0.212994 |          0.6 |                18.094032 |                     19.076181 |                          0.0 |                      1.157296 |                           1.212063 |            17.641681 |                 16.825987 |     997.358319 |   1.215237 | -2.9 | 15.723585 |  -1.058491 |             19.300549 |                  3.584388 |                 118.007125 |                 10.545193 |                      0.0 |     8.755356 |              0.0 |        107.461932 |            156.666667 |                 100.0 |               174.539275 |              172.004132 |  0.0 |  0.0 |  0.008226 |  0.008226 |      0.0 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.194709 | 0.054086 | 0.004154 |  0.0 |  0.0 |   0.0 |  nan | 8730.679448 | 78.724819 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.194709 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.054086 |
    | 1997-08-03 06:00:00 |  0.0 | 16.0 |              0.0 |                      1.0 |       57.256186 |             97.7 |       0.9 |              1015.0 | 0.0 | 0.0 | 16.533333 |             89.191667 |                   1.3 |                     14.929734 |           131.855513 |  0.0 | 16.0 | 16.533333 |    0.721432 |         1.392031 |           0.319491 |          0.9 |                18.210086 |                     18.840106 |                          0.0 |                      1.163788 |                           1.198935 |            17.791254 |                 16.803805 |     997.208746 |   1.214748 | -2.9 | 15.701487 |   -1.79108 |             50.385444 |                  9.357297 |                 116.032852 |                  10.52516 |                      0.0 |    39.860284 |              0.0 |        105.507691 |            104.444444 |                 100.0 |               173.410784 |              170.929619 |  0.0 |  0.0 |  0.034696 |  0.004154 | 0.017612 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.189679 | 0.052689 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8729.570528 | 78.707207 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.189679 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.052689 |
    | 1997-08-03 07:00:00 |  0.0 | 16.6 |              0.0 |                      1.0 |      109.332844 |             97.4 |       0.9 |              1015.0 | 0.0 | 0.0 |    16.375 |                90.125 |                   1.3 |                     14.929734 |           129.465215 |  0.0 | 16.6 |    16.375 |    0.721432 |         1.334591 |           0.319491 |          0.9 |                18.920184 |                     18.651109 |                          0.0 |                       1.20339 |                           1.188408 |            18.428259 |                 16.809312 |     996.571741 |   1.211943 | -2.9 |  15.68818 |  -5.470922 |             96.212903 |                 17.868111 |                 113.929389 |                  10.49073 |                      0.0 |    85.722173 |              0.0 |        103.438659 |            104.444444 |                 100.0 |               169.265918 |              166.978951 |  0.0 |  0.0 |  0.073333 |       0.0 | 0.043009 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.18478 | 0.051328 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  8732.14145 | 78.664198 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.18478 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.051328 |
    | 1997-08-03 08:00:00 |  0.0 | 17.4 |              0.0 |                      1.0 |      170.949153 |             96.8 |       0.9 |              1016.0 | 0.0 | 0.0 | 16.216667 |                91.275 |                   1.3 |                     14.929734 |           127.329763 |  0.0 | 17.4 | 16.216667 |    0.721432 |         1.260484 |           0.319491 |          0.9 |                19.904589 |                     18.463773 |                          0.0 |                      1.257963 |                           1.177959 |            19.267642 |                 16.852809 |     996.732358 |   1.209425 | -2.9 | 15.719031 | -10.085816 |            150.435254 |                 27.937976 |                 112.050191 |                 10.426918 |                      0.0 |   140.008337 |              0.0 |        101.623274 |            104.444444 |                 100.0 |               164.539063 |              162.465759 |  0.0 |  0.0 |  0.121207 |       0.0 | 0.072568 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.180007 | 0.050002 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8739.327265 |  78.59163 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.180007 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.050002 |
    | 1997-08-03 09:00:00 |  0.0 | 19.0 |              0.2 |                      1.0 |      311.762625 |             86.1 |       1.3 |              1016.0 | 0.0 | 0.0 |   16.1125 |             91.991667 |                   1.5 |                     14.929734 |           129.903206 |  0.0 | 19.0 |   16.1125 |    1.042069 |         1.203904 |           0.461488 |          1.3 |                22.008543 |                     18.341425 |                          0.0 |                      1.373407 |                           1.171128 |            18.949356 |                 16.872582 |     997.050644 |   1.202945 | -2.9 | 15.805261 | -19.168437 |             274.35111 |                  50.95092 |                 114.314821 |                 10.804921 |                      0.0 |   263.546189 |              0.0 |          103.5099 |             72.307692 |                 100.0 |               175.700241 |              173.109073 |  0.0 |  0.0 |  0.277889 |       0.0 | 0.144323 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.175357 |  0.04871 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8755.595702 | 78.447307 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.175357 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.04871 |
    | 1997-08-03 10:00:00 |  0.0 | 20.3 |              0.5 |                      1.0 |        501.5833 |             76.8 |       1.5 |              1016.0 | 0.0 | 0.0 | 16.083333 |             92.241667 |                   1.7 |                     14.929734 |           132.285843 |  0.0 | 20.3 | 16.083333 |    1.202387 |         1.149836 |           0.532486 |          1.5 |                23.858503 |                     18.307295 |                          0.0 |                      1.473678 |                           1.169221 |             18.32333 |                 16.886954 |      997.67667 |   1.197896 | -2.9 | 16.000482 | -25.797109 |            441.393304 |                 81.973042 |                 116.411542 |                 11.199993 |                      0.0 |   430.193311 |              0.0 |        105.211549 |             62.666667 |                 100.0 |               197.930569 |              194.170635 |  0.0 |  0.0 |  0.483999 |       0.0 | 0.227714 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.170827 | 0.047452 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8778.492812 | 78.219593 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.170827 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.047452 |
    | 1997-08-03 11:00:00 |  0.0 | 21.4 |              0.7 |                      1.0 |      615.018728 |             71.8 |       1.2 |              1016.0 | 0.0 | 0.0 |    16.125 |             92.104167 |                   2.2 |                     14.929734 |           135.919957 |  0.0 | 21.4 |    16.125 |    0.961909 |         1.089916 |           0.425988 |          1.2 |                25.528421 |                     18.356069 |                          0.0 |                      1.563281 |                           1.171946 |            18.329406 |                 16.906704 |     997.670594 |    1.19342 | -2.9 | 16.275247 | -30.748518 |             541.21648 |                100.511632 |                 119.609562 |                 12.222123 |                      0.0 |   528.994357 |              0.0 |        107.387439 |             78.333333 |                 100.0 |               224.346316 |               218.96261 |  0.0 |  0.0 |  0.579295 |       0.0 | 0.287568 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.166415 | 0.046226 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8806.341329 | 77.932025 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.166415 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.046226 |
    | 1997-08-03 12:00:00 |  0.0 | 21.3 |              0.8 |                      1.0 |      626.544325 |             67.5 |       1.3 |              1016.0 | 0.0 | 0.0 | 16.204167 |             91.729167 |                   2.8 |                     14.929734 |           145.821804 |  0.0 | 21.3 | 16.204167 |    1.042069 |         1.037502 |           0.461488 |          1.3 |                 25.37251 |                     18.449053 |                          0.0 |                       1.55495 |                           1.177138 |            17.126444 |                 16.923163 |     998.873556 |   1.194363 | -2.9 | 16.609429 | -28.143424 |            551.359006 |                102.395244 |                 128.323187 |                 13.461134 |                      0.0 |   537.897872 |              0.0 |        114.862053 |             72.307692 |                 100.0 |               245.280266 |              238.430942 |  0.0 |  0.0 |  0.610907 |       0.0 | 0.279617 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.162116 | 0.045032 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8831.584754 | 77.652407 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.162116 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.045032 |
    | 1997-08-03 13:00:00 |  0.0 | 21.8 |              0.5 |                      1.0 |      496.133417 |             66.1 |       1.5 |              1016.0 | 0.0 | 0.0 | 16.329167 |             91.104167 |                   3.2 |                     14.929734 |           152.606529 |  0.0 | 21.8 | 16.329167 |    1.202387 |         1.012602 |           0.532486 |          1.5 |                26.160453 |                      18.59671 |                          0.0 |                      1.596982 |                           1.185375 |             17.29206 |                 16.942378 |      998.70794 |   1.192265 | -2.9 |  16.91235 | -29.325898 |            436.597407 |                 81.082376 |                 134.293746 |                 14.297428 |                      0.0 |   422.299979 |              0.0 |        119.996318 |             62.666667 |                 100.0 |               259.968591 |              251.997649 |  0.0 |  0.0 |  0.531993 |       0.0 | 0.222516 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.157929 | 0.043869 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8858.010651 | 77.429892 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.157929 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.043869 |
    | 1997-08-03 14:00:00 |  0.0 | 22.9 |              0.4 |                      1.0 |      419.520994 |             63.4 |       1.9 |              1016.0 | 0.0 | 0.0 | 16.545833 |             90.058333 |                   3.4 |                     14.929734 |           157.353237 |  0.0 | 22.9 | 16.545833 |    1.523023 |         0.984394 |           0.674482 |          1.9 |                27.969419 |                     18.855098 |                          0.0 |                      1.692831 |                           1.199769 |            17.732611 |                 16.980587 |     998.267389 |   1.187639 | -2.9 | 17.229461 | -34.023233 |            369.178475 |                 68.561717 |                 138.470849 |                 14.726001 |                      0.0 |   354.452474 |              0.0 |        123.744848 |             49.473684 |                 100.0 |               297.292109 |              286.129869 |  0.0 |  0.0 |  0.525997 |       0.0 | 0.183486 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.153849 | 0.042736 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8889.133884 | 77.246406 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.153849 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.042736 |
    | 1997-08-03 15:00:00 |  0.0 | 22.7 |              0.5 |                      1.0 |      387.887353 |             62.4 |       1.9 |              1016.0 | 0.0 | 0.0 | 16.816667 |             88.816667 |                   3.8 |                     14.929734 |           164.256877 |  0.0 | 22.7 | 16.816667 |    1.523023 |         0.968687 |           0.674482 |          1.9 |                27.632633 |                     19.182495 |                          0.0 |                      1.675052 |                           1.217969 |            17.242763 |                 17.037252 |     998.757237 |    1.18866 | -2.9 |  17.60294 |  -30.58236 |             341.34087 |                 63.391876 |                 144.546052 |                 15.562805 |                      0.0 |   325.778065 |              0.0 |        128.983247 |             49.473684 |                 100.0 |               302.770249 |              291.098843 |  0.0 |  0.0 |  0.500615 |       0.0 | 0.171914 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.149875 | 0.041632 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8916.816245 | 77.074493 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.149875 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.041632 |
    | 1997-08-03 16:00:00 |  0.0 | 22.5 |              0.5 |                      1.0 |      278.496872 |             61.1 |       2.3 |              1016.0 | 0.0 | 0.0 |   17.1375 |             87.333333 |                   4.1 |                     14.929734 |           170.073414 |  0.0 | 22.5 |   17.1375 |     1.84366 |         0.991339 |           0.816478 |          2.3 |                27.299387 |                     19.576758 |                          0.0 |                      1.657431 |                            1.23983 |            16.679926 |                 17.097035 |     999.320074 |   1.189715 | -2.9 | 17.935128 |  -27.38923 |            245.077248 |                 45.514346 |                 149.664604 |                 16.204965 |                      0.0 |   228.872282 |              0.0 |        133.459638 |             40.869565 |                 100.0 |               310.894515 |              298.448973 |  0.0 |  0.0 |  0.446523 |       0.0 | 0.133843 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.146004 | 0.040557 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8941.305475 | 76.940649 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.146004 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.040557 |
    | 1997-08-03 17:00:00 |  0.0 | 21.9 |              0.3 |                      1.0 |      137.138608 |             62.1 |       2.4 |              1016.0 | 0.0 | 0.0 |   17.4875 |                85.875 |                   4.4 |                     14.929734 |           173.470856 |  0.0 | 21.9 |   17.4875 |    1.923819 |         1.017332 |           0.851977 |          2.4 |                26.320577 |                     20.014927 |                          0.0 |                      1.605502 |                           1.264057 |            16.345078 |                 17.187818 |     999.654922 |   1.192283 | -2.9 | 18.228999 | -22.026006 |            120.681975 |                 22.412367 |                 152.654353 |                 16.818923 |                      0.0 |   103.863053 |              0.0 |         135.83543 |             39.166667 |                 100.0 |               291.443446 |              280.813362 |  0.0 |  0.0 |  0.321091 |       0.0 | 0.096392 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.142232 | 0.039509 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8960.431481 | 76.844258 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.142232 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.039509 |
    | 1997-08-03 18:00:00 |  0.0 | 21.4 |              0.1 |                      1.0 |       51.080714 |             67.0 |       2.5 |              1016.0 | 0.0 | 0.0 |   17.8375 |             84.620833 |                   4.5 |                     14.929734 |           174.440885 |  0.0 | 21.4 |   17.8375 |    2.003978 |         1.054998 |           0.887476 |          2.5 |                25.528421 |                     20.461645 |                          0.0 |                      1.563281 |                           1.288683 |            17.104042 |                 17.314814 |     998.895958 |   1.193968 | -2.9 | 18.458511 | -17.648933 |             44.951028 |                  8.348048 |                 153.507979 |                 16.973018 |                      0.0 |     27.97801 |              0.0 |        136.534961 |                  37.6 |                 100.0 |               252.715137 |              245.307638 |  0.0 |  0.0 |  0.222274 |       0.0 | 0.070579 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.138558 | 0.038488 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8975.180414 | 76.773678 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.138558 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.038488 |
    | 1997-08-03 19:00:00 |  0.0 | 20.7 |              0.0 |                      1.0 |       13.632817 |             74.5 |       2.5 |              1016.0 | 0.0 | 0.0 | 18.170833 |                  83.7 |                   4.5 |                     15.429734 |           175.008919 |  0.0 | 20.7 | 18.170833 |    2.003978 |         1.100997 |           0.887476 |          2.5 |                24.454368 |                     20.895167 |                          0.0 |                      1.505746 |                           1.312512 |            18.218504 |                 17.489255 |     997.781496 |   1.196313 | -2.9 | 18.635498 |  -12.38701 |             11.996879 |                  2.227992 |                 154.007849 |                 16.556776 |                      0.0 |    -4.559897 |              0.0 |        137.451073 |                  37.6 |                 100.0 |               212.733181 |              208.094539 |  0.0 |  0.0 |  0.142971 |       0.0 | 0.049855 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.134979 | 0.037494 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8984.667424 | 76.723823 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.134979 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.037494 |
    | 1997-08-03 20:00:00 |  0.0 | 19.4 |              0.0 |                   0.1364 |        0.185944 |             81.2 |       2.2 |              1016.0 | 0.0 | 0.0 | 18.454167 |             83.066667 |                   4.5 |                     15.566134 |           175.016667 |  0.0 | 19.4 | 18.454167 |    1.763501 |         1.120309 |           0.780979 |          2.2 |                22.563931 |                      21.26995 |                          0.0 |                      1.403627 |                           1.333058 |            18.321912 |                 17.668238 |     997.678088 |   1.201582 | -2.9 | 18.749342 |  -3.903945 |              0.163631 |                  0.030389 |                 154.014667 |                 16.343497 |                      0.0 |   -16.179866 |              0.0 |         137.67117 |             42.727273 |                 100.0 |               190.511989 |               75.961192 |  0.0 |  0.0 |   0.07766 |       0.0 | 0.047497 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.131493 | 0.036526 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  8985.67137 | 76.676326 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.131493 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.036526 |
    | 1997-08-03 21:00:00 |  0.0 | 17.8 |              0.0 |                      0.0 |             0.0 |             86.9 |       1.7 |              1016.0 | 0.0 | 0.0 |     18.65 |               82.7375 |                   4.5 |                     15.566134 |           175.016667 |  0.0 | 17.8 |     18.65 |    1.362705 |         1.127089 |           0.603484 |          1.7 |                20.413369 |                     21.532411 |                          0.0 |                      1.286025 |                           1.347418 |            17.739217 |                 17.815378 |     998.260783 |   1.208454 | -2.9 |  18.76139 |   5.768339 |                   0.0 |                       0.0 |                 154.014667 |                 16.221809 |                      0.0 |   -16.221809 |              0.0 |        137.792858 |             55.294118 |                 100.0 |               185.548772 |               69.444444 |  0.0 |  0.0 |  0.033662 |       0.0 | 0.022645 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.128096 | 0.035582 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8977.003031 | 76.653681 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.128096 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.035582 |
    | 1997-08-03 22:00:00 |  0.0 | 17.0 |              0.0 |                      0.0 |             0.0 |             90.1 |       1.7 |              1017.0 | 0.0 | 0.0 | 18.808333 |             82.554167 |                   4.5 |                     15.566134 |           175.016667 |  0.0 | 17.0 | 18.808333 |    1.362705 |         1.142201 |           0.603484 |          1.7 |                19.406929 |                     21.746678 |                          0.0 |                      1.230421 |                           1.359123 |            17.485643 |                 17.952788 |     999.514357 |   1.213101 | -2.9 |  18.65737 |   9.944218 |                   0.0 |                       0.0 |                 154.014667 |                 16.101364 |                      0.0 |   -16.101364 |              0.0 |        137.913303 |             55.294118 |                 100.0 |               184.343649 |               69.444444 |  0.0 |  0.0 |  0.020833 |       0.0 | 0.013897 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.124787 | 0.034663 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8964.158813 | 76.639784 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.124787 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.034663 |
    | 1997-08-03 23:00:00 |  0.0 | 16.4 |              0.0 |                      0.0 |             0.0 |             90.9 |       2.3 |              1017.0 | 0.0 | 0.0 | 18.941667 |             82.379167 |                   4.5 |                     15.566134 |           175.016667 |  0.0 | 16.4 | 18.941667 |     1.84366 |         1.185687 |           0.816478 |          2.3 |                 18.68084 |                     21.928555 |                          0.0 |                      1.190065 |                           1.369047 |            16.980884 |                 18.064561 |    1000.019116 |   1.215845 | -2.9 | 18.503239 |  12.619435 |                   0.0 |                       0.0 |                 154.014667 |                 16.005095 |                      0.0 |   -16.005095 |              0.0 |        138.009572 |             40.869565 |                 100.0 |               187.254752 |               69.444444 |  0.0 |  0.0 |  0.027767 |       0.0 | 0.016654 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.121564 | 0.033768 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8948.639378 | 76.623131 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.121564 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.033768 |
    | 1997-08-04 00:00:00 |  0.0 | 15.9 |              0.0 |                      0.0 |             0.0 |             88.0 |       2.3 |              1017.0 | 0.0 | 0.0 |      18.9 |             82.083333 |                   4.5 |                     15.566134 |           175.016667 |  0.0 | 15.9 |      18.9 |     1.84366 |         1.235786 |           0.816478 |          2.3 |                18.094032 |                     21.871576 |                          0.0 |                      1.157296 |                           1.365939 |            15.922748 |                 17.952919 |    1001.077252 |    1.21843 | -2.9 | 18.317006 |  14.502035 |                   0.0 |                       0.0 |                 154.014667 |                 16.128293 |                      0.0 |   -16.128293 |              0.0 |        137.886374 |             40.869565 |                 100.0 |               196.846938 |               69.444444 |  0.0 |  0.0 |  0.041241 |       0.0 | 0.024575 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.118424 | 0.032895 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8931.237343 | 76.598556 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.118424 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.032895 |
    | 1997-08-04 01:00:00 |  0.0 | 15.4 |              0.0 |                      0.0 |             0.0 |             89.2 |       2.2 |              1016.0 | 0.0 | 0.0 |     18.85 |             81.845833 |                   4.5 |                     15.566134 |           175.016667 |  0.0 | 15.4 |     18.85 |    1.763501 |         1.282546 |           0.780979 |          2.2 |                17.523416 |                     21.803372 |                          0.0 |                      1.125295 |                           1.362218 |            15.630887 |                 17.845152 |    1000.369113 |   1.219467 | -2.9 | 18.108181 |  16.249089 |                   0.0 |                       0.0 |                 154.014667 |                 16.244859 |                      0.0 |   -16.244859 |              0.0 |        137.769808 |             42.727273 |                 100.0 |               198.631945 |               69.444444 |  0.0 |  0.0 |  0.032757 |       0.0 |  0.01971 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.115365 | 0.032046 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8912.088255 | 76.578846 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.115365 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.032046 |
    | 1997-08-04 02:00:00 |  0.0 | 15.2 |              0.0 |                      0.0 |             0.0 |             88.8 |       2.3 |              1016.0 | 0.0 | 0.0 |      18.8 |                 81.55 |                   4.5 |                     15.566134 |           175.016667 |  0.0 | 15.2 |      18.8 |     1.84366 |         1.332645 |           0.816478 |          2.3 |                17.299618 |                     21.735354 |                          0.0 |                      1.112706 |                           1.358505 |            15.362061 |                 17.725181 |    1000.637939 |   1.220436 | -2.9 | 17.878392 |  16.070354 |                   0.0 |                       0.0 |                 154.014667 |                 16.376926 |                      0.0 |   -16.376926 |              0.0 |        137.637741 |             40.869565 |                 100.0 |               201.151424 |               69.444444 |  0.0 |  0.0 |  0.036313 |       0.0 |  0.02144 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.112385 | 0.031218 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |   8893.1179 | 76.557406 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.112385 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.031218 |
    | 1997-08-04 03:00:00 |  0.0 | 14.9 |              0.0 |                      0.0 |             0.0 |             89.7 |       2.0 |              1016.0 | 0.0 | 0.0 | 18.741667 |             81.258333 |                   4.5 |                     15.566134 |           175.016667 |  0.0 | 14.9 | 18.741667 |    1.603182 |         1.372725 |           0.709981 |          2.0 |                16.968612 |                     21.656234 |                          0.0 |                      1.094046 |                           1.354184 |            15.220845 |                 17.597495 |    1000.779155 |   1.221771 | -2.9 | 17.650748 |  16.504489 |                   0.0 |                       0.0 |                 154.014667 |                 16.516685 |                      0.0 |   -16.516685 |              0.0 |        137.497982 |                  47.0 |                 100.0 |               202.118359 |               69.444444 |  0.0 |  0.0 |  0.026013 |       0.0 | 0.016089 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.109482 | 0.030412 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8873.713411 | 76.541317 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.109482 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.030412 |
    | 1997-08-04 04:00:00 |  0.0 | 14.3 |              0.0 |                 0.398044 |        1.590517 |             93.0 |       2.3 |              1016.0 | 0.0 | 0.0 | 18.670833 |             81.083333 |                   4.5 |                     15.534444 |           175.001952 |  0.0 | 14.3 | 18.670833 |     1.84366 |         1.422824 |           0.816478 |          2.3 |                16.323194 |                     21.560498 |                          0.0 |                      1.057522 |                           1.348953 |             15.18057 |                  17.48197 |     1000.81943 |    1.22434 | -2.9 | 17.417894 |  18.707366 |              1.399655 |                  0.259936 |                 154.001717 |                  16.65652 |                      0.0 |   -15.256865 |              0.0 |        137.345197 |             40.869565 |                 100.0 |               201.757278 |               93.626391 |  0.0 |  0.0 |  0.016871 |       0.0 | 0.008597 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.106654 | 0.029626 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8852.106046 | 76.532721 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.106654 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.029626 |
    | 1997-08-04 05:00:00 |  0.0 | 14.1 |              0.0 |                      1.0 |       19.950903 |             93.6 |       2.5 |              1017.0 | 0.0 | 0.0 | 18.595833 |             80.920833 |                   4.5 |                     15.534444 |           174.919387 |  0.0 | 14.1 | 18.595833 |    2.003978 |         1.486284 |           0.887476 |          2.5 |                16.112886 |                     21.459534 |                          0.0 |                      1.045579 |                           1.343433 |            15.081661 |                 17.365233 |    1001.918339 |   1.226451 | -2.9 | 17.158606 |  18.351635 |             17.556794 |                  3.260548 |                 153.929061 |                 16.779096 |                      0.0 |     0.777698 |              0.0 |        137.149965 |                  37.6 |                 100.0 |               202.695817 |              198.661729 |  0.0 |  0.0 |  0.029889 |       0.0 | 0.009285 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.103899 | 0.028861 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  8830.85441 | 76.523435 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.103899 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.028861 |
    | 1997-08-04 06:00:00 |  0.0 | 14.3 |              0.1 |                      1.0 |       62.985775 |             94.6 |       2.4 |              1017.0 | 0.0 | 0.0 |    18.525 |             80.791667 |                   4.6 |                     15.534444 |            175.15812 |  0.0 | 14.3 |    18.525 |    1.923819 |         1.536383 |           0.851977 |          2.4 |                16.323194 |                     21.364558 |                          0.0 |                      1.057522 |                           1.338237 |            15.441741 |                 17.260782 |    1001.558259 |   1.225432 | -2.9 | 16.903586 |  15.621518 |             55.427482 |                 10.293675 |                 154.139146 |                 17.089561 |                      0.0 |    38.337921 |              0.0 |        137.049585 |             39.166667 |                 100.0 |               199.048483 |              195.224978 |  0.0 |  0.0 |  0.056019 |       0.0 | 0.018148 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.101215 | 0.028115 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8812.332893 | 76.505287 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.101215 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.028115 |
    | 1997-08-04 07:00:00 |  0.0 | 15.4 |              0.9 |                      1.0 |      285.776389 |             93.3 |       2.5 |              1017.0 | 0.0 | 0.0 |    18.475 |             80.620833 |                   5.5 |                     15.534444 |           182.509935 |  0.0 | 15.4 |    18.475 |    2.003978 |         1.589823 |           0.887476 |          2.5 |                17.523416 |                     21.297737 |                          0.0 |                      1.125295 |                            1.33458 |            16.349347 |                 17.170413 |    1000.650653 |   1.220347 | -2.9 | 16.681328 |   7.687968 |            251.483222 |                 46.704027 |                 160.608743 |                 19.012403 |                      0.0 |    232.47082 |              0.0 |         141.59634 |                  37.6 |                 100.0 |               191.288673 |               187.89699 |  0.0 |  0.0 |  0.234872 |       0.0 | 0.077957 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.098601 | 0.027389 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8801.744925 |  76.42733 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.098601 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.027389 |
    | 1997-08-04 08:00:00 |  0.0 | 16.8 |              1.0 |                      1.0 |      475.225247 |             86.9 |       2.5 |              1017.0 | 0.0 | 0.0 |     18.45 |             80.208333 |                   6.5 |                     15.534444 |           195.188105 |  0.0 | 16.8 |     18.45 |    2.003978 |         1.643262 |           0.887476 |          2.5 |                19.162205 |                     21.264396 |                          0.0 |                      1.216842 |                           1.332754 |            16.651956 |                 17.055817 |    1000.348044 |   1.214317 | -2.9 | 16.554272 |  -1.474365 |            418.198218 |                 77.665383 |                 171.765533 |                 21.208047 |                      0.0 |   396.990171 |              0.0 |        150.557486 |                  37.6 |                 100.0 |               192.879382 |              189.400978 |  0.0 |  0.0 |  0.424445 |       0.0 | 0.144461 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.096054 | 0.026682 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |  8800.31929 | 76.282869 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.096054 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.026682 |
    | 1997-08-04 09:00:00 |  0.0 | 18.2 |              1.0 |                      1.0 |      614.006719 |             78.9 |       2.9 |              1017.0 | 0.0 | 0.0 | 18.416667 |             79.908333 |                   7.3 |                     15.534444 |           207.781609 |  0.0 | 18.2 | 18.416667 |    2.324615 |         1.696701 |           1.029472 |          2.9 |                20.933479 |                     21.220011 |                          0.0 |                      1.314615 |                           1.330323 |            16.516515 |                 16.956557 |    1000.483485 |   1.208543 | -2.9 | 16.537165 |  -9.977011 |            540.325913 |                100.346241 |                 182.847816 |                 22.996116 |                      0.0 |   517.329797 |              0.0 |          159.8517 |             32.413793 |                 100.0 |                203.80698 |              199.707772 |  0.0 |  0.0 |  0.613301 |       0.0 | 0.189606 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.093572 | 0.025992 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8807.396301 | 76.093264 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.093572 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.025992 |
    | 1997-08-04 10:00:00 |  0.0 | 19.5 |              0.9 |                      1.0 |      643.806508 |             75.8 |       3.1 |              1017.0 | 0.0 | 0.0 | 18.383333 |             79.866667 |                   7.7 |                     15.534444 |           213.707576 |  0.0 | 19.5 | 18.383333 |    2.484933 |         1.750141 |            1.10047 |          3.1 |                22.704676 |                     21.175708 |                          0.0 |                      1.411269 |                           1.327896 |            17.210144 |                 16.912332 |     999.789856 |   1.202862 | -2.9 | 16.622089 | -17.267466 |            566.549727 |                105.216378 |                 188.062667 |                 23.882018 |                      0.0 |   542.667709 |              0.0 |        164.180649 |             30.322581 |                 100.0 |                207.85156 |              203.511519 |  0.0 |  0.0 |   0.68553 |       0.0 | 0.206717 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.091155 | 0.025321 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8821.763767 | 75.886546 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.091155 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.025321 |
    | 1997-08-04 11:00:00 |  0.0 | 20.3 |              0.8 |                      1.0 |      600.262981 |             73.4 |       3.3 |              1018.0 | 0.0 | 0.0 |   18.3375 |             79.933333 |                   7.8 |                     15.534444 |           213.092753 |  0.0 | 20.3 |   18.3375 |    2.645251 |          1.82028 |           1.171468 |          3.3 |                23.858503 |                     21.114922 |                          0.0 |                      1.473678 |                           1.324565 |            17.512141 |                 16.877861 |    1000.487859 |   1.200635 | -2.9 | 16.794499 | -21.033009 |            528.231423 |                 98.100121 |                 187.521623 |                   24.1295 |                      0.0 |   504.101923 |              0.0 |        163.392123 |             28.484848 |                 100.0 |               216.995805 |              212.089353 |  0.0 |  0.0 |  0.688425 |       0.0 | 0.197417 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.088801 | 0.024667 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8839.896776 | 75.689129 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.088801 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.024667 |
    | 1997-08-04 12:00:00 |  0.0 | 21.0 |              0.9 |                      1.0 |      620.514139 |             68.3 |       3.3 |              1018.0 | 0.0 | 0.0 |    18.325 |             79.966667 |                   7.9 |                     15.534444 |           212.841496 |  0.0 | 21.0 |    18.325 |    2.645251 |         1.887079 |           1.171468 |          3.3 |                24.909749 |                     21.098371 |                          0.0 |                      1.530181 |                           1.323658 |            17.013358 |                 16.871664 |    1000.986642 |   1.198001 | -2.9 | 17.012095 | -23.927432 |            546.052442 |                101.409739 |                 187.300516 |                 24.341655 |                      0.0 |   521.710787 |              0.0 |        162.958861 |             28.484848 |                 100.0 |               245.424582 |              238.564611 |  0.0 |  0.0 |  0.754088 |       0.0 | 0.202171 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.086507 |  0.02403 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8860.924208 | 75.486958 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.086507 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.02403 |
    | 1997-08-04 13:00:00 |  0.0 | 21.8 |              0.8 |                      1.0 |      599.441386 |             61.3 |       3.2 |              1018.0 | 0.0 | 0.0 |    18.325 |             79.766667 |                   8.2 |                     15.534444 |           217.145994 |  0.0 | 21.8 |    18.325 |    2.565092 |         1.943859 |           1.135969 |          3.2 |                26.160453 |                     21.098371 |                          0.0 |                      1.596982 |                           1.323658 |            16.036358 |                 16.829467 |    1001.963642 |   1.195188 | -2.9 | 17.264424 | -27.213457 |             527.50842 |                 97.965849 |                 191.088475 |                 25.038979 |                      0.0 |   502.469441 |              0.0 |        166.049497 |                29.375 |                 100.0 |               301.588452 |              290.027764 |  0.0 |  0.0 |  0.791888 |       0.0 | 0.191434 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.084272 | 0.023409 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8885.237665 | 75.295524 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.084272 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.023409 |
    | 1997-08-04 14:00:00 |  0.0 | 21.9 |              0.9 |                      1.0 |      613.551222 |             60.6 |       2.5 |              1018.0 | 0.0 | 0.0 | 18.283333 |                 79.65 |                   8.7 |                     15.534444 |           225.230587 |  0.0 | 21.9 | 18.283333 |    2.003978 |         1.963899 |           0.887476 |          2.5 |                26.320577 |                     21.043281 |                          0.0 |                      1.605502 |                           1.320638 |             15.95027 |                 16.760973 |     1002.04973 |   1.194821 | -2.9 | 17.556185 | -26.062888 |            539.925076 |                  100.2718 |                 198.202917 |                 26.187015 |                      0.0 |   513.738061 |              0.0 |        172.015902 |                  37.6 |                 100.0 |               310.080531 |              297.713573 |  0.0 |  0.0 |  0.744374 |       0.0 | 0.208908 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.082096 | 0.022804 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8908.400553 | 75.086616 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.082096 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.022804 |
    | 1997-08-04 15:00:00 |  0.0 | 21.8 |              0.9 |                      1.0 |      517.107047 |             58.8 |       2.9 |              1017.0 | 0.0 | 0.0 | 18.245833 |                  79.5 |                   9.1 |                     15.534444 |           230.614741 |  0.0 | 21.8 | 18.245833 |    2.324615 |         1.997298 |           1.029472 |          2.9 |                26.160453 |                     20.993808 |                          0.0 |                      1.596982 |                           1.317924 |            15.382347 |                 16.690077 |    1001.617653 |   1.194298 | -2.9 |  17.83414 |  -23.79516 |            455.054202 |                 84.510066 |                 202.940972 |                 27.142101 |                      0.0 |     427.9121 |              0.0 |        175.798871 |             32.413793 |                 100.0 |               324.683496 |              310.872289 |  0.0 |  0.0 |  0.707691 |       0.0 |  0.17377 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.079975 | 0.022215 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8929.295714 | 74.912846 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.079975 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.022215 |
    | 1997-08-04 16:00:00 |  0.0 | 22.2 |              0.9 |                      1.0 |      377.836617 |             57.2 |       3.6 |              1017.0 | 0.0 | 0.0 | 18.233333 |               79.3375 |                   9.5 |                     15.534444 |           234.753897 |  0.0 | 22.2 | 18.233333 |    2.885728 |         2.040718 |           1.277965 |          3.6 |                26.806088 |                      20.97734 |                          0.0 |                      1.631293 |                           1.317021 |            15.333082 |                 16.642897 |    1001.666918 |   1.192703 | -2.9 | 18.084882 | -24.690709 |            332.496223 |                 61.749299 |                  206.58343 |                 28.068928 |                      0.0 |   304.427295 |              0.0 |        178.514502 |             26.111111 |                 100.0 |               351.945528 |               335.24437 |  0.0 |  0.0 |  0.668882 |       0.0 | 0.134178 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.077909 | 0.021641 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8951.086422 | 74.778667 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.077909 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.021641 |
    | 1997-08-04 17:00:00 |  0.0 | 21.8 |              1.0 |                      1.0 |      252.204639 |             57.2 |       4.2 |              1017.0 | 0.0 | 0.0 | 18.229167 |             79.133333 |                  10.2 |                     15.534444 |           239.548315 |  0.0 | 21.8 | 18.229167 |    3.366683 |         2.100837 |            1.49096 |          4.2 |                26.160453 |                     20.971853 |                          0.0 |                      1.596982 |                            1.31672 |            14.963779 |                 16.595726 |    1002.036221 |   1.194485 | -2.9 |  18.34637 | -20.721778 |            221.940082 |                 41.217444 |                 210.802517 |                 29.636995 |                      0.0 |   192.303087 |              0.0 |        181.165522 |             22.380952 |                 100.0 |               341.498675 |              325.934588 |  0.0 |  0.0 |  0.608204 |       0.0 | 0.109857 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.075897 | 0.021082 |      0.0 |  0.0 |  0.0 |   0.0 |  nan |   8968.9082 |  74.66881 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.075897 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.021082 |
    | 1997-08-04 18:00:00 |  0.0 | 20.9 |              1.0 |                      1.0 |      128.576714 |             59.9 |       4.2 |              1017.0 | 0.0 | 0.0 | 18.208333 |               78.8375 |                  11.1 |                     15.534444 |           242.777315 |  0.0 | 20.9 | 18.208333 |    3.366683 |         2.157616 |            1.49096 |          4.2 |                 24.75714 |                     20.944437 |                          0.0 |                      1.521999 |                           1.315216 |            14.829527 |                  16.51207 |    1002.170473 |   1.198201 | -2.9 | 18.560232 |  -14.03861 |            113.147508 |                 21.013109 |                 213.644037 |                 31.712576 |                      0.0 |    81.434932 |              0.0 |        181.931461 |             22.380952 |                 100.0 |               298.901864 |               287.59108 |  0.0 |  0.0 |  0.461374 |       0.0 | 0.089881 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.073936 | 0.020538 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8980.046809 |  74.57893 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.073936 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.020538 |
    | 1997-08-04 19:00:00 |  0.0 | 19.7 |              1.0 |                      1.0 |       42.072586 |             66.1 |       3.6 |              1017.0 | 0.0 | 0.0 | 18.166667 |               78.4875 |                  12.1 |                     15.534444 |           243.962306 |  0.0 | 19.7 | 18.166667 |    2.885728 |         2.194356 |           1.277965 |          3.6 |                22.988466 |                     20.889699 |                          0.0 |                      1.426658 |                           1.312212 |            15.195376 |                 16.395802 |    1001.804624 |   1.202947 | -2.9 | 18.693895 |   -6.03663 |             37.023876 |                  6.875863 |                 214.686829 |                 34.091963 |                      0.0 |     2.931913 |              0.0 |        180.594866 |             26.111111 |                 100.0 |                249.92486 |              242.729144 |  0.0 |  0.0 |  0.273023 |       0.0 | 0.065773 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.072026 | 0.020007 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8983.183439 | 74.513157 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.072026 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.020007 |
    | 1997-08-04 20:00:00 |  0.0 | 17.9 |              0.3 |                 0.104711 |        0.812897 |             71.7 |       3.2 |              1017.0 | 0.0 | 0.0 | 18.104167 |             78.091667 |                  12.4 |                     15.502754 |           243.988429 |  0.0 | 17.9 | 18.104167 |    2.565092 |         2.227756 |           1.135969 |          3.2 |                20.542326 |                     20.807826 |                          0.0 |                      1.293123 |                           1.307717 |            14.728848 |                 16.249178 |    1002.271152 |   1.210597 | -2.9 | 18.731535 |   4.989208 |               0.71535 |                  0.132851 |                 214.709817 |                 35.120093 |                      0.0 |   -34.404743 |              0.0 |        179.589724 |                29.375 |                 100.0 |                232.15313 |               74.878349 |  0.0 |  0.0 |   0.16074 |       0.0 | 0.083415 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.070166 | 0.019491 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8975.294232 | 74.429742 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.070166 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.019491 |
    | 1997-08-04 21:00:00 |  0.0 | 15.8 |              0.0 |                      0.0 |             0.0 |             80.6 |       2.2 |              1018.0 | 0.0 | 0.0 | 18.020833 |             77.829167 |                  12.4 |                     15.502754 |           243.988429 |  0.0 | 15.8 | 18.020833 |    1.763501 |         2.244455 |           0.780979 |          2.2 |                17.978626 |                     20.699099 |                          0.0 |                      1.150835 |                           1.301743 |            14.490772 |                 16.109936 |    1003.509228 |    1.22071 | -2.9 | 18.636864 |  17.021185 |                   0.0 |                       0.0 |                 214.709817 |                 35.424509 |                      0.0 |   -35.424509 |              0.0 |        179.285308 |             42.727273 |                 100.0 |               219.832992 |               69.444444 |  0.0 |  0.0 |  0.056348 |       0.0 | 0.034081 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.068353 | 0.018987 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8955.373047 | 74.395661 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.068353 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.018987 |
    | 1997-08-04 22:00:00 |  0.0 | 14.8 |              0.0 |                      0.0 |             0.0 |             85.9 |       2.2 |              1018.0 | 0.0 | 0.0 | 17.929167 |             77.654167 |                  12.4 |                     15.502754 |           243.988429 |  0.0 | 14.8 | 17.929167 |    1.763501 |         2.261155 |           0.780979 |          2.2 |                16.859515 |                     20.580072 |                          0.0 |                      1.087886 |                           1.295199 |            14.482324 |                 15.981284 |    1003.517676 |   1.224953 | -2.9 |  18.39781 |  21.586859 |                   0.0 |                       0.0 |                 214.709817 |                 35.698198 |                      0.0 |   -35.698198 |              0.0 |        179.011619 |             42.727273 |                 100.0 |               216.477578 |               69.444444 |  0.0 |  0.0 |  0.030446 |       0.0 | 0.018181 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.066588 | 0.018497 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8930.886187 |  74.37748 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.066588 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.018497 |
    | 1997-08-04 23:00:00 |  0.0 | 14.1 |              0.0 |                      0.0 |             0.0 |             89.3 |       2.1 |              1018.0 | 0.0 | 0.0 | 17.833333 |               77.5875 |                  12.4 |                     15.502754 |           243.988429 |  0.0 | 14.1 | 17.833333 |    1.683342 |         2.254475 |            0.74548 |          2.1 |                16.112886 |                     20.456276 |                          0.0 |                      1.045579 |                           1.288387 |            14.388807 |                 15.871513 |    1003.611193 |   1.227981 | -2.9 | 18.103968 |  24.023805 |                   0.0 |                       0.0 |                 214.709817 |                 35.921753 |                      0.0 |   -35.921753 |              0.0 |        178.788064 |             44.761905 |                 100.0 |               216.443117 |               69.444444 |  0.0 |  0.0 |  0.012637 |       0.0 | 0.007608 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.064868 | 0.018019 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8903.962382 | 74.369872 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.064868 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.018019 |
    | 1997-08-05 00:00:00 |  0.0 | 13.4 |              0.0 |                      0.0 |             0.0 |             91.4 |       1.8 |              1018.0 | 0.0 | 0.0 | 17.729167 |             77.729167 |                  12.4 |                     15.502754 |           243.988429 |  0.0 | 13.4 | 17.729167 |    1.442864 |         2.237776 |           0.638983 |          1.8 |                15.395377 |                     20.322453 |                          0.0 |                      1.004677 |                           1.281018 |            14.071374 |                 15.796473 |    1003.928626 |   1.231127 | -2.9 | 17.780882 |  26.285292 |                   0.0 |                       0.0 |                 214.709817 |                 36.050541 |                      0.0 |   -36.050541 |              0.0 |        178.659276 |             52.222222 |                 100.0 |               219.886198 |               69.444444 |  0.0 |  0.0 | -0.000854 | -0.000854 |      0.0 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.063192 | 0.017553 | 0.000854 |  0.0 |  0.0 |   0.0 |  nan |  8874.77709 | 74.369872 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.063192 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.017553 |
    | 1997-08-05 01:00:00 |  0.0 | 12.8 |              0.0 |                      0.0 |             0.0 |             93.1 |       1.8 |              1018.0 | 0.0 | 0.0 | 17.620833 |             77.891667 |                  12.4 |                     15.502754 |           243.988429 |  0.0 | 12.8 | 17.620833 |    1.442864 |         2.224416 |           0.638983 |          1.8 |                14.802811 |                     20.184089 |                          0.0 |                      0.970706 |                           1.273391 |            13.781417 |                 15.721724 |    1004.218583 |   1.233843 | -2.9 | 17.430658 |  27.783951 |                   0.0 |                       0.0 |                 214.709817 |                 36.176259 |                      0.0 |   -36.176259 |              0.0 |        178.533559 |             52.222222 |                 100.0 |               223.673889 |               69.444444 |  0.0 |  0.0 | -0.007069 | -0.007069 |      0.0 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.06156 |   0.0171 | 0.007924 |  0.0 |  0.0 |   0.0 |  nan |  8844.09314 | 74.369872 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.06156 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0171 |
    | 1997-08-05 02:00:00 |  0.0 | 12.3 |              0.0 |                      0.0 |             0.0 |             94.6 |       1.6 |              1018.0 | 0.0 | 0.0 |      17.5 |             78.133333 |                  12.4 |                     15.502754 |           243.988429 |  0.0 | 12.3 |      17.5 |    1.282546 |         2.201036 |           0.567985 |          1.6 |                14.324376 |                     20.030733 |                          0.0 |                      0.943147 |                            1.26493 |            13.550859 |                 15.650679 |    1004.449141 |   1.236111 | -2.9 | 17.062451 |  28.574706 |                   0.0 |                       0.0 |                 214.709817 |                 36.284611 |                      0.0 |   -36.284611 |              0.0 |        178.425206 |                 58.75 |                 100.0 |                227.17139 |               69.444444 |  0.0 |  0.0 | -0.013945 | -0.013945 |      0.0 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 |  0.05997 | 0.016658 | 0.021868 |  0.0 |  0.0 |   0.0 |  nan | 8812.618434 | 74.369872 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 |  0.05997 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.016658 |
    | 1997-08-05 03:00:00 |  0.0 | 12.0 |              0.0 |                      0.0 |             0.0 |             95.6 |       1.2 |              1018.0 | 0.0 | 0.0 | 17.379167 |             78.379167 |                  12.4 |                     15.502754 |           243.988429 |  0.0 | 12.0 | 17.379167 |    0.961909 |         2.174316 |           0.425988 |          1.2 |                 14.04387 |                     19.878396 |                          0.0 |                      0.926933 |                           1.256516 |            13.425939 |                 15.580521 |    1004.574061 |   1.237469 | -2.9 | 16.684755 |  28.108527 |                   0.0 |                       0.0 |                 214.709817 |                 36.390891 |                      0.0 |   -36.390891 |              0.0 |        178.318926 |             78.333333 |                 100.0 |               229.324539 |               69.444444 |  0.0 |  0.0 | -0.019233 | -0.019233 |      0.0 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.058421 | 0.016228 | 0.041102 |  0.0 |  0.0 |   0.0 |  nan | 8781.609907 | 74.369872 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.058421 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.016228 |
    | 1997-08-05 04:00:00 |  0.0 | 11.7 |              0.0 |                 0.366022 |        1.482347 |             96.2 |       1.9 |              1018.0 | 0.0 | 0.0 | 17.270833 |               78.5125 |                  12.4 |                     15.470733 |           243.983922 |  0.0 | 11.7 | 17.270833 |    1.523023 |         2.160956 |           0.674482 |          1.9 |                13.768192 |                      19.74268 |                          0.0 |                      0.910957 |                           1.249013 |            13.245001 |                 15.500472 |    1004.754999 |   1.238856 | -2.9 | 16.312652 |  27.675913 |              1.304466 |                  0.242258 |                 214.705851 |                 36.589063 |                      0.0 |   -35.284597 |              0.0 |        178.116788 |             49.473684 |                 100.0 |               232.308624 |               93.057159 |  0.0 |  0.0 | -0.016393 | -0.016393 |      0.0 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.056912 | 0.015809 | 0.057495 |  0.0 |  0.0 |   0.0 |  nan | 8751.033993 | 74.369872 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.056912 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.015809 |
    | 1997-08-05 05:00:00 |  0.0 | 11.6 |              0.0 |                      1.0 |       21.135567 |             95.7 |       1.3 |              1018.0 | 0.0 | 0.0 | 17.166667 |                  78.6 |                  12.4 |                     15.470733 |           244.033283 |  0.0 | 11.6 | 17.166667 |    1.042069 |         2.120877 |           0.461488 |          1.3 |                 13.67736 |                     19.612949 |                          0.0 |                      0.905683 |                           1.241834 |            13.089234 |                 15.415778 |    1004.910766 |   1.239363 | -2.9 | 15.945741 |  26.074448 |             18.599299 |                  3.454155 |                 214.749289 |                  36.74524 |                      0.0 |   -18.145941 |              0.0 |        178.004048 |             72.307692 |                 100.0 |               234.489857 |              228.415592 |  0.0 |  0.0 | -0.005017 | -0.005017 |      0.0 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.055442 |   0.0154 | 0.062512 |  0.0 |  0.0 |   0.0 |  nan | 8722.059546 | 74.369872 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.055442 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |   0.0154 |
    | 1997-08-05 06:00:00 |  0.0 | 12.6 |              0.6 |                      1.0 |       125.88345 |             95.1 |       1.6 |              1018.0 | 0.0 | 0.0 | 17.095833 |             78.620833 |                  12.9 |                     15.470733 |           246.654019 |  0.0 | 12.6 | 17.095833 |    1.282546 |         2.094157 |           0.567985 |          1.6 |                14.609782 |                     19.525158 |                          0.0 |                      0.959602 |                           1.236973 |            13.893903 |                 15.350842 |    1004.106097 |   1.234655 | -2.9 | 15.598048 |  17.988287 |            110.777436 |                 20.572952 |                 217.055537 |                  38.00446 |                      0.0 |    72.772976 |              0.0 |        179.051077 |                 58.75 |                 100.0 |               222.631747 |              217.361109 |  0.0 |  0.0 |  0.070823 |  0.062512 | 0.003085 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.054009 | 0.015003 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8701.171259 | 74.366788 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.054009 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.015003 |
    | 1997-08-05 07:00:00 |  0.0 | 14.8 |              0.9 |                      1.0 |      305.954944 |             96.3 |       1.9 |              1019.0 | 0.0 | 0.0 | 17.070833 |             78.745833 |                  12.9 |                     15.470733 |           247.494792 |  0.0 | 14.8 | 17.070833 |    1.523023 |         2.074117 |           0.674482 |          1.9 |                16.859515 |                     19.494255 |                          0.0 |                      1.087886 |                           1.235261 |            16.235713 |                 15.350914 |    1002.764287 |   1.225361 | -2.9 | 15.347388 |   3.284331 |            269.240351 |                 50.001779 |                 217.795417 |                 37.987562 |                      0.0 |   231.252789 |              0.0 |        179.807855 |             49.473684 |                 100.0 |               196.962413 |              193.257163 |  0.0 |  0.0 |   0.20795 |       0.0 | 0.078657 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.052614 | 0.014615 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8694.986928 |  74.28813 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.052614 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.014615 |
    | 1997-08-05 08:00:00 |  0.0 | 17.1 |              1.0 |                      1.0 |      511.104575 |             89.9 |       3.2 |              1019.0 | 0.0 | 0.0 | 17.083333 |             78.870833 |                  12.9 |                     15.470733 |           248.989764 |  0.0 | 17.1 | 17.083333 |    2.565092 |         2.097497 |           1.135969 |          3.2 |                19.530313 |                     19.509701 |                          0.0 |                      1.237258 |                           1.236116 |            17.557751 |                 15.387464 |    1001.442249 |   1.215051 | -2.9 | 15.273176 | -10.960941 |            449.772026 |                 83.529091 |                 219.110992 |                 37.896217 |                      0.0 |   411.875809 |              0.0 |        181.214776 |                29.375 |                 100.0 |               189.861573 |              186.546902 |  0.0 |  0.0 |  0.442904 |       0.0 | 0.131319 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.051255 | 0.014238 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8703.047869 | 74.156811 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.051255 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.014238 |
    | 1997-08-05 09:00:00 |  0.0 | 19.4 |              0.9 |                      1.0 |      629.519239 |             79.2 |       4.0 |              1019.0 | 0.0 | 0.0 | 17.133333 |             78.883333 |                  12.8 |                     15.470733 |           249.636119 |  0.0 | 19.4 | 17.133333 |    3.206365 |         2.134237 |           1.419962 |          4.0 |                22.563931 |                     19.571593 |                          0.0 |                      1.403627 |                           1.239544 |            17.870634 |                 15.438725 |    1001.129366 |   1.205358 | -2.9 | 15.369908 | -24.180553 |             553.97693 |                 102.88143 |                 219.679785 |                  37.56462 |                      0.0 |   516.412311 |              0.0 |        182.115165 |                  23.5 |                 100.0 |               202.878648 |              198.833875 |  0.0 |  0.0 |  0.679064 |       0.0 | 0.175108 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.049931 |  0.01387 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8724.328423 | 73.981703 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.049931 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 |  0.01387 |
    | 1997-08-05 10:00:00 |  0.0 | 20.6 |              1.0 |                      1.0 |      765.816589 |             73.4 |       4.6 |              1019.0 | 0.0 | 0.0 | 17.179167 |             78.783333 |                  12.9 |                     15.470733 |           254.719872 |  0.0 | 20.6 | 17.179167 |     3.68732 |         2.184336 |           1.632956 |          4.6 |                24.304197 |                     19.628477 |                          0.0 |                      1.497674 |                           1.242694 |            17.839281 |                 15.463969 |    1001.160719 |   1.200448 | -2.9 | 15.625274 | -29.848354 |            673.918598 |                125.156311 |                 224.153488 |                 37.751868 |                      0.0 |   636.166731 |              0.0 |         186.40162 |             20.434783 |                 100.0 |                224.43901 |              219.049161 |  0.0 |  0.0 |  0.901936 |       0.0 | 0.201966 | 0.0 |   0.0 |          nan |         0.12 |     0.0 |  0.0 |     0.0 |  0.0 |      0.0 |       0.0 |   0.0 | 0.0 | 0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.0 |  0.0 |  0.0 | 0.048642 | 0.013512 |      0.0 |  0.0 |  0.0 |   0.0 |  nan | 8751.276776 | 73.779737 |   0.0 |   0.0 |   0.0 |   0.0 |  0.0 | 0.048642 |   0.0 |   0.0 |   0.0 |  0.0 |   0.0 | 0.013512 |
    | 1997-08-05 11:00:00 |  0.0 | 21.4 |              1.0 |                      1.0 |      767.145842 |             67.3 |       4.1 |              1019.0 | 0.0 | 0.0 |    17.225 |             78.529167 |                  13.1 |                     15.470733 |           261.673325 |  0.0 | 21.4 |    17.225 |    3.286524 |         2.211056 |           1.455461 |          4.1 |           