/*
 * EMPORDA Software - More than an implementation of MHDC Recommendation for  Image Data Compression
 * Copyright (C) 2011  Group on Interactive Coding of Images (GICI)
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for  more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Group on Interactive Coding of Images (GICI)
 * Department of Information and Communication Engineering
 * Autonomous University of Barcelona
 * 08193 - Bellaterra - Cerdanyola del Valles (Barcelona)
 * Spain
 *
 * http://gici.uab.es
 * http://sourceforge.net/projects/emporda
 * gici-info@deic.uab.es
 */

package emporda;

import GiciStream.FileBitOutputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.Arrays;
import GiciContextModel.ContextModelling;
import GiciContextModel.ContextProbability;
import GiciEntropyCoder.ArithmeticCoder.ArithmeticCoderFLW;
import GiciException.ParameterException;
import GiciFile.RawImage.OrderConverter;
import GiciFile.RawImage.RawImage;
import GiciFile.RawImage.RawImageIterator;
import GiciStream.BitOutputStream;
import GiciStream.ByteStream;
import GiciPredictor.*;
import GiciQuantizer.Quantizer;

/**
 * Coder class of EMPORDA application. This is class uses the predictor and the type of coder chosen
 * in order to encode the image
 * <p>
 *
 * @author Group on Interactive Coding of Images (GICI)
 * @version 1.0
 */
public class Coder {

	private final File file;
	private final FileOutputStream fileStream;
	private final BitOutputStream fbos;

	private Parameters parameters;
	private int[] geo;
	private Predictor predictor;
	private ArithmeticCoderFLW ec;
	private ContextModelling cm;
	private ContextProbability cp;
	private ContextProbability cps;

	private int[] originalPixelOrder;
	private int[] pixelOrderTransformation;
	private String inputFile;
	private Quantizer uq;
	private int contextModel = 0;
	private int probabilityModel = 0;
	private int quantizerProbabilityLUT = 0;
	private int numBitsPrecision = 0;
	private int UPDATE_PROB0 = 0;
	private int WINDOW_PROB = 0;
	private int bands;
	private	int height;
	private int width;


	private int encoderType = 0;
	//private SamplePrediction samplePredictor = null;
	private int MAXBITS = 16;
	int numOfContexts = 0;
	int coderWordLength = 0;
	int bufferSize = 1;

	private SamplePrediction samplePredictor;
	/**
	 * Bit masks (employed when coding integers).
	 * <p>
	 * The position in the array indicates the bit for which the mask is computed.
	 */
	protected static final int[] BIT_MASKS = {1, 1 << 1, 1 << 2, 1 << 3, 1 << 4, 1 << 5, 1 << 6,
	1 << 7, 1 << 8, 1 << 9, 1 << 10, 1 << 11, 1 << 12, 1 << 13, 1 << 14, 1 << 15, 1 << 16, 1 << 17,
	1 << 18, 1 << 19, 1 << 20, 1 << 21, 1 << 22, 1 << 23, 1 << 24, 1 << 25, 1 << 26, 1 << 27,
	1 << 28, 1 << 29, 1 << 30, 1 << 31, 1 << 32};

	/**
	 * Qsteps.
	 * <p>
	 * The position in the array indicates the Qstep
	 */
	//protected static final int[] QSTEPS = {1, 2, 4, 6, 8, 10, 12, 16, 20, 24, 28, 32, 40, 48, 56, 64, 80, 96, 112, 128, 192, 256, 384, 512, 1024, 2048, 4096, 8192};
	protected static final int[] QSTEPS = {1};

	ByteStream stream;

	/**
	 * This variable indicates if debug information must be shown or not
	 */
	private boolean debugMode = false;

	/**
	 * Constructor of Coder. It receives the name of the output file and
	 * the parameters needed for the headers.
	 *
	 * @param outputFile the file where the result of the compressing will be saved
	 * @param inputFile the file that contain the image
	 * @param sampleOrder is the sample order of the image in the input file.
	 * @param parameters all the information about the compression process
	 * @param debugMode indicates if debug information must be shown
	 * @param numberOfBitplanesToDecode
	 * @throws IOException when something goes wrong and writing must be stopped
	 * @throws ParameterException when an invalid parameter is detected
	 */
	public Coder(String outputFile, String inputFile, int sampleOrder, final Parameters parameters, Quantizer uq, int contextModel, int probabilityModel, int encoderWP, int encoderUP, int encoderType)
			throws IOException, ParameterException {

		geo = parameters.getImageGeometry();
		file = new File(outputFile);
		file.delete();

		fileStream = new FileOutputStream(file);
		fbos = new FileBitOutputStream( new BufferedOutputStream(fileStream));
		this.inputFile = inputFile;
		this.uq = uq;
		this.contextModel = contextModel;
		this.probabilityModel =  probabilityModel;
		this.WINDOW_PROB = encoderWP;
		this.UPDATE_PROB0 = encoderUP;
		this.encoderType = encoderType;

		if(probabilityModel == 1 && UPDATE_PROB0 > WINDOW_PROB){
			throw new Error("For this Probability Model option (-pm) UPDATE_PROB0 must be minor than WINDOW_PROB.");
		}
		if(probabilityModel == 2 && UPDATE_PROB0 != WINDOW_PROB){
			throw new Error("For this Probability Model option (-pm) UPDATE_PROB0 equal than WINDOW_PROB.");
		}

		originalPixelOrder = OrderConverter.DIM_TRANSP_IDENTITY;
		if (parameters.sampleEncodingOrder == CONS.BAND_SEQUENTIAL) {
			pixelOrderTransformation = OrderConverter.DIM_TRANSP_IDENTITY;
		}else {
			pixelOrderTransformation = OrderConverter.DIM_TRANSP_BSQ_TO_BIL;
		}

		this.parameters = parameters;

		bands = geo[CONS.BANDS];
		height = geo[CONS.HEIGHT];
		width = geo[CONS.WIDTH];

	}


	/**
	 * Writes the header with all the needed information for the decompression process.
	 *
	 * @param parameters all information about the compression process
	 * @throws IOException when something goes wrong and writing must be stopped
	 * @throws ParameterException when an invalid parameter is detected
	 */
	public void writeHeader(final Parameters parameters) throws IOException, ParameterException {
		CoderHeader ch = new CoderHeader(fbos, debugMode, parameters);
		ch.imageHeader();
		ch.predictorMetadata();
		ch.entropyCoderMetadata();
		ch = null;
	}

	/**
	 * Compiles all the information needed to create the entropy coder,
	 * and creates it.
	 * @param verbose indicates whether to display information
	 * @throws ParameterException
	 */
	// --------- New argument added to control the entropyCoder
	private void startCoder(boolean verbose) throws ParameterException {

		cm = new ContextModelling(contextModel);
		numOfContexts = cm.getNumberOfContexts(MAXBITS);
		numBitsPrecision = 15;
		coderWordLength = 48;
		cp = new ContextProbability(probabilityModel, numOfContexts, numBitsPrecision, quantizerProbabilityLUT, parameters.entropyCoderType, WINDOW_PROB, UPDATE_PROB0);
		cps = new ContextProbability(probabilityModel, numOfContexts, numBitsPrecision, quantizerProbabilityLUT, parameters.entropyCoderType, WINDOW_PROB, UPDATE_PROB0);
		ec = new ArithmeticCoderFLW(coderWordLength, numBitsPrecision, numOfContexts);

		//initialize predictor according to the user parameters
		switch(encoderType){
		case 0:
			samplePredictor = new SamplePrediction();
			break;
		case 1:
			samplePredictor = new SamplePrediction();
			break;
		case 2:
			samplePredictor = new SamplePrediction();
			break;
		case 3:
			predictor = new Predictor(parameters);
			break;
		}
	}

	/**
	 * Runs the EMPORDA coder algorithm to compress the image.
	 *
	 * @param verbose indicates whether to display information
	 * @throws Exception
	 */
	public void code(boolean verbose) throws Exception {

		startCoder(verbose);
		codeBSQ(verbose);
		int oddBytes = (int) (file.length() % parameters.outputWordSize);
		if (oddBytes != 0) {
			byte b[] = new byte[parameters.outputWordSize - oddBytes];
			Arrays.fill(b, (byte)0);
			fileStream.write(b);
		}
		fileStream.close();

	}


	/**
	 * Encodes an image in BSQ order using Fixed Quantization Step Mode
	 *
	 * @param verbose indicates whether to display information
	 * @throws Exception
	 */
	private void codeBSQ(boolean verbose) throws Exception {
		int inputData[][][] = new int[parameters.numberPredictionBands + 1][][];

		if(verbose || debugMode) {
			System.out.println("Coding BSQ");
		}
		try {
			RawImage image = new RawImage(inputFile, geo, originalPixelOrder, RawImage.READ);
			RawImageIterator<int[]> it = (RawImageIterator<int[]>) image.getIterator(new int[0], pixelOrderTransformation, RawImage.READ, true);

			for (int z = 0; z < bands; z ++) {
				prepareBands(z, inputData, it, height, width);
				int residuals[][] = new int[height][width];
				if(z == 0){
					ec.restartEncoding();
					ec.init(z);
					cp.reset();
					cps.reset();
				}


				switch(encoderType){
					case 0://Lossless with only entropy encoder
						
						entropyEncoder(inputData[parameters.numberPredictionBands]);
						
						break;
					case 1://Lossless with predictor
						////////////////////////SESSIO 1////////////////////////
						//Afegir codi per tal de codificar la imatge mitjantçant  predictor + entropy encoder
					    	for (int y = 0; y < height; y ++) {
						for (int x = 0; x < width; x ++) {
							int prediction = samplePredictor.predict(inputData, parameters.numberPredictionBands, y, x);
							residuals[y][x] = inputData[parameters.numberPredictionBands][y][x] - prediction;
						}}
					    	entropyEncoder(residuals);
						break;
					case 2://Lossless and near-lossless with predictor
						////////////////////////SESSIO 2////////////////////////
						//Afegir codi per tal de codificar la imatge mitjantçant predictor + quantitzacio + entropy encoder
					    	for (int y = 0; y < height; y ++) {
						for (int x = 0; x < width; x ++) {
							int prediction = samplePredictor.predict(inputData, parameters.numberPredictionBands, y, x);
							residuals[y][x] = inputData[parameters.numberPredictionBands][y][x] - prediction;
							residuals[y][x] = uq.quantize(residuals[y][x]);
							int residualDequantized = uq.dequantize(residuals[y][x]);
							inputData[parameters.numberPredictionBands][y][x] = residualDequantized + prediction;
						}}
					    	entropyEncoder(residuals);
						break;
					case 3://Lossless and near-lossless with an state-of-the-art predictor
						for (int y = 0; y < height; y ++) {
						for (int x = 0; x < width; x ++) {
							residuals[y][x] = predictor.compress(inputData, z, y, x, parameters.numberPredictionBands, y, uq);
						}}
						entropyEncoder(residuals);
						break;

				}
			}
			//to save data to the final codestream
			ec.terminate();
			fileStream.write(ec.getByteStream().getByteStream(),0,(int) ec.getByteStream().getLength());

			image.close(it);

		}catch(UnsupportedOperationException e) {
			throw new Error("Unexpected exception ocurred "+e.getMessage());
		}catch(IndexOutOfBoundsException e) {
			e.printStackTrace();
			throw new Error("Unexpected exception ocurred "+e.getMessage());
		}catch(ClassCastException e) {
			throw new Error("Unexpected exception ocurred "+e.getMessage());
		}
	}


	/**
	 * BSQ Contextual Arithmetic Coder with sign coding
	 * @param predictedSamples
	 * @param Previous1
	 * @param Previous2
	 * @param z
	 */

	private void entropyEncoder(int[][] predictedSamples){

		boolean realBit = false;
		boolean signBit = false;
		int context = -1;
		int prob = -1;
		int [][] MapSign = new int [height][width];//1 --> positive, -1 --> negative
		int max = Integer.MIN_VALUE;
		int [] maxLines = new int [height];
		for (int y = 0; y < height; y ++) {
		    maxLines[y]  = Integer.MIN_VALUE;
		}
		
		for (int y = 0; y < height; y ++) {
		for (int x = 0; x < width; x ++) {
			if(predictedSamples[y][x] >= 0) {
				MapSign[y][x] = 1;
			} else {
				MapSign[y][x] = -1;
			}
			predictedSamples[y][x] = Math.abs(predictedSamples[y][x]);
			if(max < predictedSamples[y][x]) max = predictedSamples[y][x];
			if(maxLines[y] < predictedSamples[y][x]) maxLines[y] = predictedSamples[y][x];
		}}

		for (int y = 0; y < height; y ++) {
		for (int x = 0; x < width; x ++) {
			if(MapSign[y][x] == -1) {
				signBit = false;
			}
			else {
				signBit = true;
			}
			context = cm.getContextSign(MapSign, y, x);//get context
			prob = cps.getProbability(context);//get probability for the computed context
			ec.encodeBitProb(signBit, prob);//encode the bit using the specific probability
			cps.updateSymbols(signBit, context);//updates the symbols decoded to properly compute the probability
		}}

		for (int y = 0; y < height; y ++) {
		    int bitplanes = Integer.SIZE - Integer.numberOfLeadingZeros(maxLines[y]);
		    ec.encodeInteger(bitplanes, 5);
		    for (int bit = bitplanes-1; bit >= 0; bit--){
		    for (int x = 0; x < width; x ++) {
			realBit = (predictedSamples[y][x] & BIT_MASKS[bit]) != 0;
			context = cm.getContext(predictedSamples, y, x, bit, bitplanes - 1);//get context
			prob = cp.getProbability(context);//get probability for the computed context
			cp.updateSymbols(realBit, context);//updates the symbols decoded to properly compute the probability
			ec.encodeBitProb(realBit, prob);//encode the bit using the specific probability
		    }}
		}
		
	}


	
	 /*
	 private void entropyEncoder(int[][] predictedSamples){

		boolean realBit = false;
		boolean signBit = false;
		int context = -1;
		int prob = -1;
		int [][] MapSign = new int [height][width];//1 --> positive, -1 --> negative
		int max = Integer.MIN_VALUE;
		
		for (int y = 0; y < height; y ++) {
		for (int x = 0; x < width; x ++) {
			if(predictedSamples[y][x] >= 0) {
				MapSign[y][x] = 1;
			} else {
				MapSign[y][x] = -1;
			}
			predictedSamples[y][x] = Math.abs(predictedSamples[y][x]);
			if(max < predictedSamples[y][x]) max = predictedSamples[y][x];
		}}

		for (int y = 0; y < height; y ++) {
		for (int x = 0; x < width; x ++) {
			if(MapSign[y][x] == -1) {
				signBit = false;
			}
			else {
				signBit = true;
			}
			context = cm.getContextSign(MapSign, y, x);//get context
			prob = cps.getProbability(context);//get probability for the computed context
			ec.encodeBitProb(signBit, prob);//encode the bit using the specific probability
			cps.updateSymbols(signBit, context);//updates the symbols decoded to properly compute the probability
		}}

		int bitplanes = Integer.SIZE - Integer.numberOfLeadingZeros(max);
		ec.encodeInteger(bitplanes, 5);
		
		for (int bit = bitplanes-1; bit >= 0; bit--){
		for (int y = 0; y < height; y ++) {
		for (int x = 0; x < width; x ++) {
			realBit = (predictedSamples[y][x] & BIT_MASKS[bit]) != 0;
			context = cm.getContext(predictedSamples, y, x, bit, bitplanes - 1);//get context
			prob = cp.getProbability(context);//get probability for the computed context
			cp.updateSymbols(realBit, context);//updates the symbols decoded to properly compute the probability
			ec.encodeBitProb(realBit, prob);//encode the bit using the specific probability
		}}}
	}
*/	
	
	 
	/**
	 * Reads the next band of the image from the input file.
	 * @param it the BSQ iterator over the image
	 * @return the next band of the image
	 */
	private void readBand(RawImageIterator<int[]> it, int[][] band) {

		for(int i=0;i<geo[CONS.HEIGHT];i++) {
			band[i] = it.next();
		}
	}

	/**
	 * Tries to read the band z of the image, and reorders all the bands
	 * in memory.
	 * @param z the band to be read
	 * @param bands an array with all the bands needed in the compression process
	 * @param it the BSQ iterator over the image
	 */
	private void prepareBands(int z, int[][][] bands, RawImageIterator<int[]> it, int height, int width) {
		bands[0] = null;
		for(int i = 0; i < parameters.numberPredictionBands; i ++) {
			bands[i] = bands[i + 1];
		}
		bands[parameters.numberPredictionBands] = new int[height][];
		readBand(it, bands[parameters.numberPredictionBands]);
	}



}
