/*
 * EMPORDA Software - More than an implementation of MHDC Recommendation for  Image Data Compression
 * Copyright (C) 2011  Group on Interactive Coding of Images (GICI)
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for  more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Group on Interactive Coding of Images (GICI)
 * Department of Information and Communication Engineering
 * Autonomous University of Barcelona
 * 08193 - Bellaterra - Cerdanyola del Valles (Barcelona)
 * Spain
 *
 * http://gici.uab.es
 * http://sourceforge.net/projects/emporda
 * gici-info@deic.uab.es
 */

package emporda;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.EOFException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import GiciContextModel.ContextModelling;
import GiciContextModel.ContextProbability;
import GiciEntropyCoder.ArithmeticCoder.ArithmeticCoderFLW;
import GiciException.ParameterException;
import GiciFile.RawImage.OrderConverter;
import GiciFile.RawImage.RawImage;
import GiciFile.RawImage.RawImageIterator;
import GiciPredictor.Predictor;
import GiciPredictor.SamplePrediction;
import GiciQuantizer.Quantizer;
import GiciStream.BitInputStream;
import GiciStream.ByteStream;


/**
 * Decoder class of EMPORDA application. Decoder is a decoder of the Recommended Standard MHDC-123 White Book.
 * <p>
 *
 * @author Group on Interactive Coding of Images (GICI)
 * @version 1.0
 */
public class Decoder {

	private final File file;
	private FileInputStream fileStream;
	private final BitInputStream bis;
	private ArithmeticCoderFLW ec;
	private ContextModelling cm;
	private ContextProbability cp;
	private ContextProbability cps;

	private Parameters parameters = null;
	private Predictor predictor = null;
	private boolean debugMode = false;
	private int[] savedPixelOrder = null;
	private String outputFile = null;
	private int[] pixelOrderTransformation = null;
	private Quantizer uq;
	private int contextModel = 0;
	private int probabilityModel = 0;
	private int quantizerProbabilityLUT = 0;
	private int numBitsPrecision = 0;
	private int encoderType = 0;
	private int UPDATE_PROB0 = 0;
	private int WINDOW_PROB = 0;

    private SamplePrediction samplePredictor = null;
    private int MAXBITS = 32;
    private int sampleType = -1;
	
	
	/**
	 * Bit masks (employed when coding integers).
	 * <p>
	 * The position in the array indicates the bit for which the mask is computed.
	 */
	protected static final int[] BIT_MASKS = {1, 1 << 1, 1 << 2, 1 << 3, 1 << 4, 1 << 5, 1 << 6,
	1 << 7, 1 << 8, 1 << 9, 1 << 10, 1 << 11, 1 << 12, 1 << 13, 1 << 14, 1 << 15, 1 << 16, 1 << 17,
	1 << 18, 1 << 19, 1 << 20, 1 << 21, 1 << 22, 1 << 23, 1 << 24, 1 << 25, 1 << 26, 1 << 27,
	1 << 28, 1 << 29, 1 << 30, 1 << 31, 1 << 32};
	
	ByteStream stream;
	
	
	/**
	 * Constructor of Decoder. It receives the name of the input file.
	 *
	 * @param inputFile the file where is saved the image that is going
	 * to be decompressed
	 * @throws FileNotFoundException when something goes wrong and writing must be stopped
	 */
	public Decoder (String inputFile, String outputFile, int sampleType, int sampleOrder, Quantizer uq, int contextModel, int probabilityModel, int encoderType, int encoderWP, int encoderUP, int samplePrediction) throws FileNotFoundException {
		
		file = new File(inputFile);
		fileStream = new FileInputStream(file);
		bis = new BitInputStream( new BufferedInputStream( fileStream ) );
		this.sampleType = sampleType;
		this.outputFile = outputFile;

		File f = new File(outputFile);
		f.delete();
		f = null;
		this.uq = uq;
		this.contextModel = contextModel;
		this.probabilityModel =  probabilityModel;
		this.encoderType = encoderType;
		this.WINDOW_PROB = encoderWP;
		this.UPDATE_PROB0 = encoderUP;
		
	}
	

	/**
	 * Reads the header with all the needed information for  the decompression process.
	 *
	 * @return object with all the information about the compression process
	 * @throws IOException when something goes wrong and writing must be stopped
	 * @throws ParameterException when an invalid parameter is detected
	 */
	public Parameters readHeader(String optionFile) throws IOException, ParameterException {
		
	
		parameters = new Parameters(optionFile, null, true, debugMode, 0, 0);
		DecoderHeader dh;
		dh = new DecoderHeader(bis, parameters, debugMode);
		dh.readImageHeader();
		dh.finish();
		savedPixelOrder = OrderConverter.DIM_TRANSP_IDENTITY;
		if (parameters.sampleEncodingOrder == CONS.BAND_SEQUENTIAL) {
			pixelOrderTransformation = OrderConverter.DIM_TRANSP_IDENTITY;
		}else {
			pixelOrderTransformation = OrderConverter.DIM_TRANSP_BSQ_TO_BIL;
		}
		
		return parameters;
	}

	/**
	 * Compiles all the information needed to create the entropy decoder,
	 * and creates it.
	 * @param verbose indicates whether to display information
	 * @throws Exception 
	 * @throws EOFException 
	 */
	private void startDecoder(boolean verbose) throws Exception {
		
		cm = new ContextModelling(contextModel);
		int numOfContexts = cm.getNumberOfContexts(MAXBITS);
		numBitsPrecision = 15;
		int coderWordLength = 48;
		cp = new ContextProbability(probabilityModel, numOfContexts, numBitsPrecision, quantizerProbabilityLUT, parameters.entropyCoderType, WINDOW_PROB, UPDATE_PROB0);
		cps = new ContextProbability(probabilityModel, numOfContexts, numBitsPrecision, quantizerProbabilityLUT, parameters.entropyCoderType, WINDOW_PROB, UPDATE_PROB0);
		ec = new ArithmeticCoderFLW(coderWordLength, numBitsPrecision, numOfContexts);
		
		fileStream.close();
		fileStream = new FileInputStream(file);
		stream = new ByteStream(fileStream.getChannel());
		stream.putFileSegment(19,fileStream.available());//jump 19 bytes corresponding to the headers
		ec.changeStream(stream);
		ec.restartDecoding();
		ec.reset();
		
		switch(encoderType){
		case 0:
			samplePredictor = new SamplePrediction();
			break;
		case 1:
			samplePredictor = new SamplePrediction();
			break;
		case 2:
			samplePredictor = new SamplePrediction();
			break;
		case 3:
			predictor = new Predictor(parameters);
			break;
		}
		
	}

	
	/**
	 * Runs the EMPORDA decoder algorithm to decompress the image.
	 *
	 * @return the whole image decoded
	 * @throws Exception 
	 */
	public void decode(boolean verbose) throws Exception {
		
		int readBytes = bis.available();
		int[] imageGeometry = parameters.getImageGeometry();
		
		try {
			startDecoder(verbose);
		} catch (ParameterException e) {
			e.printStackTrace();
			System.exit(-1);
		}
				
		decodeBSQ(verbose);
	
		if (verbose) {
			System.out.println("\rRead " + readBytes + " bytes            ");
			System.out.println("\rRate " + readBytes*8/(float)(imageGeometry[CONS.BANDS]*imageGeometry[CONS.HEIGHT]*imageGeometry[CONS.WIDTH]) + " bpppb  ");
		}
		
		
	}

	/**
	 * 
	 * @param outoutData
	 * @throws Exception
	 */
	
	private void entropyDecoder(int outoutData[][][]) throws Exception{
		int height = outoutData[parameters.numberPredictionBands].length;
		int width = outoutData[parameters.numberPredictionBands][0].length;
		boolean realBit = false;
		boolean signBit = false;
		int context = -1;
		int prob = -1;
		int [][] MapSign = new int [height][width];//0 --> not known, 1 --> positive, -1 --> negative
		
		for (int y = 0; y < height; y ++) {
		for (int x = 0; x < width; x ++) {
			context = cm.getContextSign(MapSign, y, x);//get context
			prob = cps.getProbability(context);//get probability for the computed context
			signBit = ec.decodeBitProb(prob);//decode the bit using the specific probability
			cps.updateSymbols(signBit, context);//updates the symbols decoded to properly compute the probability
			
			if(signBit == false) {
				MapSign[y][x] = -1;
			}else {
				MapSign[y][x] = 1;
			}
		}}
		
		for (int y = 0; y < height; y ++) {
		    int bitplanes = ec.decodeInteger(5);
		    for (int bit = bitplanes-1; bit >= 0; bit--){
		    for (int x = 0; x < width; x ++) {
			realBit = false;
			context = cm.getContext(outoutData[parameters.numberPredictionBands], y, x, bit, bitplanes - 1);//get context
			prob = cp.getProbability(context);//get probability for the computed context
			realBit = ec.decodeBitProb(prob);//decode the bit using the specific probability
			cp.updateSymbols(realBit, context);//updates the symbols decoded to properly compute the probability
			outoutData[parameters.numberPredictionBands][y][x] += realBit == true ?  BIT_MASKS[bit] : 0;
		    }}
		}
		
		
		for (int y = 0; y < height; y ++) {
		for (int x = 0; x < width; x ++) {
			outoutData[parameters.numberPredictionBands][y][x] *= MapSign[y][x];
		}}
	}
	
	/*
	 private void entropyDecoder(int outoutData[][][]) throws Exception{
		int height = outoutData[parameters.numberPredictionBands].length;
		int width = outoutData[parameters.numberPredictionBands][0].length;
		boolean realBit = false;
		boolean signBit = false;
		int context = -1;
		int prob = -1;
		int [][] MapSign = new int [height][width];//0 --> not known, 1 --> positive, -1 --> negative
		
		for (int y = 0; y < height; y ++) {
		for (int x = 0; x < width; x ++) {
			context = cm.getContextSign(MapSign, y, x);//get context
			prob = cps.getProbability(context);//get probability for the computed context
			signBit = ec.decodeBitProb(prob);//decode the bit using the specific probability
			cps.updateSymbols(signBit, context);//updates the symbols decoded to properly compute the probability
			
			if(signBit == false) {
				MapSign[y][x] = -1;
			}else {
				MapSign[y][x] = 1;
			}
		}}
		
		
		int bitplanes = ec.decodeInteger(5);
		
		for (int bit = bitplanes-1; bit >= 0; bit--){
		for (int y = 0; y < height; y ++) {
		for (int x = 0; x < width; x ++) {
			realBit = false;
			context = cm.getContext(outoutData[parameters.numberPredictionBands], y, x, bit, bitplanes - 1);//get context
			prob = cp.getProbability(context);//get probability for the computed context
			realBit = ec.decodeBitProb(prob);//decode the bit using the specific probability
			cp.updateSymbols(realBit, context);//updates the symbols decoded to properly compute the probability
			outoutData[parameters.numberPredictionBands][y][x] += realBit == true ?  BIT_MASKS[bit] : 0;
		}}}
		
		for (int y = 0; y < height; y ++) {
		for (int x = 0; x < width; x ++) {
			outoutData[parameters.numberPredictionBands][y][x] *= MapSign[y][x];
		}}
	}
	 */
	
	/**
	 * Decodes an image in BSQ order
	 * @throws Exception 
	 *
	 * @params img the whole image
	 * @params bands the number of bands of the image
	 * @params height the number of lines of one band of the image
	 * @params width the number of samples of one line of the image
	 */
	private void decodeBSQ(boolean verbose) throws Exception {
		int[] imageGeometry = parameters.getImageGeometry();
		int bands = imageGeometry[CONS.BANDS];
		int height = imageGeometry[CONS.HEIGHT];
		int width = imageGeometry[CONS.WIDTH];
		//The sampleType is forced to be of the commandline not the stored in the headers. This must be modfied for a final product.
		parameters.getImageGeometry()[CONS.TYPE] = sampleType;
	
		
		int outputData[][][] = new int[parameters.numberPredictionBands + 1][height][width];
		try {
			RawImage image = new RawImage(outputFile, parameters.getImageGeometry(), savedPixelOrder, RawImage.WRITE);
			RawImageIterator<int[]> it = (RawImageIterator<int[]>) image.getIterator(new int[0], pixelOrderTransformation, RawImage.WRITE, true);
			
			for (int z = 0; z < bands; z ++) {
				for (int y = 0; y < height; y ++){
				for (int x = 0; x < width; x ++){
					outputData[parameters.numberPredictionBands][y][x] = 0;
				}}
				
				switch(this.encoderType){
				case 0://Lossless with only entropy encoder
					entropyDecoder(outputData);
					
					break;
				case 1://Lossless with predictor
					////////////////////////SESSIO 1////////////////////////
					//Afegir codi per tal de descodificar la imatge codificada utilitzant predictor + entropy decoder
				    	entropyDecoder(outputData);
					for(int y = 0; y < height; y ++){
					for(int x = 0; x < width; x ++){
						int prediction = samplePredictor.predict(outputData, parameters.numberPredictionBands, y, x);
						outputData[parameters.numberPredictionBands][y][x] = outputData[parameters.numberPredictionBands][y][x] + prediction;
					}}
					break;
				case 2://Lossless and near-lossless with predictor
					////////////////////////SESSIO 2////////////////////////
					//Afegir codi per tal de codificar la imatge  codificada utilitzant predictor + quantitzacio + entropy decoder
				    	entropyDecoder(outputData);
					for(int y = 0; y < height; y ++){
					for(int x = 0; x < width; x ++){
						int prediction = samplePredictor.predict(outputData, parameters.numberPredictionBands, y, x);
						outputData[parameters.numberPredictionBands][y][x] = uq.dequantize(outputData[parameters.numberPredictionBands][y][x]);
						outputData[parameters.numberPredictionBands][y][x] = outputData[parameters.numberPredictionBands][y][x] + prediction;
						
					}}
					break;
				case 3://Lossless and near-lossless with an stat-of-the-art predictor
					entropyDecoder(outputData);
					for(int y = 0; y < height; y ++){
					for(int x = 0; x < width; x ++){
						outputData[parameters.numberPredictionBands][y][x] = predictor.decompress(outputData, z, y, x, parameters.numberPredictionBands, y, uq);
					}}
					break;
				}
				prepareBands(outputData, it);
			}
			
		
			
			image.close(it);
			if (verbose) {
				System.out.print("\rDecoding image finished");
			}
		}catch(UnsupportedOperationException e) {
			throw new Error("Unexpected exception ocurred "+e.getMessage());
		}catch(IndexOutOfBoundsException e) {
			e.printStackTrace();
		}catch(ClassCastException e) {
			throw new Error("Unexpected exception ocurred "+e.getMessage());
		}
	}
	
	
	/**
	 * Write the next band of the image.
	 * @param it the BSQ iterator over the image
	 */
	private void writeBand(int[][] band, RawImageIterator<int[]> it) {
		int height =  parameters.getImageGeometry()[CONS.HEIGHT];

		for(int i = 0; i < height; i ++) {
			it.next();
			it.set(band[i]);
		}
	}
	
	/**
	 * Save the first band and reordered bands for prediction
	 * @param bands an array with the prediction bands
	 * @param it the BSQ iterator over the image
	 */
	private void prepareBands(int[][][] bands, RawImageIterator<int[]> it) {
		int[][] tmpBand = bands[0];
		writeBand(bands[parameters.numberPredictionBands], it);		
		for(int i = 0; i < parameters.numberPredictionBands; i ++) {
			bands[i] = bands[i + 1];
		}
		bands[parameters.numberPredictionBands] = tmpBand;
	}
	
	
	
	
	
	
	
}
