/*
 * EMPORDA Software - More than an implementation of MHDC Recommendation for Image Data Compression
 * Copyright (C) 2011  Group on Interactive Coding of Images (GICI)
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Group on Interactive Coding of Images (GICI)
 * Department of Information and Communication Engineering
 * Autonomous University of Barcelona
 * 08193 - Bellaterra - Cerdanyola del Valles (Barcelona)
 * Spain
 *
 * http://gici.uab.es
 * http://sourceforge.net/projects/emporda
 * gici-info@deic.uab.es
 */
package emporda;

import GiciException.ErrorException;
import GiciException.ParameterException;
import GiciException.WarningException;
import GiciParser.BaseArgumentsParser;


/**
 * Arguments parser for EMPORDA application (extended from ArgumentsParser).
 *
 * @author Group on Interactive Coding of Images (GICI)
 * @version 1.0
 */
public class Parser extends BaseArgumentsParser {

	//ARGUMENTS SPECIFICATION
	private String[][] coderArguments = {

			{"-c", "--compress", "", "", "1", "1",
				"Compress de input image."
			},

			{"-d", "--decompress", "", "", "1", "1",
				"Decompress de input image."
			},

			{"-i", "--inputImage", "{string}", "", "1", "1",
				"It must be a raw image"
			},

			{"-o", "--outputFile", "{string}", "", "1", "1",
				"Output file.\n"
				+ "COMPRESSING: Output file name without extension.\n"
				+ "DECOMPRESSING: Output image file WITH extension."
			},

			{"-ig", "--inputImageGeometry", "{int int int int int int}", "", "1", "1",
				"Geometry of raw image data. Only nessesary if image is raw data. Parameters are:\n" + 
				"1- zSize (number of image components)\n"+
				"2- ySize (image height)\n"+ 
				"3- xSize (image width)\n"+
				"4- data type. Possible values are:\n" + 
				"	1- unsigned int (1 byte)\n"+
				"	2- unsigned int (2 bytes)\n"+ 
				"	3- signed int (2 bytes)\n" + 
				"	4- signed int (4 bytes)\n" + 
				"5- Byte order (0 if BIG ENDIAN, 1 if LITTLE ENDIAN)\n"+
				"6- 1 if 3 first components are RGB, or 0 otherwise\n" 

			},
			{"-f", "--option-file", "{String}", "", "0", "1",
				"sets the option file for the compression process"
			},
			{"-qs", "--quantization-step", "{int}", "", "0", "1",
				"sets the quantization step. This value is 1 by default."
			},
			{"-ec", "--encoder-type", "{int}", "", "0", "1",
				"The encoder type that will encode the image. \n"
				+ "0.- Lossless without predictor + entropy encoder. \n"
				+ "1.- Lossless with predictor + entropy encoder. \n"
				+ "2.- Lossless and near-lossless with predictor predictor + entropy encoder.\n"
				+ "3.- Lossless and near-lossless with state-of-the-art predictor + entropy encoder.\n"
			},
			{"-cm", "--context model", "{int}", "", "0", "1",
				"0.- No context model is used.\n"
				+ "1.- Context modelling is used during the encoding process.\n"
			},
			{"-pm", "--probability model", "{int}", "", "0", "1",
				"Probability model employed for the entropy coder. This value is 0 by default.\n"
				+ "0.- The probability is estimated using a full division operation.\n"
				+ "1.- The probability is estimated using a division implemented through a quantized Look Up Table. This option must be used with -qlut option.\n"
				+ "2.- The probability is estimated using only bitwise operators and witout division. When this option is used -wp and -up parameters must be the same value of form 2^X.\n"
			},
			{"-wp", "--windowProbability", "{int}", "", "0", "1",
				"Indicates the maximum number of symbols within the variable-size sliding window that are employed for the Entropy Coder to compute the probability of the context. Must be of the form 2^X. This value is 8 by default.\n"
			},
			{"-up", "--updateProbability", "{int}", "", "0", "1",
				"Indicates the number of symbols coded before updating the context probability in the Entropy Coder. Must be of the form 2^X. This value is 2048 by default.\n"
			},
			{"-h", "--help", "", "", "0", "1",
				"Displays help and exits program."
			}
			
	};

	//ARGUMENTS VARIABLES + Default Values
	private int action = 0; // Indicates if the program compress 0 or decompress 1
	private String inputFile = ""; // Input File
	private String outputFile = ""; // Output File
	private int[] imageGeometry = null; // Image geometry
	private boolean verbose = false; // The program shows informatin about the compression
	private String optionFile = null; //option file
	private int sampleOrder = 0;     // sample order of the image
	private int encoderType = 0;	// Type of encoder
	private int encoderWP = 0;	// window size to compute probability in the arithmetic encoder
	private int encoderUP = 0;	// how often the probaility of the arithmetic encoder is update (in encoded bits)
	private int AC_option = 1; 		// Way of probability table's creation
	private boolean debugMode = false; //debug mode
	private int quantizationStep = 1; //quantization step
	private int contextModel = 0;
	private int probabilityModel = 0;
	
	
	/** Receives program arguments and parses it, setting to arguments variables.
	 *
	 * @param arguments the array of strings passed at the command line
	 *
	 * @throws ParameterException when an invalid parsing is detected
	 * @throws ErrorException when some problem with method invocation occurs
	 */
	public Parser(String[] arguments) throws ParameterException, ErrorException {

		parse(coderArguments, arguments);

	}

	/**
	 * Parse an argument using parse functions from super class and put its value/s to the desired variable.
	 * This function is called from parse function of the super class.
	 *
	 * @param argFound number of parameter (the index of the array coderArguments)
	 * @param options the command line options of the argument
	 *
	 * @throws ParameterException when some error about parameters passed (type, number of params, etc.) occurs
	 */
	public void parseArgument(int argFound, String[] options) throws ParameterException {

		switch(argFound) {

		case 0: // -c --compress
			action = 0;
			coderArguments[1][4] = "0";
			coderArguments[1][5] = "0";
			break;

		case 1: // -d -decompress 
			action = 1;
			coderArguments[0][4] = "0";
			coderArguments[0][5] = "0";
			coderArguments[4][4] = "0";
			coderArguments[4][5] = "0";
			
			break;

		case  2: //-i  --inputImage
			inputFile = parseString(options);
			break;

		case  3: //-o  --outputFile
			outputFile = parseString(options);
			break;

		case  4: //-ig  --inputImageGeometry
			imageGeometry = parseIntegerArray(options, 6);
	        if(imageGeometry[CONS.TYPE] < 1 || imageGeometry[CONS.TYPE] > 4) {
                throw new ParameterException("datatype in image" + 
                " geometry must be: \n" + 
                "1.- Unsigned int (1 byte)\n" + 
                "2.- Unsigned int (2 bytes)\n" + 
                "3.- Signed int (2 bytes)\n" +
                "4.- igned int (4 bytes)");
            }
            coderArguments[1][4] = "0";
			coderArguments[1][5] = "0";
			break;

		case 5: //-f --option-file
			optionFile = parseString(options);
			break;
		
		case 6: //-qs --quantization-step
			quantizationStep = parseInteger(options);
			break;
		
		case 7: //-ec --encoder
			encoderType = parseInteger(options);
			break;
		case 8: //-cm --context-model
			contextModel = parseInteger(options);
			break;	
		case 9: //-pm
			probabilityModel = parseInteger(options);
			break;
		
		case 10: //-wp
			encoderWP = parseInteger(options);
			if(Integer.SIZE != Integer.numberOfLeadingZeros(encoderWP) + Integer.numberOfTrailingZeros(encoderWP) + 1){
				 throw new ParameterException("encoderWP must be of the form 2^X");
			}
			break;
			
		case 11: //-up
			encoderUP = parseInteger(options);
			if(Integer.SIZE != Integer.numberOfLeadingZeros(encoderUP) + Integer.numberOfTrailingZeros(encoderUP) + 1){
				 throw new ParameterException("encoderUP must be of the form 2^X");
			}
			break;

		case 12: //-h  --help
			System.out.println("Emporda");
			showArgsInfo();
			System.exit(0);
			break;
		default:
			throw new ParameterException("Unknown Parameter");
		}

	}


	/**
	 * get action.
	 * 
	 * @return action tells if compression or decompression must be done
	 */
	public int getAction() {
		return action;
	}

	/**
	 * get input file.
	 *
	 * @return the input file, from where the data will be read
	 */
	public String getInputFile() {

		return inputFile;
	}

	/** 
	 * get output file.
	 *
	 * @return the output file, where the generated data will be saved
	 */
	public String getOutputFile() {
		return outputFile;
	}
	
	/**
	 * get the image geometry.
	 * 
	 * @return an array with the image geometry information
	 */
	public int[] getImageGeometry() {

		try {

			if ((imageGeometry[CONS.BANDS] <= 0) || (imageGeometry[CONS.HEIGHT] <= 0)
					|| (imageGeometry[CONS.WIDTH] <= 0)) {

				throw new WarningException("Image dimensions in \".raw\" or \".img\" data files"
						+ " must be positive (\"-h\" displays help).");
			}

			if ((imageGeometry[CONS.TYPE] < 0) || (imageGeometry[CONS.TYPE] > 7)) {

				throw new WarningException("Image type in \".raw\" or \".img\" data must be between"
						+ "0 to 7 (\"-h\" displays help).");
			}   

			if ((imageGeometry[CONS.ENDIANESS] != 0) && (imageGeometry[CONS.ENDIANESS] != 1)) {

				throw new WarningException("Image byte order  in \".raw\" or \".img\" data must be 0"
						+ "or 1 (\"-h\" displays help).");
			}  

			if ((imageGeometry[CONS.RGB] != 0) && (imageGeometry[CONS.RGB] != 1)) {

				throw new WarningException("Image RGB specification in \".raw\" or \".img\" data must "
						+ "be between 0 or 1 (\"-h\" displays help).");
			}   

		} catch (WarningException e) {

			e.printStackTrace();
		}

		return imageGeometry;
	}

	/**
	 * get sampleOrder.
	 *
	 * @return the sample order of the image.
	 */
	public int getSampleOrder() {

		try {
			if (sampleOrder < 0 || sampleOrder > 2) {

				throw new WarningException("Sample Order must be 0, 1 or 2");
			}
		} catch (WarningException e) {

			e.printStackTrace();
		}

		return sampleOrder;
	}


	/**
	 * get verbose.
	 *
	 * @return true if output information must be given, false otherwise
	 */
	public boolean getVerbose() {
		return verbose;
	}

	/**
	 * get optionFile.
	 *
	 * @return the file where can be defines parameters for the algorithms
	 */
	public String getOptionFile() {
		return optionFile;
	}
	/**
	 * get debugMode.
	 *
	 * @return the value which indicates if debug mode is used
	 */
	public boolean getDebugMode() {
		return debugMode;
	}
	/**
	 * get quantizationStep.
	 * 
	 * @return the quantization step integer value
	 */
	public int getQuantizationStep() {
		return quantizationStep;
	}

	/**
	 * get EncoderType.
	 * 
	 * @return the Encoder type integer value
	 */
	public int getEncoderType() {
		return encoderType;
	}
	
	/**
	 * get ContextModel.
	 * 
	 * @return the ContextModel type integer value
	 */
	public int getContextModel() {
		return contextModel;
	}
	
	/**
	 * get ProbabilityModel.
	 * 
	 * @return the ProbabilityModel type integer value
	 */
	public int getProbabilityModel() {
		return probabilityModel;
	}
	
	/**
	 * get window Probability update.
	 * 
	 * @return the encoderWP type integer value
	 */
	public int getEncoderWP(){
		return encoderWP;
	}
	
	/**
	 * get update probability.
	 * 
	 * @return the encoderUP type integer value
	 */
	public int getEncoderUP(){
		return encoderUP;
	}
	
	/**
	 * get AC_option
	 * 
	 * @return the way prob tables will be created
	 */
	public int getAC_option() {
		return AC_option; 
	}
}

