"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = exports.BUNDLER_DEPENDENCIES_CACHE = exports.DEPENDENCY_EXCLUDES = void 0;
const path = require("path");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const core_1 = require("aws-cdk-lib/core");
const packaging_1 = require("./packaging");
/**
 * Dependency files to exclude from the asset hash.
 */
exports.DEPENDENCY_EXCLUDES = ['*.pyc'];
/**
 * The location in the image that the bundler image caches dependencies.
 */
exports.BUNDLER_DEPENDENCIES_CACHE = '/var/dependencies';
/**
 * Produce bundled Lambda asset code
 */
class Bundling {
    static bundle(options) {
        return aws_lambda_1.Code.fromAsset(options.entry, {
            assetHash: options.assetHash,
            assetHashType: options.assetHashType,
            exclude: exports.DEPENDENCY_EXCLUDES,
            bundling: options.skip ? undefined : new Bundling(options),
        });
    }
    constructor(props) {
        const { entry, runtime, architecture = aws_lambda_1.Architecture.X86_64, outputPathSuffix = '', image, poetryIncludeHashes, commandHooks, assetExcludes = [], } = props;
        const outputPath = path.posix.join(core_1.AssetStaging.BUNDLING_OUTPUT_DIR, outputPathSuffix);
        const bundlingCommands = this.createBundlingCommand({
            entry,
            inputDir: core_1.AssetStaging.BUNDLING_INPUT_DIR,
            outputDir: outputPath,
            poetryIncludeHashes,
            commandHooks,
            assetExcludes,
        });
        this.image = image ?? core_1.DockerImage.fromBuild(path.join(__dirname, '../lib'), {
            buildArgs: {
                ...props.buildArgs,
                IMAGE: runtime.bundlingImage.image,
            },
            platform: architecture.dockerPlatform,
        });
        this.command = props.command ?? ['bash', '-c', chain(bundlingCommands)];
        this.entrypoint = props.entrypoint;
        this.volumes = props.volumes;
        this.volumesFrom = props.volumesFrom;
        this.environment = props.environment;
        this.workingDirectory = props.workingDirectory;
        this.user = props.user;
        this.securityOpt = props.securityOpt;
        this.network = props.network;
        this.bundlingFileAccess = props.bundlingFileAccess;
    }
    createBundlingCommand(options) {
        const packaging = packaging_1.Packaging.fromEntry(options.entry, options.poetryIncludeHashes);
        let bundlingCommands = [];
        bundlingCommands.push(...options.commandHooks?.beforeBundling(options.inputDir, options.outputDir) ?? []);
        const exclusionStr = options.assetExcludes?.map(item => `--exclude='${item}'`).join(' ');
        bundlingCommands.push([
            'rsync', '-rLv', exclusionStr ?? '', `${options.inputDir}/`, options.outputDir,
        ].filter(item => item).join(' '));
        bundlingCommands.push(`cd ${options.outputDir}`);
        bundlingCommands.push(packaging.exportCommand ?? '');
        if (packaging.dependenciesFile) {
            bundlingCommands.push(`python -m pip install -r ${packaging_1.DependenciesFile.PIP} -t ${options.outputDir}`);
        }
        bundlingCommands.push(...options.commandHooks?.afterBundling(options.inputDir, options.outputDir) ?? []);
        return bundlingCommands;
    }
}
exports.Bundling = Bundling;
/**
 * Chain commands
 */
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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