from typing import List, Optional, Dict, Iterator, Callable
from timeit import default_timer as timer
import tempfile
import logging
import os

import bugzoo
import bugzoo.localization.suspiciousness as metrics
from bugzoo.core.fileline import FileLine, FileLineSet
from bugzoo.core.bug import Bug
from bugzoo.core.patch import Patch
from bugzoo.core.coverage import TestSuiteCoverage
from bugzoo.core.spectra import Spectra
from bugzoo.localization import SuspiciousnessMetric, Localization
from bugzoo.testing import TestCase

import darjeeling.filters as filters
from .snippet import SnippetDatabase, Snippet
from .source import SourceFile
from .util import get_lines


class Problem(object):
    """
    Used to provide a description of a problem (i.e., a bug), and to hold
    information pertinent to its solution (e.g., coverage, transformations).
    """
    def __init__(self,
                 bz: bugzoo.BugZoo,
                 bug: Bug,
                 *,
                 suspiciousness_metric: Optional[SuspiciousnessMetric] = None,
                 in_files: List[str],
                 restrict_to_lines: Optional[FileLineSet] = None,
                 cache_coverage: bool = True,
                 verbose: bool = False,
                 logger: Optional[logging.Logger] = None,
                 line_coverage_filters: Optional[List[Callable[[str], bool]]] = None
                 ) -> None:
        """
        Constructs a Darjeeling problem description.

        Params:
            bug: A description of the faulty program.
            in_files: An optional list that can be used to restrict the set of
                transformations to those that occur in any files belonging to
                that list. If no list is provided, all source code files will
                be included.
            in_functions: An optional list that can be used to restrict the set
                of transformations to those that occur in any function whose
                name appears in the given list. If no list is provided, no
                filtering of transformations based on the function to which
                they belong will occur.
        """
        assert len(in_files) > 0
        self.__bug = bug
        self.__verbose = verbose

        # establish logging mechanism
        # * stream logging information to stdout
        if logger:
            self.__logger = logger
        else:
            self.__logger = \
                logging.getLogger('darjeeling.problem').getChild(bug.name)
            self.__logger.setLevel(logging.DEBUG)
            self.__logger.addHandler(logging.StreamHandler())
        self.__logger.debug("creating problem for bug: %s", bug.name)

        if suspiciousness_metric is None:
            self.__logger.debug("no suspiciousness metric provided: using Tarantula as a default.")
            suspiciousness_metric = metrics.tarantula

        # fetch coverage information
        if cache_coverage:
            self.__logger.debug("fetching coverage information from BugZoo")
            self.__coverage = bz.bugs.coverage(bug)
            self.__logger.debug("fetched coverage information from BugZoo")
        else:
            self.__logger.debug("computing coverage information")
            try:
                container = bz.containers.provision(bug)
                self.__coverage = bz.coverage.coverage(container, bug.tests)
            finally:
                del bz.containers[container.uid]
            self.__logger.debug("computed coverage information")

        # restrict coverage information to specified files
        self.__logger.debug("restricting coverage information to files:\n* %s",
                            '\n* '.join(in_files))
        self.__coverage = self.__coverage.restricted_to_files(in_files)
        self.__logger.debug("restricted coverage information.")

        # determine the passing and failing tests by using coverage information
        self.__logger.debug("using test execution used to generate coverage to determine passing and failing tests")

        self.__tests_failing = set()
        self.__tests_passing = set()
        for test_name in self.__coverage:
            test = bug.harness[test_name]
            test_coverage = self.__coverage[test_name]
            if test_coverage.outcome.passed:
                self.__tests_passing.add(test)
            else:
                self.__tests_failing.add(test)

        # TODO throw an error if there are no failing tests
        self.__logger.info("determined passing and failing tests")
        self.__logger.info("* passing tests: %s", ', '.join([t.name for t in self.__tests_passing]))
        self.__logger.info("* failing tests: %s", ', '.join([t.name for t in self.__tests_failing]))

        # determine the implicated lines
        # 0. we already restricted to lines that occur in specified files
        # 1. restrict to lines covered by failing tests
        # 3. restrict to lines with suspiciousness greater than zero
        # 4. restrict to lines that are optionally provided
        self.__logger.info("Determining implicated lines")
        self.__lines = self.__coverage.failing.lines

        if restrict_to_lines is not None:
            self.__lines = self.__lines.intersection(restrict_to_lines)

        # cache contents of implicated files
        t_start = timer()
        self.__logger.debug("storing contents of source code files")
        self.__sources = {} # type: Dict[str, SourceFile]
        _, fn_host_temp = tempfile.mkstemp(prefix='.darjeeling')
        ctr_source_files = bz.containers.provision(bug)
        try:
            for fn in self.__lines.files:
                fn_ctr = os.path.join(bug.source_dir, fn)
                bz.containers.copy_from(ctr_source_files, fn_ctr, fn_host_temp)
                self.__sources[fn] = SourceFile(fn, get_lines(fn_host_temp))
            duration = timer() - t_start
        finally:
            os.remove(fn_host_temp)
            del bz.containers[ctr_source_files.uid]
        self.__logger.debug("stored contents of source code files (took %.1f seconds)",
                            duration)

        # restrict attention to statements
        # FIXME for now, we approximate this -- going forward, we can use
        #   Rooibos to determine transformation targets
        line_content_filters = [
            filters.ends_with_semi_colon,
            filters.has_balanced_delimiters
        ]
        if line_coverage_filters:
            line_content_filters += line_coverage_filters
        for fltr_content in line_content_filters:
            fltr_line = \
                lambda fl: fltr_content(self.__sources[fl.filename][fl.num])
            self.__lines = self.__lines.filter(fltr_line)

        # compute fault localization
        self.__coverage = \
            self.__coverage.restricted_to_files(self.__lines.files)
        self.__spectra = Spectra.from_coverage(self.__coverage)
        self.__localization = \
            Localization.from_spectra(self.__spectra, suspiciousness_metric)
        self.__localization = \
            self.__localization.restricted_to_lines(self.__lines)
        self.__logger.info("removing non-suspicious lines from consideration")
        num_lines_before = len(self.__lines)
        self.__lines = \
            self.__lines.filter(lambda l: self.__localization.score(l) > 0)
        num_lines_after = len(self.__lines)
        num_lines_removed = num_lines_before - num_lines_after
        self.__logger.info("removed %d non-suspicious lines from consideration",
                           num_lines_removed)

        # TODO raise an exception if there are no implicated lines

        # report implicated lines and files
        self.__logger.info("implicated lines [%d]:\n%s",
                           len(self.__lines), self.__lines)
        self.__logger.info("implicated files [%d]:\n* %s",
                           len(self.__lines.files),
                           '\n* '.join(self.__lines.files))

        # construct the snippet database from the parts of the program that
        # were executed by the test suite (both passing and failing tests)
        # TODO add a snippet extractor component
        self.__logger.info("constructing snippet database")
        self.__snippets = SnippetDatabase()

        # TODO allow additional snippet filters to be provided as params
        snippet_filters = [
            filters.ends_with_semi_colon,
            filters.has_balanced_delimiters
        ]

        for line in self.__coverage.lines:
            content = self.__sources[line.filename][line.num].strip()
            if all(fltr(content) for fltr in snippet_filters):
                # self.__logger.debug("* found snippet at %s: %s", line, content)
                snippet = Snippet(content)
                self.__snippets.add(snippet, origin=line)

        self.__logger.info("construct snippet database: %d snippets",
                           len(self.__snippets))
        for fn in self.__lines.files:
            self.__logger.info("* %d unique snippets in %s",
                               len(list(self.__snippets.in_file(fn))), fn)

        self.__logger.debug("reducing memory footprint by discarding extraneous data")
        self.__coverage.restricted_to_files(self.__lines.files)
        self.__spectra.restricted_to_files(self.__lines.files)
        extraneous_source_fns = \
            set(self.__sources.keys()) - set(self.__lines.files)
        for fn in extraneous_source_fns:
            del self.__sources[fn]
        self.__logger.debug("finished reducing memory footprint")

    @property
    def snippets(self) -> SnippetDatabase:
        """
        The snippet database that should be used to generate new code.
        """
        return self.__snippets

    @property
    def bug(self) -> Bug:
        """
        A description of the bug, provided by BugZoo.
        """
        return self.__bug

    @property
    def tests(self) -> Iterator[TestCase]:
        """
        Returns an iterator over the tests for this problem.
        """
        for test in self.__tests_failing:
            yield test
        for test in self.__tests_passing:
            yield test

    @property
    def tests_failing(self) -> Iterator[TestCase]:
        """
        Returns an iterator over the failing tests for this problem.
        """
        return self.__tests_failing.__iter__()

    @property
    def tests_passing(self) -> Iterator[TestCase]:
        """
        Returns an iterator over the passing tests for this problem.
        """
        return self.__tests_passing.__iter__()

    @property
    def logger(self) -> logging.Logger:
        """
        The logger that should be used to log output for this problem.
        """
        return self.__logger

    @property
    def localization(self) -> Localization:
        """
        The fault localization for this problem is used to encode the relative
        suspiciousness of source code lines.
        """
        return self.__localization

    @property
    def coverage(self) -> TestSuiteCoverage:
        """
        Line coverage information for each test within the test suite for the
        program under repair.
        """
        return self.__coverage

    @property
    def spectra(self) -> Spectra:
        """
        Provides a concise summary of the number of passing and failing tests
        that cover each implicated line.
        """
        return self.__spectra

    @property
    def lines(self) -> Iterator[FileLine]:
        """
        Returns an iterator over the lines that are implicated by the
        description of this problem.
        """
        return self.__lines.__iter__()

    def source(self, fn: str) -> SourceFile:
        """
        Returns the contents of a given source code file for this problem.
        """
        return self.__sources[fn]

    def check_sanity(self,
                     expected_to_pass: List[TestCase],
                     expected_to_fail: List[TestCase]
                     ) -> bool:
        """

        Parameters:
            expected_to_pass: a list of test cases for the program that are
                expected to pass.
            expected_to_fail: a list o test cases for the program that are
                expected to fail.

        Returns:
            True if the outcomes of the test executions match those that are
            expected by the parameters to this method.

        Raises:
            errors.UnexpectedTestOutcomes: if the outcomes of the program
                under test, observed while computing coverage information for
                the problem, differ from those that were expected.
        """
        # determine passing and failing tests
        self.__logger.debug("sanity checking...")
        raise NotImplementedError
