# Copyright (c) Meta Platforms, Inc. and affiliates.
# All rights reserved.
#
# This source code is licensed under the terms described in the LICENSE file in
# the root directory of this source tree.

import os

import click
import yaml
from prompt_toolkit import prompt
from prompt_toolkit.validation import Validator

from llama_stack_client.lib.cli.constants import LLAMA_STACK_CLIENT_CONFIG_DIR, get_config_file_path


def get_config():
    config_file = get_config_file_path()
    if config_file.exists():
        with open(config_file, "r") as f:
            return yaml.safe_load(f)
    return None


@click.command()
@click.option("--endpoint", type=str, help="Llama Stack distribution endpoint", default="")
@click.option("--api-key", type=str, help="Llama Stack distribution API key", default="")
def configure(endpoint: str | None, api_key: str | None):
    """Configure Llama Stack Client CLI."""
    os.makedirs(LLAMA_STACK_CLIENT_CONFIG_DIR, exist_ok=True)
    config_path = get_config_file_path()

    if endpoint != "":
        final_endpoint = endpoint
    else:
        final_endpoint = prompt(
            "> Enter the endpoint of the Llama Stack distribution server: ",
            validator=Validator.from_callable(
                lambda x: len(x) > 0,
                error_message="Endpoint cannot be empty, please enter a valid endpoint",
            ),
        )

    if api_key != "":
        final_api_key = api_key
    else:
        final_api_key = prompt(
            "> Enter the API key (leave empty if no key is needed): ",
        )

    # Prepare config dict before writing it
    config_dict = {
        "endpoint": final_endpoint,
    }
    if final_api_key != "":
        config_dict["api_key"] = final_api_key

    with open(config_path, "w") as f:
        f.write(
            yaml.dump(
                config_dict,
                sort_keys=True,
            )
        )

    print(f"Done! You can now use the Llama Stack Client CLI with endpoint {final_endpoint}")
