"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.mergePrincipal = exports.AttachedPolicies = exports.generatePolicyName = exports.undefinedIfEmpty = void 0;
const core_1 = require("@aws-cdk/core");
const MAX_POLICY_NAME_LEN = 128;
function undefinedIfEmpty(f) {
    return core_1.Lazy.listValue({ produce: () => {
            const array = f();
            return (array && array.length > 0) ? array : undefined;
        } });
}
exports.undefinedIfEmpty = undefinedIfEmpty;
/**
 * Used to generate a unique policy name based on the policy resource construct.
 * The logical ID of the resource is a great candidate as long as it doesn't exceed
 * 128 characters, so we take the last 128 characters (in order to make sure the hash
 * is there).
 */
function generatePolicyName(scope, logicalId) {
    // as logicalId is itself a Token, resolve it first
    const resolvedLogicalId = core_1.Tokenization.resolve(logicalId, {
        scope,
        resolver: new core_1.DefaultTokenResolver(new core_1.StringConcat()),
    });
    return lastNCharacters(resolvedLogicalId, MAX_POLICY_NAME_LEN);
}
exports.generatePolicyName = generatePolicyName;
/**
 * Returns a string composed of the last n characters of str.
 * If str is shorter than n, returns str.
 *
 * @param str the string to return the last n characters of
 * @param n how many characters to return
 */
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex, str.length);
}
/**
 * Helper class that maintains the set of attached policies for a principal.
 */
class AttachedPolicies {
    constructor() {
        this.policies = new Array();
    }
    /**
     * Adds a policy to the list of attached policies.
     *
     * If this policy is already, attached, returns false.
     * If there is another policy attached with the same name, throws an exception.
     */
    attach(policy) {
        if (this.policies.find(p => p === policy)) {
            return; // already attached
        }
        if (this.policies.find(p => p.policyName === policy.policyName)) {
            throw new Error(`A policy named "${policy.policyName}" is already attached`);
        }
        this.policies.push(policy);
    }
}
exports.AttachedPolicies = AttachedPolicies;
/**
 * Merge two dictionaries that represent IAM principals
 */
function mergePrincipal(target, source) {
    for (const key of Object.keys(source)) {
        target[key] = target[key] || [];
        let value = source[key];
        if (!Array.isArray(value)) {
            value = [value];
        }
        target[key].push(...value);
    }
    return target;
}
exports.mergePrincipal = mergePrincipal;
//# sourceMappingURL=data:application/json;base64,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