"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaToS3 = void 0;
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
/**
 * @summary The LambdaToS3 class.
 */
class LambdaToS3 extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the LambdaToS3 class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {LambdaToS3Props} props - user provided props for the construct.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Setup the Lambda function
        this.lambdaFunction = defaults.buildLambdaFunction(this, {
            deployLambda: props.deployLambda,
            existingLambdaObj: props.existingLambdaObj,
            lambdaFunctionProps: props.lambdaFunctionProps
        });
        // Setup the S3 bucket
        this.s3Bucket = defaults.buildS3Bucket(this, {
            deployBucket: props.deployBucket,
            existingBucketObj: props.existingBucketObj,
            bucketProps: props.bucketProps
        });
        // Configure environment variables
        this.lambdaFunction.addEnvironment('S3_BUCKET_NAME', this.s3Bucket.bucketName);
        // Add the requested or default bucket permissions
        if (props.hasOwnProperty('bucketPermissions') && props.bucketPermissions) {
            if (props.bucketPermissions.includes('Delete')) {
                this.s3Bucket.grantDelete(this.lambdaFunction.grantPrincipal);
            }
            if (props.bucketPermissions.includes('Put')) {
                this.s3Bucket.grantPut(this.lambdaFunction.grantPrincipal);
            }
            if (props.bucketPermissions.includes('Read')) {
                this.s3Bucket.grantRead(this.lambdaFunction.grantPrincipal);
            }
            if (props.bucketPermissions.includes('ReadWrite')) {
                this.s3Bucket.grantReadWrite(this.lambdaFunction.grantPrincipal);
            }
            if (props.bucketPermissions.includes('Write')) {
                this.s3Bucket.grantWrite(this.lambdaFunction.grantPrincipal);
            }
        }
        else {
            this.s3Bucket.grantReadWrite(this.lambdaFunction.grantPrincipal);
        }
        // Add appropriate metadata
        const s3BucketResource = this.s3Bucket.node.findChild('Resource');
        s3BucketResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W51',
                        reason: `This S3 bucket Bucket does not need a bucket policy`
                    }]
            }
        };
    }
}
exports.LambdaToS3 = LambdaToS3;
//# sourceMappingURL=data:application/json;base64,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