/**
 * @license
 * Visual Blocks Language
 *
 * Copyright 2012 Google Inc.
 * https://developers.google.com/blockly/
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * @fileoverview Generating JavaScript for procedure blocks.
 * @author fraser@google.com (Neil Fraser)
 */
"use strict";

goog.provide("Blockly.JavaScript.procedures");

goog.require("Blockly.JavaScript");

Blockly.JavaScript["procedures_defreturn"] = function(block) {
  // Define a procedure with a return value.
  var funcName = Blockly.JavaScript.variableDB_.getName(
    block.getFieldValue("NAME"),
    Blockly.Procedures.NAME_TYPE
  );
  var branch = Blockly.JavaScript.statementToCode(block, "STACK");
  if (Blockly.JavaScript.STATEMENT_PREFIX) {
    var id = block.id.replace(/\$/g, "$$$$"); // Issue 251.
    branch =
      Blockly.JavaScript.prefixLines(
        Blockly.JavaScript.STATEMENT_PREFIX.replace(/%1/g, "'" + id + "'"),
        Blockly.JavaScript.INDENT
      ) + branch;
  }
  if (Blockly.JavaScript.INFINITE_LOOP_TRAP) {
    branch =
      Blockly.JavaScript.INFINITE_LOOP_TRAP.replace(
        /%1/g,
        "'" + block.id + "'"
      ) + branch;
  }
  var returnValue =
    Blockly.JavaScript.valueToCode(
      block,
      "RETURN",
      Blockly.JavaScript.ORDER_NONE
    ) || "";
  if (returnValue) {
    returnValue = "  return " + returnValue + ";\n";
  }
  var args = [];
  for (var i = 0; i < block.arguments_.length; i++) {
    args[i] = Blockly.JavaScript.variableDB_.getName(
      block.arguments_[i],
      Blockly.Variables.NAME_TYPE
    );
  }
  var code =
    "async function " +
    funcName +
    "(" +
    args.join(", ") +
    ") {\n" +
    branch +
    returnValue +
    "}";
  code = Blockly.JavaScript.scrub_(block, code);
  // Add % so as not to collide with helper functions in definitions list.
  Blockly.JavaScript.definitions_["%" + funcName] = code;
  return null;
};

// Defining a procedure without a return value uses the same generator as
// a procedure with a return value.
Blockly.JavaScript["procedures_defnoreturn"] =
  Blockly.JavaScript["procedures_defreturn"];

Blockly.JavaScript["procedures_callreturn"] = function(block) {
  // Call a procedure with a return value.
  var funcName = Blockly.JavaScript.variableDB_.getName(
    block.getFieldValue("NAME"),
    Blockly.Procedures.NAME_TYPE
  );
  var args = [];
  for (var i = 0; i < block.arguments_.length; i++) {
    args[i] =
      Blockly.JavaScript.valueToCode(
        block,
        "ARG" + i,
        Blockly.JavaScript.ORDER_COMMA
      ) || "null";
  }
  var code = funcName + "(" + args.join(", ") + ")";
  return [code, Blockly.JavaScript.ORDER_FUNCTION_CALL];
};

Blockly.JavaScript["procedures_callnoreturn"] = function(block) {
  // Call a procedure with no return value.
  var funcName = Blockly.JavaScript.variableDB_.getName(
    block.getFieldValue("NAME"),
    Blockly.Procedures.NAME_TYPE
  );
  var args = [];
  for (var i = 0; i < block.arguments_.length; i++) {
    args[i] =
      Blockly.JavaScript.valueToCode(
        block,
        "ARG" + i,
        Blockly.JavaScript.ORDER_COMMA
      ) || "null";
  }
  var code = funcName + "(" + args.join(", ") + ");\n";
  return code;
};

Blockly.JavaScript["procedures_ifreturn"] = function(block) {
  // Conditionally return value from a procedure.
  var condition =
    Blockly.JavaScript.valueToCode(
      block,
      "CONDITION",
      Blockly.JavaScript.ORDER_NONE
    ) || "false";
  var code = "if (" + condition + ") {\n";
  if (block.hasReturnValue_) {
    var value =
      Blockly.JavaScript.valueToCode(
        block,
        "VALUE",
        Blockly.JavaScript.ORDER_NONE
      ) || "null";
    code += "  return " + value + ";\n";
  } else {
    code += "  return;\n";
  }
  code += "}\n";
  return code;
};
