"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const principals_1 = require("./principals");
const util_1 = require("./util");
class PolicyDocument extends cdk.Token {
    /**
     * Creates a new IAM policy document.
     * @param defaultDocument An IAM policy document to use as an initial
     * policy. All statements of this document will be copied in.
     */
    constructor(baseDocument = {}) {
        super();
        this.baseDocument = baseDocument;
        this.statements = new Array();
        this._autoAssignSids = false;
    }
    /**
     * Will automatically assign a unique SID to each statement, unless an SID is provided.
     */
    autoAssignSids() {
        this._autoAssignSids = true;
    }
    resolve(_context) {
        if (this.isEmpty) {
            return undefined;
        }
        const doc = {
            ...this.baseDocument,
            Statement: (this.baseDocument.Statement || []).concat(this.statements),
            Version: this.baseDocument.Version || '2012-10-17'
        };
        return doc;
    }
    /**
     * Removes duplicate statements
     */
    postProcess(input, _context) {
        if (!input || !input.Statement) {
            return input;
        }
        const jsonStatements = new Set();
        const uniqueStatements = [];
        for (const statement of input.Statement) {
            const jsonStatement = JSON.stringify(statement);
            if (!jsonStatements.has(jsonStatement)) {
                uniqueStatements.push(statement);
                jsonStatements.add(jsonStatement);
            }
        }
        // assign unique SIDs (the statement index) if `autoAssignSids` is enabled
        const statements = uniqueStatements.map((s, i) => {
            if (this._autoAssignSids && !s.Sid) {
                s.Sid = i.toString();
            }
            return s;
        });
        return {
            ...input,
            Statement: statements
        };
    }
    get isEmpty() {
        return this.statements.length === 0;
    }
    /**
     * The number of statements already added to this policy.
     * Can be used, for example, to generate uniuqe "sid"s within the policy.
     */
    get statementCount() {
        return this.statements.length;
    }
    /**
     * Adds a statement to the policy document.
     *
     * @param statement the statement to add.
     */
    addStatement(statement) {
        this.statements.push(statement);
        return this;
    }
}
exports.PolicyDocument = PolicyDocument;
/**
 * Represents a statement in an IAM policy document.
 */
class PolicyStatement extends cdk.Token {
    constructor(effect = PolicyStatementEffect.Allow) {
        super();
        this.action = new Array();
        this.principal = {};
        this.resource = new Array();
        this.condition = {};
        this.effect = effect;
    }
    //
    // Actions
    //
    addAction(action) {
        this.action.push(action);
        return this;
    }
    addActions(...actions) {
        actions.forEach(action => this.addAction(action));
        return this;
    }
    //
    // Principal
    //
    /**
     * Indicates if this permission has a "Principal" section.
     */
    get hasPrincipal() {
        return Object.keys(this.principal).length > 0;
    }
    addPrincipal(principal) {
        const fragment = principal.policyFragment;
        util_1.mergePrincipal(this.principal, fragment.principalJson);
        this.addConditions(fragment.conditions);
        return this;
    }
    addAwsPrincipal(arn) {
        return this.addPrincipal(new principals_1.ArnPrincipal(arn));
    }
    addAwsAccountPrincipal(accountId) {
        return this.addPrincipal(new principals_1.AccountPrincipal(accountId));
    }
    addArnPrincipal(arn) {
        return this.addAwsPrincipal(arn);
    }
    /**
     * Adds a service principal to this policy statement.
     *
     * @param service the service name for which a service principal is requested (e.g: `s3.amazonaws.com`).
     * @param opts    options for adding the service principal (such as specifying a principal in a different region)
     */
    addServicePrincipal(service, opts) {
        return this.addPrincipal(new principals_1.ServicePrincipal(service, opts));
    }
    addFederatedPrincipal(federated, conditions) {
        return this.addPrincipal(new principals_1.FederatedPrincipal(federated, conditions));
    }
    addAccountRootPrincipal() {
        return this.addPrincipal(new principals_1.AccountRootPrincipal());
    }
    addCanonicalUserPrincipal(canonicalUserId) {
        return this.addPrincipal(new principals_1.CanonicalUserPrincipal(canonicalUserId));
    }
    addAnyPrincipal() {
        return this.addPrincipal(new principals_1.Anyone());
    }
    //
    // Resources
    //
    addResource(arn) {
        this.resource.push(arn);
        return this;
    }
    /**
     * Adds a ``"*"`` resource to this statement.
     */
    addAllResources() {
        return this.addResource('*');
    }
    addResources(...arns) {
        arns.forEach(r => this.addResource(r));
        return this;
    }
    /**
     * Indicates if this permission as at least one resource associated with it.
     */
    get hasResource() {
        return this.resource && this.resource.length > 0;
    }
    /**
     * @deprecated Use `statement.sid = value`
     */
    describe(sid) {
        this.sid = sid;
        return this;
    }
    //
    // Effect
    //
    /**
     * Sets the permission effect to allow access to resources.
     */
    allow() {
        this.effect = PolicyStatementEffect.Allow;
        return this;
    }
    /**
     * Sets the permission effect to deny access to resources.
     */
    deny() {
        this.effect = PolicyStatementEffect.Deny;
        return this;
    }
    //
    // Condition
    //
    /**
     * Add a condition to the Policy
     */
    addCondition(key, value) {
        this.condition[key] = value;
        return this;
    }
    /**
     * Add multiple conditions to the Policy
     */
    addConditions(conditions) {
        Object.keys(conditions).map(key => {
            this.addCondition(key, conditions[key]);
        });
        return this;
    }
    /**
     * Add a condition to the Policy.
     *
     * @deprecated For backwards compatibility. Use addCondition() instead.
     */
    setCondition(key, value) {
        return this.addCondition(key, value);
    }
    limitToAccount(accountId) {
        return this.addCondition('StringEquals', new cdk.Token(() => {
            return { 'sts:ExternalId': accountId };
        }));
    }
    //
    // Serialization
    //
    resolve(_context) {
        return this.toJson();
    }
    toJson() {
        return {
            Action: _norm(this.action),
            Condition: _norm(this.condition),
            Effect: _norm(this.effect),
            Principal: _normPrincipal(this.principal),
            Resource: _norm(this.resource),
            Sid: _norm(this.sid),
        };
        function _norm(values) {
            if (typeof (values) === 'undefined') {
                return undefined;
            }
            if (cdk.Token.isToken(values)) {
                return values;
            }
            if (Array.isArray(values)) {
                if (!values || values.length === 0) {
                    return undefined;
                }
                if (values.length === 1) {
                    return values[0];
                }
                return values;
            }
            if (typeof (values) === 'object') {
                if (Object.keys(values).length === 0) {
                    return undefined;
                }
            }
            return values;
        }
        function _normPrincipal(principal) {
            const keys = Object.keys(principal);
            if (keys.length === 0) {
                return undefined;
            }
            const result = {};
            for (const key of keys) {
                const normVal = _norm(principal[key]);
                if (normVal) {
                    result[key] = normVal;
                }
            }
            if (Object.keys(result).length === 1 && result.AWS === '*') {
                return '*';
            }
            return result;
        }
    }
}
exports.PolicyStatement = PolicyStatement;
var PolicyStatementEffect;
(function (PolicyStatementEffect) {
    PolicyStatementEffect["Allow"] = "Allow";
    PolicyStatementEffect["Deny"] = "Deny";
})(PolicyStatementEffect = exports.PolicyStatementEffect || (exports.PolicyStatementEffect = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicG9saWN5LWRvY3VtZW50LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsicG9saWN5LWRvY3VtZW50LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsb0NBQXFDO0FBQ3JDLDZDQUMrRjtBQUMvRixpQ0FBd0M7QUFFeEMsTUFBYSxjQUFlLFNBQVEsR0FBRyxDQUFDLEtBQUs7SUFJM0M7Ozs7T0FJRztJQUNILFlBQTZCLGVBQW9CLEVBQUU7UUFDakQsS0FBSyxFQUFFLENBQUM7UUFEbUIsaUJBQVksR0FBWixZQUFZLENBQVU7UUFSM0MsZUFBVSxHQUFHLElBQUksS0FBSyxFQUFtQixDQUFDO1FBQzFDLG9CQUFlLEdBQUcsS0FBSyxDQUFDO0lBU2hDLENBQUM7SUFFRDs7T0FFRztJQUNJLGNBQWM7UUFDbkIsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLENBQUM7SUFDOUIsQ0FBQztJQUVNLE9BQU8sQ0FBQyxRQUE2QjtRQUMxQyxJQUFJLElBQUksQ0FBQyxPQUFPLEVBQUU7WUFDaEIsT0FBTyxTQUFTLENBQUM7U0FDbEI7UUFFRCxNQUFNLEdBQUcsR0FBRztZQUNWLEdBQUcsSUFBSSxDQUFDLFlBQVk7WUFDcEIsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxTQUFTLElBQUksRUFBRSxDQUFDLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUM7WUFDdEUsT0FBTyxFQUFFLElBQUksQ0FBQyxZQUFZLENBQUMsT0FBTyxJQUFJLFlBQVk7U0FDbkQsQ0FBQztRQUVGLE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQztJQUVEOztPQUVHO0lBQ0ksV0FBVyxDQUFDLEtBQVUsRUFBRSxRQUE2QjtRQUMxRCxJQUFJLENBQUMsS0FBSyxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRTtZQUM5QixPQUFPLEtBQUssQ0FBQztTQUNkO1FBRUQsTUFBTSxjQUFjLEdBQUcsSUFBSSxHQUFHLEVBQVUsQ0FBQztRQUN6QyxNQUFNLGdCQUFnQixHQUFVLEVBQUUsQ0FBQztRQUVuQyxLQUFLLE1BQU0sU0FBUyxJQUFJLEtBQUssQ0FBQyxTQUFTLEVBQUU7WUFDdkMsTUFBTSxhQUFhLEdBQUcsSUFBSSxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQztZQUNoRCxJQUFJLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsRUFBRTtnQkFDdEMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO2dCQUNqQyxjQUFjLENBQUMsR0FBRyxDQUFDLGFBQWEsQ0FBQyxDQUFDO2FBQ25DO1NBQ0Y7UUFFRCwwRUFBMEU7UUFDMUUsTUFBTSxVQUFVLEdBQUcsZ0JBQWdCLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFO1lBQy9DLElBQUksSUFBSSxDQUFDLGVBQWUsSUFBSSxDQUFDLENBQUMsQ0FBQyxHQUFHLEVBQUU7Z0JBQ2xDLENBQUMsQ0FBQyxHQUFHLEdBQUcsQ0FBQyxDQUFDLFFBQVEsRUFBRSxDQUFDO2FBQ3RCO1lBRUQsT0FBTyxDQUFDLENBQUM7UUFDWCxDQUFDLENBQUMsQ0FBQztRQUVILE9BQU87WUFDTCxHQUFHLEtBQUs7WUFDUixTQUFTLEVBQUUsVUFBVTtTQUN0QixDQUFDO0lBQ0osQ0FBQztJQUVELElBQUksT0FBTztRQUNULE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLEtBQUssQ0FBQyxDQUFDO0lBQ3RDLENBQUM7SUFFRDs7O09BR0c7SUFDSCxJQUFJLGNBQWM7UUFDaEIsT0FBTyxJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQztJQUNoQyxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLFlBQVksQ0FBQyxTQUEwQjtRQUM1QyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNoQyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7Q0FDRjtBQXpGRCx3Q0F5RkM7QUFFRDs7R0FFRztBQUNILE1BQWEsZUFBZ0IsU0FBUSxHQUFHLENBQUMsS0FBSztJQVM1QyxZQUFZLFNBQWdDLHFCQUFxQixDQUFDLEtBQUs7UUFDckUsS0FBSyxFQUFFLENBQUM7UUFQRixXQUFNLEdBQUcsSUFBSSxLQUFLLEVBQU8sQ0FBQztRQUMxQixjQUFTLEdBQTZCLEVBQUUsQ0FBQztRQUN6QyxhQUFRLEdBQUcsSUFBSSxLQUFLLEVBQU8sQ0FBQztRQUM1QixjQUFTLEdBQTJCLEVBQUcsQ0FBQztRQUs5QyxJQUFJLENBQUMsTUFBTSxHQUFHLE1BQU0sQ0FBQztJQUN2QixDQUFDO0lBRUQsRUFBRTtJQUNGLFVBQVU7SUFDVixFQUFFO0lBRUssU0FBUyxDQUFDLE1BQWM7UUFDN0IsSUFBSSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDekIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRU0sVUFBVSxDQUFDLEdBQUcsT0FBaUI7UUFDcEMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxNQUFNLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztRQUNsRCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRCxFQUFFO0lBQ0YsWUFBWTtJQUNaLEVBQUU7SUFFRjs7T0FFRztJQUNILElBQVcsWUFBWTtRQUNyQixPQUFPLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUM7SUFDaEQsQ0FBQztJQUVNLFlBQVksQ0FBQyxTQUFxQjtRQUN2QyxNQUFNLFFBQVEsR0FBRyxTQUFTLENBQUMsY0FBYyxDQUFDO1FBQzFDLHFCQUFjLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDdkQsSUFBSSxDQUFDLGFBQWEsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDeEMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRU0sZUFBZSxDQUFDLEdBQVc7UUFDaEMsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUkseUJBQVksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFFTSxzQkFBc0IsQ0FBQyxTQUFpQjtRQUM3QyxPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSw2QkFBZ0IsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO0lBQzVELENBQUM7SUFFTSxlQUFlLENBQUMsR0FBVztRQUNoQyxPQUFPLElBQUksQ0FBQyxlQUFlLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7OztPQUtHO0lBQ0ksbUJBQW1CLENBQUMsT0FBZSxFQUFFLElBQTJCO1FBQ3JFLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLDZCQUFnQixDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO0lBQ2hFLENBQUM7SUFFTSxxQkFBcUIsQ0FBQyxTQUFjLEVBQUUsVUFBZ0M7UUFDM0UsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksK0JBQWtCLENBQUMsU0FBUyxFQUFFLFVBQVUsQ0FBQyxDQUFDLENBQUM7SUFDMUUsQ0FBQztJQUVNLHVCQUF1QjtRQUM1QixPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsSUFBSSxpQ0FBb0IsRUFBRSxDQUFDLENBQUM7SUFDdkQsQ0FBQztJQUVNLHlCQUF5QixDQUFDLGVBQXVCO1FBQ3RELE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLG1DQUFzQixDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUM7SUFDeEUsQ0FBQztJQUVNLGVBQWU7UUFDcEIsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksbUJBQU0sRUFBRSxDQUFDLENBQUM7SUFDekMsQ0FBQztJQUVELEVBQUU7SUFDRixZQUFZO0lBQ1osRUFBRTtJQUVLLFdBQVcsQ0FBQyxHQUFXO1FBQzVCLElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3hCLE9BQU8sSUFBSSxDQUFDO0lBQ2QsQ0FBQztJQUVEOztPQUVHO0lBQ0ksZUFBZTtRQUNwQixPQUFPLElBQUksQ0FBQyxXQUFXLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDL0IsQ0FBQztJQUVNLFlBQVksQ0FBQyxHQUFHLElBQWM7UUFDbkMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN2QyxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7T0FFRztJQUNILElBQVcsV0FBVztRQUNwQixPQUFPLElBQUksQ0FBQyxRQUFRLElBQUksSUFBSSxDQUFDLFFBQVEsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDO0lBQ25ELENBQUM7SUFFRDs7T0FFRztJQUNJLFFBQVEsQ0FBQyxHQUFXO1FBQ3pCLElBQUksQ0FBQyxHQUFHLEdBQUcsR0FBRyxDQUFDO1FBQ2YsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQsRUFBRTtJQUNGLFNBQVM7SUFDVCxFQUFFO0lBRUY7O09BRUc7SUFDSSxLQUFLO1FBQ1YsSUFBSSxDQUFDLE1BQU0sR0FBRyxxQkFBcUIsQ0FBQyxLQUFLLENBQUM7UUFDMUMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxJQUFJO1FBQ1QsSUFBSSxDQUFDLE1BQU0sR0FBRyxxQkFBcUIsQ0FBQyxJQUFJLENBQUM7UUFDekMsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQsRUFBRTtJQUNGLFlBQVk7SUFDWixFQUFFO0lBRUY7O09BRUc7SUFDSSxZQUFZLENBQUMsR0FBVyxFQUFFLEtBQVU7UUFDekMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsR0FBRyxLQUFLLENBQUM7UUFDNUIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQ7O09BRUc7SUFDSSxhQUFhLENBQUMsVUFBZ0M7UUFDbkQsTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLEVBQUU7WUFDaEMsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLEVBQUUsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7UUFDMUMsQ0FBQyxDQUFDLENBQUM7UUFDSCxPQUFPLElBQUksQ0FBQztJQUNkLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksWUFBWSxDQUFDLEdBQVcsRUFBRSxLQUFVO1FBQ3pDLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUVNLGNBQWMsQ0FBQyxTQUFpQjtRQUNyQyxPQUFPLElBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxFQUFFLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxHQUFHLEVBQUU7WUFDMUQsT0FBTyxFQUFFLGdCQUFnQixFQUFFLFNBQVMsRUFBRSxDQUFDO1FBQ3pDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDTixDQUFDO0lBRUQsRUFBRTtJQUNGLGdCQUFnQjtJQUNoQixFQUFFO0lBQ0ssT0FBTyxDQUFDLFFBQTZCO1FBQzFDLE9BQU8sSUFBSSxDQUFDLE1BQU0sRUFBRSxDQUFDO0lBQ3ZCLENBQUM7SUFFTSxNQUFNO1FBQ1gsT0FBTztZQUNMLE1BQU0sRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQztZQUMxQixTQUFTLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDaEMsTUFBTSxFQUFFLEtBQUssQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDO1lBQzFCLFNBQVMsRUFBRSxjQUFjLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQztZQUN6QyxRQUFRLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUM7WUFDOUIsR0FBRyxFQUFFLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDO1NBQ3JCLENBQUM7UUFFRixTQUFTLEtBQUssQ0FBQyxNQUFXO1lBRXhCLElBQUksT0FBTSxDQUFDLE1BQU0sQ0FBQyxLQUFLLFdBQVcsRUFBRTtnQkFDbEMsT0FBTyxTQUFTLENBQUM7YUFDbEI7WUFFRCxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFO2dCQUM3QixPQUFPLE1BQU0sQ0FBQzthQUNmO1lBRUQsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQyxFQUFFO2dCQUN6QixJQUFJLENBQUMsTUFBTSxJQUFJLE1BQU0sQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO29CQUNsQyxPQUFPLFNBQVMsQ0FBQztpQkFDbEI7Z0JBRUQsSUFBSSxNQUFNLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtvQkFDdkIsT0FBTyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUM7aUJBQ2xCO2dCQUVELE9BQU8sTUFBTSxDQUFDO2FBQ2Y7WUFFRCxJQUFJLE9BQU0sQ0FBQyxNQUFNLENBQUMsS0FBSyxRQUFRLEVBQUU7Z0JBQy9CLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO29CQUNwQyxPQUFPLFNBQVMsQ0FBQztpQkFDbEI7YUFDRjtZQUVELE9BQU8sTUFBTSxDQUFDO1FBQ2hCLENBQUM7UUFFRCxTQUFTLGNBQWMsQ0FBQyxTQUFtQztZQUN6RCxNQUFNLElBQUksR0FBRyxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1lBQ3BDLElBQUksSUFBSSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7Z0JBQUUsT0FBTyxTQUFTLENBQUM7YUFBRTtZQUM1QyxNQUFNLE1BQU0sR0FBUSxFQUFFLENBQUM7WUFDdkIsS0FBSyxNQUFNLEdBQUcsSUFBSSxJQUFJLEVBQUU7Z0JBQ3RCLE1BQU0sT0FBTyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztnQkFDdEMsSUFBSSxPQUFPLEVBQUU7b0JBQ1gsTUFBTSxDQUFDLEdBQUcsQ0FBQyxHQUFHLE9BQU8sQ0FBQztpQkFDdkI7YUFDRjtZQUNELElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxJQUFJLE1BQU0sQ0FBQyxHQUFHLEtBQUssR0FBRyxFQUFFO2dCQUMxRCxPQUFPLEdBQUcsQ0FBQzthQUNaO1lBQ0QsT0FBTyxNQUFNLENBQUM7UUFDaEIsQ0FBQztJQUNILENBQUM7Q0FDRjtBQWpQRCwwQ0FpUEM7QUFFRCxJQUFZLHFCQUdYO0FBSEQsV0FBWSxxQkFBcUI7SUFDL0Isd0NBQWUsQ0FBQTtJQUNmLHNDQUFhLENBQUE7QUFDZixDQUFDLEVBSFcscUJBQXFCLEdBQXJCLDZCQUFxQixLQUFyQiw2QkFBcUIsUUFHaEMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgY2RrID0gcmVxdWlyZSgnQGF3cy1jZGsvY2RrJyk7XG5pbXBvcnQgeyBBY2NvdW50UHJpbmNpcGFsLCBBY2NvdW50Um9vdFByaW5jaXBhbCwgQW55b25lLCBBcm5QcmluY2lwYWwsIENhbm9uaWNhbFVzZXJQcmluY2lwYWwsXG4gIEZlZGVyYXRlZFByaW5jaXBhbCwgSVByaW5jaXBhbCwgU2VydmljZVByaW5jaXBhbCwgU2VydmljZVByaW5jaXBhbE9wdHMgfSBmcm9tICcuL3ByaW5jaXBhbHMnO1xuaW1wb3J0IHsgbWVyZ2VQcmluY2lwYWwgfSBmcm9tICcuL3V0aWwnO1xuXG5leHBvcnQgY2xhc3MgUG9saWN5RG9jdW1lbnQgZXh0ZW5kcyBjZGsuVG9rZW4gaW1wbGVtZW50cyBjZGsuSVJlc29sdmVkVmFsdWVQb3N0UHJvY2Vzc29yIHtcbiAgcHJpdmF0ZSBzdGF0ZW1lbnRzID0gbmV3IEFycmF5PFBvbGljeVN0YXRlbWVudD4oKTtcbiAgcHJpdmF0ZSBfYXV0b0Fzc2lnblNpZHMgPSBmYWxzZTtcblxuICAvKipcbiAgICogQ3JlYXRlcyBhIG5ldyBJQU0gcG9saWN5IGRvY3VtZW50LlxuICAgKiBAcGFyYW0gZGVmYXVsdERvY3VtZW50IEFuIElBTSBwb2xpY3kgZG9jdW1lbnQgdG8gdXNlIGFzIGFuIGluaXRpYWxcbiAgICogcG9saWN5LiBBbGwgc3RhdGVtZW50cyBvZiB0aGlzIGRvY3VtZW50IHdpbGwgYmUgY29waWVkIGluLlxuICAgKi9cbiAgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBiYXNlRG9jdW1lbnQ6IGFueSA9IHt9KSB7XG4gICAgc3VwZXIoKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBXaWxsIGF1dG9tYXRpY2FsbHkgYXNzaWduIGEgdW5pcXVlIFNJRCB0byBlYWNoIHN0YXRlbWVudCwgdW5sZXNzIGFuIFNJRCBpcyBwcm92aWRlZC5cbiAgICovXG4gIHB1YmxpYyBhdXRvQXNzaWduU2lkcygpIHtcbiAgICB0aGlzLl9hdXRvQXNzaWduU2lkcyA9IHRydWU7XG4gIH1cblxuICBwdWJsaWMgcmVzb2x2ZShfY29udGV4dDogY2RrLklSZXNvbHZlQ29udGV4dCk6IGFueSB7XG4gICAgaWYgKHRoaXMuaXNFbXB0eSkge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG5cbiAgICBjb25zdCBkb2MgPSB7XG4gICAgICAuLi50aGlzLmJhc2VEb2N1bWVudCxcbiAgICAgIFN0YXRlbWVudDogKHRoaXMuYmFzZURvY3VtZW50LlN0YXRlbWVudCB8fCBbXSkuY29uY2F0KHRoaXMuc3RhdGVtZW50cyksXG4gICAgICBWZXJzaW9uOiB0aGlzLmJhc2VEb2N1bWVudC5WZXJzaW9uIHx8ICcyMDEyLTEwLTE3J1xuICAgIH07XG5cbiAgICByZXR1cm4gZG9jO1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbW92ZXMgZHVwbGljYXRlIHN0YXRlbWVudHNcbiAgICovXG4gIHB1YmxpYyBwb3N0UHJvY2VzcyhpbnB1dDogYW55LCBfY29udGV4dDogY2RrLklSZXNvbHZlQ29udGV4dCk6IGFueSB7XG4gICAgaWYgKCFpbnB1dCB8fCAhaW5wdXQuU3RhdGVtZW50KSB7XG4gICAgICByZXR1cm4gaW5wdXQ7XG4gICAgfVxuXG4gICAgY29uc3QganNvblN0YXRlbWVudHMgPSBuZXcgU2V0PHN0cmluZz4oKTtcbiAgICBjb25zdCB1bmlxdWVTdGF0ZW1lbnRzOiBhbnlbXSA9IFtdO1xuXG4gICAgZm9yIChjb25zdCBzdGF0ZW1lbnQgb2YgaW5wdXQuU3RhdGVtZW50KSB7XG4gICAgICBjb25zdCBqc29uU3RhdGVtZW50ID0gSlNPTi5zdHJpbmdpZnkoc3RhdGVtZW50KTtcbiAgICAgIGlmICghanNvblN0YXRlbWVudHMuaGFzKGpzb25TdGF0ZW1lbnQpKSB7XG4gICAgICAgIHVuaXF1ZVN0YXRlbWVudHMucHVzaChzdGF0ZW1lbnQpO1xuICAgICAgICBqc29uU3RhdGVtZW50cy5hZGQoanNvblN0YXRlbWVudCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgLy8gYXNzaWduIHVuaXF1ZSBTSURzICh0aGUgc3RhdGVtZW50IGluZGV4KSBpZiBgYXV0b0Fzc2lnblNpZHNgIGlzIGVuYWJsZWRcbiAgICBjb25zdCBzdGF0ZW1lbnRzID0gdW5pcXVlU3RhdGVtZW50cy5tYXAoKHMsIGkpID0+IHtcbiAgICAgIGlmICh0aGlzLl9hdXRvQXNzaWduU2lkcyAmJiAhcy5TaWQpIHtcbiAgICAgICAgcy5TaWQgPSBpLnRvU3RyaW5nKCk7XG4gICAgICB9XG5cbiAgICAgIHJldHVybiBzO1xuICAgIH0pO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIC4uLmlucHV0LFxuICAgICAgU3RhdGVtZW50OiBzdGF0ZW1lbnRzXG4gICAgfTtcbiAgfVxuXG4gIGdldCBpc0VtcHR5KCk6IGJvb2xlYW4ge1xuICAgIHJldHVybiB0aGlzLnN0YXRlbWVudHMubGVuZ3RoID09PSAwO1xuICB9XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2Ygc3RhdGVtZW50cyBhbHJlYWR5IGFkZGVkIHRvIHRoaXMgcG9saWN5LlxuICAgKiBDYW4gYmUgdXNlZCwgZm9yIGV4YW1wbGUsIHRvIGdlbmVyYXRlIHVuaXVxZSBcInNpZFwicyB3aXRoaW4gdGhlIHBvbGljeS5cbiAgICovXG4gIGdldCBzdGF0ZW1lbnRDb3VudCgpOiBudW1iZXIge1xuICAgIHJldHVybiB0aGlzLnN0YXRlbWVudHMubGVuZ3RoO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBzdGF0ZW1lbnQgdG8gdGhlIHBvbGljeSBkb2N1bWVudC5cbiAgICpcbiAgICogQHBhcmFtIHN0YXRlbWVudCB0aGUgc3RhdGVtZW50IHRvIGFkZC5cbiAgICovXG4gIHB1YmxpYyBhZGRTdGF0ZW1lbnQoc3RhdGVtZW50OiBQb2xpY3lTdGF0ZW1lbnQpOiBQb2xpY3lEb2N1bWVudCB7XG4gICAgdGhpcy5zdGF0ZW1lbnRzLnB1c2goc3RhdGVtZW50KTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxufVxuXG4vKipcbiAqIFJlcHJlc2VudHMgYSBzdGF0ZW1lbnQgaW4gYW4gSUFNIHBvbGljeSBkb2N1bWVudC5cbiAqL1xuZXhwb3J0IGNsYXNzIFBvbGljeVN0YXRlbWVudCBleHRlbmRzIGNkay5Ub2tlbiB7XG4gIHB1YmxpYyBzaWQ/OiBzdHJpbmc7XG5cbiAgcHJpdmF0ZSBhY3Rpb24gPSBuZXcgQXJyYXk8YW55PigpO1xuICBwcml2YXRlIHByaW5jaXBhbDogeyBba2V5OiBzdHJpbmddOiBhbnlbXSB9ID0ge307XG4gIHByaXZhdGUgcmVzb3VyY2UgPSBuZXcgQXJyYXk8YW55PigpO1xuICBwcml2YXRlIGNvbmRpdGlvbjogeyBba2V5OiBzdHJpbmddOiBhbnkgfSA9IHsgfTtcbiAgcHJpdmF0ZSBlZmZlY3Q/OiBQb2xpY3lTdGF0ZW1lbnRFZmZlY3Q7XG5cbiAgY29uc3RydWN0b3IoZWZmZWN0OiBQb2xpY3lTdGF0ZW1lbnRFZmZlY3QgPSBQb2xpY3lTdGF0ZW1lbnRFZmZlY3QuQWxsb3cpIHtcbiAgICBzdXBlcigpO1xuICAgIHRoaXMuZWZmZWN0ID0gZWZmZWN0O1xuICB9XG5cbiAgLy9cbiAgLy8gQWN0aW9uc1xuICAvL1xuXG4gIHB1YmxpYyBhZGRBY3Rpb24oYWN0aW9uOiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnQge1xuICAgIHRoaXMuYWN0aW9uLnB1c2goYWN0aW9uKTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIHB1YmxpYyBhZGRBY3Rpb25zKC4uLmFjdGlvbnM6IHN0cmluZ1tdKTogUG9saWN5U3RhdGVtZW50IHtcbiAgICBhY3Rpb25zLmZvckVhY2goYWN0aW9uID0+IHRoaXMuYWRkQWN0aW9uKGFjdGlvbikpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLy9cbiAgLy8gUHJpbmNpcGFsXG4gIC8vXG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyBpZiB0aGlzIHBlcm1pc3Npb24gaGFzIGEgXCJQcmluY2lwYWxcIiBzZWN0aW9uLlxuICAgKi9cbiAgcHVibGljIGdldCBoYXNQcmluY2lwYWwoKSB7XG4gICAgcmV0dXJuIE9iamVjdC5rZXlzKHRoaXMucHJpbmNpcGFsKS5sZW5ndGggPiAwO1xuICB9XG5cbiAgcHVibGljIGFkZFByaW5jaXBhbChwcmluY2lwYWw6IElQcmluY2lwYWwpOiB0aGlzIHtcbiAgICBjb25zdCBmcmFnbWVudCA9IHByaW5jaXBhbC5wb2xpY3lGcmFnbWVudDtcbiAgICBtZXJnZVByaW5jaXBhbCh0aGlzLnByaW5jaXBhbCwgZnJhZ21lbnQucHJpbmNpcGFsSnNvbik7XG4gICAgdGhpcy5hZGRDb25kaXRpb25zKGZyYWdtZW50LmNvbmRpdGlvbnMpO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgcHVibGljIGFkZEF3c1ByaW5jaXBhbChhcm46IHN0cmluZyk6IHRoaXMge1xuICAgIHJldHVybiB0aGlzLmFkZFByaW5jaXBhbChuZXcgQXJuUHJpbmNpcGFsKGFybikpO1xuICB9XG5cbiAgcHVibGljIGFkZEF3c0FjY291bnRQcmluY2lwYWwoYWNjb3VudElkOiBzdHJpbmcpOiB0aGlzIHtcbiAgICByZXR1cm4gdGhpcy5hZGRQcmluY2lwYWwobmV3IEFjY291bnRQcmluY2lwYWwoYWNjb3VudElkKSk7XG4gIH1cblxuICBwdWJsaWMgYWRkQXJuUHJpbmNpcGFsKGFybjogc3RyaW5nKTogdGhpcyB7XG4gICAgcmV0dXJuIHRoaXMuYWRkQXdzUHJpbmNpcGFsKGFybik7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIHNlcnZpY2UgcHJpbmNpcGFsIHRvIHRoaXMgcG9saWN5IHN0YXRlbWVudC5cbiAgICpcbiAgICogQHBhcmFtIHNlcnZpY2UgdGhlIHNlcnZpY2UgbmFtZSBmb3Igd2hpY2ggYSBzZXJ2aWNlIHByaW5jaXBhbCBpcyByZXF1ZXN0ZWQgKGUuZzogYHMzLmFtYXpvbmF3cy5jb21gKS5cbiAgICogQHBhcmFtIG9wdHMgICAgb3B0aW9ucyBmb3IgYWRkaW5nIHRoZSBzZXJ2aWNlIHByaW5jaXBhbCAoc3VjaCBhcyBzcGVjaWZ5aW5nIGEgcHJpbmNpcGFsIGluIGEgZGlmZmVyZW50IHJlZ2lvbilcbiAgICovXG4gIHB1YmxpYyBhZGRTZXJ2aWNlUHJpbmNpcGFsKHNlcnZpY2U6IHN0cmluZywgb3B0cz86IFNlcnZpY2VQcmluY2lwYWxPcHRzKTogdGhpcyB7XG4gICAgcmV0dXJuIHRoaXMuYWRkUHJpbmNpcGFsKG5ldyBTZXJ2aWNlUHJpbmNpcGFsKHNlcnZpY2UsIG9wdHMpKTtcbiAgfVxuXG4gIHB1YmxpYyBhZGRGZWRlcmF0ZWRQcmluY2lwYWwoZmVkZXJhdGVkOiBhbnksIGNvbmRpdGlvbnM6IHtba2V5OiBzdHJpbmddOiBhbnl9KTogdGhpcyB7XG4gICAgcmV0dXJuIHRoaXMuYWRkUHJpbmNpcGFsKG5ldyBGZWRlcmF0ZWRQcmluY2lwYWwoZmVkZXJhdGVkLCBjb25kaXRpb25zKSk7XG4gIH1cblxuICBwdWJsaWMgYWRkQWNjb3VudFJvb3RQcmluY2lwYWwoKTogdGhpcyB7XG4gICAgcmV0dXJuIHRoaXMuYWRkUHJpbmNpcGFsKG5ldyBBY2NvdW50Um9vdFByaW5jaXBhbCgpKTtcbiAgfVxuXG4gIHB1YmxpYyBhZGRDYW5vbmljYWxVc2VyUHJpbmNpcGFsKGNhbm9uaWNhbFVzZXJJZDogc3RyaW5nKTogdGhpcyB7XG4gICAgcmV0dXJuIHRoaXMuYWRkUHJpbmNpcGFsKG5ldyBDYW5vbmljYWxVc2VyUHJpbmNpcGFsKGNhbm9uaWNhbFVzZXJJZCkpO1xuICB9XG5cbiAgcHVibGljIGFkZEFueVByaW5jaXBhbCgpOiB0aGlzIHtcbiAgICByZXR1cm4gdGhpcy5hZGRQcmluY2lwYWwobmV3IEFueW9uZSgpKTtcbiAgfVxuXG4gIC8vXG4gIC8vIFJlc291cmNlc1xuICAvL1xuXG4gIHB1YmxpYyBhZGRSZXNvdXJjZShhcm46IHN0cmluZyk6IFBvbGljeVN0YXRlbWVudCB7XG4gICAgdGhpcy5yZXNvdXJjZS5wdXNoKGFybik7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIGBgXCIqXCJgYCByZXNvdXJjZSB0byB0aGlzIHN0YXRlbWVudC5cbiAgICovXG4gIHB1YmxpYyBhZGRBbGxSZXNvdXJjZXMoKTogUG9saWN5U3RhdGVtZW50IHtcbiAgICByZXR1cm4gdGhpcy5hZGRSZXNvdXJjZSgnKicpO1xuICB9XG5cbiAgcHVibGljIGFkZFJlc291cmNlcyguLi5hcm5zOiBzdHJpbmdbXSk6IFBvbGljeVN0YXRlbWVudCB7XG4gICAgYXJucy5mb3JFYWNoKHIgPT4gdGhpcy5hZGRSZXNvdXJjZShyKSk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogSW5kaWNhdGVzIGlmIHRoaXMgcGVybWlzc2lvbiBhcyBhdCBsZWFzdCBvbmUgcmVzb3VyY2UgYXNzb2NpYXRlZCB3aXRoIGl0LlxuICAgKi9cbiAgcHVibGljIGdldCBoYXNSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy5yZXNvdXJjZSAmJiB0aGlzLnJlc291cmNlLmxlbmd0aCA+IDA7XG4gIH1cblxuICAvKipcbiAgICogQGRlcHJlY2F0ZWQgVXNlIGBzdGF0ZW1lbnQuc2lkID0gdmFsdWVgXG4gICAqL1xuICBwdWJsaWMgZGVzY3JpYmUoc2lkOiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnQge1xuICAgIHRoaXMuc2lkID0gc2lkO1xuICAgIHJldHVybiB0aGlzO1xuICB9XG5cbiAgLy9cbiAgLy8gRWZmZWN0XG4gIC8vXG5cbiAgLyoqXG4gICAqIFNldHMgdGhlIHBlcm1pc3Npb24gZWZmZWN0IHRvIGFsbG93IGFjY2VzcyB0byByZXNvdXJjZXMuXG4gICAqL1xuICBwdWJsaWMgYWxsb3coKTogUG9saWN5U3RhdGVtZW50IHtcbiAgICB0aGlzLmVmZmVjdCA9IFBvbGljeVN0YXRlbWVudEVmZmVjdC5BbGxvdztcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBTZXRzIHRoZSBwZXJtaXNzaW9uIGVmZmVjdCB0byBkZW55IGFjY2VzcyB0byByZXNvdXJjZXMuXG4gICAqL1xuICBwdWJsaWMgZGVueSgpOiBQb2xpY3lTdGF0ZW1lbnQge1xuICAgIHRoaXMuZWZmZWN0ID0gUG9saWN5U3RhdGVtZW50RWZmZWN0LkRlbnk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvL1xuICAvLyBDb25kaXRpb25cbiAgLy9cblxuICAvKipcbiAgICogQWRkIGEgY29uZGl0aW9uIHRvIHRoZSBQb2xpY3lcbiAgICovXG4gIHB1YmxpYyBhZGRDb25kaXRpb24oa2V5OiBzdHJpbmcsIHZhbHVlOiBhbnkpOiBQb2xpY3lTdGF0ZW1lbnQge1xuICAgIHRoaXMuY29uZGl0aW9uW2tleV0gPSB2YWx1ZTtcbiAgICByZXR1cm4gdGhpcztcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgbXVsdGlwbGUgY29uZGl0aW9ucyB0byB0aGUgUG9saWN5XG4gICAqL1xuICBwdWJsaWMgYWRkQ29uZGl0aW9ucyhjb25kaXRpb25zOiB7W2tleTogc3RyaW5nXTogYW55fSk6IFBvbGljeVN0YXRlbWVudCB7XG4gICAgT2JqZWN0LmtleXMoY29uZGl0aW9ucykubWFwKGtleSA9PiB7XG4gICAgICB0aGlzLmFkZENvbmRpdGlvbihrZXksIGNvbmRpdGlvbnNba2V5XSk7XG4gICAgfSk7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgY29uZGl0aW9uIHRvIHRoZSBQb2xpY3kuXG4gICAqXG4gICAqIEBkZXByZWNhdGVkIEZvciBiYWNrd2FyZHMgY29tcGF0aWJpbGl0eS4gVXNlIGFkZENvbmRpdGlvbigpIGluc3RlYWQuXG4gICAqL1xuICBwdWJsaWMgc2V0Q29uZGl0aW9uKGtleTogc3RyaW5nLCB2YWx1ZTogYW55KTogUG9saWN5U3RhdGVtZW50IHtcbiAgICByZXR1cm4gdGhpcy5hZGRDb25kaXRpb24oa2V5LCB2YWx1ZSk7XG4gIH1cblxuICBwdWJsaWMgbGltaXRUb0FjY291bnQoYWNjb3VudElkOiBzdHJpbmcpOiBQb2xpY3lTdGF0ZW1lbnQge1xuICAgIHJldHVybiB0aGlzLmFkZENvbmRpdGlvbignU3RyaW5nRXF1YWxzJywgbmV3IGNkay5Ub2tlbigoKSA9PiB7XG4gICAgICByZXR1cm4geyAnc3RzOkV4dGVybmFsSWQnOiBhY2NvdW50SWQgfTtcbiAgICB9KSk7XG4gIH1cblxuICAvL1xuICAvLyBTZXJpYWxpemF0aW9uXG4gIC8vXG4gIHB1YmxpYyByZXNvbHZlKF9jb250ZXh0OiBjZGsuSVJlc29sdmVDb250ZXh0KTogYW55IHtcbiAgICByZXR1cm4gdGhpcy50b0pzb24oKTtcbiAgfVxuXG4gIHB1YmxpYyB0b0pzb24oKTogYW55IHtcbiAgICByZXR1cm4ge1xuICAgICAgQWN0aW9uOiBfbm9ybSh0aGlzLmFjdGlvbiksXG4gICAgICBDb25kaXRpb246IF9ub3JtKHRoaXMuY29uZGl0aW9uKSxcbiAgICAgIEVmZmVjdDogX25vcm0odGhpcy5lZmZlY3QpLFxuICAgICAgUHJpbmNpcGFsOiBfbm9ybVByaW5jaXBhbCh0aGlzLnByaW5jaXBhbCksXG4gICAgICBSZXNvdXJjZTogX25vcm0odGhpcy5yZXNvdXJjZSksXG4gICAgICBTaWQ6IF9ub3JtKHRoaXMuc2lkKSxcbiAgICB9O1xuXG4gICAgZnVuY3Rpb24gX25vcm0odmFsdWVzOiBhbnkpIHtcblxuICAgICAgaWYgKHR5cGVvZih2YWx1ZXMpID09PSAndW5kZWZpbmVkJykge1xuICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgfVxuXG4gICAgICBpZiAoY2RrLlRva2VuLmlzVG9rZW4odmFsdWVzKSkge1xuICAgICAgICByZXR1cm4gdmFsdWVzO1xuICAgICAgfVxuXG4gICAgICBpZiAoQXJyYXkuaXNBcnJheSh2YWx1ZXMpKSB7XG4gICAgICAgIGlmICghdmFsdWVzIHx8IHZhbHVlcy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKHZhbHVlcy5sZW5ndGggPT09IDEpIHtcbiAgICAgICAgICByZXR1cm4gdmFsdWVzWzBdO1xuICAgICAgICB9XG5cbiAgICAgICAgcmV0dXJuIHZhbHVlcztcbiAgICAgIH1cblxuICAgICAgaWYgKHR5cGVvZih2YWx1ZXMpID09PSAnb2JqZWN0Jykge1xuICAgICAgICBpZiAoT2JqZWN0LmtleXModmFsdWVzKS5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG4gICAgICB9XG5cbiAgICAgIHJldHVybiB2YWx1ZXM7XG4gICAgfVxuXG4gICAgZnVuY3Rpb24gX25vcm1QcmluY2lwYWwocHJpbmNpcGFsOiB7IFtrZXk6IHN0cmluZ106IGFueVtdIH0pIHtcbiAgICAgIGNvbnN0IGtleXMgPSBPYmplY3Qua2V5cyhwcmluY2lwYWwpO1xuICAgICAgaWYgKGtleXMubGVuZ3RoID09PSAwKSB7IHJldHVybiB1bmRlZmluZWQ7IH1cbiAgICAgIGNvbnN0IHJlc3VsdDogYW55ID0ge307XG4gICAgICBmb3IgKGNvbnN0IGtleSBvZiBrZXlzKSB7XG4gICAgICAgIGNvbnN0IG5vcm1WYWwgPSBfbm9ybShwcmluY2lwYWxba2V5XSk7XG4gICAgICAgIGlmIChub3JtVmFsKSB7XG4gICAgICAgICAgcmVzdWx0W2tleV0gPSBub3JtVmFsO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgICBpZiAoT2JqZWN0LmtleXMocmVzdWx0KS5sZW5ndGggPT09IDEgJiYgcmVzdWx0LkFXUyA9PT0gJyonKSB7XG4gICAgICAgIHJldHVybiAnKic7XG4gICAgICB9XG4gICAgICByZXR1cm4gcmVzdWx0O1xuICAgIH1cbiAgfVxufVxuXG5leHBvcnQgZW51bSBQb2xpY3lTdGF0ZW1lbnRFZmZlY3Qge1xuICBBbGxvdyA9ICdBbGxvdycsXG4gIERlbnkgPSAnRGVueScsXG59XG4iXX0=