"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk_1 = require("@aws-cdk/cdk");
const iam_generated_1 = require("./iam.generated");
const policy_document_1 = require("./policy-document");
const util_1 = require("./util");
/**
 * The AWS::IAM::Policy resource associates an IAM policy with IAM users, roles,
 * or groups. For more information about IAM policies, see [Overview of IAM
 * Policies](http://docs.aws.amazon.com/IAM/latest/UserGuide/policies_overview.html)
 * in the IAM User Guide guide.
 */
class Policy extends cdk_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        /**
         * The policy document.
         */
        this.document = new policy_document_1.PolicyDocument();
        this.roles = new Array();
        this.users = new Array();
        this.groups = new Array();
        const resource = new iam_generated_1.CfnPolicy(this, 'Resource', {
            policyDocument: this.document,
            policyName: new cdk_1.Token(() => this.policyName).toString(),
            roles: util_1.undefinedIfEmpty(() => this.roles.map(r => r.roleName)),
            users: util_1.undefinedIfEmpty(() => this.users.map(u => u.userName)),
            groups: util_1.undefinedIfEmpty(() => this.groups.map(g => g.groupName)),
        });
        // generatePolicyName will take the last 128 characters of the logical id since
        // policy names are limited to 128. the last 8 chars are a stack-unique hash, so
        // that shouod be sufficient to ensure uniqueness within a principal.
        this.policyName = props.policyName || util_1.generatePolicyName(resource.logicalId);
        if (props.users) {
            props.users.forEach(u => this.attachToUser(u));
        }
        if (props.groups) {
            props.groups.forEach(g => this.attachToGroup(g));
        }
        if (props.roles) {
            props.roles.forEach(r => this.attachToRole(r));
        }
        if (props.statements) {
            props.statements.forEach(p => this.addStatement(p));
        }
    }
    static fromPolicyName(scope, id, policyName) {
        class Import extends cdk_1.Resource {
            constructor() {
                super(...arguments);
                this.policyName = policyName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a statement to the policy document.
     */
    addStatement(statement) {
        this.document.addStatement(statement);
    }
    /**
     * Attaches this policy to a user.
     */
    attachToUser(user) {
        if (this.users.find(u => u === user)) {
            return;
        }
        this.users.push(user);
        user.attachInlinePolicy(this);
    }
    /**
     * Attaches this policy to a role.
     */
    attachToRole(role) {
        if (this.roles.find(r => r === role)) {
            return;
        }
        this.roles.push(role);
        role.attachInlinePolicy(this);
    }
    /**
     * Attaches this policy to a group.
     */
    attachToGroup(group) {
        if (this.groups.find(g => g === group)) {
            return;
        }
        this.groups.push(group);
        group.attachInlinePolicy(this);
    }
    validate() {
        const result = new Array();
        // validate that the policy document is not empty
        if (this.document.isEmpty) {
            result.push('Policy is empty. You must add statements to the policy');
        }
        // validate that the policy is attached to at least one principal (role, user or group).
        if (this.groups.length + this.users.length + this.roles.length === 0) {
            result.push(`Policy must be attached to at least one principal: user, group or role`);
        }
        return result;
    }
}
exports.Policy = Policy;
//# sourceMappingURL=data:application/json;base64,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