import { PolicyStatement } from './policy-document';
/**
 * Any object that has an associated principal that a permission can be granted to
 */
export interface IGrantable {
    /**
     * The principal to grant permissions to
     */
    readonly grantPrincipal: IPrincipal;
}
/**
 * Represents a logical IAM principal.
 *
 * An IPrincipal describes a logical entity that can perform AWS API calls
 * against sets of resources, optionally under certain conditions.
 *
 * Examples of simple principals are IAM objects that you create, such
 * as Users or Roles.
 *
 * An example of a more complex principals is a `ServicePrincipal` (such as
 * `new ServicePrincipal("sns.amazonaws.com")`, which represents the Simple
 * Notifications Service).
 *
 * A single logical Principal may also map to a set of physical principals.
 * For example, `new OrganizationPrincipal('o-1234')` represents all
 * identities that are part of the given AWS Organization.
 */
export interface IPrincipal extends IGrantable {
    /**
     * When this Principal is used in an AssumeRole policy, the action to use.
     */
    readonly assumeRoleAction: string;
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     */
    readonly policyFragment: PrincipalPolicyFragment;
    /**
     * Add to the policy of this principal.
     *
     * @returns true if the statement was added, false if the principal in
     * question does not have a policy document to add the statement to.
     */
    addToPolicy(statement: PolicyStatement): boolean;
}
/**
 * Base class for policy principals
 */
export declare abstract class PrincipalBase implements IPrincipal {
    readonly grantPrincipal: IPrincipal;
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     */
    abstract readonly policyFragment: PrincipalPolicyFragment;
    /**
     * When this Principal is used in an AssumeRole policy, the action to use.
     */
    readonly assumeRoleAction: string;
    addToPolicy(_statement: PolicyStatement): boolean;
    toString(): string;
    toJSON(): {
        [key: string]: string[];
    };
}
/**
 * A collection of the fields in a PolicyStatement that can be used to identify a principal.
 *
 * This consists of the JSON used in the "Principal" field, and optionally a
 * set of "Condition"s that need to be applied to the policy.
 */
export declare class PrincipalPolicyFragment {
    readonly principalJson: {
        [key: string]: string[];
    };
    readonly conditions: {
        [key: string]: any;
    };
    constructor(principalJson: {
        [key: string]: string[];
    }, conditions?: {
        [key: string]: any;
    });
}
export declare class ArnPrincipal extends PrincipalBase {
    readonly arn: string;
    constructor(arn: string);
    readonly policyFragment: PrincipalPolicyFragment;
    toString(): string;
}
export declare class AccountPrincipal extends ArnPrincipal {
    readonly accountId: any;
    constructor(accountId: any);
    toString(): string;
}
/**
 * Options for a service principal.
 */
export interface ServicePrincipalOpts {
    /**
     * The region in which the service is operating.
     *
     * @default the current Stack's region.
     */
    readonly region?: string;
    /**
     * Additional conditions to add to the Service Principal
     *
     * @default - No conditions
     */
    readonly conditions?: {
        [key: string]: any;
    };
}
/**
 * An IAM principal that represents an AWS service (i.e. sqs.amazonaws.com).
 */
export declare class ServicePrincipal extends PrincipalBase {
    readonly service: string;
    private readonly opts;
    constructor(service: string, opts?: ServicePrincipalOpts);
    readonly policyFragment: PrincipalPolicyFragment;
    toString(): string;
}
/**
 * A principal that represents an AWS Organization
 */
export declare class OrganizationPrincipal extends PrincipalBase {
    readonly organizationId: string;
    constructor(organizationId: string);
    readonly policyFragment: PrincipalPolicyFragment;
    toString(): string;
}
/**
 * A policy prinicipal for canonicalUserIds - useful for S3 bucket policies that use
 * Origin Access identities.
 *
 * See https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html
 *
 * and
 *
 * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 *
 * for more details.
 *
 */
export declare class CanonicalUserPrincipal extends PrincipalBase {
    readonly canonicalUserId: string;
    constructor(canonicalUserId: string);
    readonly policyFragment: PrincipalPolicyFragment;
    toString(): string;
}
export declare class FederatedPrincipal extends PrincipalBase {
    readonly federated: string;
    readonly conditions: {
        [key: string]: any;
    };
    readonly assumeRoleAction: string;
    constructor(federated: string, conditions: {
        [key: string]: any;
    }, assumeRoleAction?: string);
    readonly policyFragment: PrincipalPolicyFragment;
    toString(): string;
}
export declare class AccountRootPrincipal extends AccountPrincipal {
    constructor();
    toString(): string;
}
/**
 * A principal representing all identities in all accounts
 */
export declare class AnyPrincipal extends ArnPrincipal {
    constructor();
    toString(): string;
}
/**
 * A principal representing all identities in all accounts
 * @deprecated use `AnyPrincipal`
 */
export declare class Anyone extends AnyPrincipal {
}
export declare class CompositePrincipal extends PrincipalBase {
    readonly assumeRoleAction: string;
    private readonly principals;
    constructor(...principals: PrincipalBase[]);
    addPrincipals(...principals: PrincipalBase[]): this;
    readonly policyFragment: PrincipalPolicyFragment;
    toString(): string;
}
