"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const region_info_1 = require("@aws-cdk/region-info");
const util_1 = require("./util");
/**
 * Base class for policy principals
 */
class PrincipalBase {
    constructor() {
        this.grantPrincipal = this;
        /**
         * When this Principal is used in an AssumeRole policy, the action to use.
         */
        this.assumeRoleAction = 'sts:AssumeRole';
    }
    addToPolicy(_statement) {
        // This base class is used for non-identity principals. None of them
        // have a PolicyDocument to add to.
        return false;
    }
    toString() {
        // This is a first pass to make the object readable. Descendant principals
        // should return something nicer.
        return JSON.stringify(this.policyFragment.principalJson);
    }
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
}
exports.PrincipalBase = PrincipalBase;
/**
 * A collection of the fields in a PolicyStatement that can be used to identify a principal.
 *
 * This consists of the JSON used in the "Principal" field, and optionally a
 * set of "Condition"s that need to be applied to the policy.
 */
class PrincipalPolicyFragment {
    constructor(principalJson, conditions = {}) {
        this.principalJson = principalJson;
        this.conditions = conditions;
    }
}
exports.PrincipalPolicyFragment = PrincipalPolicyFragment;
class ArnPrincipal extends PrincipalBase {
    constructor(arn) {
        super();
        this.arn = arn;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: [this.arn] });
    }
    toString() {
        return `ArnPrincipal(${this.arn})`;
    }
}
exports.ArnPrincipal = ArnPrincipal;
class AccountPrincipal extends ArnPrincipal {
    constructor(accountId) {
        super(new StackDependentToken(stack => `arn:${stack.partition}:iam::${accountId}:root`).toString());
        this.accountId = accountId;
    }
    toString() {
        return `AccountPrincipal(${this.accountId})`;
    }
}
exports.AccountPrincipal = AccountPrincipal;
/**
 * An IAM principal that represents an AWS service (i.e. sqs.amazonaws.com).
 */
class ServicePrincipal extends PrincipalBase {
    constructor(service, opts = {}) {
        super();
        this.service = service;
        this.opts = opts;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({
            Service: [
                new ServicePrincipalToken(this.service, this.opts).toString()
            ]
        }, this.opts.conditions);
    }
    toString() {
        return `ServicePrincipal(${this.service})`;
    }
}
exports.ServicePrincipal = ServicePrincipal;
/**
 * A principal that represents an AWS Organization
 */
class OrganizationPrincipal extends PrincipalBase {
    constructor(organizationId) {
        super();
        this.organizationId = organizationId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: ['*'] }, { StringEquals: { 'aws:PrincipalOrgID': this.organizationId } });
    }
    toString() {
        return `OrganizationPrincipal(${this.organizationId})`;
    }
}
exports.OrganizationPrincipal = OrganizationPrincipal;
/**
 * A policy prinicipal for canonicalUserIds - useful for S3 bucket policies that use
 * Origin Access identities.
 *
 * See https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html
 *
 * and
 *
 * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 *
 * for more details.
 *
 */
class CanonicalUserPrincipal extends PrincipalBase {
    constructor(canonicalUserId) {
        super();
        this.canonicalUserId = canonicalUserId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ CanonicalUser: [this.canonicalUserId] });
    }
    toString() {
        return `CanonicalUserPrincipal(${this.canonicalUserId})`;
    }
}
exports.CanonicalUserPrincipal = CanonicalUserPrincipal;
class FederatedPrincipal extends PrincipalBase {
    constructor(federated, conditions, assumeRoleAction = 'sts:AssumeRole') {
        super();
        this.federated = federated;
        this.conditions = conditions;
        this.assumeRoleAction = assumeRoleAction;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `FederatedPrincipal(${this.federated})`;
    }
}
exports.FederatedPrincipal = FederatedPrincipal;
class AccountRootPrincipal extends AccountPrincipal {
    constructor() {
        super(new StackDependentToken(stack => stack.accountId).toString());
    }
    toString() {
        return `AccountRootPrincipal()`;
    }
}
exports.AccountRootPrincipal = AccountRootPrincipal;
/**
 * A principal representing all identities in all accounts
 */
class AnyPrincipal extends ArnPrincipal {
    constructor() {
        super('*');
    }
    toString() {
        return `AnyPrincipal()`;
    }
}
exports.AnyPrincipal = AnyPrincipal;
/**
 * A principal representing all identities in all accounts
 * @deprecated use `AnyPrincipal`
 */
class Anyone extends AnyPrincipal {
}
exports.Anyone = Anyone;
class CompositePrincipal extends PrincipalBase {
    constructor(...principals) {
        super();
        this.principals = new Array();
        if (principals.length === 0) {
            throw new Error('CompositePrincipals must be constructed with at least 1 Principal but none were passed.');
        }
        this.assumeRoleAction = principals[0].assumeRoleAction;
        this.addPrincipals(...principals);
    }
    addPrincipals(...principals) {
        for (const p of principals) {
            if (p.assumeRoleAction !== this.assumeRoleAction) {
                throw new Error(`Cannot add multiple principals with different "assumeRoleAction". ` +
                    `Expecting "${this.assumeRoleAction}", got "${p.assumeRoleAction}"`);
            }
            const fragment = p.policyFragment;
            if (fragment.conditions && Object.keys(fragment.conditions).length > 0) {
                throw new Error(`Components of a CompositePrincipal must not have conditions. ` +
                    `Tried to add the following fragment: ${JSON.stringify(fragment)}`);
            }
            this.principals.push(p);
        }
        return this;
    }
    get policyFragment() {
        const principalJson = {};
        for (const p of this.principals) {
            util_1.mergePrincipal(principalJson, p.policyFragment.principalJson);
        }
        return new PrincipalPolicyFragment(principalJson);
    }
    toString() {
        return `CompositePrincipal(${this.principals})`;
    }
}
exports.CompositePrincipal = CompositePrincipal;
/**
 * A lazy token that requires an instance of Stack to evaluate
 */
class StackDependentToken extends cdk.Token {
    constructor(fn) {
        super();
        this.fn = fn;
    }
    resolve(context) {
        return this.fn(context.scope.node.stack);
    }
}
class ServicePrincipalToken extends cdk.Token {
    constructor(service, opts) {
        super();
        this.service = service;
        this.opts = opts;
    }
    resolve(ctx) {
        const region = this.opts.region || ctx.scope.node.stack.region;
        const fact = region_info_1.RegionInfo.get(region).servicePrincipal(this.service);
        return fact || region_info_1.Default.servicePrincipal(this.service, region, ctx.scope.node.stack.urlSuffix);
    }
}
//# sourceMappingURL=data:application/json;base64,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